# -*- coding: latin-1; -*-
#
# PgWorksheet - PostgreSQL Front End
# http://pgworksheet.projects.postgresql.org/
#
# Copyright © 2004-2008 Henri Michelon & CML http://www.e-cml.org/
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (read LICENSE.txt).
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# $Id: UI.py,v 1.26 2008/03/12 20:26:23 hmichelon Exp $
#
import os
import ConfigParser
import pygtk
import gtk

import pgw
import pgw.Syntax
import pgw.Undo

PGW_CONNECT     = 'pgw-connect'
PGW_CONNECTNOW  = 'pgw-connectnow'
PGW_DISCONNECT  = 'pwg-disconnect'
PGW_SELECTALL   = 'pwg-selectall'
PGW_ENCODING    = 'pwg-encoding'
PGW_ABOUT       = 'pwg-about'


class UI:
  """UI Construction and management"""

  def __init__(self, app, app_name, app_version, pixmap_path):
    self.pixmap_path = pixmap_path
    self.app_version = app_version
    self.register_stock_items()
    self.file_dialog_path = None # last used path
    self.file_dialog_filename = None # last used filename
    self.define_ui(app)

    self.wndmain = gtk.Window(gtk.WINDOW_TOPLEVEL)
    self.wndmain.set_title(_('PgWorksheet - PostgreSQL SQL Tool'))
    self.wndmain.set_icon_from_file(os.path.join(pixmap_path,
                                                 "pgworksheet-32.png"))
    self.wndmain.add_accel_group(self.accelgroup)
    self.wndmain.set_default_size(640, 480)
    self.wndmain.connect('window-state-event', self.on_wndmain_state)
    self.wndmain.connect('configure-event', self.on_wndmain_configure)
    self.wndmain.connect('destroy', app.on_wndmain_destroy)
    self.wndmain.connect('delete_event', app.on_wndmain_delete)

    hbox = gtk.VBox(False)
    hbox.pack_start(self.uimanager.get_widget('/MenuBar'), False)
    hbox.pack_start(self.uimanager.get_widget('/ToolBar'), False)
    hbox.pack_start(self.vmain, True, True)
    hbox.pack_start(self.statusbar, False, False)
    self.wndmain.add(hbox)
    
    self.load_user_parameters()
    self.wndmain.show_all()
    

  def load_user_parameters(self):
    cp = ConfigParser.ConfigParser()
    try:
      cp.readfp(open(pgw.get_config_path(), 'r'))
      maximized = 0
      try:
        maximized = int(cp.get("window", "maximized"))
      except:
        pass
      if (maximized):
        self.wndmain.maximize()
      else:
        width, height = self.wndmain.get_size()
      try:
        width = int(cp.get("window", "width"))
      except:
        pass
      try:
        height = int(cp.get("window", "height"))
      except:
        pass
      self.wndmain.resize(width, height)
      x, y = self.wndmain.get_position()
      try:
        x = int(cp.get("window", "x"))
      except:
        pass
      try:
        y = int(cp.get("window", "y"))
      except:
        pass
      self.wndmain.move(x, y)
      try:
        self.vmain.set_position(int(cp.get("window", "divider")))
      except:
        pass
    except IOError:
      pass
    return cp


  def message_dialog(self, msg, type, buttons = gtk.BUTTONS_OK):
    dialog = gtk.MessageDialog(self.wndmain,
               gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
               type, buttons, msg)
    result = dialog.run()
    dialog.destroy()
    return result
    

  def message_box(self, msg):
    self.message_dialog(msg, gtk.MESSAGE_INFO)
    
    
  def error_box(self, msg):
    self.message_dialog(msg, gtk.MESSAGE_ERROR)


  def yesno_box(self, msg):
    return self.message_dialog(msg, gtk.MESSAGE_QUESTION, gtk.BUTTONS_YES_NO)


  def file_dialog(self, msg, 
                  action = gtk.FILE_CHOOSER_ACTION_OPEN,
                  button = gtk.STOCK_OPEN):
    fb = gtk.FileChooserDialog(msg, self.wndmain,
                               action,
                               (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                button, gtk.RESPONSE_OK))
    if (self.file_dialog_path is not None):
      fb.set_current_folder_uri(self.file_dialog_path)
    if (self.file_dialog_filename is not None):
      fb.set_filename(self.file_dialog_filename)
    fb.show_all()
    result = fb.run()
    if (result == gtk.RESPONSE_OK):
      filename = fb.get_filename()
    else:
      filename = None
    self.file_dialog_path = fb.get_current_folder_uri()
    self.file_dialog_filename = fb.get_filename()
    fb.destroy()
    return filename


  def register_stock_items(self):
    gtk.stock_add([
        (PGW_CONNECT, _('_Connect Server...'), gtk.gdk.CONTROL_MASK, 
         gtk.gdk.keyval_from_name('N'), None),
        (PGW_DISCONNECT, _('_Disconnect'), 0, -1, None),
        (PGW_SELECTALL, _('Select _All'), gtk.gdk.CONTROL_MASK, 
          gtk.gdk.keyval_from_name('A'), None),
        (PGW_ENCODING, _('Character _Encoding'), 0, -1, None),
        (PGW_ABOUT, _('_About...'), 0, -1, None),
        (PGW_CONNECTNOW, _('_Connect'), 0, -1, None),
                   ])
    self.factory = gtk.IconFactory()
    self.factory.add_default()
    self.factory.add(PGW_CONNECTNOW, gtk.IconSet(gtk.gdk.pixbuf_new_from_file(
                    os.path.join(self.pixmap_path, "connect.png"))))
    self.factory.add(PGW_CONNECT, gtk.IconSet(gtk.gdk.pixbuf_new_from_file(
                    os.path.join(self.pixmap_path, "connect.png"))))
    self.factory.add(PGW_DISCONNECT, gtk.IconSet(gtk.gdk.pixbuf_new_from_file(
                    os.path.join(self.pixmap_path, "disconnect.png"))))
    self.factory.add(PGW_ABOUT, gtk.IconSet(gtk.gdk.pixbuf_new_from_file(
                    os.path.join(self.pixmap_path, "about.png"))))


  def define_ui(self, app):    
    self.ui = '''<ui>
      <menubar name="MenuBar">
        <menu action="File">
          <menuitem action="Connect"/>
          <menuitem action="Disconnect"/>
          <separator/>
          <menuitem action="Open SQL"/>
          <menuitem action="Save SQL"/>
          <menuitem action="Save Results"/>
          <menuitem action="Save All Results"/>
          <separator/>
          <menuitem action="Quit"/>
        </menu>
        <menu action="Edit">
          <menuitem action="Undo"/>
          <menuitem action="Redo"/>
          <separator/>
          <menuitem action="Cut"/>
          <menuitem action="Copy"/>
          <menuitem action="Paste"/>
          <separator/>
          <menuitem action="Select All"/>
        </menu>
        <menu action="Tools">
          <menuitem action="Run SQL"/>
        </menu>
        <menu action="History">
          <menuitem action="Prev Statement"/>
          <menuitem action="Next Statement"/>
        </menu>
        <menu action="Help">
          <menuitem action="About"/>
        </menu>
      </menubar>
      <toolbar name="ToolBar">
        <toolitem action="Connect"/>
        <toolitem action="Run SQL"/>
        <toolitem action="Prev Statement"/>
        <toolitem action="Next Statement"/>
        <separator/>
        <toolitem action="Open SQL"/>
        <toolitem action="Save SQL"/>
        <separator/>
        <toolitem action="Cut"/>
        <toolitem action="Copy"/>
        <toolitem action="Paste"/>
      </toolbar>
      <accelerator action="Run SQL 2"/>
      <accelerator action="Run SQL 3"/>
    </ui>'''

    self.undo = pgw.Undo.Undo()
    self.uimanager = gtk.UIManager()
    self.accelgroup = self.uimanager.get_accel_group()
    self.actiongroup = gtk.ActionGroup('UIManagerMenuBar')
    self.actiongroup.add_actions([
        ('File', None, _('_File')),
          ('Connect', PGW_CONNECT, None, None,
            _('Connect to a database'), app.on_menu_connect),
          ('Disconnect', PGW_DISCONNECT, None, '',
            _('Disconnect from the database'), app.on_menu_disconnect),
          ('Open SQL', gtk.STOCK_OPEN, _('_Open SQL queries...'), None,
            _('Load SQL queries from a file'), app.on_menu_opensql),
          ('Save SQL', gtk.STOCK_SAVE, _('_Save SQL queries...'), None,
            _('Save SQL queries to a file'), app.on_menu_savesql),
          ('Save Results', gtk.STOCK_SAVE, _('Save _Results...'), '<control>R',
            _('Save the results to a file'), app.on_menu_saveresults),
          ('Save All Results', gtk.STOCK_SAVE, _('Save _All Results...'),
            '<control><shift>R',
            _('Save all the results to a file'), app.on_menu_saveallresults),
          ('Quit', gtk.STOCK_QUIT, None, None,
            _('Quit the Program'), app.on_wndmain_destroy),
        ('Edit', None, _('_Edit')),
          ('Undo', gtk.STOCK_UNDO, None, '<control>Z',
            _('Undo the last action'), lambda w: self.undo.undo()),
          ('Redo', gtk.STOCK_REDO, None, '<shift><control>Z',
            _('Redo the last action'), lambda w: self.undo.redo()),
          ('Cut', gtk.STOCK_CUT, None, None,
            _('Cut selection to the clipboard'), app.on_menu_cut),
          ('Copy', gtk.STOCK_COPY, None, None,
            _('Copy selection to the clipboard'), app.on_menu_copy),
          ('Paste', gtk.STOCK_PASTE, None, None,
            _('Paste the content of the clipboard'), app.on_menu_paste),
          ('Select All', PGW_SELECTALL, None, None,
            _('Select all the text'), app.on_menu_selectall),
        ('View', None, _('_View')),
          ('Encoding', PGW_ENCODING, None, None, _('Select the character \
              encoding for display'), app.on_menu_selectall),
        ('Tools', None, _('_Tools')),
          ('Run SQL', gtk.STOCK_EXECUTE, None, "<control>Return",
            _('Execute the content of the SQL buffer'), app.on_menu_runsql),
        ('History', None, _('_History')),
        ('Prev Statement', gtk.STOCK_GO_BACK, _('_Previous statements'),
            "<control>Page_Down",
            _('Go backward in the executed statements history'), 
            app.on_menu_prevsql),
          ('Next Statement', gtk.STOCK_GO_FORWARD, _('_Next statements'),
            "<control>Page_Up",
            _('Go forward in the executed statements history'), 
            app.on_menu_nextsql),
        ('Help', None, _('_Help')),
          ('About', PGW_ABOUT, None, '',
            _('More information about this application'), app.on_menu_about),
        ('Run SQL 2', gtk.STOCK_EXECUTE, None, "F9", None, app.on_menu_runsql),
        ('Run SQL 3', gtk.STOCK_EXECUTE, None, "F5", None, app.on_menu_runsql),
                                 ])
    self.uimanager.insert_action_group(self.actiongroup, 0)
    self.uimanager.add_ui_from_string(self.ui)
    self.uimanager.get_widget('/MenuBar/Help').\
      set_right_justified(True)
    self.enable_disconnect(False)
    self.enable_runsql(False)
    self.enable_prevsql(False)
    self.enable_nextsql(False)
    self.enable_saveresult(False)
    self.create_vpaned(app)
    self.create_statusbar()
    self.uimanager.get_widget('/ToolBar').set_style(gtk.TOOLBAR_ICONS)


  def create_vpaned(self, app):
    self.vmain = gtk.VPaned()
    self.vmain.connect("event", self.on_vpaned_accept)
    scroll = gtk.ScrolledWindow()
    scroll.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    scroll.set_property("height-request", 150)
    scroll.set_shadow_type(gtk.SHADOW_NONE)
    self.sqlbuffer = gtk.TextBuffer()
    self.sqlbuffer.connect('delete-range', self.undo.text_deleted)
    self.sqlbuffer.connect('insert-text', self.undo.text_inserted)
    self.syntax = pgw.Syntax.Syntax(self.sqlbuffer)
    self.buffer_handlers = []
    self.buffer_handlers.append(self.sqlbuffer.connect('delete-range',  
                                self.syntax.text_deleted))
    self.buffer_handlers.append(self.sqlbuffer.connect('insert-text', 
                                self.syntax.text_inserted))
    self.buffer_handlers.append(self.sqlbuffer.connect('changed', 
                                self.syntax.text_changed))
    self.buffer_handlers.append(self.sqlbuffer.connect('changed', 
                                app.on_text_change))
    self.sqlview = gtk.TextView(self.sqlbuffer)
    self.sqlview.connect('focus-in-event', app.on_sqlview_focus_in)
    self.sqlview.connect('focus-out-event', app.on_sqlview_focus_out)
    self.sqlview.connect('key-press-event', app.on_sqlview_keypress)
    scroll.add(self.sqlview)
    self.vmain.add1(scroll)
    self.resulttab = gtk.Notebook()
    self.resulttab.set_tab_pos(gtk.POS_BOTTOM)
    self.resulttab.set_scrollable(True)
    resultview = gtk.Viewport()
    resultview.set_shadow_type(gtk.SHADOW_IN)
    resultview.add(self.resulttab)
    self.vmain.add2(resultview)


  def create_statusbar(self):
    self.statusbar = gtk.Statusbar()
    self.statusbar.set_border_width(0)

    f = self.statusbar.get_children()[0]
    f.set_shadow_type(gtk.SHADOW_IN)
    self.status_result = f.get_children()
    self.status_result = self.status_result[0]
    self.statusbar.set_child_packing(f, True, True, 0, gtk.PACK_START)

    self.status_connect = gtk.Label();
    self.status_connect.set_justify(gtk.JUSTIFY_LEFT)
    f = gtk.Frame()
    f.set_shadow_type(gtk.SHADOW_IN)
    f.add(self.status_connect)
    self.statusbar.pack_start(f, False, False)

    self.status_version = gtk.Label();
    f = gtk.Frame()
    f.set_shadow_type(gtk.SHADOW_IN)
    f.add(self.status_version)
    self.statusbar.pack_start(f, False, False)

    f = gtk.Label('     ')
    self.statusbar.pack_start(f, False, False)


  def status(self, connect, version):
    """Update the status bar text"""
    self.status_connect.set_markup(" " + connect + " ")
    self.status_version.set_markup(" " + version + " ")


  def setfocus_sqlbuffer(self):
    self.sqlview.grab_focus()


  def enable_disconnect(self, state = True):
    self.uimanager.get_widget('/MenuBar/File/Disconnect').\
      set_sensitive(state)


  def enable_saveresult(self, state = True):
    self.uimanager.get_widget('/MenuBar/File/Save All Results').\
      set_sensitive(state)
    self.uimanager.get_widget('/MenuBar/File/Save Results').\
      set_sensitive(state)


  def enable_runsql(self, state = True):    
    self.uimanager.get_widget('/MenuBar/Tools/Run SQL').\
      set_sensitive(state)
    self.uimanager.get_widget('/ToolBar/Run SQL').\
      set_sensitive(state)


  def enable_cut(self, state = True):
    self.uimanager.get_widget('/MenuBar/Edit/Cut').\
      set_sensitive(state)
    self.uimanager.get_widget('/ToolBar/Cut').\
      set_sensitive(state)


  def enable_copy(self, state = True):
    self.uimanager.get_widget('/MenuBar/Edit/Copy').\
      set_sensitive(state)
    self.uimanager.get_widget('/ToolBar/Copy').\
      set_sensitive(state)


  def enable_selectall(self, state = True):
    self.uimanager.get_widget('/MenuBar/Edit/Select All').\
      set_sensitive(state)


  def enable_paste(self, state = True):
    self.uimanager.get_widget('/MenuBar/Edit/Paste').\
      set_sensitive(state)
    self.uimanager.get_widget('/ToolBar/Paste').\
      set_sensitive(state)


  def enable_prevsql(self, state = True):
    self.uimanager.get_widget('/MenuBar/History/Prev Statement').\
      set_sensitive(state)
    self.uimanager.get_widget('/ToolBar/Prev Statement').\
      set_sensitive(state)


  def enable_nextsql(self, state = True):
    self.uimanager.get_widget('/MenuBar/History/Next Statement').\
      set_sensitive(state)
    self.uimanager.get_widget('/ToolBar/Next Statement').\
      set_sensitive(state)


  def get_text(self, buffer):
    """Return the text of a widget"""
    c1 = buffer.get_start_iter()
    c2 = buffer.get_end_iter()
    return buffer.get_slice(c1, c2, True)


  def get_sqlbuffer_text(self):
    return self.get_text(self.sqlbuffer)


  def set_sqlbuffer_text(self, text):
    self.sqlbuffer.set_text(text)

  
  def on_connect_event(self, widget, event):
    if ((not self.entry_password.is_focus()) and
         self.viewconn.is_focus()):
        self.entry_password.grab_focus()

    
  def connect_dialog(self, app, host, port, user, database, overwrite_entry):
    dlg = gtk.Dialog(_('Database connection'),
                     self.wndmain,
                     gtk.DIALOG_MODAL or gtk.DIALOG_DESTROY_WITH_PARENT,
                     ((gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                       PGW_CONNECTNOW, gtk.RESPONSE_OK)))
    dlg.set_resizable(False)

    mbox = gtk.HBox()
    box = gtk.Frame()
    self.storeconn = gtk.ListStore(str, str)
    self.viewconn = gtk.TreeView(self.storeconn)
    cell = gtk.CellRendererText()
    column = gtk.TreeViewColumn(_('5 Previous connections'))
    column.pack_start(cell, True)
    column.add_attribute(cell, 'text', 0)
    column.set_sort_column_id(0)
    self.viewconn.append_column(column)
    self.viewconn.connect('cursor-changed', app.on_dlgconnect_change)
    box.add(self.viewconn)
    mbox.pack_start(box, True, True)

    hbox = gtk.HBox(False)
    hbox.set_border_width(10)
    hbox.set_spacing(10)

    box = gtk.VBox(True)
    box.set_spacing(10)
    lbl = gtk.Label(_('Host'))
    lbl.set_alignment(0, 0.1)
    box.pack_start(lbl, True, True)
    lbl = gtk.Label(_('Port'))
    lbl.set_alignment(0, 0.1)
    box.pack_start(lbl, True, True)
    lbl = gtk.Label(_('User'))
    lbl.set_alignment(0, 0.1)
    box.pack_start(lbl, True, True)
    lbl = gtk.Label(_('Password'))
    lbl.set_alignment(0, 0.1)
    box.pack_start(lbl, True, True)
    lbl = gtk.Label(_('Database'))
    lbl.set_alignment(0, 0.1)
    box.pack_start(lbl, True, True)
    hbox.pack_start(box, False, False)

    box = gtk.VBox(False)
    box.set_spacing(10)
    self.entry_host = gtk.Entry()
    self.entry_host.connect('activate', lambda w: dlg.response(gtk.RESPONSE_OK))
    box.pack_start(self.entry_host, True, True)
    self.entry_port = gtk.Entry()
    self.entry_port.connect('activate', lambda w: dlg.response(gtk.RESPONSE_OK))
    box.pack_start(self.entry_port, True, True)
    self.entry_user = gtk.Entry()
    self.entry_user.connect('activate', lambda w: dlg.response(gtk.RESPONSE_OK))
    box.pack_start(self.entry_user, True, True)
    self.entry_password = gtk.Entry()
    self.entry_password.set_visibility(False)
    self.entry_password.connect('activate', lambda w: dlg.response(gtk.RESPONSE_OK))
    box.pack_start(self.entry_password, True, True)
    self.entry_database = gtk.Entry()
    self.entry_database.connect('activate', lambda w: dlg.response(gtk.RESPONSE_OK))
    box.pack_start(self.entry_database, True, True)
    hbox.pack_start(box, True, True)

    mbox.pack_start(hbox, True, True)

    dlg.vbox.pack_start(mbox, True, True, 0)
    app.on_dlgconnect_map(None)
    if (overwrite_entry or (not self.storeconn.iter_n_children(None))):
      self.entry_host.set_text(host)
      self.entry_port.set_text(str(port))
      self.entry_user.set_text(user)
      self.entry_database.set_text(database)
    dlg.set_default_response(gtk.RESPONSE_OK)
    dlg.connect('event-after', self.on_connect_event)
    dlg.vbox.show_all()
    self.entry_password.grab_focus()
    result = None
    if (dlg.run() == gtk.RESPONSE_OK):
      result = (self.entry_host.get_text(), 
                self.entry_port.get_text(),
                self.entry_user.get_text(),
                self.entry_password.get_text(),
                self.entry_database.get_text())
    dlg.destroy()
    return result


  def write_config(self, cp):
    try:
      cp.write(open(pgw.get_config_path(), 'w'))
    except IOError:
      pass


  def on_vpaned_accept(self, widget, event):
    cp = ConfigParser.ConfigParser()
    try:    
      cp.readfp(open(pgw.get_config_path(), 'r'))
    except IOError:
      pass
    if (not cp.has_section("window")):
      cp.add_section("window")
    cp.set("window", "divider",  str(widget.get_position()))
    self.write_config(cp)


  def on_wndmain_state(self, widget, event):
    cp = ConfigParser.ConfigParser()
    try:    
      cp.readfp(open(pgw.get_config_path(), 'r'))
    except IOError:
      pass
    if (not cp.has_section("window")):
      cp.add_section("window")
    if (event.new_window_state == gtk.gdk.WINDOW_STATE_MAXIMIZED):  
      cp.set("window", "maximized",  "1")
    else:
      cp.set("window", "maximized",  "0")
    self.write_config(cp)


  def on_wndmain_configure(self, widget, event):
    cp = ConfigParser.ConfigParser()
    try:    
      cp.readfp(open(pgw.get_config_path(), 'r'))
    except IOError:
      pass
    if (not cp.has_section("window")):
      cp.add_section("window")
    x, y = widget.get_position()
    cp.set("window", "x",  x)
    cp.set("window", "y",  y)
    width, height = widget.get_size()
    cp.set("window", "width",  width)
    cp.set("window", "height",  height)
    self.write_config(cp)


  def about_dialog(self):
    dlg = gtk.Dialog(_('About PgWorksheet'),
                     self.wndmain,
                     gtk.DIALOG_MODAL or gtk.DIALOG_DESTROY_WITH_PARENT,
                     ((gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE)))
    nbk = gtk.Notebook()
    
    box = gtk.VBox(False)

    lbl = gtk.Label()
    lbl.set_markup('''
<b>PgWorksheet - PostgreSQL Front End</b>
Version ''' + self.app_version + '''
<u>http://pgworksheet.projects.postgresql.org/</u>

Copyright © 2004-2006, Henri Michelon and CML
<u>http://www.e-cml.org/</u>
    ''')
    lbl.set_justify(gtk.JUSTIFY_CENTER)
    lbl.set_padding(10, 10)
    box.pack_start(lbl, True)
    lbl = gtk.Label()
    lbl.set_markup('''
<b>Internationalization:</b>
French : Henri Michelon &lt;hmichelon@e-cml.org&gt;
Japanese : Tadashi Jokagi &lt;elf2000@users.sourceforge.net&gt;
    ''')
    lbl.set_padding(10, 0)
    box.pack_start(lbl, True)
    pix = gtk.Image()
    pix.set_from_file(os.path.join(self.pixmap_path, "pgworksheet.png"))
    pix.set_padding(10, 10)
    box.pack_start(pix, False)
    nbk.append_page(box, gtk.Label(_('About')))

    txt = gtk.TextBuffer()
    txt.set_text('''
		    GNU GENERAL PUBLIC LICENSE
		       Version 2, June 1991

 Copyright (C) 1989, 1991 Free Software Foundation, Inc.
                       59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 Everyone is permitted to copy and distribute verbatim copies
 of this license document, but changing it is not allowed.

			    Preamble

  The licenses for most software are designed to take away your
freedom to share and change it.  By contrast, the GNU General Public
License is intended to guarantee your freedom to share and change free
software--to make sure the software is free for all its users.  This
General Public License applies to most of the Free Software
Foundation's software and to any other program whose authors commit to
using it.  (Some other Free Software Foundation software is covered by
the GNU Library General Public License instead.)  You can apply it to
your programs, too.

  When we speak of free software, we are referring to freedom, not
price.  Our General Public Licenses are designed to make sure that you
have the freedom to distribute copies of free software (and charge for
this service if you wish), that you receive source code or can get it
if you want it, that you can change the software or use pieces of it
in new free programs; and that you know you can do these things.

  To protect your rights, we need to make restrictions that forbid
anyone to deny you these rights or to ask you to surrender the rights.
These restrictions translate to certain responsibilities for you if you
distribute copies of the software, or if you modify it.

  For example, if you distribute copies of such a program, whether
gratis or for a fee, you must give the recipients all the rights that
you have.  You must make sure that they, too, receive or can get the
source code.  And you must show them these terms so they know their
rights.

  We protect your rights with two steps: (1) copyright the software, and
(2) offer you this license which gives you legal permission to copy,
distribute and/or modify the software.

  Also, for each author's protection and ours, we want to make certain
that everyone understands that there is no warranty for this free
software.  If the software is modified by someone else and passed on, we
want its recipients to know that what they have is not the original, so
that any problems introduced by others will not reflect on the original
authors' reputations.

  Finally, any free program is threatened constantly by software
patents.  We wish to avoid the danger that redistributors of a free
program will individually obtain patent licenses, in effect making the
program proprietary.  To prevent this, we have made it clear that any
patent must be licensed for everyone's free use or not licensed at all.

  The precise terms and conditions for copying, distribution and
modification follow.

		    GNU GENERAL PUBLIC LICENSE
   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION

  0. This License applies to any program or other work which contains
a notice placed by the copyright holder saying it may be distributed
under the terms of this General Public License.  The "Program", below,
refers to any such program or work, and a "work based on the Program"
means either the Program or any derivative work under copyright law:
that is to say, a work containing the Program or a portion of it,
either verbatim or with modifications and/or translated into another
language.  (Hereinafter, translation is included without limitation in
the term "modification".)  Each licensee is addressed as "you".

Activities other than copying, distribution and modification are not
covered by this License; they are outside its scope.  The act of
running the Program is not restricted, and the output from the Program
is covered only if its contents constitute a work based on the
Program (independent of having been made by running the Program).
Whether that is true depends on what the Program does.

  1. You may copy and distribute verbatim copies of the Program's
source code as you receive it, in any medium, provided that you
conspicuously and appropriately publish on each copy an appropriate
copyright notice and disclaimer of warranty; keep intact all the
notices that refer to this License and to the absence of any warranty;
and give any other recipients of the Program a copy of this License
along with the Program.

You may charge a fee for the physical act of transferring a copy, and
you may at your option offer warranty protection in exchange for a fee.

  2. You may modify your copy or copies of the Program or any portion
of it, thus forming a work based on the Program, and copy and
distribute such modifications or work under the terms of Section 1
above, provided that you also meet all of these conditions:

    a) You must cause the modified files to carry prominent notices
    stating that you changed the files and the date of any change.

    b) You must cause any work that you distribute or publish, that in
    whole or in part contains or is derived from the Program or any
    part thereof, to be licensed as a whole at no charge to all third
    parties under the terms of this License.

    c) If the modified program normally reads commands interactively
    when run, you must cause it, when started running for such
    interactive use in the most ordinary way, to print or display an
    announcement including an appropriate copyright notice and a
    notice that there is no warranty (or else, saying that you provide
    a warranty) and that users may redistribute the program under
    these conditions, and telling the user how to view a copy of this
    License.  (Exception: if the Program itself is interactive but
    does not normally print such an announcement, your work based on
    the Program is not required to print an announcement.)

These requirements apply to the modified work as a whole.  If
identifiable sections of that work are not derived from the Program,
and can be reasonably considered independent and separate works in
themselves, then this License, and its terms, do not apply to those
sections when you distribute them as separate works.  But when you
distribute the same sections as part of a whole which is a work based
on the Program, the distribution of the whole must be on the terms of
this License, whose permissions for other licensees extend to the
entire whole, and thus to each and every part regardless of who wrote it.

Thus, it is not the intent of this section to claim rights or contest
your rights to work written entirely by you; rather, the intent is to
exercise the right to control the distribution of derivative or
collective works based on the Program.

In addition, mere aggregation of another work not based on the Program
with the Program (or with a work based on the Program) on a volume of
a storage or distribution medium does not bring the other work under
the scope of this License.

  3. You may copy and distribute the Program (or a work based on it,
under Section 2) in object code or executable form under the terms of
Sections 1 and 2 above provided that you also do one of the following:

    a) Accompany it with the complete corresponding machine-readable
    source code, which must be distributed under the terms of Sections
    1 and 2 above on a medium customarily used for software interchange; or,

    b) Accompany it with a written offer, valid for at least three
    years, to give any third party, for a charge no more than your
    cost of physically performing source distribution, a complete
    machine-readable copy of the corresponding source code, to be
    distributed under the terms of Sections 1 and 2 above on a medium
    customarily used for software interchange; or,

    c) Accompany it with the information you received as to the offer
    to distribute corresponding source code.  (This alternative is
    allowed only for noncommercial distribution and only if you
    received the program in object code or executable form with such
    an offer, in accord with Subsection b above.)

The source code for a work means the preferred form of the work for
making modifications to it.  For an executable work, complete source
code means all the source code for all modules it contains, plus any
associated interface definition files, plus the scripts used to
control compilation and installation of the executable.  However, as a
special exception, the source code distributed need not include
anything that is normally distributed (in either source or binary
form) with the major components (compiler, kernel, and so on) of the
operating system on which the executable runs, unless that component
itself accompanies the executable.

If distribution of executable or object code is made by offering
access to copy from a designated place, then offering equivalent
access to copy the source code from the same place counts as
distribution of the source code, even though third parties are not
compelled to copy the source along with the object code.

  4. You may not copy, modify, sublicense, or distribute the Program
except as expressly provided under this License.  Any attempt
otherwise to copy, modify, sublicense or distribute the Program is
void, and will automatically terminate your rights under this License.
However, parties who have received copies, or rights, from you under
this License will not have their licenses terminated so long as such
parties remain in full compliance.

  5. You are not required to accept this License, since you have not
signed it.  However, nothing else grants you permission to modify or
distribute the Program or its derivative works.  These actions are
prohibited by law if you do not accept this License.  Therefore, by
modifying or distributing the Program (or any work based on the
Program), you indicate your acceptance of this License to do so, and
all its terms and conditions for copying, distributing or modifying
the Program or works based on it.

  6. Each time you redistribute the Program (or any work based on the
Program), the recipient automatically receives a license from the
original licensor to copy, distribute or modify the Program subject to
these terms and conditions.  You may not impose any further
restrictions on the recipients' exercise of the rights granted herein.
You are not responsible for enforcing compliance by third parties to
this License.

  7. If, as a consequence of a court judgment or allegation of patent
infringement or for any other reason (not limited to patent issues),
conditions are imposed on you (whether by court order, agreement or
otherwise) that contradict the conditions of this License, they do not
excuse you from the conditions of this License.  If you cannot
distribute so as to satisfy simultaneously your obligations under this
License and any other pertinent obligations, then as a consequence you
may not distribute the Program at all.  For example, if a patent
license would not permit royalty-free redistribution of the Program by
all those who receive copies directly or indirectly through you, then
the only way you could satisfy both it and this License would be to
refrain entirely from distribution of the Program.

If any portion of this section is held invalid or unenforceable under
any particular circumstance, the balance of the section is intended to
apply and the section as a whole is intended to apply in other
circumstances.

It is not the purpose of this section to induce you to infringe any
patents or other property right claims or to contest validity of any
such claims; this section has the sole purpose of protecting the
integrity of the free software distribution system, which is
implemented by public license practices.  Many people have made
generous contributions to the wide range of software distributed
through that system in reliance on consistent application of that
system; it is up to the author/donor to decide if he or she is willing
to distribute software through any other system and a licensee cannot
impose that choice.

This section is intended to make thoroughly clear what is believed to
be a consequence of the rest of this License.

  8. If the distribution and/or use of the Program is restricted in
certain countries either by patents or by copyrighted interfaces, the
original copyright holder who places the Program under this License
may add an explicit geographical distribution limitation excluding
those countries, so that distribution is permitted only in or among
countries not thus excluded.  In such case, this License incorporates
the limitation as if written in the body of this License.

  9. The Free Software Foundation may publish revised and/or new versions
of the General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

Each version is given a distinguishing version number.  If the Program
specifies a version number of this License which applies to it and "any
later version", you have the option of following the terms and conditions
either of that version or of any later version published by the Free
Software Foundation.  If the Program does not specify a version number of
this License, you may choose any version ever published by the Free Software
Foundation.

  10. If you wish to incorporate parts of the Program into other free
programs whose distribution conditions are different, write to the author
to ask for permission.  For software which is copyrighted by the Free
Software Foundation, write to the Free Software Foundation; we sometimes
make exceptions for this.  Our decision will be guided by the two goals
of preserving the free status of all derivatives of our free software and
of promoting the sharing and reuse of software generally.

			    NO WARRANTY

  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGES.

		     END OF TERMS AND CONDITIONS

	    How to Apply These Terms to Your New Programs

  If you develop a new program, and you want it to be of the greatest
possible use to the public, the best way to achieve this is to make it
free software which everyone can redistribute and change under these terms.

  To do so, attach the following notices to the program.  It is safest
to attach them to the start of each source file to most effectively
convey the exclusion of warranty; and each file should have at least
the "copyright" line and a pointer to where the full notice is found.

    <one line to give the program's name and a brief idea of what it does.>
    Copyright (C) <year>  <name of author>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


Also add information on how to contact you by electronic and paper mail.

If the program is interactive, make it output a short notice like this
when it starts in an interactive mode:

    Gnomovision version 69, Copyright (C) year name of author
    Gnomovision comes with ABSOLUTELY NO WARRANTY; for details type `show w'.
    This is free software, and you are welcome to redistribute it
    under certain conditions; type `show c' for details.

The hypothetical commands `show w' and `show c' should show the appropriate
parts of the General Public License.  Of course, the commands you use may
be called something other than `show w' and `show c'; they could even be
mouse-clicks or menu items--whatever suits your program.

You should also get your employer (if you work as a programmer) or your
school, if any, to sign a "copyright disclaimer" for the program, if
necessary.  Here is a sample; alter the names:

  Yoyodyne, Inc., hereby disclaims all copyright interest in the program
  `Gnomovision' (which makes passes at compilers) written by James Hacker.

  <signature of Ty Coon>, 1 April 1989
  Ty Coon, President of Vice

This General Public License does not permit incorporating your program into
proprietary programs.  If your program is a subroutine library, you may
consider it more useful to permit linking proprietary applications with the
library.  If this is what you want to do, use the GNU Library General
Public License instead of this License.
    ''')
    scroll = gtk.ScrolledWindow()
    scroll.add(gtk.TextView(txt))
    nbk.append_page(scroll, gtk.Label(_('License')))
    
    dlg.vbox.pack_start(nbk, True, True, 0)
    nbk.show_all()
    dlg.run()
    dlg.destroy()
