/*
  This file is part of TALER
  (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-merchant-httpd_private-patch-units-ID.c
 * @brief implement PATCH /private/units/$UNIT
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include "taler-merchant-httpd_private-patch-units-ID.h"
#include "taler-merchant-httpd_helper.h"
#include <taler/taler_json_lib.h>

#define TMH_MAX_UNIT_PRECISION_LEVEL 6


MHD_RESULT
TMH_private_patch_units_ID (const struct TMH_RequestHandler *rh,
                            struct MHD_Connection *connection,
                            struct TMH_HandlerContext *hc)
{
  struct TMH_MerchantInstance *mi = hc->instance;
  const char *unit_id = hc->infix;
  struct TALER_MERCHANTDB_UnitDetails nud = { 0 };
  bool unit_allow_fraction_missing = true;
  bool unit_precision_missing = true;
  bool unit_active_missing = true;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("unit_name_long",
                               (const char **) &nud.unit_name_long),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("unit_name_long_i18n",
                             &nud.unit_name_long_i18n),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("unit_name_short",
                               (const char **) &nud.unit_name_short),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("unit_name_short_i18n",
                             &nud.unit_name_short_i18n),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_bool ("unit_allow_fraction",
                             &nud.unit_allow_fraction),
      &unit_allow_fraction_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint32 ("unit_precision_level",
                               &nud.unit_precision_level),
      &unit_precision_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_bool ("unit_active",
                             &nud.unit_active),
      &unit_active_missing),
    GNUNET_JSON_spec_end ()
  };
  enum GNUNET_GenericReturnValue res;
  const bool *unit_allow_fraction_ptr = NULL;
  const uint32_t *unit_precision_ptr = NULL;
  const bool *unit_active_ptr = NULL;
  enum GNUNET_DB_QueryStatus qs;
  bool no_instance = false;
  bool no_unit = false;
  bool builtin_conflict = false;
  MHD_RESULT ret = MHD_YES;

  (void) rh;
  GNUNET_assert (NULL != mi);
  GNUNET_assert (NULL != unit_id);

  res = TALER_MHD_parse_json_data (connection,
                                   hc->request_body,
                                   spec);
  if (GNUNET_OK != res)
    return (GNUNET_NO == res) ? MHD_YES : MHD_NO;

  if (NULL == nud.unit_name_long &&
      NULL == nud.unit_name_long_i18n &&
      NULL == nud.unit_name_short &&
      NULL == nud.unit_name_short_i18n &&
      unit_allow_fraction_missing &&
      unit_precision_missing &&
      unit_active_missing)
  {
    ret = TALER_MHD_reply_static (connection,
                                  MHD_HTTP_NO_CONTENT,
                                  NULL,
                                  NULL,
                                  0);
    goto cleanup;
  }

  if (! unit_precision_missing)
  {
    if (nud.unit_precision_level > TMH_MAX_UNIT_PRECISION_LEVEL)
    {
      GNUNET_break_op (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_BAD_REQUEST,
                                        TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                        "unit_precision_level");
      goto cleanup;
    }
    unit_precision_ptr = &nud.unit_precision_level;
  }

  if (! unit_allow_fraction_missing)
  {
    unit_allow_fraction_ptr = &nud.unit_allow_fraction;
    if (! nud.unit_allow_fraction)
    {
      nud.unit_precision_level = 0;
      unit_precision_missing = false;
      unit_precision_ptr = &nud.unit_precision_level;
    }
  }

  if (! unit_active_missing)
    unit_active_ptr = &nud.unit_active;

  if (NULL != nud.unit_name_long_i18n)
  {
    if (! TALER_JSON_check_i18n (nud.unit_name_long_i18n))
    {
      GNUNET_break_op (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_BAD_REQUEST,
                                        TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                        "unit_name_long_i18n");
      goto cleanup;
    }
  }

  if (NULL != nud.unit_name_short_i18n)
  {
    if (! TALER_JSON_check_i18n (nud.unit_name_short_i18n))
    {
      GNUNET_break_op (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_BAD_REQUEST,
                                        TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                        "unit_name_short_i18n");
      goto cleanup;
    }
  }

  qs = TMH_db->update_unit (TMH_db->cls,
                            mi->settings.id,
                            unit_id,
                            nud.unit_name_long,
                            nud.unit_name_long_i18n,
                            nud.unit_name_short,
                            nud.unit_name_short_i18n,
                            unit_allow_fraction_ptr,
                            unit_precision_ptr,
                            unit_active_ptr,
                            &no_instance,
                            &no_unit,
                            &builtin_conflict);
  switch (qs)
  {
  case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
    break;
  case GNUNET_DB_STATUS_SOFT_ERROR:
    GNUNET_break (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_INTERNAL_SERVER_ERROR,
                                      TALER_EC_GENERIC_DB_SOFT_FAILURE,
                                      "update_unit");
    goto cleanup;
  case GNUNET_DB_STATUS_HARD_ERROR:
  default:
    GNUNET_break (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_INTERNAL_SERVER_ERROR,
                                      TALER_EC_GENERIC_DB_STORE_FAILED,
                                      "update_unit");
    goto cleanup;
  }

  if (no_instance)
  {
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_NOT_FOUND,
                                      TALER_EC_MERCHANT_GENERIC_INSTANCE_UNKNOWN,
                                      mi->settings.id);
    goto cleanup;
  }
  if (no_unit)
  {
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_NOT_FOUND,
                                      TALER_EC_MERCHANT_GENERIC_UNIT_UNKNOWN,
                                      unit_id);
    goto cleanup;
  }
  if (builtin_conflict)
  {
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_CONFLICT,
                                      TALER_EC_MERCHANT_GENERIC_UNIT_BUILTIN,
                                      unit_id);
    goto cleanup;
  }

  ret = TALER_MHD_reply_static (connection,
                                MHD_HTTP_NO_CONTENT,
                                NULL,
                                NULL,
                                0);

cleanup:
  if (NULL != nud.unit_name_long_i18n)
  {
    json_decref (nud.unit_name_long_i18n);
    nud.unit_name_long_i18n = NULL;
  }
  if (NULL != nud.unit_name_short_i18n)
  {
    json_decref (nud.unit_name_short_i18n);
    nud.unit_name_short_i18n = NULL;
  }
  GNUNET_JSON_parse_free (spec);
  return ret;
}


/* end of taler-merchant-httpd_private-patch-units-ID.c */
