// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { encodeNamedCurves } from './ecc_encode';
import { decodeNamedCurves } from './ecc_decode';
import { frozenClass, readOnlyBinaryProperty, readOnlyProperty, } from './immutable_class';
import { publicKeyPem, privateKeyPem } from './pem_helpers';
/*
 * This public interface to the SignatureKey object is provided for
 * developers of CMMs and keyrings only. If you are a user of the AWS Encryption
 * SDK and you are not developing your own CMMs and/or keyrings, you do not
 * need to use it and you should not do so.
 */
export class SignatureKey {
    constructor(privateKey, compressPoint, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not create a SignatureKey for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        /* This is unfortunately complicated.  Node v11 crypto will accept
         * a PEM formated Buffer to sign.  But the ECDH class will still
         * return Buffers that are not PEM formated, but _only_ the points
         * on the curve.  This means I have to make a choice about
         * formating.  I chose to assume that t Buffer/Uin8Array is
         * _only_ the raw points.
         */
        if (privateKey instanceof Uint8Array) {
            const pem = privateKeyPem(namedCurve, fromBuffer(privateKey), fromBuffer(compressPoint));
            readOnlyProperty(this, 'privateKey', pem);
        }
        else {
            readOnlyProperty(this, 'privateKey', privateKey);
        }
        readOnlyBinaryProperty(this, 'compressPoint', compressPoint);
        readOnlyProperty(this, 'signatureCurve', namedCurve);
        Object.setPrototypeOf(this, SignatureKey.prototype);
        Object.freeze(this);
    }
    static encodeCompressPoint(publicKeyBytes, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not return a compress point for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        return encodeNamedCurves[namedCurve](publicKeyBytes);
    }
}
frozenClass(SignatureKey);
export class VerificationKey {
    publicKey;
    signatureCurve;
    constructor(publicKey, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not create a VerificationKey for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        /* This is unfortunately complicated.  Node v11 crypto will accept
         * a PEM formated Buffer to verify.  But the ECDH class will still
         * return Buffers that are not PEM formated, but _only_ the points
         * on the curve.  This means I have to make a choice about
         * formating.  I chose to assume that the Buffer/Uin8Array is
         * _only_ the raw points.
         */
        if (publicKey instanceof Uint8Array) {
            const pem = publicKeyPem(namedCurve, fromBuffer(publicKey));
            readOnlyProperty(this, 'publicKey', pem);
        }
        else {
            readOnlyProperty(this, 'publicKey', publicKey);
        }
        readOnlyProperty(this, 'signatureCurve', namedCurve);
        Object.setPrototypeOf(this, VerificationKey.prototype);
        Object.freeze(this);
    }
    static decodeCompressPoint(compressPoint, suite) {
        const { signatureCurve: namedCurve } = suite;
        /* Precondition: Do not decode a public key for an algorithm suite that does not have an EC named curve. */
        if (!namedCurve)
            throw new Error('Unsupported Algorithm');
        return decodeNamedCurves[namedCurve](compressPoint);
    }
}
frozenClass(VerificationKey);
function fromBuffer(uint) {
    const { buffer, byteOffset, byteLength } = uint;
    return Buffer.from(buffer, byteOffset, byteLength);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2lnbmF0dXJlX2tleS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9zaWduYXR1cmVfa2V5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7QUFPdEMsT0FBTyxFQUFFLGlCQUFpQixFQUFFLE1BQU0sY0FBYyxDQUFBO0FBQ2hELE9BQU8sRUFBRSxpQkFBaUIsRUFBRSxNQUFNLGNBQWMsQ0FBQTtBQUNoRCxPQUFPLEVBQ0wsV0FBVyxFQUNYLHNCQUFzQixFQUN0QixnQkFBZ0IsR0FDakIsTUFBTSxtQkFBbUIsQ0FBQTtBQUMxQixPQUFPLEVBQUUsWUFBWSxFQUFFLGFBQWEsRUFBRSxNQUFNLGVBQWUsQ0FBQTtBQUczRDs7Ozs7R0FLRztBQUVILE1BQU0sT0FBTyxZQUFZO0lBSXZCLFlBQ0UsVUFBMkMsRUFDM0MsYUFBeUIsRUFDekIsS0FBcUI7UUFFckIsTUFBTSxFQUFFLGNBQWMsRUFBRSxVQUFVLEVBQUUsR0FBRyxLQUFLLENBQUE7UUFDNUMsNkdBQTZHO1FBQzdHLElBQUksQ0FBQyxVQUFVO1lBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFBO1FBQ3pEOzs7Ozs7V0FNRztRQUNILElBQUksVUFBVSxZQUFZLFVBQVUsRUFBRTtZQUNwQyxNQUFNLEdBQUcsR0FBRyxhQUFhLENBQ3ZCLFVBQVUsRUFDVixVQUFVLENBQUMsVUFBVSxDQUFDLEVBQ3RCLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FDMUIsQ0FBQTtZQUNELGdCQUFnQixDQUE2QixJQUFJLEVBQUUsWUFBWSxFQUFFLEdBQUcsQ0FBQyxDQUFBO1NBQ3RFO2FBQU07WUFDTCxnQkFBZ0IsQ0FDZCxJQUFJLEVBQ0osWUFBWSxFQUNaLFVBQVUsQ0FDWCxDQUFBO1NBQ0Y7UUFDRCxzQkFBc0IsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFLGFBQWEsQ0FBQyxDQUFBO1FBQzVELGdCQUFnQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsQ0FBQTtRQUNwRCxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUE7UUFDbkQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQTtJQUNyQixDQUFDO0lBRUQsTUFBTSxDQUFDLG1CQUFtQixDQUN4QixjQUEwQixFQUMxQixLQUFxQjtRQUVyQixNQUFNLEVBQUUsY0FBYyxFQUFFLFVBQVUsRUFBRSxHQUFHLEtBQUssQ0FBQTtRQUM1QywrR0FBK0c7UUFDL0csSUFBSSxDQUFDLFVBQVU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUE7UUFDekQsT0FBTyxpQkFBaUIsQ0FBQyxVQUFVLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQTtJQUN0RCxDQUFDO0NBQ0Y7QUFDRCxXQUFXLENBQUMsWUFBWSxDQUFDLENBQUE7QUFFekIsTUFBTSxPQUFPLGVBQWU7SUFDVixTQUFTLENBQThCO0lBQ3ZDLGNBQWMsQ0FBcUM7SUFDbkUsWUFDRSxTQUEwQyxFQUMxQyxLQUFxQjtRQUVyQixNQUFNLEVBQUUsY0FBYyxFQUFFLFVBQVUsRUFBRSxHQUFHLEtBQUssQ0FBQTtRQUM1QyxnSEFBZ0g7UUFDaEgsSUFBSSxDQUFDLFVBQVU7WUFBRSxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUE7UUFDekQ7Ozs7OztXQU1HO1FBQ0gsSUFBSSxTQUFTLFlBQVksVUFBVSxFQUFFO1lBQ25DLE1BQU0sR0FBRyxHQUFHLFlBQVksQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUE7WUFDM0QsZ0JBQWdCLENBQStCLElBQUksRUFBRSxXQUFXLEVBQUUsR0FBRyxDQUFDLENBQUE7U0FDdkU7YUFBTTtZQUNMLGdCQUFnQixDQUNkLElBQUksRUFDSixXQUFXLEVBQ1gsU0FBUyxDQUNWLENBQUE7U0FDRjtRQUNELGdCQUFnQixDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsQ0FBQTtRQUNwRCxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxlQUFlLENBQUMsU0FBUyxDQUFDLENBQUE7UUFDdEQsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQTtJQUNyQixDQUFDO0lBRUQsTUFBTSxDQUFDLG1CQUFtQixDQUFDLGFBQXlCLEVBQUUsS0FBcUI7UUFDekUsTUFBTSxFQUFFLGNBQWMsRUFBRSxVQUFVLEVBQUUsR0FBRyxLQUFLLENBQUE7UUFDNUMsMkdBQTJHO1FBQzNHLElBQUksQ0FBQyxVQUFVO1lBQUUsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFBO1FBRXpELE9BQU8saUJBQWlCLENBQUMsVUFBVSxDQUFDLENBQUMsYUFBYSxDQUFDLENBQUE7SUFDckQsQ0FBQztDQUNGO0FBQ0QsV0FBVyxDQUFDLGVBQWUsQ0FBQyxDQUFBO0FBRTVCLFNBQVMsVUFBVSxDQUFDLElBQWdCO0lBQ2xDLE1BQU0sRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRSxHQUFHLElBQUksQ0FBQTtJQUMvQyxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQTtBQUNwRCxDQUFDIn0=