﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>Request to describe application versions.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/DescribeApplicationVersionsMessage">AWS
 * API Reference</a></p>
 */
class DescribeApplicationVersionsRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API DescribeApplicationVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeApplicationVersions"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Specify an application name to show only application versions for that
   * application.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  DescribeApplicationVersionsRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify a version label to show a specific application version.</p>
   */
  inline const Aws::Vector<Aws::String>& GetVersionLabels() const { return m_versionLabels; }
  inline bool VersionLabelsHasBeenSet() const { return m_versionLabelsHasBeenSet; }
  template <typename VersionLabelsT = Aws::Vector<Aws::String>>
  void SetVersionLabels(VersionLabelsT&& value) {
    m_versionLabelsHasBeenSet = true;
    m_versionLabels = std::forward<VersionLabelsT>(value);
  }
  template <typename VersionLabelsT = Aws::Vector<Aws::String>>
  DescribeApplicationVersionsRequest& WithVersionLabels(VersionLabelsT&& value) {
    SetVersionLabels(std::forward<VersionLabelsT>(value));
    return *this;
  }
  template <typename VersionLabelsT = Aws::String>
  DescribeApplicationVersionsRequest& AddVersionLabels(VersionLabelsT&& value) {
    m_versionLabelsHasBeenSet = true;
    m_versionLabels.emplace_back(std::forward<VersionLabelsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For a paginated request. Specify a maximum number of application versions to
   * include in each response.</p> <p>If no <code>MaxRecords</code> is specified, all
   * available application versions are retrieved in a single response.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeApplicationVersionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For a paginated request. Specify a token from a previous response page to
   * retrieve the next response page. All other parameter values must be identical to
   * the ones specified in the initial request.</p> <p>If no <code>NextToken</code>
   * is specified, the first page is retrieved.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeApplicationVersionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  Aws::Vector<Aws::String> m_versionLabels;

  int m_maxRecords{0};

  Aws::String m_nextToken;
  bool m_applicationNameHasBeenSet = false;
  bool m_versionLabelsHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
