﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/redshift-2012-12-01/DescribeClusterVersionsMessage">AWS
 * API Reference</a></p>
 */
class DescribeClusterVersionsRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API DescribeClusterVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeClusterVersions"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The specific cluster version to return.</p> <p>Example: <code>1.0</code> </p>
   */
  inline const Aws::String& GetClusterVersion() const { return m_clusterVersion; }
  inline bool ClusterVersionHasBeenSet() const { return m_clusterVersionHasBeenSet; }
  template <typename ClusterVersionT = Aws::String>
  void SetClusterVersion(ClusterVersionT&& value) {
    m_clusterVersionHasBeenSet = true;
    m_clusterVersion = std::forward<ClusterVersionT>(value);
  }
  template <typename ClusterVersionT = Aws::String>
  DescribeClusterVersionsRequest& WithClusterVersion(ClusterVersionT&& value) {
    SetClusterVersion(std::forward<ClusterVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of a specific cluster parameter group family to return details
   * for.</p> <p>Constraints:</p> <ul> <li> <p>Must be 1 to 255 alphanumeric
   * characters</p> </li> <li> <p>First character must be a letter</p> </li> <li>
   * <p>Cannot end with a hyphen or contain two consecutive hyphens</p> </li> </ul>
   */
  inline const Aws::String& GetClusterParameterGroupFamily() const { return m_clusterParameterGroupFamily; }
  inline bool ClusterParameterGroupFamilyHasBeenSet() const { return m_clusterParameterGroupFamilyHasBeenSet; }
  template <typename ClusterParameterGroupFamilyT = Aws::String>
  void SetClusterParameterGroupFamily(ClusterParameterGroupFamilyT&& value) {
    m_clusterParameterGroupFamilyHasBeenSet = true;
    m_clusterParameterGroupFamily = std::forward<ClusterParameterGroupFamilyT>(value);
  }
  template <typename ClusterParameterGroupFamilyT = Aws::String>
  DescribeClusterVersionsRequest& WithClusterParameterGroupFamily(ClusterParameterGroupFamilyT&& value) {
    SetClusterParameterGroupFamily(std::forward<ClusterParameterGroupFamilyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of response records to return in each call. If the number
   * of remaining response records exceeds the specified <code>MaxRecords</code>
   * value, a value is returned in a <code>marker</code> field of the response. You
   * can retrieve the next set of records by retrying the command with the returned
   * marker value. </p> <p>Default: <code>100</code> </p> <p>Constraints: minimum 20,
   * maximum 100.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeClusterVersionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional parameter that specifies the starting point to return a set of
   * response records. When the results of a <a>DescribeClusterVersions</a> request
   * exceed the value specified in <code>MaxRecords</code>, Amazon Web Services
   * returns a value in the <code>Marker</code> field of the response. You can
   * retrieve the next set of response records by providing the returned marker value
   * in the <code>Marker</code> parameter and retrying the request. </p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeClusterVersionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterVersion;

  Aws::String m_clusterParameterGroupFamily;

  int m_maxRecords{0};

  Aws::String m_marker;
  bool m_clusterVersionHasBeenSet = false;
  bool m_clusterParameterGroupFamilyHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
