﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/TagSpecification.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class CreateTrafficMirrorTargetRequest : public EC2Request {
 public:
  AWS_EC2_API CreateTrafficMirrorTargetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTrafficMirrorTarget"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The network interface ID that is associated with the target.</p>
   */
  inline const Aws::String& GetNetworkInterfaceId() const { return m_networkInterfaceId; }
  inline bool NetworkInterfaceIdHasBeenSet() const { return m_networkInterfaceIdHasBeenSet; }
  template <typename NetworkInterfaceIdT = Aws::String>
  void SetNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    m_networkInterfaceIdHasBeenSet = true;
    m_networkInterfaceId = std::forward<NetworkInterfaceIdT>(value);
  }
  template <typename NetworkInterfaceIdT = Aws::String>
  CreateTrafficMirrorTargetRequest& WithNetworkInterfaceId(NetworkInterfaceIdT&& value) {
    SetNetworkInterfaceId(std::forward<NetworkInterfaceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Network Load Balancer that is
   * associated with the target.</p>
   */
  inline const Aws::String& GetNetworkLoadBalancerArn() const { return m_networkLoadBalancerArn; }
  inline bool NetworkLoadBalancerArnHasBeenSet() const { return m_networkLoadBalancerArnHasBeenSet; }
  template <typename NetworkLoadBalancerArnT = Aws::String>
  void SetNetworkLoadBalancerArn(NetworkLoadBalancerArnT&& value) {
    m_networkLoadBalancerArnHasBeenSet = true;
    m_networkLoadBalancerArn = std::forward<NetworkLoadBalancerArnT>(value);
  }
  template <typename NetworkLoadBalancerArnT = Aws::String>
  CreateTrafficMirrorTargetRequest& WithNetworkLoadBalancerArn(NetworkLoadBalancerArnT&& value) {
    SetNetworkLoadBalancerArn(std::forward<NetworkLoadBalancerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the Traffic Mirror target.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateTrafficMirrorTargetRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to assign to the Traffic Mirror target.</p>
   */
  inline const Aws::Vector<TagSpecification>& GetTagSpecifications() const { return m_tagSpecifications; }
  inline bool TagSpecificationsHasBeenSet() const { return m_tagSpecificationsHasBeenSet; }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  void SetTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications = std::forward<TagSpecificationsT>(value);
  }
  template <typename TagSpecificationsT = Aws::Vector<TagSpecification>>
  CreateTrafficMirrorTargetRequest& WithTagSpecifications(TagSpecificationsT&& value) {
    SetTagSpecifications(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  template <typename TagSpecificationsT = TagSpecification>
  CreateTrafficMirrorTargetRequest& AddTagSpecifications(TagSpecificationsT&& value) {
    m_tagSpecificationsHasBeenSet = true;
    m_tagSpecifications.emplace_back(std::forward<TagSpecificationsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline CreateTrafficMirrorTargetRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique, case-sensitive identifier that you provide to ensure the idempotency
   * of the request. For more information, see <a
   * href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">How
   * to ensure idempotency</a>.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateTrafficMirrorTargetRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the Gateway Load Balancer endpoint.</p>
   */
  inline const Aws::String& GetGatewayLoadBalancerEndpointId() const { return m_gatewayLoadBalancerEndpointId; }
  inline bool GatewayLoadBalancerEndpointIdHasBeenSet() const { return m_gatewayLoadBalancerEndpointIdHasBeenSet; }
  template <typename GatewayLoadBalancerEndpointIdT = Aws::String>
  void SetGatewayLoadBalancerEndpointId(GatewayLoadBalancerEndpointIdT&& value) {
    m_gatewayLoadBalancerEndpointIdHasBeenSet = true;
    m_gatewayLoadBalancerEndpointId = std::forward<GatewayLoadBalancerEndpointIdT>(value);
  }
  template <typename GatewayLoadBalancerEndpointIdT = Aws::String>
  CreateTrafficMirrorTargetRequest& WithGatewayLoadBalancerEndpointId(GatewayLoadBalancerEndpointIdT&& value) {
    SetGatewayLoadBalancerEndpointId(std::forward<GatewayLoadBalancerEndpointIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_networkInterfaceId;

  Aws::String m_networkLoadBalancerArn;

  Aws::String m_description;

  Aws::Vector<TagSpecification> m_tagSpecifications;

  bool m_dryRun{false};

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_gatewayLoadBalancerEndpointId;
  bool m_networkInterfaceIdHasBeenSet = false;
  bool m_networkLoadBalancerArnHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_tagSpecificationsHasBeenSet = false;
  bool m_dryRunHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_gatewayLoadBalancerEndpointIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
