/* -*- mode: C; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
/* vim:set et sts=4: */
/* bus - The Input Bus
 * Copyright (C) 2025 Takao Fujiwara <takao.fujiwara1@gmail.com>
 * Copyright (C) 2025 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#if !defined (__IBUS_H_INSIDE__) && !defined (IBUS_COMPILATION)
#error "Only <ibus.h> can be included directly"
#endif

#ifndef __IBUS_MESSAGE_H_
#define __IBUS_MESSAGE_H_

/**
 * SECTION: ibusmessage
 * @short_description: Message data which is forwarded to IBus Panel.
 * @title: IBusMessage
 * @stability: Unstable
 *
 * An IBusMessage stores the message type, description, timeout. 
 * The message data can generated by ibus_message_new(),
 *
 * see_also: #IBusEngine, #IBuPanels
 *
 */

#include "ibusserializable.h"

/*
 * Type macros.
 */

/* define GOBJECT macros */
#define IBUS_TYPE_MESSAGE                 \
    (ibus_message_get_type ())
#define IBUS_MESSAGE(obj)                 \
    (G_TYPE_CHECK_INSTANCE_CAST ((obj), IBUS_TYPE_MESSAGE, IBusMessage))
#define IBUS_MESSAGE_CLASS(klass)         \
    (G_TYPE_CHECK_CLASS_CAST ((klass), IBUS_TYPE_MESSAGE, IBusMessageClass))
#define IBUS_IS_MESSAGE(obj)              \
    (G_TYPE_CHECK_INSTANCE_TYPE ((obj), IBUS_TYPE_MESSAGE))
#define IBUS_IS_MESSAGE_CLASS(klass)      \
    (G_TYPE_CHECK_CLASS_TYPE ((klass), IBUS_TYPE_MESSAGE))
#define IBUS_MESSAGE_GET_CLASS(obj)       \
    (G_TYPE_INSTANCE_GET_CLASS ((obj), IBUS_TYPE_MESSAGE, IBusMessageClass))

G_BEGIN_DECLS

typedef struct _IBusMessage IBusMessage;
typedef struct _IBusMessagePrivate IBusMessagePrivate;
typedef struct _IBusMessageClass IBusMessageClass;

/**
 * IBusMessage:
 *
 * Message data which is forwarded to IBus Panel.
 * You can get values with g_object_get_properties.
 *
 * Since: 1.5.33
 */
struct _IBusMessage {
    IBusSerializable parent;
    /* instance members */

    /*< public >*/
    /*< private >*/
    IBusMessagePrivate *priv;
};

struct _IBusMessageClass {
    IBusSerializableClass parent;
    /* class members */

    /*< private >*/
    /* padding */
    gpointer pdummy[5];  // We can add 5 pointers without breaking the ABI.
};

GType            ibus_message_get_type          (void);

/**
 * ibus_message_new:
 * @domain: the message domain with @IBusMessageDomain likes #GError.
 * @code: the message code with @IBusPanelServiceMsgCode likes #GError.
 * @title: the message title.
 * @description: the message description.
 * @...: the NULL-terminated arguments of the properties and values.
 *
 * Creates a new #IBusMessage.
 * You can give additional parameters to the API. E.g.
 * ibus_message_new(IBUS_MESSAGE_DOMAIN_PANEL,
 * IBUS_PANEL_SERVICE_MSG_CODE_LOADING_UNICODE, "IBus Emoji utility",
 * "Loading a Unicode dictionary", "timeout", 10, "progress", 30, NULL)
 *
 * Returns: A newly allocated IBusMessage.
 *
 * Since: 1.5.33
 */
IBusMessage     *ibus_message_new               (guint           domain,
                                                 guint           code,
                                                 const gchar    *title,
                                                 const gchar    *description,
                                                 ...) G_GNUC_NULL_TERMINATED;

/**
 * ibus_message_get_domain:
 * @msg: An #IBusMessage.
 *
 * Gets the domain property in #IBusMessage.
 *
 * Returns: domain property in #IBusMessage
 *
 * Since: 1.5.33
 */
guint            ibus_message_get_domain        (IBusMessage  *msg);

/**
 * ibus_message_get_code:
 * @msg: An #IBusMessage.
 *
 * Gets the code property in #IBusMessage.
 *
 * Returns: code property in #IBusMessage
 *
 * Since: 1.5.33
 */
guint            ibus_message_get_code          (IBusMessage  *msg);

/**
 * ibus_message_get_title:
 * @msg: An #IBuMessages.
 *
 * Gets the title property in #IBusMessage. It should not be freed.
 *
 * Returns: title property in #IBuaMessage
 *
 * Since: 1.5.33
 */
const gchar     *ibus_message_get_title         (IBusMessage  *msg);

/**
 * ibus_message_get_description:
 * @msg: An #IBuMessages.
 *
 * Gets the description property in #IBusMessage. It should not be freed.
 *
 * Returns: description property in #IBuaMessage
 *
 * Since: 1.5.33
 */
const gchar     *ibus_message_get_description   (IBusMessage  *msg);

/**
 * ibus_message_get_timeout:
 * @msg: An #IBusMessage.
 *
 * Gets the timeout property in #IBusMessage.
 *
 * Returns: timeout property in #IBusMessage
 *
 * Since: 1.5.33
 */
int              ibus_message_get_timeout       (IBusMessage  *msg);

/**
 * ibus_message_get_progress:
 * @msg: An #IBusMessage.
 *
 * Gets the progress property in #IBusMessage.
 *
 * Returns: progress property in #IBusMessage
 *
 * Since: 1.5.33
 */
int              ibus_message_get_progress      (IBusMessage  *msg);

/**
 * ibus_message_get_serial:
 * @msg: An #IBusMessage.
 *
 * Gets the serial property in #IBusMessage.
 *
 * Returns: serial property in #IBusMessage
 *
 * Since: 1.5.33
 */
guint            ibus_message_get_serial        (IBusMessage  *msg);
G_END_DECLS
#endif
