/*
 * Copyright 2024 Bloomberg Finance LP
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include <buildboxcasd_hybridfallbackexecutionservicefixture.h>

#include <build/bazel/remote/execution/v2/remote_execution.pb.h>
#include <buildboxcasd_fslocalactionstorage.h>
#include <buildboxcasd_fslocalassetstorage.h>
#include <buildboxcommon_grpctestserver.h>
#include <google/longrunning/operations.pb.h>

#include <buildboxcasd_localexecutioninstance.h>
#include <buildboxcasd_server.h>
#include <buildboxcommon_fslocalcas.h>

#include <buildboxcommon_casclient.h>
#include <buildboxcommon_digestgenerator.h>
#include <buildboxcommon_grpcclient.h>
#include <buildboxcommon_protos.h>
#include <buildboxcommon_remoteexecutionclient.h>

#include <google/rpc/code.pb.h>
#include <gtest/gtest.h>
#include <unistd.h>

using namespace buildboxcasd;
using namespace buildboxcommon;
using namespace google::longrunning;

const auto digestFunctionInitializer = []() {
    buildboxcommon::DigestGenerator::init();
    return 0;
}();

TEST_F(HybridFallbackExecutionServiceTestFixture, Execute)
{
    // Job should be executed locally and no calls to testRemoteServer will be
    // made, so a fallback to local execution will not be made since it is
    // executed locally
    const auto actionDigest = generateSimpleShellAction("echo hello, world");

    std::atomic_bool stopRequested = false;
    ActionResult actionResult =
        execClient->executeAction(actionDigest, stopRequested);
    EXPECT_EQ(actionResult.exit_code(), 0);
}

TEST_F(HybridFallbackExecutionServiceTestFixture, ExecuteMissingAction)
{
    // Job will be executed locally and no calls to testRemoteServer will be
    // made and it will fail with grpc::StatusCode::NOT_FOUND
    const auto actionDigest =
        buildboxcommon::DigestGenerator::hash("unavailable");

    std::atomic_bool stopRequested = false;
    try {
        execClient->executeAction(actionDigest, stopRequested);
        // If no exception is thrown, fail the test
        FAIL() << "Expected GrpcError exception to be thrown.";
    }
    catch (GrpcError &e) {
        EXPECT_TRUE(e.status.error_code() == grpc::StatusCode::NOT_FOUND);
    }
    ASSERT_THROW(execClient->executeAction(actionDigest, stopRequested),
                 GrpcError);
}

TEST_F(HybridFallbackExecutionServiceTestFixture, GetOperationAndWaitExecution)
{
    constexpr int TEST_TIME_SLICE = 4;
    constexpr int QUICK_TASK_SECONDS = TEST_TIME_SLICE / 4; // 1s

    // Job will be executed locally and no calls to testRemoteServer will be
    // made since it is only one job
    const auto actionDigest = generateSimpleShellAction(
        "sleep " + std::to_string(QUICK_TASK_SECONDS));

    std::atomic_bool stopRequested = false;
    auto operation =
        execClient->asyncExecuteAction(actionDigest, stopRequested);
    EXPECT_NE(operation.name(), "");
    EXPECT_FALSE(operation.done());

    operation = execClient->getOperation(operation.name());
    EXPECT_FALSE(operation.done());

    operation = execClient->waitExecution(operation.name());
    expectOperationSuccessful(operation);
}

TEST_F(HybridFallbackExecutionServiceTestFixture, GetOperation)
{
    constexpr int TEST_TIME_SLICE = 4;
    constexpr int QUICK_TASK_SECONDS = TEST_TIME_SLICE / 4;   // 1s
    constexpr int POLLING_INTERVAL_MS = TEST_TIME_SLICE * 25; // 100ms

    // Job will be executed locally and no calls to testRemoteServer will be
    // made since it is only one job
    const auto actionDigest = generateSimpleShellAction(
        "sleep " + std::to_string(QUICK_TASK_SECONDS));

    std::atomic_bool stopRequested = false;
    auto operation =
        execClient->asyncExecuteAction(actionDigest, stopRequested);
    EXPECT_NE(operation.name(), "");
    EXPECT_FALSE(operation.done());

    while (!operation.done()) {
        operation = execClient->getOperation(operation.name());
        std::this_thread::sleep_for(
            std::chrono::milliseconds(POLLING_INTERVAL_MS));
    }

    expectOperationSuccessful(operation);
}

TEST_F(HybridFallbackExecutionServiceTestFixture, CancelOperation)
{
    constexpr int TEST_TIME_SLICE = 4;
    constexpr int LONG_TASK_SECONDS = TEST_TIME_SLICE * 5; // 20s
    constexpr int CANCEL_PROOF_SECONDS = TEST_TIME_SLICE;  // 4s

    // Job will be executed locally and no calls to testRemoteServer will be
    // made since it is only one job
    const auto start = std::chrono::steady_clock::now();

    const auto actionDigest = generateSimpleShellAction(
        "sleep " + std::to_string(LONG_TASK_SECONDS));

    std::atomic_bool stopRequested = false;
    auto operation =
        execClient->asyncExecuteAction(actionDigest, stopRequested);
    EXPECT_NE(operation.name(), "");
    EXPECT_FALSE(operation.done());

    execClient->cancelOperation(operation.name());

    operation = execClient->waitExecution(operation.name());

    expectOperationCancelled(operation);

    const auto end = std::chrono::steady_clock::now();
    // Verify that cancellation took effect quickly, well before the task would
    // complete
    EXPECT_LT(end - start, std::chrono::seconds(CANCEL_PROOF_SECONDS));
}

TEST_F(HybridFallbackExecutionServiceTestFixture, Queue)
{
    constexpr int TEST_TIME_SLICE = 4;
    constexpr int PARALLEL_TASK_SECONDS = TEST_TIME_SLICE;         // 4s
    constexpr int MIN_PARALLEL_TIME_SECONDS = TEST_TIME_SLICE;     // 4s
    constexpr int MAX_PARALLEL_TIME_SECONDS = TEST_TIME_SLICE * 2; // 8s

    const auto start = std::chrono::steady_clock::now();

    const auto actionDigest1 = generateSimpleShellAction(
        "sleep " + std::to_string(PARALLEL_TASK_SECONDS) + " && echo 1");
    const auto actionDigest2 = generateSimpleShellAction(
        "sleep " + std::to_string(PARALLEL_TASK_SECONDS) + " && echo 2");
    const auto actionDigest3 = generateSimpleShellAction(
        "sleep " + std::to_string(PARALLEL_TASK_SECONDS) + " && echo 3");

    std::thread serverHandler([this, actionDigest3]() {
        auto operation = getExpectedOperation();
        operation.set_done(false);
        GrpcTestServerContext ctx(
            &testRemoteServer,
            "/build.bazel.remote.execution.v2.Execution/Execute");
        ctx.read(getExpectedExecuteRequest(actionDigest3));
        ctx.writeAndFinish(operation);
        // Sleep to fake job execution
        sleep(PARALLEL_TASK_SECONDS);
        completeOperation(&operation);
        // Handle expected WaitExecution request
        GrpcTestServerContext ctx2(
            &testRemoteServer,
            "/build.bazel.remote.execution.v2.Execution/WaitExecution");
        ctx2.read(getExpectedWaitExecutionRequest(operation));
        ctx2.writeAndFinish(operation);
    });
    try {

        std::atomic_bool stopRequested = false;
        auto operation1 =
            execClient->asyncExecuteAction(actionDigest1, stopRequested);
        auto operation2 =
            execClient->asyncExecuteAction(actionDigest2, stopRequested);
        auto operation3 =
            execClient->asyncExecuteAction(actionDigest3, stopRequested);

        operation1 = execClient->waitExecution(operation1.name());
        operation2 = execClient->waitExecution(operation2.name());
        operation3 = execClient->waitExecution(operation3.name());
        expectOperationSuccessful(operation1);
        expectOperationSuccessful(operation2);
        expectOperationSuccessful(operation3);

        const auto end = std::chrono::steady_clock::now();

        // The three sleep commands should be executed in parallel (two locally
        // and one remotely). Verify timing confirms parallel execution

        // Verify execution took at least as long as individual tasks
        EXPECT_GT(end - start,
                  std::chrono::seconds(MIN_PARALLEL_TIME_SECONDS));
        // Verify parallel execution kept total time reasonable
        EXPECT_LT(end - start,
                  std::chrono::seconds(MAX_PARALLEL_TIME_SECONDS));

        EXPECT_FALSE(isRemoteOperation(operation1));
        EXPECT_FALSE(isRemoteOperation(operation2));
        EXPECT_TRUE(isRemoteOperation(operation3));
    }
    catch (...) {
        serverHandler.join();
        throw;
    }
    serverHandler.join();
}
