use async_trait::async_trait;
use gtk::glib;

use super::{AudD, Song};
use crate::recognizer::provider::{RecognizeError, TestProvider, TestProviderMode};

#[derive(Debug)]
pub struct AudDMock;

#[async_trait(?Send)]
impl TestProvider for AudDMock {
    async fn recognize_impl(
        &self,
        _: &[u8],
        mode: TestProviderMode,
    ) -> Result<Song, RecognizeError> {
        let response_str = random_response_str(mode);
        tracing::trace!(response_str);

        AudD::build_song_from_response_bytes(response_str.as_bytes())
    }
}

fn random_response_str(mode: TestProviderMode) -> &'static str {
    let mut raw_responses = Vec::new();
    if matches!(mode, TestProviderMode::Both) || matches!(mode, TestProviderMode::ErrorOnly) {
        raw_responses.extend([
            r#"{"status":"success","result":null}"#,
            r#"{"status":"error","error":{"error_code":901,"error_message":"Recognition failed: authorization failed: no api_token passed and the limit was reached. Get an api_token from dashboard.audd.io."},"request_params":{},"request_api_method":"recognize","request_http_method":"POST","see api documentation":"https://docs.audd.io","contact us":"api@audd.io"}"#,
            r#"{"status":"error","error":{"error_code":900,"error_message":"Recognition failed: authorization failed: wrong api_token. Please check if your account is activated on dashboard.audd.io and has either a trial or an active subscription."},"request_params":{},"request_api_method":"recognize","request_http_method":"POST","see api documentation":"https://docs.audd.io","contact us":"api@audd.io"}"#,
            r#"{"status":"error","error":{"error_code":300,"error_message":"Recognition failed: a problem with fingerprints creating. Keep in mind that you should send only audio files or links to audio files. We support some of the Instagram, Twitter, TikTok and Facebook videos, and also parse html for OpenGraph and JSON-LD media and \\u003caudio\\u003e/\\u003cvideo\\u003e tags, but it's always better to send a 10-20 seconds-long audio file. For audio streams, see https://docs.audd.io/streams/"},"request_params":{},"request_api_method":"recognize","request_http_method":"POST","see api documentation":"https://docs.audd.io","contact us":"api@audd.io"}"#,
        ]);
    }
    if matches!(mode, TestProviderMode::Both) || matches!(mode, TestProviderMode::ValidOnly) {
        raw_responses.extend([
            r#"{"status":"success","result":{"artist":"The London Symphony Orchestra","title":"Eine Kleine Nachtmusik","album":"An Hour Of The London Symphony Orchestra","release_date":"2014-04-22","label":"Glory Days Music","timecode":"00:24","song_link":"https://lis.tn/EineKleineNachtmusik"}}"#,
            r#"{"status":"success","result":{"artist":"Public","title":"Make You Mine","album":"Let's Make It","release_date":"2014-10-07","label":"PUBLIC","timecode":"00:43","song_link":"https://lis.tn/FUYgUV"}}"#,
            r#"{"status":"success","result":{"artist":"5 Seconds Of Summer","title":"Amnesia","album":"Amnesia","release_date":"2014-06-24","label":"Universal Music","timecode":"01:02","song_link":"https://lis.tn/WSKAzD","spotify":{"album":{"name":"5 Seconds Of Summer","artists":[{"name":"5 Seconds of Summer","id":"5Rl15oVamLq7FbSb0NNBNy","uri":"spotify:artist:5Rl15oVamLq7FbSb0NNBNy","href":"https://api.spotify.com/v1/artists/5Rl15oVamLq7FbSb0NNBNy","external_urls":{"spotify":"https://open.spotify.com/artist/5Rl15oVamLq7FbSb0NNBNy"}}],"album_group":"","album_type":"album","id":"2LkWHNNHgD6BRNeZI2SL1L","uri":"spotify:album:2LkWHNNHgD6BRNeZI2SL1L","available_markets":null,"href":"https://api.spotify.com/v1/albums/2LkWHNNHgD6BRNeZI2SL1L","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b27393432e914046a003229378da"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e0293432e914046a003229378da"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d0000485193432e914046a003229378da"}],"external_urls":{"spotify":"https://open.spotify.com/album/2LkWHNNHgD6BRNeZI2SL1L"},"release_date":"2014-06-27","release_date_precision":"day"},"external_ids":{"isrc":"GBUM71401926"},"popularity":69,"is_playable":true,"linked_from":null,"artists":[{"name":"5 Seconds of Summer","id":"5Rl15oVamLq7FbSb0NNBNy","uri":"spotify:artist:5Rl15oVamLq7FbSb0NNBNy","href":"https://api.spotify.com/v1/artists/5Rl15oVamLq7FbSb0NNBNy","external_urls":{"spotify":"https://open.spotify.com/artist/5Rl15oVamLq7FbSb0NNBNy"}}],"available_markets":null,"disc_number":1,"duration_ms":237247,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/1JCCdiru7fhstOIF4N7WJC"},"href":"https://api.spotify.com/v1/tracks/1JCCdiru7fhstOIF4N7WJC","id":"1JCCdiru7fhstOIF4N7WJC","name":"Amnesia","preview_url":"","track_number":12,"uri":"spotify:track:1JCCdiru7fhstOIF4N7WJC"}}}"#,
            r#"{"status":"success","result":{"artist":"Alessia Cara","title":"Scars To Your Beautiful","album":"Know-It-All","release_date":"2015-11-13","label":"EP Entertainment, LLC / Def Jam","timecode":"00:28","song_link":"https://lis.tn/ScarsToYourBeautiful","spotify":{"album":{"name":"Know-It-All (Deluxe)","artists":[{"name":"Alessia Cara","id":"2wUjUUtkb5lvLKcGKsKqsR","uri":"spotify:artist:2wUjUUtkb5lvLKcGKsKqsR","href":"https://api.spotify.com/v1/artists/2wUjUUtkb5lvLKcGKsKqsR","external_urls":{"spotify":"https://open.spotify.com/artist/2wUjUUtkb5lvLKcGKsKqsR"}}],"album_group":"","album_type":"album","id":"3rDbA12I5duZnlwakqDdZa","uri":"spotify:album:3rDbA12I5duZnlwakqDdZa","available_markets":null,"href":"https://api.spotify.com/v1/albums/3rDbA12I5duZnlwakqDdZa","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b273e3ae597159d6c2541c4ee61b"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e02e3ae597159d6c2541c4ee61b"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d00004851e3ae597159d6c2541c4ee61b"}],"external_urls":{"spotify":"https://open.spotify.com/album/3rDbA12I5duZnlwakqDdZa"},"release_date":"2015-11-13","release_date_precision":"day"},"external_ids":{"isrc":"USUM71506811"},"popularity":75,"is_playable":true,"linked_from":null,"artists":[{"name":"Alessia Cara","id":"2wUjUUtkb5lvLKcGKsKqsR","uri":"spotify:artist:2wUjUUtkb5lvLKcGKsKqsR","href":"https://api.spotify.com/v1/artists/2wUjUUtkb5lvLKcGKsKqsR","external_urls":{"spotify":"https://open.spotify.com/artist/2wUjUUtkb5lvLKcGKsKqsR"}}],"available_markets":null,"disc_number":1,"duration_ms":230226,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/0prNGof3XqfTvNDxHonvdK"},"href":"https://api.spotify.com/v1/tracks/0prNGof3XqfTvNDxHonvdK","id":"0prNGof3XqfTvNDxHonvdK","name":"Scars To Your Beautiful","preview_url":"","track_number":10,"uri":"spotify:track:0prNGof3XqfTvNDxHonvdK"}}}"#,
            r#"{"status":"success","result":{"artist":"Daniel Boone","title":"Beautiful Sunday","album":"Pop Legend Vol.1","release_date":"2010-01-15","label":"Open Records","timecode":"00:33","song_link":"https://lis.tn/YTuccJ","spotify":{"album":{"name":"Cocktail Super Pop","artists":[{"name":"Various Artists","id":"0LyfQWJT6nXafLPZqxe9Of","uri":"spotify:artist:0LyfQWJT6nXafLPZqxe9Of","href":"https://api.spotify.com/v1/artists/0LyfQWJT6nXafLPZqxe9Of","external_urls":{"spotify":"https://open.spotify.com/artist/0LyfQWJT6nXafLPZqxe9Of"}}],"album_group":"","album_type":"compilation","id":"1ZsLymIsvlHEnGtQFen5xd","uri":"spotify:album:1ZsLymIsvlHEnGtQFen5xd","available_markets":null,"href":"https://api.spotify.com/v1/albums/1ZsLymIsvlHEnGtQFen5xd","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b273db8f64a52a4ec4cde9a9528a"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e02db8f64a52a4ec4cde9a9528a"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d00004851db8f64a52a4ec4cde9a9528a"}],"external_urls":{"spotify":"https://open.spotify.com/album/1ZsLymIsvlHEnGtQFen5xd"},"release_date":"2013-01-18","release_date_precision":"day"},"external_ids":{"isrc":"ES5530914999"},"popularity":0,"is_playable":true,"linked_from":null,"artists":[{"name":"Daniel Boone","id":"3M5aUsJmembbwKbUx434lS","uri":"spotify:artist:3M5aUsJmembbwKbUx434lS","href":"https://api.spotify.com/v1/artists/3M5aUsJmembbwKbUx434lS","external_urls":{"spotify":"https://open.spotify.com/artist/3M5aUsJmembbwKbUx434lS"}}],"available_markets":null,"disc_number":1,"duration_ms":176520,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/6o3AMOtlfI6APSUooekMtt"},"href":"https://api.spotify.com/v1/tracks/6o3AMOtlfI6APSUooekMtt","id":"6o3AMOtlfI6APSUooekMtt","name":"Beautiful Sunday","preview_url":"https://p.scdn.co/mp3-preview/b2fa24732fe08a251b0c8d44774f37fd55378378?cid=e44e7b8278114c7db211c00ea273ac69","track_number":16,"uri":"spotify:track:6o3AMOtlfI6APSUooekMtt"}}}"#,
            r#"{"status":"success","result":{"artist":"Kitchie Nadal","title":"Huwag Na Huwag Mong Sasabihin","album":"Kitchie Nadal","release_date":"2004-01-01","label":"12 Stone Records","timecode":"00:07","song_link":"https://lis.tn/PwQvZ","apple_music":{"previews":[{"url":"https://audio-ssl.itunes.apple.com/itunes-assets/AudioPreview115/v4/7d/2a/e5/7d2ae52b-dbde-845c-0803-9b12762cba75/mzaf_8377462403541839064.plus.aac.p.m4a"}],"artwork":{"width":1400,"height":1400,"url":"https://is4-ssl.mzstatic.com/image/thumb/Music124/v4/4b/f2/39/4bf23908-cc93-0ca5-217e-8c7e426b1990/889211800933.jpg/{w}x{h}bb.jpg","bgColor":"281800","textColor1":"d3f1b2","textColor2":"9bda12","textColor3":"b1c58e","textColor4":"84b30f"},"artistName":"Kitchie Nadal","url":"https://music.apple.com/us/album/huwag-na-huwag-mong-sasabihin/1031215547?app=music\u0026at=1000l33QU\u0026i=1031215560\u0026mt=1","discNumber":1,"genreNames":["Rock","Music","Pop"],"durationInMillis":249080,"releaseDate":"2004-01-01","name":"Huwag Na Huwag Mong Sasabihin","isrc":"USDY41595164","albumName":"Kitchie Nadal","playParams":{"id":"1031215560","kind":"song"},"trackNumber":5,"composerName":"Anna Katrina Dumilon Nadal"},"musicbrainz":[{"id":"d245a10e-f39a-4cfe-a98c-803fcf72e924","score":100,"title":"Huwag Na Huwag Mong Sasabihin","length":249080,"disambiguation":"","video":null,"artist-credit":[{"name":"Kitchie Nadal","artist":{"id":"0a7fd2c2-d03b-45d3-bc47-47547f4a91e4","name":"Kitchie Nadal","sort-name":"Nadal, Kitchie"}}],"releases":[{"id":"8d8e11b4-6859-4e78-8025-3f6254c4e4be","count":1,"title":"Kitchie Nadal","status":"Official","date":"2004-01-01","country":"PH","release-events":[{"date":"2004-01-01","area":{"id":"786532a5-2e36-315a-bdf2-221dc1b64b72","name":"Philippines","sort-name":"Philippines","iso-3166-1-codes":["PH"]}}],"track-count":12,"media":[{"position":1,"format":"Digital Media","track":[{"id":"51359cc6-197c-439b-abd4-119da1403c7a","number":"5","title":"Huwag Na Huwag Mong Sasabihin","length":249000}],"track-count":12,"track-offset":4}],"release-group":{"id":"ba030cb3-ec17-32c1-bf7b-581b6dc91592","type-id":"f529b476-6e62-324f-b0aa-1f3e33d313fc","title":"Kitchie Nadal","primary-type":"Album","secondary-types":null}},{"id":"4962e2b0-f9ad-4921-bb39-155b01084cd9","count":1,"title":"Kitchie Nadal","status":"Official","date":"2004","country":"PH","release-events":[{"date":"2004","area":{"id":"786532a5-2e36-315a-bdf2-221dc1b64b72","name":"Philippines","sort-name":"Philippines","iso-3166-1-codes":["PH"]}}],"track-count":10,"media":[{"position":1,"format":"CD","track":[{"id":"7940efe3-ec49-37d1-adb2-d5d216c89c54","number":"4","title":"Wag Na Wag Mong Sasabihin","length":250613}],"track-count":10,"track-offset":3}],"release-group":{"id":"ba030cb3-ec17-32c1-bf7b-581b6dc91592","type-id":"f529b476-6e62-324f-b0aa-1f3e33d313fc","title":"Kitchie Nadal","primary-type":"Album","secondary-types":null}},{"id":"0e42f8d2-3688-4d21-8b1f-09ec17aed74c","count":1,"title":"Kitchie Nadal","status":"Official","disambiguation":"(Special Limited Edition)","date":"2005","country":"PH","release-events":[{"date":"2005","area":{"id":"786532a5-2e36-315a-bdf2-221dc1b64b72","name":"Philippines","sort-name":"Philippines","iso-3166-1-codes":["PH"]}}],"track-count":13,"media":[{"position":1,"format":"CD","track":[{"id":"89c04c73-113b-304f-9798-b695d118a223","number":"5","title":"Huwag Na Huwag Mong Sasabihin","length":249080}],"track-count":13,"track-offset":4}],"release-group":{"id":"ba030cb3-ec17-32c1-bf7b-581b6dc91592","type-id":"f529b476-6e62-324f-b0aa-1f3e33d313fc","title":"Kitchie Nadal","primary-type":"Album","secondary-types":null}}],"isrcs":["USDY41595164"],"tags":null}],"spotify":{"album":{"name":"Kitchie Nadal","artists":[{"name":"Kitchie Nadal","id":"4OjU5UP0GFWeniBC82sGBY","uri":"spotify:artist:4OjU5UP0GFWeniBC82sGBY","href":"https://api.spotify.com/v1/artists/4OjU5UP0GFWeniBC82sGBY","external_urls":{"spotify":"https://open.spotify.com/artist/4OjU5UP0GFWeniBC82sGBY"}}],"album_group":"","album_type":"album","id":"7beVKW0o7iRoM0eRT3kGfk","uri":"spotify:album:7beVKW0o7iRoM0eRT3kGfk","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"href":"https://api.spotify.com/v1/albums/7beVKW0o7iRoM0eRT3kGfk","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b2734699830bfdf6e043cda2d294"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e024699830bfdf6e043cda2d294"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d000048514699830bfdf6e043cda2d294"}],"external_urls":{"spotify":"https://open.spotify.com/album/7beVKW0o7iRoM0eRT3kGfk"},"release_date":"2004-01-01","release_date_precision":"day"},"external_ids":{"isrc":"usdy41595164"},"popularity":55,"is_playable":null,"linked_from":null,"artists":[{"name":"Kitchie Nadal","id":"4OjU5UP0GFWeniBC82sGBY","uri":"spotify:artist:4OjU5UP0GFWeniBC82sGBY","href":"https://api.spotify.com/v1/artists/4OjU5UP0GFWeniBC82sGBY","external_urls":{"spotify":"https://open.spotify.com/artist/4OjU5UP0GFWeniBC82sGBY"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"disc_number":1,"duration_ms":249080,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/4mOjjXKDOR1sel6APULUaN"},"href":"https://api.spotify.com/v1/tracks/4mOjjXKDOR1sel6APULUaN","id":"4mOjjXKDOR1sel6APULUaN","name":"Huwag Na Huwag Mong Sasabihin","preview_url":"https://p.scdn.co/mp3-preview/4f3b0bbb8d5ed81d41a3e0d01ebbff0397840448?cid=e44e7b8278114c7db211c00ea273ac69","track_number":5,"uri":"spotify:track:4mOjjXKDOR1sel6APULUaN"}}}"#,
            r#"{"status":"success","result":{"artist":"Letter Day Story","title":"Kung Maibabalik","album":"Sama-sama","release_date":"2010-07-30","label":"SME - Musiko","timecode":"01:06","song_link":"https://lis.tn/KungMaibabalik","apple_music":{"previews":[{"url":"https://audio-ssl.itunes.apple.com/itunes-assets/AudioPreview115/v4/02/3d/6d/023d6d90-06f5-5665-0e09-dd95eda5a63c/mzaf_17349228777324501725.plus.aac.p.m4a"}],"artwork":{"width":600,"height":600,"url":"https://is2-ssl.mzstatic.com/image/thumb/Music/d7/86/e3/mzi.cpsqshnm.jpg/{w}x{h}bb.jpg","bgColor":"181d46","textColor1":"d5ecfc","textColor2":"d4c6de","textColor3":"afc3d8","textColor4":"afa4c0"},"artistName":"Letter Day Story","url":"https://music.apple.com/us/album/kung-maibabalik-duet-with-yeng-constantino/387195884?app=music\u0026at=1000l33QU\u0026i=387196208\u0026mt=1","discNumber":1,"genreNames":["Pop","Music"],"durationInMillis":292680,"releaseDate":"2010-07-30","name":"Kung Maibabalik (Duet with Yeng Constantino)","isrc":"PHS111000115","albumName":"Sama-Sama","playParams":{"id":"387196208","kind":"song"},"trackNumber":4,"composerName":"Letter Day Story"},"spotify":{"album":{"name":"Sama-sama","artists":[{"name":"Letter Day Story","id":"2DQsTcDI6YQaIKgY1FjH1M","uri":"spotify:artist:2DQsTcDI6YQaIKgY1FjH1M","href":"https://api.spotify.com/v1/artists/2DQsTcDI6YQaIKgY1FjH1M","external_urls":{"spotify":"https://open.spotify.com/artist/2DQsTcDI6YQaIKgY1FjH1M"}}],"album_group":"","album_type":"album","id":"1gTZ1F8DK1vofKEgTrQ9Ep","uri":"spotify:album:1gTZ1F8DK1vofKEgTrQ9Ep","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"href":"https://api.spotify.com/v1/albums/1gTZ1F8DK1vofKEgTrQ9Ep","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b273fc812dcdc6faf4dbec926dee"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e02fc812dcdc6faf4dbec926dee"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d00004851fc812dcdc6faf4dbec926dee"}],"external_urls":{"spotify":"https://open.spotify.com/album/1gTZ1F8DK1vofKEgTrQ9Ep"},"release_date":"2010-07-30","release_date_precision":"day"},"external_ids":{"isrc":"PHS111000115"},"popularity":41,"is_playable":null,"linked_from":null,"artists":[{"name":"Letter Day Story","id":"2DQsTcDI6YQaIKgY1FjH1M","uri":"spotify:artist:2DQsTcDI6YQaIKgY1FjH1M","href":"https://api.spotify.com/v1/artists/2DQsTcDI6YQaIKgY1FjH1M","external_urls":{"spotify":"https://open.spotify.com/artist/2DQsTcDI6YQaIKgY1FjH1M"}},{"name":"Constantino Yeng","id":"3nOJ0MZlqRyJRocFn7n7Cr","uri":"spotify:artist:3nOJ0MZlqRyJRocFn7n7Cr","href":"https://api.spotify.com/v1/artists/3nOJ0MZlqRyJRocFn7n7Cr","external_urls":{"spotify":"https://open.spotify.com/artist/3nOJ0MZlqRyJRocFn7n7Cr"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"disc_number":1,"duration_ms":292680,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/79tpIg39GSFRQcmrvyYedW"},"href":"https://api.spotify.com/v1/tracks/79tpIg39GSFRQcmrvyYedW","id":"79tpIg39GSFRQcmrvyYedW","name":"Kung Maibabalik (with Constantino Yeng)","preview_url":"https://p.scdn.co/mp3-preview/9112305c59befc6f1d1ba3dafc44bab7e1ce92ed?cid=e44e7b8278114c7db211c00ea273ac69","track_number":4,"uri":"spotify:track:79tpIg39GSFRQcmrvyYedW"}}}"#,
            r#"{"status":"success","result":{"artist":"Lord Huron","title":"The Night We Met","album":"13 Reasons Why (A Netflix Original Series Soundtrack)","release_date":"2015-02-09","label":"IAMSOUND","timecode":"00:06","song_link":"https://lis.tn/TheNightWeMet","spotify":{"album":{"name":"Strange Trails","artists":[{"name":"Lord Huron","id":"6ltzsmQQbmdoHHbLZ4ZN25","uri":"spotify:artist:6ltzsmQQbmdoHHbLZ4ZN25","href":"https://api.spotify.com/v1/artists/6ltzsmQQbmdoHHbLZ4ZN25","external_urls":{"spotify":"https://open.spotify.com/artist/6ltzsmQQbmdoHHbLZ4ZN25"}}],"album_group":"","album_type":"album","id":"4sD1qg4jwTZR4mvR4Iflk5","uri":"spotify:album:4sD1qg4jwTZR4mvR4Iflk5","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CD","CG","CH","CI","CL","CM","CO","CR","CV","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"href":"https://api.spotify.com/v1/albums/4sD1qg4jwTZR4mvR4Iflk5","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b2731fa318e90c9d4ddc6b480853"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e021fa318e90c9d4ddc6b480853"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d000048511fa318e90c9d4ddc6b480853"}],"external_urls":{"spotify":"https://open.spotify.com/album/4sD1qg4jwTZR4mvR4Iflk5"},"release_date":"2015-04-06","release_date_precision":"day"},"external_ids":{"isrc":"US53Q1200103"},"popularity":86,"is_playable":null,"linked_from":null,"artists":[{"name":"Lord Huron","id":"6ltzsmQQbmdoHHbLZ4ZN25","uri":"spotify:artist:6ltzsmQQbmdoHHbLZ4ZN25","href":"https://api.spotify.com/v1/artists/6ltzsmQQbmdoHHbLZ4ZN25","external_urls":{"spotify":"https://open.spotify.com/artist/6ltzsmQQbmdoHHbLZ4ZN25"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CD","CG","CH","CI","CL","CM","CO","CR","CV","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"disc_number":1,"duration_ms":208211,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/3hRV0jL3vUpRrcy398teAU"},"href":"https://api.spotify.com/v1/tracks/3hRV0jL3vUpRrcy398teAU","id":"3hRV0jL3vUpRrcy398teAU","name":"The Night We Met","preview_url":"https://p.scdn.co/mp3-preview/a16d7e2d837ed06f2a6a852ad694d39833deeb3e?cid=e44e7b8278114c7db211c00ea273ac69","track_number":14,"uri":"spotify:track:3hRV0jL3vUpRrcy398teAU"}}}"#,
            r#"{"status":"success","result":{"artist":"A S T R O","title":"Ten Days","album":"Summer Memories","release_date":"2021-07-30","label":"Retro Jungle","timecode":"00:46","song_link":"https://lis.tn/RlNRoN","spotify":{"album":{"name":"Ten Days","artists":[{"name":"A S T R O","id":"35iqidtNbi2CzifmDD4URB","uri":"spotify:artist:35iqidtNbi2CzifmDD4URB","href":"https://api.spotify.com/v1/artists/35iqidtNbi2CzifmDD4URB","external_urls":{"spotify":"https://open.spotify.com/artist/35iqidtNbi2CzifmDD4URB"}},{"name":"WYS","id":"2CiO7xWdwPMDlVwlt9qa1f","uri":"spotify:artist:2CiO7xWdwPMDlVwlt9qa1f","href":"https://api.spotify.com/v1/artists/2CiO7xWdwPMDlVwlt9qa1f","external_urls":{"spotify":"https://open.spotify.com/artist/2CiO7xWdwPMDlVwlt9qa1f"}}],"album_group":"","album_type":"single","id":"5SHjnvutpL73uNTFpF6kgn","uri":"spotify:album:5SHjnvutpL73uNTFpF6kgn","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"href":"https://api.spotify.com/v1/albums/5SHjnvutpL73uNTFpF6kgn","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b273e952f7d612f2b2b6486a6922"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e02e952f7d612f2b2b6486a6922"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d00004851e952f7d612f2b2b6486a6922"}],"external_urls":{"spotify":"https://open.spotify.com/album/5SHjnvutpL73uNTFpF6kgn"},"release_date":"2021-07-30","release_date_precision":"day"},"external_ids":{"isrc":"QZN882134982"},"popularity":37,"is_playable":null,"linked_from":null,"artists":[{"name":"A S T R O","id":"35iqidtNbi2CzifmDD4URB","uri":"spotify:artist:35iqidtNbi2CzifmDD4URB","href":"https://api.spotify.com/v1/artists/35iqidtNbi2CzifmDD4URB","external_urls":{"spotify":"https://open.spotify.com/artist/35iqidtNbi2CzifmDD4URB"}},{"name":"WYS","id":"2CiO7xWdwPMDlVwlt9qa1f","uri":"spotify:artist:2CiO7xWdwPMDlVwlt9qa1f","href":"https://api.spotify.com/v1/artists/2CiO7xWdwPMDlVwlt9qa1f","external_urls":{"spotify":"https://open.spotify.com/artist/2CiO7xWdwPMDlVwlt9qa1f"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"disc_number":1,"duration_ms":157307,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/6MGkXJw5R37YzKdosDHFJX"},"href":"https://api.spotify.com/v1/tracks/6MGkXJw5R37YzKdosDHFJX","id":"6MGkXJw5R37YzKdosDHFJX","name":"Ten Days","preview_url":"https://p.scdn.co/mp3-preview/8801fc62c6130531e928994022e72cfd8690f1ac?cid=e44e7b8278114c7db211c00ea273ac69","track_number":1,"uri":"spotify:track:6MGkXJw5R37YzKdosDHFJX"}}}"#,
            r#"{"status":"success","result":{"artist":"Stephen Speaks","title":"Passenger Seat","album":"Passenger Seat","release_date":"2011-12-13","label":"BELIEVE - Rippley Records","timecode":"01:14","song_link":"https://lis.tn/yihsxI","apple_music":{"previews":[{"url":"https://audio-ssl.itunes.apple.com/itunes-assets/AudioPreview126/v4/fd/54/6e/fd546e0f-99c5-dda5-ce47-64290b328847/mzaf_2247212798847815130.plus.aac.p.m4a"}],"artwork":{"width":1440,"height":1440,"url":"https://is3-ssl.mzstatic.com/image/thumb/Music126/v4/56/8b/77/568b775c-ce50-10f8-5463-32a53d81b32b/cover.jpg/{w}x{h}bb.jpg","bgColor":"050304","textColor1":"eed0a1","textColor2":"e3bb91","textColor3":"bfa782","textColor4":"b69675"},"artistName":"Stephen Speaks","url":"https://music.apple.com/us/album/passenger-seat/1604869472?app=music\u0026at=1000l33QU\u0026i=1604869987\u0026mt=1","discNumber":1,"genreNames":["Pop","Music"],"durationInMillis":274245,"releaseDate":"2011-12-13","name":"Passenger Seat","isrc":"TCABC1173267","albumName":"Passenger Seat - Single","playParams":{"id":"1604869987","kind":"song"},"trackNumber":1,"composerName":"Rockwell Ripperger \u0026 TJ McCloud"},"spotify":{"album":{"name":"Passenger Seat","artists":[{"name":"Stephen Speaks","id":"0AvtzKTYleNqdJXoZasQWG","uri":"spotify:artist:0AvtzKTYleNqdJXoZasQWG","href":"https://api.spotify.com/v1/artists/0AvtzKTYleNqdJXoZasQWG","external_urls":{"spotify":"https://open.spotify.com/artist/0AvtzKTYleNqdJXoZasQWG"}}],"album_group":"","album_type":"single","id":"6kfmY3SwNpSu8XrNeOVLd5","uri":"spotify:album:6kfmY3SwNpSu8XrNeOVLd5","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BJ","BN","BO","BR","BS","BT","BW","BZ","CA","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","MA","MC","MD","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","ZA","ZM"],"href":"https://api.spotify.com/v1/albums/6kfmY3SwNpSu8XrNeOVLd5","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b2731635b19999f7ad0d6eddc3c1"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e021635b19999f7ad0d6eddc3c1"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d000048511635b19999f7ad0d6eddc3c1"}],"external_urls":{"spotify":"https://open.spotify.com/album/6kfmY3SwNpSu8XrNeOVLd5"},"release_date":"2011-12-13","release_date_precision":"day"},"external_ids":{"isrc":"TCABC1173267"},"popularity":68,"is_playable":null,"linked_from":null,"artists":[{"name":"Stephen Speaks","id":"0AvtzKTYleNqdJXoZasQWG","uri":"spotify:artist:0AvtzKTYleNqdJXoZasQWG","href":"https://api.spotify.com/v1/artists/0AvtzKTYleNqdJXoZasQWG","external_urls":{"spotify":"https://open.spotify.com/artist/0AvtzKTYleNqdJXoZasQWG"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BJ","BN","BO","BR","BS","BT","BW","BZ","CA","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","MA","MC","MD","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RU","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","ZA","ZM"],"disc_number":1,"duration_ms":274245,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/5ZpSUdy5wL5Hvwsqz6JCAt"},"href":"https://api.spotify.com/v1/tracks/5ZpSUdy5wL5Hvwsqz6JCAt","id":"5ZpSUdy5wL5Hvwsqz6JCAt","name":"Passenger Seat","preview_url":"https://p.scdn.co/mp3-preview/172a1eabdf82bb5c932a3f59154975cdce850765?cid=e44e7b8278114c7db211c00ea273ac69","track_number":1,"uri":"spotify:track:5ZpSUdy5wL5Hvwsqz6JCAt"}}}"#,
            r#"{"status":"success","result":{"artist":"Aerosmith","title":"I Don't Want To Miss A Thing","album":"Armageddon - The Album","release_date":"1998-06-23","label":"Columbia/Sony Music Soundtrax","timecode":"00:09","song_link":"https://lis.tn/ZdAVW","lyrics":{"song_id":"320138","artist_id":"533","title":"I Don’t Want to Miss a Thing","title_with_featured":"I Don't Want to Miss a Thing","full_title":"I Don't Want to Miss a Thing by Aerosmith","artist":"Aerosmith","lyrics":"[Verse 1]\nI could stay awake just to hear you breathin'\nWatch you smile while you are sleeping\nWhile you're far away and dreaming\nI could spend my life in this sweet surrender\nI could stay lost in this moment forever\nWhere every moment spent with you is a moment I treasure\n\n[Chorus]\nDon't want to close my eyes\nI don't want to fall asleep\n'Cause I'd miss you, baby\nAnd I don't wanna miss a thing\n'Cause even when I dream of you (even when I dream)\nThe sweetest dream would never do\nI'd still miss you, babe\nAnd I don't want to miss a thing\n\n[Verse 2]\nLying close to you feeling your heart beating\nAnd I'm wondering what you're dreaming\nWonderin' if it's me you're seeing\nThen I kiss your eyes and thank God we're together\nAnd I just want to stay with you\nIn this moment forever\nForever and ever\n[Chorus]\nI don't want to close my eyes\nI don't want to fall asleep\n'Cause I'd miss you, baby\nAnd I don't wanna miss a thing\n'Cause even when I dream of you (even when I dream)\nThe sweetest dream would never do\nI'd still miss you, baby\nAnd I don't want to miss a thing\n\n[Bridge]\nI don't want to miss one smile\nI don't want to miss one kiss\nWell, I just want to be with you\nRight here with you, just like this\nI just want to hold you close\nI feel your heart so close to mine\nAnd just stay here in this moment\nFor all the rest of time\nYeah (yeah)\nYeah (yeah)\nYeah!\n\n[Chorus]\nDon't want to close my eyes\nDon't want to fall asleep\n'Cause I'd miss you, baby\nAnd I don't wanna miss a thing\n'Cause even when I dream of you (even when I dream)\nThe sweetest dream would never do\nI'd still miss you, baby\nAnd I don't want to miss a thing\n[Chorus]\nI don't want to close my eyes\nI don't want to fall asleep\n'Cause I'd miss you, baby\nAnd I don't wanna miss a thing\n'Cause even when I dream of you\nThe sweetest dream would never do\nI'd still miss you, baby\nAnd I don't want to miss a thing\n\n[Outro]\nDon't want to close my eyes\nI don't want to fall asleep, yeah\nI don't want to miss a thing","media":"[{\"provider\":\"youtube\",\"type\":\"video\",\"url\":\"http://www.youtube.com/watch?v=JkK8g6FMEXE\"},{\"provider\":\"spotify\",\"type\":\"audio\",\"url\":\"https://open.spotify.com/track/0c1gHntWjKD7QShC8s99sq\",\"native_uri\":\"spotify:track:0c1gHntWjKD7QShC8s99sq\"}]"},"apple_music":{"previews":[{"url":"https://audio-ssl.itunes.apple.com/itunes-assets/AudioPreview115/v4/92/df/59/92df5941-10fa-7356-cfc2-96346d6e2c89/mzaf_18024418854298911851.plus.aac.p.m4a"}],"artwork":{"width":600,"height":600,"url":"https://is3-ssl.mzstatic.com/image/thumb/Music115/v4/0a/f1/af/0af1af45-7deb-3252-15e2-c1394b574d64/mzi.lfxshysm.jpg/{w}x{h}bb.jpg","bgColor":"010000","textColor1":"f3a915","textColor2":"f1413c","textColor3":"c28710","textColor4":"c13430"},"artistName":"Aerosmith","url":"https://music.apple.com/us/album/i-dont-want-to-miss-a-thing/217271260?app=music\u0026at=1000l33QU\u0026i=217271368\u0026mt=1","discNumber":1,"genreNames":["Soundtrack","Music","Pop"],"durationInMillis":298760,"releaseDate":"1998-06-23","name":"I Don't Want to Miss a Thing","isrc":"USSM19801545","albumName":"Armageddon - The Album","playParams":{"id":"217271368","kind":"song"},"trackNumber":1,"composerName":"Diane Warren"},"musicbrainz":[{"id":"2e2e66bd-a016-4713-bd7f-dbb4037cc9b8","score":100,"title":"I Don’t Want to Miss a Thing","length":299000,"disambiguation":"","video":null,"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith"}}],"releases":[{"id":"3ce8e4e5-8a6c-4afa-b2c6-4be35961fbc9","count":2,"title":"Armageddon: The Album","status":"Official","date":"2016","country":"US","release-events":[{"date":"2016","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":14,"media":[{"position":1,"format":"12\" Vinyl","track":[{"id":"7b7a591a-059b-4424-8a13-2f007c10bd53","number":"A1","title":"I Don’t Want to Miss a Thing","length":298760}],"track-count":7,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"90ee3dda-176e-4666-807f-db6602531961","count":1,"title":"Best 200 Hits of the 90’s","status":"Bootleg","disambiguation":"Street market release","date":"","country":"","release-events":null,"track-count":200,"media":[{"position":1,"format":"Digital Media","track":[{"id":"e81798cf-7cda-4d0e-9da7-7eba6c104ba5","number":"41","title":"I Don’t Want to Miss a Thing","length":297189}],"track-count":200,"track-offset":40}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"4208e8cc-2b72-4f65-b1b4-7af18c874082","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Best 200 Hits of the 90’s","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"6d47e6cf-7317-45c4-9298-e267c7ed5b65","count":1,"title":"Nine Lives","status":"Official","disambiguation":"European reissue","date":"1998","country":"XE","release-events":[{"date":"1998","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":15,"media":[{"position":1,"format":"CD","track":[{"id":"e573d71d-b2ef-39e9-be4e-2d1c7634eb53","number":"15","title":"I Don’t Want To Miss a Thing","length":297174}],"track-count":15,"track-offset":14}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"7b30dd01-ee8f-392c-862d-4598e46b83f8","type-id":"f529b476-6e62-324f-b0aa-1f3e33d313fc","title":"Nine Lives","primary-type":"Album","secondary-types":null}},{"id":"634ab3fc-957d-41f9-be2e-9df3b6d3c7aa","count":1,"title":"Now That’s What I Call a Decade 1990s","status":"Official","date":"2021-04-30","country":"US","release-events":[{"date":"2021-04-30","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":18,"media":[{"position":1,"format":"CD","track":[{"id":"51596328-0122-47bd-af0e-3c3361df557a","number":"14","title":"I Don’t Want to Miss a Thing","length":297400}],"track-count":18,"track-offset":13}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"8807c5f1-4e1a-4bc3-9d40-527c5484e1f8","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Now That’s What I Call a Decade 1990s","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"9bb344f5-8b9c-4227-9750-b2e907122937","count":4,"title":"Mastermix Classic Cuts Presents: The Rock Box","status":"Promotion","date":"2007","country":"GB","release-events":[{"date":"2007","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":79,"media":[{"position":1,"format":"CD","track":[{"id":"41f523ac-dafd-33ca-a9bf-7fed51631111","number":"4","title":"I Don’t Want to Miss a Thing","length":294000}],"track-count":20,"track-offset":3}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"db980700-087f-3801-ae8f-aa703d09c66e","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Mastermix Classic Cuts Presents: The Rock Box","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"c1185a04-391d-4bb6-b5a5-8de01330c2b5","count":5,"title":"Radio 2 1000 Klassiekers, Vol. 10","status":"Official","date":"2018-11-30","country":"BE","release-events":[{"date":"2018-11-30","area":{"id":"5b8a5ee5-0bb3-34cf-9a75-c27c44e341fc","name":"Belgium","sort-name":"Belgium","iso-3166-1-codes":["BE"]}}],"track-count":76,"media":[{"position":4,"format":"Digital Media","track":[{"id":"24232470-78d7-44bd-aad2-0de10c18350a","number":"4","title":"I Don’t Want to Miss a Thing","length":298400}],"track-count":13,"track-offset":3}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"d6e87e23-1958-46db-bb6a-fadd8f586f6b","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"1000 klassiekers, Vol. 10","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"c27181da-918a-4f0c-8630-2b81c3085375","count":5,"title":"1000 klassiekers, Vol. 10","status":"Official","date":"2018-11-30","country":"BE","release-events":[{"date":"2018-11-30","area":{"id":"5b8a5ee5-0bb3-34cf-9a75-c27c44e341fc","name":"Belgium","sort-name":"Belgium","iso-3166-1-codes":["BE"]}}],"track-count":94,"media":[{"position":4,"format":"CD","track":[{"id":"70924fd9-a089-4486-98b1-f483ce80ed1e","number":"9","title":"I Don’t Want to Miss a Thing","length":298400}],"track-count":18,"track-offset":8}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"d6e87e23-1958-46db-bb6a-fadd8f586f6b","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"1000 klassiekers, Vol. 10","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"b4a85a6e-c261-43c9-b192-29aceebdde2b","count":1,"title":"Jaded","status":"Official","date":"2001","country":"GB","release-events":[{"date":"2001","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":3,"media":[{"position":1,"format":"CD","track":[{"id":"85ce7d98-7e69-4a25-a82d-35cb3fd91fe7","number":"3","title":"I Don’t Want to Miss a Thing","length":297466}],"track-count":3,"track-offset":2}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"8bcdd7a9-e0c3-3719-98c9-250f125de5a9","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"Jaded","primary-type":"Single","secondary-types":null}},{"id":"4ec869d1-10ab-403b-9556-48186edd83d6","count":1,"title":"Tough Love: Best of the Ballads","status":"Official","date":"2011-06-07","country":"DE","release-events":[{"date":"2011-06-07","area":{"id":"85752fda-13c4-31a3-bee5-0e5cb1f51dad","name":"Germany","sort-name":"Germany","iso-3166-1-codes":["DE"]}}],"track-count":16,"media":[{"position":1,"format":"CD","track":[{"id":"0a27b9aa-7a0a-46ca-beab-6483daece353","number":"1","title":"I Don't Want to Miss a Thing","length":301000}],"track-count":16,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"af4e57bd-ae6d-45db-9351-0c47835a4323","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Tough Love: Best of the Ballads","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"4721f336-f5f1-4819-964e-331b8cdaf367","count":1,"title":"Tough Love: Best of the Ballads","status":"Official","date":"2011-05-23","country":"GB","release-events":[{"date":"2011-05-23","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":16,"media":[{"position":1,"format":"CD","track":[{"id":"4b9b02e3-3373-36b0-9921-c3ab7e0b8606","number":"1","title":"I Don't Want to Miss a Thing","length":300000}],"track-count":16,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"af4e57bd-ae6d-45db-9351-0c47835a4323","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Tough Love: Best of the Ballads","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"49fee044-43f2-4009-bf11-567138769a9d","count":5,"title":"TOP 40 Hitdossier 90s","status":"Official","date":"2019","country":"NL","release-events":[{"date":"2019","area":{"id":"ef1b7cc0-cd26-36f4-8ea0-04d9623786c7","name":"Netherlands","sort-name":"Netherlands","iso-3166-1-codes":["NL"]}}],"track-count":99,"media":[{"position":2,"format":"CD","track":[{"id":"a41c1204-a4ca-4234-8918-a9f16b8d1c3a","number":"4","title":"I Don’t Want to Miss a Thing (from “Armageddon”)","length":296000}],"track-count":20,"track-offset":3}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"9f845a6a-41dd-4aa7-acf3-870744ce4309","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"TOP 40 Hitdossier 90s","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"7fb9d2db-bf96-45bf-b089-496fe2758c20","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002-08-20","country":"US","release-events":[{"date":"2002-08-20","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":30,"media":[{"position":2,"format":"SACD (2 channels)","track":[{"id":"8e8a5213-8ef0-3b2e-b4a5-cfccaa38dfdd","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"d460973b-5e14-4a0b-a987-60da4a6ed5e2","count":1,"title":"Armageddon: The Album","status":"Official","date":"","country":"","release-events":null,"track-count":14,"media":[{"position":1,"format":"Digital Media","track":[{"id":"2a5736a3-c4d2-40fe-a812-0c6845ca4057","number":"1","title":"I Don’t Want to Miss a Thing","length":299000}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"12fc772f-708a-32df-9994-29949b041092","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002","country":"GB","release-events":[{"date":"2002","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":33,"media":[{"position":2,"format":"CD","track":[{"id":"a6c0b8a8-a009-341b-80db-47b1d32e155f","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":17,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"2ca3efb3-41c6-4005-a549-eabf7aeaef09","count":2,"title":"Nine Lives","status":"Official","date":"1998","country":"XE","release-events":[{"date":"1998","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":15,"media":[{"position":2,"format":"12\" Vinyl","track":[{"id":"74dcf608-75e1-43ab-afc2-3a40c2c4c5db","number":"D3","title":"I Don't Want To Miss A Thing","length":0}],"track-count":7,"track-offset":6}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"7b30dd01-ee8f-392c-862d-4598e46b83f8","type-id":"f529b476-6e62-324f-b0aa-1f3e33d313fc","title":"Nine Lives","primary-type":"Album","secondary-types":null}},{"id":"20e562ef-c7f5-4846-8b16-65b12e06ef35","count":1,"title":"Armageddon: The Album","status":"Official","date":"","country":"XW","release-events":[{"date":"","area":{"id":"525d4e18-3d00-31b9-a58b-a146a916de8f","name":"[Worldwide]","sort-name":"[Worldwide]","iso-3166-1-codes":["XW"]}}],"track-count":14,"media":[{"position":1,"format":"Digital Media","track":[{"id":"6674f59e-3e4a-471c-8938-7092b89819af","number":"1","title":"I Don’t Want to Miss a Thing","length":298760}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"6c07894f-69a8-4b3c-aaf4-8e91ea4583e5","count":1,"title":"The 40 Greatest Power Ballads","status":"Official","date":"2015","country":"GB","release-events":[{"date":"2015","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":40,"media":[{"position":1,"format":"Digital Media","track":[{"id":"a3f6e471-ab13-4987-8dd3-d21c0d1a8fbe","number":"1","title":"I Don’t Want to Miss a Thing","length":298893}],"track-count":40,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"48eff7cc-dfa8-400d-91e9-eef4c9868f37","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"The 40 Greatest Power Ballads","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"e5b1d88d-6718-442f-a7e6-7d45adcfe72b","count":2,"title":"The Essential Aerosmith","status":"Official","date":"2011","country":"XE","release-events":[{"date":"2011","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":30,"media":[{"position":2,"format":"CD","track":[{"id":"6ea1ceab-668a-477e-9874-858bf07e519f","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"8285b67d-af15-45d0-9958-fa2feb814ca8","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"The Essential Aerosmith","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"90cc80c1-084f-49e0-b053-ef9568f64adc","count":2,"title":"The Essential Aerosmith","status":"","date":"2011-10-14","country":"DE","release-events":[{"date":"2011-10-14","area":{"id":"85752fda-13c4-31a3-bee5-0e5cb1f51dad","name":"Germany","sort-name":"Germany","iso-3166-1-codes":["DE"]}}],"track-count":30,"media":[{"position":2,"format":"CD","track":[{"id":"d4b1f18f-ee25-4398-9c77-849ac3764c4f","number":"9","title":"I Don’t Want to Miss a Thing","length":300000}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"d42329af-c448-49fa-9fcd-31241333c298","count":1,"title":"Gold Collection","status":"Official","date":"2008","country":"XE","release-events":[{"date":"2008","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":18,"media":[{"position":1,"format":"CD","track":[{"id":"228db30f-3182-4365-aa95-b939bf3a26f8","number":"16","title":"I Don't Want to Miss a Thing","length":300852}],"track-count":18,"track-offset":15}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"c6c4b2ee-2e0c-44b3-9887-1c67514d57ed","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Gold Collection","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"a00e37b2-c115-4ab0-aa3b-33fc41afa602","count":2,"title":"The Essential Aerosmith","status":"Official","disambiguation":"clean lyrics","date":"2011-09-06","country":"US","release-events":[{"date":"2011-09-06","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":30,"media":[{"position":2,"format":"Digital Media","track":[{"id":"debfefe6-d763-4168-883f-bdf33875bef0","number":"9","title":"I Don’t Want to Miss a Thing","length":298093}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"37958dba-1767-4d8a-b242-11d6a0add333","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998","country":"US","release-events":[{"date":"1998","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":4,"media":[{"position":1,"format":"CD","track":[{"id":"742a82fd-0d27-3ae5-8394-978a05b12832","number":"1","title":"I Don’t Want to Miss a Thing","length":299266}],"track-count":4,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"4576b69b-203a-4378-a0e4-f364ddd0df71","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002","country":"XE","release-events":[{"date":"2002","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":30,"media":[{"position":2,"format":"SACD","track":[{"id":"abf3aac7-35da-4bee-9cc6-c6f154b7f78c","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"3819a2f8-471c-4f28-8b24-1f598c885096","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002","country":"AU","release-events":[{"date":"2002","area":{"id":"106e0bec-b638-3b37-b731-f53d507dc00e","name":"Australia","sort-name":"Australia","iso-3166-1-codes":["AU"]}}],"track-count":33,"media":[{"position":2,"format":"CD","track":[{"id":"2da96690-2751-4e4e-81ae-d6571312110c","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":17,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"eb33aa06-6e5c-34bb-b675-2e2121220bc3","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002-06-02","country":"RU","release-events":[{"date":"2002-06-02","area":{"id":"1f1fc3a4-9500-39b8-9f10-f0a465557eef","name":"Russia","sort-name":"Russia","iso-3166-1-codes":["RU"]}}],"track-count":33,"media":[{"position":2,"format":"CD","track":[{"id":"ceed9cb9-12ad-38a7-9f43-7ac767800846","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":17,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"6e0d16d7-70c3-4bb6-a9b9-08ecd6d1e435","count":2,"title":"Knuffelrock 8","status":"Official","date":"1999-09-10","country":"NL","release-events":[{"date":"1999-09-10","area":{"id":"ef1b7cc0-cd26-36f4-8ea0-04d9623786c7","name":"Netherlands","sort-name":"Netherlands","iso-3166-1-codes":["NL"]}}],"track-count":36,"media":[{"position":1,"format":"CD","track":[{"id":"16a85988-3479-3e99-ac6b-da434fe5d353","number":"1","title":"I Don't Want to Miss a Thing","length":300666}],"track-count":18,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"6c603862-8cfa-3026-afb8-e7819df53152","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Knuffelrock 8","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"2d3d53db-79f8-330e-8114-01b3e1721716","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998-08-31","country":"GB","release-events":[{"date":"1998-08-31","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":3,"media":[{"position":1,"format":"CD","track":[{"id":"f84241ee-a2e3-371a-8cbf-03909771202a","number":"1","title":"I Don’t Want to Miss a Thing","length":299533}],"track-count":3,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"434131a8-b5d2-4f99-a4b0-d310f1fa4761","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998","country":"US","release-events":[{"date":"1998","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":3,"media":[{"position":1,"format":"CD","track":[{"id":"c5155ed3-c7ea-3d5a-9eec-b9d734c574ba","number":"1","title":"I Don’t Want to Miss a Thing","length":299533}],"track-count":3,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"ec9f2ea9-41ec-4b87-babd-e16cf08fc5ab","count":1,"title":"KISA, Volume 1: Found an Old Friend","status":"Bootleg","date":"2001-07-25","country":"US","release-events":[{"date":"2001-07-25","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":17,"media":[{"position":1,"format":"","track":[{"id":"12a507a3-b5c1-3ef7-b4e2-dcdd39e3c1f0","number":"6","title":"I Don't Want to Miss a Thing","length":299293}],"track-count":17,"track-offset":5}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"7e6df894-cba0-3129-8213-1bd0bc80a614","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"KISA, Volume 1: Found an Old Friend","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"76851b5b-5cdf-4c7f-b4e0-9353b2ee64a4","count":1,"title":"Jaded","status":"Official","date":"","country":"","release-events":null,"track-count":4,"media":[{"position":1,"format":"","track":[{"id":"fdcd8636-31d7-32d0-9a1a-7a4e0618e03f","number":"4","title":"I Don't Want to Miss a Thing","length":297000}],"track-count":4,"track-offset":3}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"8bcdd7a9-e0c3-3719-98c9-250f125de5a9","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"Jaded","primary-type":"Single","secondary-types":null}},{"id":"9ec018cb-6501-3720-9e05-fa8117d30a3d","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002-07-03","country":"JP","release-events":[{"date":"2002-07-03","area":{"id":"2db42837-c832-3c27-b4a3-08198f75693c","name":"Japan","sort-name":"Japan","iso-3166-1-codes":["JP"]}}],"track-count":34,"media":[{"position":2,"format":"CD","track":[{"id":"7dc066ad-dca3-3045-9240-4429d636a13a","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":18,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"a34c7f70-7735-4247-969c-18a2ec4508f3","count":1,"title":"Jaded","status":"Official","date":"2001-03-05","country":"GB","release-events":[{"date":"2001-03-05","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":3,"media":[{"position":1,"format":"CD","track":[{"id":"70b2c972-3502-3bcb-adbe-1c46a2cf90e2","number":"3","title":"I Don't Want to Miss a Thing","length":297466}],"track-count":3,"track-offset":2}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"8bcdd7a9-e0c3-3719-98c9-250f125de5a9","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"Jaded","primary-type":"Single","secondary-types":null}},{"id":"8afe464f-4439-4aa5-be4b-96e1698b4421","count":1,"title":"Aerosmith - The Best OF","status":"Official","date":"","country":"","release-events":null,"track-count":16,"media":[{"position":1,"format":"","track":[{"id":"918f99a8-d6c1-3060-9ef1-4a061c5159f3","number":"1","title":"I Don’t Want to Miss a Thing","length":300093}],"track-count":16,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"e86a8169-968e-3f85-8e85-db32e547618e","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Aerosmith - The Best OF","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"93602bce-0358-4a1f-b7d4-c9a2e00ef04d","count":1,"title":"Golden Collection 2000","status":"Bootleg","date":"2000","country":"RU","release-events":[{"date":"2000","area":{"id":"1f1fc3a4-9500-39b8-9f10-f0a465557eef","name":"Russia","sort-name":"Russia","iso-3166-1-codes":["RU"]}}],"track-count":17,"media":[{"position":1,"format":"CD","track":[{"id":"7e0b357a-95f8-3916-b4f8-2c3748bb03ad","number":"17","title":"I Don't Want to Miss a Thing","length":299880}],"track-count":17,"track-offset":16}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"eb6355f9-3957-394f-89dd-0ac201dfe0ac","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Golden Collection 2000","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"d004e660-2523-48e1-9e7e-9d02f34f1ae2","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998","country":"US","release-events":[{"date":"1998","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":3,"media":[{"position":1,"format":"CD","track":[{"id":"07d8ba3c-80e7-3128-8ed4-c56bbb41844d","number":"1","title":"I Don’t Want to Miss a Thing","length":302333}],"track-count":3,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"f3fec448-5a2f-405c-81c6-2d40f7edf6b0","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998-09-14","country":"SE","release-events":[{"date":"1998-09-14","area":{"id":"23d10872-f5ae-3f0c-bf55-332788a16ecb","name":"Sweden","sort-name":"Sweden","iso-3166-1-codes":["SE"]}}],"track-count":2,"media":[{"position":1,"format":"CD","track":[{"id":"63fc7d8b-7624-3dff-94b7-5c6d0d4ecfc4","number":"1","title":"I Don’t Want to Miss a Thing","length":299266}],"track-count":2,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"2699703b-9c95-45c8-821d-0e8bec2b247b","count":1,"title":"20 Years on MTV: 1998","status":"Bootleg","date":"2007","country":"XW","release-events":[{"date":"2007","area":{"id":"525d4e18-3d00-31b9-a58b-a146a916de8f","name":"[Worldwide]","sort-name":"[Worldwide]","iso-3166-1-codes":["XW"]}}],"track-count":15,"media":[{"position":1,"format":"Digital Media","track":[{"id":"96ad699f-3a44-314e-90b9-a7e3803531a4","number":"4","title":"I Don’t Want to Miss a Thing","length":296000}],"track-count":15,"track-offset":3}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"180fcde0-3a73-33b0-9790-965f245eb246","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"20 Years on MTV: 1998","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"8c3cc1ea-ff5b-49c2-9feb-9cfaaa3fb88d","count":1,"title":"Mastermix Classic Cuts 76: Rock","status":"Official","date":"","country":"GB","release-events":[{"date":"","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":17,"media":[{"position":1,"format":"CD","track":[{"id":"3415d726-f56a-3400-a71e-32b5637c6cb4","number":"2","title":"I Don't Want to Miss a Thing","length":297000}],"track-count":17,"track-offset":1}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"749b058f-23f1-3ffc-b965-a9c6d49b708e","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Mastermix Classic Cuts 76: Rock","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"a44d5923-ddaf-447a-8c08-d3307f8b24ff","count":1,"title":"Fly Away From Here","status":"Official","date":"","country":"","release-events":null,"track-count":4,"media":[{"position":1,"format":"CD","track":[{"id":"effd99ef-35d9-390c-a978-287d5b6080ba","number":"4","title":"I Don’t Want to Miss a Thing (album version)","length":298400}],"track-count":4,"track-offset":3}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"334f33a9-ab45-4231-9557-cd8af34e93c5","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"Fly Away From Here","primary-type":"Single","secondary-types":null}},{"id":"7472d623-eb64-4d43-bc4a-cbbaa9fc4800","count":2,"title":"The All Time Greatest Movie Songs of the Last 3 Decades","status":"Official","date":"2000","country":"NL","release-events":[{"date":"2000","area":{"id":"ef1b7cc0-cd26-36f4-8ea0-04d9623786c7","name":"Netherlands","sort-name":"Netherlands","iso-3166-1-codes":["NL"]}}],"track-count":41,"media":[{"position":2,"format":"CD","track":[{"id":"fd887d16-b417-4242-a2ed-0ad9dae0c87f","number":"1","title":"I Don’t Want to Miss a Thing","length":297906}],"track-count":19,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"2eb51e0c-1406-40ed-9d70-b1852def311b","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"The All Time Greatest Movie Songs of the Last 3 Decades","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"cfc31187-aebd-309f-a92f-7138c17df7c2","count":1,"title":"Armageddon: The Album","status":"Official","date":"1998","country":"DE","release-events":[{"date":"1998","area":{"id":"85752fda-13c4-31a3-bee5-0e5cb1f51dad","name":"Germany","sort-name":"Germany","iso-3166-1-codes":["DE"]}}],"track-count":14,"media":[{"position":1,"format":"CD","track":[{"id":"111dc0f7-1f4e-392a-8f57-cd152590e425","number":"1","title":"I Don’t Want to Miss a Thing","length":299000}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"29cad9d7-d8e6-4fa2-bf2c-bf75fa55509b","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002-07-02","country":"US","release-events":[{"date":"2002-07-02","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":30,"media":[{"position":2,"format":"CD","track":[{"id":"c6697209-0883-3ee3-a0d9-76bc1286b0d2","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"8ea1eb1b-28fb-4cd6-a1ea-4fcdb7c88d39","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998","country":"XE","release-events":[{"date":"1998","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":4,"media":[{"position":1,"format":"CD","track":[{"id":"1563d3bc-456f-33dd-b4a3-4d5b4014a618","number":"1","title":"I Don’t Want to Miss a Thing","length":297000}],"track-count":4,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"2eef3b7c-914c-40da-bebf-d61fa983b8fa","count":5,"title":"Mastermix Jock in a Box, Volume 2","status":"Official","date":"","country":"GB","release-events":[{"date":"","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":108,"media":[{"position":3,"format":"CD","track":[{"id":"c3bc8933-e793-305b-bcc8-005f0194a5af","number":"4","title":"I Don’t Want to Miss a Thing","length":298000}],"track-count":21,"track-offset":3}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"5475d8af-b8ab-33d6-91a0-ea5a70104752","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Mastermix Jock in a Box, Volume 2","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"32d2c3be-6654-44ac-b393-099dff1ade56","count":1,"title":"New Best Ballads","status":"","date":"","country":"","release-events":null,"track-count":16,"media":[{"position":1,"format":"","track":[{"id":"fa776f67-6287-3b13-abd3-eeeb1e5ecb6f","number":"1","title":"I Don't Want to Miss a Thing","length":299000}],"track-count":16,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"68865e27-da4d-4a90-885d-8c2eae8b77d8","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"New Best Ballads","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"e9535ce5-716b-4d66-bdfa-f8bcf07feb82","count":1,"title":"Armageddon: The Album","status":"Official","date":"","country":"XW","release-events":[{"date":"","area":{"id":"525d4e18-3d00-31b9-a58b-a146a916de8f","name":"[Worldwide]","sort-name":"[Worldwide]","iso-3166-1-codes":["XW"]}}],"track-count":14,"media":[{"position":1,"format":"Digital Media","track":[{"id":"d37e5da5-ff39-4702-b91f-99f45b684568","number":"1","title":"I Don’t Want to Miss a Thing","length":298000}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"20c068d7-b800-4ce7-ae77-8510a0b65710","count":2,"title":"Movie Music: The Definitive Performances","status":"Official","date":"1999-10-12","country":"US","release-events":[{"date":"1999-10-12","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":44,"media":[{"position":2,"format":"CD","track":[{"id":"565627b9-aad9-3458-a4a6-03310a1b88f0","number":"20","title":"I Don’t Want to Miss a Thing","length":297000}],"track-count":20,"track-offset":19}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"31ab37a0-9589-3e4a-9b8b-cf0720171ea7","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Movie Music: The Definitive Performances","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"dd98f74d-b578-4895-92aa-89f8e0106c2f","count":1,"title":"Armageddon: The Album","status":"Official","date":"1998-06-23","country":"US","release-events":[{"date":"1998-06-23","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":14,"media":[{"position":1,"format":"CD","track":[{"id":"ddbc371e-d5e0-3f27-9e69-8ed4736c7c28","number":"1","title":"I Don’t Want to Miss a Thing","length":299000}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"99a31236-ffeb-4056-a48c-1132dff12a0b","count":1,"title":"Armageddon: The Album","status":"Official","disambiguation":"BMG club edition","date":"1998-06-23","country":"US","release-events":[{"date":"1998-06-23","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":14,"media":[{"position":1,"format":"CD","track":[{"id":"6da76f95-5327-49aa-9f34-e8532a525de0","number":"1","title":"I Don’t Want to Miss a Thing","length":299000}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}},{"id":"4c2e93ea-d2b3-48d7-968a-21ecb7dc5bbf","count":2,"title":"The All Time Greatest Movie Songs, Volume Two","status":"Official","date":"1999-11-01","country":"GB","release-events":[{"date":"1999-11-01","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":36,"media":[{"position":2,"format":"CD","track":[{"id":"86f7f1e8-0c53-333e-b1c0-8c8d5758012a","number":"2","title":"I Don’t Want to Miss a Thing","length":299000}],"track-count":19,"track-offset":1}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"8839417d-b0a2-3ef7-8d85-3c925bc3d824","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"The All Time Greatest Movie Songs, Volume Two","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"2bd3d87e-0501-4061-960a-fa76418666fe","count":1,"title":"I Don’t Want to Miss a Thing","status":"Official","date":"1998","country":"AU","release-events":[{"date":"1998","area":{"id":"106e0bec-b638-3b37-b731-f53d507dc00e","name":"Australia","sort-name":"Australia","iso-3166-1-codes":["AU"]}}],"track-count":4,"media":[{"position":1,"format":"CD","track":[{"id":"e369d14e-7304-38c1-95a6-dc9e20ad252a","number":"1","title":"I Don’t Want to Miss a Thing","length":298000}],"track-count":4,"track-offset":0}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"545d7014-44a3-4254-b803-122c097525aa","count":1,"title":"I Don’t Want to Miss a Thing","status":"Promotion","date":"1998","country":"XE","release-events":[{"date":"1998","area":{"id":"89a675c2-3e37-3518-b83c-418bad59a85a","name":"Europe","sort-name":"Europe","iso-3166-1-codes":["XE"]}}],"track-count":3,"media":[{"position":1,"format":"CD","track":[{"id":"d44c5008-4235-4588-ab29-0f7429fd4bb3","number":"2","title":"I Don’t Want to Miss a Thing (pop mix)","length":298000}],"track-count":3,"track-offset":1}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"55fca8e9-74ee-352a-97a4-82fe299d3826","type-id":"d6038452-8ee0-3f68-affc-2de9a1ede0b9","title":"I Don’t Want to Miss a Thing","primary-type":"Single","secondary-types":null}},{"id":"3f419b0c-ac62-4ce6-9742-0e6fc5ddc8e3","count":2,"title":"Sony Music 100 Years: Soundtrack for a Century","status":"Official","disambiguation":"part of a 26-disc box set","date":"1999-10-12","country":"US","release-events":[{"date":"1999-10-12","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":44,"media":[{"position":14,"format":"CD","track":[{"id":"269d61a4-5813-33ee-850d-a1bb8e73366d","number":"20","title":"I Don’t Want to Miss a Thing","length":297134}],"track-count":20,"track-offset":19}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"c111aa39-0882-3c2b-ae91-7def56aaf0d1","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"Sony Music 100 Years: Soundtrack for a Century","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"a307edd1-a219-417e-9ffb-7f613752d1f1","count":2,"title":"O, Yeah! Ultimate Aerosmith Hits","status":"Official","date":"2002","country":"US","release-events":[{"date":"2002","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":30,"media":[{"position":2,"format":"CD","track":[{"id":"0f9ca2c6-0396-41a5-93b0-8cb7d000b32b","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"63ff1b14-c338-4b05-99b8-3860054c9d21","count":2,"title":"The Essential Aerosmith","status":"Official","date":"2011-09-13","country":"XW","release-events":[{"date":"2011-09-13","area":{"id":"525d4e18-3d00-31b9-a58b-a146a916de8f","name":"[Worldwide]","sort-name":"[Worldwide]","iso-3166-1-codes":["XW"]}}],"track-count":30,"media":[{"position":2,"format":"CD","track":[{"id":"b9f9cde5-5708-3069-8a3b-dfd201afa054","number":"9","title":"I Don’t Want to Miss a Thing","length":300000}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"46b0f87b-0665-3110-ad6c-8fcf1811cd3a","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"O, Yeah! Ultimate Aerosmith Hits","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"d40d9cfb-0ec6-494f-ac48-5433bf278540","count":2,"title":"The Essential Aerosmith","status":"Official","date":"2011-09-13","country":"US","release-events":[{"date":"2011-09-13","area":{"id":"489ce91b-6658-3307-9877-795b68554c98","name":"United States","sort-name":"United States","iso-3166-1-codes":["US"]}}],"track-count":30,"media":[{"position":2,"format":"CD","track":[{"id":"7cdaefc8-1d8f-4b64-9bad-1fb397ab3a0a","number":"9","title":"I Don’t Want to Miss a Thing","length":299626}],"track-count":14,"track-offset":8}],"artist-credit":[{"name":"Aerosmith","artist":{"id":"3d2b98e5-556f-4451-a3ff-c50ea18d57cb","name":"Aerosmith","sort-name":"Aerosmith","disambiguation":""}}],"release-group":{"id":"8285b67d-af15-45d0-9958-fa2feb814ca8","type-id":"dd2a21e1-0c00-3729-a7a0-de60b84eb5d1","title":"The Essential Aerosmith","primary-type":"Album","secondary-types":["Compilation"]}},{"id":"d05168e5-8403-45a4-ac3b-3fd35d888ef5","count":1,"title":"Armageddon: The Album","status":"Official","date":"2002-02-04","country":"GB","release-events":[{"date":"2002-02-04","area":{"id":"8a754a16-0027-3a29-b6d7-2b40ea0481ed","name":"United Kingdom","sort-name":"United Kingdom","iso-3166-1-codes":["GB"]}}],"track-count":14,"media":[{"position":1,"format":"CD","track":[{"id":"cf27bde9-ee6b-4f9e-b1bf-0a87e444018b","number":"1","title":"I Don't Want To Miss A Thing","length":297123}],"track-count":14,"track-offset":0}],"artist-credit":[{"name":"Various Artists","artist":{"id":"89ad4ac3-39f7-470e-963a-56509c546377","name":"Various Artists","sort-name":"Various Artists","disambiguation":"add compilations to this artist"}}],"release-group":{"id":"f51d56e4-0211-3533-a9a5-08c02d8bb04a","type-id":"22a628ad-c082-3c4f-b1b6-d41665107b88","title":"Armageddon: The Album","primary-type":"Album","secondary-types":["Soundtrack"]}}],"isrcs":["USSM19801545"],"tags":[{"count":7,"name":"rock"},{"count":4,"name":"soundtrack"},{"count":2,"name":"heavy metal"},{"count":8,"name":"hard rock"},{"count":4,"name":"blues rock"},{"count":1,"name":"southern rock"},{"count":3,"name":"classic rock"},{"count":3,"name":"album rock"},{"count":1,"name":"pop rock"},{"count":1,"name":"arena rock"},{"count":1,"name":"showtunes"},{"count":1,"name":"movie soundtrack"},{"count":1,"name":"pop-metal"},{"count":1,"name":"southern rock_soundtrack_soft rock"},{"count":1,"name":"general film music"},{"count":1,"name":"rock ballad"},{"count":0,"name":"pop/rock"},{"count":0,"name":"contemporary pop/rock"},{"count":0,"name":"soundtrack/rock"}]}],"spotify":{"album":{"name":"Armageddon - The Album","artists":[{"name":"Various Artists","id":"0LyfQWJT6nXafLPZqxe9Of","uri":"spotify:artist:0LyfQWJT6nXafLPZqxe9Of","href":"https://api.spotify.com/v1/artists/0LyfQWJT6nXafLPZqxe9Of","external_urls":{"spotify":"https://open.spotify.com/artist/0LyfQWJT6nXafLPZqxe9Of"}}],"album_group":"","album_type":"compilation","id":"4f7HKjBnjpMsDBDTHYV890","uri":"spotify:album:4f7HKjBnjpMsDBDTHYV890","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"href":"https://api.spotify.com/v1/albums/4f7HKjBnjpMsDBDTHYV890","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b273da8d92affd796f7e20af7375"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e02da8d92affd796f7e20af7375"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d00004851da8d92affd796f7e20af7375"}],"external_urls":{"spotify":"https://open.spotify.com/album/4f7HKjBnjpMsDBDTHYV890"},"release_date":"1998-06-23","release_date_precision":"day"},"external_ids":{"isrc":"USSM19801545"},"popularity":73,"is_playable":null,"linked_from":null,"artists":[{"name":"Aerosmith","id":"7Ey4PD4MYsKc5I2dolUwbH","uri":"spotify:artist:7Ey4PD4MYsKc5I2dolUwbH","href":"https://api.spotify.com/v1/artists/7Ey4PD4MYsKc5I2dolUwbH","external_urls":{"spotify":"https://open.spotify.com/artist/7Ey4PD4MYsKc5I2dolUwbH"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"disc_number":1,"duration_ms":298760,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/225xvV8r1yKMHErSWivnow"},"href":"https://api.spotify.com/v1/tracks/225xvV8r1yKMHErSWivnow","id":"225xvV8r1yKMHErSWivnow","name":"I Don't Want to Miss a Thing - From \"Armageddon\" Soundtrack","preview_url":"https://p.scdn.co/mp3-preview/0768ad35cf7d25571e02d3538c77980e75eadff0?cid=e44e7b8278114c7db211c00ea273ac69","track_number":1,"uri":"spotify:track:225xvV8r1yKMHErSWivnow"}}}"#,
            r#"{"status":"success","result":{"artist":"Bamboo","title":"Tatsulok","album":"We Stand Alone Together","label":"PolyEast Records","timecode":"00:11","song_link":"https://lis.tn/Tatsulok","lyrics":{"song_id":"3718388","artist_id":"61565","title":"Tatsulok","title_with_featured":"Tatsulok","full_title":"Tatsulok by Bamboo","artist":"Bamboo","lyrics":"Totoy bilisan mo, bilisan mo ang takbo\nIlagan ang mga bombang nakatutok sa ulo mo\nTotoy tumalon ka, dumapa kung kailangan\nAt baka tamaan pa ng mga balang ligaw\n\nTotoy makinig ka, wag kang magpa-gabi\nBaka mapagkamalan ka\u0026#39;t humandusay dyan sa tabi\nTotoy alam mo ba kung ano ang puno\u0026#39;t dulo\nNg di matapos-tapos na kaguluhang ito\n\nHindi pula\u0026#39;t dilaw tunay na magkalaban\nAng kulay at tatak ay di syang dahilan\nHangga\u0026#39;t marami ang lugmok sa kahirapan\u003c","media":"[{\"provider\":\"youtube\",\"type\":\"video\",\"url\":\"http://www.youtube.com/watch?v=hI3T_D8TCUk\"}]"},"musicbrainz":[{"id":"9996dee5-0139-4689-ad8b-6b58a6832899","score":100,"title":"Tatsulok / Englishman in New York","length":1037000,"disambiguation":"","video":null,"artist-credit":[{"name":"Bamboo","artist":{"id":"82cb9ad1-71ef-46ff-8924-3f6c36eaa2df","name":"Bamboo","sort-name":"Bamboo"}}],"releases":[{"id":"b7f7c8a3-9845-4378-85de-f4b07b4ea913","count":1,"title":"We Stand Alone Together","status":"Official","date":"2014-01-27","country":"PH","release-events":[{"date":"2014-01-27","area":{"id":"786532a5-2e36-315a-bdf2-221dc1b64b72","name":"Philippines","sort-name":"Philippines","iso-3166-1-codes":["PH"]}}],"track-count":8,"media":[{"position":1,"format":"Digital Media","track":[{"id":"5a84eed7-0715-4749-a12e-e454c67f5fc9","number":"8","title":"Tatsulok / Englishman in New York","length":1037000}],"track-count":8,"track-offset":7}],"release-group":{"id":"41d4c92d-11d3-3955-9302-d6599fd5cc32","type-id":"f529b476-6e62-324f-b0aa-1f3e33d313fc","title":"We Stand Alone Together","primary-type":"Album","secondary-types":null}}],"isrcs":["PHO010700013"],"tags":null}],"spotify":{"album":{"name":"We Stand Alone Together","artists":[{"name":"Bamboo","id":"0c7d9V8rjfT6tzojfBfRf5","uri":"spotify:artist:0c7d9V8rjfT6tzojfBfRf5","href":"https://api.spotify.com/v1/artists/0c7d9V8rjfT6tzojfBfRf5","external_urls":{"spotify":"https://open.spotify.com/artist/0c7d9V8rjfT6tzojfBfRf5"}}],"album_group":"","album_type":"album","id":"3kdYoCrhB4HN3HM4d0uzhC","uri":"spotify:album:3kdYoCrhB4HN3HM4d0uzhC","available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"href":"https://api.spotify.com/v1/albums/3kdYoCrhB4HN3HM4d0uzhC","images":[{"height":640,"width":640,"url":"https://i.scdn.co/image/ab67616d0000b27304c1502c6e319485d7ef2103"},{"height":300,"width":300,"url":"https://i.scdn.co/image/ab67616d00001e0204c1502c6e319485d7ef2103"},{"height":64,"width":64,"url":"https://i.scdn.co/image/ab67616d0000485104c1502c6e319485d7ef2103"}],"external_urls":{"spotify":"https://open.spotify.com/album/3kdYoCrhB4HN3HM4d0uzhC"},"release_date":"2007","release_date_precision":"year"},"external_ids":{"isrc":"PHO010700013"},"popularity":47,"is_playable":null,"linked_from":null,"artists":[{"name":"Bamboo","id":"0c7d9V8rjfT6tzojfBfRf5","uri":"spotify:artist:0c7d9V8rjfT6tzojfBfRf5","href":"https://api.spotify.com/v1/artists/0c7d9V8rjfT6tzojfBfRf5","external_urls":{"spotify":"https://open.spotify.com/artist/0c7d9V8rjfT6tzojfBfRf5"}}],"available_markets":["AD","AE","AG","AL","AM","AO","AR","AT","AU","AZ","BA","BB","BD","BE","BF","BG","BH","BI","BJ","BN","BO","BR","BS","BT","BW","BY","BZ","CA","CD","CG","CH","CI","CL","CM","CO","CR","CV","CW","CY","CZ","DE","DJ","DK","DM","DO","DZ","EC","EE","EG","ES","FI","FJ","FM","FR","GA","GB","GD","GE","GH","GM","GN","GQ","GR","GT","GW","GY","HK","HN","HR","HT","HU","ID","IE","IL","IN","IQ","IS","IT","JM","JO","JP","KE","KG","KH","KI","KM","KN","KR","KW","KZ","LA","LB","LC","LI","LK","LR","LS","LT","LU","LV","LY","MA","MC","MD","ME","MG","MH","MK","ML","MN","MO","MR","MT","MU","MV","MW","MX","MY","MZ","NA","NE","NG","NI","NL","NO","NP","NR","NZ","OM","PA","PE","PG","PH","PK","PL","PS","PT","PW","PY","QA","RO","RS","RW","SA","SB","SC","SE","SG","SI","SK","SL","SM","SN","SR","ST","SV","SZ","TD","TG","TH","TJ","TL","TN","TO","TR","TT","TV","TW","TZ","UA","UG","US","UY","UZ","VC","VE","VN","VU","WS","XK","ZA","ZM","ZW"],"disc_number":1,"duration_ms":1037586,"explicit":false,"external_urls":{"spotify":"https://open.spotify.com/track/69iiml9D5HulXPUZIUcTU1"},"href":"https://api.spotify.com/v1/tracks/69iiml9D5HulXPUZIUcTU1","id":"69iiml9D5HulXPUZIUcTU1","name":"Tatsulok","preview_url":"https://p.scdn.co/mp3-preview/07e5bfea5cb45b5bd3b6b4a6731817e466cdae30?cid=e44e7b8278114c7db211c00ea273ac69","track_number":8,"uri":"spotify:track:69iiml9D5HulXPUZIUcTU1"}}}"#,
        ]);
    }

    raw_responses
        .get(glib::random_int_range(0, raw_responses.len() as i32) as usize)
        .expect("index must be in range")
}
