% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_convergence.R
\name{check_convergence}
\alias{check_convergence}
\title{Convergence test for mixed effects models}
\usage{
check_convergence(model = NULL, tolerance = 0.001, x = NULL, ...)
}
\arguments{
\item{model}{A \code{merMod} or \code{glmmTMB}-object.}

\item{tolerance}{Indicates up to which value the convergence result is
accepted. The smaller \code{tolerance} is, the stricter the test will be.}

\item{x}{Deprecated, please use \code{model} instead.}

\item{...}{Currently not used.}
}
\value{
\code{TRUE} if convergence is fine and \code{FALSE} if convergence
is suspicious. Additionally, the convergence value is returned as attribute.
}
\description{
\code{check_convergence()} provides an alternative convergence
test for \code{merMod}-objects.
}
\section{Convergence and log-likelihood}{


Convergence problems typically arise when the model hasn't converged to a
solution where the log-likelihood has a true maximum. This may result in
unreliable and overly complex (or non-estimable) estimates and standard
errors.

}

\section{Inspect model convergence}{


\strong{lme4} performs a convergence-check (see \code{?lme4::convergence}), however, as
discussed \href{https://github.com/lme4/lme4/issues/120}{here} and suggested by
one of the lme4-authors in \href{https://github.com/lme4/lme4/issues/120#issuecomment-39920269}{this comment},
this check can be too strict. \code{is_converged()} (and its wrapper function,
\code{performance::check_convergence()}) thus provides an alternative convergence
test for \code{merMod}-objects.

}

\section{Resolving convergence issues}{


Convergence issues are not easy to diagnose. The help page on
\code{?lme4::convergence} provides most of the current advice about how to resolve
convergence issues. In general, convergence issues may be addressed by one or
more of the following strategies: 1. Rescale continuous predictors; 2. try a
different optimizer; 3. increase the number of iterations; or, if everything
else fails, 4. simplify the model. Another clue might be large parameter
values, e.g. estimates (on the scale of the linear predictor) larger than 10
in (non-identity link) generalized linear model \emph{might} indicate complete
separation, which can be addressed by regularization, e.g. penalized
regression or Bayesian regression with appropriate priors on the fixed
effects.

}

\section{Convergence versus Singularity}{


Note the different meaning between singularity and convergence: singularity
indicates an issue with the "true" best estimate, i.e. whether the maximum
likelihood estimation for the variance-covariance matrix of the random effects
is positive definite or only semi-definite. Convergence is a question of
whether we can assume that the numerical optimization has worked correctly
or not. A convergence failure means the optimizer (the algorithm) could not
find a stable solution (\emph{Bates et. al 2015}).

For singular models (see \code{?lme4::isSingular}), convergence is determined
based on the optimizer's convergence code. If the optimizer reports
successful convergence (convergence code 0) for a singular model,
\code{is_converged()} returns \code{TRUE}. For non-singular models, in cases where the
gradient and Hessian are not available, \code{is_converged()} returns \code{FALSE} and
prints a message to indicate that convergence cannot be assessed through the
usual gradient-based checks. Note that \code{performance::check_convergence()} is
a wrapper around \code{insight::is_converged()}.

}

\examples{
\dontshow{if (require("lme4") && require("glmmTMB")) withAutoprint(\{ # examplesIf}
data(cbpp, package = "lme4")
set.seed(1)
cbpp$x <- rnorm(nrow(cbpp))
cbpp$x2 <- runif(nrow(cbpp))

model <- lme4::glmer(
  cbind(incidence, size - incidence) ~ period + x + x2 + (1 + x | herd),
  data = cbpp,
  family = binomial()
)

check_convergence(model)

\donttest{
model <- suppressWarnings(glmmTMB::glmmTMB(
  Sepal.Length ~ poly(Petal.Width, 4) * poly(Petal.Length, 4) +
    (1 + poly(Petal.Width, 4) | Species),
  data = iris
))
check_convergence(model)
}
\dontshow{\}) # examplesIf}
}
\references{
Bates, D., Mächler, M., Bolker, B., and Walker, S. (2015). Fitting Linear
Mixed-Effects Models Using lme4. Journal of Statistical Software, 67(1),
1-48. \doi{10.18637/jss.v067.i01}
}
\seealso{
Other functions to check model assumptions and and assess model quality: 
\code{\link{check_autocorrelation}()},
\code{\link{check_collinearity}()},
\code{\link{check_heteroscedasticity}()},
\code{\link{check_homogeneity}()},
\code{\link{check_model}()},
\code{\link{check_outliers}()},
\code{\link{check_overdispersion}()},
\code{\link{check_predictions}()},
\code{\link{check_singularity}()},
\code{\link{check_zeroinflation}()}
}
\concept{functions to check model assumptions and and assess model quality}
