use isahc::http::StatusCode;
use isahc::{prelude::*, HttpClient, Request};
use serde::de::DeserializeOwned;
use url::Url;

use std::cell::RefCell;
use std::rc::Rc;

use crate::error::ClientError;
use crate::rpc::{
    RequestArgs, RpcRequest, RpcResponse, RpcResponseArguments, SessionSetArgs, TorrentActionArgs,
    TorrentAddArgs, TorrentGetArgs, TorrentRemoveArgs, TorrentSetArgs, TorrentSetLocationArgs,
};
use crate::utils;
use crate::{
    Authentication, Session, SessionMutator, SessionStats, Torrent, TorrentAdded, TorrentMutator,
    Torrents, PortTest
};

#[derive(Debug, Clone)]
pub struct Client {
    address: Url,
    authentication: Rc<RefCell<Option<Authentication>>>,
    http_client: HttpClient,
    session_id: Rc<RefCell<String>>,
}

impl Client {
    pub fn new(address: Url) -> Self {
        let mut client = Self::default();
        client.address = address;
        client
    }

    pub fn set_authentication(&self, auth: Option<Authentication>) {
        *self.authentication.borrow_mut() = auth;
    }

    pub async fn torrents(&self, ids: Option<Vec<String>>) -> Result<Vec<Torrent>, ClientError> {
        let mut args = TorrentGetArgs::default();
        args.fields = utils::torrent_fields();
        args.ids = ids;
        let request_args = Some(RequestArgs::TorrentGetArgs(args));

        let response: RpcResponse<Torrents> =
            self.send_request("torrent-get", request_args).await?;
        Ok(response.arguments.unwrap().torrents)
    }

    pub async fn torrent_set(
        &self,
        ids: Option<Vec<String>>,
        mutator: TorrentMutator,
    ) -> Result<(), ClientError> {
        let args = TorrentSetArgs { ids, mutator };
        let request_args = Some(RequestArgs::TorrentSetArgs(args));

        let _: RpcResponse<String> = self.send_request("torrent-set", request_args).await?;
        Ok(())
    }

    pub async fn torrent_add_filename(
        &self,
        filename: &str,
    ) -> Result<Option<Torrent>, ClientError> {
        let mut args = TorrentAddArgs::default();
        args.filename = Some(filename.into());
        let request_args = Some(RequestArgs::TorrentAddArgs(args));
        self.torrent_add(request_args).await
    }

    pub async fn torrent_add_metainfo(
        &self,
        metainfo: &str,
    ) -> Result<Option<Torrent>, ClientError> {
        let mut args = TorrentAddArgs::default();
        args.metainfo = Some(metainfo.into());
        let request_args = Some(RequestArgs::TorrentAddArgs(args));
        self.torrent_add(request_args).await
    }

    async fn torrent_add(
        &self,
        request_args: Option<RequestArgs>,
    ) -> Result<Option<Torrent>, ClientError> {
        let response: RpcResponse<TorrentAdded> =
            self.send_request("torrent-add", request_args).await?;

        let result_args = response.arguments.unwrap();
        if result_args.torrent_added.is_some() {
            Ok(result_args.torrent_added)
        } else {
            Ok(result_args.torrent_duplicate)
        }
    }

    pub async fn torrent_remove(
        &self,
        ids: Option<Vec<String>>,
        delete_local_data: bool,
    ) -> Result<(), ClientError> {
        let mut args = TorrentRemoveArgs::default();
        args.delete_local_data = delete_local_data;
        args.ids = ids;
        let request_args = Some(RequestArgs::TorrentRemoveArgs(args));

        let _: RpcResponse<String> = self.send_request("torrent-remove", request_args).await?;
        Ok(())
    }

    pub async fn torrent_start(
        &self,
        ids: Option<Vec<String>>,
        bypass_queue: bool,
    ) -> Result<(), ClientError> {
        let mut args = TorrentActionArgs::default();
        args.ids = ids;
        let request_args = Some(RequestArgs::TorrentActionArgs(args));

        let method_name = if bypass_queue {
            "torrent-start-now"
        } else {
            "torrent-start"
        };

        let _: RpcResponse<String> = self.send_request(method_name, request_args).await?;
        Ok(())
    }

    pub async fn torrent_stop(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("torrent-stop", ids).await?;
        Ok(())
    }

    pub async fn torrent_verify(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("torrent-verify", ids).await?;
        Ok(())
    }

    pub async fn torrent_reannounce(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("torrent-reannounce", ids).await?;
        Ok(())
    }

    pub async fn torrent_set_location(
        &self,
        ids: Option<Vec<String>>,
        location: String,
        move_data: bool,
    ) -> Result<(), ClientError> {
        let mut args = TorrentSetLocationArgs::default();
        args.ids = ids;
        args.location = location;
        args.move_data = move_data;
        let request_args = Some(RequestArgs::TorrentSetLocationArgs(args));

        let _: RpcResponse<String> = self
            .send_request("torrent-set-location", request_args)
            .await?;
        Ok(())
    }

    pub async fn queue_move_top(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("queue-move-top", ids).await?;
        Ok(())
    }

    pub async fn queue_move_up(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("queue-move-up", ids).await?;
        Ok(())
    }

    pub async fn queue_move_down(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("queue-move-down", ids).await?;
        Ok(())
    }

    pub async fn queue_move_bottom(&self, ids: Option<Vec<String>>) -> Result<(), ClientError> {
        self.send_torrent_action("queue-move-bottom", ids).await?;
        Ok(())
    }

    pub async fn session(&self) -> Result<Session, ClientError> {
        let response: RpcResponse<Session> = self.send_request("session-get", None).await?;
        Ok(response.arguments.unwrap())
    }

    pub async fn session_set(&self, mutator: SessionMutator) -> Result<(), ClientError> {
        let args = SessionSetArgs { mutator };
        let request_args = Some(RequestArgs::SessionSetArgs(args));

        let _: RpcResponse<String> = self.send_request("session-set", request_args).await?;
        Ok(())
    }

    pub async fn session_stats(&self) -> Result<SessionStats, ClientError> {
        let response: RpcResponse<SessionStats> = self.send_request("session-stats", None).await?;
        Ok(response.arguments.unwrap())
    }

    pub async fn session_close(&self) -> Result<(), ClientError> {
        let _: RpcResponse<String> = self.send_request("session-close", None).await?;
        Ok(())
    }

    pub async fn port_test(&self) -> Result<bool, ClientError> {
        let response: RpcResponse<PortTest> = self.send_request("port-test", None).await?;
        Ok(response.arguments.unwrap().port_is_open)
    }

    async fn send_torrent_action(
        &self,
        action: &str,
        ids: Option<Vec<String>>,
    ) -> Result<(), ClientError> {
        let mut args = TorrentActionArgs::default();
        args.ids = ids;
        let request_args = Some(RequestArgs::TorrentActionArgs(args));

        let _: RpcResponse<String> = self.send_request(&action, request_args).await?;
        Ok(())
    }

    async fn send_request<T: RpcResponseArguments + DeserializeOwned>(
        &self,
        method: &str,
        arguments: Option<RequestArgs>,
    ) -> Result<RpcResponse<T>, ClientError> {
        let request = RpcRequest {
            method: method.into(),
            arguments,
        };

        let body = serde_json::to_string(&request)?;
        let result = self.send_post(body).await?;

        match serde_json::from_str::<RpcResponse<T>>(&result) {
            Ok(response) => {
                if response.result != "success" {
                    return Err(ClientError::TransmissionError(response.result));
                }

                Ok(response)
            }
            Err(err) => {
                error!("Unable to parse json: {}", err.to_string());
                warn!("JSON: {:#?}", &result);

                Err(err.into())
            }
        }
    }

    async fn send_post(&self, body: String) -> Result<String, ClientError> {
        let request = self.http_request(body.clone())?;
        let mut response = self.http_client.send_async(request).await?;

        // Update session id
        let headers = response.headers();
        if let Some(session_id) = headers.get("X-Transmission-Session-Id") {
            let session_id = session_id.to_str().unwrap().to_string();
            *self.session_id.borrow_mut() = session_id;
        }

        // Check html status code
        match response.status() {
            // Invalid session id header, resend the request
            StatusCode::CONFLICT => {
                debug!("Received status code 409, resend request.");
                let request = self.http_request(body.clone())?;
                response = self.http_client.send_async(request).await?;
            }
            // Authentication needed
            StatusCode::UNAUTHORIZED => {
                return Err(ClientError::TransmissionUnauthorized);
            }
            _ => (),
        }

        Ok(response.text().await.unwrap())
    }

    fn http_request(&self, body: String) -> Result<Request<String>, ClientError> {
        let session_id = self.session_id.borrow().clone();

        let request = if let Some(auth) = &*self.authentication.borrow() {
            Request::post(self.address.to_string())
                .header("X-Transmission-Session-Id", session_id)
                .header("Authorization", auth.base64_encoded())
                .body(body)?
        } else {
            Request::post(self.address.to_string())
                .header("X-Transmission-Session-Id", session_id)
                .body(body)?
        };

        Ok(request)
    }
}

impl Default for Client {
    fn default() -> Self {
        let address = Url::parse("http://127.0.0.1:9091/transmission/rpc/").unwrap();
        let http_client = HttpClient::builder()
            .authentication(isahc::auth::Authentication::all())
            .build()
            .unwrap();
        let session_id = Rc::new(RefCell::new("0".into()));

        Self {
            address,
            authentication: Rc::default(),
            http_client,
            session_id,
        }
    }
}
