/************************************************************************
 *
 * Copyright (C) 2023-2025 IRCAD France
 *
 * This file is part of Sight.
 *
 * Sight is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Sight is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Sight. If not, see <https://www.gnu.org/licenses/>.
 *
 ***********************************************************************/

#pragma once

#include "common.hxx"
#include "io/bitmap/writer.hpp"

#ifdef SIGHT_ENABLE_NVJPEG
#include "nvjpeg_writer.hxx"
#endif

#ifdef SIGHT_ENABLE_NVJPEG2K
#include "nvjpeg2k_writer.hxx"
#endif

#include "libjpeg_writer.hxx"
#include "libtiff_writer.hxx"
#include "libpng_writer.hxx"
#include "openjpeg_writer.hxx"

// cspell:ignore nvjpeg LIBJPEG LIBTIFF LIBPNG

namespace sight::io::bitmap::detail
{

class writer_impl final
{
public:

    /// Delete default constructors and assignment operators
    writer_impl()                              = delete;
    writer_impl(const writer_impl&)            = delete;
    writer_impl(writer_impl&&)                 = delete;
    writer_impl& operator=(const writer_impl&) = delete;
    writer_impl& operator=(writer_impl&&)      = delete;

    /// Constructor
    inline explicit writer_impl(writer* const _writer) :
        m_writer(_writer)
    {
    }

    /// Default destructor
    inline ~writer_impl() noexcept = default;

    /// Main write function
    template<typename O>
    inline std::size_t write(O& _output, backend _backend, writer::mode _mode)
    {
        // Get the image pointer
        const auto& image = m_writer->get_concrete_object();
        SIGHT_THROW_IF("Source image is null", image == nullptr);

        /// @todo Should we split volume in 2D slices ?
        const auto& sizes = image->size();
        SIGHT_THROW_IF("Unsupported image dimension", sizes[0] == 0 || sizes[1] == 0);

        // Protect the image from dump
        const auto dump_lock = image->dump_lock();

#ifdef SIGHT_ENABLE_NVJPEG2K
        if(nvjpeg2k() && _backend == backend::nvjpeg2k)
        {
            try
            {
                return write<nvjpeg2k_writer>(m_nvjpeg2k, *image, _output, _mode);
            }
            catch(const std::exception& e)
            {
                SIGHT_ERROR("Failed to write image with nvjpeg2k: " << e.what() << ".");
                throw;
            }
        }
        else if(nvjpeg2k() && _backend == backend::nvjpeg2k_j2k)
        {
            try
            {
                return write<nvjpeg2k_writer>(m_nvjpeg2k, *image, _output, _mode, flag::j2k_stream);
            }
            catch(const std::exception& e)
            {
                // Same as above...
                SIGHT_ERROR("Failed to write image with nvjpeg2k: " << e.what() << ".");
                throw;
            }
        }
        else
#endif
        if(_backend == backend::openjpeg)
        {
            return write<openjpeg_writer>(m_openjpeg, *image, _output, _mode);
        }
        else if(_backend == backend::openjpeg_j2k)
        {
            return write<openjpeg_writer>(
                m_openjpeg,
                *image,
                _output,
                _mode,
                flag::j2k_stream
            );
        }
        else

#ifdef SIGHT_ENABLE_NVJPEG
        if(nvjpeg() && _backend == backend::nvjpeg)
        {
            return write<nvjpeg_writer>(m_nvjpeg, *image, _output, _mode);
        }
        else
#endif
        if(_backend == backend::libjpeg)
        {
            return write<libjpeg_writer>(m_libjpeg, *image, _output, _mode);
        }
        else if(_backend == backend::libtiff)
        {
            return write<libtiff_writer>(m_libtiff, *image, _output, _mode);
        }
        else if(_backend == backend::libpng)
        {
            return write<libpng_writer>(m_libpng, *image, _output, _mode);
        }
        else
        {
            SIGHT_THROW("No suitable backend found.");
        }
    }

private:

    //------------------------------------------------------------------------------

    template<typename W, typename O>
    static std::size_t write(
        std::unique_ptr<W>& _backend,
        const data::image& _image,
        O& _output,
        writer::mode _mode,
        flag _flag = flag::none
)
    {
        if(_backend == nullptr)
        {
            _backend = std::make_unique<W>();
            SIGHT_THROW_IF("Failed to initialize" << _backend->name() << " backend.", !_backend->valid());
        }

        return _backend->write(_image, _output, _mode, _flag);
    }

    /// Pointer to the public interface
    writer* const m_writer;

#ifdef SIGHT_ENABLE_NVJPEG
    std::unique_ptr<nvjpeg_writer> m_nvjpeg;
#endif

#ifdef SIGHT_ENABLE_NVJPEG2K
    std::unique_ptr<nvjpeg2k_writer> m_nvjpeg2k;
#endif

    std::unique_ptr<libjpeg_writer> m_libjpeg;
    std::unique_ptr<libtiff_writer> m_libtiff;
    std::unique_ptr<libpng_writer> m_libpng;
    std::unique_ptr<openjpeg_writer> m_openjpeg;
};

} // namespace sight::io::bitmap::detail
