/* -*- mode: C++; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
// SPDX-License-Identifier: MIT OR LGPL-2.0-or-later
// SPDX-FileCopyrightText: 2009 Red Hat, Inc.

#include <config.h>

#include <stdio.h>  // for stderr

#include <sstream>
#include <string>

#include <glib-object.h>
#include <glib.h>

#include <js/Printer.h>
#include <js/TypeDecls.h>
#include <js/Utility.h>  // for UniqueChars
#include <js/friend/DumpFunctions.h>

#include "gjs/auto.h"
#include "gjs/context-private.h"
#include "gjs/context.h"

void gjs_context_print_stack_stderr(GjsContext* self) {
    auto* cx = static_cast<JSContext*>(gjs_context_get_native_context(self));

    g_printerr("== Stack trace for context %p ==\n", self);
    js::DumpBacktrace(cx, stderr);
}

void gjs_dumpstack() {
    Gjs::SmartPointer<GList> contexts{gjs_context_get_all()};

    for (GList* iter = contexts; iter; iter = iter->next) {
        Gjs::AutoUnref<GjsContext> gjs_context{GJS_CONTEXT(iter->data)};
        gjs_context_print_stack_stderr(gjs_context);
    }
}

std::string gjs_dumpstack_string() {
    std::ostringstream all_traces;

    Gjs::SmartPointer<GList> contexts{gjs_context_get_all()};
    js::Sprinter printer;

    for (GList* iter = contexts; iter; iter = iter->next) {
        Gjs::AutoUnref<GjsContext> gjs_context{GJS_CONTEXT(iter->data)};
        if (!printer.init()) {
            all_traces << "No stack trace for context " << gjs_context.get()
                       << ": out of memory\n\n";
            break;
        }
        auto* cx = static_cast<JSContext*>(
            gjs_context_get_native_context(gjs_context));
        js::DumpBacktrace(cx, printer);
        JS::UniqueChars trace = printer.release();
        all_traces << "== Stack trace for context " << gjs_context.get()
                   << " ==\n"
                   << trace.get() << "\n";
    }
    std::string out = all_traces.str();
    out.resize(MAX(out.size() - 2, 0));

    return out;
}
