#! /usr/bin/perl

# 
# Copyright 1999-2006 University of Chicago
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
# http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# 

#
# Test to exercise the "put" functionality of the Globus FTP client library
# in extended block mode

use strict;
use Test::More;
use File::Basename;
use lib dirname($0);
use FtpTestLib;

my $test_exec = './extended-put-test';

my @tests;
my @todo;

my ($proto) = setup_proto();
my ($local_copy) = setup_local_source(1);
my ($dest_host, $dest_file) = setup_remote_dest();

# Test #1-11. Basic functionality: Do a put of $test_file to
# a new unique file name on localhost, varying parallelism level.
# Compare the resulting file with the real file
# Success if program returns 0, files compare,
# and no core file is generated.
sub basic_func
{
    my ($parallelism) = (shift);
    my ($errors,$rc) = ("",0);

    my $command = "$test_exec -P $parallelism -d $proto$dest_host$dest_file < $local_copy";
    $errors = run_command($command, 0);
    if($errors eq "")
    {
        my ($output) = get_remote_file($dest_host, $dest_file);
        $errors = compare_local_files($local_copy, $output);
        unlink($output);
    }

    ok($errors eq "", "basic_func $parallelism $command");
    
    clean_remote_file($dest_host, $dest_file);
}
for(my $par = 0; $par <= 10; $par++)
{
    push(@tests, "basic_func($par);");
}

# Test #12: Bad URL: Do a simple put to a bad location on the ftp server.
# Success if program returns 1 and no core file is generated.
sub bad_url
{
    my ($errors,$rc) = ("",0);

    my $command = "$test_exec -d $proto$dest_host/no/such/file/here < $local_copy";
    $errors = run_command($command, 1);
    
    ok($errors eq "", "bad_url $command");
}
push(@tests, "bad_url");

# Test #13-53: Do a simple put of $test_file to localhost, aborting
# at each possible position. Note that not all aborts may be reached.
# Success if no core file is generated for all abort points. (we could use
# a stronger measure of success here)
sub abort_test
{
    my ($errors,$rc) = ("", 0);
    my ($abort_point) = shift;

    my $command = "$test_exec -a $abort_point -d $proto$dest_host$dest_file < $local_copy";
    $errors = run_command($command, -2);
    ok($errors eq "", "abort_test $abort_point $command");
    
    clean_remote_file($dest_host, $dest_file);
}
for(my $i = 1; $i <= 43; $i++)
{
    push(@tests, "abort_test($i);");
}

# Test #54-94. Restart functionality: Do a simple put of $test_file to
# localhost, restarting at each plugin-possible point.
# Compare the resulting file with the real file
# Success if program returns 0, files compare,
# and no core file is generated.
sub restart_test
{
    my ($errors,$rc) = ("",0);
    my ($restart_point) = shift;

    my $command = "$test_exec -r $restart_point -d $proto$dest_host$dest_file < $local_copy";
    $errors = run_command($command, 0);
    if($errors eq "")
    {
        my ($output) = get_remote_file($dest_host, $dest_file);
        $errors = compare_local_files($local_copy, $output);
        unlink($output);
    }

    ok($errors eq "", "restart_test $restart_point $command");
    
    clean_remote_file($dest_host, $dest_file);
}
for(my $i = 1; $i <= 43; $i++)
{
    push(@tests, "restart_test($i);");

    if($i == 38)
    {
	push(@todo, 54 + $i);
    }
}


=head2 I<perf_test> (Test 95)

Do an extended put of $testfile, enabling perf_plugin

=back

=cut
sub perf_test
{
    my ($errors,$rc) = ("",0);

    my $command = "$test_exec -d $proto$dest_host$dest_file -M < $local_copy";
    $errors = run_command($command, 0);

    ok($errors eq "", "perf_test $command");
    
    clean_remote_file($dest_host, $dest_file);
}

push(@tests, "perf_test();");

=head2 I<throughput_test> (Test 96)

Do an extended put of $testfile, enabling throughput_plugin

=back

=cut
sub throughput_test
{
    my ($errors,$rc) = ("",0);

    my $command = "$test_exec -d $proto$dest_host$dest_file -T < $local_copy";
    $errors = run_command($command, 0);

    ok($errors eq "", "throughput_test $command");
    
    clean_remote_file($dest_host, $dest_file);
}

push(@tests, "throughput_test();");

if(defined($ENV{FTP_TEST_RANDOMIZE}))
{
    shuffle(\@tests);
}

if(@ARGV)
{
    plan tests => scalar(@ARGV);

    foreach (@ARGV)
    {
        eval "&$tests[$_-1]";
    }
}
else
{
    plan tests => scalar(@tests), todo => \@todo;

    foreach (@tests)
    {
        eval "&$_";
    }
}
