------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-environments.adb,v 1.12 2007/04/25 10:08:45 vagul Exp $

with Ada.Text_IO;

with System;

package body OCI.Environments is

   use type SWord, Ub4, OCIHandle;

   Create_Mode : Ub4 := OCI_THREADED;

   Memory_Fill_Byte : Integer := -1;
   pragma Atomic (Memory_Fill_Byte);

   Counter_Value : Integer := 0;
   pragma Atomic (Counter_Value);

   procedure Counter_Add (It : Integer);
   pragma Inline (Counter_Add);

   function Alloc_Memory
     (ctxp : in System.Address; Size : in Integer) return System.Address;
   pragma Convention (C, Alloc_Memory);

   function Realloc_Memory
     (ctxp   : in System.Address;
      Memptr : in System.Address;
      Size   : in Integer) return System.Address;
   pragma Convention (C, Realloc_Memory);

   procedure Free_Memory
     (ctxp : in System.Address; Memptr : in System.Address);
   pragma Convention (C, Free_Memory);

   function Malloc (Size : in Integer) return System.Address;
   pragma Import (C, Malloc, "__gnat_malloc");

   procedure Free (Address : in System.Address);
   pragma Import (C, Free, "__gnat_free");

   function Realloc
     (Ptr  : in System.Address; Size : in Integer) return System.Address;
   pragma Import (C, Realloc, "__gnat_realloc");

   function memset
     (S : System.Address; C : Integer; N : Integer) return System.Address;
   pragma Import (C, memset, "memset");

   ------------------
   -- Alloc_Memory --
   ------------------

   function Alloc_Memory
     (ctxp : in System.Address; Size : in Integer) return System.Address
   is
      pragma Unreferenced (ctxp);
   begin
      Counter_Add (1);
      Last_Size := Size;

      if Memory_Fill_Byte >= 0 then
         return memset (Malloc (Size), Memory_Fill_Byte, Size);
      else
         return Malloc (Size);
      end if;
   end Alloc_Memory;

   -------------
   -- Counter --
   -------------

   package body Counter is

      -----------
      -- Value --
      -----------

      function Value return Integer is
      begin
         return Counter_Value;
      end Value;

   end Counter;

   -----------------
   -- Counter_Add --
   -----------------

   procedure Counter_Add (It : Integer) is
   begin
      Counter_Value := Counter_Value + It;
   end Counter_Add;

   ------------
   -- Create --
   ------------

   function Create return Thread_Environment is
      Env         : aliased OCIEnv := OCIEnv (Empty_Handle);
      Rc          : SWord;
      Environment : Thread_Environment;
   begin
      Rc := OCIEnvCreate
        (Env'Access,
         Create_Mode,
         Malocfp => Alloc_Memory'Address,
         Ralocfp => Realloc_Memory'Address,
         Mfreefp => Free_Memory'Address);

      if Rc /= OCI_SUCCESS then
         raise Environment_Creation_Error;
      end if;

      Environment.Handle := Env;
      return Environment;
   end Create;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Thread_Environment) is
      Rc : SWord;
      use Ada.Text_IO;
   begin
      if Object.Handle /= OCIEnv (Empty_Handle) then
         Rc := OCIHandleFree (OCIHandle (Object.Handle), OCI_HTYPE_ENV);

         Object.Handle := OCIEnv (Empty_Handle);

         if Rc /= OCI_SUCCESS then
            Put_Line
              (Current_Error,
               "Warning: Return code on free environment "
                  & SWord'Image (Rc));
         end if;
      end if;
   end Destroy;

   -----------------
   -- Free_Memory --
   -----------------

   procedure Free_Memory
     (ctxp : in System.Address; Memptr : in System.Address)
   is
      pragma Unreferenced (ctxp);
   begin
      Counter_Add (-1);
      Free (Memptr);
   end Free_Memory;

   ----------------
   -- Is_Objects --
   ----------------

   function Is_Objects return Boolean is
   begin
      return (Create_Mode and OCI_OBJECT) /= 0;
   end Is_Objects;

   --------------------
   -- Realloc_Memory --
   --------------------

   function Realloc_Memory
     (ctxp   : in System.Address;
      Memptr : in System.Address;
      Size   : in Integer) return System.Address
   is
      pragma Unreferenced (ctxp);
   begin
      if Size = 0 then
         Counter_Add (-1);
      end if;

      return Realloc (Memptr, Size);
   end Realloc_Memory;

   --------------------------
   -- Set_Create_Mode_Flag --
   --------------------------

   procedure Set_Create_Mode_Flag (Flag : in Ub4) is
   begin
      Create_Mode := Create_Mode or Flag;
   end Set_Create_Mode_Flag;

   --------------------------
   -- Set_Memory_Fill_Byte --
   --------------------------

   procedure Set_Memory_Fill_Byte (Byte : Memory_Byte) is
   begin
      Memory_Fill_Byte := Integer (Byte);
   end Set_Memory_Fill_Byte;

end OCI.Environments;
