with Ada.Exceptions;
with Ada.Text_IO;
with GNAT.Float_Control;
with Interfaces.C;
with System;
with OCI.Lib;
with OCI.Thick.Gen_OCINumber;
with OCI.Thick.Number_Functions;
with Ada.Numerics.Discrete_Random;

procedure Test_Num is
   use Ada.Text_IO;
   use Interfaces;

   subtype Max_Float_Type is Long_Long_Float;

   package Max_Random is new Ada.Numerics.Discrete_Random (Integer);
   Gen : Max_Random.Generator;

   package Max_Float_IO is new Ada.Text_IO.Float_IO (Max_Float_Type);

   F : Max_Float_Type := 1.0;

   F_Stamp  : Max_Float_Type := 0.0;
   Max_Diff : Max_Float_Type := 0.0;
   Min_Diff : Max_Float_Type := 0.0;
   Direction : Boolean := True;

   generic
      type Float_Type is digits <>;
   procedure Test_Generic (F : Float_Type);

   procedure Test_Generic (F : Float_Type) is
      package Float_IO is new Ada.Text_IO.Float_IO (Float_Type);
      use OCI.Thick;
      use OCI.Thick.Number_Functions;

      type Unsigned_Type is mod System.Max_Binary_Modulus;
      function To_Number is new Gen_OCINumber.Unsigned_To_Number
                                  (Unsigned_Type);
      function To_Number
        is new Gen_OCINumber.Float_To_Number_Better (Float_Type);
      function To_Float
        is new Gen_OCINumber.Num_To_Float_Better (Float_Type);

      function To_Float_OCI
        is new Gen_OCINumber.Num_To_Float (Float_Type);

      N : OCI.Lib.OCINumber := To_Number (F);
      F2 : Float_Type := To_Float (N);
      FO : Float_Type := To_Float_OCI (N);
      Diff : Max_Float_Type := Max_Float_Type ((F2 - F) / abs F);
      Diff_O : Float_Type := (FO - F) / abs F;

   begin
      if Sign (N) = 0 and then not Is_Zero (N) then
         Put_Line ("Different zero detection " & To_String (N));
      end if;

      if abs Diff_O > 0.001 then
         Float_IO.Put (F, Aft => Float_Type'Digits + 3);
         Float_IO.Put (Float_Type'Small, Aft => Float_Type'Digits + 3);
         Float_IO.Put (Float_Type'Last, Aft => Float_Type'Digits + 3);
         raise Constraint_Error with
           Float_Type'Machine_Emin'Img & Float_Type'Machine_Emax'Img
           & F'Img & FO'Img & ' ' & To_String (N) & " abs Diff_O >"
           & Float_Type'Image (Float_Type'Small * 100.0);
      end if;

      if abs Diff > 0.000001 then
         Float_IO.Put (F, Aft => 20);
         New_Line;
         Float_IO.Put (F2, Aft => 20);
         New_Line;

         raise Constraint_Error with "too big difference "
            & F'Img & ' ' & To_String (N) & F2'Img;
      end if;

      if Diff > Max_Diff then
         Max_Diff := Diff;
         Put_Line ("Max " & Diff'Img);
         raise Constraint_Error;
      elsif Diff < Min_Diff then
         Min_Diff := Diff;
         Put_Line ("Min " & Diff'Img);
         raise Constraint_Error;
      end if;

      if N /= To_Number (F2) then
         if F = F2 then
            raise Constraint_Error with "overprecision";
         end if;

         Float_IO.Put (F, Aft => 20);
         New_Line;

         for J in 1 .. 20 loop
            Put (To_String (N) & ' '); Float_IO.Put (To_Float (N), Aft => 20);
            New_Line;
            exit when N = To_Number (To_Float (N));
            N := To_Number (To_Float (N));
         end loop;

         raise Program_Error;
      end if;
   end Test_Generic;

   ----------
   -- Test --
   ----------

   procedure Test (Item : Max_Float_Type) is
      procedure Test_Float      is new Test_Generic (Float);
      procedure Test_Long_Float is new Test_Generic (Long_Float);
      procedure Test_Max_Float  is new Test_Generic (Long_Long_Float);
   begin
      Test_Max_Float (Item);
      Test_Long_Float (Long_Float (Item));

      if abs Item <= Max_Float_Type (Float'Last)
        and then abs Item >= Float'Small
      then
         Test_Float (Float (Item));
      end if;
   end Test;

   function Random return Max_Float_Type is
   begin
      return Max_Float_Type (Max_Random.Random (Gen));
   end Random;

begin
   Max_Random.Reset (Gen);

   Test (F);
   GNAT.Float_Control.Reset;

   for J in 1 .. 388 loop
      F := -F * 2.0;
      Put (F'Img & J'Img);
      Test (F);
      Put_Line (" 1.0/");
      Test (1.0 / F);
   end loop;

   F := 1.0 / F;

   for J in 389 .. 402 loop
      F := -F / 2.0;
      Put_Line (F'Img & J'Img);
      Test (F);
   end loop;

   F := Random;

   for J in 1 .. 2000000000 loop
      declare
         use Ada.Exceptions;
      begin
         Test (F);

         if Direction then
            --  Only when Direction is true Otherwise we would leap around zero
            F := F + Random;
            Test (F);
         end if;

         for K in 1 .. 100 loop
            F := F * Random / Random;
            Test (F);
         end loop;

      exception when E : OCI.Thick.Lib_Error =>
         if F_Stamp = F then
            Put_Line ("F_Stamp " & F'Img);
            return;
         elsif F_Stamp = 0.0 then
            F_Stamp := F;
         end if;

         declare
            Errmsg : constant String := Exception_Message (E);
            Ermsg  : constant String := Errmsg (4 .. Errmsg'Last);
            --  Mask first 3 symbols because it could be OCI for object
            --  environment and ORA for deafult environment, at least in
            --  Oracle client version 10.2.
         begin
            if Ermsg = "-22053: overflow error" & ASCII.LF
              and then abs F > 1.0E100
            then
               Direction := False;
               Put (F'Img & ' ' & Errmsg);

            elsif Ermsg = "-22054: underflow error" & ASCII.LF
              and then abs F < 1.0E-100
            then
               Direction := True;
               Put (F'Img & ' ' & Errmsg);
            else
               Put_Line ("F " & J'Img & F'Img & ' ' & Errmsg);
               return;
            end if;
         end;

      when E : others =>
         Put_Line
           ("F " & J'Img & ' ' & F'Img & ' ' & Exception_Information (E));
         return;
      end;

      if Direction then
         F := F * Random;
      else
         F := F / Random;
      end if;
   end loop;
end Test_Num;
