/*
 * Algorithm testing framework and tests.
 *
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 * Copyright (c) 2002 Jean-Francois Dive <jef@linuxbe.org>
 * Copyright (c) 2007 Nokia Siemens Networks
 * Copyright (c) 2008 Herbert Xu <herbert@gondor.apana.org.au>
 *
 * Updated RFC4106 AES-GCM testing. Some test vectors were taken from
 * http://csrc.nist.gov/groups/ST/toolkit/BCM/documents/proposedmodes/
 * gcm/gcm-test-vectors.tar.gz
 *     Authors: Aidan O'Mahony (aidan.o.mahony@intel.com)
 *              Adrian Hoban <adrian.hoban@intel.com>
 *              Gabriele Paoloni <gabriele.paoloni@intel.com>
 *              Tadeusz Struk (tadeusz.struk@intel.com)
 *     Copyright (c) 2010, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 */
#ifndef _CRYPTO_TESTMGR_H
#define _CRYPTO_TESTMGR_H

#include <linux/netlink.h>

#define MAX_DIGEST_SIZE		64
#define MAX_TAP			8

#define MAX_KEYLEN		160
#define MAX_IVLEN		32

struct hash_testvec {
	/* only used with keyed hash algorithms */
	const char *key;
	const char *plaintext;
	const char *digest;
	unsigned char tap[MAX_TAP];
	unsigned short psize;
	unsigned char np;
	unsigned char ksize;
};

/*
 * cipher_testvec:	structure to describe a symmetric cipher test
 * @key:	Pointer to key
 * @klen:	Length of @key in bytes
 * @iv:		Pointer to IV (optional for some ciphers)
 * @ptext:	Pointer to plaintext
 * @ctext:	Pointer to ciphertext
 * @len:	Length of @ptext and @ctext in bytes
 * @fail:	If set to one, the test need to fail
 * @wk:		Does the test need CRYPTO_TFM_REQ_WEAK_KEY
 * 		( e.g. test needs to fail due to a weak key )
 * @np: 	numbers of SG to distribute data in (from 1 to MAX_TAP)
 * @tap:	How to distribute data in @np SGs
 * @also_non_np: 	if set to 1, the test will be also done without
 * 			splitting data in @np SGs
 * @fips_skip:	Skip the test vector in FIPS mode
 * @generates_iv: Encryption should ignore the given IV, and output @iv.
 *		  Decryption takes @iv.  Needed for AES Keywrap ("kw(aes)").
 */
struct cipher_testvec {
	const char *key;
	const char *iv;
	const char *ptext;
	const char *ctext;
	unsigned short tap[MAX_TAP];
	int np;
	unsigned char also_non_np;
	bool fail;
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short len;
	bool fips_skip;
	bool generates_iv;
};

struct aead_testvec {
	const char *key;
	const char *iv;
	const char *input;
	const char *assoc;
	const char *result;
	unsigned char tap[MAX_TAP];
	unsigned char atap[MAX_TAP];
	int np;
	int anp;
	bool fail;
	unsigned char novrfy;	/* ccm dec verification failure expected */
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short ilen;
	unsigned short alen;
	unsigned short rlen;
};

struct cprng_testvec {
	const char *key;
	const char *dt;
	const char *v;
	const char *result;
	unsigned char klen;
	unsigned short dtlen;
	unsigned short vlen;
	unsigned short rlen;
	unsigned short loops;
};

struct drbg_testvec {
	const unsigned char *entropy;
	size_t entropylen;
	const unsigned char *entpra;
	const unsigned char *entprb;
	size_t entprlen;
	const unsigned char *addtla;
	const unsigned char *addtlb;
	size_t addtllen;
	const unsigned char *pers;
	size_t perslen;
	const unsigned char *expected;
	size_t expectedlen;
};

struct akcipher_testvec {
	const unsigned char *key;
	const unsigned char *m;
	const unsigned char *c;
	unsigned int key_len;
	unsigned int m_size;
	unsigned int c_size;
	bool public_key_vec;
	bool siggen_sigver_test;
};

struct kpp_testvec {
	const unsigned char *secret;
	const unsigned char *b_secret;
	const unsigned char *b_public;
	const unsigned char *expected_a_public;
	const unsigned char *expected_ss;
	unsigned short secret_size;
	unsigned short b_secret_size;
	unsigned short b_public_size;
	unsigned short expected_a_public_size;
	unsigned short expected_ss_size;
	bool genkey;
};

static const char zeroed_string[48];

static const struct cipher_testvec gost28147_tc26z_tv_template[] = {
	{
		.key	= "\x81\x82\x83\x84\x85\x86\x87\x88"
			  "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x80"
			  "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			  "\xd9\xda\xdb\xdc\xdd\xde\xdf\xd0",
		.klen	= 32,
		.ptext	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8",
		.ctext	= "\xce\x5a\x5e\xd7\xe0\x57\x7a\x5f"
			  "\xd0\xcc\x85\xce\x31\x63\x5b\x8b",
		.len	= 16,
	} , {
		.key	= "\xcc\xdd\xee\xff\x88\x99\xaa\xbb"
			  "\x44\x55\x66\x77\x00\x11\x22\x33"
			  "\xf3\xf2\xf1\xf0\xf7\xf6\xf5\xf4"
			  "\xfb\xfa\xf9\xf8\xff\xfe\xfd\xfc",
		.klen	= 32,
		.ptext	= "\x10\x32\x54\x76\x98\xba\xdc\xfe",
		.ctext	= "\x3d\xca\xd8\xc2\xe5\x01\xe9\x4e",
		.len	= 8,
	},
};

static const struct cipher_testvec gost28147_cpa_tv_template[] = {
	{
		.key	= "\x81\x82\x83\x84\x85\x86\x87\x88"
			  "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x80"
			  "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			  "\xd9\xda\xdb\xdc\xdd\xde\xdf\xd0",
		.klen	= 32,
		.ptext	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8",
		.ctext	= "\x66\x81\x84\xae\xdc\x48\xc9\x17"
			  "\x41\x64\x34\x70\x58\x84\x5c\xac",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpb_tv_template[] = {
	{
		.key	= "\x81\x82\x83\x84\x85\x86\x87\x88"
			  "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x80"
			  "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			  "\xd9\xda\xdb\xdc\xdd\xde\xdf\xd0",
		.klen	= 32,
		.ptext	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8",
		.ctext	= "\xdb\xee\x81\x14\x7b\x74\xb0\xf2"
			  "\xdb\x5e\xf0\x0e\xff\x4b\xd5\x28",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpc_tv_template[] = {
	{
		.key	= "\x81\x82\x83\x84\x85\x86\x87\x88"
			  "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x80"
			  "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			  "\xd9\xda\xdb\xdc\xdd\xde\xdf\xd0",
		.klen	= 32,
		.ptext	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8",
		.ctext	= "\x31\xa3\x85\x9d\x0a\xee\xb8\x0e"
			  "\x4a\xfb\xd6\xce\x77\x98\xff\xa9",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpd_tv_template[] = {
	{
		.key	= "\x81\x82\x83\x84\x85\x86\x87\x88"
			  "\x89\x8a\x8b\x8c\x8d\x8e\x8f\x80"
			  "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8"
			  "\xd9\xda\xdb\xdc\xdd\xde\xdf\xd0",
		.klen	= 32,
		.ptext	= "\x01\x02\x03\x04\x05\x06\x07\x08"
			  "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8",
		.ctext	= "\xb1\x32\x3e\x0b\x21\x73\xcb\xd1"
			  "\xc5\x28\x2f\x24\x61\xe9\x7a\xa8",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_tc26z_cfb_tv_template[] = {
	{
		.key	= "\x8d\x5a\x2c\x83\xa7\xc7\x0a\x61"
			  "\xd6\x1b\x34\xb5\x1f\xdf\x42\x68"
			  "\x66\x71\xa3\x5d\x87\x4c\xfd\x84"
			  "\x99\x36\x63\xb6\x1e\xd6\x0d\xad",
		.klen	= 32,
		.iv	= "\x46\x60\x6f\x0d\x88\x34\x23\x5a",
		.ptext	= "\xd2\xfd\xf8\x3a\xc1\xb4\x39\x23"
			  "\x2e\xaa\xcc\x98\x0a\x02\xda\x33",
		.ctext	= "\xed\xa7\xf1\x41\x01\x9c\xbd\xcd"
			  "\x44\x6b\x00\x96\x87\xf7\xc7\xe6",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpa_cfb_tv_template[] = {
	{
		.key	= "\x8d\x5a\x2c\x83\xa7\xc7\x0a\x61"
			  "\xd6\x1b\x34\xb5\x1f\xdf\x42\x68"
			  "\x66\x71\xa3\x5d\x87\x4c\xfd\x84"
			  "\x99\x36\x63\xb6\x1e\xd6\x0d\xad",
		.klen	= 32,
		.iv	= "\x46\x60\x6f\x0d\x88\x34\x23\x5a",
		.ptext	= "\xd2\xfd\xf8\x3a\xc1\xb4\x39\x23"
			  "\x2e\xaa\xcc\x98\x0a\x02\xda\x33",
		.ctext	= "\x88\xb7\x75\x16\x74\xa5\xee\x2d"
			  "\x14\xfe\x91\x67\xd0\x5c\xcc\x40",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpb_cfb_tv_template[] = {
	{
		.key	= "\x48\x0c\x74\x1b\x02\x6b\x55\xd5"
			  "\xb6\x6d\xd7\x1d\x40\x48\x05\x6b"
			  "\x6d\xeb\x3c\x29\x0f\x84\x80\x23"
			  "\xee\x0d\x47\x77\xe3\xfe\x61\xc9",
		.klen	= 32,
		.iv	= "\x1f\x3f\x82\x1e\x0d\xd8\x1e\x22",
		.ptext	= "\x8c\x9c\x44\x35\xfb\xe9\xa5\xa3"
			  "\xa0\xae\x28\x56\x91\x10\x8e\x1e"
			  "\xd2\xbb\x18\x53\x81\x27\x0d\xa6"
			  "\x68\x59\x36\xc5\x81\x62\x9a\x8e"
			  "\x7d\x50\xf1\x6f\x97\x62\x29\xec"
			  "\x80\x51\xe3\x7d\x6c\xc4\x07\x95"
			  "\x28\x63\xdc\xb4\xb9\x2d\xb8\x13"
			  "\xb1\x05\xb5\xf9\xeb\x75\x37\x4e"
			  "\xf7\xbf\x51\xf1\x98\x86\x43\xc4"
			  "\xe4\x3d\x3e\xa7\x62\xec\x41\x59"
			  "\xe0\xbd\xfb\xb6\xfd\xec\xe0\x77"
			  "\x13\xd2\x59\x90\xa1\xb8\x97\x6b"
			  "\x3d\x8b\x7d\xfc\x9d\xca\x82\x73"
			  "\x32\x70\x0a\x74\x03\xc6\x0c\x26"
			  "\x7f\x56\xf0\x9d\xb2\xeb\x71\x40"
			  "\xd7\xc3\xb1\xa7\xc5\x1e\x20\x17"
			  "\xb3\x50\x1d\x8a\x6e\x19\xcb\xbe"
			  "\x20\x86\x2b\xd6\x1c\xfd\xb4\xb7"
			  "\x5d\x9a\xb3\xe3\x7d\x15\x7a\x35"
			  "\x01\x9f\x5d\x65\x89\x4b\x34\xc6"
			  "\xf4\x81\x3f\x78\x30\xcf\xe9\x15"
			  "\x90\x9a\xf9\xde\xba\x63\xd0\x19"
			  "\x14\x66\x3c\xb9\xa4\xb2\x84\x94"
			  "\x02\xcf\xce\x20\xcf\x76\xe7\xc5"
			  "\x48\xf7\x69\x3a\x5d\xec\xaf\x41"
			  "\xa7\x12\x64\x83\xf5\x99\x1e\x9e"
			  "\xb2\xab\x86\x16\x00\x23\x8e\xe6"
			  "\xd9\x80\x0b\x6d\xc5\x93\xe2\x5c"
			  "\x8c\xd8\x5e\x5a\xae\x4a\x85\xfd"
			  "\x76\x01\xea\x30\xf3\x78\x34\x10"
			  "\x72\x51\xbc\x9f\x76\xce\x1f\xd4"
			  "\x8f\x33\x50\x34\xc7\x4d\x7b\xcf"
			  "\x91\x63\x7d\x82\x9e\xa1\x23\x45"
			  "\xf5\x45\xac\x98\x7a\x48\xff\x64"
			  "\xd5\x59\x47\xde\x2b\x3f\xfa\xec"
			  "\x50\xe0\x81\x60\x8b\xc3\xfc\x80"
			  "\x98\x17\xc7\xa3\xc2\x57\x3d\xab"
			  "\x91\x67\xf5\xc4\xab\x92\xc8\xd6"
			  "\x3b\x6b\x3f\xff\x15\x6b\xcf\x53"
			  "\x65\x02\xf1\x74\xca\xa9\xbe\x24"
			  "\xd2\xf0\xb7\x26\xa8\xd7\x6d\xed"
			  "\x90\x36\x7b\x3e\x41\xa9\x7f\xa3"
			  "\x1b\xf4\x43\xc5\x51\xbe\x28\x59"
			  "\xe9\x45\x26\x49\x38\x32\xf8\xf3"
			  "\x92\x6e\x30\xcc\xb0\xa0\xf9\x01"
			  "\x14\xc8\xba\xd9\xf0\x2a\x29\xe2"
			  "\x52\x9a\x76\x95\x3a\x16\x32\xec"
			  "\xf4\x10\xec\xee\x47\x00\x70\x19"
			  "\xe4\x72\x35\x66\x44\x53\x2d\xa2"
			  "\xf3\xaa\x7e\x8a\x33\x13\xcd\xc8"
			  "\xbf\x0e\x40\x90\x00\xe4\x42\xc3"
			  "\x09\x84\xe1\x66\x17\xa2\xaf\x03"
			  "\xab\x6b\xa1\xec\xfb\x17\x72\x81"
			  "\xfe\x9a\x9f\xf4\xb2\x33\x1f\xae"
			  "\x0c\xd1\x6a\xae\x19\xb8\xaf\xec"
			  "\xe3\xea\x00\xf8\xac\x87\x07\x5f"
			  "\x6d\xb0\xac\x6b\x22\x48\x36\xbf"
			  "\x22\x18\xb0\x03\x9f\x6c\x70\x45"
			  "\x36\xf0\x6b\xc6\xc2\xa5\x72\x2c"
			  "\xd8\xe0\x27\x3d\xec\x56\x07\x05"
			  "\x7d\x83\xa1\x65\x7d\x41\x5b\xcd"
			  "\x77\x24\xe5\xaa\x76\x47\xd0\x50"
			  "\xf6\xe7\xb5\x59\x75\x31\x27\xef"
			  "\xd8\xa6\x4e\x7f\xb8\x40\xb1\xdf"
			  "\x53\x14\xed\xf1\x68\x5f\xfc\x3f"
			  "\x02\xdb\x05\xeb\x31\xe4\x2c\x7f"
			  "\x32\xb5\x70\x8e\x75\x85\xa4\x5c"
			  "\x16\x23\x37\xf2\x10\x79\xcb\xdc"
			  "\xf8\x1c\x25\xc2\xa1\x3d\x9c\x33"
			  "\x6c\xed\xc3\xe7\xf3\x02\x87\x82"
			  "\x4e\xfb\xac\xb3\x2d\xfc\xf8\x0d"
			  "\x1d\x4a\x39\xd4\xb3\x09\xbb\xe9"
			  "\x25\xc7\xec\x6a\x87\x72\x84\xed"
			  "\x12\x60\x19\x64\xeb\x16\x2a\x5b"
			  "\x10\x76\x27\xff\x7b\xe4\xae\xe5"
			  "\xa4\x04\x02\x7f\xbb\x0a\xb5\xf4"
			  "\x05\xa5\x56\x1c\x53\x31\x7a\x93"
			  "\xba\x16\x15\xab\x62\x60\xfc\xde"
			  "\x72\x36\x6e\x28\xaf\x98\x0d\xe6"
			  "\xf4\xde\x60\xa7\x7e\x06\x07\x86"
			  "\xf3\x94\xb6\x6d\x0d\x93\xa6\xbc"
			  "\x60\x70\x33\xac\x3f\xa1\xa8\x4a"
			  "\x20\x61\xb6\xb5\x43\xa3\x15\x5a"
			  "\x00\xbe\x76\x98\x57\x72\xab\x7a"
			  "\x0e\x18\x93\x82\x3a\x18\x78\x6e"
			  "\x71\x7b\x78\x4f\x7e\x8c\xde\x7a"
			  "\x62\xb5\x0a\x7c\x45\x1d\x16\xd5"
			  "\xc3\x8c\x9b\x25\xb4\x50\x90\xcd"
			  "\x96\x93\xad\x0f\xd4\x43\xcb\x49"
			  "\x0f\xfc\x5a\x31\xf4\x19\xb7\xd4"
			  "\xeb\x4d\x40\x58\xd0\x3b\xc8\xe0"
			  "\x4a\x54\x2f\xdb\x22\xc3\x29\x7b"
			  "\x40\x90\x61\x43\xd3\x7e\xe2\x30"
			  "\x2b\x48\x3c\xce\x90\x93\xb1\x8b"
			  "\x31\x96\x65\x6d\x57\x8b\x9d\x4d"
			  "\x53\xf0\x83\x1c\xe5\xa1\x9d\x55"
			  "\xe3\xbf\x7e\xca\x1a\x74\x66\x14"
			  "\xcc\x47\x43\xd9\xbb\xef\x97\x7d"
			  "\xb7\x6e\xff\xf1\x22\xf8\x10\x2d"
			  "\x3f\xcd\x49\x96\xd9\x09\x11\xb8"
			  "\x33\xd0\x23\x9a\xfa\x16\xcb\x50"
			  "\x26\x57\x24\x5c\x0e\xba\xf0\x3f"
			  "\x37\x2f\xa3\xf7\x18\x57\x48\x48"
			  "\x95\xcf\xef\x87\x67\x2a\xe9\xb6"
			  "\x8a\x21\x36\x7f\xff\x48\x6c\x46"
			  "\x35\x57\xf2\xbc\x48\x67\x8f\x63"
			  "\x23\x78\x11\x2b\xc2\x08\xde\x51"
			  "\xe8\x8b\x92\x29\xf9\x9a\x9e\xad"
			  "\xed\x0f\xeb\xa2\xd2\x40\x92\xd4"
			  "\xde\x62\x95\x76\xfd\x6e\x3c\xbf"
			  "\xc0\xd7\x0d\xe5\x1b\xa4\xc7\x18"
			  "\xe1\x58\xa4\x56\xef\x2e\x17\x1b"
			  "\x75\xcb\xbc\xf9\x2a\x95\x71\xa7"
			  "\x1d\x7f\xe7\x73\x63\x05\x6b\x19"
			  "\x4c\xf4\x22\x14\xc4\x59\x88\x66"
			  "\x92\x86\x61\x5c\x6a\xae\xec\x58"
			  "\xff\xc9\xf2\x44\xd4\xa2\xf5\x98"
			  "\xeb\x5f\x09\xbc\x8a\xbf\x3c\xb4"
			  "\x3e\xb1\x20\x05\x44\x96\x79\x0a"
			  "\x40\x92\x7f\x9d\xd1\xaf\xbc\x90"
			  "\x95\x0a\x81\xd4\xa7\xc6\xb8\xe0"
			  "\xe4\x39\x30\x1d\x79\xc0\xe5\xfa"
			  "\xb4\xe9\x63\xb4\x09\x72\x3b\x3e"
			  "\xd9\xf6\xd9\x10\x21\x18\x7e\xe5"
			  "\xad\x81\xd7\xd5\x82\xd0\x8c\x3b"
			  "\x38\x95\xf8\x92\x01\xa9\x92\x00"
			  "\x70\xd1\xa7\x88\x77\x1f\x3a\xeb"
			  "\xb5\xe4\xf5\x9d\xc7\x37\x86\xb2"
			  "\x12\x46\x34\x19\x72\x8c\xf5\x8c"
			  "\xf6\x78\x98\xe0\x7c\xd3\xf4",
		.ctext	= "\x23\xc6\x7f\x20\xa1\x23\x58\xbc"
			  "\x7b\x05\xdb\x21\x15\xcf\x96\x41"
			  "\xc7\x88\xef\x76\x5c\x49\xdb\x42"
			  "\xbf\xf3\xc0\xf5\xbd\x5d\xd9\x8e"
			  "\xaf\x3d\xf4\xe4\xda\x88\xbd\xbc"
			  "\x47\x5d\x76\x07\xc9\x5f\x54\x1d"
			  "\x1d\x6a\xa1\x2e\x18\xd6\x60\x84"
			  "\x02\x18\x37\x92\x92\x15\xab\x21"
			  "\xee\x21\xcc\x71\x6e\x51\xd9\x2b"
			  "\xcc\x81\x97\x3f\xeb\x45\x99\xb8"
			  "\x1b\xda\xff\x90\xd3\x41\x06\x9c"
			  "\x3f\xfb\xe4\xb2\xdc\xc9\x03\x0d"
			  "\xa7\xae\xd7\x7d\x02\xb8\x32\xab"
			  "\xf3\x65\xa3\x65\x6c\x4e\xe4\xa2"
			  "\x5e\x9e\xee\xcd\xde\x79\x36\x6b"
			  "\x1b\xe1\x3c\xdf\x10\xad\x4f\x02"
			  "\xe1\x14\xaa\x09\xb4\x0b\x76\xeb"
			  "\x69\x38\x20\x02\xcb\x8e\xc0\xdf"
			  "\xca\x48\x74\xc3\x31\xad\x42\x2c"
			  "\x51\x9b\xd0\x6a\xc1\x36\xd7\x21"
			  "\xdf\xb0\x45\xba\xca\x7f\x35\x20"
			  "\x28\xbb\xc1\x76\xfd\x43\x5d\x23"
			  "\x7d\x31\x84\x1a\x97\x4d\x83\xaa"
			  "\x7e\xf1\xc4\xe6\x83\xac\x0d\xef"
			  "\xef\x3c\xa4\x7c\x48\xe4\xc8\xca"
			  "\x0d\x7d\xea\x7c\x45\xd7\x73\x50"
			  "\x25\x1d\x01\xc4\x02\x1a\xcd\xe0"
			  "\x38\x5b\xa8\x5a\x16\x9a\x10\x59"
			  "\x74\xd7\x19\xc6\xf3\xb5\x17\xf6"
			  "\x59\x8d\x62\xaf\x44\xe8\xdc\xe9"
			  "\xc1\x76\xf1\xd0\xbd\x29\xd7\xec"
			  "\x1d\xac\x57\xdb\x1a\x3f\xd8\xf6"
			  "\x6e\xb6\xe6\xdf\x36\xe7\x89\xce"
			  "\x56\x35\x43\x1c\x7d\x57\x79\x0e"
			  "\xd8\xf4\xd7\xa7\x0d\xc6\x8f\x91"
			  "\x66\x67\x82\x0f\x49\xc9\xc5\x65"
			  "\x81\xa1\x39\x5a\x53\x9f\x02\xa5"
			  "\xd5\x36\x22\xa8\xa8\x1c\x37\x0e"
			  "\x76\x46\xdf\xbd\x6a\xdb\xfc\x1b"
			  "\xbd\x10\xb8\xb1\xbc\x72\x4c\x58"
			  "\x4a\xda\x6d\x66\x00\xda\x7a\x66"
			  "\xa0\xe7\x3b\x39\xa3\xf7\x05\x07"
			  "\xfa\x21\x4b\xc7\x94\xc0\xd3\x7b"
			  "\x19\x02\x5d\x4a\x10\xf1\xc2\x0f"
			  "\x19\x68\x27\xc7\x7d\xbf\x55\x03"
			  "\x57\x7d\xaf\x77\xae\x80\x2f\x7a"
			  "\xe6\x1f\x4b\xdc\x15\x18\xc0\x62"
			  "\xa1\xe8\xd9\x1c\x9e\x8c\x96\x39"
			  "\xc1\xc4\x88\xf7\x0c\xe1\x04\x84"
			  "\x68\x51\xce\xf1\x90\xda\x7f\x76"
			  "\xc8\xc0\x88\xef\x8e\x15\x25\x3e"
			  "\x7b\xe4\x79\xb5\x66\x2d\x9c\xd1"
			  "\x13\xda\xd0\xd5\x46\xd5\x8d\x46"
			  "\x18\x07\xee\xd8\xc9\x64\xe3\xbe"
			  "\x0e\x68\x27\x09\x96\x26\xf6\xe2"
			  "\x19\x61\x3f\xf4\x58\x27\x0a\xeb"
			  "\xce\x7c\xb6\x68\x92\xe7\x12\x3b"
			  "\x31\xd4\x48\xdf\x35\x8d\xf4\x86"
			  "\x42\x2a\x15\x4b\xe8\x19\x1f\x26"
			  "\x65\x9b\xa8\xda\x4b\x79\x1f\x8e"
			  "\xe6\x13\x7e\x49\x8f\xc1\xce\xdc"
			  "\x5e\x64\x74\xce\x02\x78\xe0\xcf"
			  "\xa0\xed\x5e\x31\x74\xd1\xd0\xb4"
			  "\xee\x70\x19\x14\x3c\x8f\x16\xa6"
			  "\xcf\x12\x93\x15\x88\xeb\x91\x65"
			  "\x76\x98\xfd\xa1\x94\x30\xba\x43"
			  "\x62\x65\x40\x04\x77\x9e\xd6\xab"
			  "\x8b\x0d\x93\x80\x50\x5f\xa2\x76"
			  "\x20\xa7\xd6\x9c\x27\x15\x27\xbc"
			  "\xa5\x5a\xbf\xe9\x92\x82\x05\xa8"
			  "\x41\xe9\xb5\x60\xd5\xc0\xd7\x4b"
			  "\xad\x38\xb2\xe9\xd1\xe5\x51\x5f"
			  "\x24\x78\x24\x9a\x23\xd2\xc2\x48"
			  "\xbd\x0e\xf1\x37\x72\x91\x87\xb0"
			  "\x4e\xbd\x99\x6b\x2c\x01\xb6\x79"
			  "\x69\xec\x0c\xed\xe5\x3f\x50\x64"
			  "\x7c\xb9\xdd\xe1\x92\x81\xb5\xd0"
			  "\xcb\x17\x83\x86\x8b\xea\x4f\x93"
			  "\x08\xbc\x22\x0c\xef\xe8\x0d\xf5"
			  "\x9e\x23\xe1\xf9\xb7\x6b\x45\x0b"
			  "\xcb\xa9\xb6\x4d\x28\x25\xba\x3e"
			  "\x86\xf2\x75\x47\x5d\x9d\x6b\xf6"
			  "\x8a\x05\x58\x73\x3d\x00\xde\xfd"
			  "\x69\xb1\x61\x16\xf5\x2e\xb0\x9f"
			  "\x31\x6a\x00\xb9\xef\x71\x63\x47"
			  "\xa3\xca\xe0\x40\xa8\x7e\x02\x04"
			  "\xfe\xe5\xce\x48\x73\xe3\x94\xcf"
			  "\xe2\xff\x29\x7e\xf6\x32\xbb\xb7"
			  "\x55\x12\x21\x7a\x9c\x75\x04\x0c"
			  "\xb4\x7c\xb0\x3d\x40\xb3\x11\x9a"
			  "\x7a\x9a\x13\xfb\x77\xa7\x51\x68"
			  "\xf7\x05\x47\x3b\x0f\x52\x5c\xe6"
			  "\xc2\x99\x3a\x37\x54\x5c\x4f\x2b"
			  "\xa7\x01\x08\x74\xbc\x91\xe3\xe2"
			  "\xfe\x65\x94\xfd\x3d\x18\xe0\xf0"
			  "\x62\xed\xc2\x10\x82\x9c\x58\x7f"
			  "\xb2\xa3\x87\x8a\x74\xd9\xc1\xfb"
			  "\x84\x28\x17\xc7\x2b\xcb\x53\x1f"
			  "\x4e\x8a\x82\xfc\xb4\x3f\xc1\x47"
			  "\x25\xf3\x21\xdc\x4c\x2d\x08\xfa"
			  "\xe7\x0f\x03\xa9\x68\xde\x6b\x41"
			  "\xa0\xf9\x41\x6c\x57\x4d\x3a\x0e"
			  "\xea\x51\xca\x9f\x97\x11\x7d\xf6"
			  "\x8e\x88\x63\x67\xc9\x65\x13\xca"
			  "\x38\xed\x35\xbe\xf4\x27\xa9\xfc"
			  "\xa9\xe6\xc3\x40\x86\x08\x39\x72"
			  "\x37\xee\xb2\x87\x09\x96\xb7\x40"
			  "\x87\x36\x92\xc1\x5d\x6a\x2c\x43"
			  "\xca\x25\xc8\x35\x37\x2d\xb5\xa9"
			  "\x27\x44\x50\xf2\x6d\x22\x75\x41"
			  "\x77\x2a\xdb\xb1\x8c\x6d\x05\xe8"
			  "\xc9\x99\xc7\x08\xf9\x14\x8f\x78"
			  "\xa9\x8f\xc2\x5a\x7a\x65\xc5\xd8"
			  "\x86\xbb\x72\x69\x6b\x6b\x45\x83"
			  "\x5b\xb1\xf7\xcd\x16\x73\xee\xe9"
			  "\x80\x85\xfe\x8e\xe1\xae\x53\x8f"
			  "\xde\xbe\x48\x8b\x59\xef\xf6\x7e"
			  "\xd8\xb5\xa8\x47\xc0\x4e\x15\x58"
			  "\xca\xd3\x2f\xf8\x6c\xa6\x3d\x78"
			  "\x4d\x7a\x54\xd6\x10\xe5\xcc\x05"
			  "\xe2\x29\xb5\x86\x07\x39\x7d\x78"
			  "\x8e\x5a\x8f\x83\x4c\xe7\x3d\x68"
			  "\x3e\xe5\x02\xe6\x64\x4f\x5e\xb4"
			  "\x49\x77\xf0\xc0\xfa\x6f\xc8\xfb"
			  "\x9f\x84\x6f\x55\xfb\x30\x5e\x89"
			  "\x93\xa9\xf3\xa6\xa3\xd7\x26\xbb"
			  "\xd8\xa8\xd9\x95\x1d\xfe\xfc\xd7"
			  "\xa8\x93\x66\x2f\x04\x53\x06\x64"
			  "\x7f\x31\x29\xae\xb7\x9f\xba\xc4"
			  "\x6d\x68\xd1\x24\x32\xf4\x11",
		.len = 1039,
	},
};

static const struct cipher_testvec gost28147_cpc_cfb_tv_template[] = {
	{
		.key	= "\x77\xc3\x45\x8e\xf6\x42\xe7\x04"
			  "\x8e\xfc\x08\xe4\x70\x96\xd6\x05"
			  "\x93\x59\x02\x6d\x6f\x97\xca\xe9"
			  "\xcf\x89\x44\x4b\xde\x6c\x22\x1d",
		.klen	= 32,
		.iv	= "\x43\x7c\x3e\x8e\x2f\x2a\x00\x98",
		.ptext	= "\x07\x9c\x91\xbe",
		.ctext	= "\x19\x35\x81\x34",
		.len	= 4,
	},
	{
		.key	= "\x8d\x5a\x2c\x83\xa7\xc7\x0a\x61"
			  "\xd6\x1b\x34\xb5\x1f\xdf\x42\x68"
			  "\x66\x71\xa3\x5d\x87\x4c\xfd\x84"
			  "\x99\x36\x63\xb6\x1e\xd6\x0d\xad",
		.klen	= 32,
		.iv	= "\x46\x60\x6f\x0d\x88\x34\x23\x5a",
		.ptext	= "\xd2\xfd\xf8\x3a\xc1\xb4\x39\x23"
			  "\x2e\xaa\xcc\x98\x0a\x02\xda\x33",
		.ctext	= "\x53\xec\xdf\xfc\x37\x19\x53\xcd"
			  "\x88\x2f\xd0\x06\xe9\xa3\x82\x84",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpd_cfb_tv_template[] = {
	{
		.key	= "\x38\x9f\xe8\x37\xff\x9c\x5d\x29"
			  "\xfc\x48\x55\xa0\x87\xea\xe8\x40"
			  "\x20\x87\x5b\xb2\x01\x15\x55\xa7"
			  "\xe3\x2d\xcb\x3d\xd6\x59\x04\x73",
		.klen	= 32,
		.iv	= "\xc5\xa2\xd2\x1f\x2f\xdf\xb8\xeb",
		.ptext	= "\x2f\x31\xd8\x83\xb4\x20\xe8\x6e"
			  "\xda",
		.ctext	= "\x6d\xa4\xed\x40\x08\x88\x71\xad"
			  "\x16",
		.len	= 9,
	},
	{
		.key	= "\x8d\x5a\x2c\x83\xa7\xc7\x0a\x61"
			  "\xd6\x1b\x34\xb5\x1f\xdf\x42\x68"
			  "\x66\x71\xa3\x5d\x87\x4c\xfd\x84"
			  "\x99\x36\x63\xb6\x1e\xd6\x0d\xad",
		.klen	= 32,
		.iv	= "\x46\x60\x6f\x0d\x88\x34\x23\x5a",
		.ptext	= "\xd2\xfd\xf8\x3a\xc1\xb4\x39\x23"
			  "\x2e\xaa\xcc\x98\x0a\x02\xda\x33",
		.ctext	= "\xfe\xbd\x9b\x38\x0d\x72\xfe\x1e"
			  "\x61\x93\x52\x7b\xc6\xbd\x4f\xb7",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_tc26z_cnt_tv_template[] = {
	{
		.key	= "\x59\x9f\x84\xba\xc3\xf3\xd2\xf1"
			  "\x60\xe1\xe3\xf2\x6a\x96\x1a\xf9"
			  "\x9c\x48\xb2\x4e\xbc\xbb\xbf\x7c"
			  "\xd8\xf3\xac\xcd\x96\x8d\x28\x6a",
		.klen	= 32,
		.iv	= "\x8d\xaf\xa8\xd1\x58\xed\x05\x8d",
		.ptext	= "\x90\xa2\x39\x66\xae\x01\xb9\xa3"
			  "\x52\x4e\xc8\xed\x6c\xdd\x88\x30",
		.ctext	= "\xe8\xb1\x4f\xc7\x30\xdc\x25\xbb"
			  "\x36\xba\x64\x3c\x17\xdb\xff\x99",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpa_cnt_tv_template[] = {
	{
		.key	= "\x59\x9f\x84\xba\xc3\xf3\xd2\xf1"
			  "\x60\xe1\xe3\xf2\x6a\x96\x1a\xf9"
			  "\x9c\x48\xb2\x4e\xbc\xbb\xbf\x7c"
			  "\xd8\xf3\xac\xcd\x96\x8d\x28\x6a",
		.klen	= 32,
		.iv	= "\x8d\xaf\xa8\xd1\x58\xed\x05\x8d",
		.ptext	= "\x90\xa2\x39\x66\xae\x01\xb9\xa3"
			  "\x52\x4e\xc8\xed\x6c\xdd\x88\x30",
		.ctext	= "\x6e\x72\x62\xcc\xe3\x59\x36\x90"
			  "\x83\x3a\xfe\xa9\x1b\xc9\xbe\xce",
		.len	= 16,
	},
	{
		.key	= "\x1b\x5d\xdb\x77\xcf\xf9\xec\x95"
			  "\x5e\xcc\x67\x9f\x5d\x28\xad\x4a"
			  "\x27\xf4\x32\xc6\xb2\xcb\xb1\x45"
			  "\x6a\x88\x14\x0c\x9b\x9b\x5f\x48",
		.klen	= 32,
		.iv	= "\x71\x58\x8c\xe1\x55\xf4\xf6\xb3",
		.ptext	= "\x3d\x0b\x69\xf7\xa8\xe4\xfc\x99"
			  "\x22\x2e\xee\xd1\x63\x12\xfe\xa8"
			  "\x9d\xcb\x6c\x4d\x48\x8c\xe8\xbd"
			  "\x8b\x60\xf1\xbf\x7b\xe3\x79\xd5"
			  "\x2b\x25\x97\x13\xef\x35\xda\xf4"
			  "\xbc\x77\xce\xea\xe9\x3f\xa4\xb6"
			  "\x01\xd5\x73\x29\x58\xda\xd7\x67"
			  "\x17\xac\xe4\x75\x2f\x57\x23\xac"
			  "\x96\x21\xc7\x62\x2d\xf7\x32\xb5"
			  "\x44\x5f\x72\xb1\x5f\xba\x1b\x1e"
			  "\xdb\x4a\x09\x8c\x92\x61\xa2\xb0"
			  "\x49\x68\xe5\xb3\xa2\x8f\x13\x4b"
			  "\xf5\x4d\x84\xda\xab\xa0\xb6\xd1"
			  "\x5a\x63\x19\xe8\xa2\x09\xf6\x76"
			  "\x6f\x9b\x48\x0a\x15\x5d\xb7\x20"
			  "\x21\x9a\x2e\xb9\x6d\xfa\x1e\xc2"
			  "\x0e\xef\x15\xab\x59\x01\xfe\x43"
			  "\x90\xf2\x62\xca\x4a\x9a\x48\x38"
			  "\xab\x6f\x9d\x21\xb3\xad\xa7\x60"
			  "\x46\xe3\xef\xd0\xe3\x1d\xc5\xe1"
			  "\xb8\xa1\xe2\x99\x20\xc5\x76\xcc"
			  "\xaa\x8a\xa9\x45\x55\xa0\x78\x00"
			  "\x64\xde\xcf\x5b\xdf\x26\x48\xcd"
			  "\xba\x8a\xb5\xfb\xfd\x4a\xd5\xc4"
			  "\xe0\x43\xa6\x71\x90\xa4\x8b\xca"
			  "\x2e\x88\x7b\xac\xb2\xdc\xf2\x01"
			  "\xcb\xda\x6e\x91\x27\x28\x44\x88"
			  "\x9a\xd2\x12\xf1\xa6\xf5\xb7\x61"
			  "\xce\x79\x62\x52\x3c\xe6\x14\x73"
			  "\xd1\x41\x92\x50\xbd\xdc\x3b\xd0"
			  "\xa7\x11\x8c\x3a\xe4\x2d\xf2\x52"
			  "\xd3\x2f\x7c\x8e\x54\x90\x4e\x23"
			  "\xae\xb3\xa0\xf3\x25\x7e\x66\xaa"
			  "\x0f\x6f\x81\x72\x77\xbb\xd3\x47"
			  "\xe8\x05\xff\xe1\x5b\xc9\x37\x50"
			  "\x33\x49\x17\xaf\xab\x1d\xe1\x15"
			  "\xf2\xe5\x98\x5e\x2d\x05\x1f\x0d"
			  "\x55\x97\xed\xff\x5e\xe0\x0f\xc3"
			  "\x9c\xbd\x82\xc2\x06\xbe\x45\x66"
			  "\xae\x33\xbe\x28\x48\xe9\x2d\x1a"
			  "\xe6\x65\x8e\xdf\x76\x03\x73\x4b"
			  "\xc0\x80\x71\xf9\xac\xba\xa0\xb0"
			  "\x19\x1a\x0a\xd4\x35\x12\x88\x76"
			  "\x05\x75\x8f\x7c\xb5\xf0\x19\x75"
			  "\x6d\x05\xcb\x0d\xbc\x8d\xe9\xf0"
			  "\xd4\xdb\x3c\x3c\x29\x8e\x2c\x32"
			  "\x1d\xf7\xb6\x49\xcf\xdb\x63\xee"
			  "\x3c\xfa\x33\x73\x6f\xe4\x97\x4e"
			  "\x2f\xc9\x4c\x5c\x65\xfe\xea\xfb"
			  "\xc6\xdd\xc1\x1c\x47\x3f\xf4\x50"
			  "\x2f\xde\x1b\x5b\x0b\x16\xca\xb6"
			  "\x46\x44\xf2\xc1\x0d\xa1\x1d\xa6"
			  "\xdb\xf0\x3d\xb1\x6c\x05\x31\x85"
			  "\x8e\x74\xae\xf2\x39\x26\xf7\xc1"
			  "\xe7\x4c\xdd\x9d\x40\xb8\xf3\xc5"
			  "\xc2\x16\x64\x6b\xaa\xdb\x4b\x82"
			  "\x5c\xd3\x02\xd3\x8f\x26\x79\x8d"
			  "\xb0\x78\x70\x19\x58\x0c\xb4\x31"
			  "\x88\x44\x1c\x91\x6f\xf4\x52\x39"
			  "\xa8\xf5\xc0\x1b\xfe\xf2\x0e\x4b"
			  "\xac\x0a\xc2\x7e\x9c\x9b\xeb\x5d"
			  "\x4e\x4f\x42\xd8\x71\x0a\x97\x27"
			  "\x03\x14\x96\xa6\x3d\x04\xea\x9f"
			  "\x14\x14\x27\x4c\xd9\xa2\x89\x5f"
			  "\x65\x4a\xe1\x9d\x2c\xb8\xf8\xd4"
			  "\x8f\x2a\x57\x36\xcc\x06\x9c\x2c"
			  "\xc5\x13\x16\xdf\xfc\xae\x22\x16"
			  "\xa8\x2b\x71\x6f\x1d\xb3\x47\x54"
			  "\x3f\x2d\x0a\x68\x9f\x2e\xf6\x90"
			  "\xd8\xa1\x21\x09\xd4\x97\xb9\x7b"
			  "\x7f\x9b\x6a\xed\xd1\xf0\xe3\xb6"
			  "\x28\xc7\x62\x82\x00\xc9\x38\xa1"
			  "\x82\x78\xce\x87\xc8\x53\xac\x4f"
			  "\x2e\x31\xb9\x50\x7f\x36\x00\x4a"
			  "\x32\xe6\xd8\xbb\x59\x45\x0e\x91"
			  "\x1b\x38\xa9\xbc\xb9\x5e\x6c\x6a"
			  "\x9c\x03\x01\x1c\xde\xe8\x1f\x1e"
			  "\xe3\xde\x25\xa2\x56\x79\xe1\xbd"
			  "\x58\xc4\x93\xe6\xd0\x8a\x4d\x08"
			  "\xab\xf7\xaa\xc3\x7d\xc1\xee\x68"
			  "\x37\xbc\x78\x0b\x19\x68\x2b\x2b"
			  "\x2e\x6d\xc4\x6f\xaa\x3b\xc6\x19"
			  "\xcb\xf1\x58\xb9\x60\x85\x45\xae"
			  "\x52\x97\xba\x24\x32\x13\x72\x16"
			  "\x6e\x7b\xc1\x98\xac\xb1\xed\xb4"
			  "\xcc\x6c\xcf\x45\xfc\x50\x89\x80"
			  "\x8e\x7a\xa4\xd3\x64\x50\x63\x37"
			  "\xc9\x6c\xf1\xc4\x3d\xfb\xde\x5a"
			  "\x5c\xa8\x21\x35\xe6\x2e\x8c\x2a"
			  "\x3c\x12\x17\x79\x9a\x0d\x2e\x79"
			  "\xeb\x67\x1f\x2b\xf8\x6e\xca\xc1"
			  "\xfa\x45\x18\x9e\xdf\x6a\xe6\xcb"
			  "\xe9\x5c\xc3\x09\xaf\x93\x58\x13"
			  "\xbf\x90\x84\x87\x75\xd6\x82\x28"
			  "\x8d\xe7\x2f\xa3\xfb\x97\x74\x2a"
			  "\x73\x04\x82\x06\x76\x69\xb1\x0b"
			  "\x19\xfc\xae\xb3\xdd\x2a\xe5\xc1"
			  "\x05\xd8\x80\x95\x22\x90\x71\xfc"
			  "\xc2\x92\x42\xfd\xf1\x70\xb4\x68"
			  "\x88\xa4\x9e\x0a\x24\x40\x13\xc8"
			  "\xa2\x56\x4f\x39\xe6\x06\xf1\xdc"
			  "\xf5\x13\x0e\xad\x9c\x8b\xaf\xe9"
			  "\xe3\x88\x72\xff\xa0\x6d\xda\x08"
			  "\x70\xb9\x2e\x83\xc5\xbb\x32\xa5"
			  "\x74\xc7\xfb\x7b\x76\xaf\x02\xbb"
			  "\x2b\xb8\x5e\x65\x02\xfe\x0e\xa0"
			  "\x99\xce\x01\x3b\x35\xe1\xb0\x22"
			  "\xe5\x94\xbd\xdd\x8e\xbb\xf6\x75"
			  "\xbf\xbf\xee\x7a\xb1\x58\xb4\x81"
			  "\xb8\x39\x3e\xb6\x1e\xde\xda\x1b"
			  "\xd5\xf7\xdd\x7d\x65\x9c\xaa\x56"
			  "\x93\xb8\xaf\x48\x53\xc7\x22\xe4"
			  "\x1c\xdf\xe9\x79\xb4\x20\x89\xcc"
			  "\x2a\x79\x2c\x09\xbe\x78\xcf\xcc"
			  "\xf2\x90\xd6\x65\xc5\x29\xfc\xda"
			  "\x69\xfc\xc0\xd6\x70\x99\x61\x3f"
			  "\x60\x02\xd8\x12\x22\xc8\x34\xc6"
			  "\x3b\xb3\xc2\x33\xa1\x5c\x8f\x4c"
			  "\xd1\x52\x72\xf2\x42\x05\x8e\x18"
			  "\x1f\x16\xda\xb8\x53\xa1\x5f\x01"
			  "\x32\x1b\x90\xb3\x53\x9b\xd0\x85"
			  "\x61\x2d\x17\xed\x0a\xa4\xa5\x27"
			  "\x09\x75\x7c\xbc\x30\xf7\x5e\x59"
			  "\x9a\x07\x96\x84\x28\x86\x4b\xa7"
			  "\x22\x35\x28\xc7\xed\x0d\xc3\xce"
			  "\x98\xcc\x2d\xec\xd4\x98\x09\x8e"
			  "\x52\x5f\x2b\x9a\x13\xbe\x99\x16"
			  "\x73\xd1\x1f\x81\xe5\xa2\x08\x78"
			  "\xcb\x0c\x20\xd4\xa5\xea\x4b\x5b"
			  "\x95\x5a\x92\x9a\x52",
		.ctext	= "\x8e\xcd\x8f\xc8\xac\xe1\x15\x48"
			  "\x2d\xae\x24\x8a\xc7\xfb\xba\x0f"
			  "\x1d\x8a\x95\xa2\x43\xef\xcb\xdc"
			  "\x59\x57\xa7\xc7\x0e\xe3\xe2\xb9"
			  "\x0d\x86\x29\x62\xcb\x83\x4d\x07"
			  "\x0c\x40\xd4\x7b\x2e\xca\xba\xbf"
			  "\x4a\x60\x3b\x31\x98\xc8\x88\x47"
			  "\xd9\x82\xab\xfc\x8f\x48\xe2\x46"
			  "\xab\xd3\xa1\xab\x8a\x05\x22\x8c"
			  "\xf4\xec\x9a\x1e\x76\xab\x1a\x60"
			  "\xd9\x25\x6b\xb8\x56\xe5\xb2\xea"
			  "\x10\xf3\x62\x04\x32\x5e\xaa\x3b"
			  "\x7b\x57\xbc\x3b\x8b\x43\x47\xf2"
			  "\xd5\x03\x7e\x51\x01\xff\x77\x28"
			  "\xca\x90\xa3\xfe\x7e\x2e\x70\x16"
			  "\x75\x18\x44\xf0\x1b\x85\x05\xea"
			  "\xe3\x21\xf7\x26\x86\x76\x3c\x67"
			  "\x9d\xfc\xbc\x10\x7f\x77\xe4\xed"
			  "\xd3\x12\xf8\x83\x00\x1f\x4b\x92"
			  "\x95\x92\x5c\xf3\x5a\xf3\xb7\xd0"
			  "\xa9\x5f\xf2\x18\xc4\x66\x62\xc1"
			  "\x84\x0e\x66\xe8\x80\x7d\x1f\xf0"
			  "\xba\x01\x9b\x71\xae\x93\xcc\x27"
			  "\x54\x34\x9a\xbd\xca\xee\x52\x09"
			  "\x92\x9d\xb0\xd5\xd9\xba\x2f\xb9"
			  "\x96\xdc\xfa\xbd\xce\xea\x1a\x7b"
			  "\x9a\x1d\x13\xa7\x11\xe2\x9a\x64"
			  "\xf6\xd3\xee\xc6\x33\xb7\x6e\xef"
			  "\x25\x9e\x1e\x7c\xe3\x1f\x2c\x6e"
			  "\xa9\xc0\xf8\xc1\xbf\x3b\xf8\x34"
			  "\x03\x9b\xa1\x40\x5b\x0c\x3c\x09"
			  "\x66\x9d\x63\xe2\xe2\x04\x8f\x06"
			  "\x84\x74\x68\xb2\x5c\x3b\x4c\xad"
			  "\x0b\x3f\x03\xb3\x07\x8a\x64\xa7"
			  "\x36\x56\x26\x39\x66\xda\xe9\x6d"
			  "\x1b\xd5\x88\xe8\x5c\xaf\x5a\x4c"
			  "\x49\xf7\xf5\xb7\x78\xf0\xde\xec"
			  "\xcd\x16\x23\x9e\x8c\x13\xbe\x6b"
			  "\x6f\x9b\x07\xe5\xbb\xcc\x3a\x1b"
			  "\x6f\x43\xdf\xff\x46\x2a\xae\x47"
			  "\x19\x18\x9a\x25\x09\xc9\x24\x40"
			  "\x0c\x4b\xa7\xda\x5e\x0d\xee\xfa"
			  "\x62\x45\x8e\xcc\x2f\x23\x08\x1d"
			  "\x92\xf0\xfe\x82\x0f\xd7\x11\x60"
			  "\x7e\x0b\x0b\x75\xf4\xf5\x3b\xc0"
			  "\xa4\xe8\x72\xa5\xb6\xfa\x5a\xad"
			  "\x5a\x4f\x39\xb5\xa2\x12\x96\x0a"
			  "\x32\x84\xb2\xa1\x06\x68\x56\x57"
			  "\x97\xa3\x7b\x22\x61\x76\x5d\x30"
			  "\x1a\x31\xab\x99\x06\xc5\x1a\x96"
			  "\xcf\xcf\x14\xff\xb2\xc4\xcc\x2b"
			  "\xbf\x0c\x9d\x91\x8f\x79\x5b\xbc"
			  "\xa9\x6b\x91\x6a\xb4\x93\x5c\x7b"
			  "\x5d\xc2\x8a\x75\xc0\xc1\x08\xfa"
			  "\x99\xf9\x4d\x5e\x0c\x06\x64\x60"
			  "\xa9\x01\x4a\x34\x0f\x33\x84\x95"
			  "\x69\x30\xc1\x1c\x36\xf8\xfc\x30"
			  "\x23\xb2\x71\xe5\x52\x4d\x12\x1a"
			  "\xc9\xbe\xee\xc9\xcb\x01\x85\xf3"
			  "\xdb\x30\xf9\x41\xa9\x40\xb0\x06"
			  "\x29\x77\xcd\xc5\xec\x58\x02\x48"
			  "\x83\x53\x44\x6a\xd2\xca\x05\xd8"
			  "\x5a\x08\xeb\xa9\xf4\xe6\xc7\x9d"
			  "\xd5\x7b\x74\x0b\x31\xb7\xa5\x57"
			  "\x7c\x7a\xfd\x1a\x0e\xd7\x97\x41"
			  "\xbf\xdd\xc6\x19\x6c\x77\x8c\x18"
			  "\x52\x57\x83\xba\x71\x25\xee\x39"
			  "\xbb\xe2\x43\xa0\x14\xdc\x0e\x84"
			  "\xb4\x2b\xde\x3e\xe5\x36\xb7\xa2"
			  "\x92\x98\x05\xb8\x96\xe5\xd0\x8c"
			  "\x08\x93\x35\xc2\x81\xe0\xfc\x59"
			  "\x71\xe2\x44\x49\x5d\xda\xfb\x9c"
			  "\xaa\x70\x9f\x43\xa8\xa5\xd9\x67"
			  "\xd9\x8f\xa3\x1e\xbe\x0e\xec\xdf"
			  "\x12\x2b\x6a\xe7\x1c\x12\x17\xe7"
			  "\xc4\x6d\x50\xc9\x52\x7a\xd5\xe8"
			  "\x7f\xbc\x07\x15\xac\xdb\x93\x66"
			  "\xb1\xf0\xa7\x7b\x2f\xe9\xec\xd0"
			  "\x47\x69\x59\x87\xf1\x4c\x3e\x4b"
			  "\x9b\x11\x79\x13\xe4\x96\xf6\x56"
			  "\x04\x6e\x0b\x33\xfc\x40\xf6\xc7"
			  "\xc1\x43\xb1\xbf\x0e\xb3\x87\xfd"
			  "\x0b\x1c\x63\x46\x3a\xd3\xa0\x17"
			  "\x59\x25\x94\x6c\x9c\x3d\x0c\x81"
			  "\xce\x82\x72\x42\x28\xf9\x37\x6a"
			  "\x6d\xe4\x12\xf4\x21\xaa\xf7\xfe"
			  "\x27\x55\x40\x1a\x14\xc3\x39\x5b"
			  "\xbf\x63\xc2\x5f\x10\x1f\x14\x25"
			  "\xd0\xce\xf3\x14\x48\x13\xa5\x0b"
			  "\x4d\x38\xcf\x0d\x34\xc0\x0a\x11"
			  "\xb4\xb5\x72\xc8\x4b\xc2\x6f\xe7"
			  "\x9d\x93\xf7\xdf\xb8\x43\x72\x7e"
			  "\xda\x3e\x20\x1f\xbc\x21\x2a\xce"
			  "\x00\xfa\x96\x9f\x3d\xe5\x88\x96"
			  "\xef\x29\x84\xdf\x6c\x1c\x96\xd8"
			  "\x58\x47\xaa\x92\xf3\x07\xe5\xfb"
			  "\xaf\xea\x95\x7e\x0b\x71\xcd\x81"
			  "\x0f\xb7\x0a\x59\x8f\x31\x4d\xd1"
			  "\xc3\xf3\x2f\x70\x5c\x59\x18\x97"
			  "\xaf\x77\x95\x5e\xaf\x40\x06\x12"
			  "\x81\x61\x86\x08\x4e\xbc\x89\x46"
			  "\x07\x2e\x5b\x10\xaa\x12\xf0\xa7"
			  "\x84\xe2\x9a\x08\xf1\xde\x59\xe3"
			  "\x0e\x47\x4b\xff\xc3\xc9\x18\xaf"
			  "\x95\x9c\x67\x2a\xde\x8a\x7a\x99"
			  "\x04\xc4\xb8\x97\x4c\x04\x29\x71"
			  "\x05\xda\xb3\xd6\xdb\x6c\x71\xe6"
			  "\xe8\x03\xbf\x94\x7d\xde\x3d\xc8"
			  "\x44\xfa\x7d\x62\xb4\x36\x03\xee"
			  "\x36\x52\x64\xb4\x85\x6d\xd5\x78"
			  "\xf0\x6f\x67\x2d\x0e\xe0\x2c\x88"
			  "\x9b\x55\x19\x29\x40\xf6\x8c\x12"
			  "\xbb\x2c\x83\x96\x40\xc0\x36\xf5"
			  "\x77\xff\x70\x8c\x75\x92\x0b\xad"
			  "\x05\x9b\x7e\xa2\xfc\xa9\xd1\x64"
			  "\x76\x82\x13\xba\x22\x5e\x33\x0e"
			  "\x26\x70\xa9\xbe\x74\x28\xf5\xe2"
			  "\xc4\x96\xee\x3a\xbc\x97\xa6\x2c"
			  "\x2a\xe0\x64\x8d\x35\xc6\x1a\xca"
			  "\xf4\x92\xfa\xc3\xf1\x1f\x98\xe4"
			  "\x43\x88\x69\x3a\x09\xbf\x63\xe5"
			  "\x96\x29\x0b\x9b\x62\x23\x14\x8a"
			  "\x95\xe4\x1c\x5c\x0a\xa9\xc5\xb9"
			  "\x6f\x4f\x2b\x25\x6f\x74\x1e\x18"
			  "\xd5\xfe\x27\x7d\x3f\x6e\x55\x2c"
			  "\x67\xe6\xde\xb5\xcc\xc0\x2d\xff"
			  "\xc4\xe4\x06\x21\xa5\xc8\xd3\xd6"
			  "\x6c\xa1\xc3\xfb\x88\x92\xb1\x1d"
			  "\x90\xe1\x35\x05\x9b\x29\x6d\xba"
			  "\xf1\xf4\x1e\x23\x2e",
		.len	= 1037,
	},
};

static const struct cipher_testvec gost28147_cpb_cnt_tv_template[] = {
	{
		.key	= "\x59\x9f\x84\xba\xc3\xf3\xd2\xf1"
			  "\x60\xe1\xe3\xf2\x6a\x96\x1a\xf9"
			  "\x9c\x48\xb2\x4e\xbc\xbb\xbf\x7c"
			  "\xd8\xf3\xac\xcd\x96\x8d\x28\x6a",
		.klen	= 32,
		.iv	= "\x8d\xaf\xa8\xd1\x58\xed\x05\x8d",
		.ptext	= "\x90\xa2\x39\x66\xae\x01\xb9\xa3"
			  "\x52\x4e\xc8\xed\x6c\xdd\x88\x30",
		.ctext	= "\x10\x92\x37\x31\x53\x38\xe5\x7d"
			  "\x1c\x5d\xcb\xb7\xf1\xbd\x27\x8b",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpc_cnt_tv_template[] = {
	{
		.key	= "\x59\x9f\x84\xba\xc3\xf3\xd2\xf1"
			  "\x60\xe1\xe3\xf2\x6a\x96\x1a\xf9"
			  "\x9c\x48\xb2\x4e\xbc\xbb\xbf\x7c"
			  "\xd8\xf3\xac\xcd\x96\x8d\x28\x6a",
		.klen	= 32,
		.iv	= "\x8d\xaf\xa8\xd1\x58\xed\x05\x8d",
		.ptext	= "\x90\xa2\x39\x66\xae\x01\xb9\xa3"
			  "\x52\x4e\xc8\xed\x6c\xdd\x88\x30",
		.ctext	= "\xbc\xea\xd7\xfe\x1a\xc9\xeb\xb3"
			  "\x49\x8b\xfa\xb4\xe0\x87\xf3\x6a",
		.len	= 16,
	},
};

static const struct cipher_testvec gost28147_cpd_cnt_tv_template[] = {
	{
		.key	= "\x59\x9f\x84\xba\xc3\xf3\xd2\xf1"
			  "\x60\xe1\xe3\xf2\x6a\x96\x1a\xf9"
			  "\x9c\x48\xb2\x4e\xbc\xbb\xbf\x7c"
			  "\xd8\xf3\xac\xcd\x96\x8d\x28\x6a",
		.klen	= 32,
		.iv	= "\x8d\xaf\xa8\xd1\x58\xed\x05\x8d",
		.ptext	= "\x90\xa2\x39\x66\xae\x01\xb9\xa3"
			  "\x52\x4e\xc8\xed\x6c\xdd\x88\x30",
		.ctext	= "\xa6\xde\x67\x20\x93\x72\x3c\x44"
			  "\xec\xd2\x5d\x9b\xc6\x66\x09\xf1",
		.len	= 16,
	},
};

static const struct hash_testvec gosthash94_tv_template[] = {
	{
		.plaintext = "",
		.psize = 0,
		.digest = "\x98\x1e\x5f\x3c\xa3\x0c\x84\x14"
			  "\x87\x83\x0f\x84\xfb\x43\x3e\x13"
			  "\xac\x11\x01\x56\x9b\x9c\x13\x58"
			  "\x4a\xc4\x83\x23\x4c\xd6\x56\xc0",
	}, {
		.plaintext = "a",
		.psize = 1,
		.digest = "\xe7\x4c\x52\xdd\x28\x21\x83\xbf"
			  "\x37\xaf\x00\x79\xc9\xf7\x80\x55"
			  "\x71\x5a\x10\x3f\x17\xe3\x13\x3c"
			  "\xef\xf1\xaa\xcf\x2f\x40\x30\x11",
	}, {
		.plaintext = "message digest",
		.psize = 14,
		.digest = "\xbc\x60\x41\xdd\x2a\xa4\x01\xeb"
			  "\xfa\x6e\x98\x86\x73\x41\x74\xfe"
			  "\xbd\xb4\x72\x9a\xa9\x72\xd6\x0f"
			  "\x54\x9a\xc3\x9b\x29\x72\x1b\xa0",
	}, {
		.plaintext = "The quick brown fox jumps over the lazy dog",
		.psize = 43,
		.digest = "\x90\x04\x29\x4a\x36\x1a\x50\x8c"
			  "\x58\x6f\xe5\x3d\x1f\x1b\x02\x74"
			  "\x67\x65\xe7\x1b\x76\x54\x72\x78"
			  "\x6e\x47\x70\xd5\x65\x83\x0a\x76",
	}
};

static const struct hash_testvec hmac_gosthash94_tv_template[] = {
	{
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ksize	= 32,
		.plaintext = "\x01\x26\xbd\xb8\x78\x00\xaf\x21"
			     "\x43\x41\x45\x65\x63\x78\x01\x00",
		.psize	= 16,
		.digest	= "\xba\xd7\x0b\x61\xc4\x10\x95\xbc"
			  "\x47\xe1\x14\x1c\xfa\xed\x42\x72"
			  "\x6a\x5c\xee\xbd\x62\xce\x75\xdb"
			  "\xbb\x9a\xd7\x6c\xda\x9f\x72\xf7",
	},
};

static const struct hash_testvec hmac_streebog256_tv_template[] = {
	{
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ksize	= 32,
		.plaintext = "\x01\x26\xbd\xb8\x78\x00\xaf\x21"
			     "\x43\x41\x45\x65\x63\x78\x01\x00",
		.psize	= 16,
		.digest	= "\xa1\xaa\x5f\x7d\xe4\x02\xd7\xb3"
			  "\xd3\x23\xf2\x99\x1c\x8d\x45\x34"
			  "\x01\x31\x37\x01\x0a\x83\x75\x4f"
			  "\xd0\xaf\x6d\x7c\xd4\x92\x2e\xd9",
	},
};

static const struct hash_testvec hmac_streebog512_tv_template[] = {
	{
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ksize	= 32,
		.plaintext = "\x01\x26\xbd\xb8\x78\x00\xaf\x21"
			     "\x43\x41\x45\x65\x63\x78\x01\x00",
		.psize	= 16,
		.digest	= "\xa5\x9b\xab\x22\xec\xae\x19\xc6"
			  "\x5f\xbd\xe6\xe5\xf4\xe9\xf5\xd8"
			  "\x54\x9d\x31\xf0\x37\xf9\xdf\x9b"
			  "\x90\x55\x00\xe1\x71\x92\x3a\x77"
			  "\x3d\x5f\x15\x30\xf2\xed\x7e\x96"
			  "\x4c\xb2\xee\xdc\x29\xe9\xad\x2f"
			  "\x3a\xfe\x93\xb2\x81\x4f\x79\xf5"
			  "\x00\x0f\xfc\x03\x66\xc2\x51\xe6",
	},
};

static const struct cipher_testvec kuznyechik_tv_template[] = {
	{
		.key	= "\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 32,
		.ptext	= "\x11\x22\x33\x44\x55\x66\x77\x00"
			  "\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xee\xff\x0a"
			  "\x11\x22\x33\x44\x55\x66\x77\x88"
			  "\x99\xaa\xbb\xcc\xee\xff\x0a\x00"
			  "\x22\x33\x44\x55\x66\x77\x88\x99"
			  "\xaa\xbb\xcc\xee\xff\x0a\x00\x11",
		.ctext	= "\x7f\x67\x9d\x90\xbe\xbc\x24\x30"
			  "\x5a\x46\x8d\x42\xb9\xd4\xed\xcd"
			  "\xb4\x29\x91\x2c\x6e\x00\x32\xf9"
			  "\x28\x54\x52\xd7\x67\x18\xd0\x8b"
			  "\xf0\xca\x33\x54\x9d\x24\x7c\xee"
			  "\xf3\xf5\xa5\x31\x3b\xd4\xb1\x57"
			  "\xd0\xb0\x9c\xcd\xe8\x30\xb9\xeb"
			  "\x3a\x02\xc4\xc5\xaa\x8a\xda\x98",
	},
};

static const struct cipher_testvec magma_tv_template[] = {
	{
		.key	= "\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
			  "\x77\x66\x55\x44\x33\x22\x11\x00"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.klen	= 32,
		.ptext	= "\x92\xde\xf0\x6b\x3c\x13\x0a\x59"
			  "\xdb\x54\xc7\x04\xf8\x18\x9d\x20"
			  "\x4a\x98\xfb\x2e\x67\xa8\x02\x4c"
			  "\x89\x12\x40\x9b\x17\xb5\x7e\x41",
		.ctext	= "\x2b\x07\x3f\x04\x94\xf3\x72\xa0"
			  "\xde\x70\xe7\x15\xd3\x55\x6e\x48"
			  "\x11\xd8\xd9\xe9\xea\xcf\xbc\x1e"
			  "\x7c\x68\x26\x09\x96\xc6\x7e\xfb",
		.len	= 32,
	}, {
		.key	= "\xff\xee\xdd\xcc\xbb\xaa\x99\x88"
			  "\x77\x66\x55\x44\x33\x22\x11\x00"
			  "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.klen	= 32,
		.ptext	= "\xfe\xdc\xba\x98\x76\x54\x32\x10",
		.ctext	= "\x4e\xe9\x01\xe5\xc2\xd8\xca\x3d",
		.len	= 8,
	}, {
		.key	= "\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x00",
		.ctext	= "\x3b\x9a\x2e\xaa\xbe\x78\x3b\xab",
		.len	= 8,
	}, {
		.key	= "\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
			  "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\xfe\xdc\xba\x98\x76\x54\x32\x10"
			  "\x01\x23\x45\x67\x89\xab\xcd\xef",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x01",
		.ctext	= "\x97\x0f\xd9\x08\x06\xc1\x0d\x62",
		.len	= 8,
	}, {
		.key	= "\x86\x3e\xa0\x17\x84\x2c\x3d\x37"
			  "\x2b\x18\xa8\x5a\x28\xe2\x31\x7d"
			  "\x74\xbe\xfc\x10\x77\x20\xde\x0c"
			  "\x9e\x8a\xb9\x74\xab\xd0\x0c\xa0",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x02",
		.ctext	= "\xc7\x3d\x45\x9c\x28\x7b\x3d\x1c",
		.len	= 8,
	}, {
		.key	= "\x86\x3e\xa0\x17\x84\x2c\x3d\x37"
			  "\x2b\x18\xa8\x5a\x28\xe2\x31\x7d"
			  "\x74\xbe\xfc\x10\x77\x20\xde\x0c"
			  "\x9e\x8a\xb9\x74\xab\xd0\x0c\xa0",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x03",
		.ctext	= "\x86\x36\x1c\xac\xbc\x1f\x4c\x24",
		.len	= 8,
	}, {
		.key	= "\x49\xa5\xe2\x67\x7d\xe5\x55\x98"
			  "\x2b\x8a\xd5\xe8\x26\x65\x2d\x17"
			  "\xee\xc8\x47\xbf\x5b\x39\x97\xa8"
			  "\x1c\xf7\xfe\x7f\x11\x87\xbd\x27",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x04",
		.ctext	= "\xb0\x8c\x42\x50\xcb\x8b\x64\x0a",
		.len	= 8,
	}, {
		.key	= "\x49\xa5\xe2\x67\x7d\xe5\x55\x98"
			  "\x2b\x8a\xd5\xe8\x26\x65\x2d\x17"
			  "\xee\xc8\x47\xbf\x5b\x39\x97\xa8"
			  "\x1c\xf7\xfe\x7f\x11\x87\xbd\x27",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x05",
		.ctext	= "\x32\x7e\xdc\xd4\xe8\x8d\xe6\x6f",
		.len	= 8,
	}, {
		.key	= "\x32\x56\xbf\x3f\x97\xb5\x66\x74"
			  "\x26\xa9\xfb\x1c\x5e\xaa\xbe\x41"
			  "\x89\x3c\xcd\xd5\xa8\x68\xf9\xb6"
			  "\x3b\x0a\xa9\x07\x20\xfa\x43\xc4",
		.klen	= 32,
		.ptext	= "\x12\x34\x56\x78\x00\x00\x00\x06",
		.ctext	= "\xa6\x91\xb5\x0e\x59\xbd\xfa\x58",
		.len	= 8,
	},
};

/* Test vectors: M1 and M2 are from RFC6986, carry is from gost-engine */
static const struct hash_testvec streebog_512_tv_template[] = {
	{ /* M1 */
		.plaintext =
			"01234567890123456789012345678901"
			"2345678901234567890123456789012",
		.psize = 63,
		.digest =
			"\x1b\x54\xd0\x1a\x4a\xf5\xb9\xd5"
			"\xcc\x3d\x86\xd6\x8d\x28\x54\x62"
			"\xb1\x9a\xbc\x24\x75\x22\x2f\x35"
			"\xc0\x85\x12\x2b\xe4\xba\x1f\xfa"
			"\x00\xad\x30\xf8\x76\x7b\x3a\x82"
			"\x38\x4c\x65\x74\xf0\x24\xc3\x11"
			"\xe2\xa4\x81\x33\x2b\x08\xef\x7f"
			"\x41\x79\x78\x91\xc1\x64\x6f\x48"
	}, { /* M2 */
		.plaintext =
			"\xd1\xe5\x20\xe2\xe5\xf2\xf0\xe8"
			"\x2c\x20\xd1\xf2\xf0\xe8\xe1\xee"
			"\xe6\xe8\x20\xe2\xed\xf3\xf6\xe8"
			"\x2c\x20\xe2\xe5\xfe\xf2\xfa\x20"
			"\xf1\x20\xec\xee\xf0\xff\x20\xf1"
			"\xf2\xf0\xe5\xeb\xe0\xec\xe8\x20"
			"\xed\xe0\x20\xf5\xf0\xe0\xe1\xf0"
			"\xfb\xff\x20\xef\xeb\xfa\xea\xfb"
			"\x20\xc8\xe3\xee\xf0\xe5\xe2\xfb",
		.psize = 72,
		.digest =
			"\x1e\x88\xe6\x22\x26\xbf\xca\x6f"
			"\x99\x94\xf1\xf2\xd5\x15\x69\xe0"
			"\xda\xf8\x47\x5a\x3b\x0f\xe6\x1a"
			"\x53\x00\xee\xe4\x6d\x96\x13\x76"
			"\x03\x5f\xe8\x35\x49\xad\xa2\xb8"
			"\x62\x0f\xcd\x7c\x49\x6c\xe5\xb3"
			"\x3f\x0c\xb9\xdd\xdc\x2b\x64\x60"
			"\x14\x3b\x03\xda\xba\xc9\xfb\x28"
	}, { /* carry */
		.plaintext =
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\x16\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x16",
		.psize = 128,
		.digest =
			"\x8b\x06\xf4\x1e\x59\x90\x7d\x96"
			"\x36\xe8\x92\xca\xf5\x94\x2f\xcd"
			"\xfb\x71\xfa\x31\x16\x9a\x5e\x70"
			"\xf0\xed\xb8\x73\x66\x4d\xf4\x1c"
			"\x2c\xce\x6e\x06\xdc\x67\x55\xd1"
			"\x5a\x61\xcd\xeb\x92\xbd\x60\x7c"
			"\xc4\xaa\xca\x67\x32\xbf\x35\x68"
			"\xa2\x3a\x21\x0d\xd5\x20\xfd\x41"
	},
};

static const struct hash_testvec streebog_256_tv_template[] = {
	{
		.plaintext =
			"01234567890123456789012345678901"
			"2345678901234567890123456789012",
		.psize = 63,
		.digest =
			"\x9d\x15\x1e\xef\xd8\x59\x0b\x89"
			"\xda\xa6\xba\x6c\xb7\x4a\xf9\x27"
			"\x5d\xd0\x51\x02\x6b\xb1\x49\xa4"
			"\x52\xfd\x84\xe5\xe5\x7b\x55\x00"
	}, {
		.plaintext =
			"\xd1\xe5\x20\xe2\xe5\xf2\xf0\xe8"
			"\x2c\x20\xd1\xf2\xf0\xe8\xe1\xee"
			"\xe6\xe8\x20\xe2\xed\xf3\xf6\xe8"
			"\x2c\x20\xe2\xe5\xfe\xf2\xfa\x20"
			"\xf1\x20\xec\xee\xf0\xff\x20\xf1"
			"\xf2\xf0\xe5\xeb\xe0\xec\xe8\x20"
			"\xed\xe0\x20\xf5\xf0\xe0\xe1\xf0"
			"\xfb\xff\x20\xef\xeb\xfa\xea\xfb"
			"\x20\xc8\xe3\xee\xf0\xe5\xe2\xfb",
		.psize = 72,
		.digest =
			"\x9d\xd2\xfe\x4e\x90\x40\x9e\x5d"
			"\xa8\x7f\x53\x97\x6d\x74\x05\xb0"
			"\xc0\xca\xc6\x28\xfc\x66\x9a\x74"
			"\x1d\x50\x06\x3c\x55\x7e\x8f\x50"
	}, {
		.plaintext =
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\xEE\xEE\xEE\xEE\xEE\xEE\xEE\xEE"
			"\x16\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x11"
			"\x11\x11\x11\x11\x11\x11\x11\x16",
		.psize = 128,
		.digest =
			"\x81\xbb\x63\x2f\xa3\x1f\xcc\x38"
			"\xb4\xc3\x79\xa6\x62\xdb\xc5\x8b"
			"\x9b\xed\x83\xf5\x0d\x3a\x1b\x2c"
			"\xe7\x27\x1a\xb0\x2d\x25\xba\xbb"
	},
};

static const struct hash_testvec gost28147imit_tc26z_tv_template[] = {
	{
		.key	= "\x33\xd3\xef\x01\x19\x95\x0e\x15"
			  "\xa1\x69\x75\xae\x56\x27\x17\x79"
			  "\x63\x47\xab\x62\x9d\x4a\xf0\x34"
			  "\xd3\x1e\x69\x74\xec\x31\x48\xfc",
		.ksize	= 32,
		.plaintext = "\x02\xf8\xec\x2b\x4d\x1f\xbc\x7c"
			  "\x6e\x47\xe3\x87\x22\x75\x41\xa7",
		.psize = 16,
		.digest	="\x1e\xb5\xa4\x4a",
	},
};

static const struct hash_testvec gost28147imit_cpa_tv_template[] = {
	{
		.key	= "\x9d\x05\xb7\x9e\x90\xca\xd0\x0a"
			  "\x2c\xda\xd2\x2e\xf4\xe8\x6f\x5c"
			  "\xf5\xdc\x37\x68\x19\x85\xb3\xbf"
			  "\xaa\x18\xc1\xc3\x05\x0a\x91\xa2",
		.ksize	= 32,
		.plaintext = "\xb5\xa1\xf0\xe3\xce\x2f\x02\x1d"
			  "\x67\x61\x94\x34\x5c\x41\xe3\x6e",
		.psize	= 16,
		.digest = "\xf8\x1f\x08\xa3",
	},
	{
		.key	= "\xa9\xb6\x37\xcc\x6d\x9b\x2f\x25"
			  "\xb0\xdf\x47\x04\x50\x68\xb0\x27"
			  "\x41\x27\x58\x6a\xbd\x0a\x6e\x50"
			  "\x2f\xc6\xfc\xc0\x3e\x29\x42\xa5",
		.ksize	= 32,
		.plaintext = "\x1d\xeb\xe6\x79\x0a\x59\x00\xe6"
			  "\x8e\x5c",
		.psize	= 10,
		.digest = "\x31\x7c\x16\xe4",
	}, {
		.key	= "\xb0\x6c\x48\x23\x0a\x6e\xf4\xec"
			  "\x27\x98\x01\x23\xa7\xd8\xbf\x60"
			  "\x89\xef\xad\xe8\x8f\x79\x14\x8c"
			  "\x18\x5c\x9a\xda\xef\x0b\xdd\xa0",
		.ksize	= 32,
		.plaintext = "\xef\x06\x8f\x14\xc9\x04",
		.psize	= 6,
		.digest = "\xe9\x72\xae\xbf",
	}, {
		.key	= "\x80\xd9\xa0\xdc\x21\xf9\x30\x40"
			  "\x75\xfe\x49\x1b\x9e\x71\x90\x91"
			  "\x78\x88\x21\x60\x39\xe7\xc9\x2b"
			  "\xfb\x55\x1d\xf4\xdd\x2b\x0a\x01",
		.ksize	= 32,
		.plaintext = "\xd6\xcf\x31\x96\x9c\xa1\xfb\xd6"
			  "\x8d\xa3\xdd\x01\xd9\x88\xc0\x2f"
			  "\xbc\x46\xc7\x3a\xe4\x21\x86\x96"
			  "\x8d\xe2\xca\xb6\x37\xa2\xe1\xa8"
			  "\x7e\xa7\x79\x2e\xa4\x56\x75\x7f"
			  "\x3e\x55\x8b\x43\xae\x65\xdf\xaa"
			  "\x42\xb6\x00\xa6\x61\x03\x0d\xd3"
			  "\x41\x02\x27\x23\x95\x79\x9b\x34"
			  "\x81\xa9\x86\xb5\xa7\x90\xe2\xae"
			  "\xc4\x2f\xc3\x8e\x32\x56\x13\xfa"
			  "\x4d\x4e\x9f\x15\x75\x7e\x74\xdc"
			  "\x32\x2d\xee\x4d\x67\x70\x9f\x62"
			  "\xb9\xc4\xdb\x24\x84\xcc\x16\x7b"
			  "\xda\x22\xf7\xc5\xf3\x93\x35\x73"
			  "\xc6\x03\x1c\x77\xa5\xf2\x76\x56"
			  "\xb4\x95\xd4\x7e\x0d\x20\xc6\x6e"
			  "\xee\x8f\x25\x48\xff\x7e\x01\x3a"
			  "\xb4\x1f\xaa\x35\xc0\x33\x58\x9c"
			  "\xb5\xba\x65\x4b\xd3\x51\x14\xec"
			  "\x61\xce\xe4\xba\x49\xba\x39\x32"
			  "\xab\xce\x81\x72\xce\xab\xed\xd4"
			  "\xd2\x19\x87\x85\x92\xfa\x64\x34"
			  "\xd8\x86\xf4\x8a\x08\x3c\xde\xee"
			  "\x97\x92\x92\x69\xba\x9b\x5f\x7a"
			  "\x03\xc1\x5d\x43\x02\x8c\xbe\xd2"
			  "\x46\x72\x81\x40\x7d\x68\x98\x45"
			  "\x0b\x54\x27\x1c\xaf\x80\x42\xe4"
			  "\xd5\xd4\xe4\xa2\x98\x07\x8f\x03"
			  "\xf5\x2c\x8c\x88\xca\x5a\xde\xe4"
			  "\x9f\xb1\x5f\x82\xff\x20\x67\x52"
			  "\x85\x84\x4f\xc8\xfe\xa7\x9e\xae"
			  "\x1c\xfa\xb8\x75\xd3\xf7\x9f\x0d"
			  "\xda\x2d\xe6\xcc\x86\x6b\xa4\x14"
			  "\x65\xc3\xf9\x15\xbc\x87\xf5\xae"
			  "\x8c\x10\xd4\xce\x5b\x9c\xe2\xdd"
			  "\x42\x03\x09\x87\x47\xed\x5d\xd0"
			  "\x7a\x69\x4c\xfa\x43\x7d\xbf\x07"
			  "\x85\x6a\xee\x68\xe6\x7a\x57\xb2"
			  "\x20\x8d\x80\xf2\x91\x6f\x5c\x07"
			  "\x8c\xe4\x6a\x49\x90\x85\x8b\x77"
			  "\x29\x56\x1c\x5e\xa9\x3f\xab\x8b"
			  "\x79\xa3\x6f\x6b\x34\xcb\x61\xf6"
			  "\xe6\x92\xd1\x48\x9e\x11\xa2\x82"
			  "\xc0\x4e\x23\xd2\x15\x0d\x8d\xff"
			  "\xfa\x17\x9d\x81\xb8\xbc\xd7\x5b"
			  "\x08\x81\x20\x40\xc0\x3c\x06\x8b"
			  "\x1a\x88\x0b\x4b\x7b\x31\xf5\xd4"
			  "\x4e\x09\xd1\x4d\x0d\x7f\x45\xd1"
			  "\x09\x35\xba\xce\x65\xdd\xf2\xb8"
			  "\xfb\x7a\xbc\xc4\x4b\xc8\x75\xda"
			  "\x6b\xce\x3d\xe8\x94\xcc\x23\x6f"
			  "\xb0\x3b\x4f\x7d\x07\xb9\x0f\x62"
			  "\x92\x7e\xda\x70\x50\xce\xd3\x28"
			  "\x12\x11\x00\xeb\x8d\x63\x70\x78"
			  "\xa8\x7b\x76\xab\xc6\x40\xc0\x4e"
			  "\x80\xdd\xf0\xfe\x83\x72\x56\x4c"
			  "\x09\x4c\xf1\x72\x72\x86\x26\x31"
			  "\xc3\xc2\xdc\x8e\xc7\xf4\x35\xec"
			  "\x17\x06\x63\x47\x49\x88\x47\xaf"
			  "\xb3\x38\x4f\x7e\x44\x95\xb5\xbb"
			  "\x1d\xbd\x5a\x91\x5b\xd0\x1a\xdf"
			  "\x0d\x0b\x50\xd8\xe2\x0e\xc5\x00"
			  "\x2d\x5b\x29\x19\xaa\x2b\x64\xc5"
			  "\x40\x31\x48\x11\xbc\x04\xd1\xcf"
			  "\x6d\xf9\xa5\x2f\x4a\xc9\x82\xfa"
			  "\x59\xe1\xfc\xab\x1c\x33\x26\x0a"
			  "\x5f\xef\xf2\x06\xd8\xd3\x7e\x16"
			  "\x58\x16\x78\x73\xae\xba\xeb\xe5"
			  "\x3d\xb2\x0a\xb3\x32\x2d\x14\xa4"
			  "\xfa\x3f\x1f\x43\xf9\x7b\xa9\x43"
			  "\x98\x18\x94\x07\x07\xe5\x19\x34"
			  "\xa8\x16\x5f\x71\x67\xaa\x29\xe5"
			  "\xfa\xf0\x83\x06\x1d\x9d\xfc\xfe"
			  "\xfe\x8c\xb5\xb2\xa9\xe7\xa0\x40"
			  "\x60\xb6\x71\x9e\xab\x5b\x83\xb9"
			  "\x0c\x2b\x58\x23\x80\x09\x9e\x5d"
			  "\x94\x7d\x40\x76\xa9\x16\x96\x9e"
			  "\x83\xe0\x0d\xec\xa0\xec\x76\x2a"
			  "\xb7\xa0\xff\xb8\x50\x4c\x5b\xc6"
			  "\x8b\x0a\x65\x2e\xfe\xb4\x40\x9a"
			  "\x01\xd8\xc6\xa3\xab\x99\xa2\xc5"
			  "\x0c\x08\xc4\xb7\xee\x4d\x1d\xc4"
			  "\x08\x15\xd0\xdb\xaa\x63\x4f\x31"
			  "\xeb\x14\x97\x43\xbd\xc1\x94\x08"
			  "\xe6\xde\x43\x9f\x95\x0b\x96\x7e"
			  "\x7f\x3c\x68\xba\x6f\xc4\xc9\x35"
			  "\x2b\xc4\x0e\xda\x1f\x91\x68\x64"
			  "\x63\x34\x73\xbe\x57\x75\xb9\xed"
			  "\xf7\x2d\x3b\x05\x21\x93\x28\x48"
			  "\x96\x95\x97\xa0\xd2\x7d\x78\xbb"
			  "\x6a\x49\x8f\x76\x55\x74\x63\xb9"
			  "\xc5\x36\x12\x25\xbf\x03\x82\x8f"
			  "\xf0\xf6\x80\xbb\x33\xb4\xf4\x17"
			  "\x27\x1c\xf3\x4c\x10\xa3\xe4\xd1"
			  "\x55\xd9\x68\x21\x4e\x5a\x83\x67"
			  "\xbf\xf8\x3c\x7d\x4e\x62\xd3\x28"
			  "\xa7\x26\x6f\xe9\xee\xc2\x0b\x2d"
			  "\x03\x84\xb1\xff\xd6\x68\x1f\xb6"
			  "\xf2\xe4\x0f\xda\x2d\xee\x5f\x6e"
			  "\x21\xc8\xe1\xfc\xad\x6b\x0e\x04"
			  "\x7d\xaf\xc2\x3b\xa5\x68\x9b\x0c"
			  "\xf3\x56\xf3\xda\x8d\xc8\x7d\x39"
			  "\xdc\xd5\x99\xc6\x01\x10\xce\x42"
			  "\x1b\xac\x48\xdc\x97\x78\x0a\xec"
			  "\xb3\x8f\x47\x35\xa3\x6a\x64\xb2"
			  "\x8e\x63\x69\x22\x66\xae\x2e\xe0"
			  "\x88\xf9\x40\x3c\xc9\xa2\x57\x61"
			  "\xf6\xad\xf0\xdc\x90\x56\x3f\x06"
			  "\x9b\x7d\xbd\xc2\x81\x02\xab\xb8"
			  "\x15\x09\x88\x4a\xff\x2f\x31\xbf"
			  "\x5e\xfa\x6a\x7e\xf6\xc5\xa7\xf7"
			  "\xd5\xab\x55\xac\xae\x0d\x8c\x8d"
			  "\x7f\x4b\x25\xbb\x32\xff\x11\x33"
			  "\x2e\x37\x37\x69\x96\x15\x17\xb1"
			  "\x17\x49\xe0\x9a\x9c\xd9\x5b\x8d"
			  "\x58\xa3\x1d\x92\x87\xf8\x80\xb9"
			  "\xbd\x5a\xec\x40\xe1\x00\x33\x60"
			  "\xe4\x86\x16\x6d\x61\x81\xf2\x28"
			  "\x6a\xa7\xce\x3f\x95\xae\x43\xca"
			  "\xe1\x3f\x81\x74\x7e\x1c\x47\x17"
			  "\x95\xc6\x60\xda\x74\x77\xd9\x9f"
			  "\xfa\x92\xb4\xbe\xe1\x23\x98\x18"
			  "\x95\x63\x03\x13\x4c\x1a\x2d\x41"
			  "\xcd\xe4\x84\xf7\xe6\x38\xef\xff"
			  "\x95\xb2\xe8\x7c\x8f\x58\xb5\xb5"
			  "\xed\x27\x7f\x3c\x18\xab\xbe\x7f"
			  "\x4f\xe2\x35\x15\x71\xb7\x6f\x85"
			  "\x38\x9b\x88\xf6\x9c\x8d\x43\xb5"
			  "\x58\x9e\xf2\xd1\x96\xbe\xb7\xad"
			  "\x1a\xa0\x98",
		.psize	= 1035,
		.digest	= "\x90\xf2\x11\x9a",
	}, {
		.key	= "\x42\x35\x81\x91\x0b\xa9\x99\xff"
			  "\xd9\x43\xf8\xc6\x19\x55\x1f\x2f"
			  "\x2d\x45\x40\x20\x1e\x1d\x32\x7a"
			  "\xb1\x07\x6b\x4f\x45\x90\xd9\x80",
		.ksize	= 32,
		.plaintext =  "\xf3\xb2\x29\xd2\x7a\x37\x03\x12",
		.psize	= 8,
		.digest	= "\x6e\x15\xfa\xe8",
	}, {
		.key	= "\x26\xcb\xb9\xf0\x0c\x62\x9f\xaa"
			  "\x4a\x1d\xb6\x30\x09\x01\x56\x89"
			  "\x66\xd4\xe4\x0e\xfe\xf6\x10\x6b"
			  "\x6c\xe8\x04\x3a\xe3\x61\x4b\x19",
		.ksize	= 32,
		.plaintext = "",
		.psize	= 0,
		.digest	= "\x00\x00\x00\x00",
	},
};

static const struct hash_testvec gost28147imit_cpb_tv_template[] = {
	{
		.key	= "\x33\xd3\xef\x01\x19\x95\x0e\x15"
			  "\xa1\x69\x75\xae\x56\x27\x17\x79"
			  "\x63\x47\xab\x62\x9d\x4a\xf0\x34"
			  "\xd3\x1e\x69\x74\xec\x31\x48\xfc",
		.ksize	= 32,
		.plaintext = "\x02\xf8\xec\x2b\x4d\x1f\xbc\x7c"
			  "\x6e\x47\xe3\x87\x22\x75\x41\xa7",
		.psize = 16,
		.digest	="\xf5\x55\x1f\x28",
	},
};

static const struct hash_testvec gost28147imit_cpc_tv_template[] = {
	{
		.key	= "\x33\xd3\xef\x01\x19\x95\x0e\x15"
			  "\xa1\x69\x75\xae\x56\x27\x17\x79"
			  "\x63\x47\xab\x62\x9d\x4a\xf0\x34"
			  "\xd3\x1e\x69\x74\xec\x31\x48\xfc",
		.ksize	= 32,
		.plaintext = "\x02\xf8\xec\x2b\x4d\x1f\xbc\x7c"
			  "\x6e\x47\xe3\x87\x22\x75\x41\xa7",
		.psize = 16,
		.digest	="\x3c\x28\xef\xf5",
	},
};

static const struct hash_testvec gost28147imit_cpd_tv_template[] = {
	{
		.key	= "\x33\xd3\xef\x01\x19\x95\x0e\x15"
			  "\xa1\x69\x75\xae\x56\x27\x17\x79"
			  "\x63\x47\xab\x62\x9d\x4a\xf0\x34"
			  "\xd3\x1e\x69\x74\xec\x31\x48\xfc",
		.ksize	= 32,
		.plaintext = "\x02\xf8\xec\x2b\x4d\x1f\xbc\x7c"
			  "\x6e\x47\xe3\x87\x22\x75\x41\xa7",
		.psize = 16,
		.digest	="\xf0\x94\x53\x0b",
	},
};

static const struct hash_testvec cmac_magma_tv_template[] = {
	{
		.key	= "\x50\x76\x42\xd9\x58\xc5\x20\xc6"
			  "\xd7\xee\xf5\xca\x8a\x53\x16\xd4"
			  "\xf3\x4b\x85\x5d\x2d\xd4\xbc\xbf"
			  "\x4e\x5b\xf0\xff\x64\x1a\x19\xff",
		.ksize	= 32,
		.plaintext = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x17\x03\x03\x00\x07\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.psize	= 20,
		.digest	= "\x30\x70\xb8\x64\x77\x9d\x95\x47",
	},
};

static const struct hash_testvec cmac_kuznyechik_tv_template[] = {
	{
		.key	= "\x50\x76\x42\xd9\x58\xc5\x20\xc6"
			  "\xd7\xee\xf5\xca\x8a\x53\x16\xd4"
			  "\xf3\x4b\x85\x5d\x2d\xd4\xbc\xbf"
			  "\x4e\x5b\xf0\xff\x64\x1a\x19\xff",
		.ksize	= 32,
		.plaintext = "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x17\x03\x03\x00\x0f\x00\x00\x00"
			  "\x00\x00\x00\x00\x00\x00\x00\x00"
			  "\x00\x00\x00\x00",
		.psize	= 28,
		.digest = "\x75\x53\x09\xcb\xc7\x3b\xb9\x49"
			  "\xc5\x0e\xbb\x86\x16\x0a\x0f\xee",
	},
};

#endif	/* _CRYPTO_TESTMGR_H */
