/*
 *  $Id: microprof.c 22617 2019-10-29 16:32:19Z yeti-dn $
 *  Copyright (C) 2005-2019 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

/**
 * [FILE-MAGIC-FREEDESKTOP]
 * <mime-type type="application/x-microprof-txt">
 *   <comment>MicroProf FRT text data</comment>
 *   <magic priority="80">
 *     <match type="string" offset="0" value="HeaderLines"/>
 *   </magic>
 * </mime-type>
 **/

/**
 * [FILE-MAGIC-FREEDESKTOP]
 * <mime-type type="application/x-microprof">
 *   <comment>MicroProf FRT data</comment>
 *   <magic priority="80">
 *     <match type="string" offset="0" value="FRTM_"/>
 *   </magic>
 * </mime-type>
 **/

/**
 * [FILE-MAGIC-FILEMAGIC]
 * # MicroProf FRT has binary and text data files.
 * 0 string FRTM_ MicroProf FRT profilometry data
 * 0 string HeaderLines=
 * >&0 search/80 ScanMode= MicroProf FRT profilometry text data
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * MicroProf TXT
 * .txt
 * Read
 **/

/**
 * [FILE-MAGIC-USERGUIDE]
 * MicroProf FRT
 * .frt
 * Read
 **/

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwyutils.h>
#include <libgwyddion/gwymath.h>
#include <libgwymodule/gwymodule-file.h>
#include <libprocess/datafield.h>
#include <app/data-browser.h>
#include <app/gwymoduleutils-file.h>

#include "get.h"
#include "err.h"

#define MAGIC      "FRTM_"
#define MAGIC_SIZE (sizeof(MAGIC) - 1)

/* All files I have ever seen start like this, but maybe the V1.00 part can
 * change. */
#define LONGMAGIC      "FRTM_GLIDERV1.00"
#define LONGMAGIC_SIZE (sizeof(LONGMAGIC) - 1)

#define MAGIC_TXT      "HeaderLines="
#define MAGIC_TXT_SIZE (sizeof(MAGIC_TXT) - 1)

#define EXTENSION ".frt"
#define EXTENSION_TXT ".txt"

enum {
    MICROPROF_MIN_HEADER_SIZE = 122,
    MICROPROF_MIN_TEXT_SIZE = 80,
    MAX_BLOCK_ID = 0x100,
};

typedef enum {
    MICROPROF_TOPOGRAPHY    = 0x00000004u,
    MICROPROF_INTENSITY     = 0x00000002u,
    MICROPROF_BOTTOM_SENSOR = 0x10000000u,
    /* I have only seen these two together. */
    MICROPROF_THICKNESS     = 0x00000800u,
    MICROPROF_THICKNESS2    = 0x00020000u,
} MicroProfDataType;

/* A small chunk before each image in the images data block 0x007d in the
 * multi-image part.  We synthesize it for the single-image part from block
 * 0x0066. */
typedef struct {
    MicroProfDataType datatype;
    guint xres;
    guint yres;
    guint bpp;
} MicroProfImageBlock;

/* Blocks occurring in the single-image part. */
typedef struct {
    guint size;
    const guchar *data; /* Just a pointer to file buffer. */
} MicroProfBlock000b;

typedef struct {
    gchar *text;       /* Comment1. */
} MicroProfBlock0065;

typedef struct {
    guint xres;        /* X pixel size. */
    guint yres;        /* Y pixel size. */
    guint bpp;         /* Bits per sample. */
} MicroProfBlock0066;

typedef struct {
    gdouble xrange;    /* Horizontal dimension in m. */
    gdouble yrange;    /* Vertical dimension in m. */
    gdouble float1;
    gdouble float2;
    gdouble float3;
    guint int1;
} MicroProfBlock0067;

typedef struct {
    guint int1;
    gdouble zscale;    /* Conversion factor to heights in m. */
} MicroProfBlock006c;

/* User adjustment of the false colour bar scale. */
typedef struct {
    gdouble float1;
} MicroProfBlock0071;

typedef struct {
    gchar *text;       /* Comment2. */
} MicroProfBlock0077;

/* Blocks occurring in the multi-image part. */
typedef struct {
    gdouble float1;     /* Number of the order 0.001 to 0.1. */
    gdouble float2;     /* Always the same as float1. */
    guint zero1;
    guint zero2;
    guint zero3;
    guint int1;         /* Always 14. */
    guint int2;         /* Always 300. */
} MicroProfBlock0068;

typedef struct {
    guint bitmask1;     /* All four are 0xfffffffa or 0xfffffffd. */
    guint bitmask2;
    guint bitmask3;
    guint bitmask4;
} MicroProfBlock0069;

typedef struct {
    gdouble zero1;
    gdouble float1;     /* Always 5e-8. */
    gdouble float2;     /* Always 5e-8. */
    gdouble zero2;
} MicroProfBlock006a;

typedef struct {
    guint int1;         /* Zero or 5. */
    guint int2;         /* Zero or 1.  Nonzero at the same time. */
    guint int3;         /* Zero. */
    guint int4;         /* Always 1. */
} MicroProfBlock006b;

typedef struct {
    gdouble float1;
    gdouble float2;
    guint int1;         /* 1 or 4. */
    guint int2;         /* Always 4. */
    guint int3;         /* Always 3. */
    guint int4;         /* Always 1. */
} MicroProfBlock006d;

typedef struct {
    guint int1;         /* 1 or 3. */
    guint int2;         /* 5 or 10. */
    guint int3;         /* 1 or 10. */
} MicroProfBlock006e;

typedef struct {
    guint zero1;
    guint int1;         /* 1 or 25. */
    guint zero2;
    guint int2;         /* Always 1. */
} MicroProfBlock006f;

typedef struct {
    guint int1;         /* Always 5. */
    guint bitmask1;     /* 0xfff9 or 0x7ffffff9. */
} MicroProfBlock0070;

typedef struct {
    guint timestamp1;   /* A large number, some timestamp. */
    guint timestamp2;   /* A slightly larger number, some timestamp. */
    guint meastime;     /* A number close to their difference, some measurement
                           time. */
} MicroProfBlock0072;

typedef struct {
    guint int1;         /* 6 or 10. */
} MicroProfBlock0073;

typedef struct {
    guint int1;         /* 1 or 3. */
    guint int2;         /* Zero. */
    guint int3;         /* 5 or 10. */
    guint int4;         /* 0 or 15. */
} MicroProfBlock0074;

typedef struct {
    guint palettesize;  /* Number of palette entries. */
    guint zero1;
    guint zero2;
    guint int1;         /* Always 1. */
    guint int2;         /* Always 1. */
    /* Then there are 4*palettesize bytes representing the palette entries. */
} MicroProfBlock0075;

typedef struct {
    guchar zeros[16];
} MicroProfBlock0079;

typedef struct {
    guint int1;         /* 5, 6 or 7. */
    guint int2;         /* A nice decimal number like 25, 30, 40 or 50. */
} MicroProfBlock007a;

typedef struct {
    guint int1;         /* 0 or 1. */
    guint int2;         /* 0, 5 or 7. */
} MicroProfBlock007b;

typedef struct {
    guint int1;         /* Always 1. */
} MicroProfBlock007c;

typedef struct {
    guint int1;
    guint int2;
    guint int3;
    guint int4;
    guint nimages;
    MicroProfImageBlock *imgblocks;
    const guchar **data;
} MicroProfBlock007d;

typedef struct {
    gdouble float1;     /* Typically 0.1, but possibly other number. */
    gdouble float2;     /* Typically 0.1, but possibly other number. */
} MicroProfBlock007e;

typedef struct {
    guchar calibration[0x100];      /* Always Uncalibrated. */
} MicroProfBlock007f;

typedef struct {
    guint zero1;
} MicroProfBlock0080;

typedef struct {
    guint int1;         /* Always 1. */
    gdouble float1;     /* Small number in the order of 10^{4}. */
    gdouble float2;     /* Small number in the order of 10^{4}. */
} MicroProfBlock0081;

typedef struct {
    guint zero1;
} MicroProfBlock0082;

typedef struct {
    guint zero1;
} MicroProfBlock0083;

typedef struct {
    guint int1;         /* 0 or 1. */
} MicroProfBlock0084;

typedef struct {
    guint int1;         /* Always 2. */
    guint int2;         /* 1049, 1084 or 1298, possibly other numbers. */
} MicroProfBlock0086;

typedef struct {
    guint zero1;
} MicroProfBlock0087;

typedef struct {
    guint zero1;
    guint int1;         /* Always 1. */
    guint int2;         /* Always 3. */
    guint bitmask1;     /* Always 0xffff. */
    guint zero2;
    gdouble float1;     /* Always -10. */
    gdouble float2;     /* Always 10. */
    gdouble float3;     /* Always -10. */
    gdouble float4;     /* Always 10. */
    guchar text[140];   /* Always m and then zeros.  Units? */
    gdouble float5;     /* Always 0.4. */
} MicroProfBlock0088;

typedef struct {
    guint zero1;
    guint zero2;
} MicroProfBlock0089;

typedef struct {
    guint zero1;
    guint zero2;
    guint zero3;
} MicroProfBlock008a;

typedef struct {
    guint int1;         /* 0 or 1. */
} MicroProfBlock008b;

/* User adjustments of the false colour bar scale. */
typedef struct {
    gdouble float1;
    gdouble float2;
    guint int1;
    gdouble float3;
} MicroProfBlock008c;

typedef struct {
    guint zero1;
    gdouble float1;     /* Always 100. */
    gdouble float2;     /* Always 100. */
    guint zero2;
    gdouble float3;     /* Always 25000. */
    guint int1;         /* Always 15. */
    guint int2;         /* Always 10. */
    guint zero3;
    guint zero4;
    guint zero5;
    guint zero6;
    guint zero7;
} MicroProfBlock008d;

typedef struct {
    guint textlen;
    gchar *text;       /* Sample?  Software?  */
    guint zero1;
    guint zero2;
    guint zero3;
} MicroProfBlock008e;

typedef struct {
    guint zero1;
} MicroProfBlock008f;

typedef struct {
    guint zero1;
    gdouble float1;     /* Always 1e-5. */
    guint zero2;
    guint int1;         /* Always 1. */
    gdouble float2;     /* Always 0.001. */
    guint zero3;
    gdouble float3;     /* 60 or 90. */
    gdouble float4;     /* 2 or 5. */
} MicroProfBlock0090;

typedef struct {
    guint int1;
    gdouble float1;     /* Always 10000. */
    gdouble float2;     /* Always 10. */
    gdouble float3;     /* Always 200. */
    guint zero1;
    guint zero2;
    guint zero3;
} MicroProfBlockSub0091;

typedef struct {
    gdouble zero1;
    gdouble float1;     /* Always 0.0001. */
    gdouble float2;     /* Always 0.0001. */
    gdouble zero2;
    guint int1;         /* Always 4. */
    MicroProfBlockSub0091 subblock[4];    /* The same thing repeated 4 times. */
} MicroProfBlock0091;

typedef struct {
    guchar zeros[264];
} MicroProfBlock0092;

typedef struct {
    guint int1;         /* 0 or 1. */
    guint zero1;
} MicroProfBlock0093;

typedef struct {
    gdouble float1;     /* Always 0.0078125. */
    gdouble float2;     /* Always NaN. */
    guint int1;         /* Always 80. */
    guchar zeros[80];
    guint int2;         /* Always 1. */
} MicroProfBlock0094;

typedef struct {
    guint int1;
    guint int2;
    gdouble float1;     /* Always 1. */
} MicroProfBlock0095;

typedef struct {
    guint zero1;
    guint zero2;
    guint zero3;
    gdouble float1;     /* Mostly zero, sometimes number of the order 10^{-4}. */
} MicroProfBlock0096;

typedef struct {
    guchar zeros[20];
} MicroProfBlock0097;

typedef struct {
    gdouble float1;     /* Small number of the order 0.001 to 0.1. */
} MicroProfBlock0098;

typedef struct {
    guint zero1;
    gdouble float1;     /* 100.016, 1000.16 or 300.047. */
    gdouble float2;     /* 30 or 40. */
} MicroProfBlock0099;

typedef struct {
    guint int1;         /* 0 or 1; indicates if floats are non-zero. */
    guint int2;         /* 4 or 6. */
    guint int3;         /* A nice decimal number like 25, 30, 35 or 50. */
    gdouble float1;     /* Large number in the order of 10^{3} to 10^{4}. */
    gdouble float2;     /* Large number in the order of 10^{3} to 10^{4}. */
    gdouble float3;     /* The same as float2. */
    gdouble zscale;     /* Equal to zscale (if nonzero). */
} MicroProfBlock009a;

typedef struct {
    gdouble zero1;
    gdouble float1;     /* Small number like 0.034 or 0.043. */
    gdouble float2;     /* Small number like 0.031 or 0.1. */
    gdouble float3;     /* Zero or 360.  Some angle? */
} MicroProfBlock009b;

typedef struct {
    guint zero1;
    guint zero2;
} MicroProfBlock009c;

typedef struct {
    guint zero1;
} MicroProfBlock009d;

typedef struct {
    guint zero1;
    guint zero2;
    guint zero3;
} MicroProfBlock009e;

typedef struct {
    guint textlen;
    gchar *text;        /* m, if present.  Units?  */
} MicroProfBlock009f;

typedef struct {
    guchar bytes[22];   /* Byte 6 is sometimes 1; otherwise zeros. */
} MicroProfBlock00a0;

typedef struct {
    guint zero1;
    guint zero2;
} MicroProfBlock00a1;

/* An exact duplicate of block 0x0088, including content. */
typedef struct {
    guint zero1;
    guint int1;         /* Always 1. */
    guint int2;         /* Always 3. */
    guint bitmask1;     /* Always 0xffff. */
    guint zero2;
    gdouble float1;     /* Always -10. */
    gdouble float2;     /* Always 10. */
    gdouble float3;     /* Always -10. */
    gdouble float4;     /* Always 10. */
    guchar text[140];   /* Always m and then zeros.  Units? */
    gdouble float5;     /* Always 0.4. */
} MicroProfBlock00a2;

typedef struct {
    guint int1;         /* 0 or 1; indicates if float is non-zero. */
    guint int2;         /* 0 or 255; in sync with int1. */
    guint zero1;
    guint zero2;
    gdouble float1;     /* 0.001 if non-zero. */
    guchar zeros[24];
} MicroProfBlock00a3;

typedef struct {
    guint zero1;
} MicroProfBlock00a5;

typedef struct {
    guchar zeros[264];
} MicroProfBlock00a6;

/* Have too few of these for statistics... */
typedef struct {
    guint len;
    guint int1;         /* 2. */
    guint int2;         /* 1. */
    guint int3;         /* 1. */
    guint int4;         /* 1. */
    guchar text[256];   /* Sample?  Software?  The same as in 0x008e. */
    gdouble float1;     /* 0.00066. */
    gdouble float2;     /* Equal to zscale. */
    guint int5;         /* Approx 10*32768.  Can be two 16bit integers. */
    gdouble float3;     /* Sometimes like 1.17552e-05, sometimes strange. */
    guchar zeros1[16];
    guint int6;         /* 1. */
    guint int7;         /* Zero. */
    gdouble float4;     /* 0.0065. */
    guint int8;         /* 1 */
    guint int9;         /* Zero or something very strange. */
    gdouble float5;     /* Small negative number of the order 10^{-5}. */
    guchar zeros2[24];
} MicroProfBlock368Sub00a7;

typedef struct {
    guint len;
    gdouble float1;     /* 1000.16 (equal to value in 0x0099). */
    gdouble float2;     /* 40 (equal to value in 0x0099). */
    guint zero1;
    guint int1;         /* 0 or 1. */
    guint int2;         /* 1 or 8. */
    guint zero2;
    gdouble float3;     /* 1. */
} MicroProfBlock36Sub00a7;

typedef struct {
    guint int1;
    MicroProfBlock368Sub00a7 sub368[2];
    MicroProfBlock36Sub00a7 sub36[2];
} MicroProfBlock00a7;

typedef struct {
    gdouble zero1;
    gdouble float1;     /* Small signed number of the order 0.01 to 0.1. */
    gdouble float2;     /* Small signed number of the order 0.01 to 0.1. */
    gdouble float3;     /* Small signed number of the order 0.01 to 0.1. */
    gdouble float4;     /* Small signed number of the order 0.01 to 0.1. */
} MicroProfBlock00a8;

typedef struct {
    guint zero1;
    guint zero2;
} MicroProfBlock00a9;

typedef struct {
    guchar zeros1[267];
    gdouble float1;     /* 0.001. */
    gdouble float2;     /* 3e-6. */
    guchar zeros2[28];
} MicroProfBlock00aa;

/* Is this always exactly two strings? */
typedef struct {
    guint textlen1;
    gchar *text1;       /* User name (Administrator). */
    guint textlen2;
    gchar *text2;       /* Further user specs? (empty).  */
} MicroProfBlock00ac;

typedef struct {
    guint zero1;
} MicroProfBlock00ad;

typedef struct {
    guint int1;         /* 2. */
    guint int2;         /* Zero. */
    guint int3;         /* 1. */
    guint int4;         /* Zero. */
    guint int5;         /* 0xffffffff. */
    gdouble float1;     /* 1. */
    gdouble zero1;
} MicroProfBlock00ae;

typedef struct {
    guint zero1;
    guint zero2;
    guint zero3;
    guint zero4;
    guint int1;         /* 200. */
    guint zero5;
    guint zero6;
} MicroProfBlock00af;

typedef struct {
    guchar bytes[34];   /* Sequence of 00 00 00 00 01 (yes, period is 5). */
} MicroProfBlock00b0;

/* File (single + multi). */
typedef struct {
    gboolean *seen_blocks;
    guint int1;
    MicroProfBlock000b block000b;
    MicroProfBlock0065 block0065;
    MicroProfBlock0066 block0066;
    MicroProfBlock0067 block0067;
    MicroProfBlock0068 block0068;
    MicroProfBlock0069 block0069;
    MicroProfBlock006a block006a;
    MicroProfBlock006b block006b;
    MicroProfBlock006c block006c;
    MicroProfBlock006d block006d;
    MicroProfBlock006e block006e;
    MicroProfBlock006f block006f;
    MicroProfBlock0070 block0070;
    MicroProfBlock0071 block0071;
    MicroProfBlock0072 block0072;
    MicroProfBlock0073 block0073;
    MicroProfBlock0074 block0074;
    MicroProfBlock0075 block0075;
    MicroProfBlock0077 block0077;
    MicroProfBlock0079 block0079;
    MicroProfBlock007a block007a;
    MicroProfBlock007b block007b;
    MicroProfBlock007c block007c;
    MicroProfBlock007d block007d;
    MicroProfBlock007e block007e;
    MicroProfBlock007f block007f;
    MicroProfBlock0080 block0080;
    MicroProfBlock0081 block0081;
    MicroProfBlock0082 block0082;
    MicroProfBlock0083 block0083;
    MicroProfBlock0084 block0084;
    MicroProfBlock0086 block0086;
    MicroProfBlock0087 block0087;
    MicroProfBlock0088 block0088;
    MicroProfBlock0089 block0089;
    MicroProfBlock008a block008a;
    MicroProfBlock008b block008b;
    MicroProfBlock008c block008c;
    MicroProfBlock008d block008d;
    MicroProfBlock008e block008e;
    MicroProfBlock008f block008f;
    MicroProfBlock0090 block0090;
    MicroProfBlock0091 block0091;
    MicroProfBlock0092 block0092;
    MicroProfBlock0093 block0093;
    MicroProfBlock0094 block0094;
    MicroProfBlock0095 block0095;
    MicroProfBlock0096 block0096;
    MicroProfBlock0097 block0097;
    MicroProfBlock0098 block0098;
    MicroProfBlock0099 block0099;
    MicroProfBlock009a block009a;
    MicroProfBlock009b block009b;
    MicroProfBlock009c block009c;
    MicroProfBlock009d block009d;
    MicroProfBlock009e block009e;
    MicroProfBlock009f block009f;
    MicroProfBlock00a0 block00a0;
    MicroProfBlock00a1 block00a1;
    MicroProfBlock00a2 block00a2;
    MicroProfBlock00a3 block00a3;
    MicroProfBlock00a5 block00a5;
    MicroProfBlock00a6 block00a6;
    MicroProfBlock00a7 block00a7;
    MicroProfBlock00a8 block00a8;
    MicroProfBlock00a9 block00a9;
    MicroProfBlock00aa block00aa;
    MicroProfBlock00ac block00ac;
    MicroProfBlock00ad block00ad;
    MicroProfBlock00ae block00ae;
    MicroProfBlock00af block00af;
    MicroProfBlock00b0 block00b0;
} MicroProfFile;

static gboolean      module_register          (void);
static gint          microprof_detect         (const GwyFileDetectInfo *fileinfo,
                                               gboolean only_name);
static gint          microprof_txt_detect     (const GwyFileDetectInfo *fileinfo,
                                               gboolean only_name);
static GwyContainer* microprof_load           (const gchar *filename,
                                               GwyRunType mode,
                                               GError **error);
static void          microprof_file_free      (MicroProfFile *mfile);
static gboolean      read_blocks              (const guchar *buffer,
                                               gsize size,
                                               MicroProfFile *mfile,
                                               GError **error);
static gboolean      read_images_block        (const guchar *p,
                                               gsize size,
                                               MicroProfBlock007d *block,
                                               GError **error);
static gboolean      check_imgblock           (const MicroProfImageBlock *imgblock,
                                               gsize size,
                                               GError **error);
static void          microprof_read_data_field(GwyContainer *container,
                                               gint id,
                                               const MicroProfImageBlock *imgblock,
                                               gdouble xrange,
                                               gdouble yrange,
                                               gdouble zscale,
                                               const guchar *buffer);
static GwyContainer* create_meta              (const MicroProfFile *mfile);
static GwyContainer* microprof_txt_load       (const gchar *filename,
                                               GwyRunType mode,
                                               GError **error);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Imports MicroProf FRT profilometer data files."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2006",
};

GWY_MODULE_QUERY2(module_info, microprof)

static gboolean
module_register(void)
{
    gwy_file_func_register("microprof",
                           N_("MicroProf FRT files (.frt)"),
                           (GwyFileDetectFunc)&microprof_detect,
                           (GwyFileLoadFunc)&microprof_load,
                           NULL,
                           NULL);
    gwy_file_func_register("microprof_txt",
                           N_("MicroProf FRT text files (.txt)"),
                           (GwyFileDetectFunc)&microprof_txt_detect,
                           (GwyFileLoadFunc)&microprof_txt_load,
                           NULL,
                           NULL);

    return TRUE;
}

static gint
microprof_detect(const GwyFileDetectInfo *fileinfo,
                 gboolean only_name)
{
    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION) ? 10 : 0;

    if (fileinfo->file_size < MICROPROF_MIN_HEADER_SIZE
        || fileinfo->buffer_len < MAGIC_SIZE
        || memcmp(fileinfo->head, MAGIC, MAGIC_SIZE) != 0)
        return 0;

    if (fileinfo->buffer_len > LONGMAGIC_SIZE
        && memcmp(fileinfo->head, LONGMAGIC, LONGMAGIC_SIZE) == 0)
        return 100;

    return 80;
}

static gint
microprof_txt_detect(const GwyFileDetectInfo *fileinfo,
                     gboolean only_name)
{
    GwyTextHeaderParser parser;
    GHashTable *meta;
    const guchar *p;
    gchar *buffer;
    gsize size;
    gint score = 0;

    if (only_name)
        return g_str_has_suffix(fileinfo->name_lowercase, EXTENSION_TXT) ? 10 : 0;

    if (fileinfo->buffer_len < MICROPROF_MIN_TEXT_SIZE
        || memcmp(fileinfo->head, MAGIC_TXT, MAGIC_TXT_SIZE) != 0)
        return 0;

    if (!(p = strstr(fileinfo->head, "\n\n"))
        && !(p = strstr(fileinfo->head, "\r\r"))
        && !(p = strstr(fileinfo->head, "\r\n\r\n")))
        return 0;

    size = p - (const guchar*)fileinfo->head;
    buffer = g_memdup(fileinfo->head, size);
    gwy_clear(&parser, 1);
    parser.key_value_separator = "=";
    meta = gwy_text_header_parse(buffer, &parser, NULL, NULL);
    if (g_hash_table_lookup(meta, "XSize")
        && g_hash_table_lookup(meta, "YSize")
        && g_hash_table_lookup(meta, "XRange")
        && g_hash_table_lookup(meta, "YRange")
        && g_hash_table_lookup(meta, "ZScale"))
        score = 90;

    g_free(buffer);
    if (meta)
        g_hash_table_destroy(meta);

    return score;
}

static GwyContainer*
microprof_load(const gchar *filename,
               G_GNUC_UNUSED GwyRunType mode,
               GError **error)
{
    GwyContainer *container = NULL, *meta = NULL;
    guchar *buffer = NULL;
    const guchar *p;
    gsize size = 0;
    MicroProfFile mfile;
    GError *err = NULL;
    GQuark quark;
    guint i;

    if (!gwy_file_get_contents(filename, &buffer, &size, &err)) {
        err_GET_FILE_CONTENTS(error, &err);
        return NULL;
    }

    if (size < MICROPROF_MIN_HEADER_SIZE) {
        err_TOO_SHORT(error);
        return NULL;
    }
    if (memcmp(buffer, MAGIC, MAGIC_SIZE) != 0) {
        err_FILE_TYPE(error, "MicroProf");
        return NULL;
    }

    gwy_clear(&mfile, 1);
    mfile.seen_blocks = g_new0(gboolean, MAX_BLOCK_ID);
    p = buffer + LONGMAGIC_SIZE;
    mfile.int1 = gwy_get_guint16_le(&p);
    if (!read_blocks(p, size - (p - buffer), &mfile, error))
        goto fail;

    if (!(mfile.block0067.xrange = fabs(mfile.block0067.xrange))) {
        g_warning("Real x size is 0.0, fixing to 1.0");
        mfile.block0067.xrange = 1.0;
    }
    if (!(mfile.block0067.yrange = fabs(mfile.block0067.yrange))) {
        g_warning("Real x size is 0.0, fixing to 1.0");
        mfile.block0067.yrange = 1.0;
    }

    container = gwy_container_new();
    meta = create_meta(&mfile);

    /* Create images.  If a multi-image part is present we throw away the
     * single image because it is duplicated in the multi-image part. */
    if (mfile.block007d.nimages) {
        for (i = 0; i < mfile.block007d.nimages; i++) {
            MicroProfImageBlock *imgblock = mfile.block007d.imgblocks + i;
            GwyContainer *tmpmeta;

            microprof_read_data_field(container, i, imgblock,
                                      mfile.block0067.xrange,
                                      mfile.block0067.yrange,
                                      mfile.block006c.zscale,
                                      mfile.block007d.data[i]);
            quark = gwy_app_get_data_meta_key_for_id(i);
            tmpmeta = gwy_container_duplicate(meta);
            gwy_container_set_object(container, quark, tmpmeta);
            g_object_unref(tmpmeta);
            gwy_file_channel_import_log_add(container, i, NULL, filename);
        }
    }
    else if (mfile.block000b.data) {
        MicroProfImageBlock imgblock;

        imgblock.datatype = MICROPROF_TOPOGRAPHY;
        imgblock.xres = mfile.block0066.xres;
        imgblock.yres = mfile.block0066.yres;
        imgblock.bpp = mfile.block0066.bpp;
        if (!check_imgblock(&imgblock, mfile.block000b.size, error))
            goto fail;

        microprof_read_data_field(container, 0, &imgblock,
                                  mfile.block0067.xrange,
                                  mfile.block0067.yrange,
                                  mfile.block006c.zscale,
                                  mfile.block000b.data);
        quark = gwy_app_get_data_meta_key_for_id(0);
        gwy_container_set_object(container, quark, meta);
        gwy_file_channel_import_log_add(container, 0, NULL, filename);
    }
    else {
        err_NO_DATA(error);
        goto fail;
    }

fail:
    GWY_OBJECT_UNREF(meta);
    gwy_file_abandon_contents(buffer, size, NULL);
    microprof_file_free(&mfile);
    return container;
}

static void
microprof_file_free(MicroProfFile *mfile)
{
    g_free(mfile->seen_blocks);
    g_free(mfile->block007d.imgblocks);
    g_free(mfile->block007d.data);
    g_free(mfile->block0065.text);
    g_free(mfile->block0077.text);
    g_free(mfile->block008e.text);
    g_free(mfile->block00ac.text1);
    g_free(mfile->block00ac.text2);
}

#ifdef DEBUG
static const gchar*
format_hexdump(GString *str, const guchar *p, guint len)
{
    g_string_assign(str, "data");

    while (len--) {
        g_string_append_printf(str, " %02x", *p);
        p++;
    }

    return str->str;
}
#endif

#define VS G_MAXUINT

static gboolean
read_blocks(const guchar *p, gsize size,
            MicroProfFile *mfile, GError **error)
{
    enum { BLOCK_HEADER_SIZE = 2 + 4 };

    static const guint block_sizes[MAX_BLOCK_ID] = {
      /* 0    1    2   3    4   5    6    7    8   9    a   b   c   d   e   f */
/* 0 */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0, VS,  0,  0,  0,   0,
/* 1 */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* 2 */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* 3 */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* 4 */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* 5 */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* 6 */  0,   0,   0,  0,   0, VS,  12,  44,  36, 16,  32, 16, 12, 32, 12,  16,
/* 7 */  8,   8,  12,  4,   8, VS,   0,  VS,   0, 16,   8,  8,  4, VS, 16, 256,
/* 8 */  4,  20,   4,  4,   4,  0,   4,   4, 196,  8,  12,  4, 22, 60, VS,   4,
/* 9 */ 48, 148, 264, 12, 104, 16,  20,  20,   8, 16,  44, 32,  8,  4, 12,  VS,
/* a */ 22,  10, 196, 44,   0,  4, 264, 828,  40,  8, 311,  0, VS,  4, 36,  28,
/* b */ 34,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* c */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* d */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* e */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
/* f */  0,   0,   0,  0,   0,  0,   0,   0,   0,  0,   0,  0,  0,  0,  0,   0,
    };

    GString *str = g_string_new(NULL);
    gboolean ok = FALSE;

    while (size >= BLOCK_HEADER_SIZE) {
        guint blocktype, blocksize;
        gboolean skipme = TRUE;
        const guchar *q;
        guint ii;

        blocktype = gwy_get_guint16_le(&p);
        blocksize = gwy_get_guint32_le(&p);
        q = p;
        size -= BLOCK_HEADER_SIZE;

        gwy_debug("block of type 0x%04x and size %u", blocktype, blocksize);
        if (blocksize > size) {
            gwy_debug("too long block, only %lu bytes remaining", (gulong)size);
            goto fail;
        }

#if 0
        gwy_debug("[%04x] rawdata %s",
                  blocktype, format_hexdump(str, q, MIN(blocksize, 4096)));
#endif

        if (blocktype >= MAX_BLOCK_ID) {
            g_warning("Too large block id %02x", blocktype);
        }
        else {
            guint expected_size = block_sizes[blocktype];
            if (expected_size) {
                if (expected_size == VS || blocksize == expected_size)
                    skipme = FALSE;
                else
                    g_warning("Wrong block %02x length %u (expecting %u)",
                              blocktype, blocksize, expected_size);
            }
        }

        if (skipme) {
            gwy_debug("unhandled block %s", format_hexdump(str, p, blocksize));
            p += blocksize;
            size -= blocksize;
            continue;
        }

        if (mfile->seen_blocks[blocktype]) {
            g_set_error(error, GWY_MODULE_FILE_ERROR,
                        GWY_MODULE_FILE_ERROR_DATA,
                        _("Duplicate block %02x."), blocktype);
            goto fail;
        }
        mfile->seen_blocks[blocktype] = TRUE;

        if (blocktype == 0x000b) {
            MicroProfBlock000b *block = &mfile->block000b;
            block->size = blocksize;
            block->data = q;
        }
        else if (blocktype == 0x0065 && blocksize > 0) {
            MicroProfBlock0065 *block = &mfile->block0065;
            block->text = g_strndup(q, blocksize);
            gwy_debug("[%04x] text \"%s\"", blocktype, block->text);
#if 0
            gwy_debug("[%04x] text \"%s\"",
                      blocktype,
                      gwy_strreplace(block->text, "\n", " ", (gsize)-1));
#endif
        }
        else if (blocktype == 0x0066) {
            MicroProfBlock0066 *block = &mfile->block0066;
            block->xres = gwy_get_guint32_le(&q);
            block->yres = gwy_get_guint32_le(&q);
            block->bpp = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] xres %u, yres %u, bpp %u",
                      blocktype,
                      block->xres, block->yres, block->bpp);
        }
        else if (blocktype == 0x0067) {
            MicroProfBlock0067 *block = &mfile->block0067;
            block->xrange = gwy_get_gdouble_le(&q);
            block->yrange = gwy_get_gdouble_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] xrange %g, yrange %g, "
                      "float1 %g, float2 %g, float3 %g, "
                      "int1 %u",
                      blocktype,
                      block->xrange, block->yrange,
                      block->float1, block->float2, block->float3,
                      block->int1);
        }
        else if (blocktype == 0x0068) {
            MicroProfBlock0068 *block = &mfile->block0068;
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] float1 %g, float2 %g, "
                      "zero1 %u, zero2 %u, zero3 %u, "
                      "int1 %u, int2 %u",
                      blocktype,
                      block->float1, block->float2,
                      block->zero1, block->zero2, block->zero3,
                      block->int1, block->int2);
        }
        else if (blocktype == 0x0069) {
            MicroProfBlock0069 *block = &mfile->block0069;
            block->bitmask1 = gwy_get_guint32_le(&q);
            block->bitmask2 = gwy_get_guint32_le(&q);
            block->bitmask3 = gwy_get_guint32_le(&q);
            block->bitmask4 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] bitmask1 0x%08x, bitmask2 0x%08x, "
                      "bitmask3 0x%08x, bitmask4 0x%08x",
                      blocktype,
                      block->bitmask1, block->bitmask2,
                      block->bitmask3, block->bitmask4);
        }
        else if (blocktype == 0x006a) {
            MicroProfBlock006a *block = &mfile->block006a;
            block->zero1 = gwy_get_gdouble_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->zero2 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %g, float1 %g, float2 %g, zero2 %g",
                      blocktype,
                      block->zero1, block->float1, block->float2, block->zero2);
        }
        else if (blocktype == 0x006b) {
            MicroProfBlock006b *block = &mfile->block006b;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->int3 = gwy_get_guint32_le(&q);
            block->int4 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u, int3 %u, int4 %u",
                      blocktype,
                      block->int1, block->int2, block->int3, block->int4);
        }
        else if (blocktype == 0x006c) {
            MicroProfBlock006c *block = &mfile->block006c;
            block->int1 = gwy_get_guint32_le(&q);
            block->zscale = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] int1 %u, zscale %g",
                      blocktype, block->int1, block->zscale);
        }
        else if (blocktype == 0x006d) {
            MicroProfBlock006d *block = &mfile->block006d;
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->int3 = gwy_get_guint32_le(&q);
            block->int4 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] float1 %g, float2 %g, "
                      "int1 %u, int2 %u, int3 %u, int4 %u",
                      blocktype,
                      block->float1, block->float2,
                      block->int1, block->int2, block->int3, block->int4);
        }
        else if (blocktype == 0x006e) {
            MicroProfBlock006e *block = &mfile->block006e;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->int3 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u, int3 %u",
                      blocktype, block->int1, block->int2, block->int3);
        }
        else if (blocktype == 0x006f) {
            MicroProfBlock006f *block = &mfile->block006f;
            block->zero1 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, int1 %u, zero2 %u, int2 %u",
                      blocktype,
                      block->zero1, block->int1, block->zero2, block->int2);
        }
        else if (blocktype == 0x0070) {
            MicroProfBlock0070 *block = &mfile->block0070;
            block->int1 = gwy_get_guint32_le(&q);
            block->bitmask1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u, bitmask1 0x%08x",
                      blocktype, block->int1, block->bitmask1);
        }
        else if (blocktype == 0x0071) {
            MicroProfBlock0071 *block = &mfile->block0071;
            block->float1 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] float1 %g", blocktype, block->float1);
        }
        else if (blocktype == 0x0072) {
            MicroProfBlock0072 *block = &mfile->block0072;
            block->timestamp1 = gwy_get_guint32_le(&q);
            block->timestamp2 = gwy_get_guint32_le(&q);
            block->meastime = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] timestamp1 %u, timestamp2 %u, meastime %u",
                      blocktype,
                      block->timestamp1, block->timestamp2, block->meastime);
        }
        else if (blocktype == 0x0073) {
            MicroProfBlock0073 *block = &mfile->block0073;
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u", blocktype, block->int1);
        }
        else if (blocktype == 0x0074) {
            MicroProfBlock0074 *block = &mfile->block0074;
            block->int1 = gwy_get_guint16_le(&q);
            block->int2 = gwy_get_guint16_le(&q);
            block->int3 = gwy_get_guint16_le(&q);
            block->int4 = gwy_get_guint16_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u, int3 %u, int4 %u",
                      blocktype,
                      block->int1, block->int2, block->int3, block->int4);
        }
        else if (blocktype == 0x0075 && blocksize >= 20) {
            MicroProfBlock0075 *block = &mfile->block0075;
            block->palettesize = gwy_get_guint32_le(&q);
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] palettesize %u, "
                      "zero1 %u, zero2 %u, int1 %u, int2 %u, "
                      "remainder %u bytes (expecting %u)",
                      blocktype,
                      block->palettesize,
                      block->zero1, block->zero2, block->int1, block->int2,
                      blocksize - 20, 4*block->palettesize);
        }
        else if (blocktype == 0x0077 && blocksize > 0) {
            MicroProfBlock0077 *block = &mfile->block0077;
            block->text = g_strndup(q, blocksize);
            gwy_debug("[%04x] text \"%s\"", blocktype, block->text);
        }
        else if (blocktype == 0x0079) {
            MicroProfBlock0092 *block = &mfile->block0092;
            get_CHARARRAY(block->zeros, &q);
        }
        else if (blocktype == 0x007a) {
            MicroProfBlock007a *block = &mfile->block007a;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u",
                      blocktype, block->int1, block->int2);
        }
        else if (blocktype == 0x007b) {
            MicroProfBlock007b *block = &mfile->block007b;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u",
                      blocktype, block->int1, block->int2);
        }
        else if (blocktype == 0x007c) {
            MicroProfBlock007c *block = &mfile->block007c;
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u", blocktype, block->int1);
        }
        else if (blocktype == 0x007d) {
            MicroProfBlock007d *block = &mfile->block007d;
            if (!read_images_block(p, blocksize, block, error))
                goto fail;
        }
        else if (blocktype == 0x007e) {
            MicroProfBlock007e *block = &mfile->block007e;
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] float1 %g, float2 %g",
                      blocktype, block->float1, block->float2);
        }
        else if (blocktype == 0x007f) {
            MicroProfBlock007f *block = &mfile->block007f;
            get_CHARARRAY0(block->calibration, &q);
            gwy_debug("[%04x] calibration \"%s\"",
                      blocktype, block->calibration);
        }
        else if (blocktype == 0x0080) {
            MicroProfBlock0080 *block = &mfile->block0080;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x0081) {
            MicroProfBlock0081 *block = &mfile->block0081;
            block->int1 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] int1 %u, float1 %g, float2 %g",
                      blocktype, block->int1, block->float1, block->float2);
        }
        else if (blocktype == 0x0082) {
            MicroProfBlock0082 *block = &mfile->block0082;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x0083) {
            MicroProfBlock0083 *block = &mfile->block0083;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x0084) {
            MicroProfBlock0084 *block = &mfile->block0084;
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u", blocktype, block->int1);
        }
        else if (blocktype == 0x0086) {
            MicroProfBlock0086 *block = &mfile->block0086;
            block->int1 = gwy_get_guint16_le(&q);
            block->int2 = gwy_get_guint16_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u",
                      blocktype, block->int1, block->int2);
        }
        else if (blocktype == 0x0087) {
            MicroProfBlock0087 *block = &mfile->block0087;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x0088) {
            MicroProfBlock0088 *block = &mfile->block0088;
            block->zero1 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->bitmask1 = gwy_get_guint16_le(&q);
            block->zero2 = gwy_get_guint16_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->float4 = gwy_get_gdouble_le(&q);
            get_CHARARRAY0(block->text, &q);
            block->float5 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %u, int1 %u, int2 %u, "
                      "bitmask1 0x%04x, zero2 %u, "
                      "float1 %g, float2 %g, "
                      "float3 %g, float4 %g, "
                      "text \"%s\", float5 %g",
                      blocktype, block->zero1, block->int1, block->int2,
                      block->bitmask1, block->zero2,
                      block->float1, block->float2,
                      block->float3, block->float4,
                      block->text, block->float5);
        }
        else if (blocktype == 0x0089) {
            MicroProfBlock0089 *block = &mfile->block0089;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u",
                      blocktype, block->zero1, block->zero2);
        }
        else if (blocktype == 0x008a) {
            MicroProfBlock008a *block = &mfile->block008a;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u, zero3 %u",
                      blocktype, block->zero1, block->zero2, block->zero3);
        }
        else if (blocktype == 0x008b) {
            MicroProfBlock008b *block = &mfile->block008b;
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u", blocktype, block->int1);
        }
        else if (blocktype == 0x008c) {
            MicroProfBlock008c *block = &mfile->block008c;
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->int1 = gwy_get_guint16_le(&q);
            block->float3 = gwy_get_gfloat_le(&q);
            gwy_debug("[%04x] float1 %g, float2 %g, "
                      "int1 %u, float3 %g",
                      blocktype, block->float1, block->float2,
                      block->int1, block->float3);
        }
        else if (blocktype == 0x008d) {
            MicroProfBlock008d *block = &mfile->block008d;
            block->zero1 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            block->zero4 = gwy_get_guint32_le(&q);
            block->zero5 = gwy_get_guint32_le(&q);
            block->zero6 = gwy_get_guint32_le(&q);
            block->zero7 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, float1 %g, float2 %g, "
                      "zero2 %u, float3 %g, "
                      "int1 %u, int2 %u, "
                      "zero3 %u, zero4 %u, zero5 %u, "
                      "zero6 %u, zero7 %u",
                      blocktype, block->zero1, block->float1, block->float2,
                      block->zero2, block->float3,
                      block->int1, block->int2,
                      block->zero3, block->zero4, block->zero5,
                      block->zero6, block->zero7);
        }
        else if (blocktype == 0x008e && blocksize >= 16) {
            MicroProfBlock008e *block = &mfile->block008e;
            block->textlen = gwy_get_guint32_le(&q);
            if (block->textlen == blocksize - 16) {
                block->text = g_strndup(q, block->textlen);
                q += block->textlen;
                block->zero1 = gwy_get_guint32_le(&q);
                block->zero2 = gwy_get_guint32_le(&q);
                block->zero3 = gwy_get_guint32_le(&q);
                gwy_debug("[%04x] text \"%s\", "
                          "zero1 %u, zero2 %u, zero3 %u",
                          blocktype, block->text,
                          block->zero1, block->zero2, block->zero3);
            }
        }
        else if (blocktype == 0x008f) {
            MicroProfBlock008f *block = &mfile->block008f;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x0090) {
            MicroProfBlock0090 *block = &mfile->block0090;
            block->zero1 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->float4 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %u, float1 %g, "
                      "zero2 %u, int1 %u, float2 %g, "
                      "zero3 %u, float3 %g, float4 %g",
                      blocktype, block->zero1, block->float1,
                      block->zero2, block->int1, block->float2,
                      block->zero3, block->float3, block->float4);
        }
        else if (blocktype == 0x0091) {
            MicroProfBlock0091 *block = &mfile->block0091;
            block->zero1 = gwy_get_gdouble_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->zero2 = gwy_get_gdouble_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %g, float1 %g, float2 %g, zero2 %g, "
                      "int1 %u",
                      blocktype,
                      block->zero1, block->float1, block->float2, block->zero2,
                      block->int1);
            for (ii = 0; ii < G_N_ELEMENTS(block->subblock); ii++) {
                MicroProfBlockSub0091 *sub = block->subblock + ii;
                sub->int1 = gwy_get_guint32_le(&q);
                sub->float1 = gwy_get_gfloat_le(&q);
                sub->float2 = gwy_get_gfloat_le(&q);
                sub->float3 = gwy_get_gfloat_le(&q);
                sub->zero1 = gwy_get_guint32_le(&q);
                sub->zero2 = gwy_get_guint32_le(&q);
                sub->zero3 = gwy_get_guint32_le(&q);
                gwy_debug("[%04x:%u] int1 %u, "
                          "float1 %g, float2 %g, float3 %g, "
                          "zero1 %u, zero2 %u, zero3 %u",
                          blocktype, ii, sub->int1,
                          sub->float1, sub->float2, sub->float3,
                          sub->zero1, sub->zero2, sub->zero3);
            }
        }
        else if (blocktype == 0x0092) {
            MicroProfBlock0092 *block = &mfile->block0092;
            get_CHARARRAY(block->zeros, &q);
        }
        else if (blocktype == 0x0093) {
            MicroProfBlock0093 *block = &mfile->block0093;
            block->int1 = gwy_get_guint32_le(&q);
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u, zero1 %u",
                      blocktype, block->int1, block->zero1);
        }
        else if (blocktype == 0x0094) {
            MicroProfBlock0094 *block = &mfile->block0094;
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            get_CHARARRAY(block->zeros, &q);
            block->int2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] float1 %g, float2 %g, int1 %u, int2 %u",
                      blocktype,
                      block->float1, block->float2, block->int1, block->int2);
        }
        else if (blocktype == 0x0095) {
            MicroProfBlock0095 *block = &mfile->block0095;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u, float1 %g",
                      blocktype, block->int1, block->int2, block->float1);
        }
        else if (blocktype == 0x0096) {
            MicroProfBlock0096 *block = &mfile->block0096;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u, zero3 %u, "
                      "float1 %g",
                      blocktype, block->zero1, block->zero2, block->zero3,
                      block->float1);
        }
        else if (blocktype == 0x0097) {
            MicroProfBlock0097 *block = &mfile->block0097;
            get_CHARARRAY(block->zeros, &q);
        }
        else if (blocktype == 0x0098) {
            MicroProfBlock0098 *block = &mfile->block0098;
            block->float1 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] float1 %g", blocktype, block->float1);
        }
        else if (blocktype == 0x0099) {
            MicroProfBlock0099 *block = &mfile->block0099;
            block->zero1 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gfloat_le(&q);
            gwy_debug("[%04x] zero1 %u, float1 %g, float2 %g",
                      blocktype, block->zero1, block->float1, block->float2);
        }
        else if (blocktype == 0x009a) {
            MicroProfBlock009a *block = &mfile->block009a;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->int3 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->zscale = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u, int3 %u, "
                      "float1 %g, float2 %g, float3 %g, zscale %g",
                      blocktype,
                      block->int1, block->int2, block->int3,
                      block->float1, block->float2,
                      block->float2, block->zscale);
        }
        else if (blocktype == 0x009b) {
            MicroProfBlock009b *block = &mfile->block009b;
            block->zero1 = gwy_get_gdouble_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %g, "
                      "float1 %g, float2 %g, float3 %g",
                      blocktype, block->zero1,
                      block->float1, block->float2, block->float3);
        }
        else if (blocktype == 0x009c) {
            MicroProfBlock009c *block = &mfile->block009c;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u",
                      blocktype, block->zero1, block->zero2);
        }
        else if (blocktype == 0x009d) {
            MicroProfBlock009d *block = &mfile->block009d;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x009e) {
            MicroProfBlock009e *block = &mfile->block009e;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u, zero3 %u",
                      blocktype, block->zero1, block->zero2, block->zero3);
        }
        else if (blocktype == 0x009f && blocksize >= 2) {
            MicroProfBlock009f *block = &mfile->block009f;
            block->textlen = gwy_get_guint16_le(&q);
            if (block->textlen <= blocksize - 2) {
                block->text = g_strndup(q, block->textlen);
                gwy_debug("[%04x] text \"%s\"", blocktype, block->text);
            }
        }
        else if (blocktype == 0x00a0) {
            MicroProfBlock00a0 *block = &mfile->block00a0;
            get_CHARARRAY(block->bytes, &q);
            gwy_debug("[%04x] byte6 %u",
                      blocktype, block->bytes[6]);
        }
        else if (blocktype == 0x00a1) {
            MicroProfBlock00a1 *block = &mfile->block00a1;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u",
                      blocktype, block->zero1, block->zero2);
        }
        else if (blocktype == 0x00a2) {
            MicroProfBlock00a2 *block = &mfile->block00a2;
            block->zero1 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->bitmask1 = gwy_get_guint16_le(&q);
            block->zero2 = gwy_get_guint16_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->float4 = gwy_get_gdouble_le(&q);
            get_CHARARRAY0(block->text, &q);
            block->float5 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %u, int1 %u, int2 %u, "
                      "bitmask1 0x%04x, zero2 %u, "
                      "float1 %g, float2 %g, "
                      "float3 %g, float4 %g, "
                      "text \"%s\", float5 %g",
                      blocktype, block->zero1, block->int1, block->int2,
                      block->bitmask1, block->zero2,
                      block->float1, block->float2,
                      block->float3, block->float4,
                      block->text, block->float5);
        }
        else if (blocktype == 0x00a3) {
            MicroProfBlock00a3 *block = &mfile->block00a3;
            block->int1 = gwy_get_guint16_le(&q);
            block->int2 = gwy_get_guint16_le(&q);
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            get_CHARARRAY0(block->zeros, &q);
            gwy_debug("[%04x] int1 %u, int2 %u, "
                      "zero1 %u, zero2 %u, float1 %g",
                      blocktype, block->int1, block->int2,
                      block->zero1, block->zero2, block->float1);
        }
        else if (blocktype == 0x00a5) {
            MicroProfBlock00a5 *block = &mfile->block00a5;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x00a6) {
            MicroProfBlock00a6 *block = &mfile->block00a6;
            get_CHARARRAY(block->zeros, &q);
        }
        else if (blocktype == 0x00a7) {
            /* The subblocks give their own sizes, but I have only seen a
             * fixed structure so far.  So read it so. */
            MicroProfBlock00a7 *block = &mfile->block00a7;
            block->int1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] int1 %u", blocktype, block->int1);
            for (ii = 0; ii < 2; ii++) {
                MicroProfBlock368Sub00a7 *sub368 = block->sub368 + ii;
                MicroProfBlock36Sub00a7 *sub36 = block->sub36 + ii;

                sub368->len = gwy_get_guint32_le(&q);
                sub368->int1 = gwy_get_guint32_le(&q);
                sub368->int2 = gwy_get_guint32_le(&q);
                sub368->int3 = gwy_get_guint32_le(&q);
                sub368->int4 = gwy_get_guint32_le(&q);
                get_CHARARRAY0(sub368->text, &q);
                sub368->float1 = gwy_get_gdouble_le(&q);
                sub368->float2 = gwy_get_gdouble_le(&q);
                sub368->int5 = gwy_get_guint32_le(&q);
                sub368->float3 = gwy_get_gfloat_le(&q);
                get_CHARARRAY(sub368->zeros1, &q);
                sub368->int6 = gwy_get_guint32_le(&q);
                sub368->int7 = gwy_get_guint32_le(&q);
                sub368->float4 = gwy_get_gdouble_le(&q);
                sub368->int8 = gwy_get_guint32_le(&q);
                sub368->int9 = gwy_get_guint32_le(&q);
                sub368->float5 = gwy_get_gdouble_le(&q);
                get_CHARARRAY(sub368->zeros2, &q);
                gwy_debug("[%04x:%u] len %u, "
                          "int1 %u, int2 %u, int3 %u, int4 %u, "
                          "text \"%s\", float1 %g, float2 %g, "
                          "int5 %u, float3 %g, int6 %u, int7 %u, "
                          "float4 %g, int8 %u, int9 %u, float5 %g",
                        blocktype, 2*ii, sub368->len,
                        sub368->int1, sub368->int2, sub368->int3, sub368->int4,
                        sub368->text, sub368->float1, sub368->float2,
                        sub368->int5, sub368->float3,
                        sub368->int6, sub368->int7, sub368->float4,
                        sub368->int8, sub368->int9, sub368->float5);

                sub36->len = gwy_get_guint32_le(&q);
                sub36->float1 = gwy_get_gdouble_le(&q);
                sub36->float2 = gwy_get_gfloat_le(&q);
                sub36->zero1 = gwy_get_guint32_le(&q);
                sub36->int1 = gwy_get_guint32_le(&q);
                sub36->int2 = gwy_get_guint32_le(&q);
                sub36->zero2 = gwy_get_guint32_le(&q);
                sub36->float3 = gwy_get_gdouble_le(&q);
                gwy_debug("[%04x:%u] len %u, "
                          "float1 %g, float2 %g, "
                          "zero1 %u, int1 %u, int2 %u, zero2 %u, "
                          "float3 %g",
                        blocktype, 2*ii+1, sub36->len,
                        sub36->float1, sub36->float2,
                        sub36->zero1, sub36->int1, sub36->int2, sub36->zero2,
                        sub36->float3);
            }
        }
        else if (blocktype == 0x00a8) {
            MicroProfBlock00a8 *block = &mfile->block00a8;
            block->zero1 = gwy_get_gdouble_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            block->float3 = gwy_get_gdouble_le(&q);
            block->float4 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] zero1 %g, "
                      "float1 %g, float2 %g, float3 %g, float4 %g",
                      blocktype, block->zero1,
                      block->float1, block->float2,
                      block->float3, block->float4);
        }
        else if (blocktype == 0x00a9) {
            MicroProfBlock00a9 *block = &mfile->block00a9;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u",
                      blocktype, block->zero1, block->zero2);
        }
        else if (blocktype == 0x00aa) {
            MicroProfBlock00aa *block = &mfile->block00aa;
            get_CHARARRAY(block->zeros1, &q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->float2 = gwy_get_gdouble_le(&q);
            get_CHARARRAY(block->zeros2, &q);
            gwy_debug("[%04x] float1 %g, float2 %g",
                      blocktype, block->float1, block->float2);
        }
        else if (blocktype == 0x00ac && blocksize >= 8) {
            MicroProfBlock00ac *block = &mfile->block00ac;
            block->textlen1 = gwy_get_guint32_le(&q);
            if (block->textlen1 <= blocksize - 8) {
                block->text1 = g_strndup(q, block->textlen1);
                q += block->textlen1;
                block->textlen2 = gwy_get_guint32_le(&q);
                if (block->textlen2 <= blocksize - 8 - block->textlen1) {
                    block->text2 = g_strndup(q, block->textlen2);
                    q += block->textlen2;
                    gwy_debug("[%04x] text1 \"%s\", text2 \"%s\"",
                              blocktype, block->text1, block->text2);
                }
            }
        }
        else if (blocktype == 0x00ad) {
            MicroProfBlock00ad *block = &mfile->block00ad;
            block->zero1 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u", blocktype, block->zero1);
        }
        else if (blocktype == 0x00ae) {
            MicroProfBlock00ae *block = &mfile->block00ae;
            block->int1 = gwy_get_guint32_le(&q);
            block->int2 = gwy_get_guint32_le(&q);
            block->int3 = gwy_get_guint32_le(&q);
            block->int4 = gwy_get_guint32_le(&q);
            block->int5 = gwy_get_guint32_le(&q);
            block->float1 = gwy_get_gdouble_le(&q);
            block->zero1 = gwy_get_gdouble_le(&q);
            gwy_debug("[%04x] int1 %u, int2 %u, int3 %u, int4 %u, int5 %u, "
                      "float1 %g, zero1 %g",
                      blocktype,
                      block->int1, block->int2, block->int3, block->int4,
                      block->int5,
                      block->float1, block->zero1);
        }
        else if (blocktype == 0x00af) {
            MicroProfBlock00af *block = &mfile->block00af;
            block->zero1 = gwy_get_guint32_le(&q);
            block->zero2 = gwy_get_guint32_le(&q);
            block->zero3 = gwy_get_guint32_le(&q);
            block->zero4 = gwy_get_guint32_le(&q);
            block->int1 = gwy_get_guint32_le(&q);
            block->zero5 = gwy_get_guint32_le(&q);
            block->zero6 = gwy_get_guint32_le(&q);
            gwy_debug("[%04x] zero1 %u, zero2 %u, zero3 %u, "
                      "int1 %u, zero4 %u, zero5 %u",
                      blocktype,
                      block->zero1, block->zero2, block->zero3,
                      block->int1, block->zero4, block->zero5);
        }
        else if (blocktype == 0x00b0) {
            MicroProfBlock00b0 *block = &mfile->block00b0;
            get_CHARARRAY(block->bytes, &q);
            gwy_debug("[%04x] %s",
                      blocktype, format_hexdump(str, block->bytes, blocksize));
        }
        else {
            g_warning("Failure in reading variable-sized block %04x?",
                      blocktype);
        }

        p += blocksize;
        size -= blocksize;
    }
    ok = TRUE;

fail:
    g_string_free(str, TRUE);

    return ok;
}

static gboolean
read_images_block(const guchar *p, gsize size,
                  MicroProfBlock007d *block,
                  GError **error)
{
    enum {
        BLOCK_HEADER_PREFIX_SIZE = 2*4,
        IMAGE_BLOCK_SIZE = 4*4,
    };

    GArray *imgblocks;
    GPtrArray *imgdata;

    if (size < BLOCK_HEADER_PREFIX_SIZE) {
        err_TRUNCATED_PART(error, "block 0x7d");
        return FALSE;
    }

    block->int1 = gwy_get_guint16_le(&p);
    block->int2 = gwy_get_guint16_le(&p);
    block->int3 = gwy_get_guint16_le(&p);
    block->int4 = gwy_get_guint16_le(&p);
    size -= BLOCK_HEADER_PREFIX_SIZE;
    gwy_debug("[%04x] int1 %u, int2 %u, int3 %u, int4 %u",
              0x7d, block->int1, block->int2, block->int3, block->int4);

    imgblocks = g_array_new(FALSE, FALSE, sizeof(MicroProfImageBlock));
    imgdata = g_ptr_array_new();
    while (size > IMAGE_BLOCK_SIZE) {
        MicroProfImageBlock imgblock;
        gulong datasize;

        imgblock.datatype = gwy_get_guint32_le(&p);
        imgblock.xres = gwy_get_guint32_le(&p);
        imgblock.yres = gwy_get_guint32_le(&p);
        imgblock.bpp = gwy_get_guint32_le(&p);
        gwy_debug("[%04x:%u] datatype 0x%04x, xres %u, yres %u, bpp %u",
                  0x7d, (guint)imgblocks->len,
                  imgblock.datatype,
                  imgblock.xres, imgblock.yres, imgblock.bpp);
        size -= IMAGE_BLOCK_SIZE;
        if (!check_imgblock(&imgblock, size, error))
            goto fail;

        g_array_append_val(imgblocks, imgblock);
        g_ptr_array_add(imgdata, (gpointer)p);
        datasize = imgblock.xres * imgblock.yres * (imgblock.bpp/8);
        p += datasize;
        size -= datasize;
    }

    if (size > 0)
        g_warning("Images data block was not fully consumed.");

    block->nimages = imgblocks->len;
    block->imgblocks = (MicroProfImageBlock*)g_array_free(imgblocks, FALSE);
    block->data = (const guchar**)g_ptr_array_free(imgdata, FALSE);
    return TRUE;

fail:
    g_array_free(imgblocks, TRUE);
    g_ptr_array_free(imgdata, TRUE);
    return FALSE;
}

static gboolean
check_imgblock(const MicroProfImageBlock *imgblock, gsize size,
               GError **error)
{
    gsize datasize;

    if (imgblock->bpp != 16 && imgblock->bpp != 32) {
        err_BPP(error, imgblock->bpp);
        return FALSE;
    }

    if (err_DIMENSION(error, imgblock->xres)
        || err_DIMENSION(error, imgblock->yres))
        return FALSE;

    datasize = imgblock->xres * imgblock->yres * (imgblock->bpp/8);
    if (err_SIZE_MISMATCH(error, datasize, size, FALSE))
        return FALSE;

    return TRUE;
}

static void
microprof_read_data_field(GwyContainer *container, gint id,
                          const MicroProfImageBlock *imgblock,
                          gdouble xrange, gdouble yrange, gdouble zscale,
                          const guchar *buffer)
{
    guint i, n = imgblock->xres*imgblock->yres, bpp = imgblock->bpp;
    GwyRawDataType datatype;
    GwyDataField *dfield, *mask = NULL;
    GQuark quark;
    GString *title;
    gdouble *d, *m;

    dfield = gwy_data_field_new(imgblock->xres, imgblock->yres, xrange, yrange,
                                FALSE);
    gwy_debug("bpp %d", bpp);
    datatype = (bpp == 16 ? GWY_RAW_DATA_UINT16 : GWY_RAW_DATA_UINT32);
    d = gwy_data_field_get_data(dfield);
    gwy_convert_raw_data(buffer, n, 1,
                         datatype, GWY_BYTE_ORDER_LITTLE_ENDIAN, d, 1.0, 0.0);

    gwy_data_field_invert(dfield, TRUE, FALSE, FALSE);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_xy(dfield), "m");

    /* XXX: Invalid data seem to be marked by the special value 1. */
    d = gwy_data_field_get_data(dfield);
    for (i = 0; i < n; i++) {
        if (d[i] == 1.0)
            break;
    }
    if (i < n) {
        mask = gwy_data_field_new_alike(dfield, TRUE);
        m = gwy_data_field_get_data(mask);
        for (i = 0; i < n; i++) {
            if (d[i] != 1.0)
                m[i] = 1.0;
        }
        gwy_app_channel_remove_bad_data(dfield, mask);
    }

    if (!(imgblock->datatype & MICROPROF_INTENSITY)) {
        gwy_data_field_multiply(dfield, zscale);
        gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(dfield), "m");
    }

    quark = gwy_app_get_data_key_for_id(id);
    gwy_container_set_object(container, quark, dfield);
    g_object_unref(dfield);

    if (mask) {
        quark = gwy_app_get_mask_key_for_id(id);
        gwy_container_set_object(container, quark, mask);
        g_object_unref(mask);
    }

    title = g_string_new(NULL);
    if (imgblock->datatype & MICROPROF_INTENSITY)
        g_string_append(title, " Intensity");
    if (imgblock->datatype & MICROPROF_TOPOGRAPHY)
        g_string_append(title, " Height");
    if (imgblock->datatype & MICROPROF_THICKNESS)
        g_string_append(title, " Thickness");
    if (imgblock->datatype & MICROPROF_BOTTOM_SENSOR)
        g_string_append(title, " (bottom sensor)");
    if (title->len) {
        g_string_erase(title, 0, 1);
        quark = gwy_app_get_data_title_key_for_id(id);
        gwy_container_set_const_string(container, quark, title->str);
    }
    else
        gwy_app_channel_title_fall_back(container, id);
    g_string_free(title, TRUE);
}

static void
set_meta_string(GwyContainer *meta, const gchar *name, GString *str)
{
    g_strstrip(str->str);
    if ((str->len = strlen(str->str)))
        gwy_container_set_const_string_by_name(meta, name, str->str);
}

static GwyContainer*
create_meta(const MicroProfFile *mfile)
{
    GwyContainer *meta = gwy_container_new();
    GString *str = g_string_new(NULL);

    if (mfile->block0065.text) {
        g_string_assign(str, mfile->block0065.text);
        set_meta_string(meta, "Comment 1", str);
    }
    if (mfile->block0077.text) {
        g_string_assign(str, mfile->block0077.text);
        set_meta_string(meta, "Comment 2", str);
    }

    g_string_assign(str, mfile->block007f.calibration);
    set_meta_string(meta, "Calibration", str);

    if (mfile->block008e.text) {
        g_string_assign(str, mfile->block008e.text);
        set_meta_string(meta, "Sample", str);
    }

    if (mfile->block00ac.text1) {
        g_string_assign(str, mfile->block00ac.text1);
        set_meta_string(meta, "User", str);
    }
    if (mfile->block00ac.text2) {
        g_string_assign(str, mfile->block00ac.text2);
        set_meta_string(meta, "User2", str);
    }

    if (mfile->seen_blocks[0x0072]) {
        time_t t1 = mfile->block0072.timestamp1,
               t2 = mfile->block0072.timestamp2;

        g_string_assign(str, ctime(&t1));
        set_meta_string(meta, "Task started", str);
        g_string_assign(str, ctime(&t2));
        set_meta_string(meta, "Task finished", str);
        g_string_printf(str, "%u s", mfile->block0072.meastime);
        set_meta_string(meta, "Measurement time", str);
    }

    g_string_free(str, TRUE);

    return meta;
}

static GwyContainer*
microprof_txt_load(const gchar *filename,
                   G_GNUC_UNUSED GwyRunType mode,
                   GError **error)
{
    GwyContainer *container = NULL;
    guchar *p, *buffer = NULL;
    GwyTextHeaderParser parser;
    GHashTable *meta = NULL;
    GwySIUnit *siunit;
    gchar *header = NULL, *s, *prev;
    gsize size = 0;
    GError *err = NULL;
    GwyDataField *dfield = NULL;
    gdouble xreal, yreal, zscale, v;
    gint hlines, xres, yres, i, j;
    gdouble *d;

    if (!gwy_file_get_contents(filename, &buffer, &size, &err)) {
        err_GET_FILE_CONTENTS(error, &err);
        return NULL;
    }

    if (size < MICROPROF_MIN_TEXT_SIZE
        || memcmp(buffer, MAGIC_TXT, MAGIC_TXT_SIZE) != 0) {
        err_FILE_TYPE(error, "MicroProf");
        goto fail;
    }

    hlines = atoi(buffer + MAGIC_TXT_SIZE);
    if (hlines < 7) {
        err_FILE_TYPE(error, "MicroProf");
        goto fail;
    }

    /* Skip specified number of lines */
    for (p = buffer, i = 0; i < hlines; i++) {
        while (*p != '\n' && (gsize)(p - buffer) < size)
            p++;
        if ((gsize)(p - buffer) == size) {
            err_FILE_TYPE(error, "MicroProf");
            goto fail;
        }
        /* Now skip the \n */
        p++;
    }

    header = g_memdup(buffer, p - buffer + 1);
    header[p - buffer] = '\0';

    gwy_clear(&parser, 1);
    parser.key_value_separator = "=";
    meta = gwy_text_header_parse(header, &parser, NULL, NULL);

    if (!(s = g_hash_table_lookup(meta, "XSize"))
        || !((xres = atoi(s)) > 0)) {
        err_INVALID(error, "XSize");
        goto fail;
    }

    if (!(s = g_hash_table_lookup(meta, "YSize"))
        || !((yres = atoi(s)) > 0)) {
        err_INVALID(error, "YSize");
        goto fail;
    }

    if (!(s = g_hash_table_lookup(meta, "XRange"))
        || !((xreal = g_ascii_strtod(s, NULL)) > 0.0)) {
        err_INVALID(error, "YRange");
        goto fail;
    }

    if (!(s = g_hash_table_lookup(meta, "YRange"))
        || !((yreal = g_ascii_strtod(s, NULL)) > 0.0)) {
        err_INVALID(error, "YRange");
        goto fail;
    }

    if (!(s = g_hash_table_lookup(meta, "ZScale"))
        || !((zscale = g_ascii_strtod(s, NULL)) > 0.0)) {
        err_INVALID(error, "ZScale");
        goto fail;
    }

    dfield = gwy_data_field_new(xres, yres, xreal, yreal, FALSE);
    d = gwy_data_field_get_data(dfield);
    s = (gchar*)p;
    for (i = 0; i < yres; i++) {
        for (j = 0; j < xres; j++) {
            prev = s;
            /* Skip x */
            v = strtol(s, &s, 10);
            if (v != j)
                g_warning("Column number mismatch");
            /* Skip y */
            v = strtol(s, &s, 10);
            if (v != i)
                g_warning("Row number mismatch");
            /* Read value */
            d[(yres-1 - i)*xres + j] = strtol(s, &s, 10)*zscale;

            /* Check whether we moved in the file */
            if (s == prev) {
                g_set_error(error, GWY_MODULE_FILE_ERROR,
                            GWY_MODULE_FILE_ERROR_DATA,
                            _("File contains fewer than XSize*YSize data "
                              "points."));
                goto fail;
            }
        }
    }

    siunit = gwy_si_unit_new("m");
    gwy_data_field_set_si_unit_xy(dfield, siunit);
    g_object_unref(siunit);

    siunit = gwy_si_unit_new("m");
    gwy_data_field_set_si_unit_z(dfield, siunit);
    g_object_unref(siunit);

    container = gwy_container_new();
    gwy_container_set_object_by_name(container, "/0/data", dfield);
    g_object_unref(dfield);
    gwy_container_set_string_by_name(container, "/0/data/title",
                                     g_strdup("Topography"));

    gwy_file_channel_import_log_add(container, 0, NULL, filename);

fail:
    gwy_file_abandon_contents(buffer, size, NULL);
    if (meta)
        g_hash_table_destroy(meta);
    g_free(header);

    return container;
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
