﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Text.Unicode;
using System.Threading.Tasks;
using System.Xml.Linq;
using Xunit;

namespace System.Text.RegularExpressions.Tests
{
    /// <summary>
    /// These tests have been ported from the pcre2 test suite located at https://github.com/PCRE2Project/pcre2/tree/8b133fa0ba0a1817ea45df2e8e2ad642501e2fcf/testdata
    /// in order to increase .NET's test coverage. You can find the relevant repo license in this folder's THIRD-PARTY-NOTICES.TXT file.
    /// </summary>
    public class RegexPcre2Tests
    {
        public static IEnumerable<object[]> PcreTestData()
        {
            foreach (RegexEngine engine in RegexHelpers.AvailableEngines)
            {
                (string pattern, RegexOptions options, string input, bool expectedSuccess)[] cases = PcreTestData_Cases(engine).ToArray();
                Regex[] regexes = RegexHelpers.GetRegexesAsync(engine, cases.Select(c => (c.pattern, (CultureInfo?)null, (RegexOptions?)c.options, (TimeSpan?)null)).ToArray()).Result;
                for (int i = 0; i < regexes.Length; i++)
                {
                    yield return new object[] { regexes[i], cases[i].input, cases[i].expectedSuccess };
                }
            }
        }

        public static IEnumerable<(string Pattern, RegexOptions options, string Input, bool ExpectedSuccess)> PcreTestData_Cases(RegexEngine engine)
        {
            yield return ("the quick brown fox", RegexOptions.None, "the quick brown fox", true);
            yield return ("the quick brown fox", RegexOptions.None, "What do you know about the quick brown fox?", true);
            yield return ("the quick brown fox", RegexOptions.None, "The quick brown FOX", false);
            yield return ("the quick brown fox", RegexOptions.None, "What do you know about THE QUICK BROWN FOX?", false);
            yield return ("The quick brown fox", RegexOptions.IgnoreCase, "the quick brown fox", true);
            yield return ("The quick brown fox", RegexOptions.IgnoreCase, "The quick brown FOX", true);
            yield return ("The quick brown fox", RegexOptions.IgnoreCase, "What do you know about the quick brown fox?", true);
            yield return ("The quick brown fox", RegexOptions.IgnoreCase, "What do you know about THE QUICK BROWN FOX?", true);
            yield return ("abcd\\t\\n\\r\\f\\a\\e\\071\\x3b\\$\\\\\\?caxyz", RegexOptions.None, "abcd\t\n\r\f\a\u001b9;$\\?caxyz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aabxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abcxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aabcxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqqqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqqqqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqqqqqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqqqqqqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabcxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abxyzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aabxyzzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabxyzzzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabxyzzzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abcxyzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aabcxyzzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzzzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabcxyzzzzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabcxyzzzzpqrrrabbbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabcxyzzzzpqrrrabbbxyyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypABzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypABBzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, ">>>aaabxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, ">aaaabxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, ">>>>abcxyzpqrrrabbxyyyypqAzz", true);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abxyzpqrrabbxyyyypqAzz", false);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abxyzpqrrrrabbxyyyypqAzz", false);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "abxyzpqrrrabxyyyypqAzz", false);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabcxyzzzzpqrrrabbbxyyyyyypqAzz", false);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaaabcxyzzzzpqrrrabbbxyyypqAzz", false);
            yield return ("a*abc?xyz+pqr{3}ab{2,}xy{4,5}pq{0,6}AB{0,}zz", RegexOptions.None, "aaabcxyzpqrrrabbxyyyypqqqqqqqAzz", false);
            yield return ("^(abc){1,2}zz", RegexOptions.None, "abczz", true);
            yield return ("^(abc){1,2}zz", RegexOptions.None, "abcabczz", true);
            yield return ("^(abc){1,2}zz", RegexOptions.None, "zz", false);
            yield return ("^(abc){1,2}zz", RegexOptions.None, "abcabcabczz", false);
            yield return ("^(abc){1,2}zz", RegexOptions.None, ">>abczz", false);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bc", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbc", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbbc", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bac", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbac", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "aac", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "abbbbbbbbbbbc", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbbbbbbbbbbac", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "aaac", false);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "abbbbbbbbbbbac", false);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "bc", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "bbc", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "bbbc", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "bac", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "bbac", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "aac", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "abbbbbbbbbbbc", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "bbbbbbbbbbbac", true);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "aaac", false);
            yield return ("^(b+|a){1,2}c", RegexOptions.None, "abbbbbbbbbbbac", false);
            yield return ("^(ba|b*){1,2}?bc", RegexOptions.None, "babc", true);
            yield return ("^(ba|b*){1,2}?bc", RegexOptions.None, "bbabc", true);
            yield return ("^(ba|b*){1,2}?bc", RegexOptions.None, "bababc", true);
            yield return ("^(ba|b*){1,2}?bc", RegexOptions.None, "bababbc", false);
            yield return ("^(ba|b*){1,2}?bc", RegexOptions.None, "babababc", false);
            yield return ("^[ab\\]cde]", RegexOptions.None, "athing", true);
            yield return ("^[ab\\]cde]", RegexOptions.None, "bthing", true);
            yield return ("^[ab\\]cde]", RegexOptions.None, "]thing", true);
            yield return ("^[ab\\]cde]", RegexOptions.None, "cthing", true);
            yield return ("^[ab\\]cde]", RegexOptions.None, "dthing", true);
            yield return ("^[ab\\]cde]", RegexOptions.None, "ething", true);
            yield return ("^[ab\\]cde]", RegexOptions.None, "fthing", false);
            yield return ("^[ab\\]cde]", RegexOptions.None, "[thing", false);
            yield return ("^[ab\\]cde]", RegexOptions.None, "\\thing", false);
            yield return ("^[]cde]", RegexOptions.None, "]thing", true);
            yield return ("^[]cde]", RegexOptions.None, "cthing", true);
            yield return ("^[]cde]", RegexOptions.None, "dthing", true);
            yield return ("^[]cde]", RegexOptions.None, "ething", true);
            yield return ("^[]cde]", RegexOptions.None, "athing", false);
            yield return ("^[]cde]", RegexOptions.None, "fthing", false);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "fthing", true);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "[thing", true);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "\\thing", true);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "athing", false);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "bthing", false);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "]thing", false);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "cthing", false);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "dthing", false);
            yield return ("^[^ab\\]cde]", RegexOptions.None, "ething", false);
            yield return ("^[^]cde]", RegexOptions.None, "athing", true);
            yield return ("^[^]cde]", RegexOptions.None, "fthing", true);
            yield return ("^[^]cde]", RegexOptions.None, "]thing", false);
            yield return ("^[^]cde]", RegexOptions.None, "cthing", false);
            yield return ("^[^]cde]", RegexOptions.None, "dthing", false);
            yield return ("^[^]cde]", RegexOptions.None, "ething", false);
            yield return ("^\\\u0081", RegexOptions.None, "\u0081", true);
            yield return ("^\u00FF", RegexOptions.None, "\u00FF", true);
            yield return ("^[0-9]+$", RegexOptions.None, "0", true);
            yield return ("^[0-9]+$", RegexOptions.None, "1", true);
            yield return ("^[0-9]+$", RegexOptions.None, "2", true);
            yield return ("^[0-9]+$", RegexOptions.None, "3", true);
            yield return ("^[0-9]+$", RegexOptions.None, "4", true);
            yield return ("^[0-9]+$", RegexOptions.None, "5", true);
            yield return ("^[0-9]+$", RegexOptions.None, "6", true);
            yield return ("^[0-9]+$", RegexOptions.None, "7", true);
            yield return ("^[0-9]+$", RegexOptions.None, "8", true);
            yield return ("^[0-9]+$", RegexOptions.None, "9", true);
            yield return ("^[0-9]+$", RegexOptions.None, "10", true);
            yield return ("^[0-9]+$", RegexOptions.None, "100", true);
            yield return ("^[0-9]+$", RegexOptions.None, "abc", false);
            yield return ("^.*nter", RegexOptions.None, "enter", true);
            yield return ("^.*nter", RegexOptions.None, "inter", true);
            yield return ("^.*nter", RegexOptions.None, "uponter", true);
            yield return ("^xxx[0-9]+$", RegexOptions.None, "xxx0", true);
            yield return ("^xxx[0-9]+$", RegexOptions.None, "xxx1234", true);
            yield return ("^xxx[0-9]+$", RegexOptions.None, "xxx", false);
            yield return ("^.+[0-9][0-9][0-9]$", RegexOptions.None, "x123", true);
            yield return ("^.+[0-9][0-9][0-9]$", RegexOptions.None, "x1234", true);
            yield return ("^.+[0-9][0-9][0-9]$", RegexOptions.None, "xx123", true);
            yield return ("^.+[0-9][0-9][0-9]$", RegexOptions.None, "123456", true);
            yield return ("^.+[0-9][0-9][0-9]$", RegexOptions.None, "123", false);
            yield return ("^.+?[0-9][0-9][0-9]$", RegexOptions.None, "x123", true);
            yield return ("^.+?[0-9][0-9][0-9]$", RegexOptions.None, "x1234", true);
            yield return ("^.+?[0-9][0-9][0-9]$", RegexOptions.None, "xx123", true);
            yield return ("^.+?[0-9][0-9][0-9]$", RegexOptions.None, "123456", true);
            yield return ("^.+?[0-9][0-9][0-9]$", RegexOptions.None, "123", false);
            yield return ("^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$", RegexOptions.None, "abc!pqr=apquxz.ixr.zzz.ac.uk", true);
            yield return ("^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$", RegexOptions.None, "!pqr=apquxz.ixr.zzz.ac.uk", false);
            yield return ("^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$", RegexOptions.None, "abc!=apquxz.ixr.zzz.ac.uk", false);
            yield return ("^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$", RegexOptions.None, "abc!pqr=apquxz:ixr.zzz.ac.uk", false);
            yield return ("^([^!]+)!(.+)=apquxz\\.ixr\\.zzz\\.ac\\.uk$", RegexOptions.None, "abc!pqr=apquxz.ixr.zzz.ac.ukk", false);
            yield return (":", RegexOptions.None, "Well, we need a colon: somewhere", true);
            yield return (":", RegexOptions.None, "Fail without a colon", false);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "0abc", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "abc", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "fed", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "E", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "::", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "5f03:12C0::932e", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "fed def", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "Any old stuff", true);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "0zzz", false);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "gzzz", false);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "fed ", false);
            yield return ("([\\da-f:]+)$", RegexOptions.IgnoreCase, "Any old rubbish", false);
            yield return ("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", RegexOptions.None, ".1.2.3", true);
            yield return ("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", RegexOptions.None, "A.12.123.0", true);
            yield return ("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", RegexOptions.None, ".1.2.3333", false);
            yield return ("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", RegexOptions.None, "1.2.3", false);
            yield return ("^.*\\.(\\d{1,3})\\.(\\d{1,3})\\.(\\d{1,3})$", RegexOptions.None, "1234.2.3", false);
            yield return ("^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$", RegexOptions.None, "1 IN SOA non-sp1 non-sp2(", true);
            yield return ("^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$", RegexOptions.None, "1    IN    SOA    non-sp1    non-sp2   (", true);
            yield return ("^(\\d+)\\s+IN\\s+SOA\\s+(\\S+)\\s+(\\S+)\\s*\\(\\s*$", RegexOptions.None, "1IN SOA non-sp1 non-sp2(", false);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "a.", true);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "Z.", true);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "2.", true);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "ab-c.pq-r.", true);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "sxk.zzz.ac.uk.", true);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "x-.y-.", true);
            yield return ("^[a-zA-Z\\d][a-zA-Z\\d\\-]*(\\.[a-zA-Z\\d][a-zA-z\\d\\-]*)*\\.$", RegexOptions.None, "-abc.peq.", false);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.a", true);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.b0-a", true);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.c3-b.c", true);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.c-a.b-c", true);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.0", false);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.a-", false);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.a-b.c-", false);
            yield return ("^\\*\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?(\\.[a-z]([a-z\\-\\d]*[a-z\\d]+)?)*$", RegexOptions.None, "*.c-a.0-c", false);
            yield return ("^[\\da-f](\\.[\\da-f])*$", RegexOptions.IgnoreCase, "a.b.c.d", true);
            yield return ("^[\\da-f](\\.[\\da-f])*$", RegexOptions.IgnoreCase, "A.B.C.D", true);
            yield return ("^[\\da-f](\\.[\\da-f])*$", RegexOptions.IgnoreCase, "a.b.c.1.2.3.C", true);
            yield return ("^\\\".*\\\"\\s*(;.*)?$", RegexOptions.None, "\"1234\"", true);
            yield return ("^\\\".*\\\"\\s*(;.*)?$", RegexOptions.None, "\"abcd\" ;", true);
            yield return ("^\\\".*\\\"\\s*(;.*)?$", RegexOptions.None, "\"\" ; rhubarb", true);
            yield return ("^\\\".*\\\"\\s*(;.*)?$", RegexOptions.None, "\"1234\" : things", false);
            yield return ("^$", RegexOptions.None, "", true);
            yield return ("^$", RegexOptions.None, "A non-empty line", false);
            yield return ("   ^    a   (?# begins with a)  b\\sc (?# then b c) $ (?# then end)", RegexOptions.IgnorePatternWhitespace, "ab c", true);
            yield return ("   ^    a   (?# begins with a)  b\\sc (?# then b c) $ (?# then end)", RegexOptions.IgnorePatternWhitespace, "abc", false);
            yield return ("   ^    a   (?# begins with a)  b\\sc (?# then b c) $ (?# then end)", RegexOptions.IgnorePatternWhitespace, "ab cde", false);
            yield return ("(?x)   ^    a   (?# begins with a)  b\\sc (?# then b c) $ (?# then end)", RegexOptions.None, "ab c", true);
            yield return ("(?x)   ^    a   (?# begins with a)  b\\sc (?# then b c) $ (?# then end)", RegexOptions.None, "abc", false);
            yield return ("(?x)   ^    a   (?# begins with a)  b\\sc (?# then b c) $ (?# then end)", RegexOptions.None, "ab cde", false);
            yield return ("^   a\\ b[c ]d       $", RegexOptions.IgnorePatternWhitespace, "a bcd", true);
            yield return ("^   a\\ b[c ]d       $", RegexOptions.IgnorePatternWhitespace, "a b d", true);
            yield return ("^   a\\ b[c ]d       $", RegexOptions.IgnorePatternWhitespace, "abcd", false);
            yield return ("^   a\\ b[c ]d       $", RegexOptions.IgnorePatternWhitespace, "ab d", false);
            yield return ("^(a(b(c)))(d(e(f)))(h(i(j)))(k(l(m)))$", RegexOptions.None, "abcdefhijklm", true);
            yield return ("^(?:a(b(c)))(?:d(e(f)))(?:h(i(j)))(?:k(l(m)))$", RegexOptions.None, "abcdefhijklm", true);
            yield return ("^[\\w][\\W][\\s][\\S][\\d][\\D][\\b][\\n][\\c]][\\022]", RegexOptions.None, "a+ Z0+\b\n\u001d\u0012", true);
            yield return ("^[.^$|()*+?{,}]+", RegexOptions.None, ".^$(*+)|{?,?}", true);
            yield return ("^a*\\w", RegexOptions.None, "z", true);
            yield return ("^a*\\w", RegexOptions.None, "az", true);
            yield return ("^a*\\w", RegexOptions.None, "aaaz", true);
            yield return ("^a*\\w", RegexOptions.None, "a", true);
            yield return ("^a*\\w", RegexOptions.None, "aa", true);
            yield return ("^a*\\w", RegexOptions.None, "aaaa", true);
            yield return ("^a*\\w", RegexOptions.None, "a+", true);
            yield return ("^a*\\w", RegexOptions.None, "aa+", true);
            yield return ("^a*?\\w", RegexOptions.None, "z", true);
            yield return ("^a*?\\w", RegexOptions.None, "az", true);
            yield return ("^a*?\\w", RegexOptions.None, "aaaz", true);
            yield return ("^a*?\\w", RegexOptions.None, "a", true);
            yield return ("^a*?\\w", RegexOptions.None, "aa", true);
            yield return ("^a*?\\w", RegexOptions.None, "aaaa", true);
            yield return ("^a*?\\w", RegexOptions.None, "a+", true);
            yield return ("^a*?\\w", RegexOptions.None, "aa+", true);
            yield return ("^a+\\w", RegexOptions.None, "az", true);
            yield return ("^a+\\w", RegexOptions.None, "aaaz", true);
            yield return ("^a+\\w", RegexOptions.None, "aa", true);
            yield return ("^a+\\w", RegexOptions.None, "aaaa", true);
            yield return ("^a+\\w", RegexOptions.None, "aa+", true);
            yield return ("^a+?\\w", RegexOptions.None, "az", true);
            yield return ("^a+?\\w", RegexOptions.None, "aaaz", true);
            yield return ("^a+?\\w", RegexOptions.None, "aa", true);
            yield return ("^a+?\\w", RegexOptions.None, "aaaa", true);
            yield return ("^a+?\\w", RegexOptions.None, "aa+", true);
            yield return ("^\\d{8}\\w{2,}", RegexOptions.None, "1234567890", true);
            yield return ("^\\d{8}\\w{2,}", RegexOptions.None, "12345678ab", true);
            yield return ("^\\d{8}\\w{2,}", RegexOptions.None, "12345678__", true);
            yield return ("^\\d{8}\\w{2,}", RegexOptions.None, "1234567", false);
            yield return ("^[aeiou\\d]{4,5}$", RegexOptions.None, "uoie", true);
            yield return ("^[aeiou\\d]{4,5}$", RegexOptions.None, "1234", true);
            yield return ("^[aeiou\\d]{4,5}$", RegexOptions.None, "12345", true);
            yield return ("^[aeiou\\d]{4,5}$", RegexOptions.None, "aaaaa", true);
            yield return ("^[aeiou\\d]{4,5}$", RegexOptions.None, "123456", false);
            yield return ("^[aeiou\\d]{4,5}?", RegexOptions.None, "uoie", true);
            yield return ("^[aeiou\\d]{4,5}?", RegexOptions.None, "1234", true);
            yield return ("^[aeiou\\d]{4,5}?", RegexOptions.None, "12345", true);
            yield return ("^[aeiou\\d]{4,5}?", RegexOptions.None, "aaaaa", true);
            yield return ("^[aeiou\\d]{4,5}?", RegexOptions.None, "123456", true);
            yield return ("^From +([^ ]+) +[a-zA-Z][a-zA-Z][a-zA-Z] +[a-zA-Z][a-zA-Z][a-zA-Z] +[0-9]?[0-9] +[0-9][0-9]:[0-9][0-9]", RegexOptions.None, "From abcd  Mon Sep 01 12:33:02 1997", true);
            yield return ("^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d", RegexOptions.None, "From abcd  Mon Sep 01 12:33:02 1997", true);
            yield return ("^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d", RegexOptions.None, "From abcd  Mon Sep  1 12:33:02 1997", true);
            yield return ("^From\\s+\\S+\\s+([a-zA-Z]{3}\\s+){2}\\d{1,2}\\s+\\d\\d:\\d\\d", RegexOptions.None, "From abcd  Sep 01 12:33:02 1997", false);
            yield return ("^12.34", RegexOptions.Singleline, "12\n34", true);
            yield return ("^12.34", RegexOptions.Singleline, "12\r34", true);
            yield return ("^1234(?# test newlines\n  inside)", RegexOptions.None, "1234", true);
            yield return ("^1234 #comment in extended re\n  ", RegexOptions.IgnorePatternWhitespace, "1234", true);
            yield return ("#rhubarb\n  abcd", RegexOptions.IgnorePatternWhitespace, "abcd", true);
            yield return ("^abcd#rhubarb", RegexOptions.IgnorePatternWhitespace, "abcd", true);
            yield return ("^[ab]{1,3}(ab*|b)", RegexOptions.None, "aabbbbb", true);
            yield return ("^[ab]{1,3}?(ab*|b)", RegexOptions.None, "aabbbbb", true);
            yield return ("^[ab]{1,3}?(ab*?|b)", RegexOptions.None, "aabbbbb", true);
            yield return ("^[ab]{1,3}(ab*?|b)", RegexOptions.None, "aabbbbb", true);
            yield return ("abc\\0def\\00pqr\\000xyz\\0000AB", RegexOptions.None, "abc\0def\0pqr\0xyz\00AB", true);
            yield return ("abc\\0def\\00pqr\\000xyz\\0000AB", RegexOptions.None, "abc456 abc\0def\0pqr\0xyz\00ABCDE", true);
            yield return ("abc\\x0def\\x00pqr\\x000xyz\\x0000AB", RegexOptions.None, "abc\ref\0pqr\00xyz\000AB", true);
            yield return ("abc\\x0def\\x00pqr\\x000xyz\\x0000AB", RegexOptions.None, "abc456 abc\ref\0pqr\00xyz\000ABCDE", true);
            yield return ("^[\\000-\\037]", RegexOptions.None, "\0A", true);
            yield return ("^[\\000-\\037]", RegexOptions.None, "\u0001B", true);
            yield return ("^[\\000-\\037]", RegexOptions.None, "\u001fC", true);
            yield return ("\\0*", RegexOptions.None, "\0\0\0\0", true);
            yield return ("^\\s", RegexOptions.None, " abc", true);
            yield return ("^\\s", RegexOptions.None, "\fabc", true);
            yield return ("^\\s", RegexOptions.None, "\nabc", true);
            yield return ("^\\s", RegexOptions.None, "\rabc", true);
            yield return ("^\\s", RegexOptions.None, "\tabc", true);
            yield return ("^\\s", RegexOptions.None, "abc", false);
            yield return ("^a\tb\n    \f  c", RegexOptions.IgnorePatternWhitespace, "abc", true);
            yield return ("ab{1,3}bc", RegexOptions.None, "abbbbc", true);
            yield return ("ab{1,3}bc", RegexOptions.None, "abbbc", true);
            yield return ("ab{1,3}bc", RegexOptions.None, "abbc", true);
            yield return ("ab{1,3}bc", RegexOptions.None, "abc", false);
            yield return ("ab{1,3}bc", RegexOptions.None, "abbbbbc", false);
            yield return ("([^.]*)\\.([^:]*):[T ]+(.*)", RegexOptions.None, "track1.title:TBlah blah blah", true);
            yield return ("([^.]*)\\.([^:]*):[T ]+(.*)", RegexOptions.IgnoreCase, "track1.title:TBlah blah blah", true);
            yield return ("([^.]*)\\.([^:]*):[t ]+(.*)", RegexOptions.IgnoreCase, "track1.title:TBlah blah blah", true);
            yield return ("^[W-c]+$", RegexOptions.None, "WXY_^abc", true);
            yield return ("^[W-c]+$", RegexOptions.None, "wxy", false);
            yield return ("^[W-c]+$", RegexOptions.IgnoreCase, "WXY_^abc", true);
            yield return ("^[W-c]+$", RegexOptions.IgnoreCase, "wxy_^ABC", true);
            yield return ("^[\\x3f-\\x5F]+$", RegexOptions.IgnoreCase, "WXY_^abc", true);
            yield return ("^[\\x3f-\\x5F]+$", RegexOptions.IgnoreCase, "wxy_^ABC", true);
            yield return ("^abc$", RegexOptions.Multiline, "abc", true);
            yield return ("^abc$", RegexOptions.Multiline, "qqq\nabc", true);
            yield return ("^abc$", RegexOptions.Multiline, "abc\nzzz", true);
            yield return ("^abc$", RegexOptions.Multiline, "qqq\nabc\nzzz", true);
            yield return ("^abc$", RegexOptions.None, "abc", true);
            yield return ("^abc$", RegexOptions.None, "qqq\nabc", false);
            yield return ("^abc$", RegexOptions.None, "abc\nzzz", false);
            yield return ("^abc$", RegexOptions.None, "qqq\nabc\nzzz", false);
            yield return ("\\Aabc\\Z", RegexOptions.Multiline, "abc", true);
            yield return ("\\Aabc\\Z", RegexOptions.Multiline, "abc\n", true);
            yield return ("\\Aabc\\Z", RegexOptions.Multiline, "qqq\nabc", false);
            yield return ("\\Aabc\\Z", RegexOptions.Multiline, "abc\nzzz", false);
            yield return ("\\Aabc\\Z", RegexOptions.Multiline, "qqq\nabc\nzzz", false);
            yield return ("\\A(.)*\\Z", RegexOptions.Singleline, "abc\ndef", true);
            yield return ("\\A(.)*\\Z", RegexOptions.Multiline, "abc\ndef", false);
            yield return ("(?:b)|(?::+)", RegexOptions.None, "b::c", true);
            yield return ("(?:b)|(?::+)", RegexOptions.None, "c::b", true);
            yield return ("[-az]+", RegexOptions.None, "az-", true);
            yield return ("[-az]+", RegexOptions.None, "b", false);
            yield return ("[az-]+", RegexOptions.None, "za-", true);
            yield return ("[az-]+", RegexOptions.None, "b", false);
            yield return ("[a\\-z]+", RegexOptions.None, "a-z", true);
            yield return ("[a\\-z]+", RegexOptions.None, "b", false);
            yield return ("[a-z]+", RegexOptions.None, "abcdxyz", true);
            yield return ("[\\d-]+", RegexOptions.None, "12-34", true);
            yield return ("[\\d-]+", RegexOptions.None, "aaa", false);
            yield return ("\\x5c", RegexOptions.None, "\\", true);
            yield return ("\\x20Z", RegexOptions.None, "the Zoo", true);
            yield return ("\\x20Z", RegexOptions.None, "Zulu", false);
            yield return ("abc$", RegexOptions.None, "abc", true);
            yield return ("abc$", RegexOptions.None, "abc\n", true);
            yield return ("abc$", RegexOptions.None, "abc\ndef", false);
            yield return ("(abc)\\123", RegexOptions.None, "abcS", true);
            yield return ("(abc)\\223", RegexOptions.None, "abc\u0093", true);
            yield return ("(abc)\\323", RegexOptions.None, "abc\u00D3", true);
            yield return ("(abc)\\100", RegexOptions.None, "abc@", true);
            yield return ("(abc)\\1000", RegexOptions.None, "abc@0", true);
            yield return ("(a)(b)(c)(d)(e)(f)(g)(h)(i)(j)(k)\\12\\123", RegexOptions.None, "abcdefghijk\nS", true);
            yield return ("a{0}bc", RegexOptions.None, "bc", true);
            yield return ("(a|(bc)){0,0}?xyz", RegexOptions.None, "xyz", true);
            yield return ("abc[\\10]de", RegexOptions.None, "abc\bde", true);
            yield return ("abc[\\1]de", RegexOptions.None, "abc\u0001de", true);
            yield return ("(abc)[\\1]de", RegexOptions.None, "abc\u0001de", true);
            yield return ("(?s)a.b", RegexOptions.None, "a\nb", true);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "baNOTccccd", true);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "baNOTcccd", true);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "baNOTccd", true);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "bacccd", true);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "anything", false);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "bbc", false);
            yield return ("^([^a])([^\\b])([^c]*)([^d]{3,4})", RegexOptions.None, "baccd", false);
            yield return ("[^a]", RegexOptions.None, "Abc", true);
            yield return ("[^a]", RegexOptions.IgnoreCase, "Abc", true);
            yield return ("[^a]+", RegexOptions.None, "AAAaAbc", true);
            yield return ("[^a]+", RegexOptions.IgnoreCase, "AAAaAbc", true);
            yield return ("[^a]+", RegexOptions.None, "bbb\nccc", true);
            yield return ("[^k]$", RegexOptions.None, "abc", true);
            yield return ("[^k]$", RegexOptions.None, "abk", false);
            yield return ("[^k]{2,3}$", RegexOptions.None, "abc", true);
            yield return ("[^k]{2,3}$", RegexOptions.None, "kbc", true);
            yield return ("[^k]{2,3}$", RegexOptions.None, "kabc", true);
            yield return ("[^k]{2,3}$", RegexOptions.None, "abk", false);
            yield return ("[^k]{2,3}$", RegexOptions.None, "akb", false);
            yield return ("[^k]{2,3}$", RegexOptions.None, "akk", false);
            yield return ("^\\d{8,}\\@.+[^k]$", RegexOptions.None, "12345678@a.b.c.d", true);
            yield return ("^\\d{8,}\\@.+[^k]$", RegexOptions.None, "123456789@x.y.z", true);
            yield return ("^\\d{8,}\\@.+[^k]$", RegexOptions.None, "12345678@x.y.uk", false);
            yield return ("^\\d{8,}\\@.+[^k]$", RegexOptions.None, "1234567@a.b.c.d", false);
            yield return ("[^a]", RegexOptions.None, "aaaabcd", true);
            yield return ("[^a]", RegexOptions.None, "aaAabcd", true);
            yield return ("[^a]", RegexOptions.IgnoreCase, "aaaabcd", true);
            yield return ("[^a]", RegexOptions.IgnoreCase, "aaAabcd", true);
            yield return ("[^az]", RegexOptions.None, "aaaabcd", true);
            yield return ("[^az]", RegexOptions.None, "aaAabcd", true);
            yield return ("[^az]", RegexOptions.IgnoreCase, "aaaabcd", true);
            yield return ("[^az]", RegexOptions.IgnoreCase, "aaAabcd", true);
            yield return ("\\000\\001\\002\\003\\004\\005\\006\\007\\010\\011\\012\\013\\014\\015\\016\\017\\020\\021\\022\\023\\024\\025\\026\\027\\030\\031\\032\\033\\034\\035\\036\\037\\040\\041\\042\\043\\044\\045\\046\\047\\050\\051\\052\\053\\054\\055\\056\\057\\060\\061\\062\\063\\064\\065\\066\\067\\070\\071\\072\\073\\074\\075\\076\\077\\100\\101\\102\\103\\104\\105\\106\\107\\110\\111\\112\\113\\114\\115\\116\\117\\120\\121\\122\\123\\124\\125\\126\\127\\130\\131\\132\\133\\134\\135\\136\\137\\140\\141\\142\\143\\144\\145\\146\\147\\150\\151\\152\\153\\154\\155\\156\\157\\160\\161\\162\\163\\164\\165\\166\\167\\170\\171\\172\\173\\174\\175\\176\\177\\200\\201\\202\\203\\204\\205\\206\\207\\210\\211\\212\\213\\214\\215\\216\\217\\220\\221\\222\\223\\224\\225\\226\\227\\230\\231\\232\\233\\234\\235\\236\\237\\240\\241\\242\\243\\244\\245\\246\\247\\250\\251\\252\\253\\254\\255\\256\\257\\260\\261\\262\\263\\264\\265\\266\\267\\270\\271\\272\\273\\274\\275\\276\\277\\300\\301\\302\\303\\304\\305\\306\\307\\310\\311\\312\\313\\314\\315\\316\\317\\320\\321\\322\\323\\324\\325\\326\\327\\330\\331\\332\\333\\334\\335\\336\\337\\340\\341\\342\\343\\344\\345\\346\\347\\350\\351\\352\\353\\354\\355\\356\\357\\360\\361\\362\\363\\364\\365\\366\\367\\370\\371\\372\\373\\374\\375\\376\\377", RegexOptions.None, "\0\u0001\u0002\u0003\u0004\u0005\u0006\a\b\t\n\v\f\r\u000e\u000f\u0010\u0011\u0012\u0013\u0014\u0015\u0016\u0017\u0018\u0019\u001a\u001b\u001c\u001d\u001e\u001f !\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\u007f\u0080\u0081\u0082\u0083\u0084\u0085\u0086\u0087\u0088\u0089\u008a\u008b\u008c\u008d\u008e\u008f\u0090\u0091\u0092\u0093\u0094\u0095\u0096\u0097\u0098\u0099\u009a\u009b\u009c\u009d\u009e\u009f\u00A0\u00A1\u00A2\u00A3\u00A4\u00A5\u00A6\u00A7\u00A8\u00A9\u00AA\u00AB\u00AC\u00AD\u00AE\u00AF\u00B0\u00B1\u00B2\u00B3\u00B4\u00B5\u00B6\u00B7\u00B8\u00B9\u00BA\u00BB\u00BC\u00BD\u00BE\u00BF\u00C0\u00C1\u00C2\u00C3\u00C4\u00C5\u00C6\u00C7\u00C8\u00C9\u00CA\u00CB\u00CC\u00CD\u00CE\u00CF\u00D0\u00D1\u00D2\u00D3\u00D4\u00D5\u00D6\u00D7\u00D8\u00D9\u00DA\u00DB\u00DC\u00DD\u00DE\u00DF\u00E0\u00E1\u00E2\u00E3\u00E4\u00E5\u00E6\u00E7\u00E8\u00E9\u00EA\u00EB\u00EC\u00ED\u00EE\u00EF\u00F0\u00F1\u00F2\u00F3\u00F4\u00F5\u00F6\u00F7\u00F8\u00F9\u00FA\u00FB\u00FC\u00FD\u00FE\u00FF", true);
            yield return ("P[^*]TAIRE[^*]{1,6}?LL", RegexOptions.None, "xxxxxxxxxxxPSTAIREISLLxxxxxxxxx", true);
            yield return ("P[^*]TAIRE[^*]{1,}?LL", RegexOptions.None, "xxxxxxxxxxxPSTAIREISLLxxxxxxxxx", true);
            yield return ("(\\.\\d\\d[1-9]?)\\d+", RegexOptions.None, "1.230003938", true);
            yield return ("(\\.\\d\\d[1-9]?)\\d+", RegexOptions.None, "1.875000282", true);
            yield return ("(\\.\\d\\d[1-9]?)\\d+", RegexOptions.None, "1.235", true);
            yield return ("\\b(foo)\\s+(\\w+)", RegexOptions.IgnoreCase, "Food is on the foo table", true);
            yield return ("foo(.*)bar", RegexOptions.None, "The food is under the bar in the barn.", true);
            yield return ("foo(.*?)bar", RegexOptions.None, "The food is under the bar in the barn.", true);
            yield return ("(.*)(\\d*)", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*)(\\d+)", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*?)(\\d*)", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*?)(\\d+)", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*)(\\d+)$", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*?)(\\d+)$", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*)\\b(\\d+)$", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("(.*\\D)(\\d+)$", RegexOptions.None, "I have 2 numbers: 53147", true);
            yield return ("^[W-]46]", RegexOptions.None, "W46]789", true);
            yield return ("^[W-]46]", RegexOptions.None, "-46]789", true);
            yield return ("^[W-]46]", RegexOptions.None, "Wall", false);
            yield return ("^[W-]46]", RegexOptions.None, "Zebra", false);
            yield return ("^[W-]46]", RegexOptions.None, "42", false);
            yield return ("^[W-]46]", RegexOptions.None, "[abcd]", false);
            yield return ("^[W-]46]", RegexOptions.None, "]abcd[", false);
            yield return ("^[W-\\]46]", RegexOptions.None, "W46]789", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "Wall", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "Zebra", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "Xylophone", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "42", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "[abcd]", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "]abcd[", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "\\backslash", true);
            yield return ("^[W-\\]46]", RegexOptions.None, "-46]789", false);
            yield return ("^[W-\\]46]", RegexOptions.None, "well", false);
            yield return ("\\d\\d\\/\\d\\d\\/\\d\\d\\d\\d", RegexOptions.None, "01/01/2000", true);
            yield return ("word (?:[a-zA-Z0-9]+ ){0,10}otherword", RegexOptions.None, "word cat dog elephant mussel cow horse canary baboon snake shark otherword", true);
            yield return ("word (?:[a-zA-Z0-9]+ ){0,10}otherword", RegexOptions.None, "word cat dog elephant mussel cow horse canary baboon snake shark", false);
            yield return ("word (?:[a-zA-Z0-9]+ ){0,300}otherword", RegexOptions.None, "word cat dog elephant mussel cow horse canary baboon snake shark the quick brown fox and the lazy dog and several other words getting close to thirty by now I hope", false);
            yield return ("^(a){0,0}", RegexOptions.None, "bcd", true);
            yield return ("^(a){0,0}", RegexOptions.None, "abc", true);
            yield return ("^(a){0,0}", RegexOptions.None, "aab", true);
            yield return ("^(a){0,1}", RegexOptions.None, "bcd", true);
            yield return ("^(a){0,1}", RegexOptions.None, "abc", true);
            yield return ("^(a){0,1}", RegexOptions.None, "aab", true);
            yield return ("^(a){0,2}", RegexOptions.None, "bcd", true);
            yield return ("^(a){0,2}", RegexOptions.None, "abc", true);
            yield return ("^(a){0,2}", RegexOptions.None, "aab", true);
            yield return ("^(a){0,3}", RegexOptions.None, "bcd", true);
            yield return ("^(a){0,3}", RegexOptions.None, "abc", true);
            yield return ("^(a){0,3}", RegexOptions.None, "aab", true);
            yield return ("^(a){0,3}", RegexOptions.None, "aaa", true);
            yield return ("^(a){0,}", RegexOptions.None, "bcd", true);
            yield return ("^(a){0,}", RegexOptions.None, "abc", true);
            yield return ("^(a){0,}", RegexOptions.None, "aab", true);
            yield return ("^(a){0,}", RegexOptions.None, "aaa", true);
            yield return ("^(a){0,}", RegexOptions.None, "aaaaaaaa", true);
            yield return ("^(a){1,1}", RegexOptions.None, "abc", true);
            yield return ("^(a){1,1}", RegexOptions.None, "aab", true);
            yield return ("^(a){1,1}", RegexOptions.None, "bcd", false);
            yield return ("^(a){1,2}", RegexOptions.None, "abc", true);
            yield return ("^(a){1,2}", RegexOptions.None, "aab", true);
            yield return ("^(a){1,2}", RegexOptions.None, "bcd", false);
            yield return ("^(a){1,3}", RegexOptions.None, "abc", true);
            yield return ("^(a){1,3}", RegexOptions.None, "aab", true);
            yield return ("^(a){1,3}", RegexOptions.None, "aaa", true);
            yield return ("^(a){1,3}", RegexOptions.None, "bcd", false);
            yield return ("^(a){1,}", RegexOptions.None, "abc", true);
            yield return ("^(a){1,}", RegexOptions.None, "aab", true);
            yield return ("^(a){1,}", RegexOptions.None, "aaa", true);
            yield return ("^(a){1,}", RegexOptions.None, "aaaaaaaa", true);
            yield return ("^(a){1,}", RegexOptions.None, "bcd", false);
            yield return (".*\\.gif", RegexOptions.None, "borfle\nbib.gif\nno", true);
            yield return (".{0,}\\.gif", RegexOptions.None, "borfle\nbib.gif\nno", true);
            yield return (".*\\.gif", RegexOptions.Multiline, "borfle\nbib.gif\nno", true);
            yield return (".*\\.gif", RegexOptions.Singleline, "borfle\nbib.gif\nno", true);
            yield return (".*\\.gif", RegexOptions.Multiline | RegexOptions.Singleline, "borfle\nbib.gif\nno", true);
            yield return (".*$", RegexOptions.None, "borfle\nbib.gif\nno", true);
            yield return (".*$", RegexOptions.Multiline, "borfle\nbib.gif\nno", true);
            yield return (".*$", RegexOptions.Singleline, "borfle\nbib.gif\nno", true);
            yield return (".*$", RegexOptions.Multiline | RegexOptions.Singleline, "borfle\nbib.gif\nno", true);
            yield return (".*$", RegexOptions.None, "borfle\nbib.gif\nno\n", true);
            yield return (".*$", RegexOptions.Multiline, "borfle\nbib.gif\nno\n", true);
            yield return (".*$", RegexOptions.Singleline, "borfle\nbib.gif\nno\n", true);
            yield return (".*$", RegexOptions.Multiline | RegexOptions.Singleline, "borfle\nbib.gif\nno\n", true);
            yield return ("(.*X|^B)", RegexOptions.None, "abcde\n1234Xyz", true);
            yield return ("(.*X|^B)", RegexOptions.None, "BarFoo", true);
            yield return ("(.*X|^B)", RegexOptions.None, "abcde\nBar", false);
            yield return ("(.*X|^B)", RegexOptions.Multiline, "abcde\n1234Xyz", true);
            yield return ("(.*X|^B)", RegexOptions.Multiline, "BarFoo", true);
            yield return ("(.*X|^B)", RegexOptions.Multiline, "abcde\nBar", true);
            yield return ("(.*X|^B)", RegexOptions.Singleline, "abcde\n1234Xyz", true);
            yield return ("(.*X|^B)", RegexOptions.Singleline, "BarFoo", true);
            yield return ("(.*X|^B)", RegexOptions.Singleline, "abcde\nBar", false);
            yield return ("(.*X|^B)", RegexOptions.Multiline | RegexOptions.Singleline, "abcde\n1234Xyz", true);
            yield return ("(.*X|^B)", RegexOptions.Multiline | RegexOptions.Singleline, "BarFoo", true);
            yield return ("(.*X|^B)", RegexOptions.Multiline | RegexOptions.Singleline, "abcde\nBar", true);
            yield return ("(?s)(.*X|^B)", RegexOptions.None, "abcde\n1234Xyz", true);
            yield return ("(?s)(.*X|^B)", RegexOptions.None, "BarFoo", true);
            yield return ("(?s)(.*X|^B)", RegexOptions.None, "abcde\nBar", false);
            yield return ("(?s:.*X|^B)", RegexOptions.None, "abcde\n1234Xyz", true);
            yield return ("(?s:.*X|^B)", RegexOptions.None, "BarFoo", true);
            yield return ("(?s:.*X|^B)", RegexOptions.None, "abcde\nBar", false);
            yield return ("^.*B", RegexOptions.None, "abc\nB", false);
            yield return ("(?s)^.*B", RegexOptions.None, "abc\nB", true);
            yield return ("(?m)^.*B", RegexOptions.None, "abc\nB", true);
            yield return ("(?ms)^.*B", RegexOptions.None, "abc\nB", true);
            yield return ("(?ms)^B", RegexOptions.None, "abc\nB", true);
            yield return ("(?s)B$", RegexOptions.None, "B\n", true);
            yield return ("^[0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9][0-9]", RegexOptions.None, "123456654321", true);
            yield return ("^\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d\\d", RegexOptions.None, "123456654321", true);
            yield return ("^[\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d][\\d]", RegexOptions.None, "123456654321", true);
            yield return ("^[abc]{12}", RegexOptions.None, "abcabcabcabc", true);
            yield return ("^[a-c]{12}", RegexOptions.None, "abcabcabcabc", true);
            yield return ("^(a|b|c){12}", RegexOptions.None, "abcabcabcabc", true);
            yield return ("^[abcdefghijklmnopqrstuvwxy0123456789]", RegexOptions.None, "n", true);
            yield return ("^[abcdefghijklmnopqrstuvwxy0123456789]", RegexOptions.None, "z", false);
            yield return ("abcde{0,0}", RegexOptions.None, "abcd", true);
            yield return ("abcde{0,0}", RegexOptions.None, "abce", false);
            yield return ("ab[cd]{0,0}e", RegexOptions.None, "abe", true);
            yield return ("ab[cd]{0,0}e", RegexOptions.None, "abcde", false);
            yield return ("ab(c){0,0}d", RegexOptions.None, "abd", true);
            yield return ("ab(c){0,0}d", RegexOptions.None, "abcd", false);
            yield return ("a(b*)", RegexOptions.None, "a", true);
            yield return ("a(b*)", RegexOptions.None, "ab", true);
            yield return ("a(b*)", RegexOptions.None, "abbbb", true);
            yield return ("a(b*)", RegexOptions.None, "bbbbb", false);
            yield return ("ab\\d{0}e", RegexOptions.None, "abe", true);
            yield return ("ab\\d{0}e", RegexOptions.None, "ab1e", false);
            yield return ("\"([^\\\\\"]+|\\\\.)*\"", RegexOptions.None, "the \"quick\" brown fox", true);
            yield return ("\"([^\\\\\"]+|\\\\.)*\"", RegexOptions.None, "\"the \\\"quick\\\" brown fox\"", true);
            yield return (".*?", RegexOptions.None, "abc", true);
            yield return ("\\b", RegexOptions.None, "abc", true);
            yield return ("", RegexOptions.None, "abc", true);
            yield return ("<tr([\\w\\W\\s\\d][^<>]{0,})><TD([\\w\\W\\s\\d][^<>]{0,})>([\\d]{0,}\\.)(.*)((<BR>([\\w\\W\\s\\d][^<>]{0,})|[\\s]{0,}))<\\/a><\\/TD><TD([\\w\\W\\s\\d][^<>]{0,})>([\\w\\W\\s\\d][^<>]{0,})<\\/TD><TD([\\w\\W\\s\\d][^<>]{0,})>([\\w\\W\\s\\d][^<>]{0,})<\\/TD><\\/TR>", RegexOptions.IgnoreCase | RegexOptions.Singleline, "<TR BGCOLOR='#DBE9E9'><TD align=left valign=top>43.<a href='joblist.cfm?JobID=94 6735&Keyword='>Word Processor<BR>(N-1286)</a></TD><TD align=left valign=top>Lega lstaff.com</TD><TD align=left valign=top>CA - Statewide</TD></TR>", true);
            yield return ("a[^a]b", RegexOptions.None, "acb", true);
            yield return ("a[^a]b", RegexOptions.None, "a\nb", true);
            yield return ("a.b", RegexOptions.None, "acb", true);
            yield return ("a.b", RegexOptions.None, "a\nb", false);
            yield return ("a[^a]b", RegexOptions.Singleline, "acb", true);
            yield return ("a[^a]b", RegexOptions.Singleline, "a\nb", true);
            yield return ("a.b", RegexOptions.Singleline, "acb", true);
            yield return ("a.b", RegexOptions.Singleline, "a\nb", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbbac", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbbbac", true);
            yield return ("^(b+?|a){1,2}?c", RegexOptions.None, "bbbbbac", true);
            yield return ("^(b+|a){1,2}?c", RegexOptions.None, "bac", true);
            yield return ("^(b+|a){1,2}?c", RegexOptions.None, "bbac", true);
            yield return ("^(b+|a){1,2}?c", RegexOptions.None, "bbbac", true);
            yield return ("^(b+|a){1,2}?c", RegexOptions.None, "bbbbac", true);
            yield return ("^(b+|a){1,2}?c", RegexOptions.None, "bbbbbac", true);
            yield return ("(A|B)*?CD", RegexOptions.None, "CD", true);
            yield return ("(A|B)*CD", RegexOptions.None, "CD", true);
            yield return ("\\Aabc\\z", RegexOptions.Multiline, "abc", true);
            yield return ("\\Aabc\\z", RegexOptions.Multiline, "abc\n", false);
            yield return ("\\Aabc\\z", RegexOptions.Multiline, "qqq\nabc", false);
            yield return ("\\Aabc\\z", RegexOptions.Multiline, "abc\nzzz", false);
            yield return ("\\Aabc\\z", RegexOptions.Multiline, "qqq\nabc\nzzz", false);
            yield return ("(\\d+)(\\w)", RegexOptions.None, "12345a", true);
            yield return ("(\\d+)(\\w)", RegexOptions.None, "12345+", true);
            yield return ("a(?-i)b", RegexOptions.IgnoreCase, "ab", true);
            yield return ("a(?-i)b", RegexOptions.IgnoreCase, "Ab", true);
            yield return ("a(?-i)b", RegexOptions.IgnoreCase, "aB", false);
            yield return ("a(?-i)b", RegexOptions.IgnoreCase, "AB", false);
            yield return ("(a (?x)b c)d e", RegexOptions.None, "a bcd e", true);
            yield return ("(a (?x)b c)d e", RegexOptions.None, "a b cd e", false);
            yield return ("(a (?x)b c)d e", RegexOptions.None, "abcd e", false);
            yield return ("(a (?x)b c)d e", RegexOptions.None, "a bcde", false);
            yield return ("(a b(?x)c d (?-x)e f)", RegexOptions.None, "a bcde f", true);
            yield return ("(a b(?x)c d (?-x)e f)", RegexOptions.None, "abcdef", false);
            yield return ("(a(?i)b)c", RegexOptions.None, "abc", true);
            yield return ("(a(?i)b)c", RegexOptions.None, "aBc", true);
            yield return ("(a(?i)b)c", RegexOptions.None, "abC", false);
            yield return ("(a(?i)b)c", RegexOptions.None, "aBC", false);
            yield return ("(a(?i)b)c", RegexOptions.None, "Abc", false);
            yield return ("(a(?i)b)c", RegexOptions.None, "ABc", false);
            yield return ("(a(?i)b)c", RegexOptions.None, "ABC", false);
            yield return ("(a(?i)b)c", RegexOptions.None, "AbC", false);
            yield return ("a(?i:b)c", RegexOptions.None, "abc", true);
            yield return ("a(?i:b)c", RegexOptions.None, "aBc", true);
            yield return ("a(?i:b)c", RegexOptions.None, "ABC", false);
            yield return ("a(?i:b)c", RegexOptions.None, "abC", false);
            yield return ("a(?i:b)c", RegexOptions.None, "aBC", false);
            yield return ("a(?i:b)*c", RegexOptions.None, "aBc", true);
            yield return ("a(?i:b)*c", RegexOptions.None, "aBBc", true);
            yield return ("a(?i:b)*c", RegexOptions.None, "aBC", false);
            yield return ("a(?i:b)*c", RegexOptions.None, "aBBC", false);
            yield return ("(?s-i:more.*than).*million", RegexOptions.IgnoreCase, "more than million", true);
            yield return ("(?s-i:more.*than).*million", RegexOptions.IgnoreCase, "more than MILLION", true);
            yield return ("(?s-i:more.*than).*million", RegexOptions.IgnoreCase, "more \n than Million", true);
            yield return ("(?s-i:more.*than).*million", RegexOptions.IgnoreCase, "MORE THAN MILLION", false);
            yield return ("(?s-i:more.*than).*million", RegexOptions.IgnoreCase, "more \n than \n million", false);
            yield return ("(?:(?s-i)more.*than).*million", RegexOptions.IgnoreCase, "more than million", true);
            yield return ("(?:(?s-i)more.*than).*million", RegexOptions.IgnoreCase, "more than MILLION", true);
            yield return ("(?:(?s-i)more.*than).*million", RegexOptions.IgnoreCase, "more \n than Million", true);
            yield return ("(?:(?s-i)more.*than).*million", RegexOptions.IgnoreCase, "MORE THAN MILLION", false);
            yield return ("(?:(?s-i)more.*than).*million", RegexOptions.IgnoreCase, "more \n than \n million", false);
            yield return ("(abc|)+", RegexOptions.None, "abc", true);
            yield return ("(abc|)+", RegexOptions.None, "abcabc", true);
            yield return ("(abc|)+", RegexOptions.None, "abcabcabc", true);
            yield return ("(abc|)+", RegexOptions.None, "xyz", true);
            yield return ("([a]*)*", RegexOptions.None, "a", true);
            yield return ("([a]*)*", RegexOptions.None, "aaaaa", true);
            yield return ("([ab]*)*", RegexOptions.None, "a", true);
            yield return ("([ab]*)*", RegexOptions.None, "b", true);
            yield return ("([ab]*)*", RegexOptions.None, "ababab", true);
            yield return ("([ab]*)*", RegexOptions.None, "aaaabcde", true);
            yield return ("([ab]*)*", RegexOptions.None, "bbbb", true);
            yield return ("([^a]*)*", RegexOptions.None, "b", true);
            yield return ("([^a]*)*", RegexOptions.None, "bbbb", true);
            yield return ("([^a]*)*", RegexOptions.None, "aaa", true);
            yield return ("([^ab]*)*", RegexOptions.None, "cccc", true);
            yield return ("([^ab]*)*", RegexOptions.None, "abab", true);
            yield return ("([a]*?)*", RegexOptions.None, "a", true);
            yield return ("([a]*?)*", RegexOptions.None, "aaaa", true);
            yield return ("([ab]*?)*", RegexOptions.None, "a", true);
            yield return ("([ab]*?)*", RegexOptions.None, "b", true);
            yield return ("([ab]*?)*", RegexOptions.None, "abab", true);
            yield return ("([ab]*?)*", RegexOptions.None, "baba", true);
            yield return ("([^a]*?)*", RegexOptions.None, "b", true);
            yield return ("([^a]*?)*", RegexOptions.None, "bbbb", true);
            yield return ("([^a]*?)*", RegexOptions.None, "aaa", true);
            yield return ("([^ab]*?)*", RegexOptions.None, "c", true);
            yield return ("([^ab]*?)*", RegexOptions.None, "cccc", true);
            yield return ("([^ab]*?)*", RegexOptions.None, "baba", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "saturday", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "sunday", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "Saturday", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "Sunday", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "SATURDAY", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "SUNDAY", true);
            yield return ("(?i:saturday|sunday)", RegexOptions.None, "SunDay", true);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "abcx", true);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "aBCx", true);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "bbx", true);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "BBx", true);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "abcX", false);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "aBCX", false);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "bbX", false);
            yield return ("(a(?i)bc|BB)x", RegexOptions.None, "BBX", false);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "ac", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "aC", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "bD", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "elephant", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "Europe", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "frog", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "France", true);
            yield return ("^([ab](?i)[cd]|[ef])", RegexOptions.None, "Africa", false);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "ab", true);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "aBd", true);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "xy", true);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "xY", true);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "zebra", true);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "Zambesi", true);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "aCD", false);
            yield return ("^(ab|a(?i)[b-c](?m-i)d|x(?i)y|z)", RegexOptions.None, "XY", false);
            yield return ("abc", RegexOptions.None, "abc", true);
            yield return ("abc", RegexOptions.None, "xabcy", true);
            yield return ("abc", RegexOptions.None, "ababc", true);
            yield return ("abc", RegexOptions.None, "xbc", false);
            yield return ("abc", RegexOptions.None, "axc", false);
            yield return ("abc", RegexOptions.None, "abx", false);
            yield return ("ab*c", RegexOptions.None, "abc", true);
            yield return ("ab*bc", RegexOptions.None, "abc", true);
            yield return ("ab*bc", RegexOptions.None, "abbc", true);
            yield return ("ab*bc", RegexOptions.None, "abbbbc", true);
            yield return (".{1}", RegexOptions.None, "abbbbc", true);
            yield return (".{3,4}", RegexOptions.None, "abbbbc", true);
            yield return ("ab{0,}bc", RegexOptions.None, "abbbbc", true);
            yield return ("ab+bc", RegexOptions.None, "abbc", true);
            yield return ("ab+bc", RegexOptions.None, "abc", false);
            yield return ("ab+bc", RegexOptions.None, "abq", false);
            yield return ("ab+bc", RegexOptions.None, "abbbbc", true);
            yield return ("ab{1,}bc", RegexOptions.None, "abbbbc", true);
            yield return ("ab{3,4}bc", RegexOptions.None, "abbbbc", true);
            yield return ("ab{4,5}bc", RegexOptions.None, "abq", false);
            yield return ("ab{4,5}bc", RegexOptions.None, "abbbbc", false);
            yield return ("ab?bc", RegexOptions.None, "abbc", true);
            yield return ("ab?bc", RegexOptions.None, "abc", true);
            yield return ("ab{0,1}bc", RegexOptions.None, "abc", true);
            yield return ("ab?c", RegexOptions.None, "abc", true);
            yield return ("ab{0,1}c", RegexOptions.None, "abc", true);
            yield return ("^abc$", RegexOptions.None, "abbbbc", false);
            yield return ("^abc$", RegexOptions.None, "abcc", false);
            yield return ("^abc", RegexOptions.None, "abcc", true);
            yield return ("abc$", RegexOptions.None, "aabc", true);
            yield return ("abc$", RegexOptions.None, "aabcd", false);
            yield return ("^", RegexOptions.None, "abc", true);
            yield return ("$", RegexOptions.None, "abc", true);
            yield return ("a.c", RegexOptions.None, "abc", true);
            yield return ("a.c", RegexOptions.None, "axc", true);
            yield return ("a.*c", RegexOptions.None, "axyzc", true);
            yield return ("a[bc]d", RegexOptions.None, "abd", true);
            yield return ("a[bc]d", RegexOptions.None, "axyzd", false);
            yield return ("a[bc]d", RegexOptions.None, "abc", false);
            yield return ("a[b-d]e", RegexOptions.None, "ace", true);
            yield return ("a[b-d]", RegexOptions.None, "aac", true);
            yield return ("a[-b]", RegexOptions.None, "a-", true);
            yield return ("a[b-]", RegexOptions.None, "a-", true);
            yield return ("a]", RegexOptions.None, "a]", true);
            yield return ("a[]]b", RegexOptions.None, "a]b", true);
            yield return ("a[^bc]d", RegexOptions.None, "aed", true);
            yield return ("a[^bc]d", RegexOptions.None, "abd", false);
            yield return ("a[^-b]c", RegexOptions.None, "adc", true);
            yield return ("a[^]b]c", RegexOptions.None, "adc", true);
            yield return ("a[^]b]c", RegexOptions.None, "a-c", true);
            yield return ("a[^]b]c", RegexOptions.None, "a]c", false);
            yield return ("\\ba\\b", RegexOptions.None, "a-", true);
            yield return ("\\ba\\b", RegexOptions.None, "-a", true);
            yield return ("\\ba\\b", RegexOptions.None, "-a-", true);
            yield return ("\\by\\b", RegexOptions.None, "xy", false);
            yield return ("\\by\\b", RegexOptions.None, "yz", false);
            yield return ("\\by\\b", RegexOptions.None, "xyz", false);
            yield return ("\\Ba\\B", RegexOptions.None, "a-", false);
            yield return ("\\Ba\\B", RegexOptions.None, "-a", false);
            yield return ("\\Ba\\B", RegexOptions.None, "-a-", false);
            yield return ("\\By\\b", RegexOptions.None, "xy", true);
            yield return ("\\by\\B", RegexOptions.None, "yz", true);
            yield return ("\\By\\B", RegexOptions.None, "xyz", true);
            yield return ("\\w", RegexOptions.None, "a", true);
            yield return ("\\W", RegexOptions.None, "-", true);
            yield return ("\\W", RegexOptions.None, "a", false);
            yield return ("a\\sb", RegexOptions.None, "a b", true);
            yield return ("a\\Sb", RegexOptions.None, "a-b", true);
            yield return ("a\\Sb", RegexOptions.None, "a b", false);
            yield return ("\\d", RegexOptions.None, "1", true);
            yield return ("\\D", RegexOptions.None, "-", true);
            yield return ("\\D", RegexOptions.None, "1", false);
            yield return ("[\\w]", RegexOptions.None, "a", true);
            yield return ("[\\W]", RegexOptions.None, "-", true);
            yield return ("[\\W]", RegexOptions.None, "a", false);
            yield return ("a[\\s]b", RegexOptions.None, "a b", true);
            yield return ("a[\\S]b", RegexOptions.None, "a-b", true);
            yield return ("a[\\S]b", RegexOptions.None, "a b", false);
            yield return ("[\\d]", RegexOptions.None, "1", true);
            yield return ("[\\D]", RegexOptions.None, "-", true);
            yield return ("[\\D]", RegexOptions.None, "1", false);
            yield return ("ab|cd", RegexOptions.None, "abc", true);
            yield return ("ab|cd", RegexOptions.None, "abcd", true);
            yield return ("()ef", RegexOptions.None, "def", true);
            yield return ("a\\(b", RegexOptions.None, "a(b", true);
            yield return ("a\\(*b", RegexOptions.None, "ab", true);
            yield return ("a\\(*b", RegexOptions.None, "a((b", true);
            yield return ("a\\\\b", RegexOptions.None, "a\\b", true);
            yield return ("((a))", RegexOptions.None, "abc", true);
            yield return ("(a)b(c)", RegexOptions.None, "abc", true);
            yield return ("a+b+c", RegexOptions.None, "aabbabc", true);
            yield return ("a{1,}b{1,}c", RegexOptions.None, "aabbabc", true);
            yield return ("a.+?c", RegexOptions.None, "abcabc", true);
            yield return ("(a+|b)*", RegexOptions.None, "ab", true);
            yield return ("(a+|b){0,}", RegexOptions.None, "ab", true);
            yield return ("(a+|b)+", RegexOptions.None, "ab", true);
            yield return ("(a+|b){1,}", RegexOptions.None, "ab", true);
            yield return ("(a+|b)?", RegexOptions.None, "ab", true);
            yield return ("(a+|b){0,1}", RegexOptions.None, "ab", true);
            yield return ("[^ab]*", RegexOptions.None, "cde", true);
            yield return ("abc", RegexOptions.None, "b", false);
            yield return ("a*", RegexOptions.None, "", true);
            yield return ("([abc])*d", RegexOptions.None, "abbbcd", true);
            yield return ("([abc])*bcd", RegexOptions.None, "abcd", true);
            yield return ("a|b|c|d|e", RegexOptions.None, "e", true);
            yield return ("(a|b|c|d|e)f", RegexOptions.None, "ef", true);
            yield return ("abcd*efg", RegexOptions.None, "abcdefg", true);
            yield return ("ab*", RegexOptions.None, "xabyabbbz", true);
            yield return ("ab*", RegexOptions.None, "xayabbbz", true);
            yield return ("(ab|cd)e", RegexOptions.None, "abcde", true);
            yield return ("[abhgefdc]ij", RegexOptions.None, "hij", true);
            yield return ("(abc|)ef", RegexOptions.None, "abcdef", true);
            yield return ("(a|b)c*d", RegexOptions.None, "abcd", true);
            yield return ("(ab|ab*)bc", RegexOptions.None, "abc", true);
            yield return ("a([bc]*)c*", RegexOptions.None, "abc", true);
            yield return ("a([bc]*)(c*d)", RegexOptions.None, "abcd", true);
            yield return ("a([bc]+)(c*d)", RegexOptions.None, "abcd", true);
            yield return ("a([bc]*)(c+d)", RegexOptions.None, "abcd", true);
            yield return ("a[bcd]*dcdcde", RegexOptions.None, "adcdcde", true);
            yield return ("a[bcd]+dcdcde", RegexOptions.None, "abcde", false);
            yield return ("a[bcd]+dcdcde", RegexOptions.None, "adcdcde", false);
            yield return ("(ab|a)b*c", RegexOptions.None, "abc", true);
            yield return ("((a)(b)c)(d)", RegexOptions.None, "abcd", true);
            yield return ("[a-zA-Z_][a-zA-Z0-9_]*", RegexOptions.None, "alpha", true);
            yield return ("^a(bc+|b[eh])g|.h$", RegexOptions.None, "abh", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "effgz", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "ij", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "reffgz", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "effg", false);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.None, "bcdd", false);
            yield return ("((((((((((a))))))))))", RegexOptions.None, "a", true);
            yield return ("(((((((((a)))))))))", RegexOptions.None, "a", true);
            yield return ("multiple words of text", RegexOptions.None, "aa", false);
            yield return ("multiple words of text", RegexOptions.None, "uh-uh", false);
            yield return ("multiple words", RegexOptions.None, "multiple words, yeah", true);
            yield return ("(.*)c(.*)", RegexOptions.None, "abcde", true);
            yield return ("\\((.*), (.*)\\)", RegexOptions.None, "(a, b)", true);
            yield return ("abcd", RegexOptions.None, "abcd", true);
            yield return ("a(bc)d", RegexOptions.None, "abcd", true);
            yield return ("a[-]?c", RegexOptions.None, "ac", true);
            yield return ("abc", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("abc", RegexOptions.IgnoreCase, "XABCY", true);
            yield return ("abc", RegexOptions.IgnoreCase, "ABABC", true);
            yield return ("abc", RegexOptions.IgnoreCase, "aaxabxbaxbbx", false);
            yield return ("abc", RegexOptions.IgnoreCase, "XBC", false);
            yield return ("abc", RegexOptions.IgnoreCase, "AXC", false);
            yield return ("abc", RegexOptions.IgnoreCase, "ABX", false);
            yield return ("ab*c", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("ab*bc", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("ab*bc", RegexOptions.IgnoreCase, "ABBC", true);
            yield return ("ab*?bc", RegexOptions.IgnoreCase, "ABBBBC", true);
            yield return ("ab{0,}?bc", RegexOptions.IgnoreCase, "ABBBBC", true);
            yield return ("ab+?bc", RegexOptions.IgnoreCase, "ABBC", true);
            yield return ("ab+bc", RegexOptions.IgnoreCase, "ABC", false);
            yield return ("ab+bc", RegexOptions.IgnoreCase, "ABQ", false);
            yield return ("ab+bc", RegexOptions.IgnoreCase, "ABBBBC", true);
            yield return ("ab{1,}?bc", RegexOptions.IgnoreCase, "ABBBBC", true);
            yield return ("ab{1,3}?bc", RegexOptions.IgnoreCase, "ABBBBC", true);
            yield return ("ab{3,4}?bc", RegexOptions.IgnoreCase, "ABBBBC", true);
            yield return ("ab{4,5}?bc", RegexOptions.IgnoreCase, "ABQ", false);
            yield return ("ab{4,5}?bc", RegexOptions.IgnoreCase, "ABBBBC", false);
            yield return ("ab??bc", RegexOptions.IgnoreCase, "ABBC", true);
            yield return ("ab??bc", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("ab{0,1}?bc", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("ab??c", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("ab{0,1}?c", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("^abc$", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("^abc$", RegexOptions.IgnoreCase, "ABBBBC", false);
            yield return ("^abc$", RegexOptions.IgnoreCase, "ABCC", false);
            yield return ("^abc", RegexOptions.IgnoreCase, "ABCC", true);
            yield return ("abc$", RegexOptions.IgnoreCase, "AABC", true);
            yield return ("^", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("$", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("a.c", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("a.c", RegexOptions.IgnoreCase, "AXC", true);
            yield return ("a.*?c", RegexOptions.IgnoreCase, "AXYZC", true);
            yield return ("a.*c", RegexOptions.IgnoreCase, "AABC", true);
            yield return ("a.*c", RegexOptions.IgnoreCase, "AXYZD", false);
            yield return ("a[bc]d", RegexOptions.IgnoreCase, "ABD", true);
            yield return ("a[b-d]e", RegexOptions.IgnoreCase, "ACE", true);
            yield return ("a[b-d]e", RegexOptions.IgnoreCase, "ABC", false);
            yield return ("a[b-d]e", RegexOptions.IgnoreCase, "ABD", false);
            yield return ("a[b-d]", RegexOptions.IgnoreCase, "AAC", true);
            yield return ("a[-b]", RegexOptions.IgnoreCase, "A-", true);
            yield return ("a[b-]", RegexOptions.IgnoreCase, "A-", true);
            yield return ("a]", RegexOptions.IgnoreCase, "A]", true);
            yield return ("a[]]b", RegexOptions.IgnoreCase, "A]B", true);
            yield return ("a[^bc]d", RegexOptions.IgnoreCase, "AED", true);
            yield return ("a[^-b]c", RegexOptions.IgnoreCase, "ADC", true);
            yield return ("a[^-b]c", RegexOptions.IgnoreCase, "ABD", false);
            yield return ("a[^-b]c", RegexOptions.IgnoreCase, "A-C", false);
            yield return ("a[^]b]c", RegexOptions.IgnoreCase, "ADC", true);
            yield return ("ab|cd", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("ab|cd", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("()ef", RegexOptions.IgnoreCase, "DEF", true);
            yield return ("$b", RegexOptions.IgnoreCase, "A]C", false);
            yield return ("$b", RegexOptions.IgnoreCase, "B", false);
            yield return ("a\\(b", RegexOptions.IgnoreCase, "A(B", true);
            yield return ("a\\(*b", RegexOptions.IgnoreCase, "AB", true);
            yield return ("a\\(*b", RegexOptions.IgnoreCase, "A((B", true);
            yield return ("a\\\\b", RegexOptions.IgnoreCase, "A\\b", true);
            yield return ("a\\\\b", RegexOptions.IgnoreCase, "a\\B", true);
            yield return ("((a))", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("(a)b(c)", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("a+b+c", RegexOptions.IgnoreCase, "AABBABC", true);
            yield return ("a{1,}b{1,}c", RegexOptions.IgnoreCase, "AABBABC", true);
            yield return ("a.+?c", RegexOptions.IgnoreCase, "ABCABC", true);
            yield return ("a.*?c", RegexOptions.IgnoreCase, "ABCABC", true);
            yield return ("a.{0,5}?c", RegexOptions.IgnoreCase, "ABCABC", true);
            yield return ("(a+|b)*", RegexOptions.IgnoreCase, "AB", true);
            yield return ("(a+|b){0,}", RegexOptions.IgnoreCase, "AB", true);
            yield return ("(a+|b)+", RegexOptions.IgnoreCase, "AB", true);
            yield return ("(a+|b){1,}", RegexOptions.IgnoreCase, "AB", true);
            yield return ("(a+|b)?", RegexOptions.IgnoreCase, "AB", true);
            yield return ("(a+|b){0,1}", RegexOptions.IgnoreCase, "AB", true);
            yield return ("(a+|b){0,1}?", RegexOptions.IgnoreCase, "AB", true);
            yield return ("[^ab]*", RegexOptions.IgnoreCase, "CDE", true);
            yield return ("([abc])*d", RegexOptions.IgnoreCase, "ABBBCD", true);
            yield return ("([abc])*bcd", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("a|b|c|d|e", RegexOptions.IgnoreCase, "E", true);
            yield return ("(a|b|c|d|e)f", RegexOptions.IgnoreCase, "EF", true);
            yield return ("abcd*efg", RegexOptions.IgnoreCase, "ABCDEFG", true);
            yield return ("ab*", RegexOptions.IgnoreCase, "XABYABBBZ", true);
            yield return ("ab*", RegexOptions.IgnoreCase, "XAYABBBZ", true);
            yield return ("(ab|cd)e", RegexOptions.IgnoreCase, "ABCDE", true);
            yield return ("[abhgefdc]ij", RegexOptions.IgnoreCase, "HIJ", true);
            yield return ("^(ab|cd)e", RegexOptions.IgnoreCase, "ABCDE", false);
            yield return ("(abc|)ef", RegexOptions.IgnoreCase, "ABCDEF", true);
            yield return ("(a|b)c*d", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("(ab|ab*)bc", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("a([bc]*)c*", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("a([bc]*)(c*d)", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("a([bc]+)(c*d)", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("a([bc]*)(c+d)", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("a[bcd]*dcdcde", RegexOptions.IgnoreCase, "ADCDCDE", true);
            yield return ("(ab|a)b*c", RegexOptions.IgnoreCase, "ABC", true);
            yield return ("((a)(b)c)(d)", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("[a-zA-Z_][a-zA-Z0-9_]*", RegexOptions.IgnoreCase, "ALPHA", true);
            yield return ("^a(bc+|b[eh])g|.h$", RegexOptions.IgnoreCase, "ABH", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "EFFGZ", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "IJ", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "REFFGZ", true);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "ADCDCDE", false);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "EFFG", false);
            yield return ("(bc+d$|ef*g.|h?i(j|k))", RegexOptions.IgnoreCase, "BCDD", false);
            yield return ("((((((((((a))))))))))", RegexOptions.IgnoreCase, "A", true);
            yield return ("(((((((((a)))))))))", RegexOptions.IgnoreCase, "A", true);
            yield return ("(?:(?:(?:(?:(?:(?:(?:(?:(?:(a))))))))))", RegexOptions.IgnoreCase, "A", true);
            yield return ("(?:(?:(?:(?:(?:(?:(?:(?:(?:(a|b|c))))))))))", RegexOptions.IgnoreCase, "C", true);
            yield return ("multiple words of text", RegexOptions.IgnoreCase, "AA", false);
            yield return ("multiple words of text", RegexOptions.IgnoreCase, "UH-UH", false);
            yield return ("multiple words", RegexOptions.IgnoreCase, "MULTIPLE WORDS, YEAH", true);
            yield return ("(.*)c(.*)", RegexOptions.IgnoreCase, "ABCDE", true);
            yield return ("\\((.*), (.*)\\)", RegexOptions.IgnoreCase, "(A, B)", true);
            yield return ("abcd", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("a(bc)d", RegexOptions.IgnoreCase, "ABCD", true);
            yield return ("a[-]?c", RegexOptions.IgnoreCase, "AC", true);
            yield return ("a(?:b|c|d)(.)", RegexOptions.None, "ace", true);
            yield return ("a(?:b|c|d)*(.)", RegexOptions.None, "ace", true);
            yield return ("a(?:b|c|d)+?(.)", RegexOptions.None, "ace", true);
            yield return ("a(?:b|c|d)+?(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d)+(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){2}(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){4,5}(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){4,5}?(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("((foo)|(bar))*", RegexOptions.None, "foobar", true);
            yield return ("a(?:b|c|d){6,7}(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){6,7}?(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){5,6}(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){5,6}?(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){5,7}(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|c|d){5,7}?(.)", RegexOptions.None, "acdbcdbe", true);
            yield return ("a(?:b|(c|e){1,2}?|d)+?(.)", RegexOptions.None, "ace", true);
            yield return ("^(.+)?B", RegexOptions.None, "AB", true);
            yield return ("^([^a-z])|(\\^)$", RegexOptions.None, ".", true);
            yield return ("^[<>]&", RegexOptions.None, "<&OUT", true);
            yield return ("(?:(f)(o)(o)|(b)(a)(r))*", RegexOptions.None, "foobar", true);
            yield return ("(?:..)*a", RegexOptions.None, "aba", true);
            yield return ("(?:..)*?a", RegexOptions.None, "aba", true);
            yield return ("^(){3,5}", RegexOptions.None, "abc", true);
            yield return ("^(a+)*ax", RegexOptions.None, "aax", true);
            yield return ("^((a|b)+)*ax", RegexOptions.None, "aax", true);
            yield return ("^((a|bc)+)*ax", RegexOptions.None, "aax", true);
            yield return ("(a|x)*ab", RegexOptions.None, "cab", true);
            yield return ("(a)*ab", RegexOptions.None, "cab", true);
            yield return ("(?:(?i)a)b", RegexOptions.None, "ab", true);
            yield return ("((?i)a)b", RegexOptions.None, "ab", true);
            yield return ("(?:(?i)a)b", RegexOptions.None, "Ab", true);
            yield return ("((?i)a)b", RegexOptions.None, "Ab", true);
            yield return ("(?:(?i)a)b", RegexOptions.None, "cb", false);
            yield return ("(?:(?i)a)b", RegexOptions.None, "aB", false);
            yield return ("(?i:a)b", RegexOptions.None, "ab", true);
            yield return ("((?i:a))b", RegexOptions.None, "ab", true);
            yield return ("(?i:a)b", RegexOptions.None, "Ab", true);
            yield return ("((?i:a))b", RegexOptions.None, "Ab", true);
            yield return ("(?i:a)b", RegexOptions.None, "aB", false);
            yield return ("(?:(?-i)a)b", RegexOptions.IgnoreCase, "ab", true);
            yield return ("((?-i)a)b", RegexOptions.IgnoreCase, "ab", true);
            yield return ("(?:(?-i)a)b", RegexOptions.IgnoreCase, "aB", true);
            yield return ("((?-i)a)b", RegexOptions.IgnoreCase, "aB", true);
            yield return ("(?:(?-i)a)b", RegexOptions.IgnoreCase, "Ab", false);
            yield return ("(?:(?-i)a)b", RegexOptions.IgnoreCase, "AB", false);
            yield return ("(?-i:a)b", RegexOptions.IgnoreCase, "ab", true);
            yield return ("((?-i:a))b", RegexOptions.IgnoreCase, "ab", true);
            yield return ("(?-i:a)b", RegexOptions.IgnoreCase, "aB", true);
            yield return ("((?-i:a))b", RegexOptions.IgnoreCase, "aB", true);
            yield return ("(?-i:a)b", RegexOptions.IgnoreCase, "AB", false);
            yield return ("(?-i:a)b", RegexOptions.IgnoreCase, "Ab", false);
            yield return ("((?-i:a.))b", RegexOptions.IgnoreCase, "AB", false);
            yield return ("((?-i:a.))b", RegexOptions.IgnoreCase, "a\nB", false);
            yield return ("((?s-i:a.))b", RegexOptions.IgnoreCase, "a\nB", true);
            yield return ("(?:c|d)(?:)(?:a(?:)(?:b)(?:b(?:))(?:b(?:)(?:b)))", RegexOptions.None, "cabbbb", true);
            yield return ("(?:c|d)(?:)(?:aaaaaaaa(?:)(?:bbbbbbbb)(?:bbbbbbbb(?:))(?:bbbbbbbb(?:)(?:bbbbbbbb)))", RegexOptions.None, "caaaaaaaabbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb", true);
            yield return ("foo\\w*\\d{4}baz", RegexOptions.None, "foobar1234baz", true);
            yield return ("x(~~)*(?:(?:F)?)?", RegexOptions.None, "x~~", true);
            yield return ("^a(?#xxx){3}c", RegexOptions.None, "aaac", true);
            yield return ("^a (?#xxx) (?#yyy) {3}c", RegexOptions.IgnorePatternWhitespace, "aaac", true);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "", true);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "a", true);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "ab", true);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "aaa", true);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "dbcb", false);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "a--", false);
            yield return ("^(?:a?b?)*$", RegexOptions.None, "aa--", false);
            yield return ("((?s)^a(.))((?m)^b$)", RegexOptions.None, "a\nb\nc\n", true);
            yield return ("((?m)^b$)", RegexOptions.None, "a\nb\nc\n", true);
            yield return ("(?m)^b", RegexOptions.None, "a\nb\n", true);
            yield return ("(?m)^(b)", RegexOptions.None, "a\nb\n", true);
            yield return ("((?m)^b)", RegexOptions.None, "a\nb\n", true);
            yield return ("\\n((?m)^b)", RegexOptions.None, "a\nb\n", true);
            yield return ("()^b", RegexOptions.None, "a\nb\nc\n", false);
            yield return ("((?m)^b)", RegexOptions.None, "a\nb\nc\n", true);
            yield return ("(\\w+:)+", RegexOptions.None, "one:", true);
            yield return ("([\\w:]+::)?(\\w+)$", RegexOptions.None, "abcd", true);
            yield return ("([\\w:]+::)?(\\w+)$", RegexOptions.None, "xy:z:::abcd", true);
            yield return ("^[^bcd]*(c+)", RegexOptions.None, "aexycd", true);
            yield return ("(a*)b+", RegexOptions.None, "caab", true);
            yield return ("([\\w:]+::)?(\\w+)$", RegexOptions.None, "abcd:", false);
            yield return ("([[:]+)", RegexOptions.None, "a:[b]:", true);
            yield return ("([[=]+)", RegexOptions.None, "a=[b]=", true);
            yield return ("([[.]+)", RegexOptions.None, "a.[b].", true);
            yield return ("a\\Z", RegexOptions.None, "aaab", false);
            yield return ("a\\Z", RegexOptions.None, "a\nb\n", false);
            yield return ("b\\Z", RegexOptions.None, "a\nb\n", true);
            yield return ("b\\Z", RegexOptions.None, "a\nb", true);
            yield return ("b\\z", RegexOptions.None, "a\nb", true);
            yield return ("((Z)+|A)*", RegexOptions.None, "ZABCDEFG", true);
            yield return ("(Z()|A)*", RegexOptions.None, "ZABCDEFG", true);
            yield return ("(Z(())|A)*", RegexOptions.None, "ZABCDEFG", true);
            yield return ("a*", RegexOptions.None, "abbab", true);
            yield return ("[\\s]+", RegexOptions.None, "> \t\n\f\r\v<", true);
            yield return ("\\s+", RegexOptions.None, "> \t\n\f\r\v<", true);
            // yield return ("a\vb", RegexOptions.IgnorePatternWhitespace, "ab", true); https://github.com/dotnet/runtime/issues/73206
            yield return ("abc.", RegexOptions.None, "abc1abc2xyzabc3", true);
            yield return ("a(?x: b c )d", RegexOptions.None, "XabcdY", true);
            yield return ("a(?x: b c )d", RegexOptions.None, "Xa b c d Y", false);
            yield return ("((?x)x y z | a b c)", RegexOptions.None, "XabcY", true);
            yield return ("((?x)x y z | a b c)", RegexOptions.None, "AxyzB", true);
            yield return ("(?i)AB(?-i)C", RegexOptions.None, "XabCY", true);
            yield return ("(?i)AB(?-i)C", RegexOptions.None, "XabcY", false);
            yield return ("((?i)AB(?-i)C|D)E", RegexOptions.None, "abCE", true);
            yield return ("((?i)AB(?-i)C|D)E", RegexOptions.None, "DE", true);
            yield return ("((?i)AB(?-i)C|D)E", RegexOptions.None, "abcE", false);
            yield return ("((?i)AB(?-i)C|D)E", RegexOptions.None, "abCe", false);
            yield return ("((?i)AB(?-i)C|D)E", RegexOptions.None, "dE", false);
            yield return ("((?i)AB(?-i)C|D)E", RegexOptions.None, "De", false);
            // yield return ("(a+)*b", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", false); // https://github.com/dotnet/runtime/issues/73208
            yield return ("(?i)reg(?:ul(?:[a\u00E4]|ae)r|ex)", RegexOptions.None, "REGular", true);
            yield return ("(?i)reg(?:ul(?:[a\u00E4]|ae)r|ex)", RegexOptions.None, "regulaer", true);
            yield return ("(?i)reg(?:ul(?:[a\u00E4]|ae)r|ex)", RegexOptions.None, "Regex", true);
            yield return ("(?i)reg(?:ul(?:[a\u00E4]|ae)r|ex)", RegexOptions.None, "regul\u00E4r", true);
            yield return ("\u00C5\u00E6\u00E5\u00E4[\u00E0-\u00FF\u00C0-\u00DF]+", RegexOptions.None, "\u00C5\u00E6\u00E5\u00E4\u00E0", true);
            yield return ("\u00C5\u00E6\u00E5\u00E4[\u00E0-\u00FF\u00C0-\u00DF]+", RegexOptions.None, "\u00C5\u00E6\u00E5\u00E4\u00FF", true);
            yield return ("\u00C5\u00E6\u00E5\u00E4[\u00E0-\u00FF\u00C0-\u00DF]+", RegexOptions.None, "\u00C5\u00E6\u00E5\u00E4\u00C0", true);
            yield return ("\u00C5\u00E6\u00E5\u00E4[\u00E0-\u00FF\u00C0-\u00DF]+", RegexOptions.None, "\u00C5\u00E6\u00E5\u00E4\u00DF", true);
            yield return ("ab cd (?x) de fg", RegexOptions.None, "ab cd defg", true);
            yield return ("ab cd(?x) de fg", RegexOptions.None, "ab cddefg", true);
            yield return ("ab cd(?x) de fg", RegexOptions.None, "abcddefg", false);
            yield return ("^", RegexOptions.Multiline, "a\nb\nc\n", true);
            yield return ("^", RegexOptions.Multiline, "", true);
            yield return ("[[,abc,]+]", RegexOptions.None, "abc]", true);
            yield return ("[[,abc,]+]", RegexOptions.None, "a,b]", true);
            yield return ("[[,abc,]+]", RegexOptions.None, "[a,b,c]", true);
            yield return ("(?-x: )", RegexOptions.IgnorePatternWhitespace, "A B", true);
            yield return ("(?x)(?-x: \\s*#\\s*)", RegexOptions.None, "A # B", true);
            yield return ("(?x)(?-x: \\s*#\\s*)", RegexOptions.None, "#", false);
            yield return ("(?x)(?-x: \\s*#\\s*)", RegexOptions.None, "A s#s B", false);
            yield return ("(?x-is)(?:(?-ixs) \\s*#\\s*) include", RegexOptions.None, "A #include", true);
            yield return ("(?x-is)(?:(?-ixs) \\s*#\\s*) include", RegexOptions.None, "A#include", false);
            yield return ("(?x-is)(?:(?-ixs) \\s*#\\s*) include", RegexOptions.None, "A #Include", false);
            yield return ("a*b*\\w", RegexOptions.None, "aaabbbb", true);
            yield return ("a*b*\\w", RegexOptions.None, "aaaa", true);
            yield return ("a*b*\\w", RegexOptions.None, "a", true);
            yield return ("a*b?\\w", RegexOptions.None, "aaabbbb", true);
            yield return ("a*b?\\w", RegexOptions.None, "aaaa", true);
            yield return ("a*b?\\w", RegexOptions.None, "a", true);
            yield return ("a*b{0,4}\\w", RegexOptions.None, "aaabbbb", true);
            yield return ("a*b{0,4}\\w", RegexOptions.None, "aaaa", true);
            yield return ("a*b{0,4}\\w", RegexOptions.None, "a", true);
            yield return ("a*b{0,}\\w", RegexOptions.None, "aaabbbb", true);
            yield return ("a*b{0,}\\w", RegexOptions.None, "aaaa", true);
            yield return ("a*b{0,}\\w", RegexOptions.None, "a", true);
            yield return ("a*\\d*\\w", RegexOptions.None, "0a", true);
            yield return ("a*\\d*\\w", RegexOptions.None, "a", true);
            yield return ("a*b *\\w", RegexOptions.IgnorePatternWhitespace, "a", true);
            yield return ("a*b#comment\n  *\\w", RegexOptions.IgnorePatternWhitespace, "a", true);
            yield return ("a* b *\\w", RegexOptions.IgnorePatternWhitespace, "a", true);
            yield return ("^\\w+=.*(\\\\\\n.*)*", RegexOptions.None, "abc=xyz\\\npqr", true);
            yield return ("^(a()*)*", RegexOptions.None, "aaaa", true);
            yield return ("^(?:a(?:(?:))*)*", RegexOptions.None, "aaaa", true);
            yield return ("^(a()+)+", RegexOptions.None, "aaaa", true);
            yield return ("^(?:a(?:(?:))+)+", RegexOptions.None, "aaaa", true);
            yield return ("(a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa4", true);
            yield return ("(a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", false);
            yield return ("(?:a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa4", true);
            yield return ("(?:a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", false);
            yield return ("\\Z", RegexOptions.None, "abc\n", true);
            yield return ("(.*(.)?)*", RegexOptions.None, "abcd", true);
            yield return ("[[:abcd:xyz]]", RegexOptions.None, "a]", true);
            yield return ("[[:abcd:xyz]]", RegexOptions.None, ":]", true);
            yield return ("[abc[:x\\]pqr]", RegexOptions.None, "a", true);
            yield return ("[abc[:x\\]pqr]", RegexOptions.None, "[", true);
            yield return ("[abc[:x\\]pqr]", RegexOptions.None, ":", true);
            yield return ("[abc[:x\\]pqr]", RegexOptions.None, "]", true);
            yield return ("[abc[:x\\]pqr]", RegexOptions.None, "p", true);
            yield return (".*[op][xyz]", RegexOptions.None, "fooabcfoo", false);
            yield return ("(?i)a(?-i)b|c", RegexOptions.None, "XabX", true);
            yield return ("(?i)a(?-i)b|c", RegexOptions.None, "XAbX", true);
            yield return ("(?i)a(?-i)b|c", RegexOptions.None, "CcC", true);
            yield return ("(?i)a(?-i)b|c", RegexOptions.None, "XABX", false);
            yield return ("[\\x00-\\xff\\s]+", RegexOptions.None, "\n\v\f\r", true);
            yield return ("[^a]*", RegexOptions.IgnoreCase, "12abc", true);
            yield return ("[^a]*", RegexOptions.IgnoreCase, "12ABC", true);
            yield return ("[^a]*?X", RegexOptions.IgnoreCase, "12abc", false);
            yield return ("[^a]*?X", RegexOptions.IgnoreCase, "12ABC", false);
            yield return ("[^a]+?X", RegexOptions.IgnoreCase, "12abc", false);
            yield return ("[^a]+?X", RegexOptions.IgnoreCase, "12ABC", false);
            yield return ("[^a]?X", RegexOptions.IgnoreCase, "12aXbcX", true);
            yield return ("[^a]?X", RegexOptions.IgnoreCase, "12AXBCX", true);
            yield return ("[^a]?X", RegexOptions.IgnoreCase, "BCX", true);
            yield return ("[^a]??X", RegexOptions.IgnoreCase, "12aXbcX", true);
            yield return ("[^a]??X", RegexOptions.IgnoreCase, "12AXBCX", true);
            yield return ("[^a]??X", RegexOptions.IgnoreCase, "BCX", true);
            yield return ("[^a]{2,3}", RegexOptions.IgnoreCase, "abcdef", true);
            yield return ("[^a]{2,3}", RegexOptions.IgnoreCase, "ABCDEF", true);
            yield return ("[^a]{2,3}?", RegexOptions.IgnoreCase, "abcdef", true);
            yield return ("[^a]{2,3}?", RegexOptions.IgnoreCase, "ABCDEF", true);
            yield return ("((a|)+)+Z", RegexOptions.None, "Z", true);
            yield return ("(a)b|(a)c", RegexOptions.None, "ac", true);
            yield return ("(?:a+|ab)+c", RegexOptions.None, "aabc", true);
            yield return ("^(?:a|ab)+c", RegexOptions.None, "aaaabc", true);
            yield return ("(?=abc){0}xyz", RegexOptions.None, "xyz", true);
            yield return ("^(?!a){0}\\w+", RegexOptions.None, "aaaaa", true);
            yield return ("^[:a\\d]+", RegexOptions.None, "aaaa444:::Z", true);
            yield return ("^[:a\\d:b]+", RegexOptions.None, "aaaa444:::bbbZ", true);
            yield return ("[:a]xxx[b:]", RegexOptions.None, ":xxx:", true);
            yield return ("\\A.*?(a|bc)", RegexOptions.None, "ba", true);
            yield return ("\\A.*?(?:a|bc|d)", RegexOptions.None, "ba", true);
            yield return ("((?:a?)*)*c", RegexOptions.None, "aac", true);
            yield return ("\\A.*?(?:a|bc)", RegexOptions.None, "ba", true);
            yield return ("^\\d*\\w{4}", RegexOptions.None, "1234", true);
            yield return ("^\\d*\\w{4}", RegexOptions.None, "123", false);
            yield return ("^[^b]*\\w{4}", RegexOptions.None, "aaaa", true);
            yield return ("^[^b]*\\w{4}", RegexOptions.None, "aaa", false);
            yield return ("^[^b]*\\w{4}", RegexOptions.IgnoreCase, "aaaa", true);
            yield return ("^[^b]*\\w{4}", RegexOptions.IgnoreCase, "aaa", false);
            yield return ("^a*\\w{4}", RegexOptions.None, "aaaa", true);
            yield return ("^a*\\w{4}", RegexOptions.None, "aaa", false);
            yield return ("^a*\\w{4}", RegexOptions.IgnoreCase, "aaaa", true);
            yield return ("^a*\\w{4}", RegexOptions.IgnoreCase, "aaa", false);
            yield return ("(?:x|(?:(xx|yy)+|x|x|x|x|x)|a|a|a)bc", RegexOptions.None, "acb", false);
            yield return ("\\sabc", RegexOptions.None, "\vabc", true);
            yield return ("(|ab)*?d", RegexOptions.None, "abd", true);
            yield return ("(|ab)*?d", RegexOptions.None, "xyd", true);
            yield return ("a+(?:|b)a", RegexOptions.None, "aaaa", true);
            yield return ("<(?x:[a b])>", RegexOptions.None, "< >", true);
            yield return ("<(?-x:[a b])>", RegexOptions.None, "< >", true);
            yield return ("[\\d-]+", RegexOptions.None, "12-24", true);
            yield return ("^(a?)+b", RegexOptions.None, "b", true);
            yield return ("^(a?)+b", RegexOptions.None, "ab", true);
            yield return ("^(a?)+b", RegexOptions.None, "aaab", true);
            yield return ("   (?<word> \\w+ )*    \\.   ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "pokus.", true);
            yield return ("a{1,2,3}b", RegexOptions.None, "a{1,2,3}b", true);
            yield return ("\\214748364", RegexOptions.None, ">\u008c748364<", true);
            yield return ("a{65536", RegexOptions.None, ">a{65536<", true);
            yield return ("^", RegexOptions.Multiline, "a\rb\nc\r\nxyz=aftertext", true);
            yield return ("a{1,3}b", RegexOptions.None, "ab", true);
            yield return ("(a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa4", true);
            yield return ("^a.b", RegexOptions.None, "a\rb", true);
            yield return ("^a.b", RegexOptions.None, "a\nb", false);
            yield return ("^a.b", RegexOptions.None, "a\u0085b", true);
            yield return ("^abc.", RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace, "abc1 \nabc2 \vabc3xx \fabc4 \rabc5xx \r\nabc6 \u0085abc7 JUNK", true);
            yield return ("abc.$", RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace, "abc1\n abc2\v abc3\f abc4\r abc5\r\n abc6\u0085 abc7 abc9", true);
            yield return (".+foo", RegexOptions.None, "afoo", true);
            yield return (".+foo", RegexOptions.None, "\r\nfoo", false);
            yield return (".+foo", RegexOptions.None, "\nfoo", false);
            yield return (".+foo", RegexOptions.Singleline, "afoo", true);
            yield return (".+foo", RegexOptions.Singleline, "\r\nfoo", true);
            yield return (".+foo", RegexOptions.Singleline, "\nfoo", true);
            yield return ("^$", RegexOptions.Multiline, "abc\r\nxyz", false);
            yield return ("(?m)^$", RegexOptions.None, "abc\r\n\r\n", true);
            yield return ("(?m)^$|^\\r\\n", RegexOptions.None, "abc\r\n\r\n", true);
            yield return ("(?m)$", RegexOptions.None, "abc\r\n\r\n", true);
            yield return ("abc.$", RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace, "abc1\n abc2\v abc3\f abc4\r abc5\r\n abc6\u0085 abc9", true);
            yield return ("^X", RegexOptions.Multiline, "XABC", true);
            yield return (".+A", RegexOptions.None, "\r\nA", false);
            yield return ("\\nA", RegexOptions.None, "\r\nA", true);
            yield return ("[\\r\\n]A", RegexOptions.None, "\r\nA", true);
            yield return ("(\\r|\\n)A", RegexOptions.None, "\r\nA", true);
            yield return ("TA]", RegexOptions.None, "The ACTA] comes", true);
            yield return ("(?!)?a", RegexOptions.None, "ab", true);
            yield return ("xyz", RegexOptions.None, "abc", false);
            yield return ("xyz", RegexOptions.None, "abcxypqr", false);
            yield return ("Xa{2,4}b", RegexOptions.None, "X=ps", false);
            yield return ("Xa{2,4}b", RegexOptions.None, "Xa=ps", false);
            yield return ("Xa{2,4}b", RegexOptions.None, "Xaa=ps", false);
            yield return ("Xa{2,4}b", RegexOptions.None, "Xaaa=ps", false);
            yield return ("Xa{2,4}b", RegexOptions.None, "Xaaaa=ps", false);
            yield return ("Xa{2,4}?b", RegexOptions.None, "X=ps", false);
            yield return ("Xa{2,4}?b", RegexOptions.None, "Xa=ps", false);
            yield return ("Xa{2,4}?b", RegexOptions.None, "Xaa=ps", false);
            yield return ("Xa{2,4}?b", RegexOptions.None, "Xaaa=ps", false);
            yield return ("Xa{2,4}?b", RegexOptions.None, "Xaaaa=ps", false);
            yield return ("X\\d{2,4}b", RegexOptions.None, "X=ps", false);
            yield return ("X\\d{2,4}b", RegexOptions.None, "X3=ps", false);
            yield return ("X\\d{2,4}b", RegexOptions.None, "X33=ps", false);
            yield return ("X\\d{2,4}b", RegexOptions.None, "X333=ps", false);
            yield return ("X\\d{2,4}b", RegexOptions.None, "X3333=ps", false);
            yield return ("X\\d{2,4}?b", RegexOptions.None, "X=ps", false);
            yield return ("X\\d{2,4}?b", RegexOptions.None, "X3=ps", false);
            yield return ("X\\d{2,4}?b", RegexOptions.None, "X33=ps", false);
            yield return ("X\\d{2,4}?b", RegexOptions.None, "X333=ps", false);
            yield return ("X\\d{2,4}?b", RegexOptions.None, "X3333=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "X=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "Xa=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "Xaa=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "Xaaa=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "Xaaaa=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "X=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "Xa=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "Xaa=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "Xaaa=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "Xaaaa=ps", false);
            yield return ("X[abc]{2,4}b", RegexOptions.None, "X=ps", false);
            yield return ("X[abc]{2,4}b", RegexOptions.None, "Xa=ps", false);
            yield return ("X[abc]{2,4}b", RegexOptions.None, "Xaa=ps", false);
            yield return ("X[abc]{2,4}b", RegexOptions.None, "Xaaa=ps", false);
            yield return ("X[abc]{2,4}b", RegexOptions.None, "Xaaaa=ps", false);
            yield return ("X[abc]{2,4}?b", RegexOptions.None, "X=ps", false);
            yield return ("X[abc]{2,4}?b", RegexOptions.None, "Xa=ps", false);
            yield return ("X[abc]{2,4}?b", RegexOptions.None, "Xaa=ps", false);
            yield return ("X[abc]{2,4}?b", RegexOptions.None, "Xaaa=ps", false);
            yield return ("X[abc]{2,4}?b", RegexOptions.None, "Xaaaa=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "X=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "Xz=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "Xzz=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "Xzzz=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "Xzzzz=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "X=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "Xz=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "Xzz=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "Xzzz=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "Xzzzz=ps", false);
            yield return ("Z(?!)", RegexOptions.None, "Z=ps", false);
            yield return ("Z(?!)", RegexOptions.None, "ZA=ps", false);
            yield return ("dog(sbody)?", RegexOptions.None, "dogs=ps", true);
            yield return ("dog(sbody)??", RegexOptions.None, "dogs=ps", true);
            yield return ("dog(sbody)??", RegexOptions.None, "dogs=ph", true);
            yield return ("dog|dogsbody", RegexOptions.None, "dogs=ps", true);
            yield return ("dog|dogsbody", RegexOptions.None, "dogs=ph", true);
            yield return ("dogsbody|dog", RegexOptions.None, "dogs=ps", true);
            yield return ("\\bthe cat\\b", RegexOptions.None, "the cat=ps", true);
            yield return ("abc", RegexOptions.None, "abc=ps", true);
            yield return ("abc", RegexOptions.None, "abc=ph", true);
            yield return ("\\babc\\b", RegexOptions.None, "+++abc+++", true);
            yield return ("\\babc\\b", RegexOptions.None, "+++ab=ps", false);
            yield return ("\\babc\\b", RegexOptions.None, "+++ab=ph", false);
            yield return ("a?b?", RegexOptions.None, "xyz", true);
            yield return ("a?b?", RegexOptions.None, "xyzabc", true);
            yield return ("a?b?", RegexOptions.None, "xyzabc=notempty", true);
            yield return ("a?b?", RegexOptions.None, "xyzabc=notempty_atstart", true);
            yield return ("a?b?", RegexOptions.None, "xyz=notempty_atstart", true);
            yield return ("^a?b?", RegexOptions.None, "xyz", true);
            yield return ("^a?b?", RegexOptions.None, "xyzabc", true);
            yield return ("abcd*", RegexOptions.None, "xxxxabcd=ps", true);
            yield return ("abcd*", RegexOptions.IgnoreCase, "xxxxabcd=ps", true);
            yield return ("abcd*", RegexOptions.IgnoreCase, "XXXXABCD=ph", true);
            yield return ("abc\\d*", RegexOptions.None, "xxxxabc1=ps", true);
            yield return ("abc[de]*", RegexOptions.None, "xxxxabcde=ps", true);
            yield return ("abc$", RegexOptions.None, "abc=ph", false);
            yield return ("abc$", RegexOptions.Multiline, "abc", true);
            yield return ("abc$", RegexOptions.Multiline, "abc\n", true);
            yield return ("abc$", RegexOptions.Multiline, "abc=ph", false);
            yield return ("abc$", RegexOptions.Multiline, "abc\n=ps", true);
            yield return ("abc\\z", RegexOptions.None, "abc", true);
            yield return ("abc\\z", RegexOptions.None, "abc=ph", false);
            yield return ("abc\\Z", RegexOptions.None, "abc", true);
            yield return ("abc\\Z", RegexOptions.None, "abc=ph", false);
            yield return ("abc\\b", RegexOptions.None, "abc", true);
            yield return ("abc\\b", RegexOptions.None, "abc=ps", true);
            yield return ("abc\\B", RegexOptions.None, "abc=ps", false);
            yield return ("abc\\B", RegexOptions.None, "abc=ph", false);
            yield return ("abc\\B", RegexOptions.None, "abc", false);
            yield return (".+", RegexOptions.None, "abc=offset=2", true);
            yield return (".+", RegexOptions.None, "abc=offset=3", true);
            yield return ("(a)b|ac", RegexOptions.None, "ac=ovector=1", true);
            yield return ("(a)(b)x|abc", RegexOptions.None, "abc=ovector=2", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "bazfooX", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "foobazbarX", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "barfooX", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "bazX", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "foobarbazX", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "bazfooX=ovector=0", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "bazfooX=ovector=2", true);
            yield return ("(?:(foo)|(bar)|(baz))X", RegexOptions.None, "bazfooX=ovector=3", true);
            yield return ("^a\\x41z", RegexOptions.None, "aAz", true);
            yield return ("^a\\x41z", RegexOptions.None, "ax41z", false);
            yield return ("^a[m\\x41]z", RegexOptions.None, "aAz", true);
            yield return ("^a\\u0041z", RegexOptions.None, "aAz", true);
            yield return ("^a\\u0041z", RegexOptions.None, "au0041z", false);
            yield return ("^a[m\\u0041]z", RegexOptions.None, "aAz", true);
            yield return ("^a$", RegexOptions.None, "a\r=ps", false);
            yield return ("^a$", RegexOptions.None, "a\r=ph", false);
            yield return ("^a$", RegexOptions.Multiline, "a\r=ps", false);
            yield return ("^a$", RegexOptions.Multiline, "a\r=ph", false);
            yield return ("^(a$|a\\r)", RegexOptions.None, "a\r=ps", true);
            yield return ("^(a$|a\\r)", RegexOptions.Multiline, "a\r=ps", true);
            yield return (".", RegexOptions.None, "\r=ps", true);
            yield return (".{2,3}", RegexOptions.None, "\r\r\r=ps", true);
            yield return (".{2,3}?", RegexOptions.None, "\r\r\r=ps", true);
            yield return ("(ab)x|ab", RegexOptions.None, "ab=ovector=0", true);
            yield return ("(ab)x|ab", RegexOptions.None, "ab=ovector=1", true);
            yield return ("[ab]*", RegexOptions.None, "aaaa", true);
            yield return ("[ab]*?", RegexOptions.None, "aaaa", true);
            yield return ("[ab]?", RegexOptions.None, "aaaa", true);
            yield return ("[ab]??", RegexOptions.None, "aaaa", true);
            yield return ("[ab]+", RegexOptions.None, "aaaa", true);
            yield return ("[ab]+?", RegexOptions.None, "aaaa", true);
            yield return ("[ab]{2,3}", RegexOptions.None, "aaaa", true);
            yield return ("[ab]{2,3}?", RegexOptions.None, "aaaa", true);
            yield return ("[ab]{2,}", RegexOptions.None, "aaaa", true);
            yield return ("[ab]{2,}?", RegexOptions.None, "aaaa", true);
            yield return ("[A-[!xxx!]]+", RegexOptions.None, "A]]]", true);
            yield return ("a*?b*?", RegexOptions.None, "ab", true);
            yield return ("(?<A>a)|(?<A>b)", RegexOptions.None, "a=ovector=2,copy=A,get=A,get=2", true);
            yield return ("a(b)c(d)", RegexOptions.None, "abc=ph,copy=0,copy=1,getall", false);
            yield return ("abc", RegexOptions.None, "[9x!xxx(]{9999}", false);
            yield return ("(abc)*", RegexOptions.None, "[abc]{5}", true);
            yield return ("^", RegexOptions.Multiline, "\n\n\n", true);
            yield return ("^", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace, "\n\n\n", true);
            yield return ("ab{3cd", RegexOptions.None, "ab{3cd", true);
            yield return ("ab{3,cd", RegexOptions.None, "ab{3,cd", true);
            yield return ("ab{3,4a}cd", RegexOptions.None, "ab{3,4a}cd", true);
            yield return ("{4,5a}bc", RegexOptions.None, "{4,5a}bc", true);
            yield return ("(?:ab)?(?:ab)(?:ab)", RegexOptions.None, "abab", true);
            yield return ("(?:ab)?(?:ab)(?:ab)", RegexOptions.None, "ababab", true);
            yield return ("(?:ab)?(?:ab)(?:ab)", RegexOptions.None, "aba", false);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "1234abcde=offset_limit=100", true);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "1234abcde=offset_limit=9", true);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "1234abcde=offset_limit=4", true);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "1234abcde=offset_limit=4,offset=4", true);
            yield return ("^\\w", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "\n..\naa=offset_limit=4", true);
            yield return ("abcd", RegexOptions.IgnorePatternWhitespace, "abcd=null_context", true);
            yield return ("aaa", RegexOptions.None, "[abc]{10000000000000000000000000000}", false);
            yield return ("aaa", RegexOptions.None, "[a]{3}", false);
            yield return ("(R?){65}", RegexOptions.None, "(R?){65}", true);
            yield return ("\\[(a)]{60}", RegexOptions.IgnorePatternWhitespace, "aaaa", false);
            yield return ("abc", RegexOptions.None, "xyzabc", true);
            yield return ("abc", RegexOptions.None, "xyzabc=endanchored", true);
            yield return ("abc|bcd", RegexOptions.None, "xyzabcd", true);
            yield return ("^a+?x", RegexOptions.IgnoreCase, "aaa", false);
            yield return ("^[^a]{3,}?x", RegexOptions.IgnoreCase, "bbb", false);
            yield return ("^[^a]{3,}?x", RegexOptions.IgnoreCase, "cc", false);
            yield return ("^X\\S", RegexOptions.None, "X", false);
            yield return ("^X\\W", RegexOptions.None, "X", false);
            yield return ("^X\\v", RegexOptions.None, "X", false);
            yield return ("^X\\v", RegexOptions.None, "XX", false);
            yield return ("^X.+?", RegexOptions.Singleline, "X", false);
            yield return ("^X\\D+?", RegexOptions.None, "X", false);
            yield return ("^X\\D+?", RegexOptions.None, "X9", false);
            yield return ("^X\\S+?", RegexOptions.None, "X", false);
            yield return ("^X\\S+?", RegexOptions.None, "X\n", false);
            yield return ("^X\\W+?", RegexOptions.None, "X", false);
            yield return ("^X\\W+?", RegexOptions.None, "XX", false);
            yield return ("^X.+?Z", RegexOptions.None, "XY\n", false);
            yield return ("^X\\v+?Z", RegexOptions.None, "X\n\n", false);
            yield return ("^X\\v+?Z", RegexOptions.None, "X\nY", false);
            yield return ("^X\\D+?Z", RegexOptions.None, "XY9", false);
            yield return ("^X\\D+?Z", RegexOptions.None, "XYY", false);
            yield return ("^X\\d+?Z", RegexOptions.None, "X99", false);
            yield return ("^X\\d+?Z", RegexOptions.None, "X9Y", false);
            yield return ("^X\\S+?Z", RegexOptions.None, "XY\n", false);
            yield return ("^X\\S+?Z", RegexOptions.None, "XYY", false);
            yield return ("^X\\s+?Z", RegexOptions.None, "X\n\n", false);
            yield return ("^X\\s+?Z", RegexOptions.None, "X\nY", false);
            yield return ("^X\\W+?Z", RegexOptions.None, "X.A", false);
            yield return ("^X\\W+?Z", RegexOptions.None, "X++", false);
            yield return ("^X\\w+?Z", RegexOptions.None, "Xa.", false);
            yield return ("^X\\w+?Z", RegexOptions.None, "Xaa", false);
            yield return ("^X.{1,3}Z", RegexOptions.Singleline, "Xa.bd", false);
            yield return ("a whole line", RegexOptions.IgnoreCase | RegexOptions.Multiline, "Rhubarb \na whole line\n custard", true);
            yield return ("^(b+|a){1,2}?bc", RegexOptions.None, "bbc", true);
            yield return ("^(b*|ba){1,2}?bc", RegexOptions.None, "babc", true);
            yield return ("^(b*|ba){1,2}?bc", RegexOptions.None, "bbabc", true);
            yield return ("^(b*|ba){1,2}?bc", RegexOptions.None, "bababc", true);
            yield return ("^(b*|ba){1,2}?bc", RegexOptions.None, "bababbc", false);
            yield return ("^(b*|ba){1,2}?bc", RegexOptions.None, "babababc", false);
            yield return ("\\n", RegexOptions.None, "xyz\nabc", true);
            yield return ("\\nabc", RegexOptions.None, "xyz\nabc", true);
            yield return ("a(b)c|xyz", RegexOptions.None, "abcdef=ovector=4", true);
            yield return ("a(b)c|xyz", RegexOptions.None, "abxyz=ovector=4", true);
            yield return ("abc\\rdef", RegexOptions.None, "abc\ndef", false);
            yield return ("foobar", RegexOptions.None, "the foobar thing=copy_matched_subject", true);
            yield return ("foobar", RegexOptions.None, "the foobar thing=copy_matched_subject,zero_terminate", true);
            yield return ("foobar", RegexOptions.None, "the foobar thing foobar again=copy_matched_subject", true);
            yield return ("\\Z", RegexOptions.None, "abc\n=ph", true);
            yield return ("\\sxxx\\s", RegexOptions.Singleline, "AB\u0085xxx\u00A0XYZ", true);
            yield return ("^\\w+", RegexOptions.Singleline, "\u00C3\u0089cole", true);
            yield return ("", RegexOptions.None, "=null_subject", true);
            yield return ("^[\\w]+", RegexOptions.None, "\u00C9cole", true);
            yield return ("^[\\W]+", RegexOptions.None, "\u00C9cole", false);
            yield return ("[\\b]", RegexOptions.None, "a", false);
            yield return ("^\\w+", RegexOptions.None, "\u00C9cole", true);
            yield return ("(.+)\\b(.+)", RegexOptions.None, "\u00C9cole", false);
            yield return ("\u00C9cole", RegexOptions.IgnoreCase, "\u00C9cole", true);
            yield return ("\u00C9cole", RegexOptions.IgnoreCase, "\u00E9cole", true);
            yield return ("^[\\xc8-\\xc9]", RegexOptions.IgnoreCase, "\u00C9cole", true);
            yield return ("^[\\xc8-\\xc9]", RegexOptions.IgnoreCase, "\u00E9cole", true);
            yield return ("^[\\xc8-\\xc9]", RegexOptions.None, "\u00C9cole", true);
            yield return ("^[\\xc8-\\xc9]", RegexOptions.None, "\u00E9cole", false);
            yield return ("\\W+", RegexOptions.None, ">>>\u00AA<<<", true);
            yield return ("\\W+", RegexOptions.None, ">>>\u00BA<<<", true);
            yield return ("[\\W]+", RegexOptions.None, ">>>\u00AA<<<", true);
            yield return ("[\\W]+", RegexOptions.None, ">>>\u00BA<<<", true);
            yield return ("\\w+", RegexOptions.None, ">>>\u00AA<<<", true);
            yield return ("\\w+", RegexOptions.None, ">>>\u00BA<<<", true);
            yield return ("[\\w]+", RegexOptions.None, ">>>\u00AA<<<", true);
            yield return ("[\\w]+", RegexOptions.None, ">>>\u00BA<<<", true);
            yield return ("a.b", RegexOptions.None, "a\u007fb", true);
            yield return ("a.b", RegexOptions.None, "a\u0100b", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u4000xyb", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u4000\u007fyb", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u4000\u0100yb", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u4000b", false);
            yield return ("a(.{3})b", RegexOptions.None, "ac\ncb", false);
            yield return ("a(.*?)(.)", RegexOptions.None, "a\u00C0\u0088b", true);
            yield return ("a(.*?)(.)", RegexOptions.None, "a\u0100b", true);
            yield return ("a(.*)(.)", RegexOptions.None, "a\u00C0\u0088b", true);
            yield return ("a(.*)(.)", RegexOptions.None, "a\u0100b", true);
            yield return ("a(.)(.)", RegexOptions.None, "a\u00C0\u0092bcd", true);
            yield return ("a(.)(.)", RegexOptions.None, "a\u0240bcd", true);
            yield return ("a(.?)(.)", RegexOptions.None, "a\u00C0\u0092bcd", true);
            yield return ("a(.?)(.)", RegexOptions.None, "a\u0240bcd", true);
            yield return ("a(.??)(.)", RegexOptions.None, "a\u00C0\u0092bcd", true);
            yield return ("a(.??)(.)", RegexOptions.None, "a\u0240bcd", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u1234xyb", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u1234\u4321yb", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u1234\u4321\u3412b", true);
            yield return ("a(.{3})b", RegexOptions.None, "a\u1234b", false);
            yield return ("a(.{3,})b", RegexOptions.None, "a\u1234xyb", true);
            yield return ("a(.{3,})b", RegexOptions.None, "a\u1234\u4321yb", true);
            yield return ("a(.{3,})b", RegexOptions.None, "a\u1234\u4321\u3412b", true);
            yield return ("a(.{3,})b", RegexOptions.None, "axxxxbcdefghijb", true);
            yield return ("a(.{3,})b", RegexOptions.None, "a\u1234\u4321\u3412\u3421b", true);
            yield return ("a(.{3,})b", RegexOptions.None, "a\u1234b", false);
            yield return ("a(.{3,}?)b", RegexOptions.None, "a\u1234xyb", true);
            yield return ("a(.{3,}?)b", RegexOptions.None, "a\u1234\u4321yb", true);
            yield return ("a(.{3,}?)b", RegexOptions.None, "a\u1234\u4321\u3412b", true);
            yield return ("a(.{3,}?)b", RegexOptions.None, "axxxxbcdefghijb", true);
            yield return ("a(.{3,}?)b", RegexOptions.None, "a\u1234\u4321\u3412\u3421b", true);
            yield return ("a(.{3,}?)b", RegexOptions.None, "a\u1234b", false);
            yield return ("a(.{3,5})b", RegexOptions.None, "a\u1234xyb", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "a\u1234\u4321yb", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "a\u1234\u4321\u3412b", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "axxxxbcdefghijb", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "a\u1234\u4321\u3412\u3421b", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "axbxxbcdefghijb", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "axxxxxbcdefghijb", true);
            yield return ("a(.{3,5})b", RegexOptions.None, "a\u1234b", false);
            yield return ("a(.{3,5})b", RegexOptions.None, "axxxxxxbcdefghijb", false);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "a\u1234xyb", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "a\u1234\u4321yb", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "a\u1234\u4321\u3412b", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "axxxxbcdefghijb", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "a\u1234\u4321\u3412\u3421b", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "axbxxbcdefghijb", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "axxxxxbcdefghijb", true);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "a\u1234b", false);
            yield return ("a(.{3,5}?)b", RegexOptions.None, "axxxxxxbcdefghijb", false);
            yield return ("[^a]+", RegexOptions.None, "bcd", true);
            yield return ("[^a]+", RegexOptions.None, "\u0100aY\u0256Z", true);
            yield return ("^[^a]{2}", RegexOptions.None, "\u0100bc", true);
            yield return ("^[^a]{2,}", RegexOptions.None, "\u0100bcAa", true);
            yield return ("^[^a]{2,}?", RegexOptions.None, "\u0100bca", true);
            yield return ("[^a]+", RegexOptions.IgnoreCase, "bcd", true);
            yield return ("[^a]+", RegexOptions.IgnoreCase, "\u0100aY\u0256Z", true);
            yield return ("^[^a]{2}", RegexOptions.IgnoreCase, "\u0100bc", true);
            yield return ("^[^a]{2,}", RegexOptions.IgnoreCase, "\u0100bcAa", true);
            yield return ("^[^a]{2,}?", RegexOptions.IgnoreCase, "\u0100bca", true);
            yield return ("\\D*", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", true);
            yield return ("\\D*", RegexOptions.None, "\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100\u0100", true);
            yield return ("\\D", RegexOptions.None, "1X2", true);
            yield return ("\\D", RegexOptions.None, "1\u01002", true);
            yield return (">\\S", RegexOptions.None, "> >X Y", true);
            yield return (">\\S", RegexOptions.None, "> >\u0100 Y", true);
            yield return ("\\d", RegexOptions.None, "\u01003", true);
            yield return ("\\s", RegexOptions.None, "\u0100 X", true);
            yield return ("\\D+", RegexOptions.None, "12abcd34", true);
            yield return ("\\D+", RegexOptions.None, "1234", false);
            yield return ("\\D{2,3}", RegexOptions.None, "12abcd34", true);
            yield return ("\\D{2,3}", RegexOptions.None, "12ab34", true);
            yield return ("\\D{2,3}", RegexOptions.None, "1234", false);
            yield return ("\\D{2,3}", RegexOptions.None, "12a34", false);
            yield return ("\\D{2,3}?", RegexOptions.None, "12abcd34", true);
            yield return ("\\D{2,3}?", RegexOptions.None, "12ab34", true);
            yield return ("\\D{2,3}?", RegexOptions.None, "1234", false);
            yield return ("\\D{2,3}?", RegexOptions.None, "12a34", false);
            yield return ("\\d+", RegexOptions.None, "12abcd34", true);
            yield return ("\\d{2,3}", RegexOptions.None, "12abcd34", true);
            yield return ("\\d{2,3}", RegexOptions.None, "1234abcd", true);
            yield return ("\\d{2,3}", RegexOptions.None, "1.4", false);
            yield return ("\\d{2,3}?", RegexOptions.None, "12abcd34", true);
            yield return ("\\d{2,3}?", RegexOptions.None, "1234abcd", true);
            yield return ("\\d{2,3}?", RegexOptions.None, "1.4", false);
            yield return ("\\S+", RegexOptions.None, "12abcd34", true);
            yield return ("\\S+", RegexOptions.None, "    ", false);
            yield return ("\\S{2,3}", RegexOptions.None, "12abcd34", true);
            yield return ("\\S{2,3}", RegexOptions.None, "1234abcd", true);
            yield return ("\\S{2,3}", RegexOptions.None, "     ", false);
            yield return ("\\S{2,3}?", RegexOptions.None, "12abcd34", true);
            yield return ("\\S{2,3}?", RegexOptions.None, "1234abcd", true);
            yield return ("\\S{2,3}?", RegexOptions.None, "     ", false);
            yield return (">\\s+<", RegexOptions.None, "12>      <34", true);
            yield return (">\\s{2,3}<", RegexOptions.None, "ab>  <cd", true);
            yield return (">\\s{2,3}<", RegexOptions.None, "ab>   <ce", true);
            yield return (">\\s{2,3}<", RegexOptions.None, "ab>    <cd", false);
            yield return (">\\s{2,3}?<", RegexOptions.None, "ab>  <cd", true);
            yield return (">\\s{2,3}?<", RegexOptions.None, "ab>   <ce", true);
            yield return (">\\s{2,3}?<", RegexOptions.None, "ab>    <cd", false);
            yield return ("\\w+", RegexOptions.None, "12      34", true);
            yield return ("\\w+", RegexOptions.None, "+++=*!", false);
            yield return ("\\w{2,3}", RegexOptions.None, "ab  cd", true);
            yield return ("\\w{2,3}", RegexOptions.None, "abcd ce", true);
            yield return ("\\w{2,3}", RegexOptions.None, "a.b.c", false);
            yield return ("\\w{2,3}?", RegexOptions.None, "ab  cd", true);
            yield return ("\\w{2,3}?", RegexOptions.None, "abcd ce", true);
            yield return ("\\w{2,3}?", RegexOptions.None, "a.b.c", false);
            yield return ("\\W+", RegexOptions.None, "12====34", true);
            yield return ("\\W+", RegexOptions.None, "abcd", false);
            yield return ("\\W{2,3}", RegexOptions.None, "ab====cd", true);
            yield return ("\\W{2,3}", RegexOptions.None, "ab==cd", true);
            yield return ("\\W{2,3}", RegexOptions.None, "a.b.c", false);
            yield return ("\\W{2,3}?", RegexOptions.None, "ab====cd", true);
            yield return ("\\W{2,3}?", RegexOptions.None, "ab==cd", true);
            yield return ("\\W{2,3}?", RegexOptions.None, "a.b.c", false);
            yield return ("[\\xFF]", RegexOptions.None, ">\u00FF<", true);
            yield return ("[\\xff]", RegexOptions.None, ">\u00FF<", true);
            yield return ("[^\\xFF]", RegexOptions.None, "XYZ", true);
            yield return ("[^\\xff]", RegexOptions.None, "XYZ", true);
            yield return ("[^\\xff]", RegexOptions.None, "\u0123", true);
            yield return ("^[ac]*b", RegexOptions.None, "xb", false);
            yield return ("^[^x]*b", RegexOptions.IgnoreCase, "xb", false);
            yield return ("^[^x]*b", RegexOptions.None, "xb", false);
            yield return ("^\\d*b", RegexOptions.None, "xb", false);
            yield return ("(|a)", RegexOptions.None, "catac", true);
            yield return ("(|a)", RegexOptions.None, "a\u0256a", true);
            yield return ("^\u00E1\u0088\u00B4", RegexOptions.None, "\u00E1\u0088\u00B4", true);
            yield return ("(?s)(.{1,5})", RegexOptions.None, "abcdefg", true);
            yield return ("(?s)(.{1,5})", RegexOptions.None, "ab", true);
            yield return ("\\S\\S", RegexOptions.None, "A\u00A3BC", true);
            yield return ("\\S{2}", RegexOptions.None, "A\u00A3BC", true);
            yield return ("\\W\\W", RegexOptions.None, "+\u00A3==", true);
            yield return ("\\W{2}", RegexOptions.None, "+\u00A3==", true);
            yield return ("\\S", RegexOptions.None, "\u0442\u0435\u0441\u0442", true);
            yield return ("[\\S]", RegexOptions.None, "\u0442\u0435\u0441\u0442", true);
            yield return ("\\D", RegexOptions.None, "\u0442\u0435\u0441\u0442", true);
            yield return ("[\\D]", RegexOptions.None, "\u0442\u0435\u0441\u0442", true);
            yield return ("\\W", RegexOptions.None, "\u2442\u2435\u2441\u2442", true);
            yield return ("[\\W]", RegexOptions.None, "\u2442\u2435\u2441\u2442", true);
            yield return ("[\\S\\s]*", RegexOptions.None, "abc\n\r\u0442\u0435\u0441\u0442xyz", true);
            yield return (".[^\\S].", RegexOptions.None, "abc def\u0442\u0443xyz\npqr", true);
            yield return (".[^\\S\\n].", RegexOptions.None, "abc def\u0442\u0443xyz\npqr", true);
            yield return ("^[^d]*?$", RegexOptions.None, "abc", true);
            yield return ("^[^d]*?$", RegexOptions.IgnoreCase, "abc", true);
            yield return ("A*", RegexOptions.None, "AAB\u0123BAA", true);
            yield return ("^\\p{Cc}", RegexOptions.None, "\u0017", true);
            yield return ("^\\p{Cc}", RegexOptions.None, "\u009f", true);
            yield return ("^\\p{Cc}", RegexOptions.None, "\u0600", false);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u0601", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u009f", false);
            yield return ("^\\p{Co}", RegexOptions.None, "\uF8FF", true);
            yield return ("^\\p{Co}", RegexOptions.None, "\u009f", false);
            yield return ("^\\p{Ll}", RegexOptions.None, "a", true);
            yield return ("^\\p{Ll}", RegexOptions.None, "Z", false);
            yield return ("^\\p{Ll}", RegexOptions.None, "\uE000", false);
            yield return ("^\\p{Lm}", RegexOptions.None, "\u02B0", true);
            yield return ("^\\p{Lm}", RegexOptions.None, "a", false);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u01BB", true);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u3400", true);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u3401", true);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u4D00", true);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u4DB4", true);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u4DB5", true);
            if (PlatformDetection.IsNotNetFramework)
            {
                yield return ("^\\p{Lo}", RegexOptions.None, "\u4DB6", true); // This was fixed in .NET 7
            }
            yield return ("^\\p{Lo}", RegexOptions.None, "a", false);
            yield return ("^\\p{Lo}", RegexOptions.None, "\u02B0", false);
            yield return ("^\\p{Lt}", RegexOptions.None, "\u01C5", true);
            yield return ("^\\p{Lt}", RegexOptions.None, "a", false);
            yield return ("^\\p{Lt}", RegexOptions.None, "\u02B0", false);
            yield return ("^\\p{Lu}", RegexOptions.None, "A", true);
            yield return ("^\\p{Lu}", RegexOptions.None, "\u02B0", false);
            yield return ("^\\p{Mc}", RegexOptions.None, "\u0903", true);
            yield return ("^\\p{Mc}", RegexOptions.None, "X", false);
            yield return ("^\\p{Mc}", RegexOptions.None, "\u0300", false);
            yield return ("^\\p{Me}", RegexOptions.None, "\u0488", true);
            yield return ("^\\p{Me}", RegexOptions.None, "X", false);
            yield return ("^\\p{Me}", RegexOptions.None, "\u0903", false);
            yield return ("^\\p{Me}", RegexOptions.None, "\u0300", false);
            yield return ("^\\p{Mn}", RegexOptions.None, "\u0300", true);
            yield return ("^\\p{Mn}", RegexOptions.None, "X", false);
            yield return ("^\\p{Mn}", RegexOptions.None, "\u0903", false);
            yield return ("^\\p{Nd}+", RegexOptions.None, "0123456789\u0660\u0661\u0662\u0663\u0664\u0665\u0666\u0667\u0668\u0669\u066A", true);
            yield return ("^\\p{Nd}+", RegexOptions.None, "\u06F0\u06F1\u06F2\u06F3\u06F4\u06F5\u06F6\u06F7\u06F8\u06F9\u06FA", true);
            yield return ("^\\p{Nd}+", RegexOptions.None, "\u0966\u0967\u0968\u0969\u096A\u096B\u096C\u096D\u096E\u096F\u0970", true);
            yield return ("^\\p{Nd}+", RegexOptions.None, "X", false);
            yield return ("^\\p{Nl}", RegexOptions.None, "\u16EE", true);
            yield return ("^\\p{Nl}", RegexOptions.None, "X", false);
            yield return ("^\\p{Nl}", RegexOptions.None, "\u0966", false);
            yield return ("^\\p{No}", RegexOptions.None, "\u00B2", true);
            yield return ("^\\p{No}", RegexOptions.None, "\u00B3", true);
            yield return ("^\\p{No}", RegexOptions.None, "X", false);
            yield return ("^\\p{No}", RegexOptions.None, "\u16EE", false);
            yield return ("^\\p{Pc}", RegexOptions.None, "_", true);
            yield return ("^\\p{Pc}", RegexOptions.None, "\u203F", true);
            yield return ("^\\p{Pc}", RegexOptions.None, "X", false);
            yield return ("^\\p{Pc}", RegexOptions.None, "-", false);
            yield return ("^\\p{Pc}", RegexOptions.None, "\u058A", false);
            yield return ("^\\p{Pd}", RegexOptions.None, "-", true);
            yield return ("^\\p{Pd}", RegexOptions.None, "\u058A", true);
            yield return ("^\\p{Pd}", RegexOptions.None, "X", false);
            yield return ("^\\p{Pd}", RegexOptions.None, "\u203F", false);
            yield return ("^\\p{Pe}", RegexOptions.None, ")", true);
            yield return ("^\\p{Pe}", RegexOptions.None, "]", true);
            yield return ("^\\p{Pe}", RegexOptions.None, "}", true);
            yield return ("^\\p{Pe}", RegexOptions.None, "\u0F3B", true);
            yield return ("^\\p{Pe}", RegexOptions.None, "X", false);
            yield return ("^\\p{Pe}", RegexOptions.None, "\u203F", false);
            yield return ("^\\p{Pe}", RegexOptions.None, "(", false);
            yield return ("^\\p{Pe}", RegexOptions.None, "[", false);
            yield return ("^\\p{Pe}", RegexOptions.None, "{", false);
            yield return ("^\\p{Pe}", RegexOptions.None, "\u0F3C", false);
            yield return ("^\\p{Pf}", RegexOptions.None, "\u00BB", true);
            yield return ("^\\p{Pf}", RegexOptions.None, "\u2019", true);
            yield return ("^\\p{Pf}", RegexOptions.None, "X", false);
            yield return ("^\\p{Pf}", RegexOptions.None, "\u203F", false);
            yield return ("^\\p{Pi}", RegexOptions.None, "\u00AB", true);
            yield return ("^\\p{Pi}", RegexOptions.None, "\u2018", true);
            yield return ("^\\p{Pi}", RegexOptions.None, "X", false);
            yield return ("^\\p{Pi}", RegexOptions.None, "\u203F", false);
            yield return ("^\\p{Po}", RegexOptions.None, "!", true);
            yield return ("^\\p{Po}", RegexOptions.None, "\u037E", true);
            yield return ("^\\p{Po}", RegexOptions.None, "X", false);
            yield return ("^\\p{Po}", RegexOptions.None, "\u203F", false);
            yield return ("^\\p{Ps}", RegexOptions.None, "(", true);
            yield return ("^\\p{Ps}", RegexOptions.None, "[", true);
            yield return ("^\\p{Ps}", RegexOptions.None, "{", true);
            yield return ("^\\p{Ps}", RegexOptions.None, "\u0F3C", true);
            yield return ("^\\p{Ps}", RegexOptions.None, "X", false);
            yield return ("^\\p{Ps}", RegexOptions.None, ")", false);
            yield return ("^\\p{Ps}", RegexOptions.None, "]", false);
            yield return ("^\\p{Ps}", RegexOptions.None, "}", false);
            yield return ("^\\p{Ps}", RegexOptions.None, "\u0F3B", false);
            yield return ("^\\p{Sk}", RegexOptions.None, "\u02C2", true);
            yield return ("^\\p{Sk}", RegexOptions.None, "X", false);
            yield return ("^\\p{Sk}", RegexOptions.None, "\u09F2", false);
            yield return ("^\\p{Sm}+", RegexOptions.None, "+<|~\u00AC\u2044", true);
            yield return ("^\\p{Sm}+", RegexOptions.None, "X", false);
            yield return ("^\\p{Sm}+", RegexOptions.None, "\u09F2", false);
            yield return ("^\\p{So}", RegexOptions.None, "\u00A6", true);
            yield return ("^\\p{So}", RegexOptions.None, "\u0482", true);
            yield return ("^\\p{So}", RegexOptions.None, "X", false);
            yield return ("^\\p{So}", RegexOptions.None, "\u09F2", false);
            yield return ("^\\p{Zl}", RegexOptions.None, "\u2028", true);
            yield return ("^\\p{Zl}", RegexOptions.None, "X", false);
            yield return ("^\\p{Zl}", RegexOptions.None, "\u2029", false);
            yield return ("^\\p{Zp}", RegexOptions.None, "\u2029", true);
            yield return ("^\\p{Zp}", RegexOptions.None, "X", false);
            yield return ("^\\p{Zp}", RegexOptions.None, "\u2028", false);
            yield return ("\\p{Nd}+(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}+?(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}{2,}(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}{2,}?(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}*(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}*?(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}{2}(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}{2,3}(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}{2,3}?(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}?(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("\\p{Nd}??(..)", RegexOptions.None, "\u0660\u0661\u0662ABC", true);
            yield return ("[\\P{Nd}]+", RegexOptions.None, "abcd", true);
            yield return ("[\\P{Nd}]+", RegexOptions.None, "1234", false);
            yield return ("\\D+", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", true);
            yield return ("\\D+", RegexOptions.None, "11111111111111111111111111111111111111111111111111111111111111111111111", false);
            yield return ("\\P{Nd}+", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", true);
            yield return ("\\P{Nd}+", RegexOptions.None, "11111111111111111111111111111111111111111111111111111111111111111111111", false);
            yield return ("[\\D]+", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", true);
            yield return ("[\\D]+", RegexOptions.None, "11111111111111111111111111111111111111111111111111111111111111111111111", false);
            yield return ("[\\P{Nd}]+", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", true);
            yield return ("[\\P{Nd}]+", RegexOptions.None, "11111111111111111111111111111111111111111111111111111111111111111111111", false);
            yield return ("[\\D\\P{Nd}]+", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", true);
            yield return ("[\\D\\P{Nd}]+", RegexOptions.None, "11111111111111111111111111111111111111111111111111111111111111111111111", false);
            yield return ("\\p{Lu}", RegexOptions.None, "A", true);
            yield return ("\\p{Lu}", RegexOptions.None, "aZ", true);
            yield return ("\\p{Lu}", RegexOptions.None, "abc", false);
            yield return ("\\p{Ll}", RegexOptions.None, "a", true);
            yield return ("\\p{Ll}", RegexOptions.None, "Az", true);
            yield return ("\\p{Ll}", RegexOptions.None, "ABC", false);
            yield return ("^(\\p{Z}[^\\p{C}\\p{Z}]+)*$", RegexOptions.None, "\u00A0!", true);
            yield return ("\\p{L}{4}", RegexOptions.None, "123abcdefg", true);
            yield return ("\\p{L}{4}", RegexOptions.None, "123abc\u00C4\u00C5zz", true);
            yield return ("\\P{L}?\\d", RegexOptions.None, "\u008aBCD", false);
            yield return ("(?:\\p{Lu}|\\x20)+", RegexOptions.None, "A P\u00C2T\u00C9 TODAY", true);
            yield return ("[\\p{Lu}\\x20]+", RegexOptions.None, "A P\u00C2T\u00C9 TODAY", true);
            yield return ("^\\d+", RegexOptions.None, "1\u0660\u0BEF\u16EE", true);
            yield return ("^[\\d]+", RegexOptions.None, "1\u0660\u0BEF\u16EE", true);
            yield return ("^>\\s+", RegexOptions.None, "> \u00A0\u1680\u2028\u2029\u202F\t\v", true);
            yield return ("\u00E2\u00B1\u00A5", RegexOptions.IgnoreCase, "\u00E2\u00B1\u00A5", true);
            yield return ("[\u00E2\u00B1\u00A5]", RegexOptions.IgnoreCase, "\u00E2\u00B1\u00A5", true);
            yield return ("\u00C8\u00BA", RegexOptions.IgnoreCase, "\u00C8\u00BA", true);
            yield return ("^A\\s+Z", RegexOptions.None, "A\u2005Z", true);
            yield return ("^A\\s+Z", RegexOptions.None, "A\u0085\u2005Z", true);
            yield return ("^A[\\s]+Z", RegexOptions.None, "A\u2005Z", true);
            yield return ("^A[\\s]+Z", RegexOptions.None, "A\u0085\u2005Z", true);
            yield return ("[RST]+", RegexOptions.IgnoreCase, "Ss\u017F", true);
            yield return ("[R-T]+", RegexOptions.IgnoreCase, "Ss\u017F", true);
            yield return ("[q-u]+", RegexOptions.IgnoreCase, "Ss\u017F", true);
            yield return ("^s?c", RegexOptions.IgnoreCase | RegexOptions.Multiline, "scat", true);
            yield return ("[A-`]", RegexOptions.IgnoreCase, "abcdefghijklmno", true);
            // yield return ("A\u000C\u000A\u000D\u0009\u000B\u0085B", RegexOptions.IgnorePatternWhitespace, "AB", true); https://github.com/dotnet/runtime/issues/73206
            yield return ("A\u00E2\u0080\u0082B", RegexOptions.IgnorePatternWhitespace, "AB", false);
            yield return ("\\W", RegexOptions.None, "A.B", true);
            yield return ("\\w", RegexOptions.None, "\u0100X", true);
            yield return ("^abc.", RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace, "abc1 \nabc2 \vabc3xx \fabc4 \rabc5xx \r\nabc6 \u0085abc7 \u2028abc8 \u2029abc9 JUNK", true);
            yield return ("abc.$", RegexOptions.Multiline | RegexOptions.IgnorePatternWhitespace, "abc1\n abc2\v abc3\f abc4\r abc5\r\n abc6\u0085 abc7\u2028 abc8\u2029 abc9", true);
            yield return (".*$", RegexOptions.None, "\u1EC5", true);
            yield return (".*a.*=.b.*", RegexOptions.None, "QQQ\u2029ABCaXYZ=!bPQR", true);
            yield return (".*a.*=.b.*", RegexOptions.None, "a\u2029b", false);
            yield return (".*a.*=.b.*", RegexOptions.None, "a\u00E2\u0080\u00A9b", false);
            yield return ("X", RegexOptions.None, "A\u1EC5ABCXYZ", true);
            yield return ("X\\D{2,4}b", RegexOptions.None, "X\u0123=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "X\u0123\u0123=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "X\u0123\u0123\u0123=ps", false);
            yield return ("X\\D{2,4}b", RegexOptions.None, "X\u0123\u0123\u0123\u0123=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "X\u0123=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "X\u0123\u0123=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "X\u0123\u0123\u0123=ps", false);
            yield return ("X\\D{2,4}?b", RegexOptions.None, "X\u0123\u0123\u0123\u0123=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "X\u0123=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "X\u0123\u0123=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "X\u0123\u0123\u0123=ps", false);
            yield return ("X[^a]{2,4}b", RegexOptions.None, "X\u0123\u0123\u0123\u0123=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "X\u0123=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "X\u0123\u0123=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "X\u0123\u0123\u0123=ps", false);
            yield return ("X[^a]{2,4}?b", RegexOptions.None, "X\u0123\u0123\u0123\u0123=ps", false);
            yield return ("X\\W{3}X", RegexOptions.None, "X=ps", false);
            yield return ("\\sxxx\\s", RegexOptions.Singleline, "AB\u00A0xxx\u0085XYZ", true);
            yield return ("\\S \\S", RegexOptions.Singleline, "\u00A2 \u0084", true);
            yield return ("\\v+", RegexOptions.None, "\t\u000e\u0084\u0086\u0085\n\v\f\r", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u180E", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u061C", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u2066", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u2067", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u2068", true);
            yield return ("^\\p{Cf}", RegexOptions.None, "\u2069", true);
            yield return ("^\\p{Cs}", RegexOptions.None, "\udfff=no_utf_check", true);
            yield return ("^\\p{Cs}", RegexOptions.None, "\u009f", false);
            yield return ("^\\p{Mn}", RegexOptions.None, "\u1A1B", true);
            yield return ("^\\p{Pe}", RegexOptions.None, "\u2309", true);
            yield return ("^\\p{Pe}", RegexOptions.None, "\u230B", true);
            yield return ("^\\p{Ps}", RegexOptions.None, "\u2308", true);
            yield return ("^\\p{Ps}", RegexOptions.None, "\u230A", true);
            yield return ("^\\p{Sc}+", RegexOptions.None, "$\u00A2\u00A3\u00A4\u00A5\u00A6", true);
            yield return ("^\\p{Sc}+", RegexOptions.None, "\u09F2", true);
            yield return ("^\\p{Sc}+", RegexOptions.None, "X", false);
            yield return ("^\\p{Sc}+", RegexOptions.None, "\u02C2", false);
            yield return ("^\\p{Zs}", RegexOptions.None, " ", true);
            yield return ("^\\p{Zs}", RegexOptions.None, "\u00A0", true);
            yield return ("^\\p{Zs}", RegexOptions.None, "\u1680", true);
            yield return ("^\\p{Zs}", RegexOptions.None, "\u2000", true);
            yield return ("^\\p{Zs}", RegexOptions.None, "\u2001", true);
            yield return ("^\\p{Zs}", RegexOptions.None, "\u2028", false);
            yield return ("^\\p{Zs}", RegexOptions.None, "\u200D", false);
            yield return ("\\P{Lu}", RegexOptions.IgnoreCase, "1234", true);
            yield return ("\\P{Lu}", RegexOptions.IgnoreCase, "ABC", false);
            yield return ("\\p{Ll}", RegexOptions.IgnoreCase, "a", true);
            yield return ("\\p{Ll}", RegexOptions.IgnoreCase, "Az", true);
            yield return ("\\p{Lu}", RegexOptions.IgnoreCase, "A", true);
            yield return ("\\p{Lu}", RegexOptions.IgnoreCase, "a\u10A0B", true);
            yield return ("\\p{Lu}", RegexOptions.IgnoreCase, "aZ", true);
            yield return ("[\\D]", RegexOptions.None, "1\u03C82", true);
            yield return ("[\\d]", RegexOptions.None, ">\u06F4<", true);
            yield return ("[\\S]", RegexOptions.None, "\u1680\u06F4\u1680", true);
            yield return ("[\\s]", RegexOptions.None, ">\u1680<", true);
            yield return ("[\\w]", RegexOptions.None, ">\u1723<", true);
            yield return ("\\D", RegexOptions.None, "1\u03C82", true);
            yield return ("\\d", RegexOptions.None, ">\u06F4<", true);
            yield return ("\\S", RegexOptions.None, "\u1680\u06F4\u1680", true);
            yield return ("\\s", RegexOptions.None, ">\u1680>", true);
            yield return ("\\w", RegexOptions.None, ">\u1723<", true);
            yield return ("\\b...\\B", RegexOptions.None, "abc_", true);
            yield return ("\\b...\\B", RegexOptions.None, "\u037Eabc\u0376", true);
            yield return ("\\b...\\B", RegexOptions.None, "\u037E\u0376\u0371\u0393\u0394", true);
            yield return ("\\b...\\B", RegexOptions.None, "!\u00C0++\u00C1\u00C2", true);
            yield return ("\\b...\\B", RegexOptions.None, "!\u00C0+++++", true);
            yield return ("[^\\d]+", RegexOptions.None, "abc123", true);
            yield return ("[^\\d]+", RegexOptions.None, "abc\u0123", true);
            yield return ("[^\\d]+", RegexOptions.None, "\u0660abc", true);
            yield return ("ist", RegexOptions.IgnoreCase, "ikt", false);
            yield return ("is+t", RegexOptions.IgnoreCase, "ikt", false);
            yield return ("is+?t", RegexOptions.IgnoreCase, "ikt", false);
            yield return ("is?t", RegexOptions.IgnoreCase, "ikt", false);
            yield return ("is{2}t", RegexOptions.IgnoreCase, "iskt", false);
            yield return ("", RegexOptions.None, "=zero_terminate", true);
            yield return ("^[ac]{3,}?x", RegexOptions.IgnoreCase, "aaa\u0100", false);
            yield return ("^X\\p{L}+?", RegexOptions.None, "X", false);
            yield return ("^X\\p{Lu}+?", RegexOptions.None, "X", false);
            yield return ("^X\\s+?", RegexOptions.None, "X", false);
            yield return ("^X\\s+?", RegexOptions.None, "XX", false);
            yield return ("^X\\S+?", RegexOptions.None, "XX", true);
            yield return ("^X\\w+?", RegexOptions.None, "X", false);
            yield return ("^X.+?Z", RegexOptions.Singleline, "X", false);
            yield return ("^X\\p{L}{1,3}?Z", RegexOptions.Singleline, "XY", false);
            yield return ("^X\\p{L}{1,3}?Z", RegexOptions.Singleline, "XY!", false);
            yield return ("^X\\p{Lu}{1,3}?Z", RegexOptions.Singleline, "XY", false);
            yield return ("^X\\p{Lu}{1,3}?Z", RegexOptions.Singleline, "XY!", false);
            yield return ("^X.+?Z", RegexOptions.Singleline, "XYY", false);
            yield return ("^X\\p{L}{1,3}Z", RegexOptions.None, "XY", false);
            // yield return ("A      \n\v\f\n\u0085B", RegexOptions.IgnorePatternWhitespace, "AB", true); https://github.com/dotnet/runtime/issues/73206
            yield return ("ab*c", RegexOptions.None, "abbbbc", true);
            yield return ("ab*c", RegexOptions.None, "ac", true);
            yield return ("ab+c", RegexOptions.None, "abc", true);
            yield return ("ab+c", RegexOptions.None, "abbbbbbc", true);
            yield return ("ab+c", RegexOptions.None, "ac", false);
            yield return ("ab+c", RegexOptions.None, "ab", false);
            yield return ("a*", RegexOptions.None, "a", true);
            yield return ("a*", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaa=ovector=10", true);
            yield return ("(a|abcd|african)", RegexOptions.None, "a", true);
            yield return ("(a|abcd|african)", RegexOptions.None, "abcd", true);
            yield return ("(a|abcd|african)", RegexOptions.None, "african", true);
            yield return ("^abc", RegexOptions.None, "abcdef", true);
            yield return ("^abc", RegexOptions.None, "xyzabc", false);
            yield return ("^abc", RegexOptions.None, "xyz\nabc", false);
            yield return ("^abc", RegexOptions.Multiline, "abcdef", true);
            yield return ("^abc", RegexOptions.Multiline, "xyz\nabc", true);
            yield return ("^abc", RegexOptions.Multiline, "xyzabc", false);
            yield return ("\\Aabc", RegexOptions.None, "abcdef", true);
            yield return ("\\Aabc", RegexOptions.None, "xyzabc", false);
            yield return ("\\Aabc", RegexOptions.None, "xyz\nabc", false);
            yield return ("\\Aabc", RegexOptions.Multiline, "abcdef", true);
            yield return ("\\Aabc", RegexOptions.Multiline, "xyzabc", false);
            yield return ("\\Aabc", RegexOptions.Multiline, "xyz\nabc", false);
            yield return ("x\\dy\\Dz", RegexOptions.None, "x9yzz", true);
            yield return ("x\\dy\\Dz", RegexOptions.None, "x0y+z", true);
            yield return ("x\\dy\\Dz", RegexOptions.None, "xyz", false);
            yield return ("x\\dy\\Dz", RegexOptions.None, "xxy0z", false);
            yield return ("x\\sy\\Sz", RegexOptions.None, "x yzz", true);
            yield return ("x\\sy\\Sz", RegexOptions.None, "x y+z", true);
            yield return ("x\\sy\\Sz", RegexOptions.None, "xyz", false);
            yield return ("x\\sy\\Sz", RegexOptions.None, "xxyyz", false);
            yield return ("x\\wy\\Wz", RegexOptions.None, "xxy+z", true);
            yield return ("x\\wy\\Wz", RegexOptions.None, "xxy0z", false);
            yield return ("x\\wy\\Wz", RegexOptions.None, "x+y+z", false);
            yield return ("x.y", RegexOptions.None, "x+y", true);
            yield return ("x.y", RegexOptions.None, "x-y", true);
            yield return ("x.y", RegexOptions.None, "x\ny", false);
            yield return ("x.y", RegexOptions.Singleline, "x+y", true);
            yield return ("x.y", RegexOptions.Singleline, "x-y", true);
            yield return ("x.y", RegexOptions.Singleline, "x\ny", true);
            yield return ("(a.b(?s)c.d|x.y)p.q", RegexOptions.None, "a+bc+dp+q", true);
            yield return ("(a.b(?s)c.d|x.y)p.q", RegexOptions.None, "a+bc\ndp+q", true);
            yield return ("(a.b(?s)c.d|x.y)p.q", RegexOptions.None, "x\nyp+q", true);
            yield return ("(a.b(?s)c.d|x.y)p.q", RegexOptions.None, "a\nbc\ndp+q", false);
            yield return ("(a.b(?s)c.d|x.y)p.q", RegexOptions.None, "a+bc\ndp\nq", false);
            yield return ("(a.b(?s)c.d|x.y)p.q", RegexOptions.None, "x\nyp\nq", false);
            yield return ("a\\d\\z", RegexOptions.None, "ba0", true);
            yield return ("a\\d\\z", RegexOptions.None, "ba0\n", false);
            yield return ("a\\d\\z", RegexOptions.None, "ba0\ncd", false);
            yield return ("a\\d\\z", RegexOptions.Multiline, "ba0", true);
            yield return ("a\\d\\z", RegexOptions.Multiline, "ba0\n", false);
            yield return ("a\\d\\z", RegexOptions.Multiline, "ba0\ncd", false);
            yield return ("a\\d\\Z", RegexOptions.None, "ba0", true);
            yield return ("a\\d\\Z", RegexOptions.None, "ba0\n", true);
            yield return ("a\\d\\Z", RegexOptions.None, "ba0\ncd", false);
            yield return ("a\\d\\Z", RegexOptions.Multiline, "ba0", true);
            yield return ("a\\d\\Z", RegexOptions.Multiline, "ba0\n", true);
            yield return ("a\\d\\Z", RegexOptions.Multiline, "ba0\ncd", false);
            yield return ("a\\d$", RegexOptions.None, "ba0", true);
            yield return ("a\\d$", RegexOptions.None, "ba0\n", true);
            yield return ("a\\d$", RegexOptions.None, "ba0\ncd", false);
            yield return ("a\\d$", RegexOptions.Multiline, "ba0", true);
            yield return ("a\\d$", RegexOptions.Multiline, "ba0\n", true);
            yield return ("a\\d$", RegexOptions.Multiline, "ba0\ncd", true);
            yield return ("abc", RegexOptions.IgnoreCase, "abc", true);
            yield return ("abc", RegexOptions.IgnoreCase, "aBc", true);
            yield return ("[^a]", RegexOptions.None, "abcd", true);
            yield return ("ab?\\w", RegexOptions.None, "abz", true);
            yield return ("ab?\\w", RegexOptions.None, "abbz", true);
            yield return ("ab?\\w", RegexOptions.None, "azz", true);
            yield return ("x{0,3}yz", RegexOptions.None, "ayzq", true);
            yield return ("x{0,3}yz", RegexOptions.None, "axyzq", true);
            yield return ("x{0,3}yz", RegexOptions.None, "axxyz", true);
            yield return ("x{0,3}yz", RegexOptions.None, "axxxyzq", true);
            yield return ("x{0,3}yz", RegexOptions.None, "axxxxyzq", true);
            yield return ("x{0,3}yz", RegexOptions.None, "ax", false);
            yield return ("x{0,3}yz", RegexOptions.None, "axx", false);
            yield return ("x{3}yz", RegexOptions.None, "axxxyzq", true);
            yield return ("x{3}yz", RegexOptions.None, "axxxxyzq", true);
            yield return ("x{3}yz", RegexOptions.None, "ax", false);
            yield return ("x{3}yz", RegexOptions.None, "axx", false);
            yield return ("x{3}yz", RegexOptions.None, "ayzq", false);
            yield return ("x{3}yz", RegexOptions.None, "axyzq", false);
            yield return ("x{3}yz", RegexOptions.None, "axxyz", false);
            yield return ("x{2,3}yz", RegexOptions.None, "axxyz", true);
            yield return ("x{2,3}yz", RegexOptions.None, "axxxyzq", true);
            yield return ("x{2,3}yz", RegexOptions.None, "axxxxyzq", true);
            yield return ("x{2,3}yz", RegexOptions.None, "ax", false);
            yield return ("x{2,3}yz", RegexOptions.None, "axx", false);
            yield return ("x{2,3}yz", RegexOptions.None, "ayzq", false);
            yield return ("x{2,3}yz", RegexOptions.None, "axyzq", false);
            yield return ("[^a]+", RegexOptions.None, "bac", true);
            yield return ("[^a]+", RegexOptions.None, "bcdefax", true);
            yield return ("[^a]+", RegexOptions.None, "aaaaa", false);
            yield return ("[^a]*", RegexOptions.None, "bac", true);
            yield return ("[^a]*", RegexOptions.None, "bcdefax", true);
            yield return ("[^a]*", RegexOptions.None, "aaaaa", true);
            yield return ("[^a]{3,5}", RegexOptions.None, "xyz", true);
            yield return ("[^a]{3,5}", RegexOptions.None, "awxyza", true);
            yield return ("[^a]{3,5}", RegexOptions.None, "abcdefa", true);
            yield return ("[^a]{3,5}", RegexOptions.None, "abcdefghijk", true);
            yield return ("[^a]{3,5}", RegexOptions.None, "axya", false);
            yield return ("[^a]{3,5}", RegexOptions.None, "axa", false);
            yield return ("[^a]{3,5}", RegexOptions.None, "aaaaa", false);
            yield return ("\\d*", RegexOptions.None, "1234b567", true);
            yield return ("\\d*", RegexOptions.None, "xyz", true);
            yield return ("\\D*", RegexOptions.None, "a1234b567", true);
            yield return ("\\D*", RegexOptions.None, "xyz", true);
            yield return ("\\d+", RegexOptions.None, "ab1234c56", true);
            yield return ("\\d+", RegexOptions.None, "xyz", false);
            yield return ("\\D+", RegexOptions.None, "ab123c56", true);
            yield return ("\\D+", RegexOptions.None, "789", false);
            yield return ("\\d?A", RegexOptions.None, "045ABC", true);
            yield return ("\\d?A", RegexOptions.None, "ABC", true);
            yield return ("\\d?A", RegexOptions.None, "XYZ", false);
            yield return ("\\D?A", RegexOptions.None, "ABC", true);
            yield return ("\\D?A", RegexOptions.None, "BAC", true);
            yield return ("\\D?A", RegexOptions.None, "9ABC", true);
            yield return ("a+", RegexOptions.None, "aaaa", true);
            yield return ("^.*xyz", RegexOptions.None, "xyz", true);
            yield return ("^.*xyz", RegexOptions.None, "ggggggggxyz", true);
            yield return ("^.+xyz", RegexOptions.None, "abcdxyz", true);
            yield return ("^.+xyz", RegexOptions.None, "axyz", true);
            yield return ("^.+xyz", RegexOptions.None, "xyz", false);
            yield return ("^.?xyz", RegexOptions.None, "xyz", true);
            yield return ("^.?xyz", RegexOptions.None, "cxyz", true);
            yield return ("^\\d{2,3}X", RegexOptions.None, "12X", true);
            yield return ("^\\d{2,3}X", RegexOptions.None, "123X", true);
            yield return ("^\\d{2,3}X", RegexOptions.None, "X", false);
            yield return ("^\\d{2,3}X", RegexOptions.None, "1X", false);
            yield return ("^\\d{2,3}X", RegexOptions.None, "1234X", false);
            yield return ("^[abcd]\\d", RegexOptions.None, "a45", true);
            yield return ("^[abcd]\\d", RegexOptions.None, "b93", true);
            yield return ("^[abcd]\\d", RegexOptions.None, "c99z", true);
            yield return ("^[abcd]\\d", RegexOptions.None, "d04", true);
            yield return ("^[abcd]\\d", RegexOptions.None, "e45", false);
            yield return ("^[abcd]\\d", RegexOptions.None, "abcd", false);
            yield return ("^[abcd]\\d", RegexOptions.None, "abcd1234", false);
            yield return ("^[abcd]\\d", RegexOptions.None, "1234", false);
            yield return ("^[abcd]*\\d", RegexOptions.None, "a45", true);
            yield return ("^[abcd]*\\d", RegexOptions.None, "b93", true);
            yield return ("^[abcd]*\\d", RegexOptions.None, "c99z", true);
            yield return ("^[abcd]*\\d", RegexOptions.None, "d04", true);
            yield return ("^[abcd]*\\d", RegexOptions.None, "abcd1234", true);
            yield return ("^[abcd]*\\d", RegexOptions.None, "1234", true);
            yield return ("^[abcd]*\\d", RegexOptions.None, "e45", false);
            yield return ("^[abcd]*\\d", RegexOptions.None, "abcd", false);
            yield return ("^[abcd]+\\d", RegexOptions.None, "a45", true);
            yield return ("^[abcd]+\\d", RegexOptions.None, "b93", true);
            yield return ("^[abcd]+\\d", RegexOptions.None, "c99z", true);
            yield return ("^[abcd]+\\d", RegexOptions.None, "d04", true);
            yield return ("^[abcd]+\\d", RegexOptions.None, "abcd1234", true);
            yield return ("^[abcd]+\\d", RegexOptions.None, "1234", false);
            yield return ("^[abcd]+\\d", RegexOptions.None, "e45", false);
            yield return ("^[abcd]+\\d", RegexOptions.None, "abcd", false);
            yield return ("^a+X", RegexOptions.None, "aX", true);
            yield return ("^a+X", RegexOptions.None, "aaX", true);
            yield return ("^[abcd]?\\d", RegexOptions.None, "a45", true);
            yield return ("^[abcd]?\\d", RegexOptions.None, "b93", true);
            yield return ("^[abcd]?\\d", RegexOptions.None, "c99z", true);
            yield return ("^[abcd]?\\d", RegexOptions.None, "d04", true);
            yield return ("^[abcd]?\\d", RegexOptions.None, "1234", true);
            yield return ("^[abcd]?\\d", RegexOptions.None, "abcd1234", false);
            yield return ("^[abcd]?\\d", RegexOptions.None, "e45", false);
            yield return ("^[abcd]{2,3}\\d", RegexOptions.None, "ab45", true);
            yield return ("^[abcd]{2,3}\\d", RegexOptions.None, "bcd93", true);
            yield return ("^[abcd]{2,3}\\d", RegexOptions.None, "1234", false);
            yield return ("^[abcd]{2,3}\\d", RegexOptions.None, "a36", false);
            yield return ("^[abcd]{2,3}\\d", RegexOptions.None, "abcd1234", false);
            yield return ("^[abcd]{2,3}\\d", RegexOptions.None, "ee45", false);
            yield return ("^(abc)*\\d", RegexOptions.None, "abc45", true);
            yield return ("^(abc)*\\d", RegexOptions.None, "abcabcabc45", true);
            yield return ("^(abc)*\\d", RegexOptions.None, "42xyz", true);
            yield return ("^(abc)+\\d", RegexOptions.None, "abc45", true);
            yield return ("^(abc)+\\d", RegexOptions.None, "abcabcabc45", true);
            yield return ("^(abc)+\\d", RegexOptions.None, "42xyz", false);
            yield return ("^(abc)?\\d", RegexOptions.None, "abc45", true);
            yield return ("^(abc)?\\d", RegexOptions.None, "42xyz", true);
            yield return ("^(abc)?\\d", RegexOptions.None, "abcabcabc45", false);
            yield return ("^(abc){2,3}\\d", RegexOptions.None, "abcabc45", true);
            yield return ("^(abc){2,3}\\d", RegexOptions.None, "abcabcabc45", true);
            yield return ("^(abc){2,3}\\d", RegexOptions.None, "abcabcabcabc45", false);
            yield return ("^(abc){2,3}\\d", RegexOptions.None, "abc45", false);
            yield return ("^(abc){2,3}\\d", RegexOptions.None, "42xyz", false);
            yield return ("^(a*\\w|ab)=(a*\\w|ab)", RegexOptions.None, "ab=ab", true);
            yield return ("^(a*|xyz)", RegexOptions.None, "bcd", true);
            yield return ("^(a*|xyz)", RegexOptions.None, "aaabcd", true);
            yield return ("^(a*|xyz)", RegexOptions.None, "xyz", true);
            yield return ("^(a*|xyz)", RegexOptions.None, "xyz=notempty", true);
            yield return ("xyz$", RegexOptions.None, "xyz", true);
            yield return ("xyz$", RegexOptions.None, "xyz\n", true);
            yield return ("xyz$", RegexOptions.None, "xyz=noteol", false);
            yield return ("xyz$", RegexOptions.None, "xyz\n=noteol", false);
            yield return ("xyz$", RegexOptions.Multiline, "xyz", true);
            yield return ("xyz$", RegexOptions.Multiline, "xyz\n", true);
            yield return ("xyz$", RegexOptions.Multiline, "abcxyz\npqr", true);
            yield return ("xyz$", RegexOptions.Multiline, "abcxyz\npqr=noteol", true);
            yield return ("xyz$", RegexOptions.Multiline, "xyz\n=noteol", true);
            yield return ("xyz$", RegexOptions.Multiline, "xyz=noteol", false);
            yield return ("^abcdef", RegexOptions.None, "ab=ps", false);
            yield return ("^abcdef", RegexOptions.None, "abcde=ps", false);
            yield return ("^abcdef", RegexOptions.None, "abx=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "a=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "aa=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "aa2=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "aaa=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "aaa23=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "aaaa12345=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "az=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "aaaaa=ps", false);
            yield return ("^a{2,4}\\d+z", RegexOptions.None, "a56=ps", false);
            yield return ("^abcdef", RegexOptions.None, "abc=ps", false);
            yield return ("^abcdef", RegexOptions.None, "def=dfa_restart", false);
            yield return ("(ab*(cd|ef))+X", RegexOptions.None, "lkjhlkjhlkjhlkjhabbbbbbcdaefabbbbbbbefa=ps,notbol,noteol", false);
            yield return ("(ab*(cd|ef))+X", RegexOptions.None, "cdabbbbbbbb=ps,notbol,dfa_restart,noteol", false);
            yield return ("(ab*(cd|ef))+X", RegexOptions.None, "efabbbbbbbbbbbbbbbb=ps,notbol,dfa_restart,noteol", false);
            yield return ("(ab*(cd|ef))+X", RegexOptions.None, "bbbbbbbbbbbbcdXyasdfadf=ps,notbol,dfa_restart,noteol", false);
            yield return ("(ab*(cd|ef))+X", RegexOptions.None, "adfadadaklhlkalkajhlkjahdfasdfasdfladsfjkj=ps,noteol", false);
            yield return (":", RegexOptions.None, "No match without a colon", false);
            yield return ("(a+|b+|c+)*c", RegexOptions.None, "aaabbbbccccd", true);
            yield return ("a\\\\b", RegexOptions.None, "ab", false);
            yield return ("a\\\\b", RegexOptions.IgnoreCase, "A=notbol", false);
            yield return ("line\\nbreak", RegexOptions.None, "this is a line\nbreak", true);
            yield return ("line\\nbreak", RegexOptions.None, "line one\nthis is a line\nbreak in the second line", true);
            yield return ("line\\nbreak", RegexOptions.Multiline, "this is a line\nbreak", true);
            yield return ("1234", RegexOptions.None, "123=ps", false);
            yield return ("1234", RegexOptions.None, "a4=ps,dfa_restart", false);
            yield return ("1234", RegexOptions.None, "4=ps,dfa_restart", false);
            yield return ("(?s)A?B", RegexOptions.None, "AB", true);
            yield return ("(?s)A?B", RegexOptions.None, "aB", true);
            yield return ("(?s)A*B", RegexOptions.None, "AB", true);
            yield return ("(?s)A*B", RegexOptions.None, "aB", true);
            yield return ("(?m)A?B", RegexOptions.None, "AB", true);
            yield return ("(?m)A?B", RegexOptions.None, "aB", true);
            yield return ("(?m)A*B", RegexOptions.None, "AB", true);
            yield return ("(?m)A*B", RegexOptions.None, "aB", true);
            yield return ("Content-Type\\x3A[^\\r\\n]{6,}", RegexOptions.None, "Content-Type:xxxxxyyy", true);
            yield return ("Content-Type\\x3A[^\\r\\n]{6,}z", RegexOptions.None, "Content-Type:xxxxxyyyz", true);
            yield return ("Content-Type\\x3A[^a]{6,}", RegexOptions.None, "Content-Type:xxxyyy", true);
            yield return ("Content-Type\\x3A[^a]{6,}z", RegexOptions.None, "Content-Type:xxxyyyz", true);
            yield return ("\\w+(.)(.)?def", RegexOptions.Singleline, "abc\ndef", true);
            yield return ("\\w+(.)(.)?def", RegexOptions.Singleline, "abc\rdef", true);
            yield return ("\\w+(.)(.)?def", RegexOptions.Singleline, "abc\r\ndef", true);
            yield return ("a(?!)|\\wbc", RegexOptions.None, "abc", true);
            yield return ("X$", RegexOptions.None, "X", true);
            yield return ("X$", RegexOptions.None, "X\n", true);
            yield return ("\\d+X|9+Y", RegexOptions.None, "++++123999=ps", false);
            yield return ("dog(sbody)?", RegexOptions.None, "body=dfa_restart", false);
            yield return ("\\babc\\b", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "+++abc+++", true);
            yield return ("\\babc\\b", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "+++ab=ps", false);
            yield return ("\\babc\\b", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "+++ab=ph", false);
            yield return ("abcd*", RegexOptions.None, "dddxxx=dfa_restart", false);
            yield return ("abcd*", RegexOptions.None, "xxxxabcd=ph", true);
            yield return ("abcd*", RegexOptions.None, "xxx=dfa_restart", false);
            yield return (".+", RegexOptions.None, "abc=offset=0", true);
            yield return (".+", RegexOptions.None, "abc=offset=1", true);
            yield return ("(a+)", RegexOptions.None, "aaaa=ovector=4", true);
            yield return ("abcdef", RegexOptions.None, "abc=dfa_restart", false);
            yield return ("abcd", RegexOptions.None, "abcd=ovector=0", true);
            yield return ("a\\b", RegexOptions.None, "a.=allusedtext", true);
            yield return ("(a)(b)|(c)", RegexOptions.None, "XcX=ovector=2,get=1,get=2,get=3,get=4,getall", true);
            yield return ("(?<A>aa)", RegexOptions.None, "aa=get=A", true);
            yield return ("a+", RegexOptions.None, "a=ovector=2,get=1,get=2,getall", true);
            yield return ("()()a+", RegexOptions.None, "a=allcaptures", true);
            yield return ("(02-)?[0-9]{3}-[0-9]{3}", RegexOptions.None, "02-123-123", true);
            yield return ("\\bX", RegexOptions.None, "Xoanon", true);
            yield return ("\\bX", RegexOptions.None, "+Xoanon", true);
            yield return ("\\bX", RegexOptions.None, "YXoanon", false);
            yield return ("\\BX", RegexOptions.None, "YXoanon", true);
            yield return ("\\BX", RegexOptions.None, "Xoanon", false);
            yield return ("\\BX", RegexOptions.None, "+Xoanon", false);
            yield return ("X\\b", RegexOptions.None, "X+oanon", true);
            yield return ("X\\b", RegexOptions.None, "FAX", true);
            yield return ("X\\b", RegexOptions.None, "Xoanon", false);
            yield return ("X\\B", RegexOptions.None, "Xoanon", true);
            yield return ("X\\B", RegexOptions.None, "X+oanon", false);
            yield return ("X\\B", RegexOptions.None, "FAX", false);
            yield return ("[^a]", RegexOptions.None, "a\u0100", true);
            yield return ("^\\p{Cn}", RegexOptions.None, "\u009f", false);
            yield return ("[\\p{Nd}]", RegexOptions.None, "1234", true);
            yield return ("[\\p{Nd}+-]+", RegexOptions.None, "1234", true);
            yield return ("[\\p{Nd}+-]+", RegexOptions.None, "12-34", true);
            yield return ("[\\p{Nd}+-]+", RegexOptions.None, "12+\u0661-34", true);
            yield return ("[\\p{Nd}+-]+", RegexOptions.None, "abcd", false);
            yield return ("ab", RegexOptions.None, "A\u0123B", false);
            yield return ("ab", RegexOptions.None, "Ao{443}B", false);
            yield return ("badutf", RegexOptions.None, "X\u00DF", false);
            yield return ("badutf", RegexOptions.None, "XX\u00EF", false);
            yield return ("badutf", RegexOptions.None, "XXX\u00EF\u0080", false);
            yield return ("badutf", RegexOptions.None, "X\u00F7", false);
            yield return ("badutf", RegexOptions.None, "XX\u00F7\u0080", false);
            yield return ("badutf", RegexOptions.None, "XXX\u00F7\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FB", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00DF\u007f", false);
            yield return ("badutf", RegexOptions.None, "\u00EF\u007f\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00EF\u0080\u007f", false);
            yield return ("badutf", RegexOptions.None, "\u00F7\u007f\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00F7\u0080\u007f\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00F7\u0080\u0080\u007f", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u007f\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u0080\u007f\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u0080\u0080\u007f\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FB\u0080\u0080\u0080\u007f", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u007f\u0080\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u007f\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u0080\u007f\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u0080\u0080\u007f\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FD\u0080\u0080\u0080\u0080\u007f", false);
            yield return ("badutf", RegexOptions.None, "\u00ED\u00A0\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00C0\u008f", false);
            yield return ("badutf", RegexOptions.None, "\u00E0\u0080\u008f", false);
            yield return ("badutf", RegexOptions.None, "\u00F0\u0080\u0080\u008f", false);
            yield return ("badutf", RegexOptions.None, "\u00F8\u0080\u0080\u0080\u008f", false);
            yield return ("badutf", RegexOptions.None, "\u00FC\u0080\u0080\u0080\u0080\u008f", false);
            yield return ("badutf", RegexOptions.None, "\u0080", false);
            yield return ("badutf", RegexOptions.None, "\u00FE", false);
            yield return ("badutf", RegexOptions.None, "\u00FF", false);
            yield return ("badutf", RegexOptions.None, "XX\u00FB\u0080\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "XX\u00FD\u0080\u0080\u0080\u0080\u0080", false);
            yield return ("badutf", RegexOptions.None, "XX\u00F7\u00BF\u00BF\u00BF", false);
            yield return ("shortutf", RegexOptions.None, "XX\u00DF=ph", false);
            yield return ("shortutf", RegexOptions.None, "XX\u00EF=ph", false);
            yield return ("shortutf", RegexOptions.None, "XX\u00EF\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00F7=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00F7\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00F7\u0080\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FB=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FB\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FB\u0080\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FB\u0080\u0080\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FD=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FD\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FD\u0080\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FD\u0080\u0080\u0080=ph", false);
            yield return ("shortutf", RegexOptions.None, "\u00FD\u0080\u0080\u0080\u0080=ph", false);
            yield return ("anything", RegexOptions.None, "X\u00C0\u0080", false);
            yield return ("anything", RegexOptions.None, "XX\u00C1\u008f", false);
            yield return ("anything", RegexOptions.None, "XXX\u00E0\u009f\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F0\u008f\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F8\u0087\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00FC\u0083\u0080\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00FE\u0080\u0080\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00FF\u0080\u0080\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F8\u0088\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F9\u0087\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00FC\u0084\u0080\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00FD\u0083\u0080\u0080\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00C3\u008f", false);
            yield return ("anything", RegexOptions.None, "\u00E0\u00AF\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00E1\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F0\u009f\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F1\u008f\u0080\u0080", false);
            yield return ("anything", RegexOptions.None, "\u00F8\u0088\u0080\u0080\u0080=no_utf_check", false);
            yield return ("anything", RegexOptions.None, "\u00F9\u0087\u0080\u0080\u0080=no_utf_check", false);
            yield return ("anything", RegexOptions.None, "\u00FC\u0084\u0080\u0080\u0080\u0080=no_utf_check", false);
            yield return ("anything", RegexOptions.None, "\u00FD\u0083\u0080\u0080\u0080\u0080=no_utf_check", false);
            yield return ("badutf", RegexOptions.None, "X\u00DFabcd", false);
            yield return ("badutf", RegexOptions.None, "X\u00DFabcd=offset=1", false);
            yield return ("badutf", RegexOptions.None, "X\u00DFabcd=offset=2", false);
            yield return ("X", RegexOptions.None, "\ud800", false);
            yield return ("X", RegexOptions.None, "\uda00", false);
            yield return ("X", RegexOptions.None, "\udfff", false);
            yield return ("a+", RegexOptions.None, "a\u0123aa=offset=1", true);
            yield return ("a+", RegexOptions.None, "a\u0123aa=offset=3", true);
            yield return ("a+", RegexOptions.None, "a\u0123aa=offset=4", true);
            yield return ("...", RegexOptions.IgnoreCase | RegexOptions.Multiline, "abcd\u0080wxzy\u0080pqrs", true);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Multiline, "ab\u0080ab=ph", false);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Multiline, "ab\u0080cdef=ph", false);
            yield return ("ab$", RegexOptions.IgnoreCase | RegexOptions.Multiline, "ab\u0080cdeab", true);
            yield return ("ab$", RegexOptions.IgnoreCase | RegexOptions.Multiline, "ab\u0080cde", false);
            yield return ("X$", RegexOptions.IgnoreCase | RegexOptions.Multiline, "X\u00C4", false);
            yield return ("\\bX", RegexOptions.IgnoreCase | RegexOptions.Multiline, "A\u0080X", true);
            yield return ("\\BX", RegexOptions.IgnoreCase | RegexOptions.Multiline, "A\u0080X", false);
            yield return ("x", RegexOptions.None, "abxyz", true);
            yield return ("x", RegexOptions.None, "\u0080=startchar", false);
            yield return ("x", RegexOptions.None, "abc\u0080=startchar", false);
            yield return ("x", RegexOptions.None, "abc\u0080=startchar,offset=3", false);
            yield return ("A", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "\u00E5A", true);
            yield return ("\\bch\\b", RegexOptions.IgnoreCase | RegexOptions.Multiline, "qchq=ph", false);
            yield return ("\\bch\\b", RegexOptions.IgnoreCase | RegexOptions.Multiline, "qchq=ps", false);
            yield return ("(a+)*zz", RegexOptions.None, "aaaaaaaaaaaaaz", false);
            yield return ("(a+)*zz", RegexOptions.None, "aaaaaaaaaaaaaz=match_limit=3000", false);
            yield return ("(a+)*zz", RegexOptions.None, "aaaaaaaaaaaaaz=depth_limit=10", false);
            yield return ("abcd", RegexOptions.None, "ab=ps", false);
            yield return ("abcd", RegexOptions.None, "ab=ph", false);
            yield return ("abcd", RegexOptions.None, "xyz", false);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "abcd", true);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "ab=ps", false);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "ab=ph", false);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "xyz", false);
            yield return ("abcd", RegexOptions.IgnoreCase, "abcd", true);
            yield return ("abcd", RegexOptions.IgnoreCase, "ab=ps", false);
            yield return ("abcd", RegexOptions.IgnoreCase, "ab=ph", false);
            yield return ("abcd", RegexOptions.IgnoreCase, "xyz", false);
            yield return ("abcd", RegexOptions.IgnoreCase, "xyz=ps", false);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "xyz=ps", false);
            yield return ("^12345678abcd", RegexOptions.Multiline, "12345678abcd", true);
            yield return ("abcd", RegexOptions.None, "abcd=anchored", true);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "abcd=anchored", true);
            yield return ("abcd", RegexOptions.IgnoreCase | RegexOptions.Singleline, "succeed abcd=anchored", true);
            yield return ("abc", RegexOptions.None, "abc=no_jit", true);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Singleline, "abc", true);
            yield return ("abc", RegexOptions.IgnoreCase | RegexOptions.Singleline, "abc=no_jit", true);
            yield return ("[aC]", RegexOptions.Multiline, "match\nmatch", true);
            yield return ("[aCz]", RegexOptions.Multiline, "match\nmatch", true);
            yield return ("", RegexOptions.IgnoreCase | RegexOptions.Singleline, "=null_subject", true);
            yield return ("^abc|def", RegexOptions.None, "abcdef", true);
            yield return ("^abc|def", RegexOptions.None, "abcdef=notbol", true);
            yield return (".*((abc)$|(def))", RegexOptions.None, "defabc", true);
            yield return (".*((abc)$|(def))", RegexOptions.None, "defabc=noteol", true);
            yield return ("the quick brown fox", RegexOptions.None, "The Quick Brown Fox", false);
            yield return ("the quick brown fox", RegexOptions.IgnoreCase, "the quick brown fox", true);
            yield return ("the quick brown fox", RegexOptions.IgnoreCase, "The Quick Brown Fox", true);
            yield return ("a*(b+)(z)(z)", RegexOptions.None, "aaaabbbbzzzz", true);
            yield return ("a*(b+)(z)(z)", RegexOptions.None, "aaaabbbbzzzz=ovector=2", true);
            yield return ("ab.cd", RegexOptions.Singleline, "ab-cd", true);
            yield return ("ab.cd", RegexOptions.Singleline, "ab=cd", true);
            yield return ("ab.cd", RegexOptions.Singleline, "ab\ncd", true);
            yield return ("a?|b?", RegexOptions.None, "abc", true);
            yield return ("\\w+A", RegexOptions.None, "CDAAAAB", true);
            yield return ("^d(e)$", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "acdef", false);
            yield return ("^d(e)$", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "acdef=posix_startend=2", false);
            yield return ("ABC", RegexOptions.Singleline, "ABC", true);
            yield return ("((a)(b)?(c))", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "123ace", true);
            yield return ("\\w", RegexOptions.None, "+++\u00C2", true);
            if (!RegexHelpers.IsNonBacktracking(engine)) // These following patterns use features that are not currently supported by the non-backtracking engine
            {
                yield return ("^(?=ab(de))(abd)(e)", RegexOptions.None, "abde", true);
                yield return ("^(?!(ab)de|x)(abd)(f)", RegexOptions.None, "abdf", true);
                yield return ("^(?=(ab(cd)))(ab)", RegexOptions.None, "abcd", true);
                yield return ("\\A(abc|def)=(\\1){2,3}\\Z", RegexOptions.None, "abc=abcabc", true);
                yield return ("\\A(abc|def)=(\\1){2,3}\\Z", RegexOptions.None, "def=defdefdef", true);
                yield return ("\\A(abc|def)=(\\1){2,3}\\Z", RegexOptions.None, "abc=defdef", false);
                yield return ("^(a)(b)(c)(d)(e)(f)(g)(h)(i)(j)(k)\\11*(\\3\\4)\\1(?#)2$", RegexOptions.None, "abcdefghijkcda2", true);
                yield return ("^(a)(b)(c)(d)(e)(f)(g)(h)(i)(j)(k)\\11*(\\3\\4)\\1(?#)2$", RegexOptions.None, "abcdefghijkkkkcda2", true);
                yield return ("(cat(a(ract|tonic)|erpillar)) \\1()2(3)", RegexOptions.None, "cataract cataract23", true);
                yield return ("(cat(a(ract|tonic)|erpillar)) \\1()2(3)", RegexOptions.None, "catatonic catatonic23", true);
                yield return ("(cat(a(ract|tonic)|erpillar)) \\1()2(3)", RegexOptions.None, "caterpillar caterpillar23", true);
                yield return ("\\w+(?=\\t)", RegexOptions.None, "the quick brown\t fox", true);
                yield return ("foo(?!bar)(.*)", RegexOptions.None, "foobar is foolish see?", true);
                yield return ("(?:(?!foo)...|^.{0,2})bar(.*)", RegexOptions.None, "foobar crowbar etc", true);
                yield return ("(?:(?!foo)...|^.{0,2})bar(.*)", RegexOptions.None, "barrel", true);
                yield return ("(?:(?!foo)...|^.{0,2})bar(.*)", RegexOptions.None, "2barrel", true);
                yield return ("(?:(?!foo)...|^.{0,2})bar(.*)", RegexOptions.None, "A barrel", true);
                yield return ("^(\\D*)(?=\\d)(?!123)", RegexOptions.None, "abc456", true);
                yield return ("^(\\D*)(?=\\d)(?!123)", RegexOptions.None, "abc123", false);
                yield return ("^(a)\\1{2,3}(.)", RegexOptions.None, "aaab", true);
                yield return ("^(a)\\1{2,3}(.)", RegexOptions.None, "aaaab", true);
                yield return ("^(a)\\1{2,3}(.)", RegexOptions.None, "aaaaab", true);
                yield return ("^(a)\\1{2,3}(.)", RegexOptions.None, "aaaaaab", true);
                yield return ("(?!^)abc", RegexOptions.None, "the abc", true);
                yield return ("(?!^)abc", RegexOptions.None, "abc", false);
                yield return ("(?=^)abc", RegexOptions.None, "abc", true);
                yield return ("(?=^)abc", RegexOptions.None, "the abc", false);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "Alan Other <user@dom.ain>", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "<user@dom.ain>", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "user@dom.ain", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "\"A. Other\" <user.1234@dom.ain> (a comment)", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "A. Other <user.1234@dom.ain> (a comment)", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "\"/s=user/ou=host/o=place/prmd=uu.yy/admd= /c=gb/\"@x400-re.lay", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "A missing angle <user@some.where", true);
                yield return ("  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                          # optional leading comment\n(?:    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n# address\n|                     #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)             # one word, optionally followed by....\n(?:\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037]  |  # atom and space parts, or...\n\\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)       |  # comments, or...\n\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n# quoted strings\n)*\n<  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                     # leading <\n(?:  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  ,  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n)* # further okay, if led by comma\n:                                # closing colon\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  )? #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)                    # initial word\n(?:  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\" (?:                      # opening quote...\n[^\\\\\\x80-\\xff\\n\\015\"]                #   Anything except backslash and quote\n|                     #    or\n\\\\ [^\\x80-\\xff]           #   Escaped something (something != CR)\n)* \"  # closing quote\n)  )* # further okay, if led by a period\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  @  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*    (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                           # initial subdomain\n(?:                                  #\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  \\.                        # if led by a period...\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*   (?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|   \\[                         # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*    #    stuff\n\\]                        #           ]\n)                     #   ...further okay\n)*\n#       address spec\n(?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*  > #                  trailing >\n# name and address\n)  (?: [\\040\\t] |  \\(\n(?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  |  \\( (?:  [^\\\\\\x80-\\xff\\n\\015()]  |  \\\\ [^\\x80-\\xff]  )* \\)  )*\n\\)  )*                       # optional trailing comment\n", RegexOptions.IgnorePatternWhitespace, "The quick brown fox", false);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "Alan Other <user@dom.ain>", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "<user@dom.ain>", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "user@dom.ain", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "\"A. Other\" <user.1234@dom.ain> (a comment)", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "A. Other <user.1234@dom.ain> (a comment)", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "\"/s=user/ou=host/o=place/prmd=uu.yy/admd= /c=gb/\"@x400-re.lay", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "A missing angle <user@some.where", true);
                yield return ("[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional leading comment\n(?:\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n# address\n|                             #  or\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n# leading word\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *               # \"normal\" atoms and or spaces\n(?:\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n|\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n) # \"special\" comment or quoted string\n[^()<>@,;:\".\\\\\\[\\]\\x80-\\xff\\000-\\010\\012-\\037] *            #  more \"normal\"\n)*\n<\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# <\n(?:\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n(?: ,\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n)*  # additional domains\n:\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)?     #       optional route\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n# Atom\n|                       #  or\n\"                                     # \"\n[^\\\\\\x80-\\xff\\n\\015\"] *                            #   normal\n(?:  \\\\ [^\\x80-\\xff]  [^\\\\\\x80-\\xff\\n\\015\"] * )*        #   ( special normal* )*\n\"                                     #        \"\n# Quoted string\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# additional words\n)*\n@\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n(?:\n\\.\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n(?:\n[^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]+    # some number of atom characters...\n(?![^(\\040)<>@,;:\".\\\\\\[\\]\\000-\\037\\x80-\\xff]) # ..not followed by something that could be part of an atom\n|\n\\[                            # [\n(?: [^\\\\\\x80-\\xff\\n\\015\\[\\]] |  \\\\ [^\\x80-\\xff]  )*     #    stuff\n\\]                           #           ]\n)\n[\\040\\t]*                    # Nab whitespace.\n(?:\n\\(                              #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                             #     normal*\n(?:                                 #       (\n(?:  \\\\ [^\\x80-\\xff]  |\n\\(                            #  (\n[^\\\\\\x80-\\xff\\n\\015()] *                            #     normal*\n(?:  \\\\ [^\\x80-\\xff]   [^\\\\\\x80-\\xff\\n\\015()] * )*        #     (special normal*)*\n\\)                           #                       )\n)    #         special\n[^\\\\\\x80-\\xff\\n\\015()] *                         #         normal*\n)*                                  #            )*\n\\)                             #                )\n[\\040\\t]* )*    # If comment found, allow more spaces.\n# optional trailing comments\n)*\n#       address spec\n>                    #                 >\n# name and address\n)\n", RegexOptions.IgnorePatternWhitespace, "The quick brown fox", false);
                yield return ("^(cow|)\\1(bell)", RegexOptions.None, "cowcowbell", true);
                yield return ("^(cow|)\\1(bell)", RegexOptions.None, "bell", true);
                yield return ("^(cow|)\\1(bell)", RegexOptions.None, "cowbell", false);
                yield return ("^(a|)\\1*b", RegexOptions.None, "ab", true);
                yield return ("^(a|)\\1*b", RegexOptions.None, "aaaab", true);
                yield return ("^(a|)\\1*b", RegexOptions.None, "b", true);
                yield return ("^(a|)\\1*b", RegexOptions.None, "acb", false);
                yield return ("^(a|)\\1+b", RegexOptions.None, "aab", true);
                yield return ("^(a|)\\1+b", RegexOptions.None, "aaaab", true);
                yield return ("^(a|)\\1+b", RegexOptions.None, "b", true);
                yield return ("^(a|)\\1+b", RegexOptions.None, "ab", false);
                yield return ("^(a|)\\1?b", RegexOptions.None, "ab", true);
                yield return ("^(a|)\\1?b", RegexOptions.None, "aab", true);
                yield return ("^(a|)\\1?b", RegexOptions.None, "b", true);
                yield return ("^(a|)\\1?b", RegexOptions.None, "acb", false);
                yield return ("^(a|)\\1{2}b", RegexOptions.None, "aaab", true);
                yield return ("^(a|)\\1{2}b", RegexOptions.None, "b", true);
                yield return ("^(a|)\\1{2}b", RegexOptions.None, "ab", false);
                yield return ("^(a|)\\1{2}b", RegexOptions.None, "aab", false);
                yield return ("^(a|)\\1{2}b", RegexOptions.None, "aaaab", false);
                yield return ("^(a|)\\1{2,3}b", RegexOptions.None, "aaab", true);
                yield return ("^(a|)\\1{2,3}b", RegexOptions.None, "aaaab", true);
                yield return ("^(a|)\\1{2,3}b", RegexOptions.None, "b", true);
                yield return ("^(a|)\\1{2,3}b", RegexOptions.None, "ab", false);
                yield return ("^(a|)\\1{2,3}b", RegexOptions.None, "aab", false);
                yield return ("^(a|)\\1{2,3}b", RegexOptions.None, "aaaaab", false);
                yield return ("(abc)\\1", RegexOptions.IgnoreCase, "abcabc", true);
                yield return ("(abc)\\1", RegexOptions.IgnoreCase, "ABCabc", true);
                yield return ("(abc)\\1", RegexOptions.IgnoreCase, "abcABC", true);
                yield return ("^(A)(B)(C)(D)(E)(F)(G)(H)(I)\\8\\9$", RegexOptions.None, "ABCDEFGHIHI", true);
                yield return ("(a)(b)(c)(d)(e)(f)(g)(h)(i)(j)(k)(l)\\12\\123", RegexOptions.None, "abcdefghijkllS", true);
                yield return ("(a)\\1{8,}", RegexOptions.None, "aaaaaaaaa", true);
                yield return ("(a)\\1{8,}", RegexOptions.None, "aaaaaaaaaa", true);
                yield return ("(a)\\1{8,}", RegexOptions.None, "aaaaaaa", false);
                yield return ("(\\.\\d\\d((?=0)|\\d(?=\\d)))", RegexOptions.None, "1.230003938", true);
                yield return ("(\\.\\d\\d((?=0)|\\d(?=\\d)))", RegexOptions.None, "1.875000282", true);
                yield return ("(\\.\\d\\d((?=0)|\\d(?=\\d)))", RegexOptions.None, "1.235", false);
                yield return ("^\\D*(?!123)", RegexOptions.None, "ABC123", true);
                yield return ("^(\\D*)(?=\\d)(?!123)", RegexOptions.None, "ABC445", true);
                yield return ("^(\\D*)(?=\\d)(?!123)", RegexOptions.None, "ABC123", false);
                yield return ("(?!\\A)x", RegexOptions.Multiline, "abx\n", true);
                yield return ("(?!\\A)x", RegexOptions.Multiline, "a\nx\n", true);
                yield return ("(?!\\A)x", RegexOptions.Multiline, "x\nb\n", false);
                yield return ("(AB)*?\\1", RegexOptions.None, "ABABAB", true);
                yield return ("(AB)*\\1", RegexOptions.None, "ABABAB", true);
                yield return ("(?<!bar)foo", RegexOptions.None, "foo", true);
                yield return ("(?<!bar)foo", RegexOptions.None, "catfood", true);
                yield return ("(?<!bar)foo", RegexOptions.None, "arfootle", true);
                yield return ("(?<!bar)foo", RegexOptions.None, "rfoosh", true);
                yield return ("(?<!bar)foo", RegexOptions.None, "barfoo", false);
                yield return ("(?<!bar)foo", RegexOptions.None, "towbarfoo", false);
                yield return ("\\w{3}(?<!bar)foo", RegexOptions.None, "catfood", true);
                yield return ("\\w{3}(?<!bar)foo", RegexOptions.None, "foo", false);
                yield return ("\\w{3}(?<!bar)foo", RegexOptions.None, "barfoo", false);
                yield return ("\\w{3}(?<!bar)foo", RegexOptions.None, "towbarfoo", false);
                yield return ("(?<=(foo)a)bar", RegexOptions.None, "fooabar", true);
                yield return ("(?<=(foo)a)bar", RegexOptions.None, "bar", false);
                yield return ("(?<=(foo)a)bar", RegexOptions.None, "foobbar", false);
                yield return ("(?>.*\\/)foo", RegexOptions.None, "/this/is/a/very/long/line/in/deed/with/very/many/slashes/in/and/foo", true);
                yield return ("(?>.*\\/)foo", RegexOptions.None, "/this/is/a/very/long/line/in/deed/with/very/many/slashes/in/it/you/see/", false);
                yield return ("(?>(\\.\\d\\d[1-9]?))\\d+", RegexOptions.None, "1.230003938", true);
                yield return ("(?>(\\.\\d\\d[1-9]?))\\d+", RegexOptions.None, "1.875000282", true);
                yield return ("(?>(\\.\\d\\d[1-9]?))\\d+", RegexOptions.None, "1.235", false);
                yield return ("^((?>\\w+)|(?>\\s+))*$", RegexOptions.None, "now is the time for all good men to come to the aid of the party", true);
                yield return ("^((?>\\w+)|(?>\\s+))*$", RegexOptions.None, "this is not a line with only words and spaces!", false);
                yield return ("((?>\\d+))(\\w)", RegexOptions.None, "12345a", true);
                yield return ("((?>\\d+))(\\w)", RegexOptions.None, "12345+", false);
                yield return ("(?>a+)b", RegexOptions.None, "aaab", true);
                yield return ("((?>a+)b)", RegexOptions.None, "aaab", true);
                yield return ("(?>(a+))b", RegexOptions.None, "aaab", true);
                yield return ("(?>b)+", RegexOptions.None, "aaabbbccc", true);
                yield return ("(?>a+|b+|c+)*c", RegexOptions.None, "aaabbbbccccd", true);
                yield return ("((?>[^()]+)|\\([^()]*\\))+", RegexOptions.None, "((abc(ade)ufh()()x", true);
                yield return ("\\(((?>[^()]+)|\\([^()]+\\))+\\)", RegexOptions.None, "(abc)", true);
                yield return ("\\(((?>[^()]+)|\\([^()]+\\))+\\)", RegexOptions.None, "(abc(def)xyz)", true);
                yield return ("\\(((?>[^()]+)|\\([^()]+\\))+\\)", RegexOptions.None, "((()aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", false);
                yield return ("a(?=b(?i)c)\\w\\wd", RegexOptions.None, "abcd", true);
                yield return ("a(?=b(?i)c)\\w\\wd", RegexOptions.None, "abCd", true);
                yield return ("a(?=b(?i)c)\\w\\wd", RegexOptions.None, "aBCd", false);
                yield return ("a(?=b(?i)c)\\w\\wd", RegexOptions.None, "abcD", false);
                yield return ("(?>a(?i)b+)+c", RegexOptions.None, "abc", true);
                yield return ("(?>a(?i)b+)+c", RegexOptions.None, "aBbc", true);
                yield return ("(?>a(?i)b+)+c", RegexOptions.None, "aBBc", true);
                yield return ("(?>a(?i)b+)+c", RegexOptions.None, "Abc", false);
                yield return ("(?>a(?i)b+)+c", RegexOptions.None, "abAb", false);
                yield return ("(?>a(?i)b+)+c", RegexOptions.None, "abbC", false);
                yield return ("(?=a(?i)b)\\w\\wc", RegexOptions.None, "abc", true);
                yield return ("(?=a(?i)b)\\w\\wc", RegexOptions.None, "aBc", true);
                yield return ("(?=a(?i)b)\\w\\wc", RegexOptions.None, "Ab", false);
                yield return ("(?=a(?i)b)\\w\\wc", RegexOptions.None, "abC", false);
                yield return ("(?=a(?i)b)\\w\\wc", RegexOptions.None, "aBC", false);
                yield return ("(?<=a(?i)b)(\\w\\w)c", RegexOptions.None, "abxxc", true);
                yield return ("(?<=a(?i)b)(\\w\\w)c", RegexOptions.None, "aBxxc", true);
                yield return ("(?<=a(?i)b)(\\w\\w)c", RegexOptions.None, "Abxxc", false);
                yield return ("(?<=a(?i)b)(\\w\\w)c", RegexOptions.None, "ABxxc", false);
                yield return ("(?<=a(?i)b)(\\w\\w)c", RegexOptions.None, "abxxC", false);
                yield return ("(?:(a)|b)(?(1)A|B)", RegexOptions.None, "aA", true);
                yield return ("(?:(a)|b)(?(1)A|B)", RegexOptions.None, "bB", true);
                yield return ("(?:(a)|b)(?(1)A|B)", RegexOptions.None, "aB", false);
                yield return ("(?:(a)|b)(?(1)A|B)", RegexOptions.None, "bA", false);
                yield return ("^(a)?(?(1)a|b)+$", RegexOptions.None, "aa", true);
                yield return ("^(a)?(?(1)a|b)+$", RegexOptions.None, "b", true);
                yield return ("^(a)?(?(1)a|b)+$", RegexOptions.None, "bb", true);
                yield return ("^(a)?(?(1)a|b)+$", RegexOptions.None, "ab", false);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)", RegexOptions.None, "abc:", true);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)", RegexOptions.None, "12", true);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)", RegexOptions.None, "123", true);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)", RegexOptions.None, "xyz", false);
                yield return ("^(?(?!abc)\\d\\d|\\w{3}:)$", RegexOptions.None, "abc:", true);
                yield return ("^(?(?!abc)\\d\\d|\\w{3}:)$", RegexOptions.None, "12", true);
                yield return ("^(?(?!abc)\\d\\d|\\w{3}:)$", RegexOptions.None, "123", false);
                yield return ("^(?(?!abc)\\d\\d|\\w{3}:)$", RegexOptions.None, "xyz", false);
                yield return ("(?(?<=foo)bar|cat)", RegexOptions.None, "foobar", true);
                yield return ("(?(?<=foo)bar|cat)", RegexOptions.None, "cat", true);
                yield return ("(?(?<=foo)bar|cat)", RegexOptions.None, "fcat", true);
                yield return ("(?(?<=foo)bar|cat)", RegexOptions.None, "focat", true);
                yield return ("(?(?<=foo)bar|cat)", RegexOptions.None, "foocat", false);
                yield return ("(?(?<!foo)cat|bar)", RegexOptions.None, "foobar", true);
                yield return ("(?(?<!foo)cat|bar)", RegexOptions.None, "cat", true);
                yield return ("(?(?<!foo)cat|bar)", RegexOptions.None, "fcat", true);
                yield return ("(?(?<!foo)cat|bar)", RegexOptions.None, "focat", true);
                yield return ("(?(?<!foo)cat|bar)", RegexOptions.None, "foocat", false);
                yield return ("( \\( )? [^()]+ (?(1) \\) |) ", RegexOptions.IgnorePatternWhitespace, "abcd", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) |) ", RegexOptions.IgnorePatternWhitespace, "(abcd)", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) |) ", RegexOptions.IgnorePatternWhitespace, "the quick (abcd) fox", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) |) ", RegexOptions.IgnorePatternWhitespace, "(abcd", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) ) ", RegexOptions.IgnorePatternWhitespace, "abcd", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) ) ", RegexOptions.IgnorePatternWhitespace, "(abcd)", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) ) ", RegexOptions.IgnorePatternWhitespace, "the quick (abcd) fox", true);
                yield return ("( \\( )? [^()]+ (?(1) \\) ) ", RegexOptions.IgnorePatternWhitespace, "(abcd", true);
                yield return ("^(?(2)a|(1)(2))+$", RegexOptions.None, "12", true);
                yield return ("^(?(2)a|(1)(2))+$", RegexOptions.None, "12a", true);
                yield return ("^(?(2)a|(1)(2))+$", RegexOptions.None, "12aa", true);
                yield return ("^(?(2)a|(1)(2))+$", RegexOptions.None, "1234", false);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "blah blah", true);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "BLAH BLAH", true);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "Blah Blah", true);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "blaH blaH", true);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "blah BLAH", false);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "Blah blah", false);
                yield return ("((?i)blah)\\s+\\1", RegexOptions.None, "blaH blah", false);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "blah blah", true);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "BLAH BLAH", true);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "Blah Blah", true);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "blaH blaH", true);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "blah BLAH", true);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "Blah blah", true);
                yield return ("((?i)blah)\\s+(?i:\\1)", RegexOptions.None, "blaH blah", true);
                yield return ("((?i)blah)\\s+(?m)A(?i:\\1)", RegexOptions.None, "blah ABLAH", true);
                yield return ("((?i)blah)\\s+(?m)A(?i:\\1)", RegexOptions.None, "blah aBLAH", false);
                yield return ("(?>a*)*", RegexOptions.None, "a", true);
                yield return ("(?>a*)*", RegexOptions.None, "aa", true);
                yield return ("(?>a*)*", RegexOptions.None, "aaaa", true);
                yield return ("(?>a*)*", RegexOptions.None, "aaabcde", true);
                yield return ("((?>a*))*", RegexOptions.None, "aaaaa", true);
                yield return ("((?>a*))*", RegexOptions.None, "aabbaa", true);
                yield return ("((?>a*?))*", RegexOptions.None, "aaaaa", true);
                yield return ("((?>a*?))*", RegexOptions.None, "aabbaa", true);
                yield return ("(?(?=[^a-z]+[a-z])  \\d{2}-[a-z]{3}-\\d{2}  |  \\d{2}-\\d{2}-\\d{2} ) ", RegexOptions.IgnorePatternWhitespace, "12-sep-98", true);
                yield return ("(?(?=[^a-z]+[a-z])  \\d{2}-[a-z]{3}-\\d{2}  |  \\d{2}-\\d{2}-\\d{2} ) ", RegexOptions.IgnorePatternWhitespace, "12-09-98", true);
                yield return ("(?(?=[^a-z]+[a-z])  \\d{2}-[a-z]{3}-\\d{2}  |  \\d{2}-\\d{2}-\\d{2} ) ", RegexOptions.IgnorePatternWhitespace, "sep-12-98", false);
                yield return ("(?<=(foo))bar\\1", RegexOptions.None, "foobarfoo", true);
                yield return ("(?<=(foo))bar\\1", RegexOptions.None, "foobarfootling", true);
                yield return ("(?<=(foo))bar\\1", RegexOptions.None, "foobar", false);
                yield return ("(?<=(foo))bar\\1", RegexOptions.None, "barfoo", false);
                yield return ("(?<=foo\\n)^bar", RegexOptions.Multiline, "foo\nbar", true);
                yield return ("(?<=foo\\n)^bar", RegexOptions.Multiline, "bar", false);
                yield return ("(?<=foo\\n)^bar", RegexOptions.Multiline, "baz\nbar", false);
                yield return ("(?<=(?<!foo)bar)baz", RegexOptions.None, "barbaz", true);
                yield return ("(?<=(?<!foo)bar)baz", RegexOptions.None, "barbarbaz", true);
                yield return ("(?<=(?<!foo)bar)baz", RegexOptions.None, "koobarbaz", true);
                yield return ("(?<=(?<!foo)bar)baz", RegexOptions.None, "baz", false);
                yield return ("(?<=(?<!foo)bar)baz", RegexOptions.None, "foobarbaz", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaa", true);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaa", true);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaa", true);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "a", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaaaaaaa", false);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaaaaaaaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaa", true);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaa", true);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaa", true);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaa", true);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaa", true);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "a", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaaaaaaa", false);
                yield return ("^(a\\1?)(a\\1?)(a\\2?)(a\\3?)$", RegexOptions.None, "aaaaaaaaaaaaaaaa", false);
                yield return ("((((((((((a))))))))))\\10", RegexOptions.None, "aa", true);
                yield return ("(abc)\\1", RegexOptions.None, "abcabc", true);
                yield return ("([a-c]*)\\1", RegexOptions.None, "abcabc", true);
                yield return ("(a)|\\1", RegexOptions.None, "a", true);
                yield return ("(a)|\\1", RegexOptions.None, "ab", true);
                yield return ("(a)|\\1", RegexOptions.None, "x", false);
                yield return ("(([a-c])b*?\\2)*", RegexOptions.None, "ababbbcbc", true);
                yield return ("(([a-c])b*?\\2){3}", RegexOptions.None, "ababbbcbc", true);
                yield return ("((\\3|b)\\2(a)x)+", RegexOptions.None, "aaaxabaxbaaxbbax", true);
                yield return ("((\\3|b)\\2(a)){2,}", RegexOptions.None, "bbaababbabaaaaabbaaaabba", true);
                yield return ("((((((((((a))))))))))\\10", RegexOptions.IgnoreCase, "AA", true);
                yield return ("(abc)\\1", RegexOptions.IgnoreCase, "ABCABC", true);
                yield return ("([a-c]*)\\1", RegexOptions.IgnoreCase, "ABCABC", true);
                yield return ("a(?!b).", RegexOptions.None, "abad", true);
                yield return ("a(?=d).", RegexOptions.None, "abad", true);
                yield return ("a(?=c|d).", RegexOptions.None, "abad", true);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "AB", false);
                yield return ("^(a(?(1)\\1)){4}$", RegexOptions.None, "aaaaaaaaaa", true);
                yield return ("^(a(?(1)\\1)){4}$", RegexOptions.None, "aaaaaaaaa", false);
                yield return ("^(a(?(1)\\1)){4}$", RegexOptions.None, "aaaaaaaaaaa", false);
                yield return ("(?<=a)b", RegexOptions.None, "ab", true);
                yield return ("(?<=a)b", RegexOptions.None, "cb", false);
                yield return ("(?<=a)b", RegexOptions.None, "b", false);
                yield return ("(?<!c)b", RegexOptions.None, "ab", true);
                yield return ("(?<!c)b", RegexOptions.None, "b", true);
                yield return ("^(?:b|a(?=(.)))*\\1", RegexOptions.None, "abc", true);
                yield return ("(ab)\\d\\1", RegexOptions.IgnoreCase, "Ab4ab", true);
                yield return ("(ab)\\d\\1", RegexOptions.IgnoreCase, "ab4Ab", true);
                yield return ("(?<![cd])b", RegexOptions.None, "B\nB", false);
                yield return ("(?<![cd])b", RegexOptions.None, "dbcb", false);
                yield return ("(?<![cd])[ab]", RegexOptions.None, "dbaacb", true);
                yield return ("(?<!(c|d))[ab]", RegexOptions.None, "dbaacb", true);
                yield return ("(?<!cd)[ab]", RegexOptions.None, "cdaccb", true);
                yield return ("((?s).)c(?!.)", RegexOptions.None, "a\nb\nc\n", true);
                yield return ("((?s)b.)c(?!.)", RegexOptions.None, "a\nb\nc\n", true);
                yield return ("(x)?(?(1)a|b)", RegexOptions.None, "a", false);
                yield return ("(x)?(?(1)b|a)", RegexOptions.None, "a", true);
                yield return ("()?(?(1)b|a)", RegexOptions.None, "a", true);
                yield return ("()?(?(1)a|b)", RegexOptions.None, "a", true);
                yield return ("^(\\()?blah(?(1)(\\)))$", RegexOptions.None, "(blah)", true);
                yield return ("^(\\()?blah(?(1)(\\)))$", RegexOptions.None, "blah", true);
                yield return ("^(\\()?blah(?(1)(\\)))$", RegexOptions.None, "a", false);
                yield return ("^(\\()?blah(?(1)(\\)))$", RegexOptions.None, "blah)", false);
                yield return ("^(\\()?blah(?(1)(\\)))$", RegexOptions.None, "(blah", false);
                yield return ("^(\\(+)?blah(?(1)(\\)))$", RegexOptions.None, "(blah)", true);
                yield return ("^(\\(+)?blah(?(1)(\\)))$", RegexOptions.None, "blah", true);
                yield return ("^(\\(+)?blah(?(1)(\\)))$", RegexOptions.None, "blah)", false);
                yield return ("^(\\(+)?blah(?(1)(\\)))$", RegexOptions.None, "(blah", false);
                yield return ("(?(?!a)b|a)", RegexOptions.None, "a", true);
                yield return ("(?(?=a)b|a)", RegexOptions.None, "a", false);
                yield return ("(?(?=a)a|b)", RegexOptions.None, "a", true);
                yield return ("(?=(a+?))(\\1ab)", RegexOptions.None, "aaab", true);
                yield return ("$(?<=^(a))", RegexOptions.None, "a", true);
                yield return ("^(?=(a+?))\\1ab", RegexOptions.None, "aaab", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "abc", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a-b", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "0-9", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a.b", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "5.6.7", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "the.quick.brown.fox", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a100.b200.300c", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "12-ab.1245", true);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, ".a", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "-a", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a-", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a.", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a_b", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a.-", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "a..", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "ab..bc", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "the.quick.brown.fox-", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "the.quick.brown.fox.", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "the.quick.brown.fox_", false);
                yield return ("^(?>(?(1)\\.|())[^\\W_](?>[a-z0-9-]*[^\\W_])?)+$", RegexOptions.None, "the.quick.brown.fox+", false);
                yield return ("(?>.*)(?<=(abcd|wxyz))", RegexOptions.None, "alphabetabcd", true);
                yield return ("(?>.*)(?<=(abcd|wxyz))", RegexOptions.None, "endingwxyz", true);
                yield return ("(?>.*)(?<=(abcd|wxyz))", RegexOptions.None, "a rather long string that doesn't end with one of them", false);
                yield return ("word (?>(?:(?!otherword)[a-zA-Z0-9]+ ){0,30})otherword", RegexOptions.None, "word cat dog elephant mussel cow horse canary baboon snake shark otherword", true);
                yield return ("word (?>(?:(?!otherword)[a-zA-Z0-9]+ ){0,30})otherword", RegexOptions.None, "word cat dog elephant mussel cow horse canary baboon snake shark", false);
                yield return ("word (?>[a-zA-Z0-9]+ ){0,30}otherword", RegexOptions.None, "word cat dog elephant mussel cow horse canary baboon snake shark the quick brown fox and the lazy dog and several other words getting close to thirty by now I hope", false);
                yield return ("(?<=\\d{3}(?!999))foo", RegexOptions.None, "999foo", true);
                yield return ("(?<=\\d{3}(?!999))foo", RegexOptions.None, "123999foo", true);
                yield return ("(?<=\\d{3}(?!999))foo", RegexOptions.None, "123abcfoo", false);
                yield return ("(?<=(?!...999)\\d{3})foo", RegexOptions.None, "999foo", true);
                yield return ("(?<=(?!...999)\\d{3})foo", RegexOptions.None, "123999foo", true);
                yield return ("(?<=(?!...999)\\d{3})foo", RegexOptions.None, "123abcfoo", false);
                yield return ("(?<=\\d{3}(?!999)...)foo", RegexOptions.None, "123abcfoo", true);
                yield return ("(?<=\\d{3}(?!999)...)foo", RegexOptions.None, "123456foo", true);
                yield return ("(?<=\\d{3}(?!999)...)foo", RegexOptions.None, "123999foo", false);
                yield return ("(?<=\\d{3}...)(?<!999)foo", RegexOptions.None, "123abcfoo", true);
                yield return ("(?<=\\d{3}...)(?<!999)foo", RegexOptions.None, "123456foo", true);
                yield return ("(?<=\\d{3}...)(?<!999)foo", RegexOptions.None, "123999foo", false);
                yield return ("<a[\\s]+href[\\s]*=[\\s]*          # find <a href=\n ([\\\"\\'])?                       # find single or double quote\n (?(1) (.*?)\\1 | ([^\\s]+))       # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href=abcd xyz", true);
                yield return ("<a[\\s]+href[\\s]*=[\\s]*          # find <a href=\n ([\\\"\\'])?                       # find single or double quote\n (?(1) (.*?)\\1 | ([^\\s]+))       # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href=\"abcd xyz pqr\" cats", true);
                yield return ("<a[\\s]+href[\\s]*=[\\s]*          # find <a href=\n ([\\\"\\'])?                       # find single or double quote\n (?(1) (.*?)\\1 | ([^\\s]+))       # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href='abcd xyz pqr' cats", true);
                yield return ("<a\\s+href\\s*=\\s*                # find <a href=\n ([\"'])?                         # find single or double quote\n (?(1) (.*?)\\1 | (\\S+))          # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href=abcd xyz", true);
                yield return ("<a\\s+href\\s*=\\s*                # find <a href=\n ([\"'])?                         # find single or double quote\n (?(1) (.*?)\\1 | (\\S+))          # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href=\"abcd xyz pqr\" cats", true);
                yield return ("<a\\s+href\\s*=\\s*                # find <a href=\n ([\"'])?                         # find single or double quote\n (?(1) (.*?)\\1 | (\\S+))          # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href       =       'abcd xyz pqr' cats", true);
                yield return ("<a\\s+href(?>\\s*)=(?>\\s*)        # find <a href=\n ([\"'])?                         # find single or double quote\n (?(1) (.*?)\\1 | (\\S+))          # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href=abcd xyz", true);
                yield return ("<a\\s+href(?>\\s*)=(?>\\s*)        # find <a href=\n ([\"'])?                         # find single or double quote\n (?(1) (.*?)\\1 | (\\S+))          # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href=\"abcd xyz pqr\" cats", true);
                yield return ("<a\\s+href(?>\\s*)=(?>\\s*)        # find <a href=\n ([\"'])?                         # find single or double quote\n (?(1) (.*?)\\1 | (\\S+))          # if quote found, match up to next matching\n                                 # quote, otherwise match up to next space\n", RegexOptions.IgnoreCase | RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "<a href       =       'abcd xyz pqr' cats", true);
                yield return ("((?>Z)+|A)*", RegexOptions.None, "ZABCDEFG", true);
                yield return ("((?>)+|A)*", RegexOptions.None, "ZABCDEFG", true);
                yield return ("(?!\\A)x", RegexOptions.Multiline, "a\nxb\n", true);
                yield return ("(?!^)x", RegexOptions.Multiline, "a\nxb\n", false);
                yield return ("\\Gabc", RegexOptions.None, "abc", true);
                yield return ("\\Gabc", RegexOptions.None, "xyzabc", false);
                yield return ("\\Gabc.", RegexOptions.None, "abc1abc2xyzabc3", true);
                yield return ("(.*)\\d+\\1", RegexOptions.None, "abc123abc", true);
                yield return ("(.*)\\d+\\1", RegexOptions.None, "abc123bc", true);
                yield return ("(.*)\\d+\\1", RegexOptions.Singleline, "abc123abc", true);
                yield return ("(.*)\\d+\\1", RegexOptions.Singleline, "abc123bc", true);
                yield return ("((.*))\\d+\\1", RegexOptions.None, "abc123abc", true);
                yield return ("((.*))\\d+\\1", RegexOptions.None, "abc123bc", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123::a123", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123:b342::abcd", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123:b342::324e:abcd", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123:ddde:b342::324e:abcd", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123:ddde:b342::324e:dcba:abcd", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123:ddde:9999:b342::324e:dcba:abcd", true);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "1:2:3:4:5:6:7:8", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123:bce:ddde:9999:b342::324e:dcba:abcd", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "a123::9999:b342::324e:dcba:abcd", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "abcde:2:3:4:5:6:7:8", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "::1", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "abcd:fee0:123::", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, ":1", false);
                yield return ("^(?!:)                       # colon disallowed at start\n  (?:                         # start of item\n    (?: [0-9a-f]{1,4} |       # 1-4 hex digits or\n    (?(1)0 | () ) )           # if null previously matched, fail; else null\n    :                         # followed by colon\n  ){1,7}                      # end item; 1-7 of them required               \n  [0-9a-f]{1,4} $             # final hex number at end of string\n  (?(1)|.)                    # check that there was an empty component\n  ", RegexOptions.IgnoreCase | RegexOptions.IgnorePatternWhitespace, "1:", false);
                yield return ("(?<=Z)X.", RegexOptions.None, "\u0084XAZXB", true);
                yield return ("(?<![^f]oo)(bar)", RegexOptions.None, "foobarX", true);
                yield return ("(?<![^f]oo)(bar)", RegexOptions.None, "boobarX", false);
                yield return ("(?<![^f])X", RegexOptions.None, "offX", true);
                yield return ("(?<![^f])X", RegexOptions.None, "onyX", false);
                yield return ("(?<=[^f])X", RegexOptions.None, "onyX", true);
                yield return ("(?<=[^f])X", RegexOptions.None, "offX", false);
                yield return ("(?<=C\\n)^", RegexOptions.Multiline, "A\nC\nC\n", true);
                yield return ("(?:(?(1)a|b)(X))+", RegexOptions.None, "bXaX", true);
                yield return ("(?:(?(1)\\1a|b)(X|Y))+", RegexOptions.None, "bXXaYYaY", true);
                yield return ("(?:(?(1)\\1a|b)(X|Y))+", RegexOptions.None, "bXYaXXaX", true);
                yield return ("()()()()()()()()()(?:(?(10)\\10a|b)(X|Y))+", RegexOptions.None, "bXXaYYaY", true);
                yield return ("(?=(\\w+))\\1:", RegexOptions.None, "abcd:", true);
                yield return ("^(?=(\\w+))\\1:", RegexOptions.None, "abcd:", true);
                yield return ("(a){0,3}(?(1)b|(c|))*D", RegexOptions.None, "abbD", true);
                yield return ("(a){0,3}(?(1)b|(c|))*D", RegexOptions.None, "ccccD", true);
                yield return ("(a){0,3}(?(1)b|(c|))*D", RegexOptions.None, "D", true);
                yield return ("(?>a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa4", true);
                yield return ("(?>a|)*\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", false);
                yield return ("^(?s)(?>.*)(?<!\\n)", RegexOptions.None, "abc", true);
                yield return ("^(?s)(?>.*)(?<!\\n)", RegexOptions.None, "abc\n", false);
                yield return ("^(?![^\\n]*\\n\\z)", RegexOptions.None, "abc", true);
                yield return ("^(?![^\\n]*\\n\\z)", RegexOptions.None, "abc\n", false);
                yield return ("\\z(?<!\\n)", RegexOptions.None, "abc", true);
                yield return ("\\z(?<!\\n)", RegexOptions.None, "abc\n", false);
                yield return ("( (A | (?(1)0|) )*   )", RegexOptions.IgnorePatternWhitespace, "abcd", true);
                yield return ("( ( (?(1)0|) )*   )", RegexOptions.IgnorePatternWhitespace, "abcd", true);
                yield return ("(  (?(1)0|)*   )", RegexOptions.IgnorePatternWhitespace, "abcd", true);
                yield return ("(?(?=.*b)b|^)", RegexOptions.None, "adc", true);
                yield return ("(?(?=.*b)b|^)", RegexOptions.None, "abc", true);
                yield return ("(?(?=^.*b)b|^)", RegexOptions.None, "adc", true);
                yield return ("(?(?=^.*b)b|^)", RegexOptions.None, "abc", false);
                yield return ("(?(?=.*b)b|^)*", RegexOptions.None, "adc", true);
                yield return ("(?(?=.*b)b|^)*", RegexOptions.None, "abc", true);
                yield return ("(?(?=.*b)b|^)+", RegexOptions.None, "adc", true);
                yield return ("(?(?=.*b)b|^)+", RegexOptions.None, "abc", true);
                yield return ("(?(?=b).*b|^d)", RegexOptions.None, "abc", true);
                yield return ("(?(?=.*b).*b|^d)", RegexOptions.None, "abc", true);
                yield return ("^%((?(?=[a])[^%])|b)*%$", RegexOptions.None, "%ab%", true);
                yield return ("(abc)\\1", RegexOptions.IgnoreCase, "abc", false);
                yield return ("(abc)\\1", RegexOptions.None, "abc", false);
                yield return ("(?>(a))b|(a)c", RegexOptions.None, "ac", true);
                yield return ("(?=(a))ab|(a)c", RegexOptions.None, "ac", true);
                yield return ("((?>(a))b|(a)c)", RegexOptions.None, "ac", true);
                yield return ("(?=(?>(a))b|(a)c)(..)", RegexOptions.None, "ac", true);
                yield return ("(?>(?>(a))b|(a)c)", RegexOptions.None, "ac", true);
                yield return ("(?:(?>([ab])))+a=", RegexOptions.None, "=ba=", true);
                yield return ("(?>([ab]))+a=", RegexOptions.None, "=ba=", true);
                yield return ("((?>(a+)b)+(aabab))", RegexOptions.None, "aaaabaaabaabab", true);
                yield return ("(?>a+|ab)+?c", RegexOptions.None, "aabc", false);
                yield return ("(?>a+|ab)+c", RegexOptions.None, "aabc", false);
                yield return ("(?(?=(a))a)", RegexOptions.None, "a", true);
                yield return ("(?(?=(a))a)(b)", RegexOptions.None, "ab", true);
                yield return ("(?=abc){3}abc", RegexOptions.None, "abcabcabc", true);
                yield return ("(?=abc){3}abc", RegexOptions.None, "xyz", false);
                yield return ("(?=abc)+abc", RegexOptions.None, "abcabcabc", true);
                yield return ("(?=abc)+abc", RegexOptions.None, "xyz", false);
                yield return ("(?=abc){1}xyz", RegexOptions.None, "xyz", false);
                yield return ("(?=(a))?.", RegexOptions.None, "ab", true);
                yield return ("(?=(a))?.", RegexOptions.None, "bc", true);
                yield return ("(?=(a))??.", RegexOptions.None, "ab", true);
                yield return ("(?=(a))??.", RegexOptions.None, "bc", true);
                yield return ("(?<=(abc))?xyz", RegexOptions.None, "abcxyz", true);
                yield return ("(?<=(abc))?xyz", RegexOptions.None, "pqrxyz", true);
                yield return ("(?<=a{2})b", RegexOptions.IgnoreCase, "xaabc", true);
                yield return ("(?<=a{2})b", RegexOptions.IgnoreCase, "xabc", false);
                yield return ("(?<!a{2})b", RegexOptions.IgnoreCase, "xabc", true);
                yield return ("(?<!a{2})b", RegexOptions.IgnoreCase, "xaabc", false);
                yield return ("(?<=[^a]{2})b", RegexOptions.None, "axxbc", true);
                yield return ("(?<=[^a]{2})b", RegexOptions.None, "aAAbc", true);
                yield return ("(?<=[^a]{2})b", RegexOptions.None, "xaabc", false);
                yield return ("(?<=[^a]{2})b", RegexOptions.IgnoreCase, "axxbc", true);
                yield return ("(?<=[^a]{2})b", RegexOptions.IgnoreCase, "aAAbc", false);
                yield return ("(?<=[^a]{2})b", RegexOptions.IgnoreCase, "xaabc", false);
                yield return ("(?=C)", RegexOptions.None, "ABCDECBA", true);
                yield return ("(?'abc'\\w+):\\k<abc>{2}", RegexOptions.None, "a:aaxyz", true);
                yield return ("(?'abc'\\w+):\\k<abc>{2}", RegexOptions.None, "ab:ababxyz", true);
                yield return ("(?'abc'\\w+):\\k<abc>{2}", RegexOptions.None, "a:axyz", false);
                yield return ("(?'abc'\\w+):\\k<abc>{2}", RegexOptions.None, "ab:abxyz", false);
                yield return ("^(?<ab>a)? (?(<ab>)b|c) (?('ab')d|e)", RegexOptions.IgnorePatternWhitespace, "ce", true);
                yield return ("(another)?(\\1?)test", RegexOptions.None, "hello world test", true);
                yield return ("(another)?(\\1+)test", RegexOptions.None, "hello world test", false);
                yield return ("((?>a?)*)*c", RegexOptions.None, "aac", true);
                yield return ("(?>.*?a)(?<=ba)", RegexOptions.None, "aba", true);
                yield return ("(?:.*?a)(?<=ba)", RegexOptions.None, "aba", true);
                yield return ("(?>.*?a)b", RegexOptions.Singleline, "aab", true);
                yield return ("(?>.*?a)b", RegexOptions.None, "aab", true);
                yield return ("(?>^a)b", RegexOptions.Singleline, "aab", false);
                yield return ("(?>.*?)(?<=(abcd)|(wxyz))", RegexOptions.None, "alphabetabcd", true);
                yield return ("(?>.*?)(?<=(abcd)|(wxyz))", RegexOptions.None, "endingwxyz", true);
                yield return ("(?>.*)(?<=(abcd)|(wxyz))", RegexOptions.None, "alphabetabcd", true);
                yield return ("(?>.*)(?<=(abcd)|(wxyz))", RegexOptions.None, "endingwxyz", true);
                yield return ("(?>.*)foo", RegexOptions.None, "abcdfooxyz", false);
                yield return ("(?>.*?)foo", RegexOptions.None, "abcdfooxyz", true);
                yield return ("a(?=bc).|abd", RegexOptions.None, "abd", true);
                yield return ("a(?=bc).|abd", RegexOptions.None, "abc", true);
                yield return ("a(?>bc)d|abd", RegexOptions.None, "abceabd", true);
                yield return ("^(a)?(?(1)a|b)+$", RegexOptions.None, "a", false);
                yield return ("(?:(?<n>foo)|(?<n>bar))\\k<n>", RegexOptions.None, "foofoo", true);
                yield return ("(?:(?<n>foo)|(?<n>bar))\\k<n>", RegexOptions.None, "barbar", true);
                yield return (" ^ (?> a + ) b $ ", RegexOptions.IgnorePatternWhitespace, "aaaab", true);
                yield return ("^\\w+(?>\\s*)(?<=\\w)", RegexOptions.None, "test test", true);
                yield return ("A((((((((a))))))))\\8B", RegexOptions.None, "AaaB", true);
                yield return ("A(((((((((a)))))))))\\9B", RegexOptions.None, "AaaB", true);
                yield return ("(\\2|a)(\\1)", RegexOptions.None, "aaa", true);
                yield return ("((((((((((((x))))))))))))\\12", RegexOptions.None, "xx", true);
                yield return ("(?<=a(B){0}c)X", RegexOptions.None, "acX", true);
                yield return ("(?=.*[A-Z])(?=.*[a-z])(?=.*[0-9])(?=.*[,;:])(?=.{8,16})(?!.*[\\s])", RegexOptions.None, "   Fred:099", true);
                yield return ("(?=.*X)X$", RegexOptions.None, "  X", true);
                yield return ("(?s)(?=.*?)b", RegexOptions.None, "aabc", true);
                yield return ("(Z)(a)\\2{1,2}?(?-i)\\1X", RegexOptions.IgnoreCase, "ZaAAZX", true);
                yield return ("^(?: \n(?: A| (1? (?=(?<cond>2)?) (1) 2 (?('cond')|3)) )\n(Z)\n)+$", RegexOptions.IgnorePatternWhitespace, "AZ123Z", true);
                yield return ("^(?: \n(?: A| (1? (?=(?<cond>2)?) (1) 2 (?('cond')|3)) )\n(Z)\n)+$", RegexOptions.IgnorePatternWhitespace, "AZ12Z", false);
                yield return ("(?>ab|abab){1,5}?M", RegexOptions.None, "abababababababababababM", true);
                yield return ("(?>ab|abab){2}?M", RegexOptions.None, "abababM", true);
                yield return ("((?(?=(a))a)+k)", RegexOptions.None, "bbak", true);
                yield return ("((?(?=(a))a|)+k)", RegexOptions.None, "bbak", true);
                yield return ("(?(?!(b))a|b)+k", RegexOptions.None, "ababbalbbadabak", true);
                yield return ("(?!(b))c|b", RegexOptions.None, "Ab", true);
                yield return ("(?!(b))c|b", RegexOptions.None, "Ac", true);
                yield return ("(?=(b))b|c", RegexOptions.None, "Ab", true);
                yield return ("(?=(b))b|c", RegexOptions.None, "Ac", true);
                yield return ("^(?(?=(a))abc|def)", RegexOptions.None, "abc", true);
                yield return ("^(?(?!(a))def|abc)", RegexOptions.None, "abc", true);
                yield return ("(?<=\\G.)", RegexOptions.None, "abc", true);
                yield return ("(?:a|(?=b)|.)*\\z", RegexOptions.None, "abc", true);
                yield return ("(?:a|(?=b)|.)*", RegexOptions.None, "abc", true);
                yield return ("(?<=(?=.(?<=x)))", RegexOptions.None, "abx", true);
                yield return ("(?<=(?=(?<=a)))b", RegexOptions.None, "ab", true);
                yield return ("^(a\\1?){4}$", RegexOptions.None, "aaaa", true);
                yield return ("^((\\1+)|\\d)+133X$", RegexOptions.None, "111133X", true);
                yield return ("^(?=.*(?=(([A-Z]).*(?(1)\\1)))(?!.+\\2)){26}", RegexOptions.IgnoreCase, "The quick brown fox jumps over the lazy dog.", true);
                yield return ("^(?=.*(?=(([A-Z]).*(?(1)\\1)))(?!.+\\2)){26}", RegexOptions.IgnoreCase, "Jackdaws love my big sphinx of quartz.", true);
                yield return ("^(?=.*(?=(([A-Z]).*(?(1)\\1)))(?!.+\\2)){26}", RegexOptions.IgnoreCase, "Pack my box with five dozen liquor jugs.", true);
                yield return ("^(?=.*(?=(([A-Z]).*(?(1)\\1)))(?!.+\\2)){26}", RegexOptions.IgnoreCase, "The quick brown fox jumps over the lazy cat.", false);
                yield return ("^(?=.*(?=(([A-Z]).*(?(1)\\1)))(?!.+\\2)){26}", RegexOptions.IgnoreCase, "Hackdaws love my big sphinx of quartz.", false);
                yield return ("^(?=.*(?=(([A-Z]).*(?(1)\\1)))(?!.+\\2)){26}", RegexOptions.IgnoreCase, "Pack my fox with five dozen liquor jugs.", false);
                yield return ("^(?>.*?([A-Z])(?!.*\\1)){26}", RegexOptions.IgnoreCase, "The quick brown fox jumps over the lazy dog.", true);
                yield return ("^(?>.*?([A-Z])(?!.*\\1)){26}", RegexOptions.IgnoreCase, "Jackdaws love my big sphinx of quartz.", true);
                yield return ("^(?>.*?([A-Z])(?!.*\\1)){26}", RegexOptions.IgnoreCase, "Pack my box with five dozen liquor jugs.", true);
                yield return ("^(?>.*?([A-Z])(?!.*\\1)){26}", RegexOptions.IgnoreCase, "The quick brown fox jumps over the lazy cat.", false);
                yield return ("^(?>.*?([A-Z])(?!.*\\1)){26}", RegexOptions.IgnoreCase, "Hackdaws love my big sphinx of quartz.", false);
                yield return ("^(?>.*?([A-Z])(?!.*\\1)){26}", RegexOptions.IgnoreCase, "Pack my fox with five dozen liquor jugs.", false);
                yield return ("(?<=X(?(DEFINE)(A))).", RegexOptions.None, "AXYZ", true);
                yield return ("(?<=X(?(DEFINE)(.*))Y).", RegexOptions.None, "AXYZ", true);
                if (PlatformDetection.IsNotNetFramework)
                {
                    yield return ("(?(DEFINE)(?<foo>bar))(?<![-a-z0-9])word", RegexOptions.None, "word", true); // This was fixed in .NET 7
                }
                yield return ("^(xa|=?\\1a){2}$", RegexOptions.None, "xa=xaa", true);
                yield return ("^(xa|=?\\1a){2}$", RegexOptions.None, "xa=xaaa", false);
                yield return ("^(xa|=?\\1a)+$", RegexOptions.None, "xa=xaa", true);
                yield return ("^(xa|=?\\1a)+$", RegexOptions.None, "xa=xaaa", false);
                yield return ("^(?:(?(ZZ)a|b)(?<ZZ>X))+", RegexOptions.None, "bXaX", true);
                yield return ("^(?:(?(ZZ)a|b)(?<ZZ>X))+", RegexOptions.None, "bXbX", true);
                yield return ("^(?:(?(ZZ)a|b)(?<ZZ>X))+", RegexOptions.None, "aXaX", false);
                yield return ("^(?:(?(ZZ)a|b)(?<ZZ>X))+", RegexOptions.None, "aXbX", false);
                yield return ("^((?(abc)a|b)(?<abc>x|y))+", RegexOptions.None, "bxay", true);
                yield return ("^((?(abc)a|b)(?<abc>x|y))+", RegexOptions.None, "bxby", true);
                yield return ("^((?(abc)a|b)(?<abc>x|y))+", RegexOptions.None, "axby", false);
                yield return ("^(?:(?:\\1|X)(a|b))+", RegexOptions.None, "Xaaa", true);
                yield return ("^(?:(?:\\1|X)(a|b))+", RegexOptions.None, "Xaba", true);
                yield return ("(?'abc'a|b)(?<abc>d|e)\\k<abc>{2}", RegexOptions.None, "adbb", false);
                yield return ("^(abc)?(?(-1)X|Y)", RegexOptions.None, "Y", true);
                yield return ("(\\3)(\\1)(a)", RegexOptions.None, "cat", false);
                yield return ("^\"((?(?=[a])[^\"])|b)*\"$", RegexOptions.None, "\"ab\"", true);
                yield return ("(Y)X\\1{2,4}b", RegexOptions.None, "YX=ps", false);
                yield return ("(Y)X\\1{2,4}b", RegexOptions.None, "YXY=ps", false);
                yield return ("(Y)X\\1{2,4}b", RegexOptions.None, "YXYY=ps", false);
                yield return ("(Y)X\\1{2,4}b", RegexOptions.None, "YXYYY=ps", false);
                yield return ("(Y)X\\1{2,4}b", RegexOptions.None, "YXYYYY=ps", false);
                yield return ("(Y)X\\1{2,4}?b", RegexOptions.None, "YX=ps", false);
                yield return ("(Y)X\\1{2,4}?b", RegexOptions.None, "YXY=ps", false);
                yield return ("(Y)X\\1{2,4}?b", RegexOptions.None, "YXYY=ps", false);
                yield return ("(Y)X\\1{2,4}?b", RegexOptions.None, "YXYYY=ps", false);
                yield return ("(Y)X\\1{2,4}?b", RegexOptions.None, "YXYYYY=ps", false);
                yield return ("(?<=abc)123", RegexOptions.None, "xyzabc123pqr", true);
                yield return ("(?<=abc)123", RegexOptions.None, "xyzabc12=ps", false);
                yield return ("(?<=abc)123", RegexOptions.None, "xyzabc12=ph", false);
                yield return ("(a)bc\\1*", RegexOptions.None, "xxxxabca=ps", true);
                yield return ("^(ab(c\\1)d|x){2}$", RegexOptions.None, "xabcxd", true);
                yield return ("(?<=abc)def", RegexOptions.None, "abc=ph", false);
                yield return ("((?>\\r\\n|\\n|\\x0b|\\f|\\r|\\x85)*)(.)", RegexOptions.Singleline, "\r\n", true);
                yield return ("((?>\\r\\n|\\n|\\x0b|\\f|\\r|\\x85)*)(.)", RegexOptions.Singleline, "\r\r\n\n\r", true);
                yield return ("((?>\\r\\n|\\n|\\x0b|\\f|\\r|\\x85)*)(.)", RegexOptions.Singleline, "\r\r\n\n\r\n", true);
                yield return ("^.*?(?(?=a)a|bc)", RegexOptions.None, "ba", true);
                yield return ("(?:(?>(a)))+a%", RegexOptions.None, "%aa%", true);
                yield return ("(a)bc|(a)(b)\\2", RegexOptions.None, "abc=ovector=2", true);
                yield return ("(a)bc|(a)(b)\\2", RegexOptions.None, "aba=ovector=1", true);
                yield return ("(a)bc|(a)(b)\\2", RegexOptions.None, "aba=ovector=3", true);
                yield return ("^(?>a+)(?>(z+))\\w", RegexOptions.None, "aaaazzzzb", true);
                yield return ("^(?>a+)(?>(z+))\\w", RegexOptions.None, "aazz", false);
                yield return ("(..)\\1", RegexOptions.None, "ab=ps", false);
                yield return ("(..)\\1", RegexOptions.None, "aba=ps", false);
                yield return ("(..)\\1", RegexOptions.IgnoreCase, "ab=ps", false);
                yield return ("(..)\\1", RegexOptions.IgnoreCase, "abA=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.None, "ab=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.None, "aba=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.None, "abab=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.None, "ababa=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.IgnoreCase, "ab=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.IgnoreCase, "aBa=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.IgnoreCase, "aBAb=ps", false);
                yield return ("(..)\\1{2,}", RegexOptions.IgnoreCase, "AbaBA=ps", false);
                yield return ("(..)\\1{2,}?x", RegexOptions.IgnoreCase, "ab=ps", false);
                yield return ("(..)\\1{2,}?x", RegexOptions.IgnoreCase, "abA=ps", false);
                yield return ("(..)\\1{2,}?x", RegexOptions.IgnoreCase, "aBAb=ps", false);
                yield return ("(..)\\1{2,}?x", RegexOptions.IgnoreCase, "abaBA=ps", false);
                yield return ("(..)\\1{2,}?x", RegexOptions.IgnoreCase, "abAbaB=ps", false);
                yield return ("(..)\\1{2,}?x", RegexOptions.IgnoreCase, "abaBabA=ps", false);
                yield return ("^(..)\\1", RegexOptions.None, "aba=ps", false);
                yield return ("^(..)\\1{2,3}x", RegexOptions.None, "aba=ps", false);
                yield return ("^(..)\\1{2,3}x", RegexOptions.None, "ababa=ps", false);
                yield return ("^(..)\\1{2,3}x", RegexOptions.None, "ababa=ph", false);
                yield return ("^(..)\\1{2,3}?x", RegexOptions.None, "aba=ps", false);
                yield return ("^(..)\\1{2,3}?x", RegexOptions.None, "ababa=ps", false);
                yield return ("^(..)\\1{2,3}?x", RegexOptions.None, "ababa=ph", false);
                yield return ("^(..)(\\1{2,3})ab", RegexOptions.None, "abababab", true);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)$", RegexOptions.None, "abc:", true);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)$", RegexOptions.None, "12", true);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)$", RegexOptions.None, "123", false);
                yield return ("^(?(?=abc)\\w{3}:|\\d\\d)$", RegexOptions.None, "xyz", false);
                yield return ("(?(?=ab)ab)", RegexOptions.None, "abxxx", true);
                yield return ("(?(?=ab)ab)", RegexOptions.None, "ca", true);
                if (PlatformDetection.IsNotNetFramework)
                {
                    yield return ("(?(?=ab)ab)", RegexOptions.None, "cd", true); // This was fixed in .NET 7
                }
                yield return ("(?(VERSION>=10.04)yes|no)", RegexOptions.None, "yesno", true);
                yield return ("\\k<A>*(?<A>aa)(?<A>bb)", RegexOptions.None, "aabb", true);
                yield return ("((?=(?(?=(?(?=(?(?=()))))))))", RegexOptions.None, "a", true);
                yield return ("(?(?=)==)(((((((((?=)))))))))", RegexOptions.None, "a", false);
                yield return ("^(?:(a)|b)(?(1)A|B)", RegexOptions.None, "aA123=ovector=2", true);
                yield return ("^(?:(?<AA>a)|b)(?(<AA>)A|B)", RegexOptions.None, "aA123=ovector=1", false);
                yield return ("^(?<AA>)(?:(?<AA>a)|b)(?(<AA>)A|B)", RegexOptions.None, "aA123=ovector=1", false);
                yield return ("^(?<AA>)(?:(?<AA>a)|b)(?(<AA>)A|B)", RegexOptions.None, "aA123=ovector=3", false);
                yield return ("(?(?=0)(?=00)?00765)", RegexOptions.None, "00765", true);
                yield return ("(?(?=0)(?=00)?00765|(?!3).56)", RegexOptions.None, "00765", true);
                yield return ("(?(?=0)(?=00)?00765|(?!3).56)", RegexOptions.None, "456", true);
                yield return ("(?(?=0)(?=00)?00765|(?!3).56)", RegexOptions.None, "356", false);
                yield return ("(?(?!)a|b)", RegexOptions.None, "bbb", true);
                yield return ("(?(?!)a|b)", RegexOptions.None, "aaa", false);
                yield return ("(?<=abc)", RegexOptions.IgnoreCase | RegexOptions.Multiline | RegexOptions.Singleline, "1234abc=offset_limit=7", true);
                yield return ("([ab])...(?<=\\1)z", RegexOptions.None, "a11az", true);
                yield return ("([ab])...(?<=\\1)z", RegexOptions.None, "b11bz", true);
                yield return ("([ab])...(?<=\\1)z", RegexOptions.None, "b11az", false);
                yield return ("([ab])(\\1)...(?<=\\2)z", RegexOptions.None, "aa11az", true);
                yield return ("(?<A>[ab])...(?<=\\k'A')z", RegexOptions.None, "a11az", true);
                yield return ("(?<A>[ab])...(?<=\\k'A')z", RegexOptions.None, "b11bz", true);
                yield return ("(?<A>[ab])...(?<=\\k'A')z", RegexOptions.None, "b11az", false);
                yield return ("(?<=(?=.(?<=x)))", RegexOptions.None, "ab=ph", false);
                yield return ("(?<=(?=.(?<=x)))", RegexOptions.None, "xyz", true);
                yield return ("(?<=aXb)cd", RegexOptions.None, "aXbcd", true);
                yield return ("(?<=(.))X", RegexOptions.None, "WXYZ", true);
                yield return ("(?<=(.))X", RegexOptions.None, "\u0256XYZ", true);
                yield return ("(?<=(.))X", RegexOptions.None, "XYZ", false);
                yield return ("(?<=A\\p{Nd})XYZ", RegexOptions.None, "A2XYZ", true);
                yield return ("(?<=A\\p{Nd})XYZ", RegexOptions.None, "123A5XYZPQR", true);
                yield return ("(?<=A\\p{Nd})XYZ", RegexOptions.None, "ABA\u0660XYZpqr", true);
                yield return ("(?<=A\\p{Nd})XYZ", RegexOptions.None, "AXYZ", false);
                yield return ("(?<=A\\p{Nd})XYZ", RegexOptions.None, "XYZ", false);
                yield return ("(A)\\1", RegexOptions.IgnoreCase, "AA", true);
                yield return ("(A)\\1", RegexOptions.IgnoreCase, "Aa", true);
                yield return ("(A)\\1", RegexOptions.IgnoreCase, "aa", true);
                yield return ("(A)\\1", RegexOptions.IgnoreCase, "aA", true);
                yield return ("(\u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3) \\1", RegexOptions.IgnoreCase, "\u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3 \u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3", true);
                yield return ("(\u00CF\u0083\u00CE\u00AC\u00CE\u00BC\u00CE\u00BF\u00CF\u0082) \\1", RegexOptions.IgnoreCase, "\u00CF\u0083\u00CE\u00AC\u00CE\u00BC\u00CE\u00BF\u00CF\u0082 \u00CF\u0083\u00CE\u00AC\u00CE\u00BC\u00CE\u00BF\u00CF\u0082", true);
                yield return ("(\u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3) \\1*", RegexOptions.IgnoreCase, "\u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3 ", true);
                yield return ("(\u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3) \\1*", RegexOptions.IgnoreCase, "\u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3 \u00CE\u00A3\u00CE\u0086\u00CE\u009c\u00CE\u009f\u00CE\u00A3\u00CF\u0083\u00CE\u00AC\u00CE\u00BC\u00CE\u00BF\u00CF\u0082\u00CF\u0083\u00CE\u00AC\u00CE\u00BC\u00CE\u00BF\u00CF\u0082", true);
                yield return ("(?<!^)ETA", RegexOptions.None, "ETA", false);
                yield return ("(\u00C8\u00BA\u00C8\u00BA\u00C8\u00BA)?\\1", RegexOptions.IgnoreCase, "\u00C8\u00BA\u00C8\u00BA\u00C8\u00BA\u00E2\u00B1\u00A5\u00E2\u00B1\u00A5", false);
                yield return ("(\u00C8\u00BA\u00C8\u00BA\u00C8\u00BA)\\1", RegexOptions.IgnoreCase, "\u00C8\u00BA\u00C8\u00BA\u00C8\u00BA\u00E2\u00B1\u00A5\u00E2\u00B1\u00A5", false);
                yield return ("(\\xc1)\\1", RegexOptions.IgnoreCase, "\u00C1\u00E1=no_jit", true);
                yield return ("\\Gabc", RegexOptions.None, "abcdef", true);
                yield return ("\\Gabc", RegexOptions.None, "xyzabc=offset=2", false);
                yield return ("^(?>a*)\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa9876", true);
                yield return ("^(?>a*)\\d", RegexOptions.None, "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa", false);
                yield return ("^(?=abc)\\w{5}:$", RegexOptions.None, "abcde:", true);
                yield return ("^(?=abc)\\w{5}:$", RegexOptions.None, "abc..", false);
                yield return ("^(?=abc)\\w{5}:$", RegexOptions.None, "123", false);
                yield return ("^(?=abc)\\w{5}:$", RegexOptions.None, "vwxyz", false);
                yield return ("^(?!abc)\\d\\d$", RegexOptions.None, "12", true);
                yield return ("^(?!abc)\\d\\d$", RegexOptions.None, "abcde:", false);
                yield return ("^(?!abc)\\d\\d$", RegexOptions.None, "abc..", false);
                yield return ("^(?!abc)\\d\\d$", RegexOptions.None, "123", false);
                yield return ("^(?!abc)\\d\\d$", RegexOptions.None, "vwxyz", false);
                yield return ("(?<=abc|xy)123", RegexOptions.None, "abc12345", true);
                yield return ("(?<=abc|xy)123", RegexOptions.None, "wxy123z", true);
                yield return ("(?<=abc|xy)123", RegexOptions.None, "123abc", false);
                yield return ("(?<!abc|xy)123", RegexOptions.None, "123abc", true);
                yield return ("(?<!abc|xy)123", RegexOptions.None, "mno123456", true);
                yield return ("(?<!abc|xy)123", RegexOptions.None, "abc12345", false);
                yield return ("(?<!abc|xy)123", RegexOptions.None, "wxy123z", false);
                yield return ("\\Gabc", RegexOptions.None, "defabcxyz", false);
                yield return ("(?<=foo)bar", RegexOptions.None, "foob=ps,offset=2,allusedtext", false);
                yield return ("(?>.*/)foo", RegexOptions.None, "/this/is/a/very/long/line/in/deed/with/very/many/slashes/in/it/you/see/", false);
                yield return ("(?>.*/)foo", RegexOptions.None, "/this/is/a/very/long/line/in/deed/with/very/many/slashes/in/and/foo", true);
                yield return ("^(?(2)a|(1)(2))+$", RegexOptions.None, "123a", false);
                yield return ("(?<=a|bbbb)c", RegexOptions.None, "ac", true);
                yield return ("(?<=a|bbbb)c", RegexOptions.None, "bbbbc", true);
                yield return ("^\"((?(?=[a])[^\"])|b)*\"$", RegexOptions.None, "\"ab\"=callout_none", false);
                yield return ("(?<=abc)123", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "xyzabc123pqr", true);
                yield return ("(?<=abc)123", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "xyzabc12=ps", false);
                yield return ("(?<=abc)123", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "xyzabc12=ph", false);
                yield return ("(?<=abc)def", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "abc=ph", false);
                yield return ("^(?>.{4})abc|^\\w\\w.xabcd", RegexOptions.None, "xxxxabcd", true);
                yield return ("^(?>.{4})abc|^\\w\\w.xabcd", RegexOptions.None, "xx\u00A0xabcd", true);
                yield return ("abc(?=xyz)", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "abcxyzpqr", true);
                yield return ("(?<=pqr)abc(?=xyz)", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "xyzpqrabcxyzpqr", true);
                yield return ("abc(?=abcde)(?=ab)", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "abcabcdefg", true);
                yield return ("(?<=abc|)", RegexOptions.None, "abcde=aftertext", true);
                yield return ("(?<=|abc)", RegexOptions.None, "abcde=aftertext", true);
                yield return ("(?(VERSION>=0)^B0W)", RegexOptions.None, "B0W-W0W", true);
                yield return ("(?(VERSION>=1000)^B0W|W0W)", RegexOptions.None, "B0W-W0W", true);
                yield return ("(?(VERSION>=1000)^B0W|W0W)", RegexOptions.None, "0", false);
                yield return ("(?<=pqr)abc(?=xyz)", RegexOptions.None, "123pqrabcxy=ps,allusedtext", false);
                yield return ("(?>a+b)", RegexOptions.None, "aaaa=ps", false);
                yield return ("(?(?=abc).*|Z)", RegexOptions.None, "ab=ps", false);
                yield return ("(?<=x)badutf", RegexOptions.None, "X\u00DFabcd", false);
                yield return ("(?<=x)badutf", RegexOptions.None, "X\u00DFabcd=offset=1", false);
                yield return ("(?<=x)badutf", RegexOptions.None, "X\u00DFabcd=offset=2", false);
                yield return ("(?<=x)badutf", RegexOptions.None, "X\u00DFabcd\u00DF=offset=3", false);
                yield return ("(?<=x)badutf", RegexOptions.None, "X\u00DFabcd=offset=3", false);
                yield return ("(?<=xx)badutf", RegexOptions.None, "X\u00DFabcd", false);
                yield return ("(?<=xx)badutf", RegexOptions.None, "X\u00DFabcd=offset=1", false);
                yield return ("(?<=xx)badutf", RegexOptions.None, "X\u00DFabcd=offset=2", false);
                yield return ("(?<=xx)badutf", RegexOptions.None, "X\u00DFabcd=offset=3", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabcd", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabcd=offset=1", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabcd=offset=2", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabcd=offset=3", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabc\u00DF=offset=6", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabc\u00DF=offset=7", false);
                yield return ("(?<=xxxx)badutf", RegexOptions.None, "X\u00DFabcd=offset=6", false);
                yield return ("(?<=x)..", RegexOptions.IgnoreCase | RegexOptions.Multiline, "abcd\u0080wxzy\u0080pqrs", true);
                yield return ("(?<=x)..", RegexOptions.IgnoreCase | RegexOptions.Multiline, "abcd\u0080wxzy\u0080xpqrs", true);
                yield return ("(?<=..)X", RegexOptions.IgnoreCase | RegexOptions.Multiline, "AB\u0080AQXYZ", true);
                yield return ("(?<=..)X", RegexOptions.IgnoreCase | RegexOptions.Multiline, "AB\u0080AQXYZ=offset=5", true);
                yield return ("(?<=..)X", RegexOptions.IgnoreCase | RegexOptions.Multiline, "AB\u0080\u0080AXYZXC=offset=5", true);
                yield return ("(?<=...)X", RegexOptions.IgnoreCase | RegexOptions.Multiline, "AAA\u0080BBBXYZ", true);
                yield return ("abc(?<=bc)def", RegexOptions.None, "xxxabcd=ph", false);
                yield return ("(?<=ab)cdef", RegexOptions.None, "xxabcd=ph", false);
                yield return ("(|]+){2,2452}", RegexOptions.None, "(|]+){2,2452}", true);
                yield return ("b(?<!ax)(?!cx)", RegexOptions.Singleline | RegexOptions.IgnorePatternWhitespace, "abc", true);
                yield return ("^(?>(a+))(?>b+)(?>(c+))(?>d+)(?>(e+))", RegexOptions.IgnoreCase | RegexOptions.Singleline, "aabbccddee=jitstack=1", true);
                yield return ("(abc\\1)", RegexOptions.None, "abc", false);
            }
        }

        [Theory]
        [MemberData(nameof(PcreTestData))]
        public void IsMatchTests(Regex regex, string input, bool expectSuccess)
            => Assert.Equal(expectSuccess, regex.IsMatch(input));
    }
}
