/*
  libuta - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _RECT_H
#define _RECT_H

#include <SDL/SDL.h>

#include "point.h"

namespace uta {

///
class Rect 
{
 public:
  
  ///
  Rect();
  ///
  Rect( int x, int y, int w, int h);
/*   /// */
/*   Rect( const SDL_Rect &); */
  ///
  Rect( const Point& upperLeft, const Point& lowerRight);
  ///
  virtual ~Rect();

  ///
  static Rect invalid;
  ///
  static Rect empty;
/*   /// */
/*   operator SDL_Rect*(void) const ; */
  ///
  bool isEmpty() const;
  ///
  bool isValid() const;
  ///
  bool contains( const Point &p ) const;
  ///
  bool contains( const Rect &r )  const;

  ///move the rectangle by (dx, dy)
  virtual void translate( int dx, int dy );
  ///move the rectangle, specifying new upperleft corner
  virtual void warp(const Point &newUpperLeft);
  ///this keeps the upper left corner and moves the lower right one
  virtual void resize(int w, int h);
  ///
  Rect unite( const Rect &r )    const;
  ///
  Rect intersect( const Rect &r) const;  

  ///
  const Point& upperLeft() const { return upperLeft_; };
  ///
  const Point& lowerRight() const { return lowerRight_; };

  ///
  int width() const { return lowerRight_.x - upperLeft_.x ; };
  ///
  int height() const { return lowerRight_.y - upperLeft_.y ; };

  ///
  friend bool operator==(const Rect &p, const Rect &q)
    { return ((p.upperLeft() == q.upperLeft()) && 
	      (p.lowerRight() == q.lowerRight())); };
  
  ///
  friend bool operator!=(const Rect &p, const Rect &q)
    { return ((p.upperLeft() != q.upperLeft()) || 
	      (p.lowerRight() != q.lowerRight())); };
  
  ///comparism by size of area (width * height)
  friend bool operator<(const Rect &p, const Rect &q)
    { return (p.width()*p.height()) < (q.width()*q.height()); };
  
 
  ///
  friend std::ostream& operator<<(std::ostream& s, const Rect& p);       
 
 protected:

  ///
  Point upperLeft_;
  ///
  Point lowerRight_;

};


}

#endif
