module Addressable
  # A pure Ruby implementation of IDNA.  C was eschewed for the sake of JRuby,
  # and because performance is largely irrelevant here.
  module IDNA
    # This module is loosely based on idn_actionmailer by Mick Staugaard,
    # the unicode library by Yoshida Masato, and the punycode implementation
    # by Kazuhiro Nishiyama.  Most of the code was copied verbatim, but
    # some reformatting was done, and some translation from C was done.
    #
    # Without their code to work from as a base, we'd all still be relying
    # on the presence of libidn.  Which nobody ever seems to have installed.
    #
    # Original sources:
    # http://github.com/staugaard/idn_actionmailer
    # http://www.yoshidam.net/Ruby.html#unicode
    # http://rubyforge.org/frs/?group_id=2550

    # :stopdoc:

    ACE_PREFIX = "xn--"

    UTF8_REGEX = /\A(?:
      [\x09\x0A\x0D\x20-\x7E]               # ASCII
      | [\xC2-\xDF][\x80-\xBF]              # non-overlong 2-byte
      | \xE0[\xA0-\xBF][\x80-\xBF]          # excluding overlongs
      | [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2}   # straight 3-byte
      | \xED[\x80-\x9F][\x80-\xBF]          # excluding surrogates
      | \xF0[\x90-\xBF][\x80-\xBF]{2}       # planes 1-3
      | [\xF1-\xF3][\x80-\xBF]{3}           # planes 4nil5
      | \xF4[\x80-\x8F][\x80-\xBF]{2}       # plane 16
      )*\z/mnx

    UTF8_REGEX_MULTIBYTE = /(?:
      [\xC2-\xDF][\x80-\xBF]                # non-overlong 2-byte
      | \xE0[\xA0-\xBF][\x80-\xBF]          # excluding overlongs
      | [\xE1-\xEC\xEE\xEF][\x80-\xBF]{2}   # straight 3-byte
      | \xED[\x80-\x9F][\x80-\xBF]          # excluding surrogates
      | \xF0[\x90-\xBF][\x80-\xBF]{2}       # planes 1-3
      | [\xF1-\xF3][\x80-\xBF]{3}           # planes 4nil5
      | \xF4[\x80-\x8F][\x80-\xBF]{2}       # plane 16
      )/mnx

    # :startdoc:

    # Converts from a Unicode internationalized domain name to an ASCII
    # domain name as described in RFC 3490.
    def self.to_ascii(input)
      input = input.dup
      if input.respond_to?(:force_encoding)
        input.force_encoding(Encoding::ASCII_8BIT)
      end
      if input =~ UTF8_REGEX && input =~ UTF8_REGEX_MULTIBYTE
        parts = unicode_downcase(input).split('.')
        parts.map! do |part|
          if part.respond_to?(:force_encoding)
            part.force_encoding(Encoding::ASCII_8BIT)
          end
          if part =~ UTF8_REGEX && part =~ UTF8_REGEX_MULTIBYTE
            ACE_PREFIX + punycode_encode(unicode_normalize_kc(part))
          else
            part
          end
        end
        parts.join('.')
      else
        input
      end
    end

    # Converts from an ASCII domain name to a Unicode internationalized
    # domain name as described in RFC 3490.
    def self.to_unicode(input)
      parts = input.split('.')
      parts.map! do |part|
        if part =~ /^#{ACE_PREFIX}/
          punycode_decode(part[/^#{ACE_PREFIX}(.+)/, 1])
        else
          part
        end
      end
      output = parts.join('.')
      if output.respond_to?(:force_encoding)
        output.force_encoding(Encoding::UTF_8)
      end
      output
    end

    # Unicode normalization form KC.
    def self.unicode_normalize_kc(input)
      unpacked = input.unpack("U*")
      unpacked =
        unicode_compose(unicode_sort_canonical(unicode_decompose(unpacked)))
      return unpacked.pack("U*")
    end

    ##
    # Unicode aware downcase method.
    #
    # @api private
    # @param [String] input
    #   The input string.
    # @return [String] The downcased result.
    def self.unicode_downcase(input)
      unpacked = input.unpack("U*")
      unpacked.map! { |codepoint| lookup_unicode_lowercase(codepoint) }
      return unpacked.pack("U*")
    end
    (class <<self; private :unicode_downcase; end)

    def self.unicode_compose(unpacked)
      unpacked_result = []
      length = unpacked.length

      return unpacked if length == 0

      starter = unpacked[0]
      starter_cc = lookup_unicode_combining_class(starter)
      starter_cc = 256 if starter_cc != 0
      for i in 1...length
        ch = unpacked[i]
        cc = lookup_unicode_combining_class(ch)

        if (starter_cc == 0 &&
            (composite = unicode_compose_pair(starter, ch)) != nil)
          starter = composite
          startercc = lookup_unicode_combining_class(composite)
        else
          unpacked_result << starter
          starter = ch
          startercc = cc
        end
      end
      unpacked_result << starter
      return unpacked_result
    end
    (class <<self; private :unicode_compose; end)

    def self.unicode_compose_pair(ch_one, ch_two)
      if ch_one >= HANGUL_LBASE && ch_one < HANGUL_LBASE + HANGUL_LCOUNT &&
          ch_two >= HANGUL_VBASE && ch_two < HANGUL_VBASE + HANGUL_VCOUNT
        # Hangul L + V
        return HANGUL_SBASE + (
          (ch_one - HANGUL_LBASE) * HANGUL_VCOUNT + (ch_two - HANGUL_VBASE)
        ) * HANGUL_TCOUNT
      elsif ch_one >= HANGUL_SBASE &&
          ch_one < HANGUL_SBASE + HANGUL_SCOUNT &&
          (ch_one - HANGUL_SBASE) % HANGUL_TCOUNT == 0 &&
          ch_two >= HANGUL_TBASE && ch_two < HANGUL_TBASE + HANGUL_TCOUNT
           # Hangul LV + T
        return ch_one + (ch_two - HANGUL_TBASE)
      end

      p = []
      ucs4_to_utf8 = lambda do |ch|
        # For some reason, rcov likes to drop BUS errors here.
        if ch < 128
          p << ch
        elsif ch < 2048
          p << (ch >> 6 | 192)
          p << (ch & 63 | 128)
        elsif ch < 0x10000
          p << (ch >> 12 | 224)
          p << (ch >> 6 & 63 | 128)
          p << (ch & 63 | 128)
        elsif ch < 0x200000
          p << (ch >> 18 | 240)
          p << (ch >> 12 & 63 | 128)
          p << (ch >> 6 & 63 | 128)
          p << (ch & 63 | 128)
        elsif ch < 0x4000000
          p << (ch >> 24 | 248)
          p << (ch >> 18 & 63 | 128)
          p << (ch >> 12 & 63 | 128)
          p << (ch >> 6 & 63 | 128)
          p << (ch & 63 | 128)
        elsif ch < 0x80000000
          p << (ch >> 30 | 252)
          p << (ch >> 24 & 63 | 128)
          p << (ch >> 18 & 63 | 128)
          p << (ch >> 12 & 63 | 128)
          p << (ch >> 6 & 63 | 128)
          p << (ch & 63 | 128)
        end
      end

      ucs4_to_utf8.call(ch_one)
      ucs4_to_utf8.call(ch_two)

      return lookup_unicode_composition(p)
    end
    (class <<self; private :unicode_compose_pair; end)

    def self.unicode_sort_canonical(unpacked)
      unpacked = unpacked.dup
      i = 1
      length = unpacked.length

      return unpacked if length < 2

      while i < length
        last = unpacked[i-1]
        ch = unpacked[i]
        last_cc = lookup_unicode_combining_class(last)
        cc = lookup_unicode_combining_class(ch)
        if cc != 0 && last_cc != 0 && last_cc > cc
          unpacked[i] = last
          unpacked[i-1] = ch
          i -= 1 if i > 1
        else
          i += 1
        end
      end
      return unpacked
    end
    (class <<self; private :unicode_sort_canonical; end)

    def self.unicode_decompose(unpacked)
      unpacked_result = []
      for cp in unpacked
        if cp >= HANGUL_SBASE && cp < HANGUL_SBASE + HANGUL_SCOUNT
          l, v, t = unicode_decompose_hangul(cp)
          unpacked_result << l
          unpacked_result << v if v
          unpacked_result << t if t
        else
          dc = lookup_unicode_compatibility(cp)
          unless dc
            unpacked_result << cp
          else
            unpacked_result.concat(unicode_decompose(dc.unpack("U*")))
          end
        end
      end
      return unpacked_result
    end
    (class <<self; private :unicode_decompose; end)

    def self.unicode_decompose_hangul(codepoint)
      sindex = codepoint - HANGUL_SBASE;
      if sindex < 0 || sindex >= HANGUL_SCOUNT
        l = codepoint
        v = t = nil
        return l, v, t
      end
      l = HANGUL_LBASE + sindex / HANGUL_NCOUNT
      v = HANGUL_VBASE + (sindex % HANGUL_NCOUNT) / HANGUL_TCOUNT
      t = HANGUL_TBASE + sindex % HANGUL_TCOUNT
      if t == HANGUL_TBASE
        t = nil
      end
      return l, v, t
    end
    (class <<self; private :unicode_decompose_hangul; end)

    def self.lookup_unicode_combining_class(codepoint)
      codepoint_data = UNICODE_DATA[codepoint]
      (codepoint_data ?
        (codepoint_data[UNICODE_DATA_COMBINING_CLASS] || 0) :
        0)
    end
    (class <<self; private :lookup_unicode_combining_class; end)

    def self.lookup_unicode_compatibility(codepoint)
      codepoint_data = UNICODE_DATA[codepoint]
      (codepoint_data ?
        codepoint_data[UNICODE_DATA_COMPATIBILITY] : nil)
    end
    (class <<self; private :lookup_unicode_compatibility; end)

    def self.lookup_unicode_lowercase(codepoint)
      codepoint_data = UNICODE_DATA[codepoint]
      (codepoint_data ?
        (codepoint_data[UNICODE_DATA_LOWERCASE] || codepoint) :
        codepoint)
    end
    (class <<self; private :lookup_unicode_lowercase; end)

    def self.lookup_unicode_composition(unpacked)
      return COMPOSITION_TABLE[unpacked.pack("C*")]
    end
    (class <<self; private :lookup_unicode_composition; end)

    HANGUL_SBASE =  0xac00
    HANGUL_LBASE =  0x1100
    HANGUL_LCOUNT = 19
    HANGUL_VBASE =  0x1161
    HANGUL_VCOUNT = 21
    HANGUL_TBASE =  0x11a7
    HANGUL_TCOUNT = 28
    HANGUL_NCOUNT = HANGUL_VCOUNT * HANGUL_TCOUNT # 588
    HANGUL_SCOUNT = HANGUL_LCOUNT * HANGUL_NCOUNT # 11172

    UNICODE_DATA_COMBINING_CLASS = 0
    UNICODE_DATA_EXCLUSION = 1
    UNICODE_DATA_CANONICAL = 2
    UNICODE_DATA_COMPATIBILITY = 3
    UNICODE_DATA_UPPERCASE = 4
    UNICODE_DATA_LOWERCASE = 5
    UNICODE_DATA_TITLECASE = 6

    # This is a sparse Unicode table.  Codepoints without entries are
    # assumed to have the value: [0, 0, nil, nil, nil, nil, nil]
    UNICODE_DATA = {
      0x0041 => [0, 0, nil, nil, nil, 0x0061, nil],
      0x0042 => [0, 0, nil, nil, nil, 0x0062, nil],
      0x0043 => [0, 0, nil, nil, nil, 0x0063, nil],
      0x0044 => [0, 0, nil, nil, nil, 0x0064, nil],
      0x0045 => [0, 0, nil, nil, nil, 0x0065, nil],
      0x0046 => [0, 0, nil, nil, nil, 0x0066, nil],
      0x0047 => [0, 0, nil, nil, nil, 0x0067, nil],
      0x0048 => [0, 0, nil, nil, nil, 0x0068, nil],
      0x0049 => [0, 0, nil, nil, nil, 0x0069, nil],
      0x004a => [0, 0, nil, nil, nil, 0x006a, nil],
      0x004b => [0, 0, nil, nil, nil, 0x006b, nil],
      0x004c => [0, 0, nil, nil, nil, 0x006c, nil],
      0x004d => [0, 0, nil, nil, nil, 0x006d, nil],
      0x004e => [0, 0, nil, nil, nil, 0x006e, nil],
      0x004f => [0, 0, nil, nil, nil, 0x006f, nil],
      0x0050 => [0, 0, nil, nil, nil, 0x0070, nil],
      0x0051 => [0, 0, nil, nil, nil, 0x0071, nil],
      0x0052 => [0, 0, nil, nil, nil, 0x0072, nil],
      0x0053 => [0, 0, nil, nil, nil, 0x0073, nil],
      0x0054 => [0, 0, nil, nil, nil, 0x0074, nil],
      0x0055 => [0, 0, nil, nil, nil, 0x0075, nil],
      0x0056 => [0, 0, nil, nil, nil, 0x0076, nil],
      0x0057 => [0, 0, nil, nil, nil, 0x0077, nil],
      0x0058 => [0, 0, nil, nil, nil, 0x0078, nil],
      0x0059 => [0, 0, nil, nil, nil, 0x0079, nil],
      0x005a => [0, 0, nil, nil, nil, 0x007a, nil],
      0x0061 => [0, 0, nil, nil, 0x0041, nil, 0x0041],
      0x0062 => [0, 0, nil, nil, 0x0042, nil, 0x0042],
      0x0063 => [0, 0, nil, nil, 0x0043, nil, 0x0043],
      0x0064 => [0, 0, nil, nil, 0x0044, nil, 0x0044],
      0x0065 => [0, 0, nil, nil, 0x0045, nil, 0x0045],
      0x0066 => [0, 0, nil, nil, 0x0046, nil, 0x0046],
      0x0067 => [0, 0, nil, nil, 0x0047, nil, 0x0047],
      0x0068 => [0, 0, nil, nil, 0x0048, nil, 0x0048],
      0x0069 => [0, 0, nil, nil, 0x0049, nil, 0x0049],
      0x006a => [0, 0, nil, nil, 0x004a, nil, 0x004a],
      0x006b => [0, 0, nil, nil, 0x004b, nil, 0x004b],
      0x006c => [0, 0, nil, nil, 0x004c, nil, 0x004c],
      0x006d => [0, 0, nil, nil, 0x004d, nil, 0x004d],
      0x006e => [0, 0, nil, nil, 0x004e, nil, 0x004e],
      0x006f => [0, 0, nil, nil, 0x004f, nil, 0x004f],
      0x0070 => [0, 0, nil, nil, 0x0050, nil, 0x0050],
      0x0071 => [0, 0, nil, nil, 0x0051, nil, 0x0051],
      0x0072 => [0, 0, nil, nil, 0x0052, nil, 0x0052],
      0x0073 => [0, 0, nil, nil, 0x0053, nil, 0x0053],
      0x0074 => [0, 0, nil, nil, 0x0054, nil, 0x0054],
      0x0075 => [0, 0, nil, nil, 0x0055, nil, 0x0055],
      0x0076 => [0, 0, nil, nil, 0x0056, nil, 0x0056],
      0x0077 => [0, 0, nil, nil, 0x0057, nil, 0x0057],
      0x0078 => [0, 0, nil, nil, 0x0058, nil, 0x0058],
      0x0079 => [0, 0, nil, nil, 0x0059, nil, 0x0059],
      0x007a => [0, 0, nil, nil, 0x005a, nil, 0x005a],
      0x00a0 => [0, 0, nil, " ", nil, nil, nil],
      0x00a8 => [0, 0, nil, " \314\210", nil, nil, nil],
      0x00aa => [0, 0, nil, "a", nil, nil, nil],
      0x00af => [0, 0, nil, " \314\204", nil, nil, nil],
      0x00b2 => [0, 0, nil, "2", nil, nil, nil],
      0x00b3 => [0, 0, nil, "3", nil, nil, nil],
      0x00b4 => [0, 0, nil, " \314\201", nil, nil, nil],
      0x00b5 => [0, 0, nil, "\316\274", 0x039c, nil, 0x039c],
      0x00b8 => [0, 0, nil, " \314\247", nil, nil, nil],
      0x00b9 => [0, 0, nil, "1", nil, nil, nil],
      0x00ba => [0, 0, nil, "o", nil, nil, nil],
      0x00bc => [0, 0, nil, "1\342\201\2044", nil, nil, nil],
      0x00bd => [0, 0, nil, "1\342\201\2042", nil, nil, nil],
      0x00be => [0, 0, nil, "3\342\201\2044", nil, nil, nil],
      0x00c0 => [0, 0, "A\314\200", "A\314\200", nil, 0x00e0, nil],
      0x00c1 => [0, 0, "A\314\201", "A\314\201", nil, 0x00e1, nil],
      0x00c2 => [0, 0, "A\314\202", "A\314\202", nil, 0x00e2, nil],
      0x00c3 => [0, 0, "A\314\203", "A\314\203", nil, 0x00e3, nil],
      0x00c4 => [0, 0, "A\314\210", "A\314\210", nil, 0x00e4, nil],
      0x00c5 => [0, 0, "A\314\212", "A\314\212", nil, 0x00e5, nil],
      0x00c6 => [0, 0, nil, nil, nil, 0x00e6, nil],
      0x00c7 => [0, 0, "C\314\247", "C\314\247", nil, 0x00e7, nil],
      0x00c8 => [0, 0, "E\314\200", "E\314\200", nil, 0x00e8, nil],
      0x00c9 => [0, 0, "E\314\201", "E\314\201", nil, 0x00e9, nil],
      0x00ca => [0, 0, "E\314\202", "E\314\202", nil, 0x00ea, nil],
      0x00cb => [0, 0, "E\314\210", "E\314\210", nil, 0x00eb, nil],
      0x00cc => [0, 0, "I\314\200", "I\314\200", nil, 0x00ec, nil],
      0x00cd => [0, 0, "I\314\201", "I\314\201", nil, 0x00ed, nil],
      0x00ce => [0, 0, "I\314\202", "I\314\202", nil, 0x00ee, nil],
      0x00cf => [0, 0, "I\314\210", "I\314\210", nil, 0x00ef, nil],
      0x00d0 => [0, 0, nil, nil, nil, 0x00f0, nil],
      0x00d1 => [0, 0, "N\314\203", "N\314\203", nil, 0x00f1, nil],
      0x00d2 => [0, 0, "O\314\200", "O\314\200", nil, 0x00f2, nil],
      0x00d3 => [0, 0, "O\314\201", "O\314\201", nil, 0x00f3, nil],
      0x00d4 => [0, 0, "O\314\202", "O\314\202", nil, 0x00f4, nil],
      0x00d5 => [0, 0, "O\314\203", "O\314\203", nil, 0x00f5, nil],
      0x00d6 => [0, 0, "O\314\210", "O\314\210", nil, 0x00f6, nil],
      0x00d8 => [0, 0, nil, nil, nil, 0x00f8, nil],
      0x00d9 => [0, 0, "U\314\200", "U\314\200", nil, 0x00f9, nil],
      0x00da => [0, 0, "U\314\201", "U\314\201", nil, 0x00fa, nil],
      0x00db => [0, 0, "U\314\202", "U\314\202", nil, 0x00fb, nil],
      0x00dc => [0, 0, "U\314\210", "U\314\210", nil, 0x00fc, nil],
      0x00dd => [0, 0, "Y\314\201", "Y\314\201", nil, 0x00fd, nil],
      0x00de => [0, 0, nil, nil, nil, 0x00fe, nil],
      0x00e0 => [0, 0, "a\314\200", "a\314\200", 0x00c0, nil, 0x00c0],
      0x00e1 => [0, 0, "a\314\201", "a\314\201", 0x00c1, nil, 0x00c1],
      0x00e2 => [0, 0, "a\314\202", "a\314\202", 0x00c2, nil, 0x00c2],
      0x00e3 => [0, 0, "a\314\203", "a\314\203", 0x00c3, nil, 0x00c3],
      0x00e4 => [0, 0, "a\314\210", "a\314\210", 0x00c4, nil, 0x00c4],
      0x00e5 => [0, 0, "a\314\212", "a\314\212", 0x00c5, nil, 0x00c5],
      0x00e6 => [0, 0, nil, nil, 0x00c6, nil, 0x00c6],
      0x00e7 => [0, 0, "c\314\247", "c\314\247", 0x00c7, nil, 0x00c7],
      0x00e8 => [0, 0, "e\314\200", "e\314\200", 0x00c8, nil, 0x00c8],
      0x00e9 => [0, 0, "e\314\201", "e\314\201", 0x00c9, nil, 0x00c9],
      0x00ea => [0, 0, "e\314\202", "e\314\202", 0x00ca, nil, 0x00ca],
      0x00eb => [0, 0, "e\314\210", "e\314\210", 0x00cb, nil, 0x00cb],
      0x00ec => [0, 0, "i\314\200", "i\314\200", 0x00cc, nil, 0x00cc],
      0x00ed => [0, 0, "i\314\201", "i\314\201", 0x00cd, nil, 0x00cd],
      0x00ee => [0, 0, "i\314\202", "i\314\202", 0x00ce, nil, 0x00ce],
      0x00ef => [0, 0, "i\314\210", "i\314\210", 0x00cf, nil, 0x00cf],
      0x00f0 => [0, 0, nil, nil, 0x00d0, nil, 0x00d0],
      0x00f1 => [0, 0, "n\314\203", "n\314\203", 0x00d1, nil, 0x00d1],
      0x00f2 => [0, 0, "o\314\200", "o\314\200", 0x00d2, nil, 0x00d2],
      0x00f3 => [0, 0, "o\314\201", "o\314\201", 0x00d3, nil, 0x00d3],
      0x00f4 => [0, 0, "o\314\202", "o\314\202", 0x00d4, nil, 0x00d4],
      0x00f5 => [0, 0, "o\314\203", "o\314\203", 0x00d5, nil, 0x00d5],
      0x00f6 => [0, 0, "o\314\210", "o\314\210", 0x00d6, nil, 0x00d6],
      0x00f8 => [0, 0, nil, nil, 0x00d8, nil, 0x00d8],
      0x00f9 => [0, 0, "u\314\200", "u\314\200", 0x00d9, nil, 0x00d9],
      0x00fa => [0, 0, "u\314\201", "u\314\201", 0x00da, nil, 0x00da],
      0x00fb => [0, 0, "u\314\202", "u\314\202", 0x00db, nil, 0x00db],
      0x00fc => [0, 0, "u\314\210", "u\314\210", 0x00dc, nil, 0x00dc],
      0x00fd => [0, 0, "y\314\201", "y\314\201", 0x00dd, nil, 0x00dd],
      0x00fe => [0, 0, nil, nil, 0x00de, nil, 0x00de],
      0x00ff => [0, 0, "y\314\210", "y\314\210", 0x0178, nil, 0x0178],
      0x0100 => [0, 0, "A\314\204", "A\314\204", nil, 0x0101, nil],
      0x0101 => [0, 0, "a\314\204", "a\314\204", 0x0100, nil, 0x0100],
      0x0102 => [0, 0, "A\314\206", "A\314\206", nil, 0x0103, nil],
      0x0103 => [0, 0, "a\314\206", "a\314\206", 0x0102, nil, 0x0102],
      0x0104 => [0, 0, "A\314\250", "A\314\250", nil, 0x0105, nil],
      0x0105 => [0, 0, "a\314\250", "a\314\250", 0x0104, nil, 0x0104],
      0x0106 => [0, 0, "C\314\201", "C\314\201", nil, 0x0107, nil],
      0x0107 => [0, 0, "c\314\201", "c\314\201", 0x0106, nil, 0x0106],
      0x0108 => [0, 0, "C\314\202", "C\314\202", nil, 0x0109, nil],
      0x0109 => [0, 0, "c\314\202", "c\314\202", 0x0108, nil, 0x0108],
      0x010a => [0, 0, "C\314\207", "C\314\207", nil, 0x010b, nil],
      0x010b => [0, 0, "c\314\207", "c\314\207", 0x010a, nil, 0x010a],
      0x010c => [0, 0, "C\314\214", "C\314\214", nil, 0x010d, nil],
      0x010d => [0, 0, "c\314\214", "c\314\214", 0x010c, nil, 0x010c],
      0x010e => [0, 0, "D\314\214", "D\314\214", nil, 0x010f, nil],
      0x010f => [0, 0, "d\314\214", "d\314\214", 0x010e, nil, 0x010e],
      0x0110 => [0, 0, nil, nil, nil, 0x0111, nil],
      0x0111 => [0, 0, nil, nil, 0x0110, nil, 0x0110],
      0x0112 => [0, 0, "E\314\204", "E\314\204", nil, 0x0113, nil],
      0x0113 => [0, 0, "e\314\204", "e\314\204", 0x0112, nil, 0x0112],
      0x0114 => [0, 0, "E\314\206", "E\314\206", nil, 0x0115, nil],
      0x0115 => [0, 0, "e\314\206", "e\314\206", 0x0114, nil, 0x0114],
      0x0116 => [0, 0, "E\314\207", "E\314\207", nil, 0x0117, nil],
      0x0117 => [0, 0, "e\314\207", "e\314\207", 0x0116, nil, 0x0116],
      0x0118 => [0, 0, "E\314\250", "E\314\250", nil, 0x0119, nil],
      0x0119 => [0, 0, "e\314\250", "e\314\250", 0x0118, nil, 0x0118],
      0x011a => [0, 0, "E\314\214", "E\314\214", nil, 0x011b, nil],
      0x011b => [0, 0, "e\314\214", "e\314\214", 0x011a, nil, 0x011a],
      0x011c => [0, 0, "G\314\202", "G\314\202", nil, 0x011d, nil],
      0x011d => [0, 0, "g\314\202", "g\314\202", 0x011c, nil, 0x011c],
      0x011e => [0, 0, "G\314\206", "G\314\206", nil, 0x011f, nil],
      0x011f => [0, 0, "g\314\206", "g\314\206", 0x011e, nil, 0x011e],
      0x0120 => [0, 0, "G\314\207", "G\314\207", nil, 0x0121, nil],
      0x0121 => [0, 0, "g\314\207", "g\314\207", 0x0120, nil, 0x0120],
      0x0122 => [0, 0, "G\314\247", "G\314\247", nil, 0x0123, nil],
      0x0123 => [0, 0, "g\314\247", "g\314\247", 0x0122, nil, 0x0122],
      0x0124 => [0, 0, "H\314\202", "H\314\202", nil, 0x0125, nil],
      0x0125 => [0, 0, "h\314\202", "h\314\202", 0x0124, nil, 0x0124],
      0x0126 => [0, 0, nil, nil, nil, 0x0127, nil],
      0x0127 => [0, 0, nil, nil, 0x0126, nil, 0x0126],
      0x0128 => [0, 0, "I\314\203", "I\314\203", nil, 0x0129, nil],
      0x0129 => [0, 0, "i\314\203", "i\314\203", 0x0128, nil, 0x0128],
      0x012a => [0, 0, "I\314\204", "I\314\204", nil, 0x012b, nil],
      0x012b => [0, 0, "i\314\204", "i\314\204", 0x012a, nil, 0x012a],
      0x012c => [0, 0, "I\314\206", "I\314\206", nil, 0x012d, nil],
      0x012d => [0, 0, "i\314\206", "i\314\206", 0x012c, nil, 0x012c],
      0x012e => [0, 0, "I\314\250", "I\314\250", nil, 0x012f, nil],
      0x012f => [0, 0, "i\314\250", "i\314\250", 0x012e, nil, 0x012e],
      0x0130 => [0, 0, "I\314\207", "I\314\207", nil, 0x0069, nil],
      0x0131 => [0, 0, nil, nil, 0x0049, nil, 0x0049],
      0x0132 => [0, 0, nil, "IJ", nil, 0x0133, nil],
      0x0133 => [0, 0, nil, "ij", 0x0132, nil, 0x0132],
      0x0134 => [0, 0, "J\314\202", "J\314\202", nil, 0x0135, nil],
      0x0135 => [0, 0, "j\314\202", "j\314\202", 0x0134, nil, 0x0134],
      0x0136 => [0, 0, "K\314\247", "K\314\247", nil, 0x0137, nil],
      0x0137 => [0, 0, "k\314\247", "k\314\247", 0x0136, nil, 0x0136],
      0x0139 => [0, 0, "L\314\201", "L\314\201", nil, 0x013a, nil],
      0x013a => [0, 0, "l\314\201", "l\314\201", 0x0139, nil, 0x0139],
      0x013b => [0, 0, "L\314\247", "L\314\247", nil, 0x013c, nil],
      0x013c => [0, 0, "l\314\247", "l\314\247", 0x013b, nil, 0x013b],
      0x013d => [0, 0, "L\314\214", "L\314\214", nil, 0x013e, nil],
      0x013e => [0, 0, "l\314\214", "l\314\214", 0x013d, nil, 0x013d],
      0x013f => [0, 0, nil, "L\302\267", nil, 0x0140, nil],
      0x0140 => [0, 0, nil, "l\302\267", 0x013f, nil, 0x013f],
      0x0141 => [0, 0, nil, nil, nil, 0x0142, nil],
      0x0142 => [0, 0, nil, nil, 0x0141, nil, 0x0141],
      0x0143 => [0, 0, "N\314\201", "N\314\201", nil, 0x0144, nil],
      0x0144 => [0, 0, "n\314\201", "n\314\201", 0x0143, nil, 0x0143],
      0x0145 => [0, 0, "N\314\247", "N\314\247", nil, 0x0146, nil],
      0x0146 => [0, 0, "n\314\247", "n\314\247", 0x0145, nil, 0x0145],
      0x0147 => [0, 0, "N\314\214", "N\314\214", nil, 0x0148, nil],
      0x0148 => [0, 0, "n\314\214", "n\314\214", 0x0147, nil, 0x0147],
      0x0149 => [0, 0, nil, "\312\274n", nil, nil, nil],
      0x014a => [0, 0, nil, nil, nil, 0x014b, nil],
      0x014b => [0, 0, nil, nil, 0x014a, nil, 0x014a],
      0x014c => [0, 0, "O\314\204", "O\314\204", nil, 0x014d, nil],
      0x014d => [0, 0, "o\314\204", "o\314\204", 0x014c, nil, 0x014c],
      0x014e => [0, 0, "O\314\206", "O\314\206", nil, 0x014f, nil],
      0x014f => [0, 0, "o\314\206", "o\314\206", 0x014e, nil, 0x014e],
      0x0150 => [0, 0, "O\314\213", "O\314\213", nil, 0x0151, nil],
      0x0151 => [0, 0, "o\314\213", "o\314\213", 0x0150, nil, 0x0150],
      0x0152 => [0, 0, nil, nil, nil, 0x0153, nil],
      0x0153 => [0, 0, nil, nil, 0x0152, nil, 0x0152],
      0x0154 => [0, 0, "R\314\201", "R\314\201", nil, 0x0155, nil],
      0x0155 => [0, 0, "r\314\201", "r\314\201", 0x0154, nil, 0x0154],
      0x0156 => [0, 0, "R\314\247", "R\314\247", nil, 0x0157, nil],
      0x0157 => [0, 0, "r\314\247", "r\314\247", 0x0156, nil, 0x0156],
      0x0158 => [0, 0, "R\314\214", "R\314\214", nil, 0x0159, nil],
      0x0159 => [0, 0, "r\314\214", "r\314\214", 0x0158, nil, 0x0158],
      0x015a => [0, 0, "S\314\201", "S\314\201", nil, 0x015b, nil],
      0x015b => [0, 0, "s\314\201", "s\314\201", 0x015a, nil, 0x015a],
      0x015c => [0, 0, "S\314\202", "S\314\202", nil, 0x015d, nil],
      0x015d => [0, 0, "s\314\202", "s\314\202", 0x015c, nil, 0x015c],
      0x015e => [0, 0, "S\314\247", "S\314\247", nil, 0x015f, nil],
      0x015f => [0, 0, "s\314\247", "s\314\247", 0x015e, nil, 0x015e],
      0x0160 => [0, 0, "S\314\214", "S\314\214", nil, 0x0161, nil],
      0x0161 => [0, 0, "s\314\214", "s\314\214", 0x0160, nil, 0x0160],
      0x0162 => [0, 0, "T\314\247", "T\314\247", nil, 0x0163, nil],
      0x0163 => [0, 0, "t\314\247", "t\314\247", 0x0162, nil, 0x0162],
      0x0164 => [0, 0, "T\314\214", "T\314\214", nil, 0x0165, nil],
      0x0165 => [0, 0, "t\314\214", "t\314\214", 0x0164, nil, 0x0164],
      0x0166 => [0, 0, nil, nil, nil, 0x0167, nil],
      0x0167 => [0, 0, nil, nil, 0x0166, nil, 0x0166],
      0x0168 => [0, 0, "U\314\203", "U\314\203", nil, 0x0169, nil],
      0x0169 => [0, 0, "u\314\203", "u\314\203", 0x0168, nil, 0x0168],
      0x016a => [0, 0, "U\314\204", "U\314\204", nil, 0x016b, nil],
      0x016b => [0, 0, "u\314\204", "u\314\204", 0x016a, nil, 0x016a],
      0x016c => [0, 0, "U\314\206", "U\314\206", nil, 0x016d, nil],
      0x016d => [0, 0, "u\314\206", "u\314\206", 0x016c, nil, 0x016c],
      0x016e => [0, 0, "U\314\212", "U\314\212", nil, 0x016f, nil],
      0x016f => [0, 0, "u\314\212", "u\314\212", 0x016e, nil, 0x016e],
      0x0170 => [0, 0, "U\314\213", "U\314\213", nil, 0x0171, nil],
      0x0171 => [0, 0, "u\314\213", "u\314\213", 0x0170, nil, 0x0170],
      0x0172 => [0, 0, "U\314\250", "U\314\250", nil, 0x0173, nil],
      0x0173 => [0, 0, "u\314\250", "u\314\250", 0x0172, nil, 0x0172],
      0x0174 => [0, 0, "W\314\202", "W\314\202", nil, 0x0175, nil],
      0x0175 => [0, 0, "w\314\202", "w\314\202", 0x0174, nil, 0x0174],
      0x0176 => [0, 0, "Y\314\202", "Y\314\202", nil, 0x0177, nil],
      0x0177 => [0, 0, "y\314\202", "y\314\202", 0x0176, nil, 0x0176],
      0x0178 => [0, 0, "Y\314\210", "Y\314\210", nil, 0x00ff, nil],
      0x0179 => [0, 0, "Z\314\201", "Z\314\201", nil, 0x017a, nil],
      0x017a => [0, 0, "z\314\201", "z\314\201", 0x0179, nil, 0x0179],
      0x017b => [0, 0, "Z\314\207", "Z\314\207", nil, 0x017c, nil],
      0x017c => [0, 0, "z\314\207", "z\314\207", 0x017b, nil, 0x017b],
      0x017d => [0, 0, "Z\314\214", "Z\314\214", nil, 0x017e, nil],
      0x017e => [0, 0, "z\314\214", "z\314\214", 0x017d, nil, 0x017d],
      0x017f => [0, 0, nil, "s", 0x0053, nil, 0x0053],
      0x0181 => [0, 0, nil, nil, nil, 0x0253, nil],
      0x0182 => [0, 0, nil, nil, nil, 0x0183, nil],
      0x0183 => [0, 0, nil, nil, 0x0182, nil, 0x0182],
      0x0184 => [0, 0, nil, nil, nil, 0x0185, nil],
      0x0185 => [0, 0, nil, nil, 0x0184, nil, 0x0184],
      0x0186 => [0, 0, nil, nil, nil, 0x0254, nil],
      0x0187 => [0, 0, nil, nil, nil, 0x0188, nil],
      0x0188 => [0, 0, nil, nil, 0x0187, nil, 0x0187],
      0x0189 => [0, 0, nil, nil, nil, 0x0256, nil],
      0x018a => [0, 0, nil, nil, nil, 0x0257, nil],
      0x018b => [0, 0, nil, nil, nil, 0x018c, nil],
      0x018c => [0, 0, nil, nil, 0x018b, nil, 0x018b],
      0x018e => [0, 0, nil, nil, nil, 0x01dd, nil],
      0x018f => [0, 0, nil, nil, nil, 0x0259, nil],
      0x0190 => [0, 0, nil, nil, nil, 0x025b, nil],
      0x0191 => [0, 0, nil, nil, nil, 0x0192, nil],
      0x0192 => [0, 0, nil, nil, 0x0191, nil, 0x0191],
      0x0193 => [0, 0, nil, nil, nil, 0x0260, nil],
      0x0194 => [0, 0, nil, nil, nil, 0x0263, nil],
      0x0195 => [0, 0, nil, nil, 0x01f6, nil, 0x01f6],
      0x0196 => [0, 0, nil, nil, nil, 0x0269, nil],
      0x0197 => [0, 0, nil, nil, nil, 0x0268, nil],
      0x0198 => [0, 0, nil, nil, nil, 0x0199, nil],
      0x0199 => [0, 0, nil, nil, 0x0198, nil, 0x0198],
      0x019c => [0, 0, nil, nil, nil, 0x026f, nil],
      0x019d => [0, 0, nil, nil, nil, 0x0272, nil],
      0x019f => [0, 0, nil, nil, nil, 0x0275, nil],
      0x01a0 => [0, 0, "O\314\233", "O\314\233", nil, 0x01a1, nil],
      0x01a1 => [0, 0, "o\314\233", "o\314\233", 0x01a0, nil, 0x01a0],
      0x01a2 => [0, 0, nil, nil, nil, 0x01a3, nil],
      0x01a3 => [0, 0, nil, nil, 0x01a2, nil, 0x01a2],
      0x01a4 => [0, 0, nil, nil, nil, 0x01a5, nil],
      0x01a5 => [0, 0, nil, nil, 0x01a4, nil, 0x01a4],
      0x01a6 => [0, 0, nil, nil, nil, 0x0280, nil],
      0x01a7 => [0, 0, nil, nil, nil, 0x01a8, nil],
      0x01a8 => [0, 0, nil, nil, 0x01a7, nil, 0x01a7],
      0x01a9 => [0, 0, nil, nil, nil, 0x0283, nil],
      0x01ac => [0, 0, nil, nil, nil, 0x01ad, nil],
      0x01ad => [0, 0, nil, nil, 0x01ac, nil, 0x01ac],
      0x01ae => [0, 0, nil, nil, nil, 0x0288, nil],
      0x01af => [0, 0, "U\314\233", "U\314\233", nil, 0x01b0, nil],
      0x01b0 => [0, 0, "u\314\233", "u\314\233", 0x01af, nil, 0x01af],
      0x01b1 => [0, 0, nil, nil, nil, 0x028a, nil],
      0x01b2 => [0, 0, nil, nil, nil, 0x028b, nil],
      0x01b3 => [0, 0, nil, nil, nil, 0x01b4, nil],
      0x01b4 => [0, 0, nil, nil, 0x01b3, nil, 0x01b3],
      0x01b5 => [0, 0, nil, nil, nil, 0x01b6, nil],
      0x01b6 => [0, 0, nil, nil, 0x01b5, nil, 0x01b5],
      0x01b7 => [0, 0, nil, nil, nil, 0x0292, nil],
      0x01b8 => [0, 0, nil, nil, nil, 0x01b9, nil],
      0x01b9 => [0, 0, nil, nil, 0x01b8, nil, 0x01b8],
      0x01bc => [0, 0, nil, nil, nil, 0x01bd, nil],
      0x01bd => [0, 0, nil, nil, 0x01bc, nil, 0x01bc],
      0x01bf => [0, 0, nil, nil, 0x01f7, nil, 0x01f7],
      0x01c4 => [0, 0, nil, "D\305\275", nil, 0x01c6, 0x01c5],
      0x01c5 => [0, 0, nil, "D\305\276", 0x01c4, 0x01c6, nil],
      0x01c6 => [0, 0, nil, "d\305\276", 0x01c4, nil, 0x01c5],
      0x01c7 => [0, 0, nil, "LJ", nil, 0x01c9, 0x01c8],
      0x01c8 => [0, 0, nil, "Lj", 0x01c7, 0x01c9, nil],
      0x01c9 => [0, 0, nil, "lj", 0x01c7, nil, 0x01c8],
      0x01ca => [0, 0, nil, "NJ", nil, 0x01cc, 0x01cb],
      0x01cb => [0, 0, nil, "Nj", 0x01ca, 0x01cc, nil],
      0x01cc => [0, 0, nil, "nj", 0x01ca, nil, 0x01cb],
      0x01cd => [0, 0, "A\314\214", "A\314\214", nil, 0x01ce, nil],
      0x01ce => [0, 0, "a\314\214", "a\314\214", 0x01cd, nil, 0x01cd],
      0x01cf => [0, 0, "I\314\214", "I\314\214", nil, 0x01d0, nil],
      0x01d0 => [0, 0, "i\314\214", "i\314\214", 0x01cf, nil, 0x01cf],
      0x01d1 => [0, 0, "O\314\214", "O\314\214", nil, 0x01d2, nil],
      0x01d2 => [0, 0, "o\314\214", "o\314\214", 0x01d1, nil, 0x01d1],
      0x01d3 => [0, 0, "U\314\214", "U\314\214", nil, 0x01d4, nil],
      0x01d4 => [0, 0, "u\314\214", "u\314\214", 0x01d3, nil, 0x01d3],
      0x01d5 => [0, 0, "\303\234\314\204", "\303\234\314\204", nil, 0x01d6, nil],
      0x01d6 => [0, 0, "\303\274\314\204", "\303\274\314\204", 0x01d5, nil, 0x01d5],
      0x01d7 => [0, 0, "\303\234\314\201", "\303\234\314\201", nil, 0x01d8, nil],
      0x01d8 => [0, 0, "\303\274\314\201", "\303\274\314\201", 0x01d7, nil, 0x01d7],
      0x01d9 => [0, 0, "\303\234\314\214", "\303\234\314\214", nil, 0x01da, nil],
      0x01da => [0, 0, "\303\274\314\214", "\303\274\314\214", 0x01d9, nil, 0x01d9],
      0x01db => [0, 0, "\303\234\314\200", "\303\234\314\200", nil, 0x01dc, nil],
      0x01dc => [0, 0, "\303\274\314\200", "\303\274\314\200", 0x01db, nil, 0x01db],
      0x01dd => [0, 0, nil, nil, 0x018e, nil, 0x018e],
      0x01de => [0, 0, "\303\204\314\204", "\303\204\314\204", nil, 0x01df, nil],
      0x01df => [0, 0, "\303\244\314\204", "\303\244\314\204", 0x01de, nil, 0x01de],
      0x01e0 => [0, 0, "\310\246\314\204", "\310\246\314\204", nil, 0x01e1, nil],
      0x01e1 => [0, 0, "\310\247\314\204", "\310\247\314\204", 0x01e0, nil, 0x01e0],
      0x01e2 => [0, 0, "\303\206\314\204", "\303\206\314\204", nil, 0x01e3, nil],
      0x01e3 => [0, 0, "\303\246\314\204", "\303\246\314\204", 0x01e2, nil, 0x01e2],
      0x01e4 => [0, 0, nil, nil, nil, 0x01e5, nil],
      0x01e5 => [0, 0, nil, nil, 0x01e4, nil, 0x01e4],
      0x01e6 => [0, 0, "G\314\214", "G\314\214", nil, 0x01e7, nil],
      0x01e7 => [0, 0, "g\314\214", "g\314\214", 0x01e6, nil, 0x01e6],
      0x01e8 => [0, 0, "K\314\214", "K\314\214", nil, 0x01e9, nil],
      0x01e9 => [0, 0, "k\314\214", "k\314\214", 0x01e8, nil, 0x01e8],
      0x01ea => [0, 0, "O\314\250", "O\314\250", nil, 0x01eb, nil],
      0x01eb => [0, 0, "o\314\250", "o\314\250", 0x01ea, nil, 0x01ea],
      0x01ec => [0, 0, "\307\252\314\204", "\307\252\314\204", nil, 0x01ed, nil],
      0x01ed => [0, 0, "\307\253\314\204", "\307\253\314\204", 0x01ec, nil, 0x01ec],
      0x01ee => [0, 0, "\306\267\314\214", "\306\267\314\214", nil, 0x01ef, nil],
      0x01ef => [0, 0, "\312\222\314\214", "\312\222\314\214", 0x01ee, nil, 0x01ee],
      0x01f0 => [0, 0, "j\314\214", "j\314\214", nil, nil, nil],
      0x01f1 => [0, 0, nil, "DZ", nil, 0x01f3, 0x01f2],
      0x01f2 => [0, 0, nil, "Dz", 0x01f1, 0x01f3, nil],
      0x01f3 => [0, 0, nil, "dz", 0x01f1, nil, 0x01f2],
      0x01f4 => [0, 0, "G\314\201", "G\314\201", nil, 0x01f5, nil],
      0x01f5 => [0, 0, "g\314\201", "g\314\201", 0x01f4, nil, 0x01f4],
      0x01f6 => [0, 0, nil, nil, nil, 0x0195, nil],
      0x01f7 => [0, 0, nil, nil, nil, 0x01bf, nil],
      0x01f8 => [0, 0, "N\314\200", "N\314\200", nil, 0x01f9, nil],
      0x01f9 => [0, 0, "n\314\200", "n\314\200", 0x01f8, nil, 0x01f8],
      0x01fa => [0, 0, "\303\205\314\201", "\303\205\314\201", nil, 0x01fb, nil],
      0x01fb => [0, 0, "\303\245\314\201", "\303\245\314\201", 0x01fa, nil, 0x01fa],
      0x01fc => [0, 0, "\303\206\314\201", "\303\206\314\201", nil, 0x01fd, nil],
      0x01fd => [0, 0, "\303\246\314\201", "\303\246\314\201", 0x01fc, nil, 0x01fc],
      0x01fe => [0, 0, "\303\230\314\201", "\303\230\314\201", nil, 0x01ff, nil],
      0x01ff => [0, 0, "\303\270\314\201", "\303\270\314\201", 0x01fe, nil, 0x01fe],
      0x0200 => [0, 0, "A\314\217", "A\314\217", nil, 0x0201, nil],
      0x0201 => [0, 0, "a\314\217", "a\314\217", 0x0200, nil, 0x0200],
      0x0202 => [0, 0, "A\314\221", "A\314\221", nil, 0x0203, nil],
      0x0203 => [0, 0, "a\314\221", "a\314\221", 0x0202, nil, 0x0202],
      0x0204 => [0, 0, "E\314\217", "E\314\217", nil, 0x0205, nil],
      0x0205 => [0, 0, "e\314\217", "e\314\217", 0x0204, nil, 0x0204],
      0x0206 => [0, 0, "E\314\221", "E\314\221", nil, 0x0207, nil],
      0x0207 => [0, 0, "e\314\221", "e\314\221", 0x0206, nil, 0x0206],
      0x0208 => [0, 0, "I\314\217", "I\314\217", nil, 0x0209, nil],
      0x0209 => [0, 0, "i\314\217", "i\314\217", 0x0208, nil, 0x0208],
      0x020a => [0, 0, "I\314\221", "I\314\221", nil, 0x020b, nil],
      0x020b => [0, 0, "i\314\221", "i\314\221", 0x020a, nil, 0x020a],
      0x020c => [0, 0, "O\314\217", "O\314\217", nil, 0x020d, nil],
      0x020d => [0, 0, "o\314\217", "o\314\217", 0x020c, nil, 0x020c],
      0x020e => [0, 0, "O\314\221", "O\314\221", nil, 0x020f, nil],
      0x020f => [0, 0, "o\314\221", "o\314\221", 0x020e, nil, 0x020e],
      0x0210 => [0, 0, "R\314\217", "R\314\217", nil, 0x0211, nil],
      0x0211 => [0, 0, "r\314\217", "r\314\217", 0x0210, nil, 0x0210],
      0x0212 => [0, 0, "R\314\221", "R\314\221", nil, 0x0213, nil],
      0x0213 => [0, 0, "r\314\221", "r\314\221", 0x0212, nil, 0x0212],
      0x0214 => [0, 0, "U\314\217", "U\314\217", nil, 0x0215, nil],
      0x0215 => [0, 0, "u\314\217", "u\314\217", 0x0214, nil, 0x0214],
      0x0216 => [0, 0, "U\314\221", "U\314\221", nil, 0x0217, nil],
      0x0217 => [0, 0, "u\314\221", "u\314\221", 0x0216, nil, 0x0216],
      0x0218 => [0, 0, "S\314\246", "S\314\246", nil, 0x0219, nil],
      0x0219 => [0, 0, "s\314\246", "s\314\246", 0x0218, nil, 0x0218],
      0x021a => [0, 0, "T\314\246", "T\314\246", nil, 0x021b, nil],
      0x021b => [0, 0, "t\314\246", "t\314\246", 0x021a, nil, 0x021a],
      0x021c => [0, 0, nil, nil, nil, 0x021d, nil],
      0x021d => [0, 0, nil, nil, 0x021c, nil, 0x021c],
      0x021e => [0, 0, "H\314\214", "H\314\214", nil, 0x021f, nil],
      0x021f => [0, 0, "h\314\214", "h\314\214", 0x021e, nil, 0x021e],
      0x0222 => [0, 0, nil, nil, nil, 0x0223, nil],
      0x0223 => [0, 0, nil, nil, 0x0222, nil, 0x0222],
      0x0224 => [0, 0, nil, nil, nil, 0x0225, nil],
      0x0225 => [0, 0, nil, nil, 0x0224, nil, 0x0224],
      0x0226 => [0, 0, "A\314\207", "A\314\207", nil, 0x0227, nil],
      0x0227 => [0, 0, "a\314\207", "a\314\207", 0x0226, nil, 0x0226],
      0x0228 => [0, 0, "E\314\247", "E\314\247", nil, 0x0229, nil],
      0x0229 => [0, 0, "e\314\247", "e\314\247", 0x0228, nil, 0x0228],
      0x022a => [0, 0, "\303\226\314\204", "\303\226\314\204", nil, 0x022b, nil],
      0x022b => [0, 0, "\303\266\314\204", "\303\266\314\204", 0x022a, nil, 0x022a],
      0x022c => [0, 0, "\303\225\314\204", "\303\225\314\204", nil, 0x022d, nil],
      0x022d => [0, 0, "\303\265\314\204", "\303\265\314\204", 0x022c, nil, 0x022c],
      0x022e => [0, 0, "O\314\207", "O\314\207", nil, 0x022f, nil],
      0x022f => [0, 0, "o\314\207", "o\314\207", 0x022e, nil, 0x022e],
      0x0230 => [0, 0, "\310\256\314\204", "\310\256\314\204", nil, 0x0231, nil],
      0x0231 => [0, 0, "\310\257\314\204", "\310\257\314\204", 0x0230, nil, 0x0230],
      0x0232 => [0, 0, "Y\314\204", "Y\314\204", nil, 0x0233, nil],
      0x0233 => [0, 0, "y\314\204", "y\314\204", 0x0232, nil, 0x0232],
      0x0253 => [0, 0, nil, nil, 0x0181, nil, 0x0181],
      0x0254 => [0, 0, nil, nil, 0x0186, nil, 0x0186],
      0x0256 => [0, 0, nil, nil, 0x0189, nil, 0x0189],
      0x0257 => [0, 0, nil, nil, 0x018a, nil, 0x018a],
      0x0259 => [0, 0, nil, nil, 0x018f, nil, 0x018f],
      0x025b => [0, 0, nil, nil, 0x0190, nil, 0x0190],
      0x0260 => [0, 0, nil, nil, 0x0193, nil, 0x0193],
      0x0263 => [0, 0, nil, nil, 0x0194, nil, 0x0194],
      0x0268 => [0, 0, nil, nil, 0x0197, nil, 0x0197],
      0x0269 => [0, 0, nil, nil, 0x0196, nil, 0x0196],
      0x026f => [0, 0, nil, nil, 0x019c, nil, 0x019c],
      0x0272 => [0, 0, nil, nil, 0x019d, nil, 0x019d],
      0x0275 => [0, 0, nil, nil, 0x019f, nil, 0x019f],
      0x0280 => [0, 0, nil, nil, 0x01a6, nil, 0x01a6],
      0x0283 => [0, 0, nil, nil, 0x01a9, nil, 0x01a9],
      0x0288 => [0, 0, nil, nil, 0x01ae, nil, 0x01ae],
      0x028a => [0, 0, nil, nil, 0x01b1, nil, 0x01b1],
      0x028b => [0, 0, nil, nil, 0x01b2, nil, 0x01b2],
      0x0292 => [0, 0, nil, nil, 0x01b7, nil, 0x01b7],
      0x02b0 => [0, 0, nil, "h", nil, nil, nil],
      0x02b1 => [0, 0, nil, "\311\246", nil, nil, nil],
      0x02b2 => [0, 0, nil, "j", nil, nil, nil],
      0x02b3 => [0, 0, nil, "r", nil, nil, nil],
      0x02b4 => [0, 0, nil, "\311\271", nil, nil, nil],
      0x02b5 => [0, 0, nil, "\311\273", nil, nil, nil],
      0x02b6 => [0, 0, nil, "\312\201", nil, nil, nil],
      0x02b7 => [0, 0, nil, "w", nil, nil, nil],
      0x02b8 => [0, 0, nil, "y", nil, nil, nil],
      0x02d8 => [0, 0, nil, " \314\206", nil, nil, nil],
      0x02d9 => [0, 0, nil, " \314\207", nil, nil, nil],
      0x02da => [0, 0, nil, " \314\212", nil, nil, nil],
      0x02db => [0, 0, nil, " \314\250", nil, nil, nil],
      0x02dc => [0, 0, nil, " \314\203", nil, nil, nil],
      0x02dd => [0, 0, nil, " \314\213", nil, nil, nil],
      0x02e0 => [0, 0, nil, "\311\243", nil, nil, nil],
      0x02e1 => [0, 0, nil, "l", nil, nil, nil],
      0x02e2 => [0, 0, nil, "s", nil, nil, nil],
      0x02e3 => [0, 0, nil, "x", nil, nil, nil],
      0x02e4 => [0, 0, nil, "\312\225", nil, nil, nil],
      0x0300 => [230, 0, nil, nil, nil, nil, nil],
      0x0301 => [230, 0, nil, nil, nil, nil, nil],
      0x0302 => [230, 0, nil, nil, nil, nil, nil],
      0x0303 => [230, 0, nil, nil, nil, nil, nil],
      0x0304 => [230, 0, nil, nil, nil, nil, nil],
      0x0305 => [230, 0, nil, nil, nil, nil, nil],
      0x0306 => [230, 0, nil, nil, nil, nil, nil],
      0x0307 => [230, 0, nil, nil, nil, nil, nil],
      0x0308 => [230, 0, nil, nil, nil, nil, nil],
      0x0309 => [230, 0, nil, nil, nil, nil, nil],
      0x030a => [230, 0, nil, nil, nil, nil, nil],
      0x030b => [230, 0, nil, nil, nil, nil, nil],
      0x030c => [230, 0, nil, nil, nil, nil, nil],
      0x030d => [230, 0, nil, nil, nil, nil, nil],
      0x030e => [230, 0, nil, nil, nil, nil, nil],
      0x030f => [230, 0, nil, nil, nil, nil, nil],
      0x0310 => [230, 0, nil, nil, nil, nil, nil],
      0x0311 => [230, 0, nil, nil, nil, nil, nil],
      0x0312 => [230, 0, nil, nil, nil, nil, nil],
      0x0313 => [230, 0, nil, nil, nil, nil, nil],
      0x0314 => [230, 0, nil, nil, nil, nil, nil],
      0x0315 => [232, 0, nil, nil, nil, nil, nil],
      0x0316 => [220, 0, nil, nil, nil, nil, nil],
      0x0317 => [220, 0, nil, nil, nil, nil, nil],
      0x0318 => [220, 0, nil, nil, nil, nil, nil],
      0x0319 => [220, 0, nil, nil, nil, nil, nil],
      0x031a => [232, 0, nil, nil, nil, nil, nil],
      0x031b => [216, 0, nil, nil, nil, nil, nil],
      0x031c => [220, 0, nil, nil, nil, nil, nil],
      0x031d => [220, 0, nil, nil, nil, nil, nil],
      0x031e => [220, 0, nil, nil, nil, nil, nil],
      0x031f => [220, 0, nil, nil, nil, nil, nil],
      0x0320 => [220, 0, nil, nil, nil, nil, nil],
      0x0321 => [202, 0, nil, nil, nil, nil, nil],
      0x0322 => [202, 0, nil, nil, nil, nil, nil],
      0x0323 => [220, 0, nil, nil, nil, nil, nil],
      0x0324 => [220, 0, nil, nil, nil, nil, nil],
      0x0325 => [220, 0, nil, nil, nil, nil, nil],
      0x0326 => [220, 0, nil, nil, nil, nil, nil],
      0x0327 => [202, 0, nil, nil, nil, nil, nil],
      0x0328 => [202, 0, nil, nil, nil, nil, nil],
      0x0329 => [220, 0, nil, nil, nil, nil, nil],
      0x032a => [220, 0, nil, nil, nil, nil, nil],
      0x032b => [220, 0, nil, nil, nil, nil, nil],
      0x032c => [220, 0, nil, nil, nil, nil, nil],
      0x032d => [220, 0, nil, nil, nil, nil, nil],
      0x032e => [220, 0, nil, nil, nil, nil, nil],
      0x032f => [220, 0, nil, nil, nil, nil, nil],
      0x0330 => [220, 0, nil, nil, nil, nil, nil],
      0x0331 => [220, 0, nil, nil, nil, nil, nil],
      0x0332 => [220, 0, nil, nil, nil, nil, nil],
      0x0333 => [220, 0, nil, nil, nil, nil, nil],
      0x0334 => [1, 0, nil, nil, nil, nil, nil],
      0x0335 => [1, 0, nil, nil, nil, nil, nil],
      0x0336 => [1, 0, nil, nil, nil, nil, nil],
      0x0337 => [1, 0, nil, nil, nil, nil, nil],
      0x0338 => [1, 0, nil, nil, nil, nil, nil],
      0x0339 => [220, 0, nil, nil, nil, nil, nil],
      0x033a => [220, 0, nil, nil, nil, nil, nil],
      0x033b => [220, 0, nil, nil, nil, nil, nil],
      0x033c => [220, 0, nil, nil, nil, nil, nil],
      0x033d => [230, 0, nil, nil, nil, nil, nil],
      0x033e => [230, 0, nil, nil, nil, nil, nil],
      0x033f => [230, 0, nil, nil, nil, nil, nil],
      0x0340 => [230, 2, "\314\200", "\314\200", nil, nil, nil],
      0x0341 => [230, 2, "\314\201", "\314\201", nil, nil, nil],
      0x0342 => [230, 0, nil, nil, nil, nil, nil],
      0x0343 => [230, 2, "\314\223", "\314\223", nil, nil, nil],
      0x0344 => [230, 3, "\314\210\314\201", "\314\210\314\201", nil, nil, nil],
      0x0345 => [240, 0, nil, nil, 0x0399, nil, 0x0399],
      0x0346 => [230, 0, nil, nil, nil, nil, nil],
      0x0347 => [220, 0, nil, nil, nil, nil, nil],
      0x0348 => [220, 0, nil, nil, nil, nil, nil],
      0x0349 => [220, 0, nil, nil, nil, nil, nil],
      0x034a => [230, 0, nil, nil, nil, nil, nil],
      0x034b => [230, 0, nil, nil, nil, nil, nil],
      0x034c => [230, 0, nil, nil, nil, nil, nil],
      0x034d => [220, 0, nil, nil, nil, nil, nil],
      0x034e => [220, 0, nil, nil, nil, nil, nil],
      0x0360 => [234, 0, nil, nil, nil, nil, nil],
      0x0361 => [234, 0, nil, nil, nil, nil, nil],
      0x0362 => [233, 0, nil, nil, nil, nil, nil],
      0x0374 => [0, 2, "\312\271", "\312\271", nil, nil, nil],
      0x037a => [0, 0, nil, " \315\205", nil, nil, nil],
      0x037e => [0, 2, ";", ";", nil, nil, nil],
      0x0384 => [0, 0, nil, " \314\201", nil, nil, nil],
      0x0385 => [0, 0, "\302\250\314\201", "\302\250\314\201", nil, nil, nil],
      0x0386 => [0, 0, "\316\221\314\201", "\316\221\314\201", nil, 0x03ac, nil],
      0x0387 => [0, 2, "\302\267", "\302\267", nil, nil, nil],
      0x0388 => [0, 0, "\316\225\314\201", "\316\225\314\201", nil, 0x03ad, nil],
      0x0389 => [0, 0, "\316\227\314\201", "\316\227\314\201", nil, 0x03ae, nil],
      0x038a => [0, 0, "\316\231\314\201", "\316\231\314\201", nil, 0x03af, nil],
      0x038c => [0, 0, "\316\237\314\201", "\316\237\314\201", nil, 0x03cc, nil],
      0x038e => [0, 0, "\316\245\314\201", "\316\245\314\201", nil, 0x03cd, nil],
      0x038f => [0, 0, "\316\251\314\201", "\316\251\314\201", nil, 0x03ce, nil],
      0x0390 => [0, 0, "\317\212\314\201", "\317\212\314\201", nil, nil, nil],
      0x0391 => [0, 0, nil, nil, nil, 0x03b1, nil],
      0x0392 => [0, 0, nil, nil, nil, 0x03b2, nil],
      0x0393 => [0, 0, nil, nil, nil, 0x03b3, nil],
      0x0394 => [0, 0, nil, nil, nil, 0x03b4, nil],
      0x0395 => [0, 0, nil, nil, nil, 0x03b5, nil],
      0x0396 => [0, 0, nil, nil, nil, 0x03b6, nil],
      0x0397 => [0, 0, nil, nil, nil, 0x03b7, nil],
      0x0398 => [0, 0, nil, nil, nil, 0x03b8, nil],
      0x0399 => [0, 0, nil, nil, nil, 0x03b9, nil],
      0x039a => [0, 0, nil, nil, nil, 0x03ba, nil],
      0x039b => [0, 0, nil, nil, nil, 0x03bb, nil],
      0x039c => [0, 0, nil, nil, nil, 0x03bc, nil],
      0x039d => [0, 0, nil, nil, nil, 0x03bd, nil],
      0x039e => [0, 0, nil, nil, nil, 0x03be, nil],
      0x039f => [0, 0, nil, nil, nil, 0x03bf, nil],
      0x03a0 => [0, 0, nil, nil, nil, 0x03c0, nil],
      0x03a1 => [0, 0, nil, nil, nil, 0x03c1, nil],
      0x03a3 => [0, 0, nil, nil, nil, 0x03c3, nil],
      0x03a4 => [0, 0, nil, nil, nil, 0x03c4, nil],
      0x03a5 => [0, 0, nil, nil, nil, 0x03c5, nil],
      0x03a6 => [0, 0, nil, nil, nil, 0x03c6, nil],
      0x03a7 => [0, 0, nil, nil, nil, 0x03c7, nil],
      0x03a8 => [0, 0, nil, nil, nil, 0x03c8, nil],
      0x03a9 => [0, 0, nil, nil, nil, 0x03c9, nil],
      0x03aa => [0, 0, "\316\231\314\210", "\316\231\314\210", nil, 0x03ca, nil],
      0x03ab => [0, 0, "\316\245\314\210", "\316\245\314\210", nil, 0x03cb, nil],
      0x03ac => [0, 0, "\316\261\314\201", "\316\261\314\201", 0x0386, nil, 0x0386],
      0x03ad => [0, 0, "\316\265\314\201", "\316\265\314\201", 0x0388, nil, 0x0388],
      0x03ae => [0, 0, "\316\267\314\201", "\316\267\314\201", 0x0389, nil, 0x0389],
      0x03af => [0, 0, "\316\271\314\201", "\316\271\314\201", 0x038a, nil, 0x038a],
      0x03b0 => [0, 0, "\317\213\314\201", "\317\213\314\201", nil, nil, nil],
      0x03b1 => [0, 0, nil, nil, 0x0391, nil, 0x0391],
      0x03b2 => [0, 0, nil, nil, 0x0392, nil, 0x0392],
      0x03b3 => [0, 0, nil, nil, 0x0393, nil, 0x0393],
      0x03b4 => [0, 0, nil, nil, 0x0394, nil, 0x0394],
      0x03b5 => [0, 0, nil, nil, 0x0395, nil, 0x0395],
      0x03b6 => [0, 0, nil, nil, 0x0396, nil, 0x0396],
      0x03b7 => [0, 0, nil, nil, 0x0397, nil, 0x0397],
      0x03b8 => [0, 0, nil, nil, 0x0398, nil, 0x0398],
      0x03b9 => [0, 0, nil, nil, 0x0399, nil, 0x0399],
      0x03ba => [0, 0, nil, nil, 0x039a, nil, 0x039a],
      0x03bb => [0, 0, nil, nil, 0x039b, nil, 0x039b],
      0x03bc => [0, 0, nil, nil, 0x039c, nil, 0x039c],
      0x03bd => [0, 0, nil, nil, 0x039d, nil, 0x039d],
      0x03be => [0, 0, nil, nil, 0x039e, nil, 0x039e],
      0x03bf => [0, 0, nil, nil, 0x039f, nil, 0x039f],
      0x03c0 => [0, 0, nil, nil, 0x03a0, nil, 0x03a0],
      0x03c1 => [0, 0, nil, nil, 0x03a1, nil, 0x03a1],
      0x03c2 => [0, 0, nil, nil, 0x03a3, nil, 0x03a3],
      0x03c3 => [0, 0, nil, nil, 0x03a3, nil, 0x03a3],
      0x03c4 => [0, 0, nil, nil, 0x03a4, nil, 0x03a4],
      0x03c5 => [0, 0, nil, nil, 0x03a5, nil, 0x03a5],
      0x03c6 => [0, 0, nil, nil, 0x03a6, nil, 0x03a6],
      0x03c7 => [0, 0, nil, nil, 0x03a7, nil, 0x03a7],
      0x03c8 => [0, 0, nil, nil, 0x03a8, nil, 0x03a8],
      0x03c9 => [0, 0, nil, nil, 0x03a9, nil, 0x03a9],
      0x03ca => [0, 0, "\316\271\314\210", "\316\271\314\210", 0x03aa, nil, 0x03aa],
      0x03cb => [0, 0, "\317\205\314\210", "\317\205\314\210", 0x03ab, nil, 0x03ab],
      0x03cc => [0, 0, "\316\277\314\201", "\316\277\314\201", 0x038c, nil, 0x038c],
      0x03cd => [0, 0, "\317\205\314\201", "\317\205\314\201", 0x038e, nil, 0x038e],
      0x03ce => [0, 0, "\317\211\314\201", "\317\211\314\201", 0x038f, nil, 0x038f],
      0x03d0 => [0, 0, nil, "\316\262", 0x0392, nil, 0x0392],
      0x03d1 => [0, 0, nil, "\316\270", 0x0398, nil, 0x0398],
      0x03d2 => [0, 0, nil, "\316\245", nil, nil, nil],
      0x03d3 => [0, 0, "\317\222\314\201", "\317\222\314\201", nil, nil, nil],
      0x03d4 => [0, 0, "\317\222\314\210", "\317\222\314\210", nil, nil, nil],
      0x03d5 => [0, 0, nil, "\317\206", 0x03a6, nil, 0x03a6],
      0x03d6 => [0, 0, nil, "\317\200", 0x03a0, nil, 0x03a0],
      0x03da => [0, 0, nil, nil, nil, 0x03db, nil],
      0x03db => [0, 0, nil, nil, 0x03da, nil, 0x03da],
      0x03dc => [0, 0, nil, nil, nil, 0x03dd, nil],
      0x03dd => [0, 0, nil, nil, 0x03dc, nil, 0x03dc],
      0x03de => [0, 0, nil, nil, nil, 0x03df, nil],
      0x03df => [0, 0, nil, nil, 0x03de, nil, 0x03de],
      0x03e0 => [0, 0, nil, nil, nil, 0x03e1, nil],
      0x03e1 => [0, 0, nil, nil, 0x03e0, nil, 0x03e0],
      0x03e2 => [0, 0, nil, nil, nil, 0x03e3, nil],
      0x03e3 => [0, 0, nil, nil, 0x03e2, nil, 0x03e2],
      0x03e4 => [0, 0, nil, nil, nil, 0x03e5, nil],
      0x03e5 => [0, 0, nil, nil, 0x03e4, nil, 0x03e4],
      0x03e6 => [0, 0, nil, nil, nil, 0x03e7, nil],
      0x03e7 => [0, 0, nil, nil, 0x03e6, nil, 0x03e6],
      0x03e8 => [0, 0, nil, nil, nil, 0x03e9, nil],
      0x03e9 => [0, 0, nil, nil, 0x03e8, nil, 0x03e8],
      0x03ea => [0, 0, nil, nil, nil, 0x03eb, nil],
      0x03eb => [0, 0, nil, nil, 0x03ea, nil, 0x03ea],
      0x03ec => [0, 0, nil, nil, nil, 0x03ed, nil],
      0x03ed => [0, 0, nil, nil, 0x03ec, nil, 0x03ec],
      0x03ee => [0, 0, nil, nil, nil, 0x03ef, nil],
      0x03ef => [0, 0, nil, nil, 0x03ee, nil, 0x03ee],
      0x03f0 => [0, 0, nil, "\316\272", 0x039a, nil, 0x039a],
      0x03f1 => [0, 0, nil, "\317\201", 0x03a1, nil, 0x03a1],
      0x03f2 => [0, 0, nil, "\317\202", 0x03a3, nil, 0x03a3],
      0x0400 => [0, 0, "\320\225\314\200", "\320\225\314\200", nil, 0x0450, nil],
      0x0401 => [0, 0, "\320\225\314\210", "\320\225\314\210", nil, 0x0451, nil],
      0x0402 => [0, 0, nil, nil, nil, 0x0452, nil],
      0x0403 => [0, 0, "\320\223\314\201", "\320\223\314\201", nil, 0x0453, nil],
      0x0404 => [0, 0, nil, nil, nil, 0x0454, nil],
      0x0405 => [0, 0, nil, nil, nil, 0x0455, nil],
      0x0406 => [0, 0, nil, nil, nil, 0x0456, nil],
      0x0407 => [0, 0, "\320\206\314\210", "\320\206\314\210", nil, 0x0457, nil],
      0x0408 => [0, 0, nil, nil, nil, 0x0458, nil],
      0x0409 => [0, 0, nil, nil, nil, 0x0459, nil],
      0x040a => [0, 0, nil, nil, nil, 0x045a, nil],
      0x040b => [0, 0, nil, nil, nil, 0x045b, nil],
      0x040c => [0, 0, "\320\232\314\201", "\320\232\314\201", nil, 0x045c, nil],
      0x040d => [0, 0, "\320\230\314\200", "\320\230\314\200", nil, 0x045d, nil],
      0x040e => [0, 0, "\320\243\314\206", "\320\243\314\206", nil, 0x045e, nil],
      0x040f => [0, 0, nil, nil, nil, 0x045f, nil],
      0x0410 => [0, 0, nil, nil, nil, 0x0430, nil],
      0x0411 => [0, 0, nil, nil, nil, 0x0431, nil],
      0x0412 => [0, 0, nil, nil, nil, 0x0432, nil],
      0x0413 => [0, 0, nil, nil, nil, 0x0433, nil],
      0x0414 => [0, 0, nil, nil, nil, 0x0434, nil],
      0x0415 => [0, 0, nil, nil, nil, 0x0435, nil],
      0x0416 => [0, 0, nil, nil, nil, 0x0436, nil],
      0x0417 => [0, 0, nil, nil, nil, 0x0437, nil],
      0x0418 => [0, 0, nil, nil, nil, 0x0438, nil],
      0x0419 => [0, 0, "\320\230\314\206", "\320\230\314\206", nil, 0x0439, nil],
      0x041a => [0, 0, nil, nil, nil, 0x043a, nil],
      0x041b => [0, 0, nil, nil, nil, 0x043b, nil],
      0x041c => [0, 0, nil, nil, nil, 0x043c, nil],
      0x041d => [0, 0, nil, nil, nil, 0x043d, nil],
      0x041e => [0, 0, nil, nil, nil, 0x043e, nil],
      0x041f => [0, 0, nil, nil, nil, 0x043f, nil],
      0x0420 => [0, 0, nil, nil, nil, 0x0440, nil],
      0x0421 => [0, 0, nil, nil, nil, 0x0441, nil],
      0x0422 => [0, 0, nil, nil, nil, 0x0442, nil],
      0x0423 => [0, 0, nil, nil, nil, 0x0443, nil],
      0x0424 => [0, 0, nil, nil, nil, 0x0444, nil],
      0x0425 => [0, 0, nil, nil, nil, 0x0445, nil],
      0x0426 => [0, 0, nil, nil, nil, 0x0446, nil],
      0x0427 => [0, 0, nil, nil, nil, 0x0447, nil],
      0x0428 => [0, 0, nil, nil, nil, 0x0448, nil],
      0x0429 => [0, 0, nil, nil, nil, 0x0449, nil],
      0x042a => [0, 0, nil, nil, nil, 0x044a, nil],
      0x042b => [0, 0, nil, nil, nil, 0x044b, nil],
      0x042c => [0, 0, nil, nil, nil, 0x044c, nil],
      0x042d => [0, 0, nil, nil, nil, 0x044d, nil],
      0x042e => [0, 0, nil, nil, nil, 0x044e, nil],
      0x042f => [0, 0, nil, nil, nil, 0x044f, nil],
      0x0430 => [0, 0, nil, nil, 0x0410, nil, 0x0410],
      0x0431 => [0, 0, nil, nil, 0x0411, nil, 0x0411],
      0x0432 => [0, 0, nil, nil, 0x0412, nil, 0x0412],
      0x0433 => [0, 0, nil, nil, 0x0413, nil, 0x0413],
      0x0434 => [0, 0, nil, nil, 0x0414, nil, 0x0414],
      0x0435 => [0, 0, nil, nil, 0x0415, nil, 0x0415],
      0x0436 => [0, 0, nil, nil, 0x0416, nil, 0x0416],
      0x0437 => [0, 0, nil, nil, 0x0417, nil, 0x0417],
      0x0438 => [0, 0, nil, nil, 0x0418, nil, 0x0418],
      0x0439 => [0, 0, "\320\270\314\206", "\320\270\314\206", 0x0419, nil, 0x0419],
      0x043a => [0, 0, nil, nil, 0x041a, nil, 0x041a],
      0x043b => [0, 0, nil, nil, 0x041b, nil, 0x041b],
      0x043c => [0, 0, nil, nil, 0x041c, nil, 0x041c],
      0x043d => [0, 0, nil, nil, 0x041d, nil, 0x041d],
      0x043e => [0, 0, nil, nil, 0x041e, nil, 0x041e],
      0x043f => [0, 0, nil, nil, 0x041f, nil, 0x041f],
      0x0440 => [0, 0, nil, nil, 0x0420, nil, 0x0420],
      0x0441 => [0, 0, nil, nil, 0x0421, nil, 0x0421],
      0x0442 => [0, 0, nil, nil, 0x0422, nil, 0x0422],
      0x0443 => [0, 0, nil, nil, 0x0423, nil, 0x0423],
      0x0444 => [0, 0, nil, nil, 0x0424, nil, 0x0424],
      0x0445 => [0, 0, nil, nil, 0x0425, nil, 0x0425],
      0x0446 => [0, 0, nil, nil, 0x0426, nil, 0x0426],
      0x0447 => [0, 0, nil, nil, 0x0427, nil, 0x0427],
      0x0448 => [0, 0, nil, nil, 0x0428, nil, 0x0428],
      0x0449 => [0, 0, nil, nil, 0x0429, nil, 0x0429],
      0x044a => [0, 0, nil, nil, 0x042a, nil, 0x042a],
      0x044b => [0, 0, nil, nil, 0x042b, nil, 0x042b],
      0x044c => [0, 0, nil, nil, 0x042c, nil, 0x042c],
      0x044d => [0, 0, nil, nil, 0x042d, nil, 0x042d],
      0x044e => [0, 0, nil, nil, 0x042e, nil, 0x042e],
      0x044f => [0, 0, nil, nil, 0x042f, nil, 0x042f],
      0x0450 => [0, 0, "\320\265\314\200", "\320\265\314\200", 0x0400, nil, 0x0400],
      0x0451 => [0, 0, "\320\265\314\210", "\320\265\314\210", 0x0401, nil, 0x0401],
      0x0452 => [0, 0, nil, nil, 0x0402, nil, 0x0402],
      0x0453 => [0, 0, "\320\263\314\201", "\320\263\314\201", 0x0403, nil, 0x0403],
      0x0454 => [0, 0, nil, nil, 0x0404, nil, 0x0404],
      0x0455 => [0, 0, nil, nil, 0x0405, nil, 0x0405],
      0x0456 => [0, 0, nil, nil, 0x0406, nil, 0x0406],
      0x0457 => [0, 0, "\321\226\314\210", "\321\226\314\210", 0x0407, nil, 0x0407],
      0x0458 => [0, 0, nil, nil, 0x0408, nil, 0x0408],
      0x0459 => [0, 0, nil, nil, 0x0409, nil, 0x0409],
      0x045a => [0, 0, nil, nil, 0x040a, nil, 0x040a],
      0x045b => [0, 0, nil, nil, 0x040b, nil, 0x040b],
      0x045c => [0, 0, "\320\272\314\201", "\320\272\314\201", 0x040c, nil, 0x040c],
      0x045d => [0, 0, "\320\270\314\200", "\320\270\314\200", 0x040d, nil, 0x040d],
      0x045e => [0, 0, "\321\203\314\206", "\321\203\314\206", 0x040e, nil, 0x040e],
      0x045f => [0, 0, nil, nil, 0x040f, nil, 0x040f],
      0x0460 => [0, 0, nil, nil, nil, 0x0461, nil],
      0x0461 => [0, 0, nil, nil, 0x0460, nil, 0x0460],
      0x0462 => [0, 0, nil, nil, nil, 0x0463, nil],
      0x0463 => [0, 0, nil, nil, 0x0462, nil, 0x0462],
      0x0464 => [0, 0, nil, nil, nil, 0x0465, nil],
      0x0465 => [0, 0, nil, nil, 0x0464, nil, 0x0464],
      0x0466 => [0, 0, nil, nil, nil, 0x0467, nil],
      0x0467 => [0, 0, nil, nil, 0x0466, nil, 0x0466],
      0x0468 => [0, 0, nil, nil, nil, 0x0469, nil],
      0x0469 => [0, 0, nil, nil, 0x0468, nil, 0x0468],
      0x046a => [0, 0, nil, nil, nil, 0x046b, nil],
      0x046b => [0, 0, nil, nil, 0x046a, nil, 0x046a],
      0x046c => [0, 0, nil, nil, nil, 0x046d, nil],
      0x046d => [0, 0, nil, nil, 0x046c, nil, 0x046c],
      0x046e => [0, 0, nil, nil, nil, 0x046f, nil],
      0x046f => [0, 0, nil, nil, 0x046e, nil, 0x046e],
      0x0470 => [0, 0, nil, nil, nil, 0x0471, nil],
      0x0471 => [0, 0, nil, nil, 0x0470, nil, 0x0470],
      0x0472 => [0, 0, nil, nil, nil, 0x0473, nil],
      0x0473 => [0, 0, nil, nil, 0x0472, nil, 0x0472],
      0x0474 => [0, 0, nil, nil, nil, 0x0475, nil],
      0x0475 => [0, 0, nil, nil, 0x0474, nil, 0x0474],
      0x0476 => [0, 0, "\321\264\314\217", "\321\264\314\217", nil, 0x0477, nil],
      0x0477 => [0, 0, "\321\265\314\217", "\321\265\314\217", 0x0476, nil, 0x0476],
      0x0478 => [0, 0, nil, nil, nil, 0x0479, nil],
      0x0479 => [0, 0, nil, nil, 0x0478, nil, 0x0478],
      0x047a => [0, 0, nil, nil, nil, 0x047b, nil],
      0x047b => [0, 0, nil, nil, 0x047a, nil, 0x047a],
      0x047c => [0, 0, nil, nil, nil, 0x047d, nil],
      0x047d => [0, 0, nil, nil, 0x047c, nil, 0x047c],
      0x047e => [0, 0, nil, nil, nil, 0x047f, nil],
      0x047f => [0, 0, nil, nil, 0x047e, nil, 0x047e],
      0x0480 => [0, 0, nil, nil, nil, 0x0481, nil],
      0x0481 => [0, 0, nil, nil, 0x0480, nil, 0x0480],
      0x0483 => [230, 0, nil, nil, nil, nil, nil],
      0x0484 => [230, 0, nil, nil, nil, nil, nil],
      0x0485 => [230, 0, nil, nil, nil, nil, nil],
      0x0486 => [230, 0, nil, nil, nil, nil, nil],
      0x048c => [0, 0, nil, nil, nil, 0x048d, nil],
      0x048d => [0, 0, nil, nil, 0x048c, nil, 0x048c],
      0x048e => [0, 0, nil, nil, nil, 0x048f, nil],
      0x048f => [0, 0, nil, nil, 0x048e, nil, 0x048e],
      0x0490 => [0, 0, nil, nil, nil, 0x0491, nil],
      0x0491 => [0, 0, nil, nil, 0x0490, nil, 0x0490],
      0x0492 => [0, 0, nil, nil, nil, 0x0493, nil],
      0x0493 => [0, 0, nil, nil, 0x0492, nil, 0x0492],
      0x0494 => [0, 0, nil, nil, nil, 0x0495, nil],
      0x0495 => [0, 0, nil, nil, 0x0494, nil, 0x0494],
      0x0496 => [0, 0, nil, nil, nil, 0x0497, nil],
      0x0497 => [0, 0, nil, nil, 0x0496, nil, 0x0496],
      0x0498 => [0, 0, nil, nil, nil, 0x0499, nil],
      0x0499 => [0, 0, nil, nil, 0x0498, nil, 0x0498],
      0x049a => [0, 0, nil, nil, nil, 0x049b, nil],
      0x049b => [0, 0, nil, nil, 0x049a, nil, 0x049a],
      0x049c => [0, 0, nil, nil, nil, 0x049d, nil],
      0x049d => [0, 0, nil, nil, 0x049c, nil, 0x049c],
      0x049e => [0, 0, nil, nil, nil, 0x049f, nil],
      0x049f => [0, 0, nil, nil, 0x049e, nil, 0x049e],
      0x04a0 => [0, 0, nil, nil, nil, 0x04a1, nil],
      0x04a1 => [0, 0, nil, nil, 0x04a0, nil, 0x04a0],
      0x04a2 => [0, 0, nil, nil, nil, 0x04a3, nil],
      0x04a3 => [0, 0, nil, nil, 0x04a2, nil, 0x04a2],
      0x04a4 => [0, 0, nil, nil, nil, 0x04a5, nil],
      0x04a5 => [0, 0, nil, nil, 0x04a4, nil, 0x04a4],
      0x04a6 => [0, 0, nil, nil, nil, 0x04a7, nil],
      0x04a7 => [0, 0, nil, nil, 0x04a6, nil, 0x04a6],
      0x04a8 => [0, 0, nil, nil, nil, 0x04a9, nil],
      0x04a9 => [0, 0, nil, nil, 0x04a8, nil, 0x04a8],
      0x04aa => [0, 0, nil, nil, nil, 0x04ab, nil],
      0x04ab => [0, 0, nil, nil, 0x04aa, nil, 0x04aa],
      0x04ac => [0, 0, nil, nil, nil, 0x04ad, nil],
      0x04ad => [0, 0, nil, nil, 0x04ac, nil, 0x04ac],
      0x04ae => [0, 0, nil, nil, nil, 0x04af, nil],
      0x04af => [0, 0, nil, nil, 0x04ae, nil, 0x04ae],
      0x04b0 => [0, 0, nil, nil, nil, 0x04b1, nil],
      0x04b1 => [0, 0, nil, nil, 0x04b0, nil, 0x04b0],
      0x04b2 => [0, 0, nil, nil, nil, 0x04b3, nil],
      0x04b3 => [0, 0, nil, nil, 0x04b2, nil, 0x04b2],
      0x04b4 => [0, 0, nil, nil, nil, 0x04b5, nil],
      0x04b5 => [0, 0, nil, nil, 0x04b4, nil, 0x04b4],
      0x04b6 => [0, 0, nil, nil, nil, 0x04b7, nil],
      0x04b7 => [0, 0, nil, nil, 0x04b6, nil, 0x04b6],
      0x04b8 => [0, 0, nil, nil, nil, 0x04b9, nil],
      0x04b9 => [0, 0, nil, nil, 0x04b8, nil, 0x04b8],
      0x04ba => [0, 0, nil, nil, nil, 0x04bb, nil],
      0x04bb => [0, 0, nil, nil, 0x04ba, nil, 0x04ba],
      0x04bc => [0, 0, nil, nil, nil, 0x04bd, nil],
      0x04bd => [0, 0, nil, nil, 0x04bc, nil, 0x04bc],
      0x04be => [0, 0, nil, nil, nil, 0x04bf, nil],
      0x04bf => [0, 0, nil, nil, 0x04be, nil, 0x04be],
      0x04c1 => [0, 0, "\320\226\314\206", "\320\226\314\206", nil, 0x04c2, nil],
      0x04c2 => [0, 0, "\320\266\314\206", "\320\266\314\206", 0x04c1, nil, 0x04c1],
      0x04c3 => [0, 0, nil, nil, nil, 0x04c4, nil],
      0x04c4 => [0, 0, nil, nil, 0x04c3, nil, 0x04c3],
      0x04c7 => [0, 0, nil, nil, nil, 0x04c8, nil],
      0x04c8 => [0, 0, nil, nil, 0x04c7, nil, 0x04c7],
      0x04cb => [0, 0, nil, nil, nil, 0x04cc, nil],
      0x04cc => [0, 0, nil, nil, 0x04cb, nil, 0x04cb],
      0x04d0 => [0, 0, "\320\220\314\206", "\320\220\314\206", nil, 0x04d1, nil],
      0x04d1 => [0, 0, "\320\260\314\206", "\320\260\314\206", 0x04d0, nil, 0x04d0],
      0x04d2 => [0, 0, "\320\220\314\210", "\320\220\314\210", nil, 0x04d3, nil],
      0x04d3 => [0, 0, "\320\260\314\210", "\320\260\314\210", 0x04d2, nil, 0x04d2],
      0x04d4 => [0, 0, nil, nil, nil, 0x04d5, nil],
      0x04d5 => [0, 0, nil, nil, 0x04d4, nil, 0x04d4],
      0x04d6 => [0, 0, "\320\225\314\206", "\320\225\314\206", nil, 0x04d7, nil],
      0x04d7 => [0, 0, "\320\265\314\206", "\320\265\314\206", 0x04d6, nil, 0x04d6],
      0x04d8 => [0, 0, nil, nil, nil, 0x04d9, nil],
      0x04d9 => [0, 0, nil, nil, 0x04d8, nil, 0x04d8],
      0x04da => [0, 0, "\323\230\314\210", "\323\230\314\210", nil, 0x04db, nil],
      0x04db => [0, 0, "\323\231\314\210", "\323\231\314\210", 0x04da, nil, 0x04da],
      0x04dc => [0, 0, "\320\226\314\210", "\320\226\314\210", nil, 0x04dd, nil],
      0x04dd => [0, 0, "\320\266\314\210", "\320\266\314\210", 0x04dc, nil, 0x04dc],
      0x04de => [0, 0, "\320\227\314\210", "\320\227\314\210", nil, 0x04df, nil],
      0x04df => [0, 0, "\320\267\314\210", "\320\267\314\210", 0x04de, nil, 0x04de],
      0x04e0 => [0, 0, nil, nil, nil, 0x04e1, nil],
      0x04e1 => [0, 0, nil, nil, 0x04e0, nil, 0x04e0],
      0x04e2 => [0, 0, "\320\230\314\204", "\320\230\314\204", nil, 0x04e3, nil],
      0x04e3 => [0, 0, "\320\270\314\204", "\320\270\314\204", 0x04e2, nil, 0x04e2],
      0x04e4 => [0, 0, "\320\230\314\210", "\320\230\314\210", nil, 0x04e5, nil],
      0x04e5 => [0, 0, "\320\270\314\210", "\320\270\314\210", 0x04e4, nil, 0x04e4],
      0x04e6 => [0, 0, "\320\236\314\210", "\320\236\314\210", nil, 0x04e7, nil],
      0x04e7 => [0, 0, "\320\276\314\210", "\320\276\314\210", 0x04e6, nil, 0x04e6],
      0x04e8 => [0, 0, nil, nil, nil, 0x04e9, nil],
      0x04e9 => [0, 0, nil, nil, 0x04e8, nil, 0x04e8],
      0x04ea => [0, 0, "\323\250\314\210", "\323\250\314\210", nil, 0x04eb, nil],
      0x04eb => [0, 0, "\323\251\314\210", "\323\251\314\210", 0x04ea, nil, 0x04ea],
      0x04ec => [0, 0, "\320\255\314\210", "\320\255\314\210", nil, 0x04ed, nil],
      0x04ed => [0, 0, "\321\215\314\210", "\321\215\314\210", 0x04ec, nil, 0x04ec],
      0x04ee => [0, 0, "\320\243\314\204", "\320\243\314\204", nil, 0x04ef, nil],
      0x04ef => [0, 0, "\321\203\314\204", "\321\203\314\204", 0x04ee, nil, 0x04ee],
      0x04f0 => [0, 0, "\320\243\314\210", "\320\243\314\210", nil, 0x04f1, nil],
      0x04f1 => [0, 0, "\321\203\314\210", "\321\203\314\210", 0x04f0, nil, 0x04f0],
      0x04f2 => [0, 0, "\320\243\314\213", "\320\243\314\213", nil, 0x04f3, nil],
      0x04f3 => [0, 0, "\321\203\314\213", "\321\203\314\213", 0x04f2, nil, 0x04f2],
      0x04f4 => [0, 0, "\320\247\314\210", "\320\247\314\210", nil, 0x04f5, nil],
      0x04f5 => [0, 0, "\321\207\314\210", "\321\207\314\210", 0x04f4, nil, 0x04f4],
      0x04f8 => [0, 0, "\320\253\314\210", "\320\253\314\210", nil, 0x04f9, nil],
      0x04f9 => [0, 0, "\321\213\314\210", "\321\213\314\210", 0x04f8, nil, 0x04f8],
      0x0531 => [0, 0, nil, nil, nil, 0x0561, nil],
      0x0532 => [0, 0, nil, nil, nil, 0x0562, nil],
      0x0533 => [0, 0, nil, nil, nil, 0x0563, nil],
      0x0534 => [0, 0, nil, nil, nil, 0x0564, nil],
      0x0535 => [0, 0, nil, nil, nil, 0x0565, nil],
      0x0536 => [0, 0, nil, nil, nil, 0x0566, nil],
      0x0537 => [0, 0, nil, nil, nil, 0x0567, nil],
      0x0538 => [0, 0, nil, nil, nil, 0x0568, nil],
      0x0539 => [0, 0, nil, nil, nil, 0x0569, nil],
      0x053a => [0, 0, nil, nil, nil, 0x056a, nil],
      0x053b => [0, 0, nil, nil, nil, 0x056b, nil],
      0x053c => [0, 0, nil, nil, nil, 0x056c, nil],
      0x053d => [0, 0, nil, nil, nil, 0x056d, nil],
      0x053e => [0, 0, nil, nil, nil, 0x056e, nil],
      0x053f => [0, 0, nil, nil, nil, 0x056f, nil],
      0x0540 => [0, 0, nil, nil, nil, 0x0570, nil],
      0x0541 => [0, 0, nil, nil, nil, 0x0571, nil],
      0x0542 => [0, 0, nil, nil, nil, 0x0572, nil],
      0x0543 => [0, 0, nil, nil, nil, 0x0573, nil],
      0x0544 => [0, 0, nil, nil, nil, 0x0574, nil],
      0x0545 => [0, 0, nil, nil, nil, 0x0575, nil],
      0x0546 => [0, 0, nil, nil, nil, 0x0576, nil],
      0x0547 => [0, 0, nil, nil, nil, 0x0577, nil],
      0x0548 => [0, 0, nil, nil, nil, 0x0578, nil],
      0x0549 => [0, 0, nil, nil, nil, 0x0579, nil],
      0x054a => [0, 0, nil, nil, nil, 0x057a, nil],
      0x054b => [0, 0, nil, nil, nil, 0x057b, nil],
      0x054c => [0, 0, nil, nil, nil, 0x057c, nil],
      0x054d => [0, 0, nil, nil, nil, 0x057d, nil],
      0x054e => [0, 0, nil, nil, nil, 0x057e, nil],
      0x054f => [0, 0, nil, nil, nil, 0x057f, nil],
      0x0550 => [0, 0, nil, nil, nil, 0x0580, nil],
      0x0551 => [0, 0, nil, nil, nil, 0x0581, nil],
      0x0552 => [0, 0, nil, nil, nil, 0x0582, nil],
      0x0553 => [0, 0, nil, nil, nil, 0x0583, nil],
      0x0554 => [0, 0, nil, nil, nil, 0x0584, nil],
      0x0555 => [0, 0, nil, nil, nil, 0x0585, nil],
      0x0556 => [0, 0, nil, nil, nil, 0x0586, nil],
      0x0561 => [0, 0, nil, nil, 0x0531, nil, 0x0531],
      0x0562 => [0, 0, nil, nil, 0x0532, nil, 0x0532],
      0x0563 => [0, 0, nil, nil, 0x0533, nil, 0x0533],
      0x0564 => [0, 0, nil, nil, 0x0534, nil, 0x0534],
      0x0565 => [0, 0, nil, nil, 0x0535, nil, 0x0535],
      0x0566 => [0, 0, nil, nil, 0x0536, nil, 0x0536],
      0x0567 => [0, 0, nil, nil, 0x0537, nil, 0x0537],
      0x0568 => [0, 0, nil, nil, 0x0538, nil, 0x0538],
      0x0569 => [0, 0, nil, nil, 0x0539, nil, 0x0539],
      0x056a => [0, 0, nil, nil, 0x053a, nil, 0x053a],
      0x056b => [0, 0, nil, nil, 0x053b, nil, 0x053b],
      0x056c => [0, 0, nil, nil, 0x053c, nil, 0x053c],
      0x056d => [0, 0, nil, nil, 0x053d, nil, 0x053d],
      0x056e => [0, 0, nil, nil, 0x053e, nil, 0x053e],
      0x056f => [0, 0, nil, nil, 0x053f, nil, 0x053f],
      0x0570 => [0, 0, nil, nil, 0x0540, nil, 0x0540],
      0x0571 => [0, 0, nil, nil, 0x0541, nil, 0x0541],
      0x0572 => [0, 0, nil, nil, 0x0542, nil, 0x0542],
      0x0573 => [0, 0, nil, nil, 0x0543, nil, 0x0543],
      0x0574 => [0, 0, nil, nil, 0x0544, nil, 0x0544],
      0x0575 => [0, 0, nil, nil, 0x0545, nil, 0x0545],
      0x0576 => [0, 0, nil, nil, 0x0546, nil, 0x0546],
      0x0577 => [0, 0, nil, nil, 0x0547, nil, 0x0547],
      0x0578 => [0, 0, nil, nil, 0x0548, nil, 0x0548],
      0x0579 => [0, 0, nil, nil, 0x0549, nil, 0x0549],
      0x057a => [0, 0, nil, nil, 0x054a, nil, 0x054a],
      0x057b => [0, 0, nil, nil, 0x054b, nil, 0x054b],
      0x057c => [0, 0, nil, nil, 0x054c, nil, 0x054c],
      0x057d => [0, 0, nil, nil, 0x054d, nil, 0x054d],
      0x057e => [0, 0, nil, nil, 0x054e, nil, 0x054e],
      0x057f => [0, 0, nil, nil, 0x054f, nil, 0x054f],
      0x0580 => [0, 0, nil, nil, 0x0550, nil, 0x0550],
      0x0581 => [0, 0, nil, nil, 0x0551, nil, 0x0551],
      0x0582 => [0, 0, nil, nil, 0x0552, nil, 0x0552],
      0x0583 => [0, 0, nil, nil, 0x0553, nil, 0x0553],
      0x0584 => [0, 0, nil, nil, 0x0554, nil, 0x0554],
      0x0585 => [0, 0, nil, nil, 0x0555, nil, 0x0555],
      0x0586 => [0, 0, nil, nil, 0x0556, nil, 0x0556],
      0x0587 => [0, 0, nil, "\325\245\326\202", nil, nil, nil],
      0x0591 => [220, 0, nil, nil, nil, nil, nil],
      0x0592 => [230, 0, nil, nil, nil, nil, nil],
      0x0593 => [230, 0, nil, nil, nil, nil, nil],
      0x0594 => [230, 0, nil, nil, nil, nil, nil],
      0x0595 => [230, 0, nil, nil, nil, nil, nil],
      0x0596 => [220, 0, nil, nil, nil, nil, nil],
      0x0597 => [230, 0, nil, nil, nil, nil, nil],
      0x0598 => [230, 0, nil, nil, nil, nil, nil],
      0x0599 => [230, 0, nil, nil, nil, nil, nil],
      0x059a => [222, 0, nil, nil, nil, nil, nil],
      0x059b => [220, 0, nil, nil, nil, nil, nil],
      0x059c => [230, 0, nil, nil, nil, nil, nil],
      0x059d => [230, 0, nil, nil, nil, nil, nil],
      0x059e => [230, 0, nil, nil, nil, nil, nil],
      0x059f => [230, 0, nil, nil, nil, nil, nil],
      0x05a0 => [230, 0, nil, nil, nil, nil, nil],
      0x05a1 => [230, 0, nil, nil, nil, nil, nil],
      0x05a3 => [220, 0, nil, nil, nil, nil, nil],
      0x05a4 => [220, 0, nil, nil, nil, nil, nil],
      0x05a5 => [220, 0, nil, nil, nil, nil, nil],
      0x05a6 => [220, 0, nil, nil, nil, nil, nil],
      0x05a7 => [220, 0, nil, nil, nil, nil, nil],
      0x05a8 => [230, 0, nil, nil, nil, nil, nil],
      0x05a9 => [230, 0, nil, nil, nil, nil, nil],
      0x05aa => [220, 0, nil, nil, nil, nil, nil],
      0x05ab => [230, 0, nil, nil, nil, nil, nil],
      0x05ac => [230, 0, nil, nil, nil, nil, nil],
      0x05ad => [222, 0, nil, nil, nil, nil, nil],
      0x05ae => [228, 0, nil, nil, nil, nil, nil],
      0x05af => [230, 0, nil, nil, nil, nil, nil],
      0x05b0 => [10, 0, nil, nil, nil, nil, nil],
      0x05b1 => [11, 0, nil, nil, nil, nil, nil],
      0x05b2 => [12, 0, nil, nil, nil, nil, nil],
      0x05b3 => [13, 0, nil, nil, nil, nil, nil],
      0x05b4 => [14, 0, nil, nil, nil, nil, nil],
      0x05b5 => [15, 0, nil, nil, nil, nil, nil],
      0x05b6 => [16, 0, nil, nil, nil, nil, nil],
      0x05b7 => [17, 0, nil, nil, nil, nil, nil],
      0x05b8 => [18, 0, nil, nil, nil, nil, nil],
      0x05b9 => [19, 0, nil, nil, nil, nil, nil],
      0x05bb => [20, 0, nil, nil, nil, nil, nil],
      0x05bc => [21, 0, nil, nil, nil, nil, nil],
      0x05bd => [22, 0, nil, nil, nil, nil, nil],
      0x05bf => [23, 0, nil, nil, nil, nil, nil],
      0x05c1 => [24, 0, nil, nil, nil, nil, nil],
      0x05c2 => [25, 0, nil, nil, nil, nil, nil],
      0x05c4 => [230, 0, nil, nil, nil, nil, nil],
      0x0622 => [0, 0, "\330\247\331\223", "\330\247\331\223", nil, nil, nil],
      0x0623 => [0, 0, "\330\247\331\224", "\330\247\331\224", nil, nil, nil],
      0x0624 => [0, 0, "\331\210\331\224", "\331\210\331\224", nil, nil, nil],
      0x0625 => [0, 0, "\330\247\331\225", "\330\247\331\225", nil, nil, nil],
      0x0626 => [0, 0, "\331\212\331\224", "\331\212\331\224", nil, nil, nil],
      0x064b => [27, 0, nil, nil, nil, nil, nil],
      0x064c => [28, 0, nil, nil, nil, nil, nil],
      0x064d => [29, 0, nil, nil, nil, nil, nil],
      0x064e => [30, 0, nil, nil, nil, nil, nil],
      0x064f => [31, 0, nil, nil, nil, nil, nil],
      0x0650 => [32, 0, nil, nil, nil, nil, nil],
      0x0651 => [33, 0, nil, nil, nil, nil, nil],
      0x0652 => [34, 0, nil, nil, nil, nil, nil],
      0x0653 => [230, 0, nil, nil, nil, nil, nil],
      0x0654 => [230, 0, nil, nil, nil, nil, nil],
      0x0655 => [220, 0, nil, nil, nil, nil, nil],
      0x0670 => [35, 0, nil, nil, nil, nil, nil],
      0x0675 => [0, 0, nil, "\330\247\331\264", nil, nil, nil],
      0x0676 => [0, 0, nil, "\331\210\331\264", nil, nil, nil],
      0x0677 => [0, 0, nil, "\333\207\331\264", nil, nil, nil],
      0x0678 => [0, 0, nil, "\331\212\331\264", nil, nil, nil],
      0x06c0 => [0, 0, "\333\225\331\224", "\333\225\331\224", nil, nil, nil],
      0x06c2 => [0, 0, "\333\201\331\224", "\333\201\331\224", nil, nil, nil],
      0x06d3 => [0, 0, "\333\222\331\224", "\333\222\331\224", nil, nil, nil],
      0x06d6 => [230, 0, nil, nil, nil, nil, nil],
      0x06d7 => [230, 0, nil, nil, nil, nil, nil],
      0x06d8 => [230, 0, nil, nil, nil, nil, nil],
      0x06d9 => [230, 0, nil, nil, nil, nil, nil],
      0x06da => [230, 0, nil, nil, nil, nil, nil],
      0x06db => [230, 0, nil, nil, nil, nil, nil],
      0x06dc => [230, 0, nil, nil, nil, nil, nil],
      0x06df => [230, 0, nil, nil, nil, nil, nil],
      0x06e0 => [230, 0, nil, nil, nil, nil, nil],
      0x06e1 => [230, 0, nil, nil, nil, nil, nil],
      0x06e2 => [230, 0, nil, nil, nil, nil, nil],
      0x06e3 => [220, 0, nil, nil, nil, nil, nil],
      0x06e4 => [230, 0, nil, nil, nil, nil, nil],
      0x06e7 => [230, 0, nil, nil, nil, nil, nil],
      0x06e8 => [230, 0, nil, nil, nil, nil, nil],
      0x06ea => [220, 0, nil, nil, nil, nil, nil],
      0x06eb => [230, 0, nil, nil, nil, nil, nil],
      0x06ec => [230, 0, nil, nil, nil, nil, nil],
      0x06ed => [220, 0, nil, nil, nil, nil, nil],
      0x0711 => [36, 0, nil, nil, nil, nil, nil],
      0x0730 => [230, 0, nil, nil, nil, nil, nil],
      0x0731 => [220, 0, nil, nil, nil, nil, nil],
      0x0732 => [230, 0, nil, nil, nil, nil, nil],
      0x0733 => [230, 0, nil, nil, nil, nil, nil],
      0x0734 => [220, 0, nil, nil, nil, nil, nil],
      0x0735 => [230, 0, nil, nil, nil, nil, nil],
      0x0736 => [230, 0, nil, nil, nil, nil, nil],
      0x0737 => [220, 0, nil, nil, nil, nil, nil],
      0x0738 => [220, 0, nil, nil, nil, nil, nil],
      0x0739 => [220, 0, nil, nil, nil, nil, nil],
      0x073a => [230, 0, nil, nil, nil, nil, nil],
      0x073b => [220, 0, nil, nil, nil, nil, nil],
      0x073c => [220, 0, nil, nil, nil, nil, nil],
      0x073d => [230, 0, nil, nil, nil, nil, nil],
      0x073e => [220, 0, nil, nil, nil, nil, nil],
      0x073f => [230, 0, nil, nil, nil, nil, nil],
      0x0740 => [230, 0, nil, nil, nil, nil, nil],
      0x0741 => [230, 0, nil, nil, nil, nil, nil],
      0x0742 => [220, 0, nil, nil, nil, nil, nil],
      0x0743 => [230, 0, nil, nil, nil, nil, nil],
      0x0744 => [220, 0, nil, nil, nil, nil, nil],
      0x0745 => [230, 0, nil, nil, nil, nil, nil],
      0x0746 => [220, 0, nil, nil, nil, nil, nil],
      0x0747 => [230, 0, nil, nil, nil, nil, nil],
      0x0748 => [220, 0, nil, nil, nil, nil, nil],
      0x0749 => [230, 0, nil, nil, nil, nil, nil],
      0x074a => [230, 0, nil, nil, nil, nil, nil],
      0x0929 => [0, 0, "\340\244\250\340\244\274", "\340\244\250\340\244\274", nil, nil, nil],
      0x0931 => [0, 0, "\340\244\260\340\244\274", "\340\244\260\340\244\274", nil, nil, nil],
      0x0934 => [0, 0, "\340\244\263\340\244\274", "\340\244\263\340\244\274", nil, nil, nil],
      0x093c => [7, 0, nil, nil, nil, nil, nil],
      0x094d => [9, 0, nil, nil, nil, nil, nil],
      0x0951 => [230, 0, nil, nil, nil, nil, nil],
      0x0952 => [220, 0, nil, nil, nil, nil, nil],
      0x0953 => [230, 0, nil, nil, nil, nil, nil],
      0x0954 => [230, 0, nil, nil, nil, nil, nil],
      0x0958 => [0, 1, "\340\244\225\340\244\274", "\340\244\225\340\244\274", nil, nil, nil],
      0x0959 => [0, 1, "\340\244\226\340\244\274", "\340\244\226\340\244\274", nil, nil, nil],
      0x095a => [0, 1, "\340\244\227\340\244\274", "\340\244\227\340\244\274", nil, nil, nil],
      0x095b => [0, 1, "\340\244\234\340\244\274", "\340\244\234\340\244\274", nil, nil, nil],
      0x095c => [0, 1, "\340\244\241\340\244\274", "\340\244\241\340\244\274", nil, nil, nil],
      0x095d => [0, 1, "\340\244\242\340\244\274", "\340\244\242\340\244\274", nil, nil, nil],
      0x095e => [0, 1, "\340\244\253\340\244\274", "\340\244\253\340\244\274", nil, nil, nil],
      0x095f => [0, 1, "\340\244\257\340\244\274", "\340\244\257\340\244\274", nil, nil, nil],
      0x09bc => [7, 0, nil, nil, nil, nil, nil],
      0x09cb => [0, 0, "\340\247\207\340\246\276", "\340\247\207\340\246\276", nil, nil, nil],
      0x09cc => [0, 0, "\340\247\207\340\247\227", "\340\247\207\340\247\227", nil, nil, nil],
      0x09cd => [9, 0, nil, nil, nil, nil, nil],
      0x09dc => [0, 1, "\340\246\241\340\246\274", "\340\246\241\340\246\274", nil, nil, nil],
      0x09dd => [0, 1, "\340\246\242\340\246\274", "\340\246\242\340\246\274", nil, nil, nil],
      0x09df => [0, 1, "\340\246\257\340\246\274", "\340\246\257\340\246\274", nil, nil, nil],
      0x0a33 => [0, 1, "\340\250\262\340\250\274", "\340\250\262\340\250\274", nil, nil, nil],
      0x0a36 => [0, 1, "\340\250\270\340\250\274", "\340\250\270\340\250\274", nil, nil, nil],
      0x0a3c => [7, 0, nil, nil, nil, nil, nil],
      0x0a4d => [9, 0, nil, nil, nil, nil, nil],
      0x0a59 => [0, 1, "\340\250\226\340\250\274", "\340\250\226\340\250\274", nil, nil, nil],
      0x0a5a => [0, 1, "\340\250\227\340\250\274", "\340\250\227\340\250\274", nil, nil, nil],
      0x0a5b => [0, 1, "\340\250\234\340\250\274", "\340\250\234\340\250\274", nil, nil, nil],
      0x0a5e => [0, 1, "\340\250\253\340\250\274", "\340\250\253\340\250\274", nil, nil, nil],
      0x0abc => [7, 0, nil, nil, nil, nil, nil],
      0x0acd => [9, 0, nil, nil, nil, nil, nil],
      0x0b3c => [7, 0, nil, nil, nil, nil, nil],
      0x0b48 => [0, 0, "\340\255\207\340\255\226", "\340\255\207\340\255\226", nil, nil, nil],
      0x0b4b => [0, 0, "\340\255\207\340\254\276", "\340\255\207\340\254\276", nil, nil, nil],
      0x0b4c => [0, 0, "\340\255\207\340\255\227", "\340\255\207\340\255\227", nil, nil, nil],
      0x0b4d => [9, 0, nil, nil, nil, nil, nil],
      0x0b5c => [0, 1, "\340\254\241\340\254\274", "\340\254\241\340\254\274", nil, nil, nil],
      0x0b5d => [0, 1, "\340\254\242\340\254\274", "\340\254\242\340\254\274", nil, nil, nil],
      0x0b94 => [0, 0, "\340\256\222\340\257\227", "\340\256\222\340\257\227", nil, nil, nil],
      0x0bca => [0, 0, "\340\257\206\340\256\276", "\340\257\206\340\256\276", nil, nil, nil],
      0x0bcb => [0, 0, "\340\257\207\340\256\276", "\340\257\207\340\256\276", nil, nil, nil],
      0x0bcc => [0, 0, "\340\257\206\340\257\227", "\340\257\206\340\257\227", nil, nil, nil],
      0x0bcd => [9, 0, nil, nil, nil, nil, nil],
      0x0c48 => [0, 0, "\340\261\206\340\261\226", "\340\261\206\340\261\226", nil, nil, nil],
      0x0c4d => [9, 0, nil, nil, nil, nil, nil],
      0x0c55 => [84, 0, nil, nil, nil, nil, nil],
      0x0c56 => [91, 0, nil, nil, nil, nil, nil],
      0x0cc0 => [0, 0, "\340\262\277\340\263\225", "\340\262\277\340\263\225", nil, nil, nil],
      0x0cc7 => [0, 0, "\340\263\206\340\263\225", "\340\263\206\340\263\225", nil, nil, nil],
      0x0cc8 => [0, 0, "\340\263\206\340\263\226", "\340\263\206\340\263\226", nil, nil, nil],
      0x0cca => [0, 0, "\340\263\206\340\263\202", "\340\263\206\340\263\202", nil, nil, nil],
      0x0ccb => [0, 0, "\340\263\212\340\263\225", "\340\263\212\340\263\225", nil, nil, nil],
      0x0ccd => [9, 0, nil, nil, nil, nil, nil],
      0x0d4a => [0, 0, "\340\265\206\340\264\276", "\340\265\206\340\264\276", nil, nil, nil],
      0x0d4b => [0, 0, "\340\265\207\340\264\276", "\340\265\207\340\264\276", nil, nil, nil],
      0x0d4c => [0, 0, "\340\265\206\340\265\227", "\340\265\206\340\265\227", nil, nil, nil],
      0x0d4d => [9, 0, nil, nil, nil, nil, nil],
      0x0dca => [9, 0, nil, nil, nil, nil, nil],
      0x0dda => [0, 0, "\340\267\231\340\267\212", "\340\267\231\340\267\212", nil, nil, nil],
      0x0ddc => [0, 0, "\340\267\231\340\267\217", "\340\267\231\340\267\217", nil, nil, nil],
      0x0ddd => [0, 0, "\340\267\234\340\267\212", "\340\267\234\340\267\212", nil, nil, nil],
      0x0dde => [0, 0, "\340\267\231\340\267\237", "\340\267\231\340\267\237", nil, nil, nil],
      0x0e33 => [0, 0, nil, "\340\271\215\340\270\262", nil, nil, nil],
      0x0e38 => [103, 0, nil, nil, nil, nil, nil],
      0x0e39 => [103, 0, nil, nil, nil, nil, nil],
      0x0e3a => [9, 0, nil, nil, nil, nil, nil],
      0x0e48 => [107, 0, nil, nil, nil, nil, nil],
      0x0e49 => [107, 0, nil, nil, nil, nil, nil],
      0x0e4a => [107, 0, nil, nil, nil, nil, nil],
      0x0e4b => [107, 0, nil, nil, nil, nil, nil],
      0x0eb3 => [0, 0, nil, "\340\273\215\340\272\262", nil, nil, nil],
      0x0eb8 => [118, 0, nil, nil, nil, nil, nil],
      0x0eb9 => [118, 0, nil, nil, nil, nil, nil],
      0x0ec8 => [122, 0, nil, nil, nil, nil, nil],
      0x0ec9 => [122, 0, nil, nil, nil, nil, nil],
      0x0eca => [122, 0, nil, nil, nil, nil, nil],
      0x0ecb => [122, 0, nil, nil, nil, nil, nil],
      0x0edc => [0, 0, nil, "\340\272\253\340\272\231", nil, nil, nil],
      0x0edd => [0, 0, nil, "\340\272\253\340\272\241", nil, nil, nil],
      0x0f0c => [0, 0, nil, "\340\274\213", nil, nil, nil],
      0x0f18 => [220, 0, nil, nil, nil, nil, nil],
      0x0f19 => [220, 0, nil, nil, nil, nil, nil],
      0x0f35 => [220, 0, nil, nil, nil, nil, nil],
      0x0f37 => [220, 0, nil, nil, nil, nil, nil],
      0x0f39 => [216, 0, nil, nil, nil, nil, nil],
      0x0f43 => [0, 1, "\340\275\202\340\276\267", "\340\275\202\340\276\267", nil, nil, nil],
      0x0f4d => [0, 1, "\340\275\214\340\276\267", "\340\275\214\340\276\267", nil, nil, nil],
      0x0f52 => [0, 1, "\340\275\221\340\276\267", "\340\275\221\340\276\267", nil, nil, nil],
      0x0f57 => [0, 1, "\340\275\226\340\276\267", "\340\275\226\340\276\267", nil, nil, nil],
      0x0f5c => [0, 1, "\340\275\233\340\276\267", "\340\275\233\340\276\267", nil, nil, nil],
      0x0f69 => [0, 1, "\340\275\200\340\276\265", "\340\275\200\340\276\265", nil, nil, nil],
      0x0f71 => [129, 0, nil, nil, nil, nil, nil],
      0x0f72 => [130, 0, nil, nil, nil, nil, nil],
      0x0f73 => [0, 3, "\340\275\261\340\275\262", "\340\275\261\340\275\262", nil, nil, nil],
      0x0f74 => [132, 0, nil, nil, nil, nil, nil],
      0x0f75 => [0, 3, "\340\275\261\340\275\264", "\340\275\261\340\275\264", nil, nil, nil],
      0x0f76 => [0, 1, "\340\276\262\340\276\200", "\340\276\262\340\276\200", nil, nil, nil],
      0x0f77 => [0, 0, nil, "\340\276\262\340\276\201", nil, nil, nil],
      0x0f78 => [0, 1, "\340\276\263\340\276\200", "\340\276\263\340\276\200", nil, nil, nil],
      0x0f79 => [0, 0, nil, "\340\276\263\340\276\201", nil, nil, nil],
      0x0f7a => [130, 0, nil, nil, nil, nil, nil],
      0x0f7b => [130, 0, nil, nil, nil, nil, nil],
      0x0f7c => [130, 0, nil, nil, nil, nil, nil],
      0x0f7d => [130, 0, nil, nil, nil, nil, nil],
      0x0f80 => [130, 0, nil, nil, nil, nil, nil],
      0x0f81 => [0, 3, "\340\275\261\340\276\200", "\340\275\261\340\276\200", nil, nil, nil],
      0x0f82 => [230, 0, nil, nil, nil, nil, nil],
      0x0f83 => [230, 0, nil, nil, nil, nil, nil],
      0x0f84 => [9, 0, nil, nil, nil, nil, nil],
      0x0f86 => [230, 0, nil, nil, nil, nil, nil],
      0x0f87 => [230, 0, nil, nil, nil, nil, nil],
      0x0f93 => [0, 1, "\340\276\222\340\276\267", "\340\276\222\340\276\267", nil, nil, nil],
      0x0f9d => [0, 1, "\340\276\234\340\276\267", "\340\276\234\340\276\267", nil, nil, nil],
      0x0fa2 => [0, 1, "\340\276\241\340\276\267", "\340\276\241\340\276\267", nil, nil, nil],
      0x0fa7 => [0, 1, "\340\276\246\340\276\267", "\340\276\246\340\276\267", nil, nil, nil],
      0x0fac => [0, 1, "\340\276\253\340\276\267", "\340\276\253\340\276\267", nil, nil, nil],
      0x0fb9 => [0, 1, "\340\276\220\340\276\265", "\340\276\220\340\276\265", nil, nil, nil],
      0x0fc6 => [220, 0, nil, nil, nil, nil, nil],
      0x1026 => [0, 0, "\341\200\245\341\200\256", "\341\200\245\341\200\256", nil, nil, nil],
      0x1037 => [7, 0, nil, nil, nil, nil, nil],
      0x1039 => [9, 0, nil, nil, nil, nil, nil],
      0x17d2 => [9, 0, nil, nil, nil, nil, nil],
      0x18a9 => [228, 0, nil, nil, nil, nil, nil],
      0x1e00 => [0, 0, "A\314\245", "A\314\245", nil, 0x1e01, nil],
      0x1e01 => [0, 0, "a\314\245", "a\314\245", 0x1e00, nil, 0x1e00],
      0x1e02 => [0, 0, "B\314\207", "B\314\207", nil, 0x1e03, nil],
      0x1e03 => [0, 0, "b\314\207", "b\314\207", 0x1e02, nil, 0x1e02],
      0x1e04 => [0, 0, "B\314\243", "B\314\243", nil, 0x1e05, nil],
      0x1e05 => [0, 0, "b\314\243", "b\314\243", 0x1e04, nil, 0x1e04],
      0x1e06 => [0, 0, "B\314\261", "B\314\261", nil, 0x1e07, nil],
      0x1e07 => [0, 0, "b\314\261", "b\314\261", 0x1e06, nil, 0x1e06],
      0x1e08 => [0, 0, "\303\207\314\201", "\303\207\314\201", nil, 0x1e09, nil],
      0x1e09 => [0, 0, "\303\247\314\201", "\303\247\314\201", 0x1e08, nil, 0x1e08],
      0x1e0a => [0, 0, "D\314\207", "D\314\207", nil, 0x1e0b, nil],
      0x1e0b => [0, 0, "d\314\207", "d\314\207", 0x1e0a, nil, 0x1e0a],
      0x1e0c => [0, 0, "D\314\243", "D\314\243", nil, 0x1e0d, nil],
      0x1e0d => [0, 0, "d\314\243", "d\314\243", 0x1e0c, nil, 0x1e0c],
      0x1e0e => [0, 0, "D\314\261", "D\314\261", nil, 0x1e0f, nil],
      0x1e0f => [0, 0, "d\314\261", "d\314\261", 0x1e0e, nil, 0x1e0e],
      0x1e10 => [0, 0, "D\314\247", "D\314\247", nil, 0x1e11, nil],
      0x1e11 => [0, 0, "d\314\247", "d\314\247", 0x1e10, nil, 0x1e10],
      0x1e12 => [0, 0, "D\314\255", "D\314\255", nil, 0x1e13, nil],
      0x1e13 => [0, 0, "d\314\255", "d\314\255", 0x1e12, nil, 0x1e12],
      0x1e14 => [0, 0, "\304\222\314\200", "\304\222\314\200", nil, 0x1e15, nil],
      0x1e15 => [0, 0, "\304\223\314\200", "\304\223\314\200", 0x1e14, nil, 0x1e14],
      0x1e16 => [0, 0, "\304\222\314\201", "\304\222\314\201", nil, 0x1e17, nil],
      0x1e17 => [0, 0, "\304\223\314\201", "\304\223\314\201", 0x1e16, nil, 0x1e16],
      0x1e18 => [0, 0, "E\314\255", "E\314\255", nil, 0x1e19, nil],
      0x1e19 => [0, 0, "e\314\255", "e\314\255", 0x1e18, nil, 0x1e18],
      0x1e1a => [0, 0, "E\314\260", "E\314\260", nil, 0x1e1b, nil],
      0x1e1b => [0, 0, "e\314\260", "e\314\260", 0x1e1a, nil, 0x1e1a],
      0x1e1c => [0, 0, "\310\250\314\206", "\310\250\314\206", nil, 0x1e1d, nil],
      0x1e1d => [0, 0, "\310\251\314\206", "\310\251\314\206", 0x1e1c, nil, 0x1e1c],
      0x1e1e => [0, 0, "F\314\207", "F\314\207", nil, 0x1e1f, nil],
      0x1e1f => [0, 0, "f\314\207", "f\314\207", 0x1e1e, nil, 0x1e1e],
      0x1e20 => [0, 0, "G\314\204", "G\314\204", nil, 0x1e21, nil],
      0x1e21 => [0, 0, "g\314\204", "g\314\204", 0x1e20, nil, 0x1e20],
      0x1e22 => [0, 0, "H\314\207", "H\314\207", nil, 0x1e23, nil],
      0x1e23 => [0, 0, "h\314\207", "h\314\207", 0x1e22, nil, 0x1e22],
      0x1e24 => [0, 0, "H\314\243", "H\314\243", nil, 0x1e25, nil],
      0x1e25 => [0, 0, "h\314\243", "h\314\243", 0x1e24, nil, 0x1e24],
      0x1e26 => [0, 0, "H\314\210", "H\314\210", nil, 0x1e27, nil],
      0x1e27 => [0, 0, "h\314\210", "h\314\210", 0x1e26, nil, 0x1e26],
      0x1e28 => [0, 0, "H\314\247", "H\314\247", nil, 0x1e29, nil],
      0x1e29 => [0, 0, "h\314\247", "h\314\247", 0x1e28, nil, 0x1e28],
      0x1e2a => [0, 0, "H\314\256", "H\314\256", nil, 0x1e2b, nil],
      0x1e2b => [0, 0, "h\314\256", "h\314\256", 0x1e2a, nil, 0x1e2a],
      0x1e2c => [0, 0, "I\314\260", "I\314\260", nil, 0x1e2d, nil],
      0x1e2d => [0, 0, "i\314\260", "i\314\260", 0x1e2c, nil, 0x1e2c],
      0x1e2e => [0, 0, "\303\217\314\201", "\303\217\314\201", nil, 0x1e2f, nil],
      0x1e2f => [0, 0, "\303\257\314\201", "\303\257\314\201", 0x1e2e, nil, 0x1e2e],
      0x1e30 => [0, 0, "K\314\201", "K\314\201", nil, 0x1e31, nil],
      0x1e31 => [0, 0, "k\314\201", "k\314\201", 0x1e30, nil, 0x1e30],
      0x1e32 => [0, 0, "K\314\243", "K\314\243", nil, 0x1e33, nil],
      0x1e33 => [0, 0, "k\314\243", "k\314\243", 0x1e32, nil, 0x1e32],
      0x1e34 => [0, 0, "K\314\261", "K\314\261", nil, 0x1e35, nil],
      0x1e35 => [0, 0, "k\314\261", "k\314\261", 0x1e34, nil, 0x1e34],
      0x1e36 => [0, 0, "L\314\243", "L\314\243", nil, 0x1e37, nil],
      0x1e37 => [0, 0, "l\314\243", "l\314\243", 0x1e36, nil, 0x1e36],
      0x1e38 => [0, 0, "\341\270\266\314\204", "\341\270\266\314\204", nil, 0x1e39, nil],
      0x1e39 => [0, 0, "\341\270\267\314\204", "\341\270\267\314\204", 0x1e38, nil, 0x1e38],
      0x1e3a => [0, 0, "L\314\261", "L\314\261", nil, 0x1e3b, nil],
      0x1e3b => [0, 0, "l\314\261", "l\314\261", 0x1e3a, nil, 0x1e3a],
      0x1e3c => [0, 0, "L\314\255", "L\314\255", nil, 0x1e3d, nil],
      0x1e3d => [0, 0, "l\314\255", "l\314\255", 0x1e3c, nil, 0x1e3c],
      0x1e3e => [0, 0, "M\314\201", "M\314\201", nil, 0x1e3f, nil],
      0x1e3f => [0, 0, "m\314\201", "m\314\201", 0x1e3e, nil, 0x1e3e],
      0x1e40 => [0, 0, "M\314\207", "M\314\207", nil, 0x1e41, nil],
      0x1e41 => [0, 0, "m\314\207", "m\314\207", 0x1e40, nil, 0x1e40],
      0x1e42 => [0, 0, "M\314\243", "M\314\243", nil, 0x1e43, nil],
      0x1e43 => [0, 0, "m\314\243", "m\314\243", 0x1e42, nil, 0x1e42],
      0x1e44 => [0, 0, "N\314\207", "N\314\207", nil, 0x1e45, nil],
      0x1e45 => [0, 0, "n\314\207", "n\314\207", 0x1e44, nil, 0x1e44],
      0x1e46 => [0, 0, "N\314\243", "N\314\243", nil, 0x1e47, nil],
      0x1e47 => [0, 0, "n\314\243", "n\314\243", 0x1e46, nil, 0x1e46],
      0x1e48 => [0, 0, "N\314\261", "N\314\261", nil, 0x1e49, nil],
      0x1e49 => [0, 0, "n\314\261", "n\314\261", 0x1e48, nil, 0x1e48],
      0x1e4a => [0, 0, "N\314\255", "N\314\255", nil, 0x1e4b, nil],
      0x1e4b => [0, 0, "n\314\255", "n\314\255", 0x1e4a, nil, 0x1e4a],
      0x1e4c => [0, 0, "\303\225\314\201", "\303\225\314\201", nil, 0x1e4d, nil],
      0x1e4d => [0, 0, "\303\265\314\201", "\303\265\314\201", 0x1e4c, nil, 0x1e4c],
      0x1e4e => [0, 0, "\303\225\314\210", "\303\225\314\210", nil, 0x1e4f, nil],
      0x1e4f => [0, 0, "\303\265\314\210", "\303\265\314\210", 0x1e4e, nil, 0x1e4e],
      0x1e50 => [0, 0, "\305\214\314\200", "\305\214\314\200", nil, 0x1e51, nil],
      0x1e51 => [0, 0, "\305\215\314\200", "\305\215\314\200", 0x1e50, nil, 0x1e50],
      0x1e52 => [0, 0, "\305\214\314\201", "\305\214\314\201", nil, 0x1e53, nil],
      0x1e53 => [0, 0, "\305\215\314\201", "\305\215\314\201", 0x1e52, nil, 0x1e52],
      0x1e54 => [0, 0, "P\314\201", "P\314\201", nil, 0x1e55, nil],
      0x1e55 => [0, 0, "p\314\201", "p\314\201", 0x1e54, nil, 0x1e54],
      0x1e56 => [0, 0, "P\314\207", "P\314\207", nil, 0x1e57, nil],
      0x1e57 => [0, 0, "p\314\207", "p\314\207", 0x1e56, nil, 0x1e56],
      0x1e58 => [0, 0, "R\314\207", "R\314\207", nil, 0x1e59, nil],
      0x1e59 => [0, 0, "r\314\207", "r\314\207", 0x1e58, nil, 0x1e58],
      0x1e5a => [0, 0, "R\314\243", "R\314\243", nil, 0x1e5b, nil],
      0x1e5b => [0, 0, "r\314\243", "r\314\243", 0x1e5a, nil, 0x1e5a],
      0x1e5c => [0, 0, "\341\271\232\314\204", "\341\271\232\314\204", nil, 0x1e5d, nil],
      0x1e5d => [0, 0, "\341\271\233\314\204", "\341\271\233\314\204", 0x1e5c, nil, 0x1e5c],
      0x1e5e => [0, 0, "R\314\261", "R\314\261", nil, 0x1e5f, nil],
      0x1e5f => [0, 0, "r\314\261", "r\314\261", 0x1e5e, nil, 0x1e5e],
      0x1e60 => [0, 0, "S\314\207", "S\314\207", nil, 0x1e61, nil],
      0x1e61 => [0, 0, "s\314\207", "s\314\207", 0x1e60, nil, 0x1e60],
      0x1e62 => [0, 0, "S\314\243", "S\314\243", nil, 0x1e63, nil],
      0x1e63 => [0, 0, "s\314\243", "s\314\243", 0x1e62, nil, 0x1e62],
      0x1e64 => [0, 0, "\305\232\314\207", "\305\232\314\207", nil, 0x1e65, nil],
      0x1e65 => [0, 0, "\305\233\314\207", "\305\233\314\207", 0x1e64, nil, 0x1e64],
      0x1e66 => [0, 0, "\305\240\314\207", "\305\240\314\207", nil, 0x1e67, nil],
      0x1e67 => [0, 0, "\305\241\314\207", "\305\241\314\207", 0x1e66, nil, 0x1e66],
      0x1e68 => [0, 0, "\341\271\242\314\207", "\341\271\242\314\207", nil, 0x1e69, nil],
      0x1e69 => [0, 0, "\341\271\243\314\207", "\341\271\243\314\207", 0x1e68, nil, 0x1e68],
      0x1e6a => [0, 0, "T\314\207", "T\314\207", nil, 0x1e6b, nil],
      0x1e6b => [0, 0, "t\314\207", "t\314\207", 0x1e6a, nil, 0x1e6a],
      0x1e6c => [0, 0, "T\314\243", "T\314\243", nil, 0x1e6d, nil],
      0x1e6d => [0, 0, "t\314\243", "t\314\243", 0x1e6c, nil, 0x1e6c],
      0x1e6e => [0, 0, "T\314\261", "T\314\261", nil, 0x1e6f, nil],
      0x1e6f => [0, 0, "t\314\261", "t\314\261", 0x1e6e, nil, 0x1e6e],
      0x1e70 => [0, 0, "T\314\255", "T\314\255", nil, 0x1e71, nil],
      0x1e71 => [0, 0, "t\314\255", "t\314\255", 0x1e70, nil, 0x1e70],
      0x1e72 => [0, 0, "U\314\244", "U\314\244", nil, 0x1e73, nil],
      0x1e73 => [0, 0, "u\314\244", "u\314\244", 0x1e72, nil, 0x1e72],
      0x1e74 => [0, 0, "U\314\260", "U\314\260", nil, 0x1e75, nil],
      0x1e75 => [0, 0, "u\314\260", "u\314\260", 0x1e74, nil, 0x1e74],
      0x1e76 => [0, 0, "U\314\255", "U\314\255", nil, 0x1e77, nil],
      0x1e77 => [0, 0, "u\314\255", "u\314\255", 0x1e76, nil, 0x1e76],
      0x1e78 => [0, 0, "\305\250\314\201", "\305\250\314\201", nil, 0x1e79, nil],
      0x1e79 => [0, 0, "\305\251\314\201", "\305\251\314\201", 0x1e78, nil, 0x1e78],
      0x1e7a => [0, 0, "\305\252\314\210", "\305\252\314\210", nil, 0x1e7b, nil],
      0x1e7b => [0, 0, "\305\253\314\210", "\305\253\314\210", 0x1e7a, nil, 0x1e7a],
      0x1e7c => [0, 0, "V\314\203", "V\314\203", nil, 0x1e7d, nil],
      0x1e7d => [0, 0, "v\314\203", "v\314\203", 0x1e7c, nil, 0x1e7c],
      0x1e7e => [0, 0, "V\314\243", "V\314\243", nil, 0x1e7f, nil],
      0x1e7f => [0, 0, "v\314\243", "v\314\243", 0x1e7e, nil, 0x1e7e],
      0x1e80 => [0, 0, "W\314\200", "W\314\200", nil, 0x1e81, nil],
      0x1e81 => [0, 0, "w\314\200", "w\314\200", 0x1e80, nil, 0x1e80],
      0x1e82 => [0, 0, "W\314\201", "W\314\201", nil, 0x1e83, nil],
      0x1e83 => [0, 0, "w\314\201", "w\314\201", 0x1e82, nil, 0x1e82],
      0x1e84 => [0, 0, "W\314\210", "W\314\210", nil, 0x1e85, nil],
      0x1e85 => [0, 0, "w\314\210", "w\314\210", 0x1e84, nil, 0x1e84],
      0x1e86 => [0, 0, "W\314\207", "W\314\207", nil, 0x1e87, nil],
      0x1e87 => [0, 0, "w\314\207", "w\314\207", 0x1e86, nil, 0x1e86],
      0x1e88 => [0, 0, "W\314\243", "W\314\243", nil, 0x1e89, nil],
      0x1e89 => [0, 0, "w\314\243", "w\314\243", 0x1e88, nil, 0x1e88],
      0x1e8a => [0, 0, "X\314\207", "X\314\207", nil, 0x1e8b, nil],
      0x1e8b => [0, 0, "x\314\207", "x\314\207", 0x1e8a, nil, 0x1e8a],
      0x1e8c => [0, 0, "X\314\210", "X\314\210", nil, 0x1e8d, nil],
      0x1e8d => [0, 0, "x\314\210", "x\314\210", 0x1e8c, nil, 0x1e8c],
      0x1e8e => [0, 0, "Y\314\207", "Y\314\207", nil, 0x1e8f, nil],
      0x1e8f => [0, 0, "y\314\207", "y\314\207", 0x1e8e, nil, 0x1e8e],
      0x1e90 => [0, 0, "Z\314\202", "Z\314\202", nil, 0x1e91, nil],
      0x1e91 => [0, 0, "z\314\202", "z\314\202", 0x1e90, nil, 0x1e90],
      0x1e92 => [0, 0, "Z\314\243", "Z\314\243", nil, 0x1e93, nil],
      0x1e93 => [0, 0, "z\314\243", "z\314\243", 0x1e92, nil, 0x1e92],
      0x1e94 => [0, 0, "Z\314\261", "Z\314\261", nil, 0x1e95, nil],
      0x1e95 => [0, 0, "z\314\261", "z\314\261", 0x1e94, nil, 0x1e94],
      0x1e96 => [0, 0, "h\314\261", "h\314\261", nil, nil, nil],
      0x1e97 => [0, 0, "t\314\210", "t\314\210", nil, nil, nil],
      0x1e98 => [0, 0, "w\314\212", "w\314\212", nil, nil, nil],
      0x1e99 => [0, 0, "y\314\212", "y\314\212", nil, nil, nil],
      0x1e9a => [0, 0, nil, "a\312\276", nil, nil, nil],
      0x1e9b => [0, 0, "\305\277\314\207", "\305\277\314\207", 0x1e60, nil, 0x1e60],
      0x1ea0 => [0, 0, "A\314\243", "A\314\243", nil, 0x1ea1, nil],
      0x1ea1 => [0, 0, "a\314\243", "a\314\243", 0x1ea0, nil, 0x1ea0],
      0x1ea2 => [0, 0, "A\314\211", "A\314\211", nil, 0x1ea3, nil],
      0x1ea3 => [0, 0, "a\314\211", "a\314\211", 0x1ea2, nil, 0x1ea2],
      0x1ea4 => [0, 0, "\303\202\314\201", "\303\202\314\201", nil, 0x1ea5, nil],
      0x1ea5 => [0, 0, "\303\242\314\201", "\303\242\314\201", 0x1ea4, nil, 0x1ea4],
      0x1ea6 => [0, 0, "\303\202\314\200", "\303\202\314\200", nil, 0x1ea7, nil],
      0x1ea7 => [0, 0, "\303\242\314\200", "\303\242\314\200", 0x1ea6, nil, 0x1ea6],
      0x1ea8 => [0, 0, "\303\202\314\211", "\303\202\314\211", nil, 0x1ea9, nil],
      0x1ea9 => [0, 0, "\303\242\314\211", "\303\242\314\211", 0x1ea8, nil, 0x1ea8],
      0x1eaa => [0, 0, "\303\202\314\203", "\303\202\314\203", nil, 0x1eab, nil],
      0x1eab => [0, 0, "\303\242\314\203", "\303\242\314\203", 0x1eaa, nil, 0x1eaa],
      0x1eac => [0, 0, "\341\272\240\314\202", "\341\272\240\314\202", nil, 0x1ead, nil],
      0x1ead => [0, 0, "\341\272\241\314\202", "\341\272\241\314\202", 0x1eac, nil, 0x1eac],
      0x1eae => [0, 0, "\304\202\314\201", "\304\202\314\201", nil, 0x1eaf, nil],
      0x1eaf => [0, 0, "\304\203\314\201", "\304\203\314\201", 0x1eae, nil, 0x1eae],
      0x1eb0 => [0, 0, "\304\202\314\200", "\304\202\314\200", nil, 0x1eb1, nil],
      0x1eb1 => [0, 0, "\304\203\314\200", "\304\203\314\200", 0x1eb0, nil, 0x1eb0],
      0x1eb2 => [0, 0, "\304\202\314\211", "\304\202\314\211", nil, 0x1eb3, nil],
      0x1eb3 => [0, 0, "\304\203\314\211", "\304\203\314\211", 0x1eb2, nil, 0x1eb2],
      0x1eb4 => [0, 0, "\304\202\314\203", "\304\202\314\203", nil, 0x1eb5, nil],
      0x1eb5 => [0, 0, "\304\203\314\203", "\304\203\314\203", 0x1eb4, nil, 0x1eb4],
      0x1eb6 => [0, 0, "\341\272\240\314\206", "\341\272\240\314\206", nil, 0x1eb7, nil],
      0x1eb7 => [0, 0, "\341\272\241\314\206", "\341\272\241\314\206", 0x1eb6, nil, 0x1eb6],
      0x1eb8 => [0, 0, "E\314\243", "E\314\243", nil, 0x1eb9, nil],
      0x1eb9 => [0, 0, "e\314\243", "e\314\243", 0x1eb8, nil, 0x1eb8],
      0x1eba => [0, 0, "E\314\211", "E\314\211", nil, 0x1ebb, nil],
      0x1ebb => [0, 0, "e\314\211", "e\314\211", 0x1eba, nil, 0x1eba],
      0x1ebc => [0, 0, "E\314\203", "E\314\203", nil, 0x1ebd, nil],
      0x1ebd => [0, 0, "e\314\203", "e\314\203", 0x1ebc, nil, 0x1ebc],
      0x1ebe => [0, 0, "\303\212\314\201", "\303\212\314\201", nil, 0x1ebf, nil],
      0x1ebf => [0, 0, "\303\252\314\201", "\303\252\314\201", 0x1ebe, nil, 0x1ebe],
      0x1ec0 => [0, 0, "\303\212\314\200", "\303\212\314\200", nil, 0x1ec1, nil],
      0x1ec1 => [0, 0, "\303\252\314\200", "\303\252\314\200", 0x1ec0, nil, 0x1ec0],
      0x1ec2 => [0, 0, "\303\212\314\211", "\303\212\314\211", nil, 0x1ec3, nil],
      0x1ec3 => [0, 0, "\303\252\314\211", "\303\252\314\211", 0x1ec2, nil, 0x1ec2],
      0x1ec4 => [0, 0, "\303\212\314\203", "\303\212\314\203", nil, 0x1ec5, nil],
      0x1ec5 => [0, 0, "\303\252\314\203", "\303\252\314\203", 0x1ec4, nil, 0x1ec4],
      0x1ec6 => [0, 0, "\341\272\270\314\202", "\341\272\270\314\202", nil, 0x1ec7, nil],
      0x1ec7 => [0, 0, "\341\272\271\314\202", "\341\272\271\314\202", 0x1ec6, nil, 0x1ec6],
      0x1ec8 => [0, 0, "I\314\211", "I\314\211", nil, 0x1ec9, nil],
      0x1ec9 => [0, 0, "i\314\211", "i\314\211", 0x1ec8, nil, 0x1ec8],
      0x1eca => [0, 0, "I\314\243", "I\314\243", nil, 0x1ecb, nil],
      0x1ecb => [0, 0, "i\314\243", "i\314\243", 0x1eca, nil, 0x1eca],
      0x1ecc => [0, 0, "O\314\243", "O\314\243", nil, 0x1ecd, nil],
      0x1ecd => [0, 0, "o\314\243", "o\314\243", 0x1ecc, nil, 0x1ecc],
      0x1ece => [0, 0, "O\314\211", "O\314\211", nil, 0x1ecf, nil],
      0x1ecf => [0, 0, "o\314\211", "o\314\211", 0x1ece, nil, 0x1ece],
      0x1ed0 => [0, 0, "\303\224\314\201", "\303\224\314\201", nil, 0x1ed1, nil],
      0x1ed1 => [0, 0, "\303\264\314\201", "\303\264\314\201", 0x1ed0, nil, 0x1ed0],
      0x1ed2 => [0, 0, "\303\224\314\200", "\303\224\314\200", nil, 0x1ed3, nil],
      0x1ed3 => [0, 0, "\303\264\314\200", "\303\264\314\200", 0x1ed2, nil, 0x1ed2],
      0x1ed4 => [0, 0, "\303\224\314\211", "\303\224\314\211", nil, 0x1ed5, nil],
      0x1ed5 => [0, 0, "\303\264\314\211", "\303\264\314\211", 0x1ed4, nil, 0x1ed4],
      0x1ed6 => [0, 0, "\303\224\314\203", "\303\224\314\203", nil, 0x1ed7, nil],
      0x1ed7 => [0, 0, "\303\264\314\203", "\303\264\314\203", 0x1ed6, nil, 0x1ed6],
      0x1ed8 => [0, 0, "\341\273\214\314\202", "\341\273\214\314\202", nil, 0x1ed9, nil],
      0x1ed9 => [0, 0, "\341\273\215\314\202", "\341\273\215\314\202", 0x1ed8, nil, 0x1ed8],
      0x1eda => [0, 0, "\306\240\314\201", "\306\240\314\201", nil, 0x1edb, nil],
      0x1edb => [0, 0, "\306\241\314\201", "\306\241\314\201", 0x1eda, nil, 0x1eda],
      0x1edc => [0, 0, "\306\240\314\200", "\306\240\314\200", nil, 0x1edd, nil],
      0x1edd => [0, 0, "\306\241\314\200", "\306\241\314\200", 0x1edc, nil, 0x1edc],
      0x1ede => [0, 0, "\306\240\314\211", "\306\240\314\211", nil, 0x1edf, nil],
      0x1edf => [0, 0, "\306\241\314\211", "\306\241\314\211", 0x1ede, nil, 0x1ede],
      0x1ee0 => [0, 0, "\306\240\314\203", "\306\240\314\203", nil, 0x1ee1, nil],
      0x1ee1 => [0, 0, "\306\241\314\203", "\306\241\314\203", 0x1ee0, nil, 0x1ee0],
      0x1ee2 => [0, 0, "\306\240\314\243", "\306\240\314\243", nil, 0x1ee3, nil],
      0x1ee3 => [0, 0, "\306\241\314\243", "\306\241\314\243", 0x1ee2, nil, 0x1ee2],
      0x1ee4 => [0, 0, "U\314\243", "U\314\243", nil, 0x1ee5, nil],
      0x1ee5 => [0, 0, "u\314\243", "u\314\243", 0x1ee4, nil, 0x1ee4],
      0x1ee6 => [0, 0, "U\314\211", "U\314\211", nil, 0x1ee7, nil],
      0x1ee7 => [0, 0, "u\314\211", "u\314\211", 0x1ee6, nil, 0x1ee6],
      0x1ee8 => [0, 0, "\306\257\314\201", "\306\257\314\201", nil, 0x1ee9, nil],
      0x1ee9 => [0, 0, "\306\260\314\201", "\306\260\314\201", 0x1ee8, nil, 0x1ee8],
      0x1eea => [0, 0, "\306\257\314\200", "\306\257\314\200", nil, 0x1eeb, nil],
      0x1eeb => [0, 0, "\306\260\314\200", "\306\260\314\200", 0x1eea, nil, 0x1eea],
      0x1eec => [0, 0, "\306\257\314\211", "\306\257\314\211", nil, 0x1eed, nil],
      0x1eed => [0, 0, "\306\260\314\211", "\306\260\314\211", 0x1eec, nil, 0x1eec],
      0x1eee => [0, 0, "\306\257\314\203", "\306\257\314\203", nil, 0x1eef, nil],
      0x1eef => [0, 0, "\306\260\314\203", "\306\260\314\203", 0x1eee, nil, 0x1eee],
      0x1ef0 => [0, 0, "\306\257\314\243", "\306\257\314\243", nil, 0x1ef1, nil],
      0x1ef1 => [0, 0, "\306\260\314\243", "\306\260\314\243", 0x1ef0, nil, 0x1ef0],
      0x1ef2 => [0, 0, "Y\314\200", "Y\314\200", nil, 0x1ef3, nil],
      0x1ef3 => [0, 0, "y\314\200", "y\314\200", 0x1ef2, nil, 0x1ef2],
      0x1ef4 => [0, 0, "Y\314\243", "Y\314\243", nil, 0x1ef5, nil],
      0x1ef5 => [0, 0, "y\314\243", "y\314\243", 0x1ef4, nil, 0x1ef4],
      0x1ef6 => [0, 0, "Y\314\211", "Y\314\211", nil, 0x1ef7, nil],
      0x1ef7 => [0, 0, "y\314\211", "y\314\211", 0x1ef6, nil, 0x1ef6],
      0x1ef8 => [0, 0, "Y\314\203", "Y\314\203", nil, 0x1ef9, nil],
      0x1ef9 => [0, 0, "y\314\203", "y\314\203", 0x1ef8, nil, 0x1ef8],
      0x1f00 => [0, 0, "\316\261\314\223", "\316\261\314\223", 0x1f08, nil, 0x1f08],
      0x1f01 => [0, 0, "\316\261\314\224", "\316\261\314\224", 0x1f09, nil, 0x1f09],
      0x1f02 => [0, 0, "\341\274\200\314\200", "\341\274\200\314\200", 0x1f0a, nil, 0x1f0a],
      0x1f03 => [0, 0, "\341\274\201\314\200", "\341\274\201\314\200", 0x1f0b, nil, 0x1f0b],
      0x1f04 => [0, 0, "\341\274\200\314\201", "\341\274\200\314\201", 0x1f0c, nil, 0x1f0c],
      0x1f05 => [0, 0, "\341\274\201\314\201", "\341\274\201\314\201", 0x1f0d, nil, 0x1f0d],
      0x1f06 => [0, 0, "\341\274\200\315\202", "\341\274\200\315\202", 0x1f0e, nil, 0x1f0e],
      0x1f07 => [0, 0, "\341\274\201\315\202", "\341\274\201\315\202", 0x1f0f, nil, 0x1f0f],
      0x1f08 => [0, 0, "\316\221\314\223", "\316\221\314\223", nil, 0x1f00, nil],
      0x1f09 => [0, 0, "\316\221\314\224", "\316\221\314\224", nil, 0x1f01, nil],
      0x1f0a => [0, 0, "\341\274\210\314\200", "\341\274\210\314\200", nil, 0x1f02, nil],
      0x1f0b => [0, 0, "\341\274\211\314\200", "\341\274\211\314\200", nil, 0x1f03, nil],
      0x1f0c => [0, 0, "\341\274\210\314\201", "\341\274\210\314\201", nil, 0x1f04, nil],
      0x1f0d => [0, 0, "\341\274\211\314\201", "\341\274\211\314\201", nil, 0x1f05, nil],
      0x1f0e => [0, 0, "\341\274\210\315\202", "\341\274\210\315\202", nil, 0x1f06, nil],
      0x1f0f => [0, 0, "\341\274\211\315\202", "\341\274\211\315\202", nil, 0x1f07, nil],
      0x1f10 => [0, 0, "\316\265\314\223", "\316\265\314\223", 0x1f18, nil, 0x1f18],
      0x1f11 => [0, 0, "\316\265\314\224", "\316\265\314\224", 0x1f19, nil, 0x1f19],
      0x1f12 => [0, 0, "\341\274\220\314\200", "\341\274\220\314\200", 0x1f1a, nil, 0x1f1a],
      0x1f13 => [0, 0, "\341\274\221\314\200", "\341\274\221\314\200", 0x1f1b, nil, 0x1f1b],
      0x1f14 => [0, 0, "\341\274\220\314\201", "\341\274\220\314\201", 0x1f1c, nil, 0x1f1c],
      0x1f15 => [0, 0, "\341\274\221\314\201", "\341\274\221\314\201", 0x1f1d, nil, 0x1f1d],
      0x1f18 => [0, 0, "\316\225\314\223", "\316\225\314\223", nil, 0x1f10, nil],
      0x1f19 => [0, 0, "\316\225\314\224", "\316\225\314\224", nil, 0x1f11, nil],
      0x1f1a => [0, 0, "\341\274\230\314\200", "\341\274\230\314\200", nil, 0x1f12, nil],
      0x1f1b => [0, 0, "\341\274\231\314\200", "\341\274\231\314\200", nil, 0x1f13, nil],
      0x1f1c => [0, 0, "\341\274\230\314\201", "\341\274\230\314\201", nil, 0x1f14, nil],
      0x1f1d => [0, 0, "\341\274\231\314\201", "\341\274\231\314\201", nil, 0x1f15, nil],
      0x1f20 => [0, 0, "\316\267\314\223", "\316\267\314\223", 0x1f28, nil, 0x1f28],
      0x1f21 => [0, 0, "\316\267\314\224", "\316\267\314\224", 0x1f29, nil, 0x1f29],
      0x1f22 => [0, 0, "\341\274\240\314\200", "\341\274\240\314\200", 0x1f2a, nil, 0x1f2a],
      0x1f23 => [0, 0, "\341\274\241\314\200", "\341\274\241\314\200", 0x1f2b, nil, 0x1f2b],
      0x1f24 => [0, 0, "\341\274\240\314\201", "\341\274\240\314\201", 0x1f2c, nil, 0x1f2c],
      0x1f25 => [0, 0, "\341\274\241\314\201", "\341\274\241\314\201", 0x1f2d, nil, 0x1f2d],
      0x1f26 => [0, 0, "\341\274\240\315\202", "\341\274\240\315\202", 0x1f2e, nil, 0x1f2e],
      0x1f27 => [0, 0, "\341\274\241\315\202", "\341\274\241\315\202", 0x1f2f, nil, 0x1f2f],
      0x1f28 => [0, 0, "\316\227\314\223", "\316\227\314\223", nil, 0x1f20, nil],
      0x1f29 => [0, 0, "\316\227\314\224", "\316\227\314\224", nil, 0x1f21, nil],
      0x1f2a => [0, 0, "\341\274\250\314\200", "\341\274\250\314\200", nil, 0x1f22, nil],
      0x1f2b => [0, 0, "\341\274\251\314\200", "\341\274\251\314\200", nil, 0x1f23, nil],
      0x1f2c => [0, 0, "\341\274\250\314\201", "\341\274\250\314\201", nil, 0x1f24, nil],
      0x1f2d => [0, 0, "\341\274\251\314\201", "\341\274\251\314\201", nil, 0x1f25, nil],
      0x1f2e => [0, 0, "\341\274\250\315\202", "\341\274\250\315\202", nil, 0x1f26, nil],
      0x1f2f => [0, 0, "\341\274\251\315\202", "\341\274\251\315\202", nil, 0x1f27, nil],
      0x1f30 => [0, 0, "\316\271\314\223", "\316\271\314\223", 0x1f38, nil, 0x1f38],
      0x1f31 => [0, 0, "\316\271\314\224", "\316\271\314\224", 0x1f39, nil, 0x1f39],
      0x1f32 => [0, 0, "\341\274\260\314\200", "\341\274\260\314\200", 0x1f3a, nil, 0x1f3a],
      0x1f33 => [0, 0, "\341\274\261\314\200", "\341\274\261\314\200", 0x1f3b, nil, 0x1f3b],
      0x1f34 => [0, 0, "\341\274\260\314\201", "\341\274\260\314\201", 0x1f3c, nil, 0x1f3c],
      0x1f35 => [0, 0, "\341\274\261\314\201", "\341\274\261\314\201", 0x1f3d, nil, 0x1f3d],
      0x1f36 => [0, 0, "\341\274\260\315\202", "\341\274\260\315\202", 0x1f3e, nil, 0x1f3e],
      0x1f37 => [0, 0, "\341\274\261\315\202", "\341\274\261\315\202", 0x1f3f, nil, 0x1f3f],
      0x1f38 => [0, 0, "\316\231\314\223", "\316\231\314\223", nil, 0x1f30, nil],
      0x1f39 => [0, 0, "\316\231\314\224", "\316\231\314\224", nil, 0x1f31, nil],
      0x1f3a => [0, 0, "\341\274\270\314\200", "\341\274\270\314\200", nil, 0x1f32, nil],
      0x1f3b => [0, 0, "\341\274\271\314\200", "\341\274\271\314\200", nil, 0x1f33, nil],
      0x1f3c => [0, 0, "\341\274\270\314\201", "\341\274\270\314\201", nil, 0x1f34, nil],
      0x1f3d => [0, 0, "\341\274\271\314\201", "\341\274\271\314\201", nil, 0x1f35, nil],
      0x1f3e => [0, 0, "\341\274\270\315\202", "\341\274\270\315\202", nil, 0x1f36, nil],
      0x1f3f => [0, 0, "\341\274\271\315\202", "\341\274\271\315\202", nil, 0x1f37, nil],
      0x1f40 => [0, 0, "\316\277\314\223", "\316\277\314\223", 0x1f48, nil, 0x1f48],
      0x1f41 => [0, 0, "\316\277\314\224", "\316\277\314\224", 0x1f49, nil, 0x1f49],
      0x1f42 => [0, 0, "\341\275\200\314\200", "\341\275\200\314\200", 0x1f4a, nil, 0x1f4a],
      0x1f43 => [0, 0, "\341\275\201\314\200", "\341\275\201\314\200", 0x1f4b, nil, 0x1f4b],
      0x1f44 => [0, 0, "\341\275\200\314\201", "\341\275\200\314\201", 0x1f4c, nil, 0x1f4c],
      0x1f45 => [0, 0, "\341\275\201\314\201", "\341\275\201\314\201", 0x1f4d, nil, 0x1f4d],
      0x1f48 => [0, 0, "\316\237\314\223", "\316\237\314\223", nil, 0x1f40, nil],
      0x1f49 => [0, 0, "\316\237\314\224", "\316\237\314\224", nil, 0x1f41, nil],
      0x1f4a => [0, 0, "\341\275\210\314\200", "\341\275\210\314\200", nil, 0x1f42, nil],
      0x1f4b => [0, 0, "\341\275\211\314\200", "\341\275\211\314\200", nil, 0x1f43, nil],
      0x1f4c => [0, 0, "\341\275\210\314\201", "\341\275\210\314\201", nil, 0x1f44, nil],
      0x1f4d => [0, 0, "\341\275\211\314\201", "\341\275\211\314\201", nil, 0x1f45, nil],
      0x1f50 => [0, 0, "\317\205\314\223", "\317\205\314\223", nil, nil, nil],
      0x1f51 => [0, 0, "\317\205\314\224", "\317\205\314\224", 0x1f59, nil, 0x1f59],
      0x1f52 => [0, 0, "\341\275\220\314\200", "\341\275\220\314\200", nil, nil, nil],
      0x1f53 => [0, 0, "\341\275\221\314\200", "\341\275\221\314\200", 0x1f5b, nil, 0x1f5b],
      0x1f54 => [0, 0, "\341\275\220\314\201", "\341\275\220\314\201", nil, nil, nil],
      0x1f55 => [0, 0, "\341\275\221\314\201", "\341\275\221\314\201", 0x1f5d, nil, 0x1f5d],
      0x1f56 => [0, 0, "\341\275\220\315\202", "\341\275\220\315\202", nil, nil, nil],
      0x1f57 => [0, 0, "\341\275\221\315\202", "\341\275\221\315\202", 0x1f5f, nil, 0x1f5f],
      0x1f59 => [0, 0, "\316\245\314\224", "\316\245\314\224", nil, 0x1f51, nil],
      0x1f5b => [0, 0, "\341\275\231\314\200", "\341\275\231\314\200", nil, 0x1f53, nil],
      0x1f5d => [0, 0, "\341\275\231\314\201", "\341\275\231\314\201", nil, 0x1f55, nil],
      0x1f5f => [0, 0, "\341\275\231\315\202", "\341\275\231\315\202", nil, 0x1f57, nil],
      0x1f60 => [0, 0, "\317\211\314\223", "\317\211\314\223", 0x1f68, nil, 0x1f68],
      0x1f61 => [0, 0, "\317\211\314\224", "\317\211\314\224", 0x1f69, nil, 0x1f69],
      0x1f62 => [0, 0, "\341\275\240\314\200", "\341\275\240\314\200", 0x1f6a, nil, 0x1f6a],
      0x1f63 => [0, 0, "\341\275\241\314\200", "\341\275\241\314\200", 0x1f6b, nil, 0x1f6b],
      0x1f64 => [0, 0, "\341\275\240\314\201", "\341\275\240\314\201", 0x1f6c, nil, 0x1f6c],
      0x1f65 => [0, 0, "\341\275\241\314\201", "\341\275\241\314\201", 0x1f6d, nil, 0x1f6d],
      0x1f66 => [0, 0, "\341\275\240\315\202", "\341\275\240\315\202", 0x1f6e, nil, 0x1f6e],
      0x1f67 => [0, 0, "\341\275\241\315\202", "\341\275\241\315\202", 0x1f6f, nil, 0x1f6f],
      0x1f68 => [0, 0, "\316\251\314\223", "\316\251\314\223", nil, 0x1f60, nil],
      0x1f69 => [0, 0, "\316\251\314\224", "\316\251\314\224", nil, 0x1f61, nil],
      0x1f6a => [0, 0, "\341\275\250\314\200", "\341\275\250\314\200", nil, 0x1f62, nil],
      0x1f6b => [0, 0, "\341\275\251\314\200", "\341\275\251\314\200", nil, 0x1f63, nil],
      0x1f6c => [0, 0, "\341\275\250\314\201", "\341\275\250\314\201", nil, 0x1f64, nil],
      0x1f6d => [0, 0, "\341\275\251\314\201", "\341\275\251\314\201", nil, 0x1f65, nil],
      0x1f6e => [0, 0, "\341\275\250\315\202", "\341\275\250\315\202", nil, 0x1f66, nil],
      0x1f6f => [0, 0, "\341\275\251\315\202", "\341\275\251\315\202", nil, 0x1f67, nil],
      0x1f70 => [0, 0, "\316\261\314\200", "\316\261\314\200", 0x1fba, nil, 0x1fba],
      0x1f71 => [0, 2, "\316\254", "\316\254", 0x1fbb, nil, 0x1fbb],
      0x1f72 => [0, 0, "\316\265\314\200", "\316\265\314\200", 0x1fc8, nil, 0x1fc8],
      0x1f73 => [0, 2, "\316\255", "\316\255", 0x1fc9, nil, 0x1fc9],
      0x1f74 => [0, 0, "\316\267\314\200", "\316\267\314\200", 0x1fca, nil, 0x1fca],
      0x1f75 => [0, 2, "\316\256", "\316\256", 0x1fcb, nil, 0x1fcb],
      0x1f76 => [0, 0, "\316\271\314\200", "\316\271\314\200", 0x1fda, nil, 0x1fda],
      0x1f77 => [0, 2, "\316\257", "\316\257", 0x1fdb, nil, 0x1fdb],
      0x1f78 => [0, 0, "\316\277\314\200", "\316\277\314\200", 0x1ff8, nil, 0x1ff8],
      0x1f79 => [0, 2, "\317\214", "\317\214", 0x1ff9, nil, 0x1ff9],
      0x1f7a => [0, 0, "\317\205\314\200", "\317\205\314\200", 0x1fea, nil, 0x1fea],
      0x1f7b => [0, 2, "\317\215", "\317\215", 0x1feb, nil, 0x1feb],
      0x1f7c => [0, 0, "\317\211\314\200", "\317\211\314\200", 0x1ffa, nil, 0x1ffa],
      0x1f7d => [0, 2, "\317\216", "\317\216", 0x1ffb, nil, 0x1ffb],
      0x1f80 => [0, 0, "\341\274\200\315\205", "\341\274\200\315\205", 0x1f88, nil, 0x1f88],
      0x1f81 => [0, 0, "\341\274\201\315\205", "\341\274\201\315\205", 0x1f89, nil, 0x1f89],
      0x1f82 => [0, 0, "\341\274\202\315\205", "\341\274\202\315\205", 0x1f8a, nil, 0x1f8a],
      0x1f83 => [0, 0, "\341\274\203\315\205", "\341\274\203\315\205", 0x1f8b, nil, 0x1f8b],
      0x1f84 => [0, 0, "\341\274\204\315\205", "\341\274\204\315\205", 0x1f8c, nil, 0x1f8c],
      0x1f85 => [0, 0, "\341\274\205\315\205", "\341\274\205\315\205", 0x1f8d, nil, 0x1f8d],
      0x1f86 => [0, 0, "\341\274\206\315\205", "\341\274\206\315\205", 0x1f8e, nil, 0x1f8e],
      0x1f87 => [0, 0, "\341\274\207\315\205", "\341\274\207\315\205", 0x1f8f, nil, 0x1f8f],
      0x1f88 => [0, 0, "\341\274\210\315\205", "\341\274\210\315\205", nil, 0x1f80, nil],
      0x1f89 => [0, 0, "\341\274\211\315\205", "\341\274\211\315\205", nil, 0x1f81, nil],
      0x1f8a => [0, 0, "\341\274\212\315\205", "\341\274\212\315\205", nil, 0x1f82, nil],
      0x1f8b => [0, 0, "\341\274\213\315\205", "\341\274\213\315\205", nil, 0x1f83, nil],
      0x1f8c => [0, 0, "\341\274\214\315\205", "\341\274\214\315\205", nil, 0x1f84, nil],
      0x1f8d => [0, 0, "\341\274\215\315\205", "\341\274\215\315\205", nil, 0x1f85, nil],
      0x1f8e => [0, 0, "\341\274\216\315\205", "\341\274\216\315\205", nil, 0x1f86, nil],
      0x1f8f => [0, 0, "\341\274\217\315\205", "\341\274\217\315\205", nil, 0x1f87, nil],
      0x1f90 => [0, 0, "\341\274\240\315\205", "\341\274\240\315\205", 0x1f98, nil, 0x1f98],
      0x1f91 => [0, 0, "\341\274\241\315\205", "\341\274\241\315\205", 0x1f99, nil, 0x1f99],
      0x1f92 => [0, 0, "\341\274\242\315\205", "\341\274\242\315\205", 0x1f9a, nil, 0x1f9a],
      0x1f93 => [0, 0, "\341\274\243\315\205", "\341\274\243\315\205", 0x1f9b, nil, 0x1f9b],
      0x1f94 => [0, 0, "\341\274\244\315\205", "\341\274\244\315\205", 0x1f9c, nil, 0x1f9c],
      0x1f95 => [0, 0, "\341\274\245\315\205", "\341\274\245\315\205", 0x1f9d, nil, 0x1f9d],
      0x1f96 => [0, 0, "\341\274\246\315\205", "\341\274\246\315\205", 0x1f9e, nil, 0x1f9e],
      0x1f97 => [0, 0, "\341\274\247\315\205", "\341\274\247\315\205", 0x1f9f, nil, 0x1f9f],
      0x1f98 => [0, 0, "\341\274\250\315\205", "\341\274\250\315\205", nil, 0x1f90, nil],
      0x1f99 => [0, 0, "\341\274\251\315\205", "\341\274\251\315\205", nil, 0x1f91, nil],
      0x1f9a => [0, 0, "\341\274\252\315\205", "\341\274\252\315\205", nil, 0x1f92, nil],
      0x1f9b => [0, 0, "\341\274\253\315\205", "\341\274\253\315\205", nil, 0x1f93, nil],
      0x1f9c => [0, 0, "\341\274\254\315\205", "\341\274\254\315\205", nil, 0x1f94, nil],
      0x1f9d => [0, 0, "\341\274\255\315\205", "\341\274\255\315\205", nil, 0x1f95, nil],
      0x1f9e => [0, 0, "\341\274\256\315\205", "\341\274\256\315\205", nil, 0x1f96, nil],
      0x1f9f => [0, 0, "\341\274\257\315\205", "\341\274\257\315\205", nil, 0x1f97, nil],
      0x1fa0 => [0, 0, "\341\275\240\315\205", "\341\275\240\315\205", 0x1fa8, nil, 0x1fa8],
      0x1fa1 => [0, 0, "\341\275\241\315\205", "\341\275\241\315\205", 0x1fa9, nil, 0x1fa9],
      0x1fa2 => [0, 0, "\341\275\242\315\205", "\341\275\242\315\205", 0x1faa, nil, 0x1faa],
      0x1fa3 => [0, 0, "\341\275\243\315\205", "\341\275\243\315\205", 0x1fab, nil, 0x1fab],
      0x1fa4 => [0, 0, "\341\275\244\315\205", "\341\275\244\315\205", 0x1fac, nil, 0x1fac],
      0x1fa5 => [0, 0, "\341\275\245\315\205", "\341\275\245\315\205", 0x1fad, nil, 0x1fad],
      0x1fa6 => [0, 0, "\341\275\246\315\205", "\341\275\246\315\205", 0x1fae, nil, 0x1fae],
      0x1fa7 => [0, 0, "\341\275\247\315\205", "\341\275\247\315\205", 0x1faf, nil, 0x1faf],
      0x1fa8 => [0, 0, "\341\275\250\315\205", "\341\275\250\315\205", nil, 0x1fa0, nil],
      0x1fa9 => [0, 0, "\341\275\251\315\205", "\341\275\251\315\205", nil, 0x1fa1, nil],
      0x1faa => [0, 0, "\341\275\252\315\205", "\341\275\252\315\205", nil, 0x1fa2, nil],
      0x1fab => [0, 0, "\341\275\253\315\205", "\341\275\253\315\205", nil, 0x1fa3, nil],
      0x1fac => [0, 0, "\341\275\254\315\205", "\341\275\254\315\205", nil, 0x1fa4, nil],
      0x1fad => [0, 0, "\341\275\255\315\205", "\341\275\255\315\205", nil, 0x1fa5, nil],
      0x1fae => [0, 0, "\341\275\256\315\205", "\341\275\256\315\205", nil, 0x1fa6, nil],
      0x1faf => [0, 0, "\341\275\257\315\205", "\341\275\257\315\205", nil, 0x1fa7, nil],
      0x1fb0 => [0, 0, "\316\261\314\206", "\316\261\314\206", 0x1fb8, nil, 0x1fb8],
      0x1fb1 => [0, 0, "\316\261\314\204", "\316\261\314\204", 0x1fb9, nil, 0x1fb9],
      0x1fb2 => [0, 0, "\341\275\260\315\205", "\341\275\260\315\205", nil, nil, nil],
      0x1fb3 => [0, 0, "\316\261\315\205", "\316\261\315\205", 0x1fbc, nil, 0x1fbc],
      0x1fb4 => [0, 0, "\316\254\315\205", "\316\254\315\205", nil, nil, nil],
      0x1fb6 => [0, 0, "\316\261\315\202", "\316\261\315\202", nil, nil, nil],
      0x1fb7 => [0, 0, "\341\276\266\315\205", "\341\276\266\315\205", nil, nil, nil],
      0x1fb8 => [0, 0, "\316\221\314\206", "\316\221\314\206", nil, 0x1fb0, nil],
      0x1fb9 => [0, 0, "\316\221\314\204", "\316\221\314\204", nil, 0x1fb1, nil],
      0x1fba => [0, 0, "\316\221\314\200", "\316\221\314\200", nil, 0x1f70, nil],
      0x1fbb => [0, 2, "\316\206", "\316\206", nil, 0x1f71, nil],
      0x1fbc => [0, 0, "\316\221\315\205", "\316\221\315\205", nil, 0x1fb3, nil],
      0x1fbd => [0, 0, nil, " \314\223", nil, nil, nil],
      0x1fbe => [0, 2, "\316\271", "\316\271", 0x0399, nil, 0x0399],
      0x1fbf => [0, 0, nil, " \314\223", nil, nil, nil],
      0x1fc0 => [0, 0, nil, " \315\202", nil, nil, nil],
      0x1fc1 => [0, 0, "\302\250\315\202", "\302\250\315\202", nil, nil, nil],
      0x1fc2 => [0, 0, "\341\275\264\315\205", "\341\275\264\315\205", nil, nil, nil],
      0x1fc3 => [0, 0, "\316\267\315\205", "\316\267\315\205", 0x1fcc, nil, 0x1fcc],
      0x1fc4 => [0, 0, "\316\256\315\205", "\316\256\315\205", nil, nil, nil],
      0x1fc6 => [0, 0, "\316\267\315\202", "\316\267\315\202", nil, nil, nil],
      0x1fc7 => [0, 0, "\341\277\206\315\205", "\341\277\206\315\205", nil, nil, nil],
      0x1fc8 => [0, 0, "\316\225\314\200", "\316\225\314\200", nil, 0x1f72, nil],
      0x1fc9 => [0, 2, "\316\210", "\316\210", nil, 0x1f73, nil],
      0x1fca => [0, 0, "\316\227\314\200", "\316\227\314\200", nil, 0x1f74, nil],
      0x1fcb => [0, 2, "\316\211", "\316\211", nil, 0x1f75, nil],
      0x1fcc => [0, 0, "\316\227\315\205", "\316\227\315\205", nil, 0x1fc3, nil],
      0x1fcd => [0, 0, "\341\276\277\314\200", "\341\276\277\314\200", nil, nil, nil],
      0x1fce => [0, 0, "\341\276\277\314\201", "\341\276\277\314\201", nil, nil, nil],
      0x1fcf => [0, 0, "\341\276\277\315\202", "\341\276\277\315\202", nil, nil, nil],
      0x1fd0 => [0, 0, "\316\271\314\206", "\316\271\314\206", 0x1fd8, nil, 0x1fd8],
      0x1fd1 => [0, 0, "\316\271\314\204", "\316\271\314\204", 0x1fd9, nil, 0x1fd9],
      0x1fd2 => [0, 0, "\317\212\314\200", "\317\212\314\200", nil, nil, nil],
      0x1fd3 => [0, 2, "\316\220", "\316\220", nil, nil, nil],
      0x1fd6 => [0, 0, "\316\271\315\202", "\316\271\315\202", nil, nil, nil],
      0x1fd7 => [0, 0, "\317\212\315\202", "\317\212\315\202", nil, nil, nil],
      0x1fd8 => [0, 0, "\316\231\314\206", "\316\231\314\206", nil, 0x1fd0, nil],
      0x1fd9 => [0, 0, "\316\231\314\204", "\316\231\314\204", nil, 0x1fd1, nil],
      0x1fda => [0, 0, "\316\231\314\200", "\316\231\314\200", nil, 0x1f76, nil],
      0x1fdb => [0, 2, "\316\212", "\316\212", nil, 0x1f77, nil],
      0x1fdd => [0, 0, "\341\277\276\314\200", "\341\277\276\314\200", nil, nil, nil],
      0x1fde => [0, 0, "\341\277\276\314\201", "\341\277\276\314\201", nil, nil, nil],
      0x1fdf => [0, 0, "\341\277\276\315\202", "\341\277\276\315\202", nil, nil, nil],
      0x1fe0 => [0, 0, "\317\205\314\206", "\317\205\314\206", 0x1fe8, nil, 0x1fe8],
      0x1fe1 => [0, 0, "\317\205\314\204", "\317\205\314\204", 0x1fe9, nil, 0x1fe9],
      0x1fe2 => [0, 0, "\317\213\314\200", "\317\213\314\200", nil, nil, nil],
      0x1fe3 => [0, 2, "\316\260", "\316\260", nil, nil, nil],
      0x1fe4 => [0, 0, "\317\201\314\223", "\317\201\314\223", nil, nil, nil],
      0x1fe5 => [0, 0, "\317\201\314\224", "\317\201\314\224", 0x1fec, nil, 0x1fec],
      0x1fe6 => [0, 0, "\317\205\315\202", "\317\205\315\202", nil, nil, nil],
      0x1fe7 => [0, 0, "\317\213\315\202", "\317\213\315\202", nil, nil, nil],
      0x1fe8 => [0, 0, "\316\245\314\206", "\316\245\314\206", nil, 0x1fe0, nil],
      0x1fe9 => [0, 0, "\316\245\314\204", "\316\245\314\204", nil, 0x1fe1, nil],
      0x1fea => [0, 0, "\316\245\314\200", "\316\245\314\200", nil, 0x1f7a, nil],
      0x1feb => [0, 2, "\316\216", "\316\216", nil, 0x1f7b, nil],
      0x1fec => [0, 0, "\316\241\314\224", "\316\241\314\224", nil, 0x1fe5, nil],
      0x1fed => [0, 0, "\302\250\314\200", "\302\250\314\200", nil, nil, nil],
      0x1fee => [0, 2, "\316\205", "\316\205", nil, nil, nil],
      0x1fef => [0, 2, "`", "`", nil, nil, nil],
      0x1ff2 => [0, 0, "\341\275\274\315\205", "\341\275\274\315\205", nil, nil, nil],
      0x1ff3 => [0, 0, "\317\211\315\205", "\317\211\315\205", 0x1ffc, nil, 0x1ffc],
      0x1ff4 => [0, 0, "\317\216\315\205", "\317\216\315\205", nil, nil, nil],
      0x1ff6 => [0, 0, "\317\211\315\202", "\317\211\315\202", nil, nil, nil],
      0x1ff7 => [0, 0, "\341\277\266\315\205", "\341\277\266\315\205", nil, nil, nil],
      0x1ff8 => [0, 0, "\316\237\314\200", "\316\237\314\200", nil, 0x1f78, nil],
      0x1ff9 => [0, 2, "\316\214", "\316\214", nil, 0x1f79, nil],
      0x1ffa => [0, 0, "\316\251\314\200", "\316\251\314\200", nil, 0x1f7c, nil],
      0x1ffb => [0, 2, "\316\217", "\316\217", nil, 0x1f7d, nil],
      0x1ffc => [0, 0, "\316\251\315\205", "\316\251\315\205", nil, 0x1ff3, nil],
      0x1ffd => [0, 2, "\302\264", "\302\264", nil, nil, nil],
      0x1ffe => [0, 0, nil, " \314\224", nil, nil, nil],
      0x2000 => [0, 2, "\342\200\202", "\342\200\202", nil, nil, nil],
      0x2001 => [0, 2, "\342\200\203", "\342\200\203", nil, nil, nil],
      0x2002 => [0, 0, nil, " ", nil, nil, nil],
      0x2003 => [0, 0, nil, " ", nil, nil, nil],
      0x2004 => [0, 0, nil, " ", nil, nil, nil],
      0x2005 => [0, 0, nil, " ", nil, nil, nil],
      0x2006 => [0, 0, nil, " ", nil, nil, nil],
      0x2007 => [0, 0, nil, " ", nil, nil, nil],
      0x2008 => [0, 0, nil, " ", nil, nil, nil],
      0x2009 => [0, 0, nil, " ", nil, nil, nil],
      0x200a => [0, 0, nil, " ", nil, nil, nil],
      0x2011 => [0, 0, nil, "\342\200\220", nil, nil, nil],
      0x2017 => [0, 0, nil, " \314\263", nil, nil, nil],
      0x2024 => [0, 0, nil, ".", nil, nil, nil],
      0x2025 => [0, 0, nil, "..", nil, nil, nil],
      0x2026 => [0, 0, nil, "...", nil, nil, nil],
      0x202f => [0, 0, nil, " ", nil, nil, nil],
      0x2033 => [0, 0, nil, "\342\200\262\342\200\262", nil, nil, nil],
      0x2034 => [0, 0, nil, "\342\200\262\342\200\262\342\200\262", nil, nil, nil],
      0x2036 => [0, 0, nil, "\342\200\265\342\200\265", nil, nil, nil],
      0x2037 => [0, 0, nil, "\342\200\265\342\200\265\342\200\265", nil, nil, nil],
      0x203c => [0, 0, nil, "!!", nil, nil, nil],
      0x203e => [0, 0, nil, " \314\205", nil, nil, nil],
      0x2048 => [0, 0, nil, "?!", nil, nil, nil],
      0x2049 => [0, 0, nil, "!?", nil, nil, nil],
      0x2070 => [0, 0, nil, "0", nil, nil, nil],
      0x2074 => [0, 0, nil, "4", nil, nil, nil],
      0x2075 => [0, 0, nil, "5", nil, nil, nil],
      0x2076 => [0, 0, nil, "6", nil, nil, nil],
      0x2077 => [0, 0, nil, "7", nil, nil, nil],
      0x2078 => [0, 0, nil, "8", nil, nil, nil],
      0x2079 => [0, 0, nil, "9", nil, nil, nil],
      0x207a => [0, 0, nil, "+", nil, nil, nil],
      0x207b => [0, 0, nil, "\342\210\222", nil, nil, nil],
      0x207c => [0, 0, nil, "=", nil, nil, nil],
      0x207d => [0, 0, nil, "(", nil, nil, nil],
      0x207e => [0, 0, nil, ")", nil, nil, nil],
      0x207f => [0, 0, nil, "n", nil, nil, nil],
      0x2080 => [0, 0, nil, "0", nil, nil, nil],
      0x2081 => [0, 0, nil, "1", nil, nil, nil],
      0x2082 => [0, 0, nil, "2", nil, nil, nil],
      0x2083 => [0, 0, nil, "3", nil, nil, nil],
      0x2084 => [0, 0, nil, "4", nil, nil, nil],
      0x2085 => [0, 0, nil, "5", nil, nil, nil],
      0x2086 => [0, 0, nil, "6", nil, nil, nil],
      0x2087 => [0, 0, nil, "7", nil, nil, nil],
      0x2088 => [0, 0, nil, "8", nil, nil, nil],
      0x2089 => [0, 0, nil, "9", nil, nil, nil],
      0x208a => [0, 0, nil, "+", nil, nil, nil],
      0x208b => [0, 0, nil, "\342\210\222", nil, nil, nil],
      0x208c => [0, 0, nil, "=", nil, nil, nil],
      0x208d => [0, 0, nil, "(", nil, nil, nil],
      0x208e => [0, 0, nil, ")", nil, nil, nil],
      0x20a8 => [0, 0, nil, "Rs", nil, nil, nil],
      0x20d0 => [230, 0, nil, nil, nil, nil, nil],
      0x20d1 => [230, 0, nil, nil, nil, nil, nil],
      0x20d2 => [1, 0, nil, nil, nil, nil, nil],
      0x20d3 => [1, 0, nil, nil, nil, nil, nil],
      0x20d4 => [230, 0, nil, nil, nil, nil, nil],
      0x20d5 => [230, 0, nil, nil, nil, nil, nil],
      0x20d6 => [230, 0, nil, nil, nil, nil, nil],
      0x20d7 => [230, 0, nil, nil, nil, nil, nil],
      0x20d8 => [1, 0, nil, nil, nil, nil, nil],
      0x20d9 => [1, 0, nil, nil, nil, nil, nil],
      0x20da => [1, 0, nil, nil, nil, nil, nil],
      0x20db => [230, 0, nil, nil, nil, nil, nil],
      0x20dc => [230, 0, nil, nil, nil, nil, nil],
      0x20e1 => [230, 0, nil, nil, nil, nil, nil],
      0x2100 => [0, 0, nil, "a/c", nil, nil, nil],
      0x2101 => [0, 0, nil, "a/s", nil, nil, nil],
      0x2102 => [0, 0, nil, "C", nil, nil, nil],
      0x2103 => [0, 0, nil, "\302\260C", nil, nil, nil],
      0x2105 => [0, 0, nil, "c/o", nil, nil, nil],
      0x2106 => [0, 0, nil, "c/u", nil, nil, nil],
      0x2107 => [0, 0, nil, "\306\220", nil, nil, nil],
      0x2109 => [0, 0, nil, "\302\260F", nil, nil, nil],
      0x210a => [0, 0, nil, "g", nil, nil, nil],
      0x210b => [0, 0, nil, "H", nil, nil, nil],
      0x210c => [0, 0, nil, "H", nil, nil, nil],
      0x210d => [0, 0, nil, "H", nil, nil, nil],
      0x210e => [0, 0, nil, "h", nil, nil, nil],
      0x210f => [0, 0, nil, "\304\247", nil, nil, nil],
      0x2110 => [0, 0, nil, "I", nil, nil, nil],
      0x2111 => [0, 0, nil, "I", nil, nil, nil],
      0x2112 => [0, 0, nil, "L", nil, nil, nil],
      0x2113 => [0, 0, nil, "l", nil, nil, nil],
      0x2115 => [0, 0, nil, "N", nil, nil, nil],
      0x2116 => [0, 0, nil, "No", nil, nil, nil],
      0x2119 => [0, 0, nil, "P", nil, nil, nil],
      0x211a => [0, 0, nil, "Q", nil, nil, nil],
      0x211b => [0, 0, nil, "R", nil, nil, nil],
      0x211c => [0, 0, nil, "R", nil, nil, nil],
      0x211d => [0, 0, nil, "R", nil, nil, nil],
      0x2120 => [0, 0, nil, "SM", nil, nil, nil],
      0x2121 => [0, 0, nil, "TEL", nil, nil, nil],
      0x2122 => [0, 0, nil, "TM", nil, nil, nil],
      0x2124 => [0, 0, nil, "Z", nil, nil, nil],
      0x2126 => [0, 2, "\316\251", "\316\251", nil, 0x03c9, nil],
      0x2128 => [0, 0, nil, "Z", nil, nil, nil],
      0x212a => [0, 2, "K", "K", nil, 0x006b, nil],
      0x212b => [0, 2, "\303\205", "\303\205", nil, 0x00e5, nil],
      0x212c => [0, 0, nil, "B", nil, nil, nil],
      0x212d => [0, 0, nil, "C", nil, nil, nil],
      0x212f => [0, 0, nil, "e", nil, nil, nil],
      0x2130 => [0, 0, nil, "E", nil, nil, nil],
      0x2131 => [0, 0, nil, "F", nil, nil, nil],
      0x2133 => [0, 0, nil, "M", nil, nil, nil],
      0x2134 => [0, 0, nil, "o", nil, nil, nil],
      0x2135 => [0, 0, nil, "\327\220", nil, nil, nil],
      0x2136 => [0, 0, nil, "\327\221", nil, nil, nil],
      0x2137 => [0, 0, nil, "\327\222", nil, nil, nil],
      0x2138 => [0, 0, nil, "\327\223", nil, nil, nil],
      0x2139 => [0, 0, nil, "i", nil, nil, nil],
      0x2153 => [0, 0, nil, "1\342\201\2043", nil, nil, nil],
      0x2154 => [0, 0, nil, "2\342\201\2043", nil, nil, nil],
      0x2155 => [0, 0, nil, "1\342\201\2045", nil, nil, nil],
      0x2156 => [0, 0, nil, "2\342\201\2045", nil, nil, nil],
      0x2157 => [0, 0, nil, "3\342\201\2045", nil, nil, nil],
      0x2158 => [0, 0, nil, "4\342\201\2045", nil, nil, nil],
      0x2159 => [0, 0, nil, "1\342\201\2046", nil, nil, nil],
      0x215a => [0, 0, nil, "5\342\201\2046", nil, nil, nil],
      0x215b => [0, 0, nil, "1\342\201\2048", nil, nil, nil],
      0x215c => [0, 0, nil, "3\342\201\2048", nil, nil, nil],
      0x215d => [0, 0, nil, "5\342\201\2048", nil, nil, nil],
      0x215e => [0, 0, nil, "7\342\201\2048", nil, nil, nil],
      0x215f => [0, 0, nil, "1\342\201\204", nil, nil, nil],
      0x2160 => [0, 0, nil, "I", nil, 0x2170, nil],
      0x2161 => [0, 0, nil, "II", nil, 0x2171, nil],
      0x2162 => [0, 0, nil, "III", nil, 0x2172, nil],
      0x2163 => [0, 0, nil, "IV", nil, 0x2173, nil],
      0x2164 => [0, 0, nil, "V", nil, 0x2174, nil],
      0x2165 => [0, 0, nil, "VI", nil, 0x2175, nil],
      0x2166 => [0, 0, nil, "VII", nil, 0x2176, nil],
      0x2167 => [0, 0, nil, "VIII", nil, 0x2177, nil],
      0x2168 => [0, 0, nil, "IX", nil, 0x2178, nil],
      0x2169 => [0, 0, nil, "X", nil, 0x2179, nil],
      0x216a => [0, 0, nil, "XI", nil, 0x217a, nil],
      0x216b => [0, 0, nil, "XII", nil, 0x217b, nil],
      0x216c => [0, 0, nil, "L", nil, 0x217c, nil],
      0x216d => [0, 0, nil, "C", nil, 0x217d, nil],
      0x216e => [0, 0, nil, "D", nil, 0x217e, nil],
      0x216f => [0, 0, nil, "M", nil, 0x217f, nil],
      0x2170 => [0, 0, nil, "i", 0x2160, nil, 0x2160],
      0x2171 => [0, 0, nil, "ii", 0x2161, nil, 0x2161],
      0x2172 => [0, 0, nil, "iii", 0x2162, nil, 0x2162],
      0x2173 => [0, 0, nil, "iv", 0x2163, nil, 0x2163],
      0x2174 => [0, 0, nil, "v", 0x2164, nil, 0x2164],
      0x2175 => [0, 0, nil, "vi", 0x2165, nil, 0x2165],
      0x2176 => [0, 0, nil, "vii", 0x2166, nil, 0x2166],
      0x2177 => [0, 0, nil, "viii", 0x2167, nil, 0x2167],
      0x2178 => [0, 0, nil, "ix", 0x2168, nil, 0x2168],
      0x2179 => [0, 0, nil, "x", 0x2169, nil, 0x2169],
      0x217a => [0, 0, nil, "xi", 0x216a, nil, 0x216a],
      0x217b => [0, 0, nil, "xii", 0x216b, nil, 0x216b],
      0x217c => [0, 0, nil, "l", 0x216c, nil, 0x216c],
      0x217d => [0, 0, nil, "c", 0x216d, nil, 0x216d],
      0x217e => [0, 0, nil, "d", 0x216e, nil, 0x216e],
      0x217f => [0, 0, nil, "m", 0x216f, nil, 0x216f],
      0x219a => [0, 0, "\342\206\220\314\270", "\342\206\220\314\270", nil, nil, nil],
      0x219b => [0, 0, "\342\206\222\314\270", "\342\206\222\314\270", nil, nil, nil],
      0x21ae => [0, 0, "\342\206\224\314\270", "\342\206\224\314\270", nil, nil, nil],
      0x21cd => [0, 0, "\342\207\220\314\270", "\342\207\220\314\270", nil, nil, nil],
      0x21ce => [0, 0, "\342\207\224\314\270", "\342\207\224\314\270", nil, nil, nil],
      0x21cf => [0, 0, "\342\207\222\314\270", "\342\207\222\314\270", nil, nil, nil],
      0x2204 => [0, 0, "\342\210\203\314\270", "\342\210\203\314\270", nil, nil, nil],
      0x2209 => [0, 0, "\342\210\210\314\270", "\342\210\210\314\270", nil, nil, nil],
      0x220c => [0, 0, "\342\210\213\314\270", "\342\210\213\314\270", nil, nil, nil],
      0x2224 => [0, 0, "\342\210\243\314\270", "\342\210\243\314\270", nil, nil, nil],
      0x2226 => [0, 0, "\342\210\245\314\270", "\342\210\245\314\270", nil, nil, nil],
      0x222c => [0, 0, nil, "\342\210\253\342\210\253", nil, nil, nil],
      0x222d => [0, 0, nil, "\342\210\253\342\210\253\342\210\253", nil, nil, nil],
      0x222f => [0, 0, nil, "\342\210\256\342\210\256", nil, nil, nil],
      0x2230 => [0, 0, nil, "\342\210\256\342\210\256\342\210\256", nil, nil, nil],
      0x2241 => [0, 0, "\342\210\274\314\270", "\342\210\274\314\270", nil, nil, nil],
      0x2244 => [0, 0, "\342\211\203\314\270", "\342\211\203\314\270", nil, nil, nil],
      0x2247 => [0, 0, "\342\211\205\314\270", "\342\211\205\314\270", nil, nil, nil],
      0x2249 => [0, 0, "\342\211\210\314\270", "\342\211\210\314\270", nil, nil, nil],
      0x2260 => [0, 0, "=\314\270", "=\314\270", nil, nil, nil],
      0x2262 => [0, 0, "\342\211\241\314\270", "\342\211\241\314\270", nil, nil, nil],
      0x226d => [0, 0, "\342\211\215\314\270", "\342\211\215\314\270", nil, nil, nil],
      0x226e => [0, 0, "<\314\270", "<\314\270", nil, nil, nil],
      0x226f => [0, 0, ">\314\270", ">\314\270", nil, nil, nil],
      0x2270 => [0, 0, "\342\211\244\314\270", "\342\211\244\314\270", nil, nil, nil],
      0x2271 => [0, 0, "\342\211\245\314\270", "\342\211\245\314\270", nil, nil, nil],
      0x2274 => [0, 0, "\342\211\262\314\270", "\342\211\262\314\270", nil, nil, nil],
      0x2275 => [0, 0, "\342\211\263\314\270", "\342\211\263\314\270", nil, nil, nil],
      0x2278 => [0, 0, "\342\211\266\314\270", "\342\211\266\314\270", nil, nil, nil],
      0x2279 => [0, 0, "\342\211\267\314\270", "\342\211\267\314\270", nil, nil, nil],
      0x2280 => [0, 0, "\342\211\272\314\270", "\342\211\272\314\270", nil, nil, nil],
      0x2281 => [0, 0, "\342\211\273\314\270", "\342\211\273\314\270", nil, nil, nil],
      0x2284 => [0, 0, "\342\212\202\314\270", "\342\212\202\314\270", nil, nil, nil],
      0x2285 => [0, 0, "\342\212\203\314\270", "\342\212\203\314\270", nil, nil, nil],
      0x2288 => [0, 0, "\342\212\206\314\270", "\342\212\206\314\270", nil, nil, nil],
      0x2289 => [0, 0, "\342\212\207\314\270", "\342\212\207\314\270", nil, nil, nil],
      0x22ac => [0, 0, "\342\212\242\314\270", "\342\212\242\314\270", nil, nil, nil],
      0x22ad => [0, 0, "\342\212\250\314\270", "\342\212\250\314\270", nil, nil, nil],
      0x22ae => [0, 0, "\342\212\251\314\270", "\342\212\251\314\270", nil, nil, nil],
      0x22af => [0, 0, "\342\212\253\314\270", "\342\212\253\314\270", nil, nil, nil],
      0x22e0 => [0, 0, "\342\211\274\314\270", "\342\211\274\314\270", nil, nil, nil],
      0x22e1 => [0, 0, "\342\211\275\314\270", "\342\211\275\314\270", nil, nil, nil],
      0x22e2 => [0, 0, "\342\212\221\314\270", "\342\212\221\314\270", nil, nil, nil],
      0x22e3 => [0, 0, "\342\212\222\314\270", "\342\212\222\314\270", nil, nil, nil],
      0x22ea => [0, 0, "\342\212\262\314\270", "\342\212\262\314\270", nil, nil, nil],
      0x22eb => [0, 0, "\342\212\263\314\270", "\342\212\263\314\270", nil, nil, nil],
      0x22ec => [0, 0, "\342\212\264\314\270", "\342\212\264\314\270", nil, nil, nil],
      0x22ed => [0, 0, "\342\212\265\314\270", "\342\212\265\314\270", nil, nil, nil],
      0x2329 => [0, 2, "\343\200\210", "\343\200\210", nil, nil, nil],
      0x232a => [0, 2, "\343\200\211", "\343\200\211", nil, nil, nil],
      0x2460 => [0, 0, nil, "1", nil, nil, nil],
      0x2461 => [0, 0, nil, "2", nil, nil, nil],
      0x2462 => [0, 0, nil, "3", nil, nil, nil],
      0x2463 => [0, 0, nil, "4", nil, nil, nil],
      0x2464 => [0, 0, nil, "5", nil, nil, nil],
      0x2465 => [0, 0, nil, "6", nil, nil, nil],
      0x2466 => [0, 0, nil, "7", nil, nil, nil],
      0x2467 => [0, 0, nil, "8", nil, nil, nil],
      0x2468 => [0, 0, nil, "9", nil, nil, nil],
      0x2469 => [0, 0, nil, "10", nil, nil, nil],
      0x246a => [0, 0, nil, "11", nil, nil, nil],
      0x246b => [0, 0, nil, "12", nil, nil, nil],
      0x246c => [0, 0, nil, "13", nil, nil, nil],
      0x246d => [0, 0, nil, "14", nil, nil, nil],
      0x246e => [0, 0, nil, "15", nil, nil, nil],
      0x246f => [0, 0, nil, "16", nil, nil, nil],
      0x2470 => [0, 0, nil, "17", nil, nil, nil],
      0x2471 => [0, 0, nil, "18", nil, nil, nil],
      0x2472 => [0, 0, nil, "19", nil, nil, nil],
      0x2473 => [0, 0, nil, "20", nil, nil, nil],
      0x2474 => [0, 0, nil, "(1)", nil, nil, nil],
      0x2475 => [0, 0, nil, "(2)", nil, nil, nil],
      0x2476 => [0, 0, nil, "(3)", nil, nil, nil],
      0x2477 => [0, 0, nil, "(4)", nil, nil, nil],
      0x2478 => [0, 0, nil, "(5)", nil, nil, nil],
      0x2479 => [0, 0, nil, "(6)", nil, nil, nil],
      0x247a => [0, 0, nil, "(7)", nil, nil, nil],
      0x247b => [0, 0, nil, "(8)", nil, nil, nil],
      0x247c => [0, 0, nil, "(9)", nil, nil, nil],
      0x247d => [0, 0, nil, "(10)", nil, nil, nil],
      0x247e => [0, 0, nil, "(11)", nil, nil, nil],
      0x247f => [0, 0, nil, "(12)", nil, nil, nil],
      0x2480 => [0, 0, nil, "(13)", nil, nil, nil],
      0x2481 => [0, 0, nil, "(14)", nil, nil, nil],
      0x2482 => [0, 0, nil, "(15)", nil, nil, nil],
      0x2483 => [0, 0, nil, "(16)", nil, nil, nil],
      0x2484 => [0, 0, nil, "(17)", nil, nil, nil],
      0x2485 => [0, 0, nil, "(18)", nil, nil, nil],
      0x2486 => [0, 0, nil, "(19)", nil, nil, nil],
      0x2487 => [0, 0, nil, "(20)", nil, nil, nil],
      0x2488 => [0, 0, nil, "1.", nil, nil, nil],
      0x2489 => [0, 0, nil, "2.", nil, nil, nil],
      0x248a => [0, 0, nil, "3.", nil, nil, nil],
      0x248b => [0, 0, nil, "4.", nil, nil, nil],
      0x248c => [0, 0, nil, "5.", nil, nil, nil],
      0x248d => [0, 0, nil, "6.", nil, nil, nil],
      0x248e => [0, 0, nil, "7.", nil, nil, nil],
      0x248f => [0, 0, nil, "8.", nil, nil, nil],
      0x2490 => [0, 0, nil, "9.", nil, nil, nil],
      0x2491 => [0, 0, nil, "10.", nil, nil, nil],
      0x2492 => [0, 0, nil, "11.", nil, nil, nil],
      0x2493 => [0, 0, nil, "12.", nil, nil, nil],
      0x2494 => [0, 0, nil, "13.", nil, nil, nil],
      0x2495 => [0, 0, nil, "14.", nil, nil, nil],
      0x2496 => [0, 0, nil, "15.", nil, nil, nil],
      0x2497 => [0, 0, nil, "16.", nil, nil, nil],
      0x2498 => [0, 0, nil, "17.", nil, nil, nil],
      0x2499 => [0, 0, nil, "18.", nil, nil, nil],
      0x249a => [0, 0, nil, "19.", nil, nil, nil],
      0x249b => [0, 0, nil, "20.", nil, nil, nil],
      0x249c => [0, 0, nil, "(a)", nil, nil, nil],
      0x249d => [0, 0, nil, "(b)", nil, nil, nil],
      0x249e => [0, 0, nil, "(c)", nil, nil, nil],
      0x249f => [0, 0, nil, "(d)", nil, nil, nil],
      0x24a0 => [0, 0, nil, "(e)", nil, nil, nil],
      0x24a1 => [0, 0, nil, "(f)", nil, nil, nil],
      0x24a2 => [0, 0, nil, "(g)", nil, nil, nil],
      0x24a3 => [0, 0, nil, "(h)", nil, nil, nil],
      0x24a4 => [0, 0, nil, "(i)", nil, nil, nil],
      0x24a5 => [0, 0, nil, "(j)", nil, nil, nil],
      0x24a6 => [0, 0, nil, "(k)", nil, nil, nil],
      0x24a7 => [0, 0, nil, "(l)", nil, nil, nil],
      0x24a8 => [0, 0, nil, "(m)", nil, nil, nil],
      0x24a9 => [0, 0, nil, "(n)", nil, nil, nil],
      0x24aa => [0, 0, nil, "(o)", nil, nil, nil],
      0x24ab => [0, 0, nil, "(p)", nil, nil, nil],
      0x24ac => [0, 0, nil, "(q)", nil, nil, nil],
      0x24ad => [0, 0, nil, "(r)", nil, nil, nil],
      0x24ae => [0, 0, nil, "(s)", nil, nil, nil],
      0x24af => [0, 0, nil, "(t)", nil, nil, nil],
      0x24b0 => [0, 0, nil, "(u)", nil, nil, nil],
      0x24b1 => [0, 0, nil, "(v)", nil, nil, nil],
      0x24b2 => [0, 0, nil, "(w)", nil, nil, nil],
      0x24b3 => [0, 0, nil, "(x)", nil, nil, nil],
      0x24b4 => [0, 0, nil, "(y)", nil, nil, nil],
      0x24b5 => [0, 0, nil, "(z)", nil, nil, nil],
      0x24b6 => [0, 0, nil, "A", nil, 0x24d0, nil],
      0x24b7 => [0, 0, nil, "B", nil, 0x24d1, nil],
      0x24b8 => [0, 0, nil, "C", nil, 0x24d2, nil],
      0x24b9 => [0, 0, nil, "D", nil, 0x24d3, nil],
      0x24ba => [0, 0, nil, "E", nil, 0x24d4, nil],
      0x24bb => [0, 0, nil, "F", nil, 0x24d5, nil],
      0x24bc => [0, 0, nil, "G", nil, 0x24d6, nil],
      0x24bd => [0, 0, nil, "H", nil, 0x24d7, nil],
      0x24be => [0, 0, nil, "I", nil, 0x24d8, nil],
      0x24bf => [0, 0, nil, "J", nil, 0x24d9, nil],
      0x24c0 => [0, 0, nil, "K", nil, 0x24da, nil],
      0x24c1 => [0, 0, nil, "L", nil, 0x24db, nil],
      0x24c2 => [0, 0, nil, "M", nil, 0x24dc, nil],
      0x24c3 => [0, 0, nil, "N", nil, 0x24dd, nil],
      0x24c4 => [0, 0, nil, "O", nil, 0x24de, nil],
      0x24c5 => [0, 0, nil, "P", nil, 0x24df, nil],
      0x24c6 => [0, 0, nil, "Q", nil, 0x24e0, nil],
      0x24c7 => [0, 0, nil, "R", nil, 0x24e1, nil],
      0x24c8 => [0, 0, nil, "S", nil, 0x24e2, nil],
      0x24c9 => [0, 0, nil, "T", nil, 0x24e3, nil],
      0x24ca => [0, 0, nil, "U", nil, 0x24e4, nil],
      0x24cb => [0, 0, nil, "V", nil, 0x24e5, nil],
      0x24cc => [0, 0, nil, "W", nil, 0x24e6, nil],
      0x24cd => [0, 0, nil, "X", nil, 0x24e7, nil],
      0x24ce => [0, 0, nil, "Y", nil, 0x24e8, nil],
      0x24cf => [0, 0, nil, "Z", nil, 0x24e9, nil],
      0x24d0 => [0, 0, nil, "a", 0x24b6, nil, 0x24b6],
      0x24d1 => [0, 0, nil, "b", 0x24b7, nil, 0x24b7],
      0x24d2 => [0, 0, nil, "c", 0x24b8, nil, 0x24b8],
      0x24d3 => [0, 0, nil, "d", 0x24b9, nil, 0x24b9],
      0x24d4 => [0, 0, nil, "e", 0x24ba, nil, 0x24ba],
      0x24d5 => [0, 0, nil, "f", 0x24bb, nil, 0x24bb],
      0x24d6 => [0, 0, nil, "g", 0x24bc, nil, 0x24bc],
      0x24d7 => [0, 0, nil, "h", 0x24bd, nil, 0x24bd],
      0x24d8 => [0, 0, nil, "i", 0x24be, nil, 0x24be],
      0x24d9 => [0, 0, nil, "j", 0x24bf, nil, 0x24bf],
      0x24da => [0, 0, nil, "k", 0x24c0, nil, 0x24c0],
      0x24db => [0, 0, nil, "l", 0x24c1, nil, 0x24c1],
      0x24dc => [0, 0, nil, "m", 0x24c2, nil, 0x24c2],
      0x24dd => [0, 0, nil, "n", 0x24c3, nil, 0x24c3],
      0x24de => [0, 0, nil, "o", 0x24c4, nil, 0x24c4],
      0x24df => [0, 0, nil, "p", 0x24c5, nil, 0x24c5],
      0x24e0 => [0, 0, nil, "q", 0x24c6, nil, 0x24c6],
      0x24e1 => [0, 0, nil, "r", 0x24c7, nil, 0x24c7],
      0x24e2 => [0, 0, nil, "s", 0x24c8, nil, 0x24c8],
      0x24e3 => [0, 0, nil, "t", 0x24c9, nil, 0x24c9],
      0x24e4 => [0, 0, nil, "u", 0x24ca, nil, 0x24ca],
      0x24e5 => [0, 0, nil, "v", 0x24cb, nil, 0x24cb],
      0x24e6 => [0, 0, nil, "w", 0x24cc, nil, 0x24cc],
      0x24e7 => [0, 0, nil, "x", 0x24cd, nil, 0x24cd],
      0x24e8 => [0, 0, nil, "y", 0x24ce, nil, 0x24ce],
      0x24e9 => [0, 0, nil, "z", 0x24cf, nil, 0x24cf],
      0x24ea => [0, 0, nil, "0", nil, nil, nil],
      0x2e9f => [0, 0, nil, "\346\257\215", nil, nil, nil],
      0x2ef3 => [0, 0, nil, "\351\276\237", nil, nil, nil],
      0x2f00 => [0, 0, nil, "\344\270\200", nil, nil, nil],
      0x2f01 => [0, 0, nil, "\344\270\250", nil, nil, nil],
      0x2f02 => [0, 0, nil, "\344\270\266", nil, nil, nil],
      0x2f03 => [0, 0, nil, "\344\270\277", nil, nil, nil],
      0x2f04 => [0, 0, nil, "\344\271\231", nil, nil, nil],
      0x2f05 => [0, 0, nil, "\344\272\205", nil, nil, nil],
      0x2f06 => [0, 0, nil, "\344\272\214", nil, nil, nil],
      0x2f07 => [0, 0, nil, "\344\272\240", nil, nil, nil],
      0x2f08 => [0, 0, nil, "\344\272\272", nil, nil, nil],
      0x2f09 => [0, 0, nil, "\345\204\277", nil, nil, nil],
      0x2f0a => [0, 0, nil, "\345\205\245", nil, nil, nil],
      0x2f0b => [0, 0, nil, "\345\205\253", nil, nil, nil],
      0x2f0c => [0, 0, nil, "\345\206\202", nil, nil, nil],
      0x2f0d => [0, 0, nil, "\345\206\226", nil, nil, nil],
      0x2f0e => [0, 0, nil, "\345\206\253", nil, nil, nil],
      0x2f0f => [0, 0, nil, "\345\207\240", nil, nil, nil],
      0x2f10 => [0, 0, nil, "\345\207\265", nil, nil, nil],
      0x2f11 => [0, 0, nil, "\345\210\200", nil, nil, nil],
      0x2f12 => [0, 0, nil, "\345\212\233", nil, nil, nil],
      0x2f13 => [0, 0, nil, "\345\213\271", nil, nil, nil],
      0x2f14 => [0, 0, nil, "\345\214\225", nil, nil, nil],
      0x2f15 => [0, 0, nil, "\345\214\232", nil, nil, nil],
      0x2f16 => [0, 0, nil, "\345\214\270", nil, nil, nil],
      0x2f17 => [0, 0, nil, "\345\215\201", nil, nil, nil],
      0x2f18 => [0, 0, nil, "\345\215\234", nil, nil, nil],
      0x2f19 => [0, 0, nil, "\345\215\251", nil, nil, nil],
      0x2f1a => [0, 0, nil, "\345\216\202", nil, nil, nil],
      0x2f1b => [0, 0, nil, "\345\216\266", nil, nil, nil],
      0x2f1c => [0, 0, nil, "\345\217\210", nil, nil, nil],
      0x2f1d => [0, 0, nil, "\345\217\243", nil, nil, nil],
      0x2f1e => [0, 0, nil, "\345\233\227", nil, nil, nil],
      0x2f1f => [0, 0, nil, "\345\234\237", nil, nil, nil],
      0x2f20 => [0, 0, nil, "\345\243\253", nil, nil, nil],
      0x2f21 => [0, 0, nil, "\345\244\202", nil, nil, nil],
      0x2f22 => [0, 0, nil, "\345\244\212", nil, nil, nil],
      0x2f23 => [0, 0, nil, "\345\244\225", nil, nil, nil],
      0x2f24 => [0, 0, nil, "\345\244\247", nil, nil, nil],
      0x2f25 => [0, 0, nil, "\345\245\263", nil, nil, nil],
      0x2f26 => [0, 0, nil, "\345\255\220", nil, nil, nil],
      0x2f27 => [0, 0, nil, "\345\256\200", nil, nil, nil],
      0x2f28 => [0, 0, nil, "\345\257\270", nil, nil, nil],
      0x2f29 => [0, 0, nil, "\345\260\217", nil, nil, nil],
      0x2f2a => [0, 0, nil, "\345\260\242", nil, nil, nil],
      0x2f2b => [0, 0, nil, "\345\260\270", nil, nil, nil],
      0x2f2c => [0, 0, nil, "\345\261\256", nil, nil, nil],
      0x2f2d => [0, 0, nil, "\345\261\261", nil, nil, nil],
      0x2f2e => [0, 0, nil, "\345\267\233", nil, nil, nil],
      0x2f2f => [0, 0, nil, "\345\267\245", nil, nil, nil],
      0x2f30 => [0, 0, nil, "\345\267\261", nil, nil, nil],
      0x2f31 => [0, 0, nil, "\345\267\276", nil, nil, nil],
      0x2f32 => [0, 0, nil, "\345\271\262", nil, nil, nil],
      0x2f33 => [0, 0, nil, "\345\271\272", nil, nil, nil],
      0x2f34 => [0, 0, nil, "\345\271\277", nil, nil, nil],
      0x2f35 => [0, 0, nil, "\345\273\264", nil, nil, nil],
      0x2f36 => [0, 0, nil, "\345\273\276", nil, nil, nil],
      0x2f37 => [0, 0, nil, "\345\274\213", nil, nil, nil],
      0x2f38 => [0, 0, nil, "\345\274\223", nil, nil, nil],
      0x2f39 => [0, 0, nil, "\345\275\220", nil, nil, nil],
      0x2f3a => [0, 0, nil, "\345\275\241", nil, nil, nil],
      0x2f3b => [0, 0, nil, "\345\275\263", nil, nil, nil],
      0x2f3c => [0, 0, nil, "\345\277\203", nil, nil, nil],
      0x2f3d => [0, 0, nil, "\346\210\210", nil, nil, nil],
      0x2f3e => [0, 0, nil, "\346\210\266", nil, nil, nil],
      0x2f3f => [0, 0, nil, "\346\211\213", nil, nil, nil],
      0x2f40 => [0, 0, nil, "\346\224\257", nil, nil, nil],
      0x2f41 => [0, 0, nil, "\346\224\264", nil, nil, nil],
      0x2f42 => [0, 0, nil, "\346\226\207", nil, nil, nil],
      0x2f43 => [0, 0, nil, "\346\226\227", nil, nil, nil],
      0x2f44 => [0, 0, nil, "\346\226\244", nil, nil, nil],
      0x2f45 => [0, 0, nil, "\346\226\271", nil, nil, nil],
      0x2f46 => [0, 0, nil, "\346\227\240", nil, nil, nil],
      0x2f47 => [0, 0, nil, "\346\227\245", nil, nil, nil],
      0x2f48 => [0, 0, nil, "\346\233\260", nil, nil, nil],
      0x2f49 => [0, 0, nil, "\346\234\210", nil, nil, nil],
      0x2f4a => [0, 0, nil, "\346\234\250", nil, nil, nil],
      0x2f4b => [0, 0, nil, "\346\254\240", nil, nil, nil],
      0x2f4c => [0, 0, nil, "\346\255\242", nil, nil, nil],
      0x2f4d => [0, 0, nil, "\346\255\271", nil, nil, nil],
      0x2f4e => [0, 0, nil, "\346\256\263", nil, nil, nil],
      0x2f4f => [0, 0, nil, "\346\257\213", nil, nil, nil],
      0x2f50 => [0, 0, nil, "\346\257\224", nil, nil, nil],
      0x2f51 => [0, 0, nil, "\346\257\233", nil, nil, nil],
      0x2f52 => [0, 0, nil, "\346\260\217", nil, nil, nil],
      0x2f53 => [0, 0, nil, "\346\260\224", nil, nil, nil],
      0x2f54 => [0, 0, nil, "\346\260\264", nil, nil, nil],
      0x2f55 => [0, 0, nil, "\347\201\253", nil, nil, nil],
      0x2f56 => [0, 0, nil, "\347\210\252", nil, nil, nil],
      0x2f57 => [0, 0, nil, "\347\210\266", nil, nil, nil],
      0x2f58 => [0, 0, nil, "\347\210\273", nil, nil, nil],
      0x2f59 => [0, 0, nil, "\347\210\277", nil, nil, nil],
      0x2f5a => [0, 0, nil, "\347\211\207", nil, nil, nil],
      0x2f5b => [0, 0, nil, "\347\211\231", nil, nil, nil],
      0x2f5c => [0, 0, nil, "\347\211\233", nil, nil, nil],
      0x2f5d => [0, 0, nil, "\347\212\254", nil, nil, nil],
      0x2f5e => [0, 0, nil, "\347\216\204", nil, nil, nil],
      0x2f5f => [0, 0, nil, "\347\216\211", nil, nil, nil],
      0x2f60 => [0, 0, nil, "\347\223\234", nil, nil, nil],
      0x2f61 => [0, 0, nil, "\347\223\246", nil, nil, nil],
      0x2f62 => [0, 0, nil, "\347\224\230", nil, nil, nil],
      0x2f63 => [0, 0, nil, "\347\224\237", nil, nil, nil],
      0x2f64 => [0, 0, nil, "\347\224\250", nil, nil, nil],
      0x2f65 => [0, 0, nil, "\347\224\260", nil, nil, nil],
      0x2f66 => [0, 0, nil, "\347\226\213", nil, nil, nil],
      0x2f67 => [0, 0, nil, "\347\226\222", nil, nil, nil],
      0x2f68 => [0, 0, nil, "\347\231\266", nil, nil, nil],
      0x2f69 => [0, 0, nil, "\347\231\275", nil, nil, nil],
      0x2f6a => [0, 0, nil, "\347\232\256", nil, nil, nil],
      0x2f6b => [0, 0, nil, "\347\232\277", nil, nil, nil],
      0x2f6c => [0, 0, nil, "\347\233\256", nil, nil, nil],
      0x2f6d => [0, 0, nil, "\347\237\233", nil, nil, nil],
      0x2f6e => [0, 0, nil, "\347\237\242", nil, nil, nil],
      0x2f6f => [0, 0, nil, "\347\237\263", nil, nil, nil],
      0x2f70 => [0, 0, nil, "\347\244\272", nil, nil, nil],
      0x2f71 => [0, 0, nil, "\347\246\270", nil, nil, nil],
      0x2f72 => [0, 0, nil, "\347\246\276", nil, nil, nil],
      0x2f73 => [0, 0, nil, "\347\251\264", nil, nil, nil],
      0x2f74 => [0, 0, nil, "\347\253\213", nil, nil, nil],
      0x2f75 => [0, 0, nil, "\347\253\271", nil, nil, nil],
      0x2f76 => [0, 0, nil, "\347\261\263", nil, nil, nil],
      0x2f77 => [0, 0, nil, "\347\263\270", nil, nil, nil],
      0x2f78 => [0, 0, nil, "\347\274\266", nil, nil, nil],
      0x2f79 => [0, 0, nil, "\347\275\221", nil, nil, nil],
      0x2f7a => [0, 0, nil, "\347\276\212", nil, nil, nil],
      0x2f7b => [0, 0, nil, "\347\276\275", nil, nil, nil],
      0x2f7c => [0, 0, nil, "\350\200\201", nil, nil, nil],
      0x2f7d => [0, 0, nil, "\350\200\214", nil, nil, nil],
      0x2f7e => [0, 0, nil, "\350\200\222", nil, nil, nil],
      0x2f7f => [0, 0, nil, "\350\200\263", nil, nil, nil],
      0x2f80 => [0, 0, nil, "\350\201\277", nil, nil, nil],
      0x2f81 => [0, 0, nil, "\350\202\211", nil, nil, nil],
      0x2f82 => [0, 0, nil, "\350\207\243", nil, nil, nil],
      0x2f83 => [0, 0, nil, "\350\207\252", nil, nil, nil],
      0x2f84 => [0, 0, nil, "\350\207\263", nil, nil, nil],
      0x2f85 => [0, 0, nil, "\350\207\274", nil, nil, nil],
      0x2f86 => [0, 0, nil, "\350\210\214", nil, nil, nil],
      0x2f87 => [0, 0, nil, "\350\210\233", nil, nil, nil],
      0x2f88 => [0, 0, nil, "\350\210\237", nil, nil, nil],
      0x2f89 => [0, 0, nil, "\350\211\256", nil, nil, nil],
      0x2f8a => [0, 0, nil, "\350\211\262", nil, nil, nil],
      0x2f8b => [0, 0, nil, "\350\211\270", nil, nil, nil],
      0x2f8c => [0, 0, nil, "\350\231\215", nil, nil, nil],
      0x2f8d => [0, 0, nil, "\350\231\253", nil, nil, nil],
      0x2f8e => [0, 0, nil, "\350\241\200", nil, nil, nil],
      0x2f8f => [0, 0, nil, "\350\241\214", nil, nil, nil],
      0x2f90 => [0, 0, nil, "\350\241\243", nil, nil, nil],
      0x2f91 => [0, 0, nil, "\350\245\276", nil, nil, nil],
      0x2f92 => [0, 0, nil, "\350\246\213", nil, nil, nil],
      0x2f93 => [0, 0, nil, "\350\247\222", nil, nil, nil],
      0x2f94 => [0, 0, nil, "\350\250\200", nil, nil, nil],
      0x2f95 => [0, 0, nil, "\350\260\267", nil, nil, nil],
      0x2f96 => [0, 0, nil, "\350\261\206", nil, nil, nil],
      0x2f97 => [0, 0, nil, "\350\261\225", nil, nil, nil],
      0x2f98 => [0, 0, nil, "\350\261\270", nil, nil, nil],
      0x2f99 => [0, 0, nil, "\350\262\235", nil, nil, nil],
      0x2f9a => [0, 0, nil, "\350\265\244", nil, nil, nil],
      0x2f9b => [0, 0, nil, "\350\265\260", nil, nil, nil],
      0x2f9c => [0, 0, nil, "\350\266\263", nil, nil, nil],
      0x2f9d => [0, 0, nil, "\350\272\253", nil, nil, nil],
      0x2f9e => [0, 0, nil, "\350\273\212", nil, nil, nil],
      0x2f9f => [0, 0, nil, "\350\276\233", nil, nil, nil],
      0x2fa0 => [0, 0, nil, "\350\276\260", nil, nil, nil],
      0x2fa1 => [0, 0, nil, "\350\276\265", nil, nil, nil],
      0x2fa2 => [0, 0, nil, "\351\202\221", nil, nil, nil],
      0x2fa3 => [0, 0, nil, "\351\205\211", nil, nil, nil],
      0x2fa4 => [0, 0, nil, "\351\207\206", nil, nil, nil],
      0x2fa5 => [0, 0, nil, "\351\207\214", nil, nil, nil],
      0x2fa6 => [0, 0, nil, "\351\207\221", nil, nil, nil],
      0x2fa7 => [0, 0, nil, "\351\225\267", nil, nil, nil],
      0x2fa8 => [0, 0, nil, "\351\226\200", nil, nil, nil],
      0x2fa9 => [0, 0, nil, "\351\230\234", nil, nil, nil],
      0x2faa => [0, 0, nil, "\351\232\266", nil, nil, nil],
      0x2fab => [0, 0, nil, "\351\232\271", nil, nil, nil],
      0x2fac => [0, 0, nil, "\351\233\250", nil, nil, nil],
      0x2fad => [0, 0, nil, "\351\235\221", nil, nil, nil],
      0x2fae => [0, 0, nil, "\351\235\236", nil, nil, nil],
      0x2faf => [0, 0, nil, "\351\235\242", nil, nil, nil],
      0x2fb0 => [0, 0, nil, "\351\235\251", nil, nil, nil],
      0x2fb1 => [0, 0, nil, "\351\237\213", nil, nil, nil],
      0x2fb2 => [0, 0, nil, "\351\237\255", nil, nil, nil],
      0x2fb3 => [0, 0, nil, "\351\237\263", nil, nil, nil],
      0x2fb4 => [0, 0, nil, "\351\240\201", nil, nil, nil],
      0x2fb5 => [0, 0, nil, "\351\242\250", nil, nil, nil],
      0x2fb6 => [0, 0, nil, "\351\243\233", nil, nil, nil],
      0x2fb7 => [0, 0, nil, "\351\243\237", nil, nil, nil],
      0x2fb8 => [0, 0, nil, "\351\246\226", nil, nil, nil],
      0x2fb9 => [0, 0, nil, "\351\246\231", nil, nil, nil],
      0x2fba => [0, 0, nil, "\351\246\254", nil, nil, nil],
      0x2fbb => [0, 0, nil, "\351\252\250", nil, nil, nil],
      0x2fbc => [0, 0, nil, "\351\253\230", nil, nil, nil],
      0x2fbd => [0, 0, nil, "\351\253\237", nil, nil, nil],
      0x2fbe => [0, 0, nil, "\351\254\245", nil, nil, nil],
      0x2fbf => [0, 0, nil, "\351\254\257", nil, nil, nil],
      0x2fc0 => [0, 0, nil, "\351\254\262", nil, nil, nil],
      0x2fc1 => [0, 0, nil, "\351\254\274", nil, nil, nil],
      0x2fc2 => [0, 0, nil, "\351\255\232", nil, nil, nil],
      0x2fc3 => [0, 0, nil, "\351\263\245", nil, nil, nil],
      0x2fc4 => [0, 0, nil, "\351\271\265", nil, nil, nil],
      0x2fc5 => [0, 0, nil, "\351\271\277", nil, nil, nil],
      0x2fc6 => [0, 0, nil, "\351\272\245", nil, nil, nil],
      0x2fc7 => [0, 0, nil, "\351\272\273", nil, nil, nil],
      0x2fc8 => [0, 0, nil, "\351\273\203", nil, nil, nil],
      0x2fc9 => [0, 0, nil, "\351\273\215", nil, nil, nil],
      0x2fca => [0, 0, nil, "\351\273\221", nil, nil, nil],
      0x2fcb => [0, 0, nil, "\351\273\271", nil, nil, nil],
      0x2fcc => [0, 0, nil, "\351\273\275", nil, nil, nil],
      0x2fcd => [0, 0, nil, "\351\274\216", nil, nil, nil],
      0x2fce => [0, 0, nil, "\351\274\223", nil, nil, nil],
      0x2fcf => [0, 0, nil, "\351\274\240", nil, nil, nil],
      0x2fd0 => [0, 0, nil, "\351\274\273", nil, nil, nil],
      0x2fd1 => [0, 0, nil, "\351\275\212", nil, nil, nil],
      0x2fd2 => [0, 0, nil, "\351\275\222", nil, nil, nil],
      0x2fd3 => [0, 0, nil, "\351\276\215", nil, nil, nil],
      0x2fd4 => [0, 0, nil, "\351\276\234", nil, nil, nil],
      0x2fd5 => [0, 0, nil, "\351\276\240", nil, nil, nil],
      0x3000 => [0, 0, nil, " ", nil, nil, nil],
      0x302a => [218, 0, nil, nil, nil, nil, nil],
      0x302b => [228, 0, nil, nil, nil, nil, nil],
      0x302c => [232, 0, nil, nil, nil, nil, nil],
      0x302d => [222, 0, nil, nil, nil, nil, nil],
      0x302e => [224, 0, nil, nil, nil, nil, nil],
      0x302f => [224, 0, nil, nil, nil, nil, nil],
      0x3036 => [0, 0, nil, "\343\200\222", nil, nil, nil],
      0x3038 => [0, 0, nil, "\345\215\201", nil, nil, nil],
      0x3039 => [0, 0, nil, "\345\215\204", nil, nil, nil],
      0x303a => [0, 0, nil, "\345\215\205", nil, nil, nil],
      0x304c => [0, 0, "\343\201\213\343\202\231", "\343\201\213\343\202\231", nil, nil, nil],
      0x304e => [0, 0, "\343\201\215\343\202\231", "\343\201\215\343\202\231", nil, nil, nil],
      0x3050 => [0, 0, "\343\201\217\343\202\231", "\343\201\217\343\202\231", nil, nil, nil],
      0x3052 => [0, 0, "\343\201\221\343\202\231", "\343\201\221\343\202\231", nil, nil, nil],
      0x3054 => [0, 0, "\343\201\223\343\202\231", "\343\201\223\343\202\231", nil, nil, nil],
      0x3056 => [0, 0, "\343\201\225\343\202\231", "\343\201\225\343\202\231", nil, nil, nil],
      0x3058 => [0, 0, "\343\201\227\343\202\231", "\343\201\227\343\202\231", nil, nil, nil],
      0x305a => [0, 0, "\343\201\231\343\202\231", "\343\201\231\343\202\231", nil, nil, nil],
      0x305c => [0, 0, "\343\201\233\343\202\231", "\343\201\233\343\202\231", nil, nil, nil],
      0x305e => [0, 0, "\343\201\235\343\202\231", "\343\201\235\343\202\231", nil, nil, nil],
      0x3060 => [0, 0, "\343\201\237\343\202\231", "\343\201\237\343\202\231", nil, nil, nil],
      0x3062 => [0, 0, "\343\201\241\343\202\231", "\343\201\241\343\202\231", nil, nil, nil],
      0x3065 => [0, 0, "\343\201\244\343\202\231", "\343\201\244\343\202\231", nil, nil, nil],
      0x3067 => [0, 0, "\343\201\246\343\202\231", "\343\201\246\343\202\231", nil, nil, nil],
      0x3069 => [0, 0, "\343\201\250\343\202\231", "\343\201\250\343\202\231", nil, nil, nil],
      0x3070 => [0, 0, "\343\201\257\343\202\231", "\343\201\257\343\202\231", nil, nil, nil],
      0x3071 => [0, 0, "\343\201\257\343\202\232", "\343\201\257\343\202\232", nil, nil, nil],
      0x3073 => [0, 0, "\343\201\262\343\202\231", "\343\201\262\343\202\231", nil, nil, nil],
      0x3074 => [0, 0, "\343\201\262\343\202\232", "\343\201\262\343\202\232", nil, nil, nil],
      0x3076 => [0, 0, "\343\201\265\343\202\231", "\343\201\265\343\202\231", nil, nil, nil],
      0x3077 => [0, 0, "\343\201\265\343\202\232", "\343\201\265\343\202\232", nil, nil, nil],
      0x3079 => [0, 0, "\343\201\270\343\202\231", "\343\201\270\343\202\231", nil, nil, nil],
      0x307a => [0, 0, "\343\201\270\343\202\232", "\343\201\270\343\202\232", nil, nil, nil],
      0x307c => [0, 0, "\343\201\273\343\202\231", "\343\201\273\343\202\231", nil, nil, nil],
      0x307d => [0, 0, "\343\201\273\343\202\232", "\343\201\273\343\202\232", nil, nil, nil],
      0x3094 => [0, 0, "\343\201\206\343\202\231", "\343\201\206\343\202\231", nil, nil, nil],
      0x3099 => [8, 0, nil, nil, nil, nil, nil],
      0x309a => [8, 0, nil, nil, nil, nil, nil],
      0x309b => [0, 0, nil, " \343\202\231", nil, nil, nil],
      0x309c => [0, 0, nil, " \343\202\232", nil, nil, nil],
      0x309e => [0, 0, "\343\202\235\343\202\231", "\343\202\235\343\202\231", nil, nil, nil],
      0x30ac => [0, 0, "\343\202\253\343\202\231", "\343\202\253\343\202\231", nil, nil, nil],
      0x30ae => [0, 0, "\343\202\255\343\202\231", "\343\202\255\343\202\231", nil, nil, nil],
      0x30b0 => [0, 0, "\343\202\257\343\202\231", "\343\202\257\343\202\231", nil, nil, nil],
      0x30b2 => [0, 0, "\343\202\261\343\202\231", "\343\202\261\343\202\231", nil, nil, nil],
      0x30b4 => [0, 0, "\343\202\263\343\202\231", "\343\202\263\343\202\231", nil, nil, nil],
      0x30b6 => [0, 0, "\343\202\265\343\202\231", "\343\202\265\343\202\231", nil, nil, nil],
      0x30b8 => [0, 0, "\343\202\267\343\202\231", "\343\202\267\343\202\231", nil, nil, nil],
      0x30ba => [0, 0, "\343\202\271\343\202\231", "\343\202\271\343\202\231", nil, nil, nil],
      0x30bc => [0, 0, "\343\202\273\343\202\231", "\343\202\273\343\202\231", nil, nil, nil],
      0x30be => [0, 0, "\343\202\275\343\202\231", "\343\202\275\343\202\231", nil, nil, nil],
      0x30c0 => [0, 0, "\343\202\277\343\202\231", "\343\202\277\343\202\231", nil, nil, nil],
      0x30c2 => [0, 0, "\343\203\201\343\202\231", "\343\203\201\343\202\231", nil, nil, nil],
      0x30c5 => [0, 0, "\343\203\204\343\202\231", "\343\203\204\343\202\231", nil, nil, nil],
      0x30c7 => [0, 0, "\343\203\206\343\202\231", "\343\203\206\343\202\231", nil, nil, nil],
      0x30c9 => [0, 0, "\343\203\210\343\202\231", "\343\203\210\343\202\231", nil, nil, nil],
      0x30d0 => [0, 0, "\343\203\217\343\202\231", "\343\203\217\343\202\231", nil, nil, nil],
      0x30d1 => [0, 0, "\343\203\217\343\202\232", "\343\203\217\343\202\232", nil, nil, nil],
      0x30d3 => [0, 0, "\343\203\222\343\202\231", "\343\203\222\343\202\231", nil, nil, nil],
      0x30d4 => [0, 0, "\343\203\222\343\202\232", "\343\203\222\343\202\232", nil, nil, nil],
      0x30d6 => [0, 0, "\343\203\225\343\202\231", "\343\203\225\343\202\231", nil, nil, nil],
      0x30d7 => [0, 0, "\343\203\225\343\202\232", "\343\203\225\343\202\232", nil, nil, nil],
      0x30d9 => [0, 0, "\343\203\230\343\202\231", "\343\203\230\343\202\231", nil, nil, nil],
      0x30da => [0, 0, "\343\203\230\343\202\232", "\343\203\230\343\202\232", nil, nil, nil],
      0x30dc => [0, 0, "\343\203\233\343\202\231", "\343\203\233\343\202\231", nil, nil, nil],
      0x30dd => [0, 0, "\343\203\233\343\202\232", "\343\203\233\343\202\232", nil, nil, nil],
      0x30f4 => [0, 0, "\343\202\246\343\202\231", "\343\202\246\343\202\231", nil, nil, nil],
      0x30f7 => [0, 0, "\343\203\257\343\202\231", "\343\203\257\343\202\231", nil, nil, nil],
      0x30f8 => [0, 0, "\343\203\260\343\202\231", "\343\203\260\343\202\231", nil, nil, nil],
      0x30f9 => [0, 0, "\343\203\261\343\202\231", "\343\203\261\343\202\231", nil, nil, nil],
      0x30fa => [0, 0, "\343\203\262\343\202\231", "\343\203\262\343\202\231", nil, nil, nil],
      0x30fe => [0, 0, "\343\203\275\343\202\231", "\343\203\275\343\202\231", nil, nil, nil],
      0x3131 => [0, 0, nil, "\341\204\200", nil, nil, nil],
      0x3132 => [0, 0, nil, "\341\204\201", nil, nil, nil],
      0x3133 => [0, 0, nil, "\341\206\252", nil, nil, nil],
      0x3134 => [0, 0, nil, "\341\204\202", nil, nil, nil],
      0x3135 => [0, 0, nil, "\341\206\254", nil, nil, nil],
      0x3136 => [0, 0, nil, "\341\206\255", nil, nil, nil],
      0x3137 => [0, 0, nil, "\341\204\203", nil, nil, nil],
      0x3138 => [0, 0, nil, "\341\204\204", nil, nil, nil],
      0x3139 => [0, 0, nil, "\341\204\205", nil, nil, nil],
      0x313a => [0, 0, nil, "\341\206\260", nil, nil, nil],
      0x313b => [0, 0, nil, "\341\206\261", nil, nil, nil],
      0x313c => [0, 0, nil, "\341\206\262", nil, nil, nil],
      0x313d => [0, 0, nil, "\341\206\263", nil, nil, nil],
      0x313e => [0, 0, nil, "\341\206\264", nil, nil, nil],
      0x313f => [0, 0, nil, "\341\206\265", nil, nil, nil],
      0x3140 => [0, 0, nil, "\341\204\232", nil, nil, nil],
      0x3141 => [0, 0, nil, "\341\204\206", nil, nil, nil],
      0x3142 => [0, 0, nil, "\341\204\207", nil, nil, nil],
      0x3143 => [0, 0, nil, "\341\204\210", nil, nil, nil],
      0x3144 => [0, 0, nil, "\341\204\241", nil, nil, nil],
      0x3145 => [0, 0, nil, "\341\204\211", nil, nil, nil],
      0x3146 => [0, 0, nil, "\341\204\212", nil, nil, nil],
      0x3147 => [0, 0, nil, "\341\204\213", nil, nil, nil],
      0x3148 => [0, 0, nil, "\341\204\214", nil, nil, nil],
      0x3149 => [0, 0, nil, "\341\204\215", nil, nil, nil],
      0x314a => [0, 0, nil, "\341\204\216", nil, nil, nil],
      0x314b => [0, 0, nil, "\341\204\217", nil, nil, nil],
      0x314c => [0, 0, nil, "\341\204\220", nil, nil, nil],
      0x314d => [0, 0, nil, "\341\204\221", nil, nil, nil],
      0x314e => [0, 0, nil, "\341\204\222", nil, nil, nil],
      0x314f => [0, 0, nil, "\341\205\241", nil, nil, nil],
      0x3150 => [0, 0, nil, "\341\205\242", nil, nil, nil],
      0x3151 => [0, 0, nil, "\341\205\243", nil, nil, nil],
      0x3152 => [0, 0, nil, "\341\205\244", nil, nil, nil],
      0x3153 => [0, 0, nil, "\341\205\245", nil, nil, nil],
      0x3154 => [0, 0, nil, "\341\205\246", nil, nil, nil],
      0x3155 => [0, 0, nil, "\341\205\247", nil, nil, nil],
      0x3156 => [0, 0, nil, "\341\205\250", nil, nil, nil],
      0x3157 => [0, 0, nil, "\341\205\251", nil, nil, nil],
      0x3158 => [0, 0, nil, "\341\205\252", nil, nil, nil],
      0x3159 => [0, 0, nil, "\341\205\253", nil, nil, nil],
      0x315a => [0, 0, nil, "\341\205\254", nil, nil, nil],
      0x315b => [0, 0, nil, "\341\205\255", nil, nil, nil],
      0x315c => [0, 0, nil, "\341\205\256", nil, nil, nil],
      0x315d => [0, 0, nil, "\341\205\257", nil, nil, nil],
      0x315e => [0, 0, nil, "\341\205\260", nil, nil, nil],
      0x315f => [0, 0, nil, "\341\205\261", nil, nil, nil],
      0x3160 => [0, 0, nil, "\341\205\262", nil, nil, nil],
      0x3161 => [0, 0, nil, "\341\205\263", nil, nil, nil],
      0x3162 => [0, 0, nil, "\341\205\264", nil, nil, nil],
      0x3163 => [0, 0, nil, "\341\205\265", nil, nil, nil],
      0x3164 => [0, 0, nil, "\341\205\240", nil, nil, nil],
      0x3165 => [0, 0, nil, "\341\204\224", nil, nil, nil],
      0x3166 => [0, 0, nil, "\341\204\225", nil, nil, nil],
      0x3167 => [0, 0, nil, "\341\207\207", nil, nil, nil],
      0x3168 => [0, 0, nil, "\341\207\210", nil, nil, nil],
      0x3169 => [0, 0, nil, "\341\207\214", nil, nil, nil],
      0x316a => [0, 0, nil, "\341\207\216", nil, nil, nil],
      0x316b => [0, 0, nil, "\341\207\223", nil, nil, nil],
      0x316c => [0, 0, nil, "\341\207\227", nil, nil, nil],
      0x316d => [0, 0, nil, "\341\207\231", nil, nil, nil],
      0x316e => [0, 0, nil, "\341\204\234", nil, nil, nil],
      0x316f => [0, 0, nil, "\341\207\235", nil, nil, nil],
      0x3170 => [0, 0, nil, "\341\207\237", nil, nil, nil],
      0x3171 => [0, 0, nil, "\341\204\235", nil, nil, nil],
      0x3172 => [0, 0, nil, "\341\204\236", nil, nil, nil],
      0x3173 => [0, 0, nil, "\341\204\240", nil, nil, nil],
      0x3174 => [0, 0, nil, "\341\204\242", nil, nil, nil],
      0x3175 => [0, 0, nil, "\341\204\243", nil, nil, nil],
      0x3176 => [0, 0, nil, "\341\204\247", nil, nil, nil],
      0x3177 => [0, 0, nil, "\341\204\251", nil, nil, nil],
      0x3178 => [0, 0, nil, "\341\204\253", nil, nil, nil],
      0x3179 => [0, 0, nil, "\341\204\254", nil, nil, nil],
      0x317a => [0, 0, nil, "\341\204\255", nil, nil, nil],
      0x317b => [0, 0, nil, "\341\204\256", nil, nil, nil],
      0x317c => [0, 0, nil, "\341\204\257", nil, nil, nil],
      0x317d => [0, 0, nil, "\341\204\262", nil, nil, nil],
      0x317e => [0, 0, nil, "\341\204\266", nil, nil, nil],
      0x317f => [0, 0, nil, "\341\205\200", nil, nil, nil],
      0x3180 => [0, 0, nil, "\341\205\207", nil, nil, nil],
      0x3181 => [0, 0, nil, "\341\205\214", nil, nil, nil],
      0x3182 => [0, 0, nil, "\341\207\261", nil, nil, nil],
      0x3183 => [0, 0, nil, "\341\207\262", nil, nil, nil],
      0x3184 => [0, 0, nil, "\341\205\227", nil, nil, nil],
      0x3185 => [0, 0, nil, "\341\205\230", nil, nil, nil],
      0x3186 => [0, 0, nil, "\341\205\231", nil, nil, nil],
      0x3187 => [0, 0, nil, "\341\206\204", nil, nil, nil],
      0x3188 => [0, 0, nil, "\341\206\205", nil, nil, nil],
      0x3189 => [0, 0, nil, "\341\206\210", nil, nil, nil],
      0x318a => [0, 0, nil, "\341\206\221", nil, nil, nil],
      0x318b => [0, 0, nil, "\341\206\222", nil, nil, nil],
      0x318c => [0, 0, nil, "\341\206\224", nil, nil, nil],
      0x318d => [0, 0, nil, "\341\206\236", nil, nil, nil],
      0x318e => [0, 0, nil, "\341\206\241", nil, nil, nil],
      0x3192 => [0, 0, nil, "\344\270\200", nil, nil, nil],
      0x3193 => [0, 0, nil, "\344\272\214", nil, nil, nil],
      0x3194 => [0, 0, nil, "\344\270\211", nil, nil, nil],
      0x3195 => [0, 0, nil, "\345\233\233", nil, nil, nil],
      0x3196 => [0, 0, nil, "\344\270\212", nil, nil, nil],
      0x3197 => [0, 0, nil, "\344\270\255", nil, nil, nil],
      0x3198 => [0, 0, nil, "\344\270\213", nil, nil, nil],
      0x3199 => [0, 0, nil, "\347\224\262", nil, nil, nil],
      0x319a => [0, 0, nil, "\344\271\231", nil, nil, nil],
      0x319b => [0, 0, nil, "\344\270\231", nil, nil, nil],
      0x319c => [0, 0, nil, "\344\270\201", nil, nil, nil],
      0x319d => [0, 0, nil, "\345\244\251", nil, nil, nil],
      0x319e => [0, 0, nil, "\345\234\260", nil, nil, nil],
      0x319f => [0, 0, nil, "\344\272\272", nil, nil, nil],
      0x3200 => [0, 0, nil, "(\341\204\200)", nil, nil, nil],
      0x3201 => [0, 0, nil, "(\341\204\202)", nil, nil, nil],
      0x3202 => [0, 0, nil, "(\341\204\203)", nil, nil, nil],
      0x3203 => [0, 0, nil, "(\341\204\205)", nil, nil, nil],
      0x3204 => [0, 0, nil, "(\341\204\206)", nil, nil, nil],
      0x3205 => [0, 0, nil, "(\341\204\207)", nil, nil, nil],
      0x3206 => [0, 0, nil, "(\341\204\211)", nil, nil, nil],
      0x3207 => [0, 0, nil, "(\341\204\213)", nil, nil, nil],
      0x3208 => [0, 0, nil, "(\341\204\214)", nil, nil, nil],
      0x3209 => [0, 0, nil, "(\341\204\216)", nil, nil, nil],
      0x320a => [0, 0, nil, "(\341\204\217)", nil, nil, nil],
      0x320b => [0, 0, nil, "(\341\204\220)", nil, nil, nil],
      0x320c => [0, 0, nil, "(\341\204\221)", nil, nil, nil],
      0x320d => [0, 0, nil, "(\341\204\222)", nil, nil, nil],
      0x320e => [0, 0, nil, "(\341\204\200\341\205\241)", nil, nil, nil],
      0x320f => [0, 0, nil, "(\341\204\202\341\205\241)", nil, nil, nil],
      0x3210 => [0, 0, nil, "(\341\204\203\341\205\241)", nil, nil, nil],
      0x3211 => [0, 0, nil, "(\341\204\205\341\205\241)", nil, nil, nil],
      0x3212 => [0, 0, nil, "(\341\204\206\341\205\241)", nil, nil, nil],
      0x3213 => [0, 0, nil, "(\341\204\207\341\205\241)", nil, nil, nil],
      0x3214 => [0, 0, nil, "(\341\204\211\341\205\241)", nil, nil, nil],
      0x3215 => [0, 0, nil, "(\341\204\213\341\205\241)", nil, nil, nil],
      0x3216 => [0, 0, nil, "(\341\204\214\341\205\241)", nil, nil, nil],
      0x3217 => [0, 0, nil, "(\341\204\216\341\205\241)", nil, nil, nil],
      0x3218 => [0, 0, nil, "(\341\204\217\341\205\241)", nil, nil, nil],
      0x3219 => [0, 0, nil, "(\341\204\220\341\205\241)", nil, nil, nil],
      0x321a => [0, 0, nil, "(\341\204\221\341\205\241)", nil, nil, nil],
      0x321b => [0, 0, nil, "(\341\204\222\341\205\241)", nil, nil, nil],
      0x321c => [0, 0, nil, "(\341\204\214\341\205\256)", nil, nil, nil],
      0x3220 => [0, 0, nil, "(\344\270\200)", nil, nil, nil],
      0x3221 => [0, 0, nil, "(\344\272\214)", nil, nil, nil],
      0x3222 => [0, 0, nil, "(\344\270\211)", nil, nil, nil],
      0x3223 => [0, 0, nil, "(\345\233\233)", nil, nil, nil],
      0x3224 => [0, 0, nil, "(\344\272\224)", nil, nil, nil],
      0x3225 => [0, 0, nil, "(\345\205\255)", nil, nil, nil],
      0x3226 => [0, 0, nil, "(\344\270\203)", nil, nil, nil],
      0x3227 => [0, 0, nil, "(\345\205\253)", nil, nil, nil],
      0x3228 => [0, 0, nil, "(\344\271\235)", nil, nil, nil],
      0x3229 => [0, 0, nil, "(\345\215\201)", nil, nil, nil],
      0x322a => [0, 0, nil, "(\346\234\210)", nil, nil, nil],
      0x322b => [0, 0, nil, "(\347\201\253)", nil, nil, nil],
      0x322c => [0, 0, nil, "(\346\260\264)", nil, nil, nil],
      0x322d => [0, 0, nil, "(\346\234\250)", nil, nil, nil],
      0x322e => [0, 0, nil, "(\351\207\221)", nil, nil, nil],
      0x322f => [0, 0, nil, "(\345\234\237)", nil, nil, nil],
      0x3230 => [0, 0, nil, "(\346\227\245)", nil, nil, nil],
      0x3231 => [0, 0, nil, "(\346\240\252)", nil, nil, nil],
      0x3232 => [0, 0, nil, "(\346\234\211)", nil, nil, nil],
      0x3233 => [0, 0, nil, "(\347\244\276)", nil, nil, nil],
      0x3234 => [0, 0, nil, "(\345\220\215)", nil, nil, nil],
      0x3235 => [0, 0, nil, "(\347\211\271)", nil, nil, nil],
      0x3236 => [0, 0, nil, "(\350\262\241)", nil, nil, nil],
      0x3237 => [0, 0, nil, "(\347\245\235)", nil, nil, nil],
      0x3238 => [0, 0, nil, "(\345\212\264)", nil, nil, nil],
      0x3239 => [0, 0, nil, "(\344\273\243)", nil, nil, nil],
      0x323a => [0, 0, nil, "(\345\221\274)", nil, nil, nil],
      0x323b => [0, 0, nil, "(\345\255\246)", nil, nil, nil],
      0x323c => [0, 0, nil, "(\347\233\243)", nil, nil, nil],
      0x323d => [0, 0, nil, "(\344\274\201)", nil, nil, nil],
      0x323e => [0, 0, nil, "(\350\263\207)", nil, nil, nil],
      0x323f => [0, 0, nil, "(\345\215\224)", nil, nil, nil],
      0x3240 => [0, 0, nil, "(\347\245\255)", nil, nil, nil],
      0x3241 => [0, 0, nil, "(\344\274\221)", nil, nil, nil],
      0x3242 => [0, 0, nil, "(\350\207\252)", nil, nil, nil],
      0x3243 => [0, 0, nil, "(\350\207\263)", nil, nil, nil],
      0x3260 => [0, 0, nil, "\341\204\200", nil, nil, nil],
      0x3261 => [0, 0, nil, "\341\204\202", nil, nil, nil],
      0x3262 => [0, 0, nil, "\341\204\203", nil, nil, nil],
      0x3263 => [0, 0, nil, "\341\204\205", nil, nil, nil],
      0x3264 => [0, 0, nil, "\341\204\206", nil, nil, nil],
      0x3265 => [0, 0, nil, "\341\204\207", nil, nil, nil],
      0x3266 => [0, 0, nil, "\341\204\211", nil, nil, nil],
      0x3267 => [0, 0, nil, "\341\204\213", nil, nil, nil],
      0x3268 => [0, 0, nil, "\341\204\214", nil, nil, nil],
      0x3269 => [0, 0, nil, "\341\204\216", nil, nil, nil],
      0x326a => [0, 0, nil, "\341\204\217", nil, nil, nil],
      0x326b => [0, 0, nil, "\341\204\220", nil, nil, nil],
      0x326c => [0, 0, nil, "\341\204\221", nil, nil, nil],
      0x326d => [0, 0, nil, "\341\204\222", nil, nil, nil],
      0x326e => [0, 0, nil, "\341\204\200\341\205\241", nil, nil, nil],
      0x326f => [0, 0, nil, "\341\204\202\341\205\241", nil, nil, nil],
      0x3270 => [0, 0, nil, "\341\204\203\341\205\241", nil, nil, nil],
      0x3271 => [0, 0, nil, "\341\204\205\341\205\241", nil, nil, nil],
      0x3272 => [0, 0, nil, "\341\204\206\341\205\241", nil, nil, nil],
      0x3273 => [0, 0, nil, "\341\204\207\341\205\241", nil, nil, nil],
      0x3274 => [0, 0, nil, "\341\204\211\341\205\241", nil, nil, nil],
      0x3275 => [0, 0, nil, "\341\204\213\341\205\241", nil, nil, nil],
      0x3276 => [0, 0, nil, "\341\204\214\341\205\241", nil, nil, nil],
      0x3277 => [0, 0, nil, "\341\204\216\341\205\241", nil, nil, nil],
      0x3278 => [0, 0, nil, "\341\204\217\341\205\241", nil, nil, nil],
      0x3279 => [0, 0, nil, "\341\204\220\341\205\241", nil, nil, nil],
      0x327a => [0, 0, nil, "\341\204\221\341\205\241", nil, nil, nil],
      0x327b => [0, 0, nil, "\341\204\222\341\205\241", nil, nil, nil],
      0x3280 => [0, 0, nil, "\344\270\200", nil, nil, nil],
      0x3281 => [0, 0, nil, "\344\272\214", nil, nil, nil],
      0x3282 => [0, 0, nil, "\344\270\211", nil, nil, nil],
      0x3283 => [0, 0, nil, "\345\233\233", nil, nil, nil],
      0x3284 => [0, 0, nil, "\344\272\224", nil, nil, nil],
      0x3285 => [0, 0, nil, "\345\205\255", nil, nil, nil],
      0x3286 => [0, 0, nil, "\344\270\203", nil, nil, nil],
      0x3287 => [0, 0, nil, "\345\205\253", nil, nil, nil],
      0x3288 => [0, 0, nil, "\344\271\235", nil, nil, nil],
      0x3289 => [0, 0, nil, "\345\215\201", nil, nil, nil],
      0x328a => [0, 0, nil, "\346\234\210", nil, nil, nil],
      0x328b => [0, 0, nil, "\347\201\253", nil, nil, nil],
      0x328c => [0, 0, nil, "\346\260\264", nil, nil, nil],
      0x328d => [0, 0, nil, "\346\234\250", nil, nil, nil],
      0x328e => [0, 0, nil, "\351\207\221", nil, nil, nil],
      0x328f => [0, 0, nil, "\345\234\237", nil, nil, nil],
      0x3290 => [0, 0, nil, "\346\227\245", nil, nil, nil],
      0x3291 => [0, 0, nil, "\346\240\252", nil, nil, nil],
      0x3292 => [0, 0, nil, "\346\234\211", nil, nil, nil],
      0x3293 => [0, 0, nil, "\347\244\276", nil, nil, nil],
      0x3294 => [0, 0, nil, "\345\220\215", nil, nil, nil],
      0x3295 => [0, 0, nil, "\347\211\271", nil, nil, nil],
      0x3296 => [0, 0, nil, "\350\262\241", nil, nil, nil],
      0x3297 => [0, 0, nil, "\347\245\235", nil, nil, nil],
      0x3298 => [0, 0, nil, "\345\212\264", nil, nil, nil],
      0x3299 => [0, 0, nil, "\347\247\230", nil, nil, nil],
      0x329a => [0, 0, nil, "\347\224\267", nil, nil, nil],
      0x329b => [0, 0, nil, "\345\245\263", nil, nil, nil],
      0x329c => [0, 0, nil, "\351\201\251", nil, nil, nil],
      0x329d => [0, 0, nil, "\345\204\252", nil, nil, nil],
      0x329e => [0, 0, nil, "\345\215\260", nil, nil, nil],
      0x329f => [0, 0, nil, "\346\263\250", nil, nil, nil],
      0x32a0 => [0, 0, nil, "\351\240\205", nil, nil, nil],
      0x32a1 => [0, 0, nil, "\344\274\221", nil, nil, nil],
      0x32a2 => [0, 0, nil, "\345\206\231", nil, nil, nil],
      0x32a3 => [0, 0, nil, "\346\255\243", nil, nil, nil],
      0x32a4 => [0, 0, nil, "\344\270\212", nil, nil, nil],
      0x32a5 => [0, 0, nil, "\344\270\255", nil, nil, nil],
      0x32a6 => [0, 0, nil, "\344\270\213", nil, nil, nil],
      0x32a7 => [0, 0, nil, "\345\267\246", nil, nil, nil],
      0x32a8 => [0, 0, nil, "\345\217\263", nil, nil, nil],
      0x32a9 => [0, 0, nil, "\345\214\273", nil, nil, nil],
      0x32aa => [0, 0, nil, "\345\256\227", nil, nil, nil],
      0x32ab => [0, 0, nil, "\345\255\246", nil, nil, nil],
      0x32ac => [0, 0, nil, "\347\233\243", nil, nil, nil],
      0x32ad => [0, 0, nil, "\344\274\201", nil, nil, nil],
      0x32ae => [0, 0, nil, "\350\263\207", nil, nil, nil],
      0x32af => [0, 0, nil, "\345\215\224", nil, nil, nil],
      0x32b0 => [0, 0, nil, "\345\244\234", nil, nil, nil],
      0x32c0 => [0, 0, nil, "1\346\234\210", nil, nil, nil],
      0x32c1 => [0, 0, nil, "2\346\234\210", nil, nil, nil],
      0x32c2 => [0, 0, nil, "3\346\234\210", nil, nil, nil],
      0x32c3 => [0, 0, nil, "4\346\234\210", nil, nil, nil],
      0x32c4 => [0, 0, nil, "5\346\234\210", nil, nil, nil],
      0x32c5 => [0, 0, nil, "6\346\234\210", nil, nil, nil],
      0x32c6 => [0, 0, nil, "7\346\234\210", nil, nil, nil],
      0x32c7 => [0, 0, nil, "8\346\234\210", nil, nil, nil],
      0x32c8 => [0, 0, nil, "9\346\234\210", nil, nil, nil],
      0x32c9 => [0, 0, nil, "10\346\234\210", nil, nil, nil],
      0x32ca => [0, 0, nil, "11\346\234\210", nil, nil, nil],
      0x32cb => [0, 0, nil, "12\346\234\210", nil, nil, nil],
      0x32d0 => [0, 0, nil, "\343\202\242", nil, nil, nil],
      0x32d1 => [0, 0, nil, "\343\202\244", nil, nil, nil],
      0x32d2 => [0, 0, nil, "\343\202\246", nil, nil, nil],
      0x32d3 => [0, 0, nil, "\343\202\250", nil, nil, nil],
      0x32d4 => [0, 0, nil, "\343\202\252", nil, nil, nil],
      0x32d5 => [0, 0, nil, "\343\202\253", nil, nil, nil],
      0x32d6 => [0, 0, nil, "\343\202\255", nil, nil, nil],
      0x32d7 => [0, 0, nil, "\343\202\257", nil, nil, nil],
      0x32d8 => [0, 0, nil, "\343\202\261", nil, nil, nil],
      0x32d9 => [0, 0, nil, "\343\202\263", nil, nil, nil],
      0x32da => [0, 0, nil, "\343\202\265", nil, nil, nil],
      0x32db => [0, 0, nil, "\343\202\267", nil, nil, nil],
      0x32dc => [0, 0, nil, "\343\202\271", nil, nil, nil],
      0x32dd => [0, 0, nil, "\343\202\273", nil, nil, nil],
      0x32de => [0, 0, nil, "\343\202\275", nil, nil, nil],
      0x32df => [0, 0, nil, "\343\202\277", nil, nil, nil],
      0x32e0 => [0, 0, nil, "\343\203\201", nil, nil, nil],
      0x32e1 => [0, 0, nil, "\343\203\204", nil, nil, nil],
      0x32e2 => [0, 0, nil, "\343\203\206", nil, nil, nil],
      0x32e3 => [0, 0, nil, "\343\203\210", nil, nil, nil],
      0x32e4 => [0, 0, nil, "\343\203\212", nil, nil, nil],
      0x32e5 => [0, 0, nil, "\343\203\213", nil, nil, nil],
      0x32e6 => [0, 0, nil, "\343\203\214", nil, nil, nil],
      0x32e7 => [0, 0, nil, "\343\203\215", nil, nil, nil],
      0x32e8 => [0, 0, nil, "\343\203\216", nil, nil, nil],
      0x32e9 => [0, 0, nil, "\343\203\217", nil, nil, nil],
      0x32ea => [0, 0, nil, "\343\203\222", nil, nil, nil],
      0x32eb => [0, 0, nil, "\343\203\225", nil, nil, nil],
      0x32ec => [0, 0, nil, "\343\203\230", nil, nil, nil],
      0x32ed => [0, 0, nil, "\343\203\233", nil, nil, nil],
      0x32ee => [0, 0, nil, "\343\203\236", nil, nil, nil],
      0x32ef => [0, 0, nil, "\343\203\237", nil, nil, nil],
      0x32f0 => [0, 0, nil, "\343\203\240", nil, nil, nil],
      0x32f1 => [0, 0, nil, "\343\203\241", nil, nil, nil],
      0x32f2 => [0, 0, nil, "\343\203\242", nil, nil, nil],
      0x32f3 => [0, 0, nil, "\343\203\244", nil, nil, nil],
      0x32f4 => [0, 0, nil, "\343\203\246", nil, nil, nil],
      0x32f5 => [0, 0, nil, "\343\203\250", nil, nil, nil],
      0x32f6 => [0, 0, nil, "\343\203\251", nil, nil, nil],
      0x32f7 => [0, 0, nil, "\343\203\252", nil, nil, nil],
      0x32f8 => [0, 0, nil, "\343\203\253", nil, nil, nil],
      0x32f9 => [0, 0, nil, "\343\203\254", nil, nil, nil],
      0x32fa => [0, 0, nil, "\343\203\255", nil, nil, nil],
      0x32fb => [0, 0, nil, "\343\203\257", nil, nil, nil],
      0x32fc => [0, 0, nil, "\343\203\260", nil, nil, nil],
      0x32fd => [0, 0, nil, "\343\203\261", nil, nil, nil],
      0x32fe => [0, 0, nil, "\343\203\262", nil, nil, nil],
      0x3300 => [0, 0, nil, "\343\202\242\343\203\221\343\203\274\343\203\210", nil, nil, nil],
      0x3301 => [0, 0, nil, "\343\202\242\343\203\253\343\203\225\343\202\241", nil, nil, nil],
      0x3302 => [0, 0, nil, "\343\202\242\343\203\263\343\203\232\343\202\242", nil, nil, nil],
      0x3303 => [0, 0, nil, "\343\202\242\343\203\274\343\203\253", nil, nil, nil],
      0x3304 => [0, 0, nil, "\343\202\244\343\203\213\343\203\263\343\202\260", nil, nil, nil],
      0x3305 => [0, 0, nil, "\343\202\244\343\203\263\343\203\201", nil, nil, nil],
      0x3306 => [0, 0, nil, "\343\202\246\343\202\251\343\203\263", nil, nil, nil],
      0x3307 => [0, 0, nil, "\343\202\250\343\202\271\343\202\257\343\203\274\343\203\211", nil, nil, nil],
      0x3308 => [0, 0, nil, "\343\202\250\343\203\274\343\202\253\343\203\274", nil, nil, nil],
      0x3309 => [0, 0, nil, "\343\202\252\343\203\263\343\202\271", nil, nil, nil],
      0x330a => [0, 0, nil, "\343\202\252\343\203\274\343\203\240", nil, nil, nil],
      0x330b => [0, 0, nil, "\343\202\253\343\202\244\343\203\252", nil, nil, nil],
      0x330c => [0, 0, nil, "\343\202\253\343\203\251\343\203\203\343\203\210", nil, nil, nil],
      0x330d => [0, 0, nil, "\343\202\253\343\203\255\343\203\252\343\203\274", nil, nil, nil],
      0x330e => [0, 0, nil, "\343\202\254\343\203\255\343\203\263", nil, nil, nil],
      0x330f => [0, 0, nil, "\343\202\254\343\203\263\343\203\236", nil, nil, nil],
      0x3310 => [0, 0, nil, "\343\202\256\343\202\254", nil, nil, nil],
      0x3311 => [0, 0, nil, "\343\202\256\343\203\213\343\203\274", nil, nil, nil],
      0x3312 => [0, 0, nil, "\343\202\255\343\203\245\343\203\252\343\203\274", nil, nil, nil],
      0x3313 => [0, 0, nil, "\343\202\256\343\203\253\343\203\200\343\203\274", nil, nil, nil],
      0x3314 => [0, 0, nil, "\343\202\255\343\203\255", nil, nil, nil],
      0x3315 => [0, 0, nil, "\343\202\255\343\203\255\343\202\260\343\203\251\343\203\240", nil, nil, nil],
      0x3316 => [0, 0, nil, "\343\202\255\343\203\255\343\203\241\343\203\274\343\203\210\343\203\253", nil, nil, nil],
      0x3317 => [0, 0, nil, "\343\202\255\343\203\255\343\203\257\343\203\203\343\203\210", nil, nil, nil],
      0x3318 => [0, 0, nil, "\343\202\260\343\203\251\343\203\240", nil, nil, nil],
      0x3319 => [0, 0, nil, "\343\202\260\343\203\251\343\203\240\343\203\210\343\203\263", nil, nil, nil],
      0x331a => [0, 0, nil, "\343\202\257\343\203\253\343\202\274\343\202\244\343\203\255", nil, nil, nil],
      0x331b => [0, 0, nil, "\343\202\257\343\203\255\343\203\274\343\203\215", nil, nil, nil],
      0x331c => [0, 0, nil, "\343\202\261\343\203\274\343\202\271", nil, nil, nil],
      0x331d => [0, 0, nil, "\343\202\263\343\203\253\343\203\212", nil, nil, nil],
      0x331e => [0, 0, nil, "\343\202\263\343\203\274\343\203\235", nil, nil, nil],
      0x331f => [0, 0, nil, "\343\202\265\343\202\244\343\202\257\343\203\253", nil, nil, nil],
      0x3320 => [0, 0, nil, "\343\202\265\343\203\263\343\203\201\343\203\274\343\203\240", nil, nil, nil],
      0x3321 => [0, 0, nil, "\343\202\267\343\203\252\343\203\263\343\202\260", nil, nil, nil],
      0x3322 => [0, 0, nil, "\343\202\273\343\203\263\343\203\201", nil, nil, nil],
      0x3323 => [0, 0, nil, "\343\202\273\343\203\263\343\203\210", nil, nil, nil],
      0x3324 => [0, 0, nil, "\343\203\200\343\203\274\343\202\271", nil, nil, nil],
      0x3325 => [0, 0, nil, "\343\203\207\343\202\267", nil, nil, nil],
      0x3326 => [0, 0, nil, "\343\203\211\343\203\253", nil, nil, nil],
      0x3327 => [0, 0, nil, "\343\203\210\343\203\263", nil, nil, nil],
      0x3328 => [0, 0, nil, "\343\203\212\343\203\216", nil, nil, nil],
      0x3329 => [0, 0, nil, "\343\203\216\343\203\203\343\203\210", nil, nil, nil],
      0x332a => [0, 0, nil, "\343\203\217\343\202\244\343\203\204", nil, nil, nil],
      0x332b => [0, 0, nil, "\343\203\221\343\203\274\343\202\273\343\203\263\343\203\210", nil, nil, nil],
      0x332c => [0, 0, nil, "\343\203\221\343\203\274\343\203\204", nil, nil, nil],
      0x332d => [0, 0, nil, "\343\203\220\343\203\274\343\203\254\343\203\253", nil, nil, nil],
      0x332e => [0, 0, nil, "\343\203\224\343\202\242\343\202\271\343\203\210\343\203\253", nil, nil, nil],
      0x332f => [0, 0, nil, "\343\203\224\343\202\257\343\203\253", nil, nil, nil],
      0x3330 => [0, 0, nil, "\343\203\224\343\202\263", nil, nil, nil],
      0x3331 => [0, 0, nil, "\343\203\223\343\203\253", nil, nil, nil],
      0x3332 => [0, 0, nil, "\343\203\225\343\202\241\343\203\251\343\203\203\343\203\211", nil, nil, nil],
      0x3333 => [0, 0, nil, "\343\203\225\343\202\243\343\203\274\343\203\210", nil, nil, nil],
      0x3334 => [0, 0, nil, "\343\203\226\343\203\203\343\202\267\343\202\247\343\203\253", nil, nil, nil],
      0x3335 => [0, 0, nil, "\343\203\225\343\203\251\343\203\263", nil, nil, nil],
      0x3336 => [0, 0, nil, "\343\203\230\343\202\257\343\202\277\343\203\274\343\203\253", nil, nil, nil],
      0x3337 => [0, 0, nil, "\343\203\232\343\202\275", nil, nil, nil],
      0x3338 => [0, 0, nil, "\343\203\232\343\203\213\343\203\222", nil, nil, nil],
      0x3339 => [0, 0, nil, "\343\203\230\343\203\253\343\203\204", nil, nil, nil],
      0x333a => [0, 0, nil, "\343\203\232\343\203\263\343\202\271", nil, nil, nil],
      0x333b => [0, 0, nil, "\343\203\232\343\203\274\343\202\270", nil, nil, nil],
      0x333c => [0, 0, nil, "\343\203\231\343\203\274\343\202\277", nil, nil, nil],
      0x333d => [0, 0, nil, "\343\203\235\343\202\244\343\203\263\343\203\210", nil, nil, nil],
      0x333e => [0, 0, nil, "\343\203\234\343\203\253\343\203\210", nil, nil, nil],
      0x333f => [0, 0, nil, "\343\203\233\343\203\263", nil, nil, nil],
      0x3340 => [0, 0, nil, "\343\203\235\343\203\263\343\203\211", nil, nil, nil],
      0x3341 => [0, 0, nil, "\343\203\233\343\203\274\343\203\253", nil, nil, nil],
      0x3342 => [0, 0, nil, "\343\203\233\343\203\274\343\203\263", nil, nil, nil],
      0x3343 => [0, 0, nil, "\343\203\236\343\202\244\343\202\257\343\203\255", nil, nil, nil],
      0x3344 => [0, 0, nil, "\343\203\236\343\202\244\343\203\253", nil, nil, nil],
      0x3345 => [0, 0, nil, "\343\203\236\343\203\203\343\203\217", nil, nil, nil],
      0x3346 => [0, 0, nil, "\343\203\236\343\203\253\343\202\257", nil, nil, nil],
      0x3347 => [0, 0, nil, "\343\203\236\343\203\263\343\202\267\343\203\247\343\203\263", nil, nil, nil],
      0x3348 => [0, 0, nil, "\343\203\237\343\202\257\343\203\255\343\203\263", nil, nil, nil],
      0x3349 => [0, 0, nil, "\343\203\237\343\203\252", nil, nil, nil],
      0x334a => [0, 0, nil, "\343\203\237\343\203\252\343\203\220\343\203\274\343\203\253", nil, nil, nil],
      0x334b => [0, 0, nil, "\343\203\241\343\202\254", nil, nil, nil],
      0x334c => [0, 0, nil, "\343\203\241\343\202\254\343\203\210\343\203\263", nil, nil, nil],
      0x334d => [0, 0, nil, "\343\203\241\343\203\274\343\203\210\343\203\253", nil, nil, nil],
      0x334e => [0, 0, nil, "\343\203\244\343\203\274\343\203\211", nil, nil, nil],
      0x334f => [0, 0, nil, "\343\203\244\343\203\274\343\203\253", nil, nil, nil],
      0x3350 => [0, 0, nil, "\343\203\246\343\202\242\343\203\263", nil, nil, nil],
      0x3351 => [0, 0, nil, "\343\203\252\343\203\203\343\203\210\343\203\253", nil, nil, nil],
      0x3352 => [0, 0, nil, "\343\203\252\343\203\251", nil, nil, nil],
      0x3353 => [0, 0, nil, "\343\203\253\343\203\224\343\203\274", nil, nil, nil],
      0x3354 => [0, 0, nil, "\343\203\253\343\203\274\343\203\226\343\203\253", nil, nil, nil],
      0x3355 => [0, 0, nil, "\343\203\254\343\203\240", nil, nil, nil],
      0x3356 => [0, 0, nil, "\343\203\254\343\203\263\343\203\210\343\202\262\343\203\263", nil, nil, nil],
      0x3357 => [0, 0, nil, "\343\203\257\343\203\203\343\203\210", nil, nil, nil],
      0x3358 => [0, 0, nil, "0\347\202\271", nil, nil, nil],
      0x3359 => [0, 0, nil, "1\347\202\271", nil, nil, nil],
      0x335a => [0, 0, nil, "2\347\202\271", nil, nil, nil],
      0x335b => [0, 0, nil, "3\347\202\271", nil, nil, nil],
      0x335c => [0, 0, nil, "4\347\202\271", nil, nil, nil],
      0x335d => [0, 0, nil, "5\347\202\271", nil, nil, nil],
      0x335e => [0, 0, nil, "6\347\202\271", nil, nil, nil],
      0x335f => [0, 0, nil, "7\347\202\271", nil, nil, nil],
      0x3360 => [0, 0, nil, "8\347\202\271", nil, nil, nil],
      0x3361 => [0, 0, nil, "9\347\202\271", nil, nil, nil],
      0x3362 => [0, 0, nil, "10\347\202\271", nil, nil, nil],
      0x3363 => [0, 0, nil, "11\347\202\271", nil, nil, nil],
      0x3364 => [0, 0, nil, "12\347\202\271", nil, nil, nil],
      0x3365 => [0, 0, nil, "13\347\202\271", nil, nil, nil],
      0x3366 => [0, 0, nil, "14\347\202\271", nil, nil, nil],
      0x3367 => [0, 0, nil, "15\347\202\271", nil, nil, nil],
      0x3368 => [0, 0, nil, "16\347\202\271", nil, nil, nil],
      0x3369 => [0, 0, nil, "17\347\202\271", nil, nil, nil],
      0x336a => [0, 0, nil, "18\347\202\271", nil, nil, nil],
      0x336b => [0, 0, nil, "19\347\202\271", nil, nil, nil],
      0x336c => [0, 0, nil, "20\347\202\271", nil, nil, nil],
      0x336d => [0, 0, nil, "21\347\202\271", nil, nil, nil],
      0x336e => [0, 0, nil, "22\347\202\271", nil, nil, nil],
      0x336f => [0, 0, nil, "23\347\202\271", nil, nil, nil],
      0x3370 => [0, 0, nil, "24\347\202\271", nil, nil, nil],
      0x3371 => [0, 0, nil, "hPa", nil, nil, nil],
      0x3372 => [0, 0, nil, "da", nil, nil, nil],
      0x3373 => [0, 0, nil, "AU", nil, nil, nil],
      0x3374 => [0, 0, nil, "bar", nil, nil, nil],
      0x3375 => [0, 0, nil, "oV", nil, nil, nil],
      0x3376 => [0, 0, nil, "pc", nil, nil, nil],
      0x337b => [0, 0, nil, "\345\271\263\346\210\220", nil, nil, nil],
      0x337c => [0, 0, nil, "\346\230\255\345\222\214", nil, nil, nil],
      0x337d => [0, 0, nil, "\345\244\247\346\255\243", nil, nil, nil],
      0x337e => [0, 0, nil, "\346\230\216\346\262\273", nil, nil, nil],
      0x337f => [0, 0, nil, "\346\240\252\345\274\217\344\274\232\347\244\276", nil, nil, nil],
      0x3380 => [0, 0, nil, "pA", nil, nil, nil],
      0x3381 => [0, 0, nil, "nA", nil, nil, nil],
      0x3382 => [0, 0, nil, "\316\274A", nil, nil, nil],
      0x3383 => [0, 0, nil, "mA", nil, nil, nil],
      0x3384 => [0, 0, nil, "kA", nil, nil, nil],
      0x3385 => [0, 0, nil, "KB", nil, nil, nil],
      0x3386 => [0, 0, nil, "MB", nil, nil, nil],
      0x3387 => [0, 0, nil, "GB", nil, nil, nil],
      0x3388 => [0, 0, nil, "cal", nil, nil, nil],
      0x3389 => [0, 0, nil, "kcal", nil, nil, nil],
      0x338a => [0, 0, nil, "pF", nil, nil, nil],
      0x338b => [0, 0, nil, "nF", nil, nil, nil],
      0x338c => [0, 0, nil, "\316\274F", nil, nil, nil],
      0x338d => [0, 0, nil, "\316\274g", nil, nil, nil],
      0x338e => [0, 0, nil, "mg", nil, nil, nil],
      0x338f => [0, 0, nil, "kg", nil, nil, nil],
      0x3390 => [0, 0, nil, "Hz", nil, nil, nil],
      0x3391 => [0, 0, nil, "kHz", nil, nil, nil],
      0x3392 => [0, 0, nil, "MHz", nil, nil, nil],
      0x3393 => [0, 0, nil, "GHz", nil, nil, nil],
      0x3394 => [0, 0, nil, "THz", nil, nil, nil],
      0x3395 => [0, 0, nil, "\316\274\342\204\223", nil, nil, nil],
      0x3396 => [0, 0, nil, "m\342\204\223", nil, nil, nil],
      0x3397 => [0, 0, nil, "d\342\204\223", nil, nil, nil],
      0x3398 => [0, 0, nil, "k\342\204\223", nil, nil, nil],
      0x3399 => [0, 0, nil, "fm", nil, nil, nil],
      0x339a => [0, 0, nil, "nm", nil, nil, nil],
      0x339b => [0, 0, nil, "\316\274m", nil, nil, nil],
      0x339c => [0, 0, nil, "mm", nil, nil, nil],
      0x339d => [0, 0, nil, "cm", nil, nil, nil],
      0x339e => [0, 0, nil, "km", nil, nil, nil],
      0x339f => [0, 0, nil, "mm\302\262", nil, nil, nil],
      0x33a0 => [0, 0, nil, "cm\302\262", nil, nil, nil],
      0x33a1 => [0, 0, nil, "m\302\262", nil, nil, nil],
      0x33a2 => [0, 0, nil, "km\302\262", nil, nil, nil],
      0x33a3 => [0, 0, nil, "mm\302\263", nil, nil, nil],
      0x33a4 => [0, 0, nil, "cm\302\263", nil, nil, nil],
      0x33a5 => [0, 0, nil, "m\302\263", nil, nil, nil],
      0x33a6 => [0, 0, nil, "km\302\263", nil, nil, nil],
      0x33a7 => [0, 0, nil, "m\342\210\225s", nil, nil, nil],
      0x33a8 => [0, 0, nil, "m\342\210\225s\302\262", nil, nil, nil],
      0x33a9 => [0, 0, nil, "Pa", nil, nil, nil],
      0x33aa => [0, 0, nil, "kPa", nil, nil, nil],
      0x33ab => [0, 0, nil, "MPa", nil, nil, nil],
      0x33ac => [0, 0, nil, "GPa", nil, nil, nil],
      0x33ad => [0, 0, nil, "rad", nil, nil, nil],
      0x33ae => [0, 0, nil, "rad\342\210\225s", nil, nil, nil],
      0x33af => [0, 0, nil, "rad\342\210\225s\302\262", nil, nil, nil],
      0x33b0 => [0, 0, nil, "ps", nil, nil, nil],
      0x33b1 => [0, 0, nil, "ns", nil, nil, nil],
      0x33b2 => [0, 0, nil, "\316\274s", nil, nil, nil],
      0x33b3 => [0, 0, nil, "ms", nil, nil, nil],
      0x33b4 => [0, 0, nil, "pV", nil, nil, nil],
      0x33b5 => [0, 0, nil, "nV", nil, nil, nil],
      0x33b6 => [0, 0, nil, "\316\274V", nil, nil, nil],
      0x33b7 => [0, 0, nil, "mV", nil, nil, nil],
      0x33b8 => [0, 0, nil, "kV", nil, nil, nil],
      0x33b9 => [0, 0, nil, "MV", nil, nil, nil],
      0x33ba => [0, 0, nil, "pW", nil, nil, nil],
      0x33bb => [0, 0, nil, "nW", nil, nil, nil],
      0x33bc => [0, 0, nil, "\316\274W", nil, nil, nil],
      0x33bd => [0, 0, nil, "mW", nil, nil, nil],
      0x33be => [0, 0, nil, "kW", nil, nil, nil],
      0x33bf => [0, 0, nil, "MW", nil, nil, nil],
      0x33c0 => [0, 0, nil, "k\316\251", nil, nil, nil],
      0x33c1 => [0, 0, nil, "M\316\251", nil, nil, nil],
      0x33c2 => [0, 0, nil, "a.m.", nil, nil, nil],
      0x33c3 => [0, 0, nil, "Bq", nil, nil, nil],
      0x33c4 => [0, 0, nil, "cc", nil, nil, nil],
      0x33c5 => [0, 0, nil, "cd", nil, nil, nil],
      0x33c6 => [0, 0, nil, "C\342\210\225kg", nil, nil, nil],
      0x33c7 => [0, 0, nil, "Co.", nil, nil, nil],
      0x33c8 => [0, 0, nil, "dB", nil, nil, nil],
      0x33c9 => [0, 0, nil, "Gy", nil, nil, nil],
      0x33ca => [0, 0, nil, "ha", nil, nil, nil],
      0x33cb => [0, 0, nil, "HP", nil, nil, nil],
      0x33cc => [0, 0, nil, "in", nil, nil, nil],
      0x33cd => [0, 0, nil, "KK", nil, nil, nil],
      0x33ce => [0, 0, nil, "KM", nil, nil, nil],
      0x33cf => [0, 0, nil, "kt", nil, nil, nil],
      0x33d0 => [0, 0, nil, "lm", nil, nil, nil],
      0x33d1 => [0, 0, nil, "ln", nil, nil, nil],
      0x33d2 => [0, 0, nil, "log", nil, nil, nil],
      0x33d3 => [0, 0, nil, "lx", nil, nil, nil],
      0x33d4 => [0, 0, nil, "mb", nil, nil, nil],
      0x33d5 => [0, 0, nil, "mil", nil, nil, nil],
      0x33d6 => [0, 0, nil, "mol", nil, nil, nil],
      0x33d7 => [0, 0, nil, "PH", nil, nil, nil],
      0x33d8 => [0, 0, nil, "p.m.", nil, nil, nil],
      0x33d9 => [0, 0, nil, "PPM", nil, nil, nil],
      0x33da => [0, 0, nil, "PR", nil, nil, nil],
      0x33db => [0, 0, nil, "sr", nil, nil, nil],
      0x33dc => [0, 0, nil, "Sv", nil, nil, nil],
      0x33dd => [0, 0, nil, "Wb", nil, nil, nil],
      0x33e0 => [0, 0, nil, "1\346\227\245", nil, nil, nil],
      0x33e1 => [0, 0, nil, "2\346\227\245", nil, nil, nil],
      0x33e2 => [0, 0, nil, "3\346\227\245", nil, nil, nil],
      0x33e3 => [0, 0, nil, "4\346\227\245", nil, nil, nil],
      0x33e4 => [0, 0, nil, "5\346\227\245", nil, nil, nil],
      0x33e5 => [0, 0, nil, "6\346\227\245", nil, nil, nil],
      0x33e6 => [0, 0, nil, "7\346\227\245", nil, nil, nil],
      0x33e7 => [0, 0, nil, "8\346\227\245", nil, nil, nil],
      0x33e8 => [0, 0, nil, "9\346\227\245", nil, nil, nil],
      0x33e9 => [0, 0, nil, "10\346\227\245", nil, nil, nil],
      0x33ea => [0, 0, nil, "11\346\227\245", nil, nil, nil],
      0x33eb => [0, 0, nil, "12\346\227\245", nil, nil, nil],
      0x33ec => [0, 0, nil, "13\346\227\245", nil, nil, nil],
      0x33ed => [0, 0, nil, "14\346\227\245", nil, nil, nil],
      0x33ee => [0, 0, nil, "15\346\227\245", nil, nil, nil],
      0x33ef => [0, 0, nil, "16\346\227\245", nil, nil, nil],
      0x33f0 => [0, 0, nil, "17\346\227\245", nil, nil, nil],
      0x33f1 => [0, 0, nil, "18\346\227\245", nil, nil, nil],
      0x33f2 => [0, 0, nil, "19\346\227\245", nil, nil, nil],
      0x33f3 => [0, 0, nil, "20\346\227\245", nil, nil, nil],
      0x33f4 => [0, 0, nil, "21\346\227\245", nil, nil, nil],
      0x33f5 => [0, 0, nil, "22\346\227\245", nil, nil, nil],
      0x33f6 => [0, 0, nil, "23\346\227\245", nil, nil, nil],
      0x33f7 => [0, 0, nil, "24\346\227\245", nil, nil, nil],
      0x33f8 => [0, 0, nil, "25\346\227\245", nil, nil, nil],
      0x33f9 => [0, 0, nil, "26\346\227\245", nil, nil, nil],
      0x33fa => [0, 0, nil, "27\346\227\245", nil, nil, nil],
      0x33fb => [0, 0, nil, "28\346\227\245", nil, nil, nil],
      0x33fc => [0, 0, nil, "29\346\227\245", nil, nil, nil],
      0x33fd => [0, 0, nil, "30\346\227\245", nil, nil, nil],
      0x33fe => [0, 0, nil, "31\346\227\245", nil, nil, nil],
      0xf900 => [0, 2, "\350\261\210", "\350\261\210", nil, nil, nil],
      0xf901 => [0, 2, "\346\233\264", "\346\233\264", nil, nil, nil],
      0xf902 => [0, 2, "\350\273\212", "\350\273\212", nil, nil, nil],
      0xf903 => [0, 2, "\350\263\210", "\350\263\210", nil, nil, nil],
      0xf904 => [0, 2, "\346\273\221", "\346\273\221", nil, nil, nil],
      0xf905 => [0, 2, "\344\270\262", "\344\270\262", nil, nil, nil],
      0xf906 => [0, 2, "\345\217\245", "\345\217\245", nil, nil, nil],
      0xf907 => [0, 2, "\351\276\234", "\351\276\234", nil, nil, nil],
      0xf908 => [0, 2, "\351\276\234", "\351\276\234", nil, nil, nil],
      0xf909 => [0, 2, "\345\245\221", "\345\245\221", nil, nil, nil],
      0xf90a => [0, 2, "\351\207\221", "\351\207\221", nil, nil, nil],
      0xf90b => [0, 2, "\345\226\207", "\345\226\207", nil, nil, nil],
      0xf90c => [0, 2, "\345\245\210", "\345\245\210", nil, nil, nil],
      0xf90d => [0, 2, "\346\207\266", "\346\207\266", nil, nil, nil],
      0xf90e => [0, 2, "\347\231\251", "\347\231\251", nil, nil, nil],
      0xf90f => [0, 2, "\347\276\205", "\347\276\205", nil, nil, nil],
      0xf910 => [0, 2, "\350\230\277", "\350\230\277", nil, nil, nil],
      0xf911 => [0, 2, "\350\236\272", "\350\236\272", nil, nil, nil],
      0xf912 => [0, 2, "\350\243\270", "\350\243\270", nil, nil, nil],
      0xf913 => [0, 2, "\351\202\217", "\351\202\217", nil, nil, nil],
      0xf914 => [0, 2, "\346\250\202", "\346\250\202", nil, nil, nil],
      0xf915 => [0, 2, "\346\264\233", "\346\264\233", nil, nil, nil],
      0xf916 => [0, 2, "\347\203\231", "\347\203\231", nil, nil, nil],
      0xf917 => [0, 2, "\347\217\236", "\347\217\236", nil, nil, nil],
      0xf918 => [0, 2, "\350\220\275", "\350\220\275", nil, nil, nil],
      0xf919 => [0, 2, "\351\205\252", "\351\205\252", nil, nil, nil],
      0xf91a => [0, 2, "\351\247\261", "\351\247\261", nil, nil, nil],
      0xf91b => [0, 2, "\344\272\202", "\344\272\202", nil, nil, nil],
      0xf91c => [0, 2, "\345\215\265", "\345\215\265", nil, nil, nil],
      0xf91d => [0, 2, "\346\254\204", "\346\254\204", nil, nil, nil],
      0xf91e => [0, 2, "\347\210\233", "\347\210\233", nil, nil, nil],
      0xf91f => [0, 2, "\350\230\255", "\350\230\255", nil, nil, nil],
      0xf920 => [0, 2, "\351\270\236", "\351\270\236", nil, nil, nil],
      0xf921 => [0, 2, "\345\265\220", "\345\265\220", nil, nil, nil],
      0xf922 => [0, 2, "\346\277\253", "\346\277\253", nil, nil, nil],
      0xf923 => [0, 2, "\350\227\215", "\350\227\215", nil, nil, nil],
      0xf924 => [0, 2, "\350\245\244", "\350\245\244", nil, nil, nil],
      0xf925 => [0, 2, "\346\213\211", "\346\213\211", nil, nil, nil],
      0xf926 => [0, 2, "\350\207\230", "\350\207\230", nil, nil, nil],
      0xf927 => [0, 2, "\350\240\237", "\350\240\237", nil, nil, nil],
      0xf928 => [0, 2, "\345\273\212", "\345\273\212", nil, nil, nil],
      0xf929 => [0, 2, "\346\234\227", "\346\234\227", nil, nil, nil],
      0xf92a => [0, 2, "\346\265\252", "\346\265\252", nil, nil, nil],
      0xf92b => [0, 2, "\347\213\274", "\347\213\274", nil, nil, nil],
      0xf92c => [0, 2, "\351\203\216", "\351\203\216", nil, nil, nil],
      0xf92d => [0, 2, "\344\276\206", "\344\276\206", nil, nil, nil],
      0xf92e => [0, 2, "\345\206\267", "\345\206\267", nil, nil, nil],
      0xf92f => [0, 2, "\345\213\236", "\345\213\236", nil, nil, nil],
      0xf930 => [0, 2, "\346\223\204", "\346\223\204", nil, nil, nil],
      0xf931 => [0, 2, "\346\253\223", "\346\253\223", nil, nil, nil],
      0xf932 => [0, 2, "\347\210\220", "\347\210\220", nil, nil, nil],
      0xf933 => [0, 2, "\347\233\247", "\347\233\247", nil, nil, nil],
      0xf934 => [0, 2, "\350\200\201", "\350\200\201", nil, nil, nil],
      0xf935 => [0, 2, "\350\230\206", "\350\230\206", nil, nil, nil],
      0xf936 => [0, 2, "\350\231\234", "\350\231\234", nil, nil, nil],
      0xf937 => [0, 2, "\350\267\257", "\350\267\257", nil, nil, nil],
      0xf938 => [0, 2, "\351\234\262", "\351\234\262", nil, nil, nil],
      0xf939 => [0, 2, "\351\255\257", "\351\255\257", nil, nil, nil],
      0xf93a => [0, 2, "\351\267\272", "\351\267\272", nil, nil, nil],
      0xf93b => [0, 2, "\347\242\214", "\347\242\214", nil, nil, nil],
      0xf93c => [0, 2, "\347\245\277", "\347\245\277", nil, nil, nil],
      0xf93d => [0, 2, "\347\266\240", "\347\266\240", nil, nil, nil],
      0xf93e => [0, 2, "\350\217\211", "\350\217\211", nil, nil, nil],
      0xf93f => [0, 2, "\351\214\204", "\351\214\204", nil, nil, nil],
      0xf940 => [0, 2, "\351\271\277", "\351\271\277", nil, nil, nil],
      0xf941 => [0, 2, "\350\253\226", "\350\253\226", nil, nil, nil],
      0xf942 => [0, 2, "\345\243\237", "\345\243\237", nil, nil, nil],
      0xf943 => [0, 2, "\345\274\204", "\345\274\204", nil, nil, nil],
      0xf944 => [0, 2, "\347\261\240", "\347\261\240", nil, nil, nil],
      0xf945 => [0, 2, "\350\201\276", "\350\201\276", nil, nil, nil],
      0xf946 => [0, 2, "\347\211\242", "\347\211\242", nil, nil, nil],
      0xf947 => [0, 2, "\347\243\212", "\347\243\212", nil, nil, nil],
      0xf948 => [0, 2, "\350\263\202", "\350\263\202", nil, nil, nil],
      0xf949 => [0, 2, "\351\233\267", "\351\233\267", nil, nil, nil],
      0xf94a => [0, 2, "\345\243\230", "\345\243\230", nil, nil, nil],
      0xf94b => [0, 2, "\345\261\242", "\345\261\242", nil, nil, nil],
      0xf94c => [0, 2, "\346\250\223", "\346\250\223", nil, nil, nil],
      0xf94d => [0, 2, "\346\267\232", "\346\267\232", nil, nil, nil],
      0xf94e => [0, 2, "\346\274\217", "\346\274\217", nil, nil, nil],
      0xf94f => [0, 2, "\347\264\257", "\347\264\257", nil, nil, nil],
      0xf950 => [0, 2, "\347\270\267", "\347\270\267", nil, nil, nil],
      0xf951 => [0, 2, "\351\233\273", "\351\233\273", nil, nil, nil],
      0xf952 => [0, 2, "\345\213\222", "\345\213\222", nil, nil, nil],
      0xf953 => [0, 2, "\350\202\213", "\350\202\213", nil, nil, nil],
      0xf954 => [0, 2, "\345\207\234", "\345\207\234", nil, nil, nil],
      0xf955 => [0, 2, "\345\207\214", "\345\207\214", nil, nil, nil],
      0xf956 => [0, 2, "\347\250\234", "\347\250\234", nil, nil, nil],
      0xf957 => [0, 2, "\347\266\276", "\347\266\276", nil, nil, nil],
      0xf958 => [0, 2, "\350\217\261", "\350\217\261", nil, nil, nil],
      0xf959 => [0, 2, "\351\231\265", "\351\231\265", nil, nil, nil],
      0xf95a => [0, 2, "\350\256\200", "\350\256\200", nil, nil, nil],
      0xf95b => [0, 2, "\346\213\217", "\346\213\217", nil, nil, nil],
      0xf95c => [0, 2, "\346\250\202", "\346\250\202", nil, nil, nil],
      0xf95d => [0, 2, "\350\253\276", "\350\253\276", nil, nil, nil],
      0xf95e => [0, 2, "\344\270\271", "\344\270\271", nil, nil, nil],
      0xf95f => [0, 2, "\345\257\247", "\345\257\247", nil, nil, nil],
      0xf960 => [0, 2, "\346\200\222", "\346\200\222", nil, nil, nil],
      0xf961 => [0, 2, "\347\216\207", "\347\216\207", nil, nil, nil],
      0xf962 => [0, 2, "\347\225\260", "\347\225\260", nil, nil, nil],
      0xf963 => [0, 2, "\345\214\227", "\345\214\227", nil, nil, nil],
      0xf964 => [0, 2, "\347\243\273", "\347\243\273", nil, nil, nil],
      0xf965 => [0, 2, "\344\276\277", "\344\276\277", nil, nil, nil],
      0xf966 => [0, 2, "\345\276\251", "\345\276\251", nil, nil, nil],
      0xf967 => [0, 2, "\344\270\215", "\344\270\215", nil, nil, nil],
      0xf968 => [0, 2, "\346\263\214", "\346\263\214", nil, nil, nil],
      0xf969 => [0, 2, "\346\225\270", "\346\225\270", nil, nil, nil],
      0xf96a => [0, 2, "\347\264\242", "\347\264\242", nil, nil, nil],
      0xf96b => [0, 2, "\345\217\203", "\345\217\203", nil, nil, nil],
      0xf96c => [0, 2, "\345\241\236", "\345\241\236", nil, nil, nil],
      0xf96d => [0, 2, "\347\234\201", "\347\234\201", nil, nil, nil],
      0xf96e => [0, 2, "\350\221\211", "\350\221\211", nil, nil, nil],
      0xf96f => [0, 2, "\350\252\252", "\350\252\252", nil, nil, nil],
      0xf970 => [0, 2, "\346\256\272", "\346\256\272", nil, nil, nil],
      0xf971 => [0, 2, "\350\276\260", "\350\276\260", nil, nil, nil],
      0xf972 => [0, 2, "\346\262\210", "\346\262\210", nil, nil, nil],
      0xf973 => [0, 2, "\346\213\276", "\346\213\276", nil, nil, nil],
      0xf974 => [0, 2, "\350\213\245", "\350\213\245", nil, nil, nil],
      0xf975 => [0, 2, "\346\216\240", "\346\216\240", nil, nil, nil],
      0xf976 => [0, 2, "\347\225\245", "\347\225\245", nil, nil, nil],
      0xf977 => [0, 2, "\344\272\256", "\344\272\256", nil, nil, nil],
      0xf978 => [0, 2, "\345\205\251", "\345\205\251", nil, nil, nil],
      0xf979 => [0, 2, "\345\207\211", "\345\207\211", nil, nil, nil],
      0xf97a => [0, 2, "\346\242\201", "\346\242\201", nil, nil, nil],
      0xf97b => [0, 2, "\347\263\247", "\347\263\247", nil, nil, nil],
      0xf97c => [0, 2, "\350\211\257", "\350\211\257", nil, nil, nil],
      0xf97d => [0, 2, "\350\253\222", "\350\253\222", nil, nil, nil],
      0xf97e => [0, 2, "\351\207\217", "\351\207\217", nil, nil, nil],
      0xf97f => [0, 2, "\345\213\265", "\345\213\265", nil, nil, nil],
      0xf980 => [0, 2, "\345\221\202", "\345\221\202", nil, nil, nil],
      0xf981 => [0, 2, "\345\245\263", "\345\245\263", nil, nil, nil],
      0xf982 => [0, 2, "\345\273\254", "\345\273\254", nil, nil, nil],
      0xf983 => [0, 2, "\346\227\205", "\346\227\205", nil, nil, nil],
      0xf984 => [0, 2, "\346\277\276", "\346\277\276", nil, nil, nil],
      0xf985 => [0, 2, "\347\244\252", "\347\244\252", nil, nil, nil],
      0xf986 => [0, 2, "\351\226\255", "\351\226\255", nil, nil, nil],
      0xf987 => [0, 2, "\351\251\252", "\351\251\252", nil, nil, nil],
      0xf988 => [0, 2, "\351\272\227", "\351\272\227", nil, nil, nil],
      0xf989 => [0, 2, "\351\273\216", "\351\273\216", nil, nil, nil],
      0xf98a => [0, 2, "\345\212\233", "\345\212\233", nil, nil, nil],
      0xf98b => [0, 2, "\346\233\206", "\346\233\206", nil, nil, nil],
      0xf98c => [0, 2, "\346\255\267", "\346\255\267", nil, nil, nil],
      0xf98d => [0, 2, "\350\275\242", "\350\275\242", nil, nil, nil],
      0xf98e => [0, 2, "\345\271\264", "\345\271\264", nil, nil, nil],
      0xf98f => [0, 2, "\346\206\220", "\346\206\220", nil, nil, nil],
      0xf990 => [0, 2, "\346\210\200", "\346\210\200", nil, nil, nil],
      0xf991 => [0, 2, "\346\222\232", "\346\222\232", nil, nil, nil],
      0xf992 => [0, 2, "\346\274\243", "\346\274\243", nil, nil, nil],
      0xf993 => [0, 2, "\347\205\211", "\347\205\211", nil, nil, nil],
      0xf994 => [0, 2, "\347\222\211", "\347\222\211", nil, nil, nil],
      0xf995 => [0, 2, "\347\247\212", "\347\247\212", nil, nil, nil],
      0xf996 => [0, 2, "\347\267\264", "\347\267\264", nil, nil, nil],
      0xf997 => [0, 2, "\350\201\257", "\350\201\257", nil, nil, nil],
      0xf998 => [0, 2, "\350\274\246", "\350\274\246", nil, nil, nil],
      0xf999 => [0, 2, "\350\223\256", "\350\223\256", nil, nil, nil],
      0xf99a => [0, 2, "\351\200\243", "\351\200\243", nil, nil, nil],
      0xf99b => [0, 2, "\351\215\212", "\351\215\212", nil, nil, nil],
      0xf99c => [0, 2, "\345\210\227", "\345\210\227", nil, nil, nil],
      0xf99d => [0, 2, "\345\212\243", "\345\212\243", nil, nil, nil],
      0xf99e => [0, 2, "\345\222\275", "\345\222\275", nil, nil, nil],
      0xf99f => [0, 2, "\347\203\210", "\347\203\210", nil, nil, nil],
      0xf9a0 => [0, 2, "\350\243\202", "\350\243\202", nil, nil, nil],
      0xf9a1 => [0, 2, "\350\252\252", "\350\252\252", nil, nil, nil],
      0xf9a2 => [0, 2, "\345\273\211", "\345\273\211", nil, nil, nil],
      0xf9a3 => [0, 2, "\345\277\265", "\345\277\265", nil, nil, nil],
      0xf9a4 => [0, 2, "\346\215\273", "\346\215\273", nil, nil, nil],
      0xf9a5 => [0, 2, "\346\256\256", "\346\256\256", nil, nil, nil],
      0xf9a6 => [0, 2, "\347\260\276", "\347\260\276", nil, nil, nil],
      0xf9a7 => [0, 2, "\347\215\265", "\347\215\265", nil, nil, nil],
      0xf9a8 => [0, 2, "\344\273\244", "\344\273\244", nil, nil, nil],
      0xf9a9 => [0, 2, "\345\233\271", "\345\233\271", nil, nil, nil],
      0xf9aa => [0, 2, "\345\257\247", "\345\257\247", nil, nil, nil],
      0xf9ab => [0, 2, "\345\266\272", "\345\266\272", nil, nil, nil],
      0xf9ac => [0, 2, "\346\200\234", "\346\200\234", nil, nil, nil],
      0xf9ad => [0, 2, "\347\216\262", "\347\216\262", nil, nil, nil],
      0xf9ae => [0, 2, "\347\221\251", "\347\221\251", nil, nil, nil],
      0xf9af => [0, 2, "\347\276\232", "\347\276\232", nil, nil, nil],
      0xf9b0 => [0, 2, "\350\201\206", "\350\201\206", nil, nil, nil],
      0xf9b1 => [0, 2, "\351\210\264", "\351\210\264", nil, nil, nil],
      0xf9b2 => [0, 2, "\351\233\266", "\351\233\266", nil, nil, nil],
      0xf9b3 => [0, 2, "\351\235\210", "\351\235\210", nil, nil, nil],
      0xf9b4 => [0, 2, "\351\240\230", "\351\240\230", nil, nil, nil],
      0xf9b5 => [0, 2, "\344\276\213", "\344\276\213", nil, nil, nil],
      0xf9b6 => [0, 2, "\347\246\256", "\347\246\256", nil, nil, nil],
      0xf9b7 => [0, 2, "\351\206\264", "\351\206\264", nil, nil, nil],
      0xf9b8 => [0, 2, "\351\232\270", "\351\232\270", nil, nil, nil],
      0xf9b9 => [0, 2, "\346\203\241", "\346\203\241", nil, nil, nil],
      0xf9ba => [0, 2, "\344\272\206", "\344\272\206", nil, nil, nil],
      0xf9bb => [0, 2, "\345\203\232", "\345\203\232", nil, nil, nil],
      0xf9bc => [0, 2, "\345\257\256", "\345\257\256", nil, nil, nil],
      0xf9bd => [0, 2, "\345\260\277", "\345\260\277", nil, nil, nil],
      0xf9be => [0, 2, "\346\226\231", "\346\226\231", nil, nil, nil],
      0xf9bf => [0, 2, "\346\250\202", "\346\250\202", nil, nil, nil],
      0xf9c0 => [0, 2, "\347\207\216", "\347\207\216", nil, nil, nil],
      0xf9c1 => [0, 2, "\347\231\202", "\347\231\202", nil, nil, nil],
      0xf9c2 => [0, 2, "\350\223\274", "\350\223\274", nil, nil, nil],
      0xf9c3 => [0, 2, "\351\201\274", "\351\201\274", nil, nil, nil],
      0xf9c4 => [0, 2, "\351\276\215", "\351\276\215", nil, nil, nil],
      0xf9c5 => [0, 2, "\346\232\210", "\346\232\210", nil, nil, nil],
      0xf9c6 => [0, 2, "\351\230\256", "\351\230\256", nil, nil, nil],
      0xf9c7 => [0, 2, "\345\212\211", "\345\212\211", nil, nil, nil],
      0xf9c8 => [0, 2, "\346\235\273", "\346\235\273", nil, nil, nil],
      0xf9c9 => [0, 2, "\346\237\263", "\346\237\263", nil, nil, nil],
      0xf9ca => [0, 2, "\346\265\201", "\346\265\201", nil, nil, nil],
      0xf9cb => [0, 2, "\346\272\234", "\346\272\234", nil, nil, nil],
      0xf9cc => [0, 2, "\347\220\211", "\347\220\211", nil, nil, nil],
      0xf9cd => [0, 2, "\347\225\231", "\347\225\231", nil, nil, nil],
      0xf9ce => [0, 2, "\347\241\253", "\347\241\253", nil, nil, nil],
      0xf9cf => [0, 2, "\347\264\220", "\347\264\220", nil, nil, nil],
      0xf9d0 => [0, 2, "\351\241\236", "\351\241\236", nil, nil, nil],
      0xf9d1 => [0, 2, "\345\205\255", "\345\205\255", nil, nil, nil],
      0xf9d2 => [0, 2, "\346\210\256", "\346\210\256", nil, nil, nil],
      0xf9d3 => [0, 2, "\351\231\270", "\351\231\270", nil, nil, nil],
      0xf9d4 => [0, 2, "\345\200\253", "\345\200\253", nil, nil, nil],
      0xf9d5 => [0, 2, "\345\264\231", "\345\264\231", nil, nil, nil],
      0xf9d6 => [0, 2, "\346\267\252", "\346\267\252", nil, nil, nil],
      0xf9d7 => [0, 2, "\350\274\252", "\350\274\252", nil, nil, nil],
      0xf9d8 => [0, 2, "\345\276\213", "\345\276\213", nil, nil, nil],
      0xf9d9 => [0, 2, "\346\205\204", "\346\205\204", nil, nil, nil],
      0xf9da => [0, 2, "\346\240\227", "\346\240\227", nil, nil, nil],
      0xf9db => [0, 2, "\347\216\207", "\347\216\207", nil, nil, nil],
      0xf9dc => [0, 2, "\351\232\206", "\351\232\206", nil, nil, nil],
      0xf9dd => [0, 2, "\345\210\251", "\345\210\251", nil, nil, nil],
      0xf9de => [0, 2, "\345\220\217", "\345\220\217", nil, nil, nil],
      0xf9df => [0, 2, "\345\261\245", "\345\261\245", nil, nil, nil],
      0xf9e0 => [0, 2, "\346\230\223", "\346\230\223", nil, nil, nil],
      0xf9e1 => [0, 2, "\346\235\216", "\346\235\216", nil, nil, nil],
      0xf9e2 => [0, 2, "\346\242\250", "\346\242\250", nil, nil, nil],
      0xf9e3 => [0, 2, "\346\263\245", "\346\263\245", nil, nil, nil],
      0xf9e4 => [0, 2, "\347\220\206", "\347\220\206", nil, nil, nil],
      0xf9e5 => [0, 2, "\347\227\242", "\347\227\242", nil, nil, nil],
      0xf9e6 => [0, 2, "\347\275\271", "\347\275\271", nil, nil, nil],
      0xf9e7 => [0, 2, "\350\243\217", "\350\243\217", nil, nil, nil],
      0xf9e8 => [0, 2, "\350\243\241", "\350\243\241", nil, nil, nil],
      0xf9e9 => [0, 2, "\351\207\214", "\351\207\214", nil, nil, nil],
      0xf9ea => [0, 2, "\351\233\242", "\351\233\242", nil, nil, nil],
      0xf9eb => [0, 2, "\345\214\277", "\345\214\277", nil, nil, nil],
      0xf9ec => [0, 2, "\346\272\272", "\346\272\272", nil, nil, nil],
      0xf9ed => [0, 2, "\345\220\235", "\345\220\235", nil, nil, nil],
      0xf9ee => [0, 2, "\347\207\220", "\347\207\220", nil, nil, nil],
      0xf9ef => [0, 2, "\347\222\230", "\347\222\230", nil, nil, nil],
      0xf9f0 => [0, 2, "\350\227\272", "\350\227\272", nil, nil, nil],
      0xf9f1 => [0, 2, "\351\232\243", "\351\232\243", nil, nil, nil],
      0xf9f2 => [0, 2, "\351\261\227", "\351\261\227", nil, nil, nil],
      0xf9f3 => [0, 2, "\351\272\237", "\351\272\237", nil, nil, nil],
      0xf9f4 => [0, 2, "\346\236\227", "\346\236\227", nil, nil, nil],
      0xf9f5 => [0, 2, "\346\267\213", "\346\267\213", nil, nil, nil],
      0xf9f6 => [0, 2, "\350\207\250", "\350\207\250", nil, nil, nil],
      0xf9f7 => [0, 2, "\347\253\213", "\347\253\213", nil, nil, nil],
      0xf9f8 => [0, 2, "\347\254\240", "\347\254\240", nil, nil, nil],
      0xf9f9 => [0, 2, "\347\262\222", "\347\262\222", nil, nil, nil],
      0xf9fa => [0, 2, "\347\213\200", "\347\213\200", nil, nil, nil],
      0xf9fb => [0, 2, "\347\202\231", "\347\202\231", nil, nil, nil],
      0xf9fc => [0, 2, "\350\255\230", "\350\255\230", nil, nil, nil],
      0xf9fd => [0, 2, "\344\273\200", "\344\273\200", nil, nil, nil],
      0xf9fe => [0, 2, "\350\214\266", "\350\214\266", nil, nil, nil],
      0xf9ff => [0, 2, "\345\210\272", "\345\210\272", nil, nil, nil],
      0xfa00 => [0, 2, "\345\210\207", "\345\210\207", nil, nil, nil],
      0xfa01 => [0, 2, "\345\272\246", "\345\272\246", nil, nil, nil],
      0xfa02 => [0, 2, "\346\213\223", "\346\213\223", nil, nil, nil],
      0xfa03 => [0, 2, "\347\263\226", "\347\263\226", nil, nil, nil],
      0xfa04 => [0, 2, "\345\256\205", "\345\256\205", nil, nil, nil],
      0xfa05 => [0, 2, "\346\264\236", "\346\264\236", nil, nil, nil],
      0xfa06 => [0, 2, "\346\232\264", "\346\232\264", nil, nil, nil],
      0xfa07 => [0, 2, "\350\274\273", "\350\274\273", nil, nil, nil],
      0xfa08 => [0, 2, "\350\241\214", "\350\241\214", nil, nil, nil],
      0xfa09 => [0, 2, "\351\231\215", "\351\231\215", nil, nil, nil],
      0xfa0a => [0, 2, "\350\246\213", "\350\246\213", nil, nil, nil],
      0xfa0b => [0, 2, "\345\273\223", "\345\273\223", nil, nil, nil],
      0xfa0c => [0, 2, "\345\205\200", "\345\205\200", nil, nil, nil],
      0xfa0d => [0, 2, "\345\227\200", "\345\227\200", nil, nil, nil],
      0xfa10 => [0, 2, "\345\241\232", "\345\241\232", nil, nil, nil],
      0xfa12 => [0, 2, "\346\231\264", "\346\231\264", nil, nil, nil],
      0xfa15 => [0, 2, "\345\207\236", "\345\207\236", nil, nil, nil],
      0xfa16 => [0, 2, "\347\214\252", "\347\214\252", nil, nil, nil],
      0xfa17 => [0, 2, "\347\233\212", "\347\233\212", nil, nil, nil],
      0xfa18 => [0, 2, "\347\244\274", "\347\244\274", nil, nil, nil],
      0xfa19 => [0, 2, "\347\245\236", "\347\245\236", nil, nil, nil],
      0xfa1a => [0, 2, "\347\245\245", "\347\245\245", nil, nil, nil],
      0xfa1b => [0, 2, "\347\246\217", "\347\246\217", nil, nil, nil],
      0xfa1c => [0, 2, "\351\235\226", "\351\235\226", nil, nil, nil],
      0xfa1d => [0, 2, "\347\262\276", "\347\262\276", nil, nil, nil],
      0xfa1e => [0, 2, "\347\276\275", "\347\276\275", nil, nil, nil],
      0xfa20 => [0, 2, "\350\230\222", "\350\230\222", nil, nil, nil],
      0xfa22 => [0, 2, "\350\253\270", "\350\253\270", nil, nil, nil],
      0xfa25 => [0, 2, "\351\200\270", "\351\200\270", nil, nil, nil],
      0xfa26 => [0, 2, "\351\203\275", "\351\203\275", nil, nil, nil],
      0xfa2a => [0, 2, "\351\243\257", "\351\243\257", nil, nil, nil],
      0xfa2b => [0, 2, "\351\243\274", "\351\243\274", nil, nil, nil],
      0xfa2c => [0, 2, "\351\244\250", "\351\244\250", nil, nil, nil],
      0xfa2d => [0, 2, "\351\266\264", "\351\266\264", nil, nil, nil],
      0xfb00 => [0, 0, nil, "ff", nil, nil, nil],
      0xfb01 => [0, 0, nil, "fi", nil, nil, nil],
      0xfb02 => [0, 0, nil, "fl", nil, nil, nil],
      0xfb03 => [0, 0, nil, "ffi", nil, nil, nil],
      0xfb04 => [0, 0, nil, "ffl", nil, nil, nil],
      0xfb05 => [0, 0, nil, "\305\277t", nil, nil, nil],
      0xfb06 => [0, 0, nil, "st", nil, nil, nil],
      0xfb13 => [0, 0, nil, "\325\264\325\266", nil, nil, nil],
      0xfb14 => [0, 0, nil, "\325\264\325\245", nil, nil, nil],
      0xfb15 => [0, 0, nil, "\325\264\325\253", nil, nil, nil],
      0xfb16 => [0, 0, nil, "\325\276\325\266", nil, nil, nil],
      0xfb17 => [0, 0, nil, "\325\264\325\255", nil, nil, nil],
      0xfb1d => [0, 0, "\327\231\326\264", "\327\231\326\264", nil, nil, nil],
      0xfb1e => [26, 0, nil, nil, nil, nil, nil],
      0xfb1f => [0, 1, "\327\262\326\267", "\327\262\326\267", nil, nil, nil],
      0xfb20 => [0, 0, nil, "\327\242", nil, nil, nil],
      0xfb21 => [0, 0, nil, "\327\220", nil, nil, nil],
      0xfb22 => [0, 0, nil, "\327\223", nil, nil, nil],
      0xfb23 => [0, 0, nil, "\327\224", nil, nil, nil],
      0xfb24 => [0, 0, nil, "\327\233", nil, nil, nil],
      0xfb25 => [0, 0, nil, "\327\234", nil, nil, nil],
      0xfb26 => [0, 0, nil, "\327\235", nil, nil, nil],
      0xfb27 => [0, 0, nil, "\327\250", nil, nil, nil],
      0xfb28 => [0, 0, nil, "\327\252", nil, nil, nil],
      0xfb29 => [0, 0, nil, "+", nil, nil, nil],
      0xfb2a => [0, 1, "\327\251\327\201", "\327\251\327\201", nil, nil, nil],
      0xfb2b => [0, 1, "\327\251\327\202", "\327\251\327\202", nil, nil, nil],
      0xfb2c => [0, 1, "\357\255\211\327\201", "\357\255\211\327\201", nil, nil, nil],
      0xfb2d => [0, 1, "\357\255\211\327\202", "\357\255\211\327\202", nil, nil, nil],
      0xfb2e => [0, 1, "\327\220\326\267", "\327\220\326\267", nil, nil, nil],
      0xfb2f => [0, 1, "\327\220\326\270", "\327\220\326\270", nil, nil, nil],
      0xfb30 => [0, 1, "\327\220\326\274", "\327\220\326\274", nil, nil, nil],
      0xfb31 => [0, 1, "\327\221\326\274", "\327\221\326\274", nil, nil, nil],
      0xfb32 => [0, 1, "\327\222\326\274", "\327\222\326\274", nil, nil, nil],
      0xfb33 => [0, 1, "\327\223\326\274", "\327\223\326\274", nil, nil, nil],
      0xfb34 => [0, 1, "\327\224\326\274", "\327\224\326\274", nil, nil, nil],
      0xfb35 => [0, 1, "\327\225\326\274", "\327\225\326\274", nil, nil, nil],
      0xfb36 => [0, 1, "\327\226\326\274", "\327\226\326\274", nil, nil, nil],
      0xfb38 => [0, 1, "\327\230\326\274", "\327\230\326\274", nil, nil, nil],
      0xfb39 => [0, 1, "\327\231\326\274", "\327\231\326\274", nil, nil, nil],
      0xfb3a => [0, 1, "\327\232\326\274", "\327\232\326\274", nil, nil, nil],
      0xfb3b => [0, 1, "\327\233\326\274", "\327\233\326\274", nil, nil, nil],
      0xfb3c => [0, 1, "\327\234\326\274", "\327\234\326\274", nil, nil, nil],
      0xfb3e => [0, 1, "\327\236\326\274", "\327\236\326\274", nil, nil, nil],
      0xfb40 => [0, 1, "\327\240\326\274", "\327\240\326\274", nil, nil, nil],
      0xfb41 => [0, 1, "\327\241\326\274", "\327\241\326\274", nil, nil, nil],
      0xfb43 => [0, 1, "\327\243\326\274", "\327\243\326\274", nil, nil, nil],
      0xfb44 => [0, 1, "\327\244\326\274", "\327\244\326\274", nil, nil, nil],
      0xfb46 => [0, 1, "\327\246\326\274", "\327\246\326\274", nil, nil, nil],
      0xfb47 => [0, 1, "\327\247\326\274", "\327\247\326\274", nil, nil, nil],
      0xfb48 => [0, 1, "\327\250\326\274", "\327\250\326\274", nil, nil, nil],
      0xfb49 => [0, 1, "\327\251\326\274", "\327\251\326\274", nil, nil, nil],
      0xfb4a => [0, 1, "\327\252\326\274", "\327\252\326\274", nil, nil, nil],
      0xfb4b => [0, 1, "\327\225\326\271", "\327\225\326\271", nil, nil, nil],
      0xfb4c => [0, 1, "\327\221\326\277", "\327\221\326\277", nil, nil, nil],
      0xfb4d => [0, 1, "\327\233\326\277", "\327\233\326\277", nil, nil, nil],
      0xfb4e => [0, 1, "\327\244\326\277", "\327\244\326\277", nil, nil, nil],
      0xfb4f => [0, 0, nil, "\327\220\327\234", nil, nil, nil],
      0xfb50 => [0, 0, nil, "\331\261", nil, nil, nil],
      0xfb51 => [0, 0, nil, "\331\261", nil, nil, nil],
      0xfb52 => [0, 0, nil, "\331\273", nil, nil, nil],
      0xfb53 => [0, 0, nil, "\331\273", nil, nil, nil],
      0xfb54 => [0, 0, nil, "\331\273", nil, nil, nil],
      0xfb55 => [0, 0, nil, "\331\273", nil, nil, nil],
      0xfb56 => [0, 0, nil, "\331\276", nil, nil, nil],
      0xfb57 => [0, 0, nil, "\331\276", nil, nil, nil],
      0xfb58 => [0, 0, nil, "\331\276", nil, nil, nil],
      0xfb59 => [0, 0, nil, "\331\276", nil, nil, nil],
      0xfb5a => [0, 0, nil, "\332\200", nil, nil, nil],
      0xfb5b => [0, 0, nil, "\332\200", nil, nil, nil],
      0xfb5c => [0, 0, nil, "\332\200", nil, nil, nil],
      0xfb5d => [0, 0, nil, "\332\200", nil, nil, nil],
      0xfb5e => [0, 0, nil, "\331\272", nil, nil, nil],
      0xfb5f => [0, 0, nil, "\331\272", nil, nil, nil],
      0xfb60 => [0, 0, nil, "\331\272", nil, nil, nil],
      0xfb61 => [0, 0, nil, "\331\272", nil, nil, nil],
      0xfb62 => [0, 0, nil, "\331\277", nil, nil, nil],
      0xfb63 => [0, 0, nil, "\331\277", nil, nil, nil],
      0xfb64 => [0, 0, nil, "\331\277", nil, nil, nil],
      0xfb65 => [0, 0, nil, "\331\277", nil, nil, nil],
      0xfb66 => [0, 0, nil, "\331\271", nil, nil, nil],
      0xfb67 => [0, 0, nil, "\331\271", nil, nil, nil],
      0xfb68 => [0, 0, nil, "\331\271", nil, nil, nil],
      0xfb69 => [0, 0, nil, "\331\271", nil, nil, nil],
      0xfb6a => [0, 0, nil, "\332\244", nil, nil, nil],
      0xfb6b => [0, 0, nil, "\332\244", nil, nil, nil],
      0xfb6c => [0, 0, nil, "\332\244", nil, nil, nil],
      0xfb6d => [0, 0, nil, "\332\244", nil, nil, nil],
      0xfb6e => [0, 0, nil, "\332\246", nil, nil, nil],
      0xfb6f => [0, 0, nil, "\332\246", nil, nil, nil],
      0xfb70 => [0, 0, nil, "\332\246", nil, nil, nil],
      0xfb71 => [0, 0, nil, "\332\246", nil, nil, nil],
      0xfb72 => [0, 0, nil, "\332\204", nil, nil, nil],
      0xfb73 => [0, 0, nil, "\332\204", nil, nil, nil],
      0xfb74 => [0, 0, nil, "\332\204", nil, nil, nil],
      0xfb75 => [0, 0, nil, "\332\204", nil, nil, nil],
      0xfb76 => [0, 0, nil, "\332\203", nil, nil, nil],
      0xfb77 => [0, 0, nil, "\332\203", nil, nil, nil],
      0xfb78 => [0, 0, nil, "\332\203", nil, nil, nil],
      0xfb79 => [0, 0, nil, "\332\203", nil, nil, nil],
      0xfb7a => [0, 0, nil, "\332\206", nil, nil, nil],
      0xfb7b => [0, 0, nil, "\332\206", nil, nil, nil],
      0xfb7c => [0, 0, nil, "\332\206", nil, nil, nil],
      0xfb7d => [0, 0, nil, "\332\206", nil, nil, nil],
      0xfb7e => [0, 0, nil, "\332\207", nil, nil, nil],
      0xfb7f => [0, 0, nil, "\332\207", nil, nil, nil],
      0xfb80 => [0, 0, nil, "\332\207", nil, nil, nil],
      0xfb81 => [0, 0, nil, "\332\207", nil, nil, nil],
      0xfb82 => [0, 0, nil, "\332\215", nil, nil, nil],
      0xfb83 => [0, 0, nil, "\332\215", nil, nil, nil],
      0xfb84 => [0, 0, nil, "\332\214", nil, nil, nil],
      0xfb85 => [0, 0, nil, "\332\214", nil, nil, nil],
      0xfb86 => [0, 0, nil, "\332\216", nil, nil, nil],
      0xfb87 => [0, 0, nil, "\332\216", nil, nil, nil],
      0xfb88 => [0, 0, nil, "\332\210", nil, nil, nil],
      0xfb89 => [0, 0, nil, "\332\210", nil, nil, nil],
      0xfb8a => [0, 0, nil, "\332\230", nil, nil, nil],
      0xfb8b => [0, 0, nil, "\332\230", nil, nil, nil],
      0xfb8c => [0, 0, nil, "\332\221", nil, nil, nil],
      0xfb8d => [0, 0, nil, "\332\221", nil, nil, nil],
      0xfb8e => [0, 0, nil, "\332\251", nil, nil, nil],
      0xfb8f => [0, 0, nil, "\332\251", nil, nil, nil],
      0xfb90 => [0, 0, nil, "\332\251", nil, nil, nil],
      0xfb91 => [0, 0, nil, "\332\251", nil, nil, nil],
      0xfb92 => [0, 0, nil, "\332\257", nil, nil, nil],
      0xfb93 => [0, 0, nil, "\332\257", nil, nil, nil],
      0xfb94 => [0, 0, nil, "\332\257", nil, nil, nil],
      0xfb95 => [0, 0, nil, "\332\257", nil, nil, nil],
      0xfb96 => [0, 0, nil, "\332\263", nil, nil, nil],
      0xfb97 => [0, 0, nil, "\332\263", nil, nil, nil],
      0xfb98 => [0, 0, nil, "\332\263", nil, nil, nil],
      0xfb99 => [0, 0, nil, "\332\263", nil, nil, nil],
      0xfb9a => [0, 0, nil, "\332\261", nil, nil, nil],
      0xfb9b => [0, 0, nil, "\332\261", nil, nil, nil],
      0xfb9c => [0, 0, nil, "\332\261", nil, nil, nil],
      0xfb9d => [0, 0, nil, "\332\261", nil, nil, nil],
      0xfb9e => [0, 0, nil, "\332\272", nil, nil, nil],
      0xfb9f => [0, 0, nil, "\332\272", nil, nil, nil],
      0xfba0 => [0, 0, nil, "\332\273", nil, nil, nil],
      0xfba1 => [0, 0, nil, "\332\273", nil, nil, nil],
      0xfba2 => [0, 0, nil, "\332\273", nil, nil, nil],
      0xfba3 => [0, 0, nil, "\332\273", nil, nil, nil],
      0xfba4 => [0, 0, nil, "\333\200", nil, nil, nil],
      0xfba5 => [0, 0, nil, "\333\200", nil, nil, nil],
      0xfba6 => [0, 0, nil, "\333\201", nil, nil, nil],
      0xfba7 => [0, 0, nil, "\333\201", nil, nil, nil],
      0xfba8 => [0, 0, nil, "\333\201", nil, nil, nil],
      0xfba9 => [0, 0, nil, "\333\201", nil, nil, nil],
      0xfbaa => [0, 0, nil, "\332\276", nil, nil, nil],
      0xfbab => [0, 0, nil, "\332\276", nil, nil, nil],
      0xfbac => [0, 0, nil, "\332\276", nil, nil, nil],
      0xfbad => [0, 0, nil, "\332\276", nil, nil, nil],
      0xfbae => [0, 0, nil, "\333\222", nil, nil, nil],
      0xfbaf => [0, 0, nil, "\333\222", nil, nil, nil],
      0xfbb0 => [0, 0, nil, "\333\223", nil, nil, nil],
      0xfbb1 => [0, 0, nil, "\333\223", nil, nil, nil],
      0xfbd3 => [0, 0, nil, "\332\255", nil, nil, nil],
      0xfbd4 => [0, 0, nil, "\332\255", nil, nil, nil],
      0xfbd5 => [0, 0, nil, "\332\255", nil, nil, nil],
      0xfbd6 => [0, 0, nil, "\332\255", nil, nil, nil],
      0xfbd7 => [0, 0, nil, "\333\207", nil, nil, nil],
      0xfbd8 => [0, 0, nil, "\333\207", nil, nil, nil],
      0xfbd9 => [0, 0, nil, "\333\206", nil, nil, nil],
      0xfbda => [0, 0, nil, "\333\206", nil, nil, nil],
      0xfbdb => [0, 0, nil, "\333\210", nil, nil, nil],
      0xfbdc => [0, 0, nil, "\333\210", nil, nil, nil],
      0xfbdd => [0, 0, nil, "\331\267", nil, nil, nil],
      0xfbde => [0, 0, nil, "\333\213", nil, nil, nil],
      0xfbdf => [0, 0, nil, "\333\213", nil, nil, nil],
      0xfbe0 => [0, 0, nil, "\333\205", nil, nil, nil],
      0xfbe1 => [0, 0, nil, "\333\205", nil, nil, nil],
      0xfbe2 => [0, 0, nil, "\333\211", nil, nil, nil],
      0xfbe3 => [0, 0, nil, "\333\211", nil, nil, nil],
      0xfbe4 => [0, 0, nil, "\333\220", nil, nil, nil],
      0xfbe5 => [0, 0, nil, "\333\220", nil, nil, nil],
      0xfbe6 => [0, 0, nil, "\333\220", nil, nil, nil],
      0xfbe7 => [0, 0, nil, "\333\220", nil, nil, nil],
      0xfbe8 => [0, 0, nil, "\331\211", nil, nil, nil],
      0xfbe9 => [0, 0, nil, "\331\211", nil, nil, nil],
      0xfbea => [0, 0, nil, "\330\246\330\247", nil, nil, nil],
      0xfbeb => [0, 0, nil, "\330\246\330\247", nil, nil, nil],
      0xfbec => [0, 0, nil, "\330\246\333\225", nil, nil, nil],
      0xfbed => [0, 0, nil, "\330\246\333\225", nil, nil, nil],
      0xfbee => [0, 0, nil, "\330\246\331\210", nil, nil, nil],
      0xfbef => [0, 0, nil, "\330\246\331\210", nil, nil, nil],
      0xfbf0 => [0, 0, nil, "\330\246\333\207", nil, nil, nil],
      0xfbf1 => [0, 0, nil, "\330\246\333\207", nil, nil, nil],
      0xfbf2 => [0, 0, nil, "\330\246\333\206", nil, nil, nil],
      0xfbf3 => [0, 0, nil, "\330\246\333\206", nil, nil, nil],
      0xfbf4 => [0, 0, nil, "\330\246\333\210", nil, nil, nil],
      0xfbf5 => [0, 0, nil, "\330\246\333\210", nil, nil, nil],
      0xfbf6 => [0, 0, nil, "\330\246\333\220", nil, nil, nil],
      0xfbf7 => [0, 0, nil, "\330\246\333\220", nil, nil, nil],
      0xfbf8 => [0, 0, nil, "\330\246\333\220", nil, nil, nil],
      0xfbf9 => [0, 0, nil, "\330\246\331\211", nil, nil, nil],
      0xfbfa => [0, 0, nil, "\330\246\331\211", nil, nil, nil],
      0xfbfb => [0, 0, nil, "\330\246\331\211", nil, nil, nil],
      0xfbfc => [0, 0, nil, "\333\214", nil, nil, nil],
      0xfbfd => [0, 0, nil, "\333\214", nil, nil, nil],
      0xfbfe => [0, 0, nil, "\333\214", nil, nil, nil],
      0xfbff => [0, 0, nil, "\333\214", nil, nil, nil],
      0xfc00 => [0, 0, nil, "\330\246\330\254", nil, nil, nil],
      0xfc01 => [0, 0, nil, "\330\246\330\255", nil, nil, nil],
      0xfc02 => [0, 0, nil, "\330\246\331\205", nil, nil, nil],
      0xfc03 => [0, 0, nil, "\330\246\331\211", nil, nil, nil],
      0xfc04 => [0, 0, nil, "\330\246\331\212", nil, nil, nil],
      0xfc05 => [0, 0, nil, "\330\250\330\254", nil, nil, nil],
      0xfc06 => [0, 0, nil, "\330\250\330\255", nil, nil, nil],
      0xfc07 => [0, 0, nil, "\330\250\330\256", nil, nil, nil],
      0xfc08 => [0, 0, nil, "\330\250\331\205", nil, nil, nil],
      0xfc09 => [0, 0, nil, "\330\250\331\211", nil, nil, nil],
      0xfc0a => [0, 0, nil, "\330\250\331\212", nil, nil, nil],
      0xfc0b => [0, 0, nil, "\330\252\330\254", nil, nil, nil],
      0xfc0c => [0, 0, nil, "\330\252\330\255", nil, nil, nil],
      0xfc0d => [0, 0, nil, "\330\252\330\256", nil, nil, nil],
      0xfc0e => [0, 0, nil, "\330\252\331\205", nil, nil, nil],
      0xfc0f => [0, 0, nil, "\330\252\331\211", nil, nil, nil],
      0xfc10 => [0, 0, nil, "\330\252\331\212", nil, nil, nil],
      0xfc11 => [0, 0, nil, "\330\253\330\254", nil, nil, nil],
      0xfc12 => [0, 0, nil, "\330\253\331\205", nil, nil, nil],
      0xfc13 => [0, 0, nil, "\330\253\331\211", nil, nil, nil],
      0xfc14 => [0, 0, nil, "\330\253\331\212", nil, nil, nil],
      0xfc15 => [0, 0, nil, "\330\254\330\255", nil, nil, nil],
      0xfc16 => [0, 0, nil, "\330\254\331\205", nil, nil, nil],
      0xfc17 => [0, 0, nil, "\330\255\330\254", nil, nil, nil],
      0xfc18 => [0, 0, nil, "\330\255\331\205", nil, nil, nil],
      0xfc19 => [0, 0, nil, "\330\256\330\254", nil, nil, nil],
      0xfc1a => [0, 0, nil, "\330\256\330\255", nil, nil, nil],
      0xfc1b => [0, 0, nil, "\330\256\331\205", nil, nil, nil],
      0xfc1c => [0, 0, nil, "\330\263\330\254", nil, nil, nil],
      0xfc1d => [0, 0, nil, "\330\263\330\255", nil, nil, nil],
      0xfc1e => [0, 0, nil, "\330\263\330\256", nil, nil, nil],
      0xfc1f => [0, 0, nil, "\330\263\331\205", nil, nil, nil],
      0xfc20 => [0, 0, nil, "\330\265\330\255", nil, nil, nil],
      0xfc21 => [0, 0, nil, "\330\265\331\205", nil, nil, nil],
      0xfc22 => [0, 0, nil, "\330\266\330\254", nil, nil, nil],
      0xfc23 => [0, 0, nil, "\330\266\330\255", nil, nil, nil],
      0xfc24 => [0, 0, nil, "\330\266\330\256", nil, nil, nil],
      0xfc25 => [0, 0, nil, "\330\266\331\205", nil, nil, nil],
      0xfc26 => [0, 0, nil, "\330\267\330\255", nil, nil, nil],
      0xfc27 => [0, 0, nil, "\330\267\331\205", nil, nil, nil],
      0xfc28 => [0, 0, nil, "\330\270\331\205", nil, nil, nil],
      0xfc29 => [0, 0, nil, "\330\271\330\254", nil, nil, nil],
      0xfc2a => [0, 0, nil, "\330\271\331\205", nil, nil, nil],
      0xfc2b => [0, 0, nil, "\330\272\330\254", nil, nil, nil],
      0xfc2c => [0, 0, nil, "\330\272\331\205", nil, nil, nil],
      0xfc2d => [0, 0, nil, "\331\201\330\254", nil, nil, nil],
      0xfc2e => [0, 0, nil, "\331\201\330\255", nil, nil, nil],
      0xfc2f => [0, 0, nil, "\331\201\330\256", nil, nil, nil],
      0xfc30 => [0, 0, nil, "\331\201\331\205", nil, nil, nil],
      0xfc31 => [0, 0, nil, "\331\201\331\211", nil, nil, nil],
      0xfc32 => [0, 0, nil, "\331\201\331\212", nil, nil, nil],
      0xfc33 => [0, 0, nil, "\331\202\330\255", nil, nil, nil],
      0xfc34 => [0, 0, nil, "\331\202\331\205", nil, nil, nil],
      0xfc35 => [0, 0, nil, "\331\202\331\211", nil, nil, nil],
      0xfc36 => [0, 0, nil, "\331\202\331\212", nil, nil, nil],
      0xfc37 => [0, 0, nil, "\331\203\330\247", nil, nil, nil],
      0xfc38 => [0, 0, nil, "\331\203\330\254", nil, nil, nil],
      0xfc39 => [0, 0, nil, "\331\203\330\255", nil, nil, nil],
      0xfc3a => [0, 0, nil, "\331\203\330\256", nil, nil, nil],
      0xfc3b => [0, 0, nil, "\331\203\331\204", nil, nil, nil],
      0xfc3c => [0, 0, nil, "\331\203\331\205", nil, nil, nil],
      0xfc3d => [0, 0, nil, "\331\203\331\211", nil, nil, nil],
      0xfc3e => [0, 0, nil, "\331\203\331\212", nil, nil, nil],
      0xfc3f => [0, 0, nil, "\331\204\330\254", nil, nil, nil],
      0xfc40 => [0, 0, nil, "\331\204\330\255", nil, nil, nil],
      0xfc41 => [0, 0, nil, "\331\204\330\256", nil, nil, nil],
      0xfc42 => [0, 0, nil, "\331\204\331\205", nil, nil, nil],
      0xfc43 => [0, 0, nil, "\331\204\331\211", nil, nil, nil],
      0xfc44 => [0, 0, nil, "\331\204\331\212", nil, nil, nil],
      0xfc45 => [0, 0, nil, "\331\205\330\254", nil, nil, nil],
      0xfc46 => [0, 0, nil, "\331\205\330\255", nil, nil, nil],
      0xfc47 => [0, 0, nil, "\331\205\330\256", nil, nil, nil],
      0xfc48 => [0, 0, nil, "\331\205\331\205", nil, nil, nil],
      0xfc49 => [0, 0, nil, "\331\205\331\211", nil, nil, nil],
      0xfc4a => [0, 0, nil, "\331\205\331\212", nil, nil, nil],
      0xfc4b => [0, 0, nil, "\331\206\330\254", nil, nil, nil],
      0xfc4c => [0, 0, nil, "\331\206\330\255", nil, nil, nil],
      0xfc4d => [0, 0, nil, "\331\206\330\256", nil, nil, nil],
      0xfc4e => [0, 0, nil, "\331\206\331\205", nil, nil, nil],
      0xfc4f => [0, 0, nil, "\331\206\331\211", nil, nil, nil],
      0xfc50 => [0, 0, nil, "\331\206\331\212", nil, nil, nil],
      0xfc51 => [0, 0, nil, "\331\207\330\254", nil, nil, nil],
      0xfc52 => [0, 0, nil, "\331\207\331\205", nil, nil, nil],
      0xfc53 => [0, 0, nil, "\331\207\331\211", nil, nil, nil],
      0xfc54 => [0, 0, nil, "\331\207\331\212", nil, nil, nil],
      0xfc55 => [0, 0, nil, "\331\212\330\254", nil, nil, nil],
      0xfc56 => [0, 0, nil, "\331\212\330\255", nil, nil, nil],
      0xfc57 => [0, 0, nil, "\331\212\330\256", nil, nil, nil],
      0xfc58 => [0, 0, nil, "\331\212\331\205", nil, nil, nil],
      0xfc59 => [0, 0, nil, "\331\212\331\211", nil, nil, nil],
      0xfc5a => [0, 0, nil, "\331\212\331\212", nil, nil, nil],
      0xfc5b => [0, 0, nil, "\330\260\331\260", nil, nil, nil],
      0xfc5c => [0, 0, nil, "\330\261\331\260", nil, nil, nil],
      0xfc5d => [0, 0, nil, "\331\211\331\260", nil, nil, nil],
      0xfc5e => [0, 0, nil, " \331\214\331\221", nil, nil, nil],
      0xfc5f => [0, 0, nil, " \331\215\331\221", nil, nil, nil],
      0xfc60 => [0, 0, nil, " \331\216\331\221", nil, nil, nil],
      0xfc61 => [0, 0, nil, " \331\217\331\221", nil, nil, nil],
      0xfc62 => [0, 0, nil, " \331\220\331\221", nil, nil, nil],
      0xfc63 => [0, 0, nil, " \331\221\331\260", nil, nil, nil],
      0xfc64 => [0, 0, nil, "\330\246\330\261", nil, nil, nil],
      0xfc65 => [0, 0, nil, "\330\246\330\262", nil, nil, nil],
      0xfc66 => [0, 0, nil, "\330\246\331\205", nil, nil, nil],
      0xfc67 => [0, 0, nil, "\330\246\331\206", nil, nil, nil],
      0xfc68 => [0, 0, nil, "\330\246\331\211", nil, nil, nil],
      0xfc69 => [0, 0, nil, "\330\246\331\212", nil, nil, nil],
      0xfc6a => [0, 0, nil, "\330\250\330\261", nil, nil, nil],
      0xfc6b => [0, 0, nil, "\330\250\330\262", nil, nil, nil],
      0xfc6c => [0, 0, nil, "\330\250\331\205", nil, nil, nil],
      0xfc6d => [0, 0, nil, "\330\250\331\206", nil, nil, nil],
      0xfc6e => [0, 0, nil, "\330\250\331\211", nil, nil, nil],
      0xfc6f => [0, 0, nil, "\330\250\331\212", nil, nil, nil],
      0xfc70 => [0, 0, nil, "\330\252\330\261", nil, nil, nil],
      0xfc71 => [0, 0, nil, "\330\252\330\262", nil, nil, nil],
      0xfc72 => [0, 0, nil, "\330\252\331\205", nil, nil, nil],
      0xfc73 => [0, 0, nil, "\330\252\331\206", nil, nil, nil],
      0xfc74 => [0, 0, nil, "\330\252\331\211", nil, nil, nil],
      0xfc75 => [0, 0, nil, "\330\252\331\212", nil, nil, nil],
      0xfc76 => [0, 0, nil, "\330\253\330\261", nil, nil, nil],
      0xfc77 => [0, 0, nil, "\330\253\330\262", nil, nil, nil],
      0xfc78 => [0, 0, nil, "\330\253\331\205", nil, nil, nil],
      0xfc79 => [0, 0, nil, "\330\253\331\206", nil, nil, nil],
      0xfc7a => [0, 0, nil, "\330\253\331\211", nil, nil, nil],
      0xfc7b => [0, 0, nil, "\330\253\331\212", nil, nil, nil],
      0xfc7c => [0, 0, nil, "\331\201\331\211", nil, nil, nil],
      0xfc7d => [0, 0, nil, "\331\201\331\212", nil, nil, nil],
      0xfc7e => [0, 0, nil, "\331\202\331\211", nil, nil, nil],
      0xfc7f => [0, 0, nil, "\331\202\331\212", nil, nil, nil],
      0xfc80 => [0, 0, nil, "\331\203\330\247", nil, nil, nil],
      0xfc81 => [0, 0, nil, "\331\203\331\204", nil, nil, nil],
      0xfc82 => [0, 0, nil, "\331\203\331\205", nil, nil, nil],
      0xfc83 => [0, 0, nil, "\331\203\331\211", nil, nil, nil],
      0xfc84 => [0, 0, nil, "\331\203\331\212", nil, nil, nil],
      0xfc85 => [0, 0, nil, "\331\204\331\205", nil, nil, nil],
      0xfc86 => [0, 0, nil, "\331\204\331\211", nil, nil, nil],
      0xfc87 => [0, 0, nil, "\331\204\331\212", nil, nil, nil],
      0xfc88 => [0, 0, nil, "\331\205\330\247", nil, nil, nil],
      0xfc89 => [0, 0, nil, "\331\205\331\205", nil, nil, nil],
      0xfc8a => [0, 0, nil, "\331\206\330\261", nil, nil, nil],
      0xfc8b => [0, 0, nil, "\331\206\330\262", nil, nil, nil],
      0xfc8c => [0, 0, nil, "\331\206\331\205", nil, nil, nil],
      0xfc8d => [0, 0, nil, "\331\206\331\206", nil, nil, nil],
      0xfc8e => [0, 0, nil, "\331\206\331\211", nil, nil, nil],
      0xfc8f => [0, 0, nil, "\331\206\331\212", nil, nil, nil],
      0xfc90 => [0, 0, nil, "\331\211\331\260", nil, nil, nil],
      0xfc91 => [0, 0, nil, "\331\212\330\261", nil, nil, nil],
      0xfc92 => [0, 0, nil, "\331\212\330\262", nil, nil, nil],
      0xfc93 => [0, 0, nil, "\331\212\331\205", nil, nil, nil],
      0xfc94 => [0, 0, nil, "\331\212\331\206", nil, nil, nil],
      0xfc95 => [0, 0, nil, "\331\212\331\211", nil, nil, nil],
      0xfc96 => [0, 0, nil, "\331\212\331\212", nil, nil, nil],
      0xfc97 => [0, 0, nil, "\330\246\330\254", nil, nil, nil],
      0xfc98 => [0, 0, nil, "\330\246\330\255", nil, nil, nil],
      0xfc99 => [0, 0, nil, "\330\246\330\256", nil, nil, nil],
      0xfc9a => [0, 0, nil, "\330\246\331\205", nil, nil, nil],
      0xfc9b => [0, 0, nil, "\330\246\331\207", nil, nil, nil],
      0xfc9c => [0, 0, nil, "\330\250\330\254", nil, nil, nil],
      0xfc9d => [0, 0, nil, "\330\250\330\255", nil, nil, nil],
      0xfc9e => [0, 0, nil, "\330\250\330\256", nil, nil, nil],
      0xfc9f => [0, 0, nil, "\330\250\331\205", nil, nil, nil],
      0xfca0 => [0, 0, nil, "\330\250\331\207", nil, nil, nil],
      0xfca1 => [0, 0, nil, "\330\252\330\254", nil, nil, nil],
      0xfca2 => [0, 0, nil, "\330\252\330\255", nil, nil, nil],
      0xfca3 => [0, 0, nil, "\330\252\330\256", nil, nil, nil],
      0xfca4 => [0, 0, nil, "\330\252\331\205", nil, nil, nil],
      0xfca5 => [0, 0, nil, "\330\252\331\207", nil, nil, nil],
      0xfca6 => [0, 0, nil, "\330\253\331\205", nil, nil, nil],
      0xfca7 => [0, 0, nil, "\330\254\330\255", nil, nil, nil],
      0xfca8 => [0, 0, nil, "\330\254\331\205", nil, nil, nil],
      0xfca9 => [0, 0, nil, "\330\255\330\254", nil, nil, nil],
      0xfcaa => [0, 0, nil, "\330\255\331\205", nil, nil, nil],
      0xfcab => [0, 0, nil, "\330\256\330\254", nil, nil, nil],
      0xfcac => [0, 0, nil, "\330\256\331\205", nil, nil, nil],
      0xfcad => [0, 0, nil, "\330\263\330\254", nil, nil, nil],
      0xfcae => [0, 0, nil, "\330\263\330\255", nil, nil, nil],
      0xfcaf => [0, 0, nil, "\330\263\330\256", nil, nil, nil],
      0xfcb0 => [0, 0, nil, "\330\263\331\205", nil, nil, nil],
      0xfcb1 => [0, 0, nil, "\330\265\330\255", nil, nil, nil],
      0xfcb2 => [0, 0, nil, "\330\265\330\256", nil, nil, nil],
      0xfcb3 => [0, 0, nil, "\330\265\331\205", nil, nil, nil],
      0xfcb4 => [0, 0, nil, "\330\266\330\254", nil, nil, nil],
      0xfcb5 => [0, 0, nil, "\330\266\330\255", nil, nil, nil],
      0xfcb6 => [0, 0, nil, "\330\266\330\256", nil, nil, nil],
      0xfcb7 => [0, 0, nil, "\330\266\331\205", nil, nil, nil],
      0xfcb8 => [0, 0, nil, "\330\267\330\255", nil, nil, nil],
      0xfcb9 => [0, 0, nil, "\330\270\331\205", nil, nil, nil],
      0xfcba => [0, 0, nil, "\330\271\330\254", nil, nil, nil],
      0xfcbb => [0, 0, nil, "\330\271\331\205", nil, nil, nil],
      0xfcbc => [0, 0, nil, "\330\272\330\254", nil, nil, nil],
      0xfcbd => [0, 0, nil, "\330\272\331\205", nil, nil, nil],
      0xfcbe => [0, 0, nil, "\331\201\330\254", nil, nil, nil],
      0xfcbf => [0, 0, nil, "\331\201\330\255", nil, nil, nil],
      0xfcc0 => [0, 0, nil, "\331\201\330\256", nil, nil, nil],
      0xfcc1 => [0, 0, nil, "\331\201\331\205", nil, nil, nil],
      0xfcc2 => [0, 0, nil, "\331\202\330\255", nil, nil, nil],
      0xfcc3 => [0, 0, nil, "\331\202\331\205", nil, nil, nil],
      0xfcc4 => [0, 0, nil, "\331\203\330\254", nil, nil, nil],
      0xfcc5 => [0, 0, nil, "\331\203\330\255", nil, nil, nil],
      0xfcc6 => [0, 0, nil, "\331\203\330\256", nil, nil, nil],
      0xfcc7 => [0, 0, nil, "\331\203\331\204", nil, nil, nil],
      0xfcc8 => [0, 0, nil, "\331\203\331\205", nil, nil, nil],
      0xfcc9 => [0, 0, nil, "\331\204\330\254", nil, nil, nil],
      0xfcca => [0, 0, nil, "\331\204\330\255", nil, nil, nil],
      0xfccb => [0, 0, nil, "\331\204\330\256", nil, nil, nil],
      0xfccc => [0, 0, nil, "\331\204\331\205", nil, nil, nil],
      0xfccd => [0, 0, nil, "\331\204\331\207", nil, nil, nil],
      0xfcce => [0, 0, nil, "\331\205\330\254", nil, nil, nil],
      0xfccf => [0, 0, nil, "\331\205\330\255", nil, nil, nil],
      0xfcd0 => [0, 0, nil, "\331\205\330\256", nil, nil, nil],
      0xfcd1 => [0, 0, nil, "\331\205\331\205", nil, nil, nil],
      0xfcd2 => [0, 0, nil, "\331\206\330\254", nil, nil, nil],
      0xfcd3 => [0, 0, nil, "\331\206\330\255", nil, nil, nil],
      0xfcd4 => [0, 0, nil, "\331\206\330\256", nil, nil, nil],
      0xfcd5 => [0, 0, nil, "\331\206\331\205", nil, nil, nil],
      0xfcd6 => [0, 0, nil, "\331\206\331\207", nil, nil, nil],
      0xfcd7 => [0, 0, nil, "\331\207\330\254", nil, nil, nil],
      0xfcd8 => [0, 0, nil, "\331\207\331\205", nil, nil, nil],
      0xfcd9 => [0, 0, nil, "\331\207\331\260", nil, nil, nil],
      0xfcda => [0, 0, nil, "\331\212\330\254", nil, nil, nil],
      0xfcdb => [0, 0, nil, "\331\212\330\255", nil, nil, nil],
      0xfcdc => [0, 0, nil, "\331\212\330\256", nil, nil, nil],
      0xfcdd => [0, 0, nil, "\331\212\331\205", nil, nil, nil],
      0xfcde => [0, 0, nil, "\331\212\331\207", nil, nil, nil],
      0xfcdf => [0, 0, nil, "\330\246\331\205", nil, nil, nil],
      0xfce0 => [0, 0, nil, "\330\246\331\207", nil, nil, nil],
      0xfce1 => [0, 0, nil, "\330\250\331\205", nil, nil, nil],
      0xfce2 => [0, 0, nil, "\330\250\331\207", nil, nil, nil],
      0xfce3 => [0, 0, nil, "\330\252\331\205", nil, nil, nil],
      0xfce4 => [0, 0, nil, "\330\252\331\207", nil, nil, nil],
      0xfce5 => [0, 0, nil, "\330\253\331\205", nil, nil, nil],
      0xfce6 => [0, 0, nil, "\330\253\331\207", nil, nil, nil],
      0xfce7 => [0, 0, nil, "\330\263\331\205", nil, nil, nil],
      0xfce8 => [0, 0, nil, "\330\263\331\207", nil, nil, nil],
      0xfce9 => [0, 0, nil, "\330\264\331\205", nil, nil, nil],
      0xfcea => [0, 0, nil, "\330\264\331\207", nil, nil, nil],
      0xfceb => [0, 0, nil, "\331\203\331\204", nil, nil, nil],
      0xfcec => [0, 0, nil, "\331\203\331\205", nil, nil, nil],
      0xfced => [0, 0, nil, "\331\204\331\205", nil, nil, nil],
      0xfcee => [0, 0, nil, "\331\206\331\205", nil, nil, nil],
      0xfcef => [0, 0, nil, "\331\206\331\207", nil, nil, nil],
      0xfcf0 => [0, 0, nil, "\331\212\331\205", nil, nil, nil],
      0xfcf1 => [0, 0, nil, "\331\212\331\207", nil, nil, nil],
      0xfcf2 => [0, 0, nil, "\331\200\331\216\331\221", nil, nil, nil],
      0xfcf3 => [0, 0, nil, "\331\200\331\217\331\221", nil, nil, nil],
      0xfcf4 => [0, 0, nil, "\331\200\331\220\331\221", nil, nil, nil],
      0xfcf5 => [0, 0, nil, "\330\267\331\211", nil, nil, nil],
      0xfcf6 => [0, 0, nil, "\330\267\331\212", nil, nil, nil],
      0xfcf7 => [0, 0, nil, "\330\271\331\211", nil, nil, nil],
      0xfcf8 => [0, 0, nil, "\330\271\331\212", nil, nil, nil],
      0xfcf9 => [0, 0, nil, "\330\272\331\211", nil, nil, nil],
      0xfcfa => [0, 0, nil, "\330\272\331\212", nil, nil, nil],
      0xfcfb => [0, 0, nil, "\330\263\331\211", nil, nil, nil],
      0xfcfc => [0, 0, nil, "\330\263\331\212", nil, nil, nil],
      0xfcfd => [0, 0, nil, "\330\264\331\211", nil, nil, nil],
      0xfcfe => [0, 0, nil, "\330\264\331\212", nil, nil, nil],
      0xfcff => [0, 0, nil, "\330\255\331\211", nil, nil, nil],
      0xfd00 => [0, 0, nil, "\330\255\331\212", nil, nil, nil],
      0xfd01 => [0, 0, nil, "\330\254\331\211", nil, nil, nil],
      0xfd02 => [0, 0, nil, "\330\254\331\212", nil, nil, nil],
      0xfd03 => [0, 0, nil, "\330\256\331\211", nil, nil, nil],
      0xfd04 => [0, 0, nil, "\330\256\331\212", nil, nil, nil],
      0xfd05 => [0, 0, nil, "\330\265\331\211", nil, nil, nil],
      0xfd06 => [0, 0, nil, "\330\265\331\212", nil, nil, nil],
      0xfd07 => [0, 0, nil, "\330\266\331\211", nil, nil, nil],
      0xfd08 => [0, 0, nil, "\330\266\331\212", nil, nil, nil],
      0xfd09 => [0, 0, nil, "\330\264\330\254", nil, nil, nil],
      0xfd0a => [0, 0, nil, "\330\264\330\255", nil, nil, nil],
      0xfd0b => [0, 0, nil, "\330\264\330\256", nil, nil, nil],
      0xfd0c => [0, 0, nil, "\330\264\331\205", nil, nil, nil],
      0xfd0d => [0, 0, nil, "\330\264\330\261", nil, nil, nil],
      0xfd0e => [0, 0, nil, "\330\263\330\261", nil, nil, nil],
      0xfd0f => [0, 0, nil, "\330\265\330\261", nil, nil, nil],
      0xfd10 => [0, 0, nil, "\330\266\330\261", nil, nil, nil],
      0xfd11 => [0, 0, nil, "\330\267\331\211", nil, nil, nil],
      0xfd12 => [0, 0, nil, "\330\267\331\212", nil, nil, nil],
      0xfd13 => [0, 0, nil, "\330\271\331\211", nil, nil, nil],
      0xfd14 => [0, 0, nil, "\330\271\331\212", nil, nil, nil],
      0xfd15 => [0, 0, nil, "\330\272\331\211", nil, nil, nil],
      0xfd16 => [0, 0, nil, "\330\272\331\212", nil, nil, nil],
      0xfd17 => [0, 0, nil, "\330\263\331\211", nil, nil, nil],
      0xfd18 => [0, 0, nil, "\330\263\331\212", nil, nil, nil],
      0xfd19 => [0, 0, nil, "\330\264\331\211", nil, nil, nil],
      0xfd1a => [0, 0, nil, "\330\264\331\212", nil, nil, nil],
      0xfd1b => [0, 0, nil, "\330\255\331\211", nil, nil, nil],
      0xfd1c => [0, 0, nil, "\330\255\331\212", nil, nil, nil],
      0xfd1d => [0, 0, nil, "\330\254\331\211", nil, nil, nil],
      0xfd1e => [0, 0, nil, "\330\254\331\212", nil, nil, nil],
      0xfd1f => [0, 0, nil, "\330\256\331\211", nil, nil, nil],
      0xfd20 => [0, 0, nil, "\330\256\331\212", nil, nil, nil],
      0xfd21 => [0, 0, nil, "\330\265\331\211", nil, nil, nil],
      0xfd22 => [0, 0, nil, "\330\265\331\212", nil, nil, nil],
      0xfd23 => [0, 0, nil, "\330\266\331\211", nil, nil, nil],
      0xfd24 => [0, 0, nil, "\330\266\331\212", nil, nil, nil],
      0xfd25 => [0, 0, nil, "\330\264\330\254", nil, nil, nil],
      0xfd26 => [0, 0, nil, "\330\264\330\255", nil, nil, nil],
      0xfd27 => [0, 0, nil, "\330\264\330\256", nil, nil, nil],
      0xfd28 => [0, 0, nil, "\330\264\331\205", nil, nil, nil],
      0xfd29 => [0, 0, nil, "\330\264\330\261", nil, nil, nil],
      0xfd2a => [0, 0, nil, "\330\263\330\261", nil, nil, nil],
      0xfd2b => [0, 0, nil, "\330\265\330\261", nil, nil, nil],
      0xfd2c => [0, 0, nil, "\330\266\330\261", nil, nil, nil],
      0xfd2d => [0, 0, nil, "\330\264\330\254", nil, nil, nil],
      0xfd2e => [0, 0, nil, "\330\264\330\255", nil, nil, nil],
      0xfd2f => [0, 0, nil, "\330\264\330\256", nil, nil, nil],
      0xfd30 => [0, 0, nil, "\330\264\331\205", nil, nil, nil],
      0xfd31 => [0, 0, nil, "\330\263\331\207", nil, nil, nil],
      0xfd32 => [0, 0, nil, "\330\264\331\207", nil, nil, nil],
      0xfd33 => [0, 0, nil, "\330\267\331\205", nil, nil, nil],
      0xfd34 => [0, 0, nil, "\330\263\330\254", nil, nil, nil],
      0xfd35 => [0, 0, nil, "\330\263\330\255", nil, nil, nil],
      0xfd36 => [0, 0, nil, "\330\263\330\256", nil, nil, nil],
      0xfd37 => [0, 0, nil, "\330\264\330\254", nil, nil, nil],
      0xfd38 => [0, 0, nil, "\330\264\330\255", nil, nil, nil],
      0xfd39 => [0, 0, nil, "\330\264\330\256", nil, nil, nil],
      0xfd3a => [0, 0, nil, "\330\267\331\205", nil, nil, nil],
      0xfd3b => [0, 0, nil, "\330\270\331\205", nil, nil, nil],
      0xfd3c => [0, 0, nil, "\330\247\331\213", nil, nil, nil],
      0xfd3d => [0, 0, nil, "\330\247\331\213", nil, nil, nil],
      0xfd50 => [0, 0, nil, "\330\252\330\254\331\205", nil, nil, nil],
      0xfd51 => [0, 0, nil, "\330\252\330\255\330\254", nil, nil, nil],
      0xfd52 => [0, 0, nil, "\330\252\330\255\330\254", nil, nil, nil],
      0xfd53 => [0, 0, nil, "\330\252\330\255\331\205", nil, nil, nil],
      0xfd54 => [0, 0, nil, "\330\252\330\256\331\205", nil, nil, nil],
      0xfd55 => [0, 0, nil, "\330\252\331\205\330\254", nil, nil, nil],
      0xfd56 => [0, 0, nil, "\330\252\331\205\330\255", nil, nil, nil],
      0xfd57 => [0, 0, nil, "\330\252\331\205\330\256", nil, nil, nil],
      0xfd58 => [0, 0, nil, "\330\254\331\205\330\255", nil, nil, nil],
      0xfd59 => [0, 0, nil, "\330\254\331\205\330\255", nil, nil, nil],
      0xfd5a => [0, 0, nil, "\330\255\331\205\331\212", nil, nil, nil],
      0xfd5b => [0, 0, nil, "\330\255\331\205\331\211", nil, nil, nil],
      0xfd5c => [0, 0, nil, "\330\263\330\255\330\254", nil, nil, nil],
      0xfd5d => [0, 0, nil, "\330\263\330\254\330\255", nil, nil, nil],
      0xfd5e => [0, 0, nil, "\330\263\330\254\331\211", nil, nil, nil],
      0xfd5f => [0, 0, nil, "\330\263\331\205\330\255", nil, nil, nil],
      0xfd60 => [0, 0, nil, "\330\263\331\205\330\255", nil, nil, nil],
      0xfd61 => [0, 0, nil, "\330\263\331\205\330\254", nil, nil, nil],
      0xfd62 => [0, 0, nil, "\330\263\331\205\331\205", nil, nil, nil],
      0xfd63 => [0, 0, nil, "\330\263\331\205\331\205", nil, nil, nil],
      0xfd64 => [0, 0, nil, "\330\265\330\255\330\255", nil, nil, nil],
      0xfd65 => [0, 0, nil, "\330\265\330\255\330\255", nil, nil, nil],
      0xfd66 => [0, 0, nil, "\330\265\331\205\331\205", nil, nil, nil],
      0xfd67 => [0, 0, nil, "\330\264\330\255\331\205", nil, nil, nil],
      0xfd68 => [0, 0, nil, "\330\264\330\255\331\205", nil, nil, nil],
      0xfd69 => [0, 0, nil, "\330\264\330\254\331\212", nil, nil, nil],
      0xfd6a => [0, 0, nil, "\330\264\331\205\330\256", nil, nil, nil],
      0xfd6b => [0, 0, nil, "\330\264\331\205\330\256", nil, nil, nil],
      0xfd6c => [0, 0, nil, "\330\264\331\205\331\205", nil, nil, nil],
      0xfd6d => [0, 0, nil, "\330\264\331\205\331\205", nil, nil, nil],
      0xfd6e => [0, 0, nil, "\330\266\330\255\331\211", nil, nil, nil],
      0xfd6f => [0, 0, nil, "\330\266\330\256\331\205", nil, nil, nil],
      0xfd70 => [0, 0, nil, "\330\266\330\256\331\205", nil, nil, nil],
      0xfd71 => [0, 0, nil, "\330\267\331\205\330\255", nil, nil, nil],
      0xfd72 => [0, 0, nil, "\330\267\331\205\330\255", nil, nil, nil],
      0xfd73 => [0, 0, nil, "\330\267\331\205\331\205", nil, nil, nil],
      0xfd74 => [0, 0, nil, "\330\267\331\205\331\212", nil, nil, nil],
      0xfd75 => [0, 0, nil, "\330\271\330\254\331\205", nil, nil, nil],
      0xfd76 => [0, 0, nil, "\330\271\331\205\331\205", nil, nil, nil],
      0xfd77 => [0, 0, nil, "\330\271\331\205\331\205", nil, nil, nil],
      0xfd78 => [0, 0, nil, "\330\271\331\205\331\211", nil, nil, nil],
      0xfd79 => [0, 0, nil, "\330\272\331\205\331\205", nil, nil, nil],
      0xfd7a => [0, 0, nil, "\330\272\331\205\331\212", nil, nil, nil],
      0xfd7b => [0, 0, nil, "\330\272\331\205\331\211", nil, nil, nil],
      0xfd7c => [0, 0, nil, "\331\201\330\256\331\205", nil, nil, nil],
      0xfd7d => [0, 0, nil, "\331\201\330\256\331\205", nil, nil, nil],
      0xfd7e => [0, 0, nil, "\331\202\331\205\330\255", nil, nil, nil],
      0xfd7f => [0, 0, nil, "\331\202\331\205\331\205", nil, nil, nil],
      0xfd80 => [0, 0, nil, "\331\204\330\255\331\205", nil, nil, nil],
      0xfd81 => [0, 0, nil, "\331\204\330\255\331\212", nil, nil, nil],
      0xfd82 => [0, 0, nil, "\331\204\330\255\331\211", nil, nil, nil],
      0xfd83 => [0, 0, nil, "\331\204\330\254\330\254", nil, nil, nil],
      0xfd84 => [0, 0, nil, "\331\204\330\254\330\254", nil, nil, nil],
      0xfd85 => [0, 0, nil, "\331\204\330\256\331\205", nil, nil, nil],
      0xfd86 => [0, 0, nil, "\331\204\330\256\331\205", nil, nil, nil],
      0xfd87 => [0, 0, nil, "\331\204\331\205\330\255", nil, nil, nil],
      0xfd88 => [0, 0, nil, "\331\204\331\205\330\255", nil, nil, nil],
      0xfd89 => [0, 0, nil, "\331\205\330\255\330\254", nil, nil, nil],
      0xfd8a => [0, 0, nil, "\331\205\330\255\331\205", nil, nil, nil],
      0xfd8b => [0, 0, nil, "\331\205\330\255\331\212", nil, nil, nil],
      0xfd8c => [0, 0, nil, "\331\205\330\254\330\255", nil, nil, nil],
      0xfd8d => [0, 0, nil, "\331\205\330\254\331\205", nil, nil, nil],
      0xfd8e => [0, 0, nil, "\331\205\330\256\330\254", nil, nil, nil],
      0xfd8f => [0, 0, nil, "\331\205\330\256\331\205", nil, nil, nil],
      0xfd92 => [0, 0, nil, "\331\205\330\254\330\256", nil, nil, nil],
      0xfd93 => [0, 0, nil, "\331\207\331\205\330\254", nil, nil, nil],
      0xfd94 => [0, 0, nil, "\331\207\331\205\331\205", nil, nil, nil],
      0xfd95 => [0, 0, nil, "\331\206\330\255\331\205", nil, nil, nil],
      0xfd96 => [0, 0, nil, "\331\206\330\255\331\211", nil, nil, nil],
      0xfd97 => [0, 0, nil, "\331\206\330\254\331\205", nil, nil, nil],
      0xfd98 => [0, 0, nil, "\331\206\330\254\331\205", nil, nil, nil],
      0xfd99 => [0, 0, nil, "\331\206\330\254\331\211", nil, nil, nil],
      0xfd9a => [0, 0, nil, "\331\206\331\205\331\212", nil, nil, nil],
      0xfd9b => [0, 0, nil, "\331\206\331\205\331\211", nil, nil, nil],
      0xfd9c => [0, 0, nil, "\331\212\331\205\331\205", nil, nil, nil],
      0xfd9d => [0, 0, nil, "\331\212\331\205\331\205", nil, nil, nil],
      0xfd9e => [0, 0, nil, "\330\250\330\256\331\212", nil, nil, nil],
      0xfd9f => [0, 0, nil, "\330\252\330\254\331\212", nil, nil, nil],
      0xfda0 => [0, 0, nil, "\330\252\330\254\331\211", nil, nil, nil],
      0xfda1 => [0, 0, nil, "\330\252\330\256\331\212", nil, nil, nil],
      0xfda2 => [0, 0, nil, "\330\252\330\256\331\211", nil, nil, nil],
      0xfda3 => [0, 0, nil, "\330\252\331\205\331\212", nil, nil, nil],
      0xfda4 => [0, 0, nil, "\330\252\331\205\331\211", nil, nil, nil],
      0xfda5 => [0, 0, nil, "\330\254\331\205\331\212", nil, nil, nil],
      0xfda6 => [0, 0, nil, "\330\254\330\255\331\211", nil, nil, nil],
      0xfda7 => [0, 0, nil, "\330\254\331\205\331\211", nil, nil, nil],
      0xfda8 => [0, 0, nil, "\330\263\330\256\331\211", nil, nil, nil],
      0xfda9 => [0, 0, nil, "\330\265\330\255\331\212", nil, nil, nil],
      0xfdaa => [0, 0, nil, "\330\264\330\255\331\212", nil, nil, nil],
      0xfdab => [0, 0, nil, "\330\266\330\255\331\212", nil, nil, nil],
      0xfdac => [0, 0, nil, "\331\204\330\254\331\212", nil, nil, nil],
      0xfdad => [0, 0, nil, "\331\204\331\205\331\212", nil, nil, nil],
      0xfdae => [0, 0, nil, "\331\212\330\255\331\212", nil, nil, nil],
      0xfdaf => [0, 0, nil, "\331\212\330\254\331\212", nil, nil, nil],
      0xfdb0 => [0, 0, nil, "\331\212\331\205\331\212", nil, nil, nil],
      0xfdb1 => [0, 0, nil, "\331\205\331\205\331\212", nil, nil, nil],
      0xfdb2 => [0, 0, nil, "\331\202\331\205\331\212", nil, nil, nil],
      0xfdb3 => [0, 0, nil, "\331\206\330\255\331\212", nil, nil, nil],
      0xfdb4 => [0, 0, nil, "\331\202\331\205\330\255", nil, nil, nil],
      0xfdb5 => [0, 0, nil, "\331\204\330\255\331\205", nil, nil, nil],
      0xfdb6 => [0, 0, nil, "\330\271\331\205\331\212", nil, nil, nil],
      0xfdb7 => [0, 0, nil, "\331\203\331\205\331\212", nil, nil, nil],
      0xfdb8 => [0, 0, nil, "\331\206\330\254\330\255", nil, nil, nil],
      0xfdb9 => [0, 0, nil, "\331\205\330\256\331\212", nil, nil, nil],
      0xfdba => [0, 0, nil, "\331\204\330\254\331\205", nil, nil, nil],
      0xfdbb => [0, 0, nil, "\331\203\331\205\331\205", nil, nil, nil],
      0xfdbc => [0, 0, nil, "\331\204\330\254\331\205", nil, nil, nil],
      0xfdbd => [0, 0, nil, "\331\206\330\254\330\255", nil, nil, nil],
      0xfdbe => [0, 0, nil, "\330\254\330\255\331\212", nil, nil, nil],
      0xfdbf => [0, 0, nil, "\330\255\330\254\331\212", nil, nil, nil],
      0xfdc0 => [0, 0, nil, "\331\205\330\254\331\212", nil, nil, nil],
      0xfdc1 => [0, 0, nil, "\331\201\331\205\331\212", nil, nil, nil],
      0xfdc2 => [0, 0, nil, "\330\250\330\255\331\212", nil, nil, nil],
      0xfdc3 => [0, 0, nil, "\331\203\331\205\331\205", nil, nil, nil],
      0xfdc4 => [0, 0, nil, "\330\271\330\254\331\205", nil, nil, nil],
      0xfdc5 => [0, 0, nil, "\330\265\331\205\331\205", nil, nil, nil],
      0xfdc6 => [0, 0, nil, "\330\263\330\256\331\212", nil, nil, nil],
      0xfdc7 => [0, 0, nil, "\331\206\330\254\331\212", nil, nil, nil],
      0xfdf0 => [0, 0, nil, "\330\265\331\204\333\222", nil, nil, nil],
      0xfdf1 => [0, 0, nil, "\331\202\331\204\333\222", nil, nil, nil],
      0xfdf2 => [0, 0, nil, "\330\247\331\204\331\204\331\207", nil, nil, nil],
      0xfdf3 => [0, 0, nil, "\330\247\331\203\330\250\330\261", nil, nil, nil],
      0xfdf4 => [0, 0, nil, "\331\205\330\255\331\205\330\257", nil, nil, nil],
      0xfdf5 => [0, 0, nil, "\330\265\331\204\330\271\331\205", nil, nil, nil],
      0xfdf6 => [0, 0, nil, "\330\261\330\263\331\210\331\204", nil, nil, nil],
      0xfdf7 => [0, 0, nil, "\330\271\331\204\331\212\331\207", nil, nil, nil],
      0xfdf8 => [0, 0, nil, "\331\210\330\263\331\204\331\205", nil, nil, nil],
      0xfdf9 => [0, 0, nil, "\330\265\331\204\331\211", nil, nil, nil],
      0xfdfa => [0, 0, nil, "\330\265\331\204\331\211 \330\247\331\204\331\204\331\207 \330\271\331\204\331\212\331\207 \331\210\330\263\331\204\331\205", nil, nil, nil],
      0xfdfb => [0, 0, nil, "\330\254\331\204 \330\254\331\204\330\247\331\204\331\207", nil, nil, nil],
      0xfe20 => [230, 0, nil, nil, nil, nil, nil],
      0xfe21 => [230, 0, nil, nil, nil, nil, nil],
      0xfe22 => [230, 0, nil, nil, nil, nil, nil],
      0xfe23 => [230, 0, nil, nil, nil, nil, nil],
      0xfe30 => [0, 0, nil, "\342\200\245", nil, nil, nil],
      0xfe31 => [0, 0, nil, "\342\200\224", nil, nil, nil],
      0xfe32 => [0, 0, nil, "\342\200\223", nil, nil, nil],
      0xfe33 => [0, 0, nil, "_", nil, nil, nil],
      0xfe34 => [0, 0, nil, "_", nil, nil, nil],
      0xfe35 => [0, 0, nil, "(", nil, nil, nil],
      0xfe36 => [0, 0, nil, ")", nil, nil, nil],
      0xfe37 => [0, 0, nil, "{", nil, nil, nil],
      0xfe38 => [0, 0, nil, "}", nil, nil, nil],
      0xfe39 => [0, 0, nil, "\343\200\224", nil, nil, nil],
      0xfe3a => [0, 0, nil, "\343\200\225", nil, nil, nil],
      0xfe3b => [0, 0, nil, "\343\200\220", nil, nil, nil],
      0xfe3c => [0, 0, nil, "\343\200\221", nil, nil, nil],
      0xfe3d => [0, 0, nil, "\343\200\212", nil, nil, nil],
      0xfe3e => [0, 0, nil, "\343\200\213", nil, nil, nil],
      0xfe3f => [0, 0, nil, "\343\200\210", nil, nil, nil],
      0xfe40 => [0, 0, nil, "\343\200\211", nil, nil, nil],
      0xfe41 => [0, 0, nil, "\343\200\214", nil, nil, nil],
      0xfe42 => [0, 0, nil, "\343\200\215", nil, nil, nil],
      0xfe43 => [0, 0, nil, "\343\200\216", nil, nil, nil],
      0xfe44 => [0, 0, nil, "\343\200\217", nil, nil, nil],
      0xfe49 => [0, 0, nil, "\342\200\276", nil, nil, nil],
      0xfe4a => [0, 0, nil, "\342\200\276", nil, nil, nil],
      0xfe4b => [0, 0, nil, "\342\200\276", nil, nil, nil],
      0xfe4c => [0, 0, nil, "\342\200\276", nil, nil, nil],
      0xfe4d => [0, 0, nil, "_", nil, nil, nil],
      0xfe4e => [0, 0, nil, "_", nil, nil, nil],
      0xfe4f => [0, 0, nil, "_", nil, nil, nil],
      0xfe50 => [0, 0, nil, ",", nil, nil, nil],
      0xfe51 => [0, 0, nil, "\343\200\201", nil, nil, nil],
      0xfe52 => [0, 0, nil, ".", nil, nil, nil],
      0xfe54 => [0, 0, nil, ";", nil, nil, nil],
      0xfe55 => [0, 0, nil, ":", nil, nil, nil],
      0xfe56 => [0, 0, nil, "?", nil, nil, nil],
      0xfe57 => [0, 0, nil, "!", nil, nil, nil],
      0xfe58 => [0, 0, nil, "\342\200\224", nil, nil, nil],
      0xfe59 => [0, 0, nil, "(", nil, nil, nil],
      0xfe5a => [0, 0, nil, ")", nil, nil, nil],
      0xfe5b => [0, 0, nil, "{", nil, nil, nil],
      0xfe5c => [0, 0, nil, "}", nil, nil, nil],
      0xfe5d => [0, 0, nil, "\343\200\224", nil, nil, nil],
      0xfe5e => [0, 0, nil, "\343\200\225", nil, nil, nil],
      0xfe5f => [0, 0, nil, "#", nil, nil, nil],
      0xfe60 => [0, 0, nil, "&", nil, nil, nil],
      0xfe61 => [0, 0, nil, "*", nil, nil, nil],
      0xfe62 => [0, 0, nil, "+", nil, nil, nil],
      0xfe63 => [0, 0, nil, "-", nil, nil, nil],
      0xfe64 => [0, 0, nil, "<", nil, nil, nil],
      0xfe65 => [0, 0, nil, ">", nil, nil, nil],
      0xfe66 => [0, 0, nil, "=", nil, nil, nil],
      0xfe68 => [0, 0, nil, "\134", nil, nil, nil],
      0xfe69 => [0, 0, nil, "$", nil, nil, nil],
      0xfe6a => [0, 0, nil, "%", nil, nil, nil],
      0xfe6b => [0, 0, nil, "@", nil, nil, nil],
      0xfe70 => [0, 0, nil, " \331\213", nil, nil, nil],
      0xfe71 => [0, 0, nil, "\331\200\331\213", nil, nil, nil],
      0xfe72 => [0, 0, nil, " \331\214", nil, nil, nil],
      0xfe74 => [0, 0, nil, " \331\215", nil, nil, nil],
      0xfe76 => [0, 0, nil, " \331\216", nil, nil, nil],
      0xfe77 => [0, 0, nil, "\331\200\331\216", nil, nil, nil],
      0xfe78 => [0, 0, nil, " \331\217", nil, nil, nil],
      0xfe79 => [0, 0, nil, "\331\200\331\217", nil, nil, nil],
      0xfe7a => [0, 0, nil, " \331\220", nil, nil, nil],
      0xfe7b => [0, 0, nil, "\331\200\331\220", nil, nil, nil],
      0xfe7c => [0, 0, nil, " \331\221", nil, nil, nil],
      0xfe7d => [0, 0, nil, "\331\200\331\221", nil, nil, nil],
      0xfe7e => [0, 0, nil, " \331\222", nil, nil, nil],
      0xfe7f => [0, 0, nil, "\331\200\331\222", nil, nil, nil],
      0xfe80 => [0, 0, nil, "\330\241", nil, nil, nil],
      0xfe81 => [0, 0, nil, "\330\242", nil, nil, nil],
      0xfe82 => [0, 0, nil, "\330\242", nil, nil, nil],
      0xfe83 => [0, 0, nil, "\330\243", nil, nil, nil],
      0xfe84 => [0, 0, nil, "\330\243", nil, nil, nil],
      0xfe85 => [0, 0, nil, "\330\244", nil, nil, nil],
      0xfe86 => [0, 0, nil, "\330\244", nil, nil, nil],
      0xfe87 => [0, 0, nil, "\330\245", nil, nil, nil],
      0xfe88 => [0, 0, nil, "\330\245", nil, nil, nil],
      0xfe89 => [0, 0, nil, "\330\246", nil, nil, nil],
      0xfe8a => [0, 0, nil, "\330\246", nil, nil, nil],
      0xfe8b => [0, 0, nil, "\330\246", nil, nil, nil],
      0xfe8c => [0, 0, nil, "\330\246", nil, nil, nil],
      0xfe8d => [0, 0, nil, "\330\247", nil, nil, nil],
      0xfe8e => [0, 0, nil, "\330\247", nil, nil, nil],
      0xfe8f => [0, 0, nil, "\330\250", nil, nil, nil],
      0xfe90 => [0, 0, nil, "\330\250", nil, nil, nil],
      0xfe91 => [0, 0, nil, "\330\250", nil, nil, nil],
      0xfe92 => [0, 0, nil, "\330\250", nil, nil, nil],
      0xfe93 => [0, 0, nil, "\330\251", nil, nil, nil],
      0xfe94 => [0, 0, nil, "\330\251", nil, nil, nil],
      0xfe95 => [0, 0, nil, "\330\252", nil, nil, nil],
      0xfe96 => [0, 0, nil, "\330\252", nil, nil, nil],
      0xfe97 => [0, 0, nil, "\330\252", nil, nil, nil],
      0xfe98 => [0, 0, nil, "\330\252", nil, nil, nil],
      0xfe99 => [0, 0, nil, "\330\253", nil, nil, nil],
      0xfe9a => [0, 0, nil, "\330\253", nil, nil, nil],
      0xfe9b => [0, 0, nil, "\330\253", nil, nil, nil],
      0xfe9c => [0, 0, nil, "\330\253", nil, nil, nil],
      0xfe9d => [0, 0, nil, "\330\254", nil, nil, nil],
      0xfe9e => [0, 0, nil, "\330\254", nil, nil, nil],
      0xfe9f => [0, 0, nil, "\330\254", nil, nil, nil],
      0xfea0 => [0, 0, nil, "\330\254", nil, nil, nil],
      0xfea1 => [0, 0, nil, "\330\255", nil, nil, nil],
      0xfea2 => [0, 0, nil, "\330\255", nil, nil, nil],
      0xfea3 => [0, 0, nil, "\330\255", nil, nil, nil],
      0xfea4 => [0, 0, nil, "\330\255", nil, nil, nil],
      0xfea5 => [0, 0, nil, "\330\256", nil, nil, nil],
      0xfea6 => [0, 0, nil, "\330\256", nil, nil, nil],
      0xfea7 => [0, 0, nil, "\330\256", nil, nil, nil],
      0xfea8 => [0, 0, nil, "\330\256", nil, nil, nil],
      0xfea9 => [0, 0, nil, "\330\257", nil, nil, nil],
      0xfeaa => [0, 0, nil, "\330\257", nil, nil, nil],
      0xfeab => [0, 0, nil, "\330\260", nil, nil, nil],
      0xfeac => [0, 0, nil, "\330\260", nil, nil, nil],
      0xfead => [0, 0, nil, "\330\261", nil, nil, nil],
      0xfeae => [0, 0, nil, "\330\261", nil, nil, nil],
      0xfeaf => [0, 0, nil, "\330\262", nil, nil, nil],
      0xfeb0 => [0, 0, nil, "\330\262", nil, nil, nil],
      0xfeb1 => [0, 0, nil, "\330\263", nil, nil, nil],
      0xfeb2 => [0, 0, nil, "\330\263", nil, nil, nil],
      0xfeb3 => [0, 0, nil, "\330\263", nil, nil, nil],
      0xfeb4 => [0, 0, nil, "\330\263", nil, nil, nil],
      0xfeb5 => [0, 0, nil, "\330\264", nil, nil, nil],
      0xfeb6 => [0, 0, nil, "\330\264", nil, nil, nil],
      0xfeb7 => [0, 0, nil, "\330\264", nil, nil, nil],
      0xfeb8 => [0, 0, nil, "\330\264", nil, nil, nil],
      0xfeb9 => [0, 0, nil, "\330\265", nil, nil, nil],
      0xfeba => [0, 0, nil, "\330\265", nil, nil, nil],
      0xfebb => [0, 0, nil, "\330\265", nil, nil, nil],
      0xfebc => [0, 0, nil, "\330\265", nil, nil, nil],
      0xfebd => [0, 0, nil, "\330\266", nil, nil, nil],
      0xfebe => [0, 0, nil, "\330\266", nil, nil, nil],
      0xfebf => [0, 0, nil, "\330\266", nil, nil, nil],
      0xfec0 => [0, 0, nil, "\330\266", nil, nil, nil],
      0xfec1 => [0, 0, nil, "\330\267", nil, nil, nil],
      0xfec2 => [0, 0, nil, "\330\267", nil, nil, nil],
      0xfec3 => [0, 0, nil, "\330\267", nil, nil, nil],
      0xfec4 => [0, 0, nil, "\330\267", nil, nil, nil],
      0xfec5 => [0, 0, nil, "\330\270", nil, nil, nil],
      0xfec6 => [0, 0, nil, "\330\270", nil, nil, nil],
      0xfec7 => [0, 0, nil, "\330\270", nil, nil, nil],
      0xfec8 => [0, 0, nil, "\330\270", nil, nil, nil],
      0xfec9 => [0, 0, nil, "\330\271", nil, nil, nil],
      0xfeca => [0, 0, nil, "\330\271", nil, nil, nil],
      0xfecb => [0, 0, nil, "\330\271", nil, nil, nil],
      0xfecc => [0, 0, nil, "\330\271", nil, nil, nil],
      0xfecd => [0, 0, nil, "\330\272", nil, nil, nil],
      0xfece => [0, 0, nil, "\330\272", nil, nil, nil],
      0xfecf => [0, 0, nil, "\330\272", nil, nil, nil],
      0xfed0 => [0, 0, nil, "\330\272", nil, nil, nil],
      0xfed1 => [0, 0, nil, "\331\201", nil, nil, nil],
      0xfed2 => [0, 0, nil, "\331\201", nil, nil, nil],
      0xfed3 => [0, 0, nil, "\331\201", nil, nil, nil],
      0xfed4 => [0, 0, nil, "\331\201", nil, nil, nil],
      0xfed5 => [0, 0, nil, "\331\202", nil, nil, nil],
      0xfed6 => [0, 0, nil, "\331\202", nil, nil, nil],
      0xfed7 => [0, 0, nil, "\331\202", nil, nil, nil],
      0xfed8 => [0, 0, nil, "\331\202", nil, nil, nil],
      0xfed9 => [0, 0, nil, "\331\203", nil, nil, nil],
      0xfeda => [0, 0, nil, "\331\203", nil, nil, nil],
      0xfedb => [0, 0, nil, "\331\203", nil, nil, nil],
      0xfedc => [0, 0, nil, "\331\203", nil, nil, nil],
      0xfedd => [0, 0, nil, "\331\204", nil, nil, nil],
      0xfede => [0, 0, nil, "\331\204", nil, nil, nil],
      0xfedf => [0, 0, nil, "\331\204", nil, nil, nil],
      0xfee0 => [0, 0, nil, "\331\204", nil, nil, nil],
      0xfee1 => [0, 0, nil, "\331\205", nil, nil, nil],
      0xfee2 => [0, 0, nil, "\331\205", nil, nil, nil],
      0xfee3 => [0, 0, nil, "\331\205", nil, nil, nil],
      0xfee4 => [0, 0, nil, "\331\205", nil, nil, nil],
      0xfee5 => [0, 0, nil, "\331\206", nil, nil, nil],
      0xfee6 => [0, 0, nil, "\331\206", nil, nil, nil],
      0xfee7 => [0, 0, nil, "\331\206", nil, nil, nil],
      0xfee8 => [0, 0, nil, "\331\206", nil, nil, nil],
      0xfee9 => [0, 0, nil, "\331\207", nil, nil, nil],
      0xfeea => [0, 0, nil, "\331\207", nil, nil, nil],
      0xfeeb => [0, 0, nil, "\331\207", nil, nil, nil],
      0xfeec => [0, 0, nil, "\331\207", nil, nil, nil],
      0xfeed => [0, 0, nil, "\331\210", nil, nil, nil],
      0xfeee => [0, 0, nil, "\331\210", nil, nil, nil],
      0xfeef => [0, 0, nil, "\331\211", nil, nil, nil],
      0xfef0 => [0, 0, nil, "\331\211", nil, nil, nil],
      0xfef1 => [0, 0, nil, "\331\212", nil, nil, nil],
      0xfef2 => [0, 0, nil, "\331\212", nil, nil, nil],
      0xfef3 => [0, 0, nil, "\331\212", nil, nil, nil],
      0xfef4 => [0, 0, nil, "\331\212", nil, nil, nil],
      0xfef5 => [0, 0, nil, "\331\204\330\242", nil, nil, nil],
      0xfef6 => [0, 0, nil, "\331\204\330\242", nil, nil, nil],
      0xfef7 => [0, 0, nil, "\331\204\330\243", nil, nil, nil],
      0xfef8 => [0, 0, nil, "\331\204\330\243", nil, nil, nil],
      0xfef9 => [0, 0, nil, "\331\204\330\245", nil, nil, nil],
      0xfefa => [0, 0, nil, "\331\204\330\245", nil, nil, nil],
      0xfefb => [0, 0, nil, "\331\204\330\247", nil, nil, nil],
      0xfefc => [0, 0, nil, "\331\204\330\247", nil, nil, nil],
      0xff01 => [0, 0, nil, "!", nil, nil, nil],
      0xff02 => [0, 0, nil, "\042", nil, nil, nil],
      0xff03 => [0, 0, nil, "#", nil, nil, nil],
      0xff04 => [0, 0, nil, "$", nil, nil, nil],
      0xff05 => [0, 0, nil, "%", nil, nil, nil],
      0xff06 => [0, 0, nil, "&", nil, nil, nil],
      0xff07 => [0, 0, nil, "'", nil, nil, nil],
      0xff08 => [0, 0, nil, "(", nil, nil, nil],
      0xff09 => [0, 0, nil, ")", nil, nil, nil],
      0xff0a => [0, 0, nil, "*", nil, nil, nil],
      0xff0b => [0, 0, nil, "+", nil, nil, nil],
      0xff0c => [0, 0, nil, ",", nil, nil, nil],
      0xff0d => [0, 0, nil, "-", nil, nil, nil],
      0xff0e => [0, 0, nil, ".", nil, nil, nil],
      0xff0f => [0, 0, nil, "/", nil, nil, nil],
      0xff10 => [0, 0, nil, "0", nil, nil, nil],
      0xff11 => [0, 0, nil, "1", nil, nil, nil],
      0xff12 => [0, 0, nil, "2", nil, nil, nil],
      0xff13 => [0, 0, nil, "3", nil, nil, nil],
      0xff14 => [0, 0, nil, "4", nil, nil, nil],
      0xff15 => [0, 0, nil, "5", nil, nil, nil],
      0xff16 => [0, 0, nil, "6", nil, nil, nil],
      0xff17 => [0, 0, nil, "7", nil, nil, nil],
      0xff18 => [0, 0, nil, "8", nil, nil, nil],
      0xff19 => [0, 0, nil, "9", nil, nil, nil],
      0xff1a => [0, 0, nil, ":", nil, nil, nil],
      0xff1b => [0, 0, nil, ";", nil, nil, nil],
      0xff1c => [0, 0, nil, "<", nil, nil, nil],
      0xff1d => [0, 0, nil, "=", nil, nil, nil],
      0xff1e => [0, 0, nil, ">", nil, nil, nil],
      0xff1f => [0, 0, nil, "?", nil, nil, nil],
      0xff20 => [0, 0, nil, "@", nil, nil, nil],
      0xff21 => [0, 0, nil, "A", nil, 0xff41, nil],
      0xff22 => [0, 0, nil, "B", nil, 0xff42, nil],
      0xff23 => [0, 0, nil, "C", nil, 0xff43, nil],
      0xff24 => [0, 0, nil, "D", nil, 0xff44, nil],
      0xff25 => [0, 0, nil, "E", nil, 0xff45, nil],
      0xff26 => [0, 0, nil, "F", nil, 0xff46, nil],
      0xff27 => [0, 0, nil, "G", nil, 0xff47, nil],
      0xff28 => [0, 0, nil, "H", nil, 0xff48, nil],
      0xff29 => [0, 0, nil, "I", nil, 0xff49, nil],
      0xff2a => [0, 0, nil, "J", nil, 0xff4a, nil],
      0xff2b => [0, 0, nil, "K", nil, 0xff4b, nil],
      0xff2c => [0, 0, nil, "L", nil, 0xff4c, nil],
      0xff2d => [0, 0, nil, "M", nil, 0xff4d, nil],
      0xff2e => [0, 0, nil, "N", nil, 0xff4e, nil],
      0xff2f => [0, 0, nil, "O", nil, 0xff4f, nil],
      0xff30 => [0, 0, nil, "P", nil, 0xff50, nil],
      0xff31 => [0, 0, nil, "Q", nil, 0xff51, nil],
      0xff32 => [0, 0, nil, "R", nil, 0xff52, nil],
      0xff33 => [0, 0, nil, "S", nil, 0xff53, nil],
      0xff34 => [0, 0, nil, "T", nil, 0xff54, nil],
      0xff35 => [0, 0, nil, "U", nil, 0xff55, nil],
      0xff36 => [0, 0, nil, "V", nil, 0xff56, nil],
      0xff37 => [0, 0, nil, "W", nil, 0xff57, nil],
      0xff38 => [0, 0, nil, "X", nil, 0xff58, nil],
      0xff39 => [0, 0, nil, "Y", nil, 0xff59, nil],
      0xff3a => [0, 0, nil, "Z", nil, 0xff5a, nil],
      0xff3b => [0, 0, nil, "[", nil, nil, nil],
      0xff3c => [0, 0, nil, "\134", nil, nil, nil],
      0xff3d => [0, 0, nil, "]", nil, nil, nil],
      0xff3e => [0, 0, nil, "^", nil, nil, nil],
      0xff3f => [0, 0, nil, "_", nil, nil, nil],
      0xff40 => [0, 0, nil, "`", nil, nil, nil],
      0xff41 => [0, 0, nil, "a", 0xff21, nil, 0xff21],
      0xff42 => [0, 0, nil, "b", 0xff22, nil, 0xff22],
      0xff43 => [0, 0, nil, "c", 0xff23, nil, 0xff23],
      0xff44 => [0, 0, nil, "d", 0xff24, nil, 0xff24],
      0xff45 => [0, 0, nil, "e", 0xff25, nil, 0xff25],
      0xff46 => [0, 0, nil, "f", 0xff26, nil, 0xff26],
      0xff47 => [0, 0, nil, "g", 0xff27, nil, 0xff27],
      0xff48 => [0, 0, nil, "h", 0xff28, nil, 0xff28],
      0xff49 => [0, 0, nil, "i", 0xff29, nil, 0xff29],
      0xff4a => [0, 0, nil, "j", 0xff2a, nil, 0xff2a],
      0xff4b => [0, 0, nil, "k", 0xff2b, nil, 0xff2b],
      0xff4c => [0, 0, nil, "l", 0xff2c, nil, 0xff2c],
      0xff4d => [0, 0, nil, "m", 0xff2d, nil, 0xff2d],
      0xff4e => [0, 0, nil, "n", 0xff2e, nil, 0xff2e],
      0xff4f => [0, 0, nil, "o", 0xff2f, nil, 0xff2f],
      0xff50 => [0, 0, nil, "p", 0xff30, nil, 0xff30],
      0xff51 => [0, 0, nil, "q", 0xff31, nil, 0xff31],
      0xff52 => [0, 0, nil, "r", 0xff32, nil, 0xff32],
      0xff53 => [0, 0, nil, "s", 0xff33, nil, 0xff33],
      0xff54 => [0, 0, nil, "t", 0xff34, nil, 0xff34],
      0xff55 => [0, 0, nil, "u", 0xff35, nil, 0xff35],
      0xff56 => [0, 0, nil, "v", 0xff36, nil, 0xff36],
      0xff57 => [0, 0, nil, "w", 0xff37, nil, 0xff37],
      0xff58 => [0, 0, nil, "x", 0xff38, nil, 0xff38],
      0xff59 => [0, 0, nil, "y", 0xff39, nil, 0xff39],
      0xff5a => [0, 0, nil, "z", 0xff3a, nil, 0xff3a],
      0xff5b => [0, 0, nil, "{", nil, nil, nil],
      0xff5c => [0, 0, nil, "|", nil, nil, nil],
      0xff5d => [0, 0, nil, "}", nil, nil, nil],
      0xff5e => [0, 0, nil, "~", nil, nil, nil],
      0xff61 => [0, 0, nil, "\343\200\202", nil, nil, nil],
      0xff62 => [0, 0, nil, "\343\200\214", nil, nil, nil],
      0xff63 => [0, 0, nil, "\343\200\215", nil, nil, nil],
      0xff64 => [0, 0, nil, "\343\200\201", nil, nil, nil],
      0xff65 => [0, 0, nil, "\343\203\273", nil, nil, nil],
      0xff66 => [0, 0, nil, "\343\203\262", nil, nil, nil],
      0xff67 => [0, 0, nil, "\343\202\241", nil, nil, nil],
      0xff68 => [0, 0, nil, "\343\202\243", nil, nil, nil],
      0xff69 => [0, 0, nil, "\343\202\245", nil, nil, nil],
      0xff6a => [0, 0, nil, "\343\202\247", nil, nil, nil],
      0xff6b => [0, 0, nil, "\343\202\251", nil, nil, nil],
      0xff6c => [0, 0, nil, "\343\203\243", nil, nil, nil],
      0xff6d => [0, 0, nil, "\343\203\245", nil, nil, nil],
      0xff6e => [0, 0, nil, "\343\203\247", nil, nil, nil],
      0xff6f => [0, 0, nil, "\343\203\203", nil, nil, nil],
      0xff70 => [0, 0, nil, "\343\203\274", nil, nil, nil],
      0xff71 => [0, 0, nil, "\343\202\242", nil, nil, nil],
      0xff72 => [0, 0, nil, "\343\202\244", nil, nil, nil],
      0xff73 => [0, 0, nil, "\343\202\246", nil, nil, nil],
      0xff74 => [0, 0, nil, "\343\202\250", nil, nil, nil],
      0xff75 => [0, 0, nil, "\343\202\252", nil, nil, nil],
      0xff76 => [0, 0, nil, "\343\202\253", nil, nil, nil],
      0xff77 => [0, 0, nil, "\343\202\255", nil, nil, nil],
      0xff78 => [0, 0, nil, "\343\202\257", nil, nil, nil],
      0xff79 => [0, 0, nil, "\343\202\261", nil, nil, nil],
      0xff7a => [0, 0, nil, "\343\202\263", nil, nil, nil],
      0xff7b => [0, 0, nil, "\343\202\265", nil, nil, nil],
      0xff7c => [0, 0, nil, "\343\202\267", nil, nil, nil],
      0xff7d => [0, 0, nil, "\343\202\271", nil, nil, nil],
      0xff7e => [0, 0, nil, "\343\202\273", nil, nil, nil],
      0xff7f => [0, 0, nil, "\343\202\275", nil, nil, nil],
      0xff80 => [0, 0, nil, "\343\202\277", nil, nil, nil],
      0xff81 => [0, 0, nil, "\343\203\201", nil, nil, nil],
      0xff82 => [0, 0, nil, "\343\203\204", nil, nil, nil],
      0xff83 => [0, 0, nil, "\343\203\206", nil, nil, nil],
      0xff84 => [0, 0, nil, "\343\203\210", nil, nil, nil],
      0xff85 => [0, 0, nil, "\343\203\212", nil, nil, nil],
      0xff86 => [0, 0, nil, "\343\203\213", nil, nil, nil],
      0xff87 => [0, 0, nil, "\343\203\214", nil, nil, nil],
      0xff88 => [0, 0, nil, "\343\203\215", nil, nil, nil],
      0xff89 => [0, 0, nil, "\343\203\216", nil, nil, nil],
      0xff8a => [0, 0, nil, "\343\203\217", nil, nil, nil],
      0xff8b => [0, 0, nil, "\343\203\222", nil, nil, nil],
      0xff8c => [0, 0, nil, "\343\203\225", nil, nil, nil],
      0xff8d => [0, 0, nil, "\343\203\230", nil, nil, nil],
      0xff8e => [0, 0, nil, "\343\203\233", nil, nil, nil],
      0xff8f => [0, 0, nil, "\343\203\236", nil, nil, nil],
      0xff90 => [0, 0, nil, "\343\203\237", nil, nil, nil],
      0xff91 => [0, 0, nil, "\343\203\240", nil, nil, nil],
      0xff92 => [0, 0, nil, "\343\203\241", nil, nil, nil],
      0xff93 => [0, 0, nil, "\343\203\242", nil, nil, nil],
      0xff94 => [0, 0, nil, "\343\203\244", nil, nil, nil],
      0xff95 => [0, 0, nil, "\343\203\246", nil, nil, nil],
      0xff96 => [0, 0, nil, "\343\203\250", nil, nil, nil],
      0xff97 => [0, 0, nil, "\343\203\251", nil, nil, nil],
      0xff98 => [0, 0, nil, "\343\203\252", nil, nil, nil],
      0xff99 => [0, 0, nil, "\343\203\253", nil, nil, nil],
      0xff9a => [0, 0, nil, "\343\203\254", nil, nil, nil],
      0xff9b => [0, 0, nil, "\343\203\255", nil, nil, nil],
      0xff9c => [0, 0, nil, "\343\203\257", nil, nil, nil],
      0xff9d => [0, 0, nil, "\343\203\263", nil, nil, nil],
      0xff9e => [0, 0, nil, "\343\202\231", nil, nil, nil],
      0xff9f => [0, 0, nil, "\343\202\232", nil, nil, nil],
      0xffa0 => [0, 0, nil, "\343\205\244", nil, nil, nil],
      0xffa1 => [0, 0, nil, "\343\204\261", nil, nil, nil],
      0xffa2 => [0, 0, nil, "\343\204\262", nil, nil, nil],
      0xffa3 => [0, 0, nil, "\343\204\263", nil, nil, nil],
      0xffa4 => [0, 0, nil, "\343\204\264", nil, nil, nil],
      0xffa5 => [0, 0, nil, "\343\204\265", nil, nil, nil],
      0xffa6 => [0, 0, nil, "\343\204\266", nil, nil, nil],
      0xffa7 => [0, 0, nil, "\343\204\267", nil, nil, nil],
      0xffa8 => [0, 0, nil, "\343\204\270", nil, nil, nil],
      0xffa9 => [0, 0, nil, "\343\204\271", nil, nil, nil],
      0xffaa => [0, 0, nil, "\343\204\272", nil, nil, nil],
      0xffab => [0, 0, nil, "\343\204\273", nil, nil, nil],
      0xffac => [0, 0, nil, "\343\204\274", nil, nil, nil],
      0xffad => [0, 0, nil, "\343\204\275", nil, nil, nil],
      0xffae => [0, 0, nil, "\343\204\276", nil, nil, nil],
      0xffaf => [0, 0, nil, "\343\204\277", nil, nil, nil],
      0xffb0 => [0, 0, nil, "\343\205\200", nil, nil, nil],
      0xffb1 => [0, 0, nil, "\343\205\201", nil, nil, nil],
      0xffb2 => [0, 0, nil, "\343\205\202", nil, nil, nil],
      0xffb3 => [0, 0, nil, "\343\205\203", nil, nil, nil],
      0xffb4 => [0, 0, nil, "\343\205\204", nil, nil, nil],
      0xffb5 => [0, 0, nil, "\343\205\205", nil, nil, nil],
      0xffb6 => [0, 0, nil, "\343\205\206", nil, nil, nil],
      0xffb7 => [0, 0, nil, "\343\205\207", nil, nil, nil],
      0xffb8 => [0, 0, nil, "\343\205\210", nil, nil, nil],
      0xffb9 => [0, 0, nil, "\343\205\211", nil, nil, nil],
      0xffba => [0, 0, nil, "\343\205\212", nil, nil, nil],
      0xffbb => [0, 0, nil, "\343\205\213", nil, nil, nil],
      0xffbc => [0, 0, nil, "\343\205\214", nil, nil, nil],
      0xffbd => [0, 0, nil, "\343\205\215", nil, nil, nil],
      0xffbe => [0, 0, nil, "\343\205\216", nil, nil, nil],
      0xffc2 => [0, 0, nil, "\343\205\217", nil, nil, nil],
      0xffc3 => [0, 0, nil, "\343\205\220", nil, nil, nil],
      0xffc4 => [0, 0, nil, "\343\205\221", nil, nil, nil],
      0xffc5 => [0, 0, nil, "\343\205\222", nil, nil, nil],
      0xffc6 => [0, 0, nil, "\343\205\223", nil, nil, nil],
      0xffc7 => [0, 0, nil, "\343\205\224", nil, nil, nil],
      0xffca => [0, 0, nil, "\343\205\225", nil, nil, nil],
      0xffcb => [0, 0, nil, "\343\205\226", nil, nil, nil],
      0xffcc => [0, 0, nil, "\343\205\227", nil, nil, nil],
      0xffcd => [0, 0, nil, "\343\205\230", nil, nil, nil],
      0xffce => [0, 0, nil, "\343\205\231", nil, nil, nil],
      0xffcf => [0, 0, nil, "\343\205\232", nil, nil, nil],
      0xffd2 => [0, 0, nil, "\343\205\233", nil, nil, nil],
      0xffd3 => [0, 0, nil, "\343\205\234", nil, nil, nil],
      0xffd4 => [0, 0, nil, "\343\205\235", nil, nil, nil],
      0xffd5 => [0, 0, nil, "\343\205\236", nil, nil, nil],
      0xffd6 => [0, 0, nil, "\343\205\237", nil, nil, nil],
      0xffd7 => [0, 0, nil, "\343\205\240", nil, nil, nil],
      0xffda => [0, 0, nil, "\343\205\241", nil, nil, nil],
      0xffdb => [0, 0, nil, "\343\205\242", nil, nil, nil],
      0xffdc => [0, 0, nil, "\343\205\243", nil, nil, nil],
      0xffe0 => [0, 0, nil, "\302\242", nil, nil, nil],
      0xffe1 => [0, 0, nil, "\302\243", nil, nil, nil],
      0xffe2 => [0, 0, nil, "\302\254", nil, nil, nil],
      0xffe3 => [0, 0, nil, "\302\257", nil, nil, nil],
      0xffe4 => [0, 0, nil, "\302\246", nil, nil, nil],
      0xffe5 => [0, 0, nil, "\302\245", nil, nil, nil],
      0xffe6 => [0, 0, nil, "\342\202\251", nil, nil, nil],
      0xffe8 => [0, 0, nil, "\342\224\202", nil, nil, nil],
      0xffe9 => [0, 0, nil, "\342\206\220", nil, nil, nil],
      0xffea => [0, 0, nil, "\342\206\221", nil, nil, nil],
      0xffeb => [0, 0, nil, "\342\206\222", nil, nil, nil],
      0xffec => [0, 0, nil, "\342\206\223", nil, nil, nil],
      0xffed => [0, 0, nil, "\342\226\240", nil, nil, nil],
      0xffee => [0, 0, nil, "\342\227\213", nil, nil, nil]
    }

    COMPOSITION_TABLE = {}
    for codepoint, data in UNICODE_DATA
      canonical = data[UNICODE_DATA_CANONICAL]
      exclusion = data[UNICODE_DATA_EXCLUSION]

      if canonical && exclusion == 0
        COMPOSITION_TABLE[canonical] = codepoint
      end
    end

    UNICODE_MAX_LENGTH = 256
    ACE_MAX_LENGTH = 256

    PUNYCODE_BASE = 36
    PUNYCODE_TMIN = 1
    PUNYCODE_TMAX = 26
    PUNYCODE_SKEW = 38
    PUNYCODE_DAMP = 700
    PUNYCODE_INITIAL_BIAS = 72
    PUNYCODE_INITIAL_N = 0x80
    PUNYCODE_DELIMITER = 0x2D

    PUNYCODE_MAXINT = 1 << 64

    PUNYCODE_PRINT_ASCII =
      "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n" +
      "\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n" +
      " !\"\#$%&'()*+,-./" +
      "0123456789:;<=>?" +
      "@ABCDEFGHIJKLMNO" +
      "PQRSTUVWXYZ[\\]^_" +
      "`abcdefghijklmno" +
      "pqrstuvwxyz{|}~\n"

    # Input is invalid.
    class PunycodeBadInput < StandardError; end
    # Output would exceed the space provided.
    class PunycodeBigOutput < StandardError; end
    # Input needs wider integers to process.
    class PunycodeOverflow < StandardError; end

    def self.punycode_encode(unicode)
      input = unicode.unpack("U*")
      output = [0] * (ACE_MAX_LENGTH + 1)
      input_length = input.size
      output_length = [ACE_MAX_LENGTH]

      # Initialize the state
      n = PUNYCODE_INITIAL_N
      delta = out = 0
      max_out = output_length[0]
      bias = PUNYCODE_INITIAL_BIAS

      # Handle the basic code points:
      input_length.times do |j|
        if punycode_basic?(input[j])
          if max_out - out < 2
            raise PunycodeBigOutput,
              "Output would exceed the space provided."
          end
          output[out] = input[j]
          out += 1
        end
      end

      h = b = out

      # h is the number of code points that have been handled, b is the
      # number of basic code points, and out is the number of characters
      # that have been output.

      if b > 0
        output[out] = PUNYCODE_DELIMITER
        out += 1
      end

      # Main encoding loop:

      while h < input_length
        # All non-basic code points < n have been
        # handled already.  Find the next larger one:

        m = PUNYCODE_MAXINT
        input_length.times do |j|
          m = input[j] if (n...m) === input[j]
        end

        # Increase delta enough to advance the decoder's
        # <n,i> state to <m,0>, but guard against overflow:

        if m - n > (PUNYCODE_MAXINT - delta) / (h + 1)
          raise PunycodeOverflow, "Input needs wider integers to process."
        end
        delta += (m - n) * (h + 1)
        n = m

        input_length.times do |j|
          # Punycode does not need to check whether input[j] is basic:
          if input[j] < n
            delta += 1
            if delta == 0
              raise PunycodeOverflow,
                "Input needs wider integers to process."
            end
          end

          if input[j] == n
            # Represent delta as a generalized variable-length integer:

            q = delta; k = PUNYCODE_BASE
            while true
              if out >= max_out
                raise PunycodeBigOutput,
                  "Output would exceed the space provided."
              end
              t = (
                if k <= bias
                  PUNYCODE_TMIN
                elsif k >= bias + PUNYCODE_TMAX
                  PUNYCODE_TMAX
                else
                  k - bias
                end
              )
              break if q < t
              output[out] =
                punycode_encode_digit(t + (q - t) % (PUNYCODE_BASE - t))
              out += 1
              q = (q - t) / (PUNYCODE_BASE - t)
              k += PUNYCODE_BASE
            end

            output[out] = punycode_encode_digit(q)
            out += 1
            bias = punycode_adapt(delta, h + 1, h == b)
            delta = 0
            h += 1
          end
        end

        delta += 1
        n += 1
      end

      output_length[0] = out

      outlen = out
      outlen.times do |j|
        c = output[j]
        unless c >= 0 && c <= 127
          raise Exception, "Invalid output char."
        end
        unless PUNYCODE_PRINT_ASCII[c]
          raise PunycodeBadInput, "Input is invalid."
        end
      end

      output[0..outlen].map { |x| x.chr }.join("").sub(/\0+\z/, "")
    end
    (class <<self; private :punycode_encode; end)

    def self.punycode_decode(punycode)
      input = []
      output = []

      if ACE_MAX_LENGTH * 2 < punycode.size
        raise PunycodeBigOutput, "Output would exceed the space provided."
      end
      punycode.each_byte do |c|
        unless c >= 0 && c <= 127
          raise PunycodeBadInput, "Input is invalid."
        end
        input.push(c)
      end

      input_length = input.length
      output_length = [UNICODE_MAX_LENGTH]

      # Initialize the state
      n = PUNYCODE_INITIAL_N

      out = i = 0
      max_out = output_length[0]
      bias = PUNYCODE_INITIAL_BIAS

      # Handle the basic code points:  Let b be the number of input code
      # points before the last delimiter, or 0 if there is none, then
      # copy the first b code points to the output.

      b = 0
      input_length.times do |j|
        b = j if punycode_delimiter?(input[j])
      end
      if b > max_out
        raise PunycodeBigOutput, "Output would exceed the space provided."
      end

      b.times do |j|
        unless punycode_basic?(input[j])
          raise PunycodeBadInput, "Input is invalid."
        end
        output[out] = input[j]
        out+=1
      end

      # Main decoding loop:  Start just after the last delimiter if any
      # basic code points were copied; start at the beginning otherwise.

      in_ = b > 0 ? b + 1 : 0
      while in_ < input_length

        # in_ is the index of the next character to be consumed, and
        # out is the number of code points in the output array.

        # Decode a generalized variable-length integer into delta,
        # which gets added to i.  The overflow checking is easier
        # if we increase i as we go, then subtract off its starting
        # value at the end to obtain delta.

        oldi = i; w = 1; k = PUNYCODE_BASE
        while true
          if in_ >= input_length
            raise PunycodeBadInput, "Input is invalid."
          end
          digit = punycode_decode_digit(input[in_])
          in_+=1
          if digit >= PUNYCODE_BASE
            raise PunycodeBadInput, "Input is invalid."
          end
          if digit > (PUNYCODE_MAXINT - i) / w
            raise PunycodeOverflow, "Input needs wider integers to process."
          end
          i += digit * w
          t = (
            if k <= bias
              PUNYCODE_TMIN
            elsif k >= bias + PUNYCODE_TMAX
              PUNYCODE_TMAX
            else
              k - bias
            end
          )
          break if digit < t
          if w > PUNYCODE_MAXINT / (PUNYCODE_BASE - t)
            raise PunycodeOverflow, "Input needs wider integers to process."
          end
          w *= PUNYCODE_BASE - t
          k += PUNYCODE_BASE
        end

        bias = punycode_adapt(i - oldi, out + 1, oldi == 0)

        # I was supposed to wrap around from out + 1 to 0,
        # incrementing n each time, so we'll fix that now:

        if i / (out + 1) > PUNYCODE_MAXINT - n
          raise PunycodeOverflow, "Input needs wider integers to process."
        end
        n += i / (out + 1)
        i %= out + 1

        # Insert n at position i of the output:

        # not needed for Punycode:
        # raise PUNYCODE_INVALID_INPUT if decode_digit(n) <= base
        if out >= max_out
          raise PunycodeBigOutput, "Output would exceed the space provided."
        end

        #memmove(output + i + 1, output + i, (out - i) * sizeof *output)
        output[i + 1, out - i] = output[i, out - i]
        output[i] = n
        i += 1

        out += 1
      end

      output_length[0] = out

      output.pack("U*")
    end
    (class <<self; private :punycode_decode; end)

    def self.punycode_basic?(codepoint)
      codepoint < 0x80
    end
    (class <<self; private :punycode_basic?; end)

    def self.punycode_delimiter?(codepoint)
      codepoint == PUNYCODE_DELIMITER
    end
    (class <<self; private :punycode_delimiter?; end)

    def self.punycode_encode_digit(d)
      d + 22 + 75 * ((d < 26) ? 1 : 0)
    end
    (class <<self; private :punycode_encode_digit; end)

    # Returns the numeric value of a basic codepoint
    # (for use in representing integers) in the range 0 to
    # base - 1, or PUNYCODE_BASE if codepoint does not represent a value.
    def self.punycode_decode_digit(codepoint)
      if codepoint - 48 < 10
        codepoint - 22
      elsif codepoint - 65 < 26
        codepoint - 65
      elsif codepoint - 97 < 26
        codepoint - 97
      else
        PUNYCODE_BASE
      end
    end
    (class <<self; private :punycode_decode_digit; end)

    # Bias adaptation method
    def self.punycode_adapt(delta, numpoints, firsttime)
      delta = firsttime ? delta / PUNYCODE_DAMP : delta >> 1
      # delta >> 1 is a faster way of doing delta / 2
      delta += delta / numpoints

      k = 0
      while delta > ((PUNYCODE_BASE - PUNYCODE_TMIN) * PUNYCODE_TMAX) / 2
        delta /= PUNYCODE_BASE - PUNYCODE_TMIN
        k += PUNYCODE_BASE
      end

      k + (PUNYCODE_BASE - PUNYCODE_TMIN + 1) *
        delta / (delta + PUNYCODE_SKEW)
    end
    (class <<self; private :punycode_adapt; end)
  end
  # :startdoc:
end
