#!/usr/bin/perl

$me = $0;

$usage = qq{
Usage:	$me <dumpfile>
	Restore a custom dump (pg_dump -Fc) of a PostGIS-enabled database.
	First dump the old database: pg_dump -Fc <olddb> > <olddb.dmp>
	Then create a new database: createdb <newdb>
	Then install PostGIS in the new database: psql -f <path>/postgis.sql
	Finally, run this script on the old dump: $me <olddb.dmp> | psql <newdb>

};

die $usage if (@ARGV != 1);

$dumpfile = $ARGV[0];
$manifest = $dumpfile . ".lst";

die "$me:\tUnable to find 'pg_dump' on the path.\n" if ! `pg_dump --version`;
die "$me:\tUnable to find 'pg_restore' on the path.\n" if ! `pg_restore --version`;
die "$me:\tUnable to open dump file '$dumpfile'.\n" if ! -f $dumpfile;

$DEBUG = 0;

print STDERR "Converting $dumpfile to ASCII on stdout...\n";

######################################################################
# Load the signatures of things to skip.
#

print STDERR "  Reading list of functions to ignore...\n";

my %skip = ();
while(my $l = <DATA>) {
  print STDERR "    $l" if $DEBUG;
  $l =~ s/\s//g;
  $skip{$l} = 1;
}

######################################################################
# Write a new manifest for the dump file, skipping the things that
# are part of PostGIS
#

print STDERR "  Writing manifest of things to read from dump file...\n";

open( DUMP, "pg_restore -l $dumpfile |" ) || die "$me:\tCannot open dump file '$dumpfile'\n";
open( MANIFEST, ">$manifest" ) || die "$me:\tCannot open manifest file '$outputfile'\n";
while( my $l = <DUMP> ) {

  next if $l =~ /^\;/;
  my $sig = linesignature($l);
  $sig =~ s/\s//g;
  print STDERR "    $sig\n" if $DEBUG;
  next if $skip{$sig};
  print STDERR "    $l" if $DEBUG;
  print MANIFEST $l;

}
close(MANIFEST);
close(DUMP);

######################################################################
# Convert the dump file into an ASCII file, stripping out the 
# unwanted bits.
#
print STDERR "  Writing ASCII to stdout...\n";
open( INPUT, "pg_restore -L $manifest $dumpfile |") || die "$me:\tCan't run pg_restore\n";
while( my $l = <INPUT> ) {

  next if $l =~ /^ *--/;

  if ( $l =~ /^SET search_path/ ) {
    $l =~ s/; *$/, public;/; 
  }

  print STDOUT $l;

}


print STDERR "Done.\n";

######################################################################
# Strip a dump file manifest line down to the unique elements of
# type and signature.
#
sub linesignature {

  my $line = shift;
  my $sig;

  $line =~ s/\n$//;
  $line =~ s/\r$//;
  $line =~ s/OPERATOR CLASS/OPERATORCLASS/;
  $line =~ s/TABLE DATA/TABLEDATA/;
  $line =~ s/SHELL TYPE/SHELLTYPE/;
  $line =~ s/PROCEDURAL LANGUAGE/PROCEDURALLANGUAGE/;

  if( $line =~ /^(\d+)\; (\d+) (\d+) (\w+) (\w+) (.*) (\w*)/ ) {
    $sig = $4 . "\t" . $6;
  }
  elsif( $line =~ /PROCEDURALLANGUAGE.*plpgsql/ ) {
    $sig = "PROCEDURALLANGUAGE\tplpgsql";
  }

  $sig;

}


######################################################################
# Here are all the signatures we want to skip.
#
__END__
PROCEDURALLANGUAGE	plpgsql
AGGREGATE	accum(geometry)	
AGGREGATE	accum_old(geometry)	
AGGREGATE	collect(geometry)	
AGGREGATE	extent(geometry)	
AGGREGATE	extent3d(geometry)	
AGGREGATE	geomunion(geometry)	
AGGREGATE	geomunion_old(geometry)	
AGGREGATE	makeline(geometry)	
AGGREGATE	memcollect(geometry)	
AGGREGATE	memgeomunion(geometry)	
AGGREGATE	polygonize(geometry)	
AGGREGATE	st_accum(geometry)	
AGGREGATE	st_accum_old(geometry)	
AGGREGATE	st_collect(geometry)	
AGGREGATE	st_extent(geometry)	
AGGREGATE	st_extent3d(geometry)	
AGGREGATE	st_makeline(geometry)	
AGGREGATE	st_memcollect(geometry)	
AGGREGATE	st_memunion(geometry)	
AGGREGATE	st_polygonize(geometry)	
AGGREGATE	st_union(geometry)	
AGGREGATE	st_union_old(geometry)	
CAST	CAST (boolean AS text)	
CAST	CAST (bytea AS public.geometry)	
CAST	CAST (public.box2d AS public.box3d)	
CAST	CAST (public.box2d AS public.geometry)	
CAST	CAST (public.box3d AS box)	
CAST	CAST (public.box3d AS public.box2d)	
CAST	CAST (public.box3d AS public.geometry)	
CAST	CAST (public.chip AS public.geometry)	
CAST	CAST (public.geometry AS box)	
CAST	CAST (public.geometry AS bytea)	
CAST	CAST (public.geometry AS public.box2d)	
CAST	CAST (public.geometry AS public.box3d)	
CAST	CAST (public.geometry AS text)	
CAST	CAST (text AS public.geometry)	
FUNCTION	_st_asgeojson(integer, geometry, integer, integer)	
FUNCTION	_st_asgml(integer, geometry, integer)	
FUNCTION	_st_askml(integer, geometry, integer)	
FUNCTION	_st_contains(geometry, geometry)	
FUNCTION	_st_containsproperly(geometry, geometry)	
FUNCTION	_st_coveredby(geometry, geometry)	
FUNCTION	_st_covers(geometry, geometry)	
FUNCTION	_st_crosses(geometry, geometry)	
FUNCTION	_st_dwithin(geometry, geometry, double precision)	
FUNCTION	_st_intersects(geometry, geometry)	
FUNCTION	_st_linecrossingdirection(geometry, geometry)	
FUNCTION	_st_overlaps(geometry, geometry)	
FUNCTION	_st_touches(geometry, geometry)	
FUNCTION	_st_within(geometry, geometry)	
FUNCTION	addauth(text)	
FUNCTION	addbbox(geometry)	
FUNCTION	addgeometrycolumn(character varying, character varying, character varying, character varying, integer, character varying, integer)	
FUNCTION	addgeometrycolumn(character varying, character varying, character varying, integer, character varying, integer)	
FUNCTION	addgeometrycolumn(character varying, character varying, integer, character varying, integer)	
FUNCTION	addpoint(geometry, geometry)	
FUNCTION	addpoint(geometry, geometry, integer)	
FUNCTION	affine(geometry, double precision, double precision, double precision, double precision, double precision, double precision)	
FUNCTION	affine(geometry, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision)	
FUNCTION	area(geometry)	
FUNCTION	area2d(geometry)	
FUNCTION	asbinary(geometry)	
FUNCTION	asbinary(geometry, text)	
FUNCTION	asewkb(geometry)	
FUNCTION	asewkb(geometry, text)	
FUNCTION	asewkt(geometry)	
FUNCTION	asgml(geometry)	
FUNCTION	asgml(geometry, integer)	
FUNCTION	asgml(geometry, integer, integer)	
FUNCTION	ashexewkb(geometry)	
FUNCTION	ashexewkb(geometry, text)	
FUNCTION	askml(geometry)	
FUNCTION	askml(geometry, integer)	
FUNCTION	askml(geometry, integer, integer)	
FUNCTION	askml(integer, geometry, integer)	
FUNCTION	assvg(geometry)	
FUNCTION	assvg(geometry, integer)	
FUNCTION	assvg(geometry, integer, integer)	
FUNCTION	astext(geometry)	
FUNCTION	asukml(geometry)	
FUNCTION	asukml(geometry, integer)	
FUNCTION	asukml(geometry, integer, integer)	
FUNCTION	azimuth(geometry, geometry)	
FUNCTION	bdmpolyfromtext(text, integer)	
FUNCTION	bdpolyfromtext(text, integer)	
FUNCTION	boundary(geometry)	
FUNCTION	box(box3d)	
FUNCTION	box(geometry)	
FUNCTION	box2d(box3d)	
FUNCTION	box2d(geometry)	
FUNCTION	box2d_contain(box2d, box2d)	
FUNCTION	box2d_contained(box2d, box2d)	
FUNCTION	box2d_in(cstring)	
FUNCTION	box2d_intersects(box2d, box2d)	
FUNCTION	box2d_left(box2d, box2d)	
FUNCTION	box2d_out(box2d)	
FUNCTION	box2d_overlap(box2d, box2d)	
FUNCTION	box2d_overleft(box2d, box2d)	
FUNCTION	box2d_overright(box2d, box2d)	
FUNCTION	box2d_right(box2d, box2d)	
FUNCTION	box2d_same(box2d, box2d)	
FUNCTION	box3d(box2d)	
FUNCTION	box3d(geometry)	
FUNCTION	box3d_in(cstring)	
FUNCTION	box3d_out(box3d)	
FUNCTION	box3dtobox(box3d)	
FUNCTION	buffer(geometry, double precision)	
FUNCTION	buffer(geometry, double precision, integer)	
FUNCTION	build_histogram2d(histogram2d, text, text)	
FUNCTION	build_histogram2d(histogram2d, text, text, text)	
FUNCTION	buildarea(geometry)	
FUNCTION	bytea(geometry)	
FUNCTION	cache_bbox()	
FUNCTION	centroid(geometry)	
FUNCTION	checkauth(text, text)	
FUNCTION	checkauth(text, text, text)	
FUNCTION	checkauthtrigger()	
FUNCTION	chip_in(cstring)	
FUNCTION	chip_out(chip)	
FUNCTION	collect(geometry, geometry)	
FUNCTION	collect_garray(geometry[])	
FUNCTION	collector(geometry, geometry)	
FUNCTION	combine_bbox(box2d, geometry)	
FUNCTION	combine_bbox(box3d, geometry)	
FUNCTION	compression(chip)	
FUNCTION	contains(geometry, geometry)	
FUNCTION	convexhull(geometry)	
FUNCTION	create_histogram2d(box2d, integer)	
FUNCTION	crosses(geometry, geometry)	
FUNCTION	datatype(chip)	
FUNCTION	difference(geometry, geometry)	
FUNCTION	dimension(geometry)	
FUNCTION	disablelongtransactions()	
FUNCTION	disjoint(geometry, geometry)	
FUNCTION	distance(geometry, geometry)	
FUNCTION	distance_sphere(geometry, geometry)	
FUNCTION	distance_spheroid(geometry, geometry, spheroid)	
FUNCTION	dropbbox(geometry)	
FUNCTION	dropgeometrycolumn(character varying, character varying)	
FUNCTION	dropgeometrycolumn(character varying, character varying, character varying)	
FUNCTION	dropgeometrycolumn(character varying, character varying, character varying, character varying)	
FUNCTION	dropgeometrytable(character varying)	
FUNCTION	dropgeometrytable(character varying, character varying)	
FUNCTION	dropgeometrytable(character varying, character varying, character varying)	
FUNCTION	dump(geometry)	
FUNCTION	dumprings(geometry)	
FUNCTION	enablelongtransactions()	
FUNCTION	endpoint(geometry)	
FUNCTION	envelope(geometry)	
FUNCTION	equals(geometry, geometry)	
FUNCTION	estimate_histogram2d(histogram2d, box2d)	
FUNCTION	estimated_extent(text, text)	
FUNCTION	estimated_extent(text, text, text)	
FUNCTION	expand(box2d, double precision)	
FUNCTION	expand(box3d, double precision)	
FUNCTION	expand(geometry, double precision)	
FUNCTION	explode_histogram2d(histogram2d, text)	
FUNCTION	exteriorring(geometry)	
FUNCTION	factor(chip)	
FUNCTION	find_extent(text, text)	
FUNCTION	find_extent(text, text, text)	
FUNCTION	find_srid(character varying, character varying, character varying)	
FUNCTION	fix_geometry_columns()	
FUNCTION	force_2d(geometry)	
FUNCTION	force_3d(geometry)	
FUNCTION	force_3dm(geometry)	
FUNCTION	force_3dz(geometry)	
FUNCTION	force_4d(geometry)	
FUNCTION	force_collection(geometry)	
FUNCTION	forcerhr(geometry)	
FUNCTION	geom_accum(geometry[], geometry)	
FUNCTION	geomcollfromtext(text)	
FUNCTION	geomcollfromtext(text, integer)	
FUNCTION	geomcollfromwkb(bytea)	
FUNCTION	geomcollfromwkb(bytea, integer)	
FUNCTION	geometry(box2d)	
FUNCTION	geometry(box3d)	
FUNCTION	geometry(bytea)	
FUNCTION	geometry(chip)	
FUNCTION	geometry(text)	
FUNCTION	geometry_above(geometry, geometry)	
FUNCTION	geometry_analyze(internal)	
FUNCTION	geometry_below(geometry, geometry)	
FUNCTION	geometry_cmp(geometry, geometry)	
FUNCTION	geometry_contain(geometry, geometry)	
FUNCTION	geometry_contained(geometry, geometry)	
FUNCTION	geometry_eq(geometry, geometry)	
FUNCTION	geometry_ge(geometry, geometry)	
FUNCTION	geometry_gt(geometry, geometry)	
FUNCTION	geometry_in(cstring)	
FUNCTION	geometry_le(geometry, geometry)	
FUNCTION	geometry_left(geometry, geometry)	
FUNCTION	geometry_lt(geometry, geometry)	
FUNCTION	geometry_out(geometry)	
FUNCTION	geometry_overabove(geometry, geometry)	
FUNCTION	geometry_overbelow(geometry, geometry)	
FUNCTION	geometry_overlap(geometry, geometry)	
FUNCTION	geometry_overleft(geometry, geometry)	
FUNCTION	geometry_overright(geometry, geometry)	
FUNCTION	geometry_recv(internal)	
FUNCTION	geometry_right(geometry, geometry)	
FUNCTION	geometry_same(geometry, geometry)	
FUNCTION	geometry_send(geometry)	
FUNCTION	geometryfromtext(text)	
FUNCTION	geometryfromtext(text, integer)	
FUNCTION	geometryn(geometry, integer)	
FUNCTION	geometrytype(geometry)	
FUNCTION	geomfromewkb(bytea)	
FUNCTION	geomfromewkt(text)	
FUNCTION	geomfromtext(text)	
FUNCTION	geomfromtext(text, integer)	
FUNCTION	geomfromwkb(bytea)	
FUNCTION	geomfromwkb(bytea, integer)	
FUNCTION	geomunion(geometry, geometry)	
FUNCTION	geosnoop(geometry)	
FUNCTION	get_proj4_from_srid(integer)	
FUNCTION	getbbox(geometry)	
FUNCTION	getsrid(geometry)	
FUNCTION	gettransactionid()	
FUNCTION	hasbbox(geometry)	
FUNCTION	height(chip)	
FUNCTION	histogram2d_in(cstring)	
FUNCTION	histogram2d_out(histogram2d)	
FUNCTION	interiorringn(geometry, integer)	
FUNCTION	intersection(geometry, geometry)	
FUNCTION	intersects(geometry, geometry)	
FUNCTION	isclosed(geometry)	
FUNCTION	isempty(geometry)	
FUNCTION	isring(geometry)	
FUNCTION	issimple(geometry)	
FUNCTION	isvalid(geometry)	
FUNCTION	jtsnoop(geometry)	
FUNCTION	length(geometry)	
FUNCTION	length2d(geometry)	
FUNCTION	length2d_spheroid(geometry, spheroid)	
FUNCTION	length3d(geometry)	
FUNCTION	length3d_spheroid(geometry, spheroid)	
FUNCTION	length_spheroid(geometry, spheroid)	
FUNCTION	line_interpolate_point(geometry, double precision)	
FUNCTION	line_locate_point(geometry, geometry)	
FUNCTION	line_substring(geometry, double precision, double precision)	
FUNCTION	linefrommultipoint(geometry)	
FUNCTION	linefromtext(text)	
FUNCTION	linefromtext(text, integer)	
FUNCTION	linefromwkb(bytea)	
FUNCTION	linefromwkb(bytea, integer)	
FUNCTION	linemerge(geometry)	
FUNCTION	linestringfromtext(text)	
FUNCTION	linestringfromtext(text, integer)	
FUNCTION	linestringfromwkb(bytea)	
FUNCTION	linestringfromwkb(bytea, integer)	
FUNCTION	locate_along_measure(geometry, double precision)	
FUNCTION	locate_between_measures(geometry, double precision, double precision)	
FUNCTION	lockrow(text, text, text)	
FUNCTION	lockrow(text, text, text, text)	
FUNCTION	lockrow(text, text, text, text, timestamp without time zone)	
FUNCTION	lockrow(text, text, text, timestamp without time zone)	
FUNCTION	longtransactionsenabled()	
FUNCTION	lwgeom_gist_compress(internal)	
FUNCTION	lwgeom_gist_consistent(internal, geometry, integer)	
FUNCTION	lwgeom_gist_decompress(internal)	
FUNCTION	lwgeom_gist_penalty(internal, internal, internal)	
FUNCTION	lwgeom_gist_picksplit(internal, internal)	
FUNCTION	lwgeom_gist_same(box2d, box2d, internal)	
FUNCTION	lwgeom_gist_union(bytea, internal)	
FUNCTION	m(geometry)	
FUNCTION	makebox2d(geometry, geometry)	
FUNCTION	makebox3d(geometry, geometry)	
FUNCTION	makeline(geometry, geometry)	
FUNCTION	makeline_garray(geometry[])	
FUNCTION	makepoint(double precision, double precision)	
FUNCTION	makepoint(double precision, double precision, double precision)	
FUNCTION	makepoint(double precision, double precision, double precision, double precision)	
FUNCTION	makepointm(double precision, double precision, double precision)	
FUNCTION	makepolygon(geometry)	
FUNCTION	makepolygon(geometry, geometry[])	
FUNCTION	max_distance(geometry, geometry)	
FUNCTION	mem_size(geometry)	
FUNCTION	mlinefromtext(text)	
FUNCTION	mlinefromtext(text, integer)	
FUNCTION	mlinefromwkb(bytea)	
FUNCTION	mlinefromwkb(bytea, integer)	
FUNCTION	mpointfromtext(text)	
FUNCTION	mpointfromtext(text, integer)	
FUNCTION	mpointfromwkb(bytea)	
FUNCTION	mpointfromwkb(bytea, integer)	
FUNCTION	mpolyfromtext(text)	
FUNCTION	mpolyfromtext(text, integer)	
FUNCTION	mpolyfromwkb(bytea)	
FUNCTION	mpolyfromwkb(bytea, integer)	
FUNCTION	multi(geometry)	
FUNCTION	multilinefromwkb(bytea)	
FUNCTION	multilinefromwkb(bytea, integer)	
FUNCTION	multilinestringfromtext(text)	
FUNCTION	multilinestringfromtext(text, integer)	
FUNCTION	multipointfromtext(text)	
FUNCTION	multipointfromtext(text, integer)	
FUNCTION	multipointfromwkb(bytea)	
FUNCTION	multipointfromwkb(bytea, integer)	
FUNCTION	multipolyfromwkb(bytea)	
FUNCTION	multipolyfromwkb(bytea, integer)	
FUNCTION	multipolygonfromtext(text)	
FUNCTION	multipolygonfromtext(text, integer)	
FUNCTION	ndims(geometry)	
FUNCTION	noop(geometry)	
FUNCTION	npoints(geometry)	
FUNCTION	nrings(geometry)	
FUNCTION	numgeometries(geometry)	
FUNCTION	numinteriorring(geometry)	
FUNCTION	numinteriorrings(geometry)	
FUNCTION	numpoints(geometry)	
FUNCTION	overlaps(geometry, geometry)	
FUNCTION	perimeter(geometry)	
FUNCTION	perimeter2d(geometry)	
FUNCTION	perimeter3d(geometry)	
FUNCTION	pgis_abs_in(cstring)	
FUNCTION	pgis_abs_out(pgis_abs)	
FUNCTION	pgis_geometry_accum_finalfn(pgis_abs)	
FUNCTION	pgis_geometry_accum_transfn(pgis_abs, geometry)	
FUNCTION	pgis_geometry_collect_finalfn(pgis_abs)	
FUNCTION	pgis_geometry_makeline_finalfn(pgis_abs)	
FUNCTION	pgis_geometry_polygonize_finalfn(pgis_abs)	
FUNCTION	pgis_geometry_union_finalfn(pgis_abs)	
FUNCTION	point_inside_circle(geometry, double precision, double precision, double precision)	
FUNCTION	pointfromtext(text)	
FUNCTION	pointfromtext(text, integer)	
FUNCTION	pointfromwkb(bytea)	
FUNCTION	pointfromwkb(bytea, integer)	
FUNCTION	pointn(geometry, integer)	
FUNCTION	pointonsurface(geometry)	
FUNCTION	polyfromtext(text)	
FUNCTION	polyfromtext(text, integer)	
FUNCTION	polyfromwkb(bytea)	
FUNCTION	polyfromwkb(bytea, integer)	
FUNCTION	polygonfromtext(text)	
FUNCTION	polygonfromtext(text, integer)	
FUNCTION	polygonfromwkb(bytea)	
FUNCTION	polygonfromwkb(bytea, integer)	
FUNCTION	polygonize_garray(geometry[])	
FUNCTION	populate_geometry_columns()	
FUNCTION	populate_geometry_columns(oid)	
FUNCTION	postgis_full_version()	
FUNCTION	postgis_geos_version()	
FUNCTION	postgis_gist_joinsel(internal, oid, internal, smallint)	
FUNCTION	postgis_gist_sel(internal, oid, internal, integer)	
FUNCTION	postgis_jts_version()	
FUNCTION	postgis_lib_build_date()	
FUNCTION	postgis_lib_version()	
FUNCTION	postgis_proj_version()	
FUNCTION	postgis_scripts_build_date()	
FUNCTION	postgis_scripts_installed()	
FUNCTION	postgis_scripts_released()	
FUNCTION	postgis_uses_stats()	
FUNCTION	postgis_version()	
FUNCTION	probe_geometry_columns()	
FUNCTION	relate(geometry, geometry)	
FUNCTION	relate(geometry, geometry, text)	
FUNCTION	removepoint(geometry, integer)	
FUNCTION	rename_geometry_table_constraints()	
FUNCTION	reverse(geometry)	
FUNCTION	rotate(geometry, double precision)	
FUNCTION	rotatex(geometry, double precision)	
FUNCTION	rotatey(geometry, double precision)	
FUNCTION	rotatez(geometry, double precision)	
FUNCTION	scale(geometry, double precision, double precision)	
FUNCTION	scale(geometry, double precision, double precision, double precision)	
FUNCTION	se_envelopesintersect(geometry, geometry)	
FUNCTION	se_is3d(geometry)	
FUNCTION	se_ismeasured(geometry)	
FUNCTION	se_locatealong(geometry, double precision)	
FUNCTION	se_locatebetween(geometry, double precision, double precision)	
FUNCTION	se_m(geometry)	
FUNCTION	se_z(geometry)	
FUNCTION	segmentize(geometry, double precision)	
FUNCTION	setfactor(chip, real)	
FUNCTION	setpoint(geometry, integer, geometry)	
FUNCTION	setsrid(chip, integer)	
FUNCTION	setsrid(geometry, integer)	
FUNCTION	shift_longitude(geometry)	
FUNCTION	simplify(geometry, double precision)	
FUNCTION	snaptogrid(geometry, double precision)	
FUNCTION	snaptogrid(geometry, double precision, double precision)	
FUNCTION	snaptogrid(geometry, double precision, double precision, double precision, double precision)	
FUNCTION	snaptogrid(geometry, geometry, double precision, double precision, double precision, double precision)	
FUNCTION	spheroid_in(cstring)	
FUNCTION	spheroid_out(spheroid)	
FUNCTION	srid(chip)	
FUNCTION	srid(geometry)	
FUNCTION	st_addbbox(geometry)	
FUNCTION	st_addpoint(geometry, geometry)	
FUNCTION	st_addpoint(geometry, geometry, integer)	
FUNCTION	st_affine(geometry, double precision, double precision, double precision, double precision, double precision, double precision)	
FUNCTION	st_affine(geometry, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision, double precision)	
FUNCTION	st_area(geometry)	
FUNCTION	st_area2d(geometry)	
FUNCTION	st_asbinary(geometry)	
FUNCTION	st_asbinary(geometry, text)	
FUNCTION	st_asewkb(geometry)	
FUNCTION	st_asewkb(geometry, text)	
FUNCTION	st_asewkt(geometry)	
FUNCTION	st_asgeojson(geometry)	
FUNCTION	st_asgeojson(geometry, integer)	
FUNCTION	st_asgeojson(geometry, integer, integer)	
FUNCTION	st_asgeojson(integer, geometry)	
FUNCTION	st_asgeojson(integer, geometry, integer)	
FUNCTION	st_asgeojson(integer, geometry, integer, integer)	
FUNCTION	st_asgml(geometry)	
FUNCTION	st_asgml(geometry, integer)	
FUNCTION	st_asgml(integer, geometry)	
FUNCTION	st_asgml(integer, geometry, integer)	
FUNCTION	st_ashexewkb(geometry)	
FUNCTION	st_ashexewkb(geometry, text)	
FUNCTION	st_askml(geometry)	
FUNCTION	st_askml(geometry, integer)	
FUNCTION	st_askml(integer, geometry)	
FUNCTION	st_askml(integer, geometry, integer)	
FUNCTION	st_assvg(geometry)	
FUNCTION	st_assvg(geometry, integer)	
FUNCTION	st_assvg(geometry, integer, integer)	
FUNCTION	st_astext(geometry)	
FUNCTION	st_azimuth(geometry, geometry)	
FUNCTION	st_bdmpolyfromtext(text, integer)	
FUNCTION	st_bdpolyfromtext(text, integer)	
FUNCTION	st_boundary(geometry)	
FUNCTION	st_box(box3d)	
FUNCTION	st_box(geometry)	
FUNCTION	st_box2d(box3d)	
FUNCTION	st_box2d(geometry)	
FUNCTION	st_box2d_contain(box2d, box2d)	
FUNCTION	st_box2d_contained(box2d, box2d)	
FUNCTION	st_box2d_in(cstring)	
FUNCTION	st_box2d_intersects(box2d, box2d)	
FUNCTION	st_box2d_left(box2d, box2d)	
FUNCTION	st_box2d_out(box2d)	
FUNCTION	st_box2d_overlap(box2d, box2d)	
FUNCTION	st_box2d_overleft(box2d, box2d)	
FUNCTION	st_box2d_overright(box2d, box2d)	
FUNCTION	st_box2d_right(box2d, box2d)	
FUNCTION	st_box2d_same(box2d, box2d)	
FUNCTION	st_box3d(box2d)	
FUNCTION	st_box3d(geometry)	
FUNCTION	st_box3d_in(cstring)	
FUNCTION	st_box3d_out(box3d)	
FUNCTION	st_buffer(geometry, double precision)	
FUNCTION	st_buffer(geometry, double precision, integer)	
FUNCTION	st_build_histogram2d(histogram2d, text, text)	
FUNCTION	st_build_histogram2d(histogram2d, text, text, text)	
FUNCTION	st_buildarea(geometry)	
FUNCTION	st_bytea(geometry)	
FUNCTION	st_cache_bbox()	
FUNCTION	st_centroid(geometry)	
FUNCTION	st_chip_in(cstring)	
FUNCTION	st_chip_out(chip)	
FUNCTION	st_collect(geometry, geometry)	
FUNCTION	st_collect(geometry[])	
FUNCTION	st_collect_garray(geometry[])	
FUNCTION	st_collector(geometry, geometry)	
FUNCTION	st_combine_bbox(box2d, geometry)	
FUNCTION	st_combine_bbox(box3d, geometry)	
FUNCTION	st_compression(chip)	
FUNCTION	st_contains(geometry, geometry)	
FUNCTION	st_containsproperly(geometry, geometry)	
FUNCTION	st_convexhull(geometry)	
FUNCTION	st_coorddim(geometry)	
FUNCTION	st_coveredby(geometry, geometry)	
FUNCTION	st_covers(geometry, geometry)	
FUNCTION	st_create_histogram2d(box2d, integer)	
FUNCTION	st_crosses(geometry, geometry)	
FUNCTION	st_curvetoline(geometry)	
FUNCTION	st_curvetoline(geometry, integer)	
FUNCTION	st_datatype(chip)	
FUNCTION	st_difference(geometry, geometry)	
FUNCTION	st_dimension(geometry)	
FUNCTION	st_disjoint(geometry, geometry)	
FUNCTION	st_distance(geometry, geometry)	
FUNCTION	st_distance_sphere(geometry, geometry)	
FUNCTION	st_distance_spheroid(geometry, geometry, spheroid)	
FUNCTION	st_dropbbox(geometry)	
FUNCTION	st_dump(geometry)	
FUNCTION	st_dumprings(geometry)	
FUNCTION	st_dwithin(geometry, geometry, double precision)	
FUNCTION	st_endpoint(geometry)	
FUNCTION	st_envelope(geometry)	
FUNCTION	st_equals(geometry, geometry)	
FUNCTION	st_estimate_histogram2d(histogram2d, box2d)	
FUNCTION	st_estimated_extent(text, text)	
FUNCTION	st_estimated_extent(text, text, text)	
FUNCTION	st_expand(box2d, double precision)	
FUNCTION	st_expand(box3d, double precision)	
FUNCTION	st_expand(geometry, double precision)	
FUNCTION	st_explode_histogram2d(histogram2d, text)	
FUNCTION	st_exteriorring(geometry)	
FUNCTION	st_factor(chip)	
FUNCTION	st_find_extent(text, text)	
FUNCTION	st_find_extent(text, text, text)	
FUNCTION	st_force_2d(geometry)	
FUNCTION	st_force_3d(geometry)	
FUNCTION	st_force_3dm(geometry)	
FUNCTION	st_force_3dz(geometry)	
FUNCTION	st_force_4d(geometry)	
FUNCTION	st_force_collection(geometry)	
FUNCTION	st_forcerhr(geometry)	
FUNCTION	st_geom_accum(geometry[], geometry)	
FUNCTION	st_geomcollfromtext(text)	
FUNCTION	st_geomcollfromtext(text, integer)	
FUNCTION	st_geomcollfromwkb(bytea)	
FUNCTION	st_geomcollfromwkb(bytea, integer)	
FUNCTION	st_geometry(box2d)	
FUNCTION	st_geometry(box3d)	
FUNCTION	st_geometry(bytea)	
FUNCTION	st_geometry(chip)	
FUNCTION	st_geometry(text)	
FUNCTION	st_geometry_above(geometry, geometry)	
FUNCTION	st_geometry_analyze(internal)	
FUNCTION	st_geometry_below(geometry, geometry)	
FUNCTION	st_geometry_cmp(geometry, geometry)	
FUNCTION	st_geometry_contain(geometry, geometry)	
FUNCTION	st_geometry_contained(geometry, geometry)	
FUNCTION	st_geometry_eq(geometry, geometry)	
FUNCTION	st_geometry_ge(geometry, geometry)	
FUNCTION	st_geometry_gt(geometry, geometry)	
FUNCTION	st_geometry_in(cstring)	
FUNCTION	st_geometry_le(geometry, geometry)	
FUNCTION	st_geometry_left(geometry, geometry)	
FUNCTION	st_geometry_lt(geometry, geometry)	
FUNCTION	st_geometry_out(geometry)	
FUNCTION	st_geometry_overabove(geometry, geometry)	
FUNCTION	st_geometry_overbelow(geometry, geometry)	
FUNCTION	st_geometry_overlap(geometry, geometry)	
FUNCTION	st_geometry_overleft(geometry, geometry)	
FUNCTION	st_geometry_overright(geometry, geometry)	
FUNCTION	st_geometry_recv(internal)	
FUNCTION	st_geometry_right(geometry, geometry)	
FUNCTION	st_geometry_same(geometry, geometry)	
FUNCTION	st_geometry_send(geometry)	
FUNCTION	st_geometryfromtext(text)	
FUNCTION	st_geometryfromtext(text, integer)	
FUNCTION	st_geometryn(geometry, integer)	
FUNCTION	st_geometrytype(geometry)	
FUNCTION	st_geomfromewkb(bytea)	
FUNCTION	st_geomfromewkt(text)	
FUNCTION	st_geomfromtext(text)	
FUNCTION	st_geomfromtext(text, integer)	
FUNCTION	st_geomfromwkb(bytea)	
FUNCTION	st_geomfromwkb(bytea, integer)	
FUNCTION	st_hasarc(geometry)	
FUNCTION	st_hasbbox(geometry)	
FUNCTION	st_height(chip)	
FUNCTION	st_histogram2d_in(cstring)	
FUNCTION	st_histogram2d_out(histogram2d)	
FUNCTION	st_interiorringn(geometry, integer)	
FUNCTION	st_intersection(geometry, geometry)	
FUNCTION	st_intersects(geometry, geometry)	
FUNCTION	st_isclosed(geometry)	
FUNCTION	st_isempty(geometry)	
FUNCTION	st_isring(geometry)	
FUNCTION	st_issimple(geometry)	
FUNCTION	st_isvalid(geometry)	
FUNCTION	st_isvalidreason(geometry)	
FUNCTION	st_length(geometry)	
FUNCTION	st_length2d(geometry)	
FUNCTION	st_length2d_spheroid(geometry, spheroid)	
FUNCTION	st_length3d(geometry)	
FUNCTION	st_length3d_spheroid(geometry, spheroid)	
FUNCTION	st_length_spheroid(geometry, spheroid)	
FUNCTION	st_line_interpolate_point(geometry, double precision)	
FUNCTION	st_line_locate_point(geometry, geometry)	
FUNCTION	st_line_substring(geometry, double precision, double precision)	
FUNCTION	st_linecrossingdirection(geometry, geometry)	
FUNCTION	st_linefrommultipoint(geometry)	
FUNCTION	st_linefromtext(text)	
FUNCTION	st_linefromtext(text, integer)	
FUNCTION	st_linefromwkb(bytea)	
FUNCTION	st_linefromwkb(bytea, integer)	
FUNCTION	st_linemerge(geometry)	
FUNCTION	st_linestringfromwkb(bytea)	
FUNCTION	st_linestringfromwkb(bytea, integer)	
FUNCTION	st_linetocurve(geometry)	
FUNCTION	st_locate_along_measure(geometry, double precision)	
FUNCTION	st_locate_between_measures(geometry, double precision, double precision)	
FUNCTION	st_locatebetweenelevations(geometry, double precision, double precision)	
FUNCTION	st_m(geometry)	
FUNCTION	st_makebox2d(geometry, geometry)	
FUNCTION	st_makebox3d(geometry, geometry)	
FUNCTION	st_makeline(geometry, geometry)	
FUNCTION	st_makeline(geometry[])	
FUNCTION	st_makeline_garray(geometry[])	
FUNCTION	st_makepoint(double precision, double precision)	
FUNCTION	st_makepoint(double precision, double precision, double precision)	
FUNCTION	st_makepoint(double precision, double precision, double precision, double precision)	
FUNCTION	st_makepointm(double precision, double precision, double precision)	
FUNCTION	st_makepolygon(geometry)	
FUNCTION	st_makepolygon(geometry, geometry[])	
FUNCTION	st_max_distance(geometry, geometry)	
FUNCTION	st_mem_size(geometry)	
FUNCTION	st_minimumboundingcircle(geometry)	
FUNCTION	st_minimumboundingcircle(geometry, integer)	
FUNCTION	st_mlinefromtext(text)	
FUNCTION	st_mlinefromtext(text, integer)	
FUNCTION	st_mlinefromwkb(bytea)	
FUNCTION	st_mlinefromwkb(bytea, integer)	
FUNCTION	st_mpointfromtext(text)	
FUNCTION	st_mpointfromtext(text, integer)	
FUNCTION	st_mpointfromwkb(bytea)	
FUNCTION	st_mpointfromwkb(bytea, integer)	
FUNCTION	st_mpolyfromtext(text)	
FUNCTION	st_mpolyfromtext(text, integer)	
FUNCTION	st_mpolyfromwkb(bytea)	
FUNCTION	st_mpolyfromwkb(bytea, integer)	
FUNCTION	st_multi(geometry)	
FUNCTION	st_multilinefromwkb(bytea)	
FUNCTION	st_multilinestringfromtext(text)	
FUNCTION	st_multilinestringfromtext(text, integer)	
FUNCTION	st_multipointfromtext(text)	
FUNCTION	st_multipointfromwkb(bytea)	
FUNCTION	st_multipointfromwkb(bytea, integer)	
FUNCTION	st_multipolyfromwkb(bytea)	
FUNCTION	st_multipolyfromwkb(bytea, integer)	
FUNCTION	st_multipolygonfromtext(text)	
FUNCTION	st_multipolygonfromtext(text, integer)	
FUNCTION	st_ndims(geometry)	
FUNCTION	st_noop(geometry)	
FUNCTION	st_npoints(geometry)	
FUNCTION	st_nrings(geometry)	
FUNCTION	st_numgeometries(geometry)	
FUNCTION	st_numinteriorring(geometry)	
FUNCTION	st_numinteriorrings(geometry)	
FUNCTION	st_numpoints(geometry)	
FUNCTION	st_orderingequals(geometry, geometry)	
FUNCTION	st_overlaps(geometry, geometry)	
FUNCTION	st_perimeter(geometry)	
FUNCTION	st_perimeter2d(geometry)	
FUNCTION	st_perimeter3d(geometry)	
FUNCTION	st_point(double precision, double precision)	
FUNCTION	st_point_inside_circle(geometry, double precision, double precision, double precision)	
FUNCTION	st_pointfromtext(text)	
FUNCTION	st_pointfromtext(text, integer)	
FUNCTION	st_pointfromwkb(bytea)	
FUNCTION	st_pointfromwkb(bytea, integer)	
FUNCTION	st_pointn(geometry)	
FUNCTION	st_pointn(geometry, integer)	
FUNCTION	st_pointonsurface(geometry)	
FUNCTION	st_polyfromtext(text)	
FUNCTION	st_polyfromtext(text, integer)	
FUNCTION	st_polyfromwkb(bytea)	
FUNCTION	st_polyfromwkb(bytea, integer)	
FUNCTION	st_polygon(geometry, integer)	
FUNCTION	st_polygonfromtext(text)	
FUNCTION	st_polygonfromtext(text, integer)	
FUNCTION	st_polygonfromwkb(bytea)	
FUNCTION	st_polygonfromwkb(bytea, integer)	
FUNCTION	st_polygonize(geometry[])	
FUNCTION	st_polygonize_garray(geometry[])	
FUNCTION	st_postgis_gist_joinsel(internal, oid, internal, smallint)	
FUNCTION	st_postgis_gist_sel(internal, oid, internal, integer)	
FUNCTION	st_relate(geometry, geometry)	
FUNCTION	st_relate(geometry, geometry, text)	
FUNCTION	st_removepoint(geometry, integer)	
FUNCTION	st_reverse(geometry)	
FUNCTION	st_rotate(geometry, double precision)	
FUNCTION	st_rotatex(geometry, double precision)	
FUNCTION	st_rotatey(geometry, double precision)	
FUNCTION	st_rotatez(geometry, double precision)	
FUNCTION	st_scale(geometry, double precision, double precision)	
FUNCTION	st_scale(geometry, double precision, double precision, double precision)	
FUNCTION	st_segmentize(geometry, double precision)	
FUNCTION	st_setfactor(chip, real)	
FUNCTION	st_setpoint(geometry, integer, geometry)	
FUNCTION	st_setsrid(geometry, integer)	
FUNCTION	st_shift_longitude(geometry)	
FUNCTION	st_simplify(geometry, double precision)	
FUNCTION	st_simplifypreservetopology(geometry, double precision)	
FUNCTION	st_snaptogrid(geometry, double precision)	
FUNCTION	st_snaptogrid(geometry, double precision, double precision)	
FUNCTION	st_snaptogrid(geometry, double precision, double precision, double precision, double precision)	
FUNCTION	st_snaptogrid(geometry, geometry, double precision, double precision, double precision, double precision)	
FUNCTION	st_spheroid_in(cstring)	
FUNCTION	st_spheroid_out(spheroid)	
FUNCTION	st_srid(chip)	
FUNCTION	st_srid(geometry)	
FUNCTION	st_startpoint(geometry)	
FUNCTION	st_summary(geometry)	
FUNCTION	st_symdifference(geometry, geometry)	
FUNCTION	st_symmetricdifference(geometry, geometry)	
FUNCTION	st_text(boolean)	
FUNCTION	st_text(geometry)	
FUNCTION	st_touches(geometry, geometry)	
FUNCTION	st_transform(geometry, integer)	
FUNCTION	st_translate(geometry, double precision, double precision)	
FUNCTION	st_translate(geometry, double precision, double precision, double precision)	
FUNCTION	st_transscale(geometry, double precision, double precision, double precision, double precision)	
FUNCTION	st_union(geometry, geometry)	
FUNCTION	st_union(geometry[])	
FUNCTION	st_unite_garray(geometry[])	
FUNCTION	st_width(chip)	
FUNCTION	st_within(geometry, geometry)	
FUNCTION	st_wkbtosql(bytea)	
FUNCTION	st_wkttosql(text)	
FUNCTION	st_x(geometry)	
FUNCTION	st_xmax(box3d)	
FUNCTION	st_xmin(box3d)	
FUNCTION	st_y(geometry)	
FUNCTION	st_ymax(box3d)	
FUNCTION	st_ymin(box3d)	
FUNCTION	st_z(geometry)	
FUNCTION	st_zmax(box3d)	
FUNCTION	st_zmflag(geometry)	
FUNCTION	st_zmin(box3d)	
FUNCTION	startpoint(geometry)	
FUNCTION	summary(geometry)	
FUNCTION	symdifference(geometry, geometry)	
FUNCTION	symmetricdifference(geometry, geometry)	
FUNCTION	text(boolean)	
FUNCTION	text(geometry)	
FUNCTION	touches(geometry, geometry)	
FUNCTION	transform(geometry, integer)	
FUNCTION	transform_geometry(geometry, text, text, integer)	
FUNCTION	translate(geometry, double precision, double precision)	
FUNCTION	translate(geometry, double precision, double precision, double precision)	
FUNCTION	transscale(geometry, double precision, double precision, double precision, double precision)	
FUNCTION	unite_garray(geometry[])	
FUNCTION	unlockrows(text)	
FUNCTION	update_geometry_stats()	
FUNCTION	update_geometry_stats(character varying, character varying)	
FUNCTION	updategeometrysrid(character varying, character varying, character varying, character varying, integer)	
FUNCTION	updategeometrysrid(character varying, character varying, character varying, integer)	
FUNCTION	updategeometrysrid(character varying, character varying, integer)	
FUNCTION	width(chip)	
FUNCTION	within(geometry, geometry)	
FUNCTION	x(geometry)	
FUNCTION	xmax(box3d)	
FUNCTION	xmin(box3d)	
FUNCTION	y(geometry)	
FUNCTION	ymax(box3d)	
FUNCTION	ymin(box3d)	
FUNCTION	z(geometry)	
FUNCTION	zmax(box3d)	
FUNCTION	zmflag(geometry)	
FUNCTION	zmin(box3d)	
OPERATOR	&&	
OPERATOR	&<	
OPERATOR	&<|	
OPERATOR	&>	
OPERATOR	<	
OPERATOR	<<	
OPERATOR	<<|	
OPERATOR	<=	
OPERATOR	=	
OPERATOR	>	
OPERATOR	>=	
OPERATOR	>>	
OPERATOR	@	
OPERATOR	|&>	
OPERATOR	|>>	
OPERATOR	~	
OPERATOR	~=	
OPERATORCLASS	btree_geometry_ops	
OPERATORCLASS	gist_geometry_ops	
TYPE	box2d	
TYPE	box3d	
TYPE	chip	
TYPE	geometry	
TYPE	geometry_dump	
TYPE	histogram2d	
TYPE	pgis_abs	
TYPE	spheroid	
TABLE       geometry_columns        
TABLE       spatial_ref_sys 
CONSTRAINT	geometry_columns_pk
CONSTRAINT	spatial_ref_sys_pkey
SHELLTYPE	box2d
SHELLTYPE	box3d
SHELLTYPE	chip
SHELLTYPE	geometry
SHELLTYPE	pgis_abs
SHELLTYPE	spheroid
