# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python binding implicit animation example
#
# Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: Florian Boucault <florian@fluendo.com>

import sys, pgm, gobject
from pgm.timing import implicit, controller, ticker

# Update animations
def on_update_pass(viewport, tck):
    tck.tick()

# Handles the key presses
def on_key_press(viewport, event, drawable, help):
    if event.keyval == pgm.keysyms.q or event.keyval == pgm.keysyms.Escape:
        pgm.main_quit()

    elif event.keyval == pgm.keysyms.Right:
        drawable.x += 0.5

    elif event.keyval == pgm.keysyms.Left:
        drawable.x -= 0.5

    elif event.keyval == pgm.keysyms.Down:
        drawable.y += 0.5

    elif event.keyval == pgm.keysyms.Up:
        drawable.y -= 0.5

    elif event.keyval == pgm.keysyms.o:
        if drawable.opacity != 0:
            drawable.opacity = 0
        else:
            drawable.opacity = 255

    elif event.keyval == pgm.keysyms.h:
        if help.opacity != 0:
            help.opacity = 0
        else:
            help.opacity = 220

# Terminate the mainloop on a delete event
def on_delete(viewport, event):
    pgm.main_quit()

def main(args):
    # GObject threads initialization
    gobject.threads_init()

    # OpenGL viewport creation
    gl = pgm.viewport_factory_make('opengl')
    gl.title = 'Implicit animation'

    # Canvas, image drawables creation
    cvs = pgm.Canvas()
    img = pgm.Image()

    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(cvs)

    # Image properties
    img.position = (1.0, 1.5, 0.0)
    img.size = (1.0, 1.0)
    img.bg_a = 0
    img.set_from_file("examples/pictures/fluendo.png")

    # A drawable needs to be shown
    img.show()

    # Add it to the middle layer of the canvas
    cvs.add(pgm.DRAWABLE_MIDDLE, img)

    # Set the ticker used by controllers
    tck = ticker.Ticker()
    controller.Controller.set_ticker(tck)

    # Implicity animated image wrapping img
    animated_img = implicit.AnimatedObject(img)
    animated_img.mode = implicit.REPLACE

    # Help drawable creation
    help = pgm.Text("<b><tt>Implicit animations are great !</tt></b>\n\n"   \
                    "Keys:\n\n" \
                    "'arrows':\t move the drawable\n" \
                    "'o':\t set the drawable's opacity to 0 or 255\n" \
                    "'h':\t get rid of the help\n")

    # Help text properties tweaking
    help.position = (0.8, 1.0, 0.0)
    help.size = (2.0, 1.0)
    help.font_height = 0.1
    help.fg_color = (220, 220, 220, 255)
    help.bg_color = (20, 20, 20, 255)
    help.opacity = 0

    # A drawable needs to be shown
    help.show()

    # Add it to the middle layer of the canvas
    cvs.add(pgm.DRAWABLE_MIDDLE, help)

    # Implicity animated text wrapping help
    animated_help = implicit.AnimatedObject(help)
    animated_help.mode = implicit.REPLACE
    animated_help.opacity = 220

    # Let's start the mainloop
    gl.connect('delete-event', on_delete)
    gl.connect('key-press-event', on_key_press, animated_img, animated_help)
    gl.connect('update-pass', on_update_pass, tck)
    gl.show()
    pgm.main()

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))
