/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.common.ui.properties.internal.view;

import org.eclipse.jface.util.Assert;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.IContributedContentsView;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.wst.common.ui.properties.internal.provisional.IDynamicTabItem;
import org.eclipse.wst.common.ui.properties.internal.provisional.IDynamicTabListContentProvider;
import org.eclipse.wst.common.ui.properties.internal.provisional.IDynamicTabListSection;
import org.eclipse.wst.common.ui.properties.internal.provisional.ITabSelectionListener;
import org.eclipse.wst.common.ui.properties.internal.provisional.TabbedPropertySheetPage;


/**
 * An implementation of the content provider for the tabbed property sheet
 * page's list of tabs. This implementation dynamically overrides what tabs are
 * displayed, rather than using the default list of tabs read from the
 * PropertyTab extension point.
 * 
 * @author Anthony Hunter <a
 *         href="mailto:anthonyh@ca.ibm.com">anthonyh@ca.ibm.com </a>
 */
public class DynamicTabListContentProvider
	implements IDynamicTabListContentProvider, ITabSelectionListener {

	private IWorkbenchPart currentPart;

	private TabbedPropertyRegistry tabbedPropertyRegistry;

	private TabbedPropertySheetPage tabbedPropertySheetPage;

	private TabbedPropertyViewer tabbedPropertyViewer;

	/**
	 * Constructor for DynamicTabListContentProvider
	 */
	public DynamicTabListContentProvider(
			TabbedPropertyRegistry tabbedPropertyRegistry) {
		this.tabbedPropertyRegistry = tabbedPropertyRegistry;
	}

	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement) {
		if (tabbedPropertySheetPage.getCurrentTab() == null) {
			/*
			 * In this case, return the default list of tabs from the registry.
			 * The contributor will not have had a chance to load and override
			 * the tabs.
			 */
			return tabbedPropertyRegistry.getTabDescriptors(currentPart,
				(ISelection) inputElement);
		}
		return getOverrideTabs(inputElement);
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose() {
		stopListening();
		this.tabbedPropertyViewer = null;
		this.currentPart = null;
		this.tabbedPropertySheetPage = null;
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
	 *      java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if (tabbedPropertyViewer == null) {
			Assert.isTrue(viewer instanceof TabbedPropertyViewer);
			init((TabbedPropertyViewer) viewer);
		}
	}

	/**
	 * Initialize the content provider for a tabbed property viewer.
	 * 
	 * @param tabbedPropertyViewer
	 *            a tabbed property viewer.
	 */
	private void init(TabbedPropertyViewer newTabbedPropertyViewer) {
		this.tabbedPropertyViewer = newTabbedPropertyViewer;
		currentPart = tabbedPropertyViewer.getWorkbenchPart();
		if (currentPart.getAdapter(IPropertySheetPage.class) != null) {
			tabbedPropertySheetPage = (TabbedPropertySheetPage) currentPart
				.getAdapter(IPropertySheetPage.class);
		} else {
			/*
			 * Is the part is a IContributedContentsView for the contributor,
			 * for example, outline view.
			 */
			IContributedContentsView view = (IContributedContentsView) currentPart
				.getAdapter(IContributedContentsView.class);
			if (view != null) {
				IWorkbenchPart part = view.getContributingPart();
				if (part != null) {
					tabbedPropertySheetPage = (TabbedPropertySheetPage) part
						.getAdapter(IPropertySheetPage.class);
				}
			}
		}
		Assert
			.isNotNull(
				tabbedPropertySheetPage,
				"DynamicTabListContentProvider could not find the TabbedPropertySheetPage for the active part");//$NON-NLS-1$
		startListening();
	}

	/**
	 * Start listening to tab selection change events.
	 */
	private void startListening() {
		tabbedPropertySheetPage.addTabSelectionListener(this);
	}

	/**
	 * Stop listening to tab selection change events.
	 */
	private void stopListening() {
		tabbedPropertySheetPage.removeTabSelectionListener(this);
	}

	/**
	 * @see org.eclipse.wst.common.ui.properties.internal.provisional.ITabSelectionListener#tabSelected(org.eclipse.wst.common.ui.properties.internal.provisional.internal.view.TabDescriptor)
	 */
	public void tabSelected(TabDescriptor tabDescriptor) {
		IDynamicTabListSection section = getDynamicTabListSection();
		Assert.isNotNull(section);
		int selectedIndex = tabbedPropertyViewer.getSelectionIndex();
		section.selectTab(selectedIndex);
	}

	/**
	 * set the selected tab from the list of tabs provided by the section.
	 */
	private void setSelectedTab() {
		TabDescriptor currentSelectedTabInList = null;
		IStructuredSelection selectionFromList = (IStructuredSelection) tabbedPropertyViewer
			.getSelection();
		if (!selectionFromList.equals(StructuredSelection.EMPTY)) {
			currentSelectedTabInList = (TabDescriptor) selectionFromList
				.getFirstElement();
		}
		IDynamicTabListSection section = getDynamicTabListSection();
		if (section == null) {
			if (currentSelectedTabInList == null) {
				/*
				 * make sure there is a selected tab in the list.
				 */
				TabDescriptor newSelectedTab = (TabDescriptor) tabbedPropertyViewer
					.getElementAt(0);
				if (newSelectedTab != null) {
					tabbedPropertyViewer.setSelection(new StructuredSelection(
						newSelectedTab), true);
				}
			}
			return;
		}
		IDynamicTabItem[] dynamicTabItems = section.getTabs();
		if (dynamicTabItems == null) {
			/*
			 * if the section does not provide overridden tabs, return.
			 */
			return;
		}
		int selectedTabIndex = -1;
		for (int i = 0; i < dynamicTabItems.length; i++) {
			if (dynamicTabItems[i].isSelected()) {
				selectedTabIndex = i;
			}
		}
		if (currentSelectedTabInList == null
			|| !currentSelectedTabInList.getText().equals(
				dynamicTabItems[selectedTabIndex].getText())) {
			TabDescriptor newSelectedTab = (TabDescriptor) tabbedPropertyViewer
				.getElementAt(selectedTabIndex);
			tabbedPropertyViewer.setSelection(new StructuredSelection(
				newSelectedTab), true);
		}
	}

	/**
	 * Get the list of tabs to display in the tabbed property sheet page.
	 * 
	 * @param inputElement
	 *            The current input element.
	 * @return the list of tabs.
	 */
	private TabDescriptor[] getOverrideTabs(Object inputElement) {
		TabDescriptor tabDescriptors[] = tabbedPropertyRegistry
			.getTabDescriptors(currentPart, (ISelection) inputElement);
		if (tabDescriptors == TabbedPropertyRegistry.EMPTY_DESCRIPTOR_ARRAY) {
			/*
			 * We clone the dynamic tabs from the single tab in the registry. If
			 * the registry has no tabs for the selection, return.
			 */
			return tabDescriptors;
		}
		IDynamicTabListSection section = getDynamicTabListSection();
		Assert.isNotNull(section);
		IDynamicTabItem[] dynamicTabItems = section.getTabs();
		if (dynamicTabItems == null) {
			/*
			 * if the section does not provide overridden tabs, return the
			 * default tabs from the registry.
			 */
			return tabDescriptors;
		}
		TabDescriptor[] overrideTabDescriptors = new TabDescriptor[dynamicTabItems.length];
		TabDescriptor target = tabDescriptors[0];
		for (int i = 0; i < dynamicTabItems.length; i++) {
			overrideTabDescriptors[i] = (TabDescriptor) target.clone();
			overrideTabDescriptors[i].setLabel(dynamicTabItems[i].getText());
			overrideTabDescriptors[i].setImage(dynamicTabItems[i].getImage());
			overrideTabDescriptors[i].setIndented(dynamicTabItems[i]
				.isIndented());
			overrideTabDescriptors[i].setSelected(dynamicTabItems[i]
				.isSelected());
			//System.out.print("override " + i + " [" +
			// dynamicTabItems[i].getText() + "]");
			//if (dynamicTabItems[i].isSelected()) {
			//	System.out.print(" selected");
			//}
			//System.out.println("");
		}
		return overrideTabDescriptors;
	}

	/**
	 * @see org.eclipse.wst.common.ui.properties.internal.provisional.IDynamicTabListContentProvider#overrideTabs()
	 */
	public void overrideTabs() {
		stopListening();
		Object input = tabbedPropertyViewer.getInput();
		tabbedPropertyViewer.setInput(input);
		setSelectedTab();
		startListening();
	}

	/**
	 * Get the section implementation that provides the list of tabs. In our
	 * implementation, all the sections provide the list of tabs, so we select
	 * the first section from the tab descriptor.
	 * 
	 * @return the section.
	 */
	private IDynamicTabListSection getDynamicTabListSection() {
		Tab tab = tabbedPropertySheetPage.getCurrentTab();
		if (tab != null) {
			return (IDynamicTabListSection) tab.getSectionAtIndex(0);
		}
		return null;
	}
}
