/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal;

import org.eclipse.core.resources.IStorage;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;

/**
 * This class implements the <code>ISQLEditorInput</code> interface on a
 * <code>IStorageEditorInput</code> base.  It is provided as a convenience to 
 * callers of the SQL Editor who want to open the editor on an input that 
 * isn't a file.
 */
public class SQLEditorStorageEditorInput implements IStorageEditorInput, ISQLEditorInput, IPersistableElement {

    /** The connection information for this SQL editor input. */
    private ConnectionInfo fConnInfo;
    /** The database catalog associated with this object. */
    private Database fDatabase;
    /** The name of the default schema to use with the associated database.  */
    private String fDefaultSchemaName;
    /** The name of ths editor input. */
    private String fName;
    /** The storage object used by this editor input. */
    private IStorage fStorage;

    /**
     * Constructs an instance of this class with the given string as the editor
     * input source.
     * 
     * @param storageSource the editor input source string
     */
    public SQLEditorStorageEditorInput( String storageSource ) {
        this( new SQLEditorStorage( storageSource ) );
    }

    /**
     * Constructs an instance of this class with the given <code>IStorage</code>
     * object as the editor input source.
     * 
     * @param storage the storage object for this editor input
     */
    public SQLEditorStorageEditorInput( IStorage storage ) {
		if (storage == null) {
			throw new IllegalArgumentException();
        }
        setStorage( storage );
        setName( storage.getName() );
        setConnectionInfo( null );
    }    

    /**
     * Gets whether the editor input exists.
     * 
     * @see org.eclipse.ui.IEditorInput#exists()
     */
    public boolean exists() {
        if (fStorage != null)
            return true;
        
        return false;
    }

    /**
     * Gets an object which is an instance of the given class associated with
     * this object. Returns <code>null</code> if no such object can be found.
     * This default implementation returns null.
     * 
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter( Class adapter ) {
        return null;
    }

    /**
     * Gets the <code>ConnectionInfo</code> object for this editor input.
     * 
     * @return the current <code>ConnectionInfo</code> object
     */
    public ConnectionInfo getConnectionInfo() {
        return fConnInfo;
    }

	/**
	 * Gets the <code>Database</code> object associated with this input.
	 * 
	 * @return the <code>Database</code> object associated with this input
	 */
	public Database getDatabase() {
	    return fDatabase;
	}
	
	/**
     * Gets the default schema name to use with the connection or database
     * associated with this input.
     * 
     * @return the default schema name to use with this input, or null if none
     */
    public String getDefaultSchemaName() {
        return fDefaultSchemaName;
    }

    /**
     * Gets the image descriptor for this input.
     * 
     * @see org.eclipse.ui.IEditorInput#getImageDescriptor()
     */
    public ImageDescriptor getImageDescriptor() {
        return null;
    }

    /**
     * Gets the name of this editor input for display purposes.
     * 
     * @see org.eclipse.ui.IEditorInput#getName()
     */
    public String getName() {
        return fName;
    }

    /**
     * Gets an object that can be used to save the state of this editor input.
     * 
     * @see org.eclipse.ui.IEditorInput#getPersistable()
     */
    public IPersistableElement getPersistable() {
        //return this;
        return null;
    }

    /**
     * Gets the underlying <code>IStorage</code> object. The default storage
     * object is implemented as a <code>InputStream</code>.)
     * 
     * @see org.eclipse.ui.IStorageEditorInput#getStorage()
     */
    public IStorage getStorage() {
        return fStorage;
    }

    /**
     * Gets the tool tip text for this editor input.
     * 
     * @see org.eclipse.ui.IEditorInput#getToolTipText()
     */
    public String getToolTipText() {        
        StringBuffer sb = new StringBuffer(30);
        if (fConnInfo != null) {
        	sb.append(fConnInfo.getName());
            sb.append("/");
        }
        sb.append(fName);
        return sb.toString();        
    }

    /**
     * Returns the id of the element factory which should be used to re-create this
     * object.
     * 
     * @see org.eclipse.ui.IPersistableElement#getFactoryId()
     */
    public String getFactoryId() {
        return SQLEditorInputFactory.ID_FACTORY;
    }

    /**
     * Saves the state of the object in the given memento.
     * 
     * @param memento the storage area for object's state
     * @see org.eclipse.ui.IPersistableElement#saveState(org.eclipse.ui.IMemento)
     */
    public void saveState(IMemento memento) {
        SQLEditorInputFactory.saveState( memento, this );
    }

    /**
     * Sets the <code>ConnectionInfo</code> object for this editor input to
     * the given object.
     * 
     * @param connInfo the <code>ConnectionInfo</code> object to set
     */
    public void setConnectionInfo( ConnectionInfo connInfo ) {
        fConnInfo = connInfo;
    }

	/**
	 * Sets the <code>Database</code> object associated with this input to the
	 * given object.
	 * 
	 * @param database the <code>Database</code> object to set
	 */
	public void setDatabase( Database database ) {
	    fDatabase = database;
	}

	/**
     * Sets the default schema name to use with the connection or database
     * associated with this input.
     * 
     * @param schemaName the default schema name to set
     */
    public void setDefaultSchemaName( String schemaName ) {
        fDefaultSchemaName = schemaName;
    }

    /**
     * Sets the name of this editor input to the given name.
     * 
     * @param name the name to set
     */
    public void setName( String name ) {
        fName = name;
    }

    /**
     * Sets the underlying <code>IStorage</code> object.
     * 
     * @param storage the storage object to use
     */
    public void setStorage( IStorage storage ) {
        fStorage = storage;
    }

} // end class
