#!/bin/sh
#
# Automatic generation of driver files for French Cursive.
#
#  Emmanuel Beffara, 2001--2004. This script is covered by the GPL licence,
# and is part of the French Cursive font sources.

calc()
{
	( echo "scale=3"; echo $* ) | bc -l
}

make_driver()
{
	DESCR=""    # textual description
	IDENT=""    # font identifier
	SIZE=$1     # specific size in points

	EX=0        # value of ex# in sharp points
	THIN=0.2    # thickness of thin drawings
	MED=0.4     # thickness of medium drawings
	THICK=0.6   # thickness of thick drawings
	DOT=0.8     # diameter of dots

	WIDTH=0.8   # base width
	SLANT=0     # slant factor

	ACCW=0.7    # relative width of accents

	FIXED=false # fixed-thickness version ?

	shift
	while [ "$#" != 0 ]; do
		case $1 in
		bold)	IDENT="${IDENT}B"; MED=0.55; THICK=0.8; DOT=1.25;
			DESCR="${DESCR}bold ";;
		callig)	IDENT="${IDENT}C"; MED=0.3; THICK=0.8;
			DESCR="${DESCR}calligraphic ";;
		ext)	IDENT="${IDENT}X"; WIDTH=0.9;
			DESCR="${DESCR}extended ";;
		slant)	IDENT="${IDENT}SL"; SLANT=0.3;
			DESCR="${DESCR}slanted ";;
		fixed)	IDENT="${IDENT}F"; FIXED=true;
			THIN=$MED; THICK=$MED; DOT=`calc "2*$MED"`;
			DESCR="${DESCR}fixed-thickness ";;
		esac
		shift
	done

	# The value of ex# is computed the same way as for Computer Modern:
	EX=`calc "$SIZE*15.5"`/36

	# The base width is multiplied by a scale factor related to height
	WIDTH=`calc "$WIDTH/e(l($SIZE/10)/6)"`

	# Correct the thickness values according to the size:
	THIN=`calc "$THIN*$SIZE/10"`
	MED=`calc "$MED*$SIZE/10"`
	THICK=`calc "$THICK*$SIZE/10"`
	DOT=`calc "$DOT*$SIZE/10"`

	# The identifier is "FRC" followed by the letters above (or "R")
	IDENT="FRC${IDENT:-R}"

	# And here we go...

	cat <<EOF
%%% This file is part of the French Cursive font definition.
%%% This is a parameter file for the ${DESCR:-regular }variant

font_identifier := "$IDENT";
font_size $SIZE pt#;

mode_setup;

ex# := $EX pt#;
med# := $MED pt#;
thin# := $THIN pt#;
thick# := $THICK pt#;
dot_size# := $DOT pt#;

base_width := $WIDTH;
slant := $SLANT;
accent_wd := $ACCW;
link_width := 0.13;
straight_ascend := 2;
loop_ascend := 2.5;
uc_ascend := 2.5;
straight_descend := 1.2;
loop_descend := 1.5;
dot_height := 1.4;
accent_bot := 1.2;
accent_top := 1.6;
cedilla_dp = 0.2;

EOF

	# When using a fixed-thickness version, we have to add special code to use
	# the "draw" macro instead of "penstroke":

	$FIXED && cat <<EOF
def penstroke text t =
  begingroup
    save e; def e = enddef;
    draw t withpen pencircle scaled med;
  endgroup
enddef;

EOF

	# And the final line:
	echo "input frcursive"
}


make_series()
{
	local NAME=$1
	shift

	local OPTS=""
	while [ -n "$1" -a "$1" != "at" ]; do
		OPTS="$OPTS $1"
		shift
	done

	if [ -n "$1" ]; then
		shift
		SIZES="$*"
	else
		echo "syntax: ./mkdrv.sh <base name> <modifiers> at <sizes>"
		exit 1
	fi

	local SIZE
	for SIZE in $SIZES; do
		local RS=$(echo "x=$SIZE;y=x-x%1;if(x%1>.5)y+1else y" | bc)
		local FN="$NAME${RS%%.*}.mf"
		echo "making $FN"
		make_driver $SIZE $OPTS > $FN
	done
}

if [ "$#" = 0 ]; then
	cat <<EOF
The syntax is one of:
    $(basename $0) <base name> <modifiers> at <sizes>
    $(basename $0) default

The second form produces a set of standard variants and sizes.

The modifiers are words among:
    bold   = thicker stems and hairlines
    callig = more contrasted stems and hairlines
    fixed  = fixed thickness
    ext    = slightly extended width
    slant  = well, slanted...

The sizes are expressed in points, they can be non-integer. The file names
produced are of the form <base name><integer part of size>.mf.
EOF
elif [ "$1" = default ]; then
	make_series frcr                   at 7 8 9 10 10.95 12 14.4 17.28
	make_series frcsl   slant          at 7 8 9 10 10.95 12 14.4 17.28
	make_series frcbx   bold ext       at 7 8 9 10 10.95 12 14.4 17.28
	make_series frcslbx slant bold ext at 7 8 9 10 10.95 12 14.4 17.28
	make_series frcc    callig         at 10 10.95 12 14.4
	make_series frcf    fixed          at 10 10.95 12 14.4
else
	make_series $@
fi
