/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.data.internal.ui.editor;

import org.eclipse.ui.IActionFilter;
import org.eclipse.wst.rdb.data.internal.core.editor.IRowData;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;

/**
 * Represents a selected cell of the Table Data editor. 
 * This class is used to expose the table data editor selection, 
 * and to allow adding context actions using an object contribution.
 * 
 * The TableDataCell contains the row object and the column index.
 * If the cell corresponds to an actual table value, the row is an IRowTable data.
 * If the cell cell is part of the 'insertion row' then the row is just a dummy Object.
 * 
 * The actual cell value can be accessed using getValue().
 * The class also implements IActionFilter so it can be used for object contrinution enablement.
 * 
 * @author groux
 */
public class TableDataCell implements IActionFilter {

	protected TableDataEditor editor;
    protected Object row;
    protected int col;
    
    public TableDataCell(TableDataEditor editor, Object row, int col)
    {
    	this.editor = editor;
        this.row = row;
        this.col = col;
    }

    public int getCol() {
		return col;
	}

    public Object getRow() {
		return row;
	}
    
    public TableDataEditor getEditor() {
		return editor;
	}

    /**
     * Returns the actual value of the cell (may be null) or null if the cell is part of the insertion row.
     * @return
     */
    public Object getValue() {
    	if (row instanceof IRowData)
    		return ((IRowData)row).getValue(col);
    	else
    		return null;
    }
    

    /**
     * Tests whether the specific value matches the state of the cell.
     * The following properties are supported:
     *  - vendor: Product name of the underlying database
     *  - version: Version name of the underlying database
     *  - dataType: Data type name of the column (precision/scale/length are not part of the name)
     *  - nullable: true if the column is nullable, false otherwise
     *  - insertionCell: true if the cell is part of the insertion row
     */
	public boolean testAttribute(Object target, String name, String value) {
		
		if (name.equals("vendor")) { //$NON-NLS-1$
			String vendor = editor.getSqlTable().getSchema().getDatabase().getVendor();
			return value.equals(vendor);
		} else if (name.equals("version")) { //$NON-NLS-1$
			String version = editor.getSqlTable().getSchema().getDatabase().getVersion();
			return value.equals(version);
		} else if (name.equals("dataType")) {//$NON-NLS-1$
			Column sqlCol = (Column) editor.getSqlTable().getColumns().get(col);
			return value.equals(sqlCol.getDataType().getName());
		} else if (name.equals("nullable")) { //$NON-NLS-1$
			Column sqlCol = (Column) editor.getSqlTable().getColumns().get(col);
			return new Boolean(value).booleanValue() == sqlCol.isNullable();
		} else if (name.equals("insertionCell")) { //$NON-NLS-1$	
			boolean b = row instanceof IRowData;
			return new Boolean(value).booleanValue()!=b;
		} else
			return false;
	}

}
