/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.derby.internal.ui.connection.dialogs;

import java.io.File;

import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.wst.rdb.connection.internal.ui.factories.RSCCoreUIWidgetFactory;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionInformation;
import org.eclipse.wst.rdb.connection.internal.ui.providers.IServerConnectionUIProvider;
import org.eclipse.wst.rdb.connection.internal.ui.wizards.NewConnectionWizard;
import org.eclipse.wst.rdb.derby.internal.ui.util.ResourceLoader;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;


/**
 * @author ljulien
 */
public class DerbyNewConnectionProvider implements IServerConnectionUIProvider, Listener
{
    protected static final RSCCoreUIWidgetFactory factory = RSCCoreUIWidgetFactory.INSTANCE;

    protected static final ResourceLoader resource = ResourceLoader.INSTANCE;
    protected static final String CUI_NEWCW_DBLOCATION_LBL_UI_ = resource.queryString("CUI_NEWCW_DBLOCATION_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_DBBROWSE_BTN_UI_ = resource.queryString("CUI_NEWCW_DBBROWSE_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_CREATEDB_BTN_UI_ = resource.queryString("CUI_NEWCW_CREATEDB_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_UPGRADEDB_BTN_UI_ = resource.queryString("CUI_NEWCW_UPGRADEDB_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_ = resource.queryString("CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_ = resource.queryString("CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_JARBROWSE_BTN_UI_ = resource.queryString("CUI_NEWCW_JARBROWSE_BTN_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_CONNECTIONURL_LBL_UI_ = resource.queryString("CUI_NEWCW_CONNECTIONURL_LBL_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_DBLOCATION_CLOUDSCAPE_UI_ = resource.queryString("CUI_NEWCW_DBLOCATION_CLOUDSCAPE_UI_"); //$NON-NLS-1$
    protected static final String CUI_NEWCW_DEFDBNAME_VAL_UI_ = resource.queryString("CUI_NEWCW_DEFDBNAME_VAL_UI_"); //$NON-NLS-1$
    
    protected static final String CREATE_EQUALS_TRUE_TEXT = "create=true"; //$NON-NLS-1$
    protected static final String UPGRADE_EQUALS_TRUE_TEXT = "upgrade=true"; //$NON-NLS-1$
    protected static final String DRIVER_CLASS_NAME = "org.apache.derby.jdbc.EmbeddedDriver"; //$NON-NLS-1$
   
    protected Composite parentComposite;
    protected Label databaseLabel;
    protected Combo databaseCombo;
    protected Button browseDBLocation;
    protected Button createCheck;
    protected boolean displayUpgrade = false;
    protected Button upgradeCheck;
    protected Label driverLabel;
    protected Combo driverCombo;
    protected Text driverDisplay;
    protected Label locationLabel;
    protected Combo locationCombo;
    protected Button browseLocation;
    protected Label urlLabel;
    protected Text urlDisplay;
    protected WizardPage parentPage;

    protected IServerConnectionInformation connectionInformation;
    protected DatabaseDefinition definition;
    protected String databaseName;
    
    protected void refresh()
    {
        this.databaseName = null;
    }

    public boolean determinePageCompletion()
    {
        return validateControl();
    }

    protected void setConnectionInformation()
    {
        connectionInformation.setDriverClassName(DRIVER_CLASS_NAME);
        connectionInformation.setDatabaseName(databaseName);
        connectionInformation.setLoadingPath(locationCombo.getText());
        if (determinePageCompletion()) {
            connectionInformation.setInformationComplete(true);
        }
    }

    protected boolean validateControl()
    {
        if (databaseCombo.getText() == null || databaseCombo.getText().equals("")) //$NON-NLS-1$
        {
            parentPage.setErrorMessage(NewConnectionWizard.getResourceString("CUI_NEWCW_VALIDATE_DBLOCATION_REQ_UI_")); //$NON-NLS-1$
            return false;
        }
        else if (locationCombo != null && locationCombo.getText().trim().length() < 1)
        {
            parentPage.setErrorMessage(NewConnectionWizard.getResourceString("CUI_NEWCW_VALIDATE_CLASSLOCATION_REQ_UI_")); //$NON-NLS-1$
            return false;
        }
        return true;
    }

    protected void initialize()
    {
        driverDisplay.setText(DRIVER_CLASS_NAME);
        this.connectionInformation.setDriverClassName(DRIVER_CLASS_NAME);

        updateURL();
        addListeners();

        determinePageCompletion();
    }
    
    protected void updateURL()
    {
        String url = "jdbc:derby:" + databaseCombo.getText(); //$NON-NLS-1$
        url += !createCheck.getSelection() ? "" : ";create=true"; //$NON-NLS-1$ //$NON-NLS-2$
        if (displayUpgrade)
            url += !upgradeCheck.getSelection() ? "" : ";upgrade=true"; //$NON-NLS-1$ //$NON-NLS-2$
        urlDisplay.setText(url);
        connectionInformation.setURL(url);
    }
    
    protected String getDirectory(String message)
    {
        DirectoryDialog dialog = new DirectoryDialog(parentPage.getWizard().getContainer().getShell(), SWT.SINGLE);
        dialog.setMessage(message);
        dialog.setFilterPath(databaseCombo.getText().trim());
        return dialog.open();
    }

    public void addListeners()
    {
        locationCombo.addListener(SWT.Modify, this);
        browseLocation.addListener(SWT.Selection, this);
        databaseCombo.addListener(SWT.Modify, this);
        browseDBLocation.addListener(SWT.Selection, this);
        createCheck.addListener(SWT.Selection, this);
        if (displayUpgrade)
            upgradeCheck.addListener(SWT.Selection, this);
        urlDisplay.addListener(SWT.Selection, this);
    }

    protected void removeListeners() {
        locationCombo.removeListener(SWT.Modify, this);
        browseLocation.removeListener(SWT.Selection, this);
        databaseCombo.removeListener(SWT.Modify, this);
        browseDBLocation.removeListener(SWT.Selection, this);
        createCheck.removeListener(SWT.Selection, this);
        if (displayUpgrade)
            upgradeCheck.removeListener(SWT.Selection, this);
        urlDisplay.removeListener(SWT.Selection, this);
    }

    public Composite getServerConnectionUI(Composite parent)
    {
        if (parentComposite == null || parentComposite.isDisposed()) {
            refresh();
            parentComposite = createServerConnectionUI(parent);
        }
        return parentComposite;
    }
    
    protected Composite createServerConnectionUI(Composite parent)
    {
        parentComposite = factory.createComposite(parent, SWT.NULL);
        GridLayout layout = new GridLayout(3, false);
        parentComposite.setLayout(layout);

        GridData gd;

        databaseLabel = factory.createLabel(parentComposite, SWT.NONE);
        databaseLabel.setText(CUI_NEWCW_DBLOCATION_LBL_UI_);
        
        databaseCombo = factory.createCombo(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        databaseCombo.setLayoutData(gd);
        
        browseDBLocation = factory.createButton(parentComposite, SWT.PUSH);
        browseDBLocation.setText(CUI_NEWCW_DBBROWSE_BTN_UI_);
        
        createCheck = factory.createButton(parentComposite, SWT.CHECK);
        createCheck.setText(CUI_NEWCW_CREATEDB_BTN_UI_);
        createCheck.setSelection(true);
        gd = new GridData();
        gd.horizontalSpan = 3;
        createCheck.setLayoutData(gd);
        
        if (displayUpgrade) {
            upgradeCheck = factory.createButton(parentComposite, SWT.CHECK);
            upgradeCheck.setText(CUI_NEWCW_UPGRADEDB_BTN_UI_);
            upgradeCheck.setSelection(false);
            gd = new GridData();
            gd.horizontalSpan = 3;
            upgradeCheck.setLayoutData(gd);
        }

        driverLabel = factory.createLabel(parentComposite, SWT.NONE);
        driverLabel.setText(CUI_NEWCW_JDBCDRIVERCLS_LBL_UI_);

        driverDisplay = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        driverDisplay.setLayoutData(gd);

        locationLabel = factory.createLabel(parentComposite, SWT.NONE);
        locationLabel.setText(CUI_NEWCW_JDBCCLSLOCATION_LBL_UI_);
        
        locationCombo = factory.createCombo(parentComposite, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        locationCombo.setLayoutData(gd);

        browseLocation = factory.createButton(parentComposite, SWT.PUSH);
        browseLocation.setText(CUI_NEWCW_JARBROWSE_BTN_UI_);
        
        urlLabel = factory.createLabel(parentComposite, SWT.NONE);
        urlLabel.setText(CUI_NEWCW_CONNECTIONURL_LBL_UI_);

        urlDisplay = factory.createText(parentComposite, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
        gd = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
        gd.horizontalSpan = 2;
        urlDisplay.setLayoutData(gd);

        initialize();
        
        return parentComposite;
    }

    public void setServerConnectionInformation(IServerConnectionInformation connectionInformation)
    {
        this.connectionInformation = connectionInformation;
    }

    public void setWizardPage(WizardPage parentPage)
    {
        this.parentPage = parentPage;
    }
    
    public void setDatabaseDefinition(DatabaseDefinition definition)
    {
        this.definition = definition;
    }

    protected void updateDatabaseName() {
        if (databaseCombo.getText() != null
                && !databaseCombo.getText().equals("")) { //$NON-NLS-1$
            databaseName = databaseCombo.getText().substring(
                    databaseCombo.getText().lastIndexOf(File.separator) + 1);
        }
    }
    public void handleEvent(Event event)
    {
        Widget source = event.widget;
        if (source == databaseCombo || source == createCheck || source == upgradeCheck) {
        	updateDatabaseName();
            updateURL();
        } else if (source == browseDBLocation) {
            String directory = getDirectory(CUI_NEWCW_DBLOCATION_CLOUDSCAPE_UI_);
            if (directory != null && directory.length() > 0) {
                databaseCombo.setText(directory);
                databaseName = directory.substring(directory.lastIndexOf(File.separator) + 1);
            }
        } else if (source == locationCombo) {
        } else if (source == browseLocation) {
            String file = NewConnectionWizard.getFile((Wizard) parentPage.getWizard(), NewConnectionWizard.fExtensions);
            if (file != null && file.length() > 0) {
                locationCombo.removeListener(SWT.Modify, this);
                locationCombo.setText(file);
                locationCombo.addListener(SWT.Modify, this);
                connectionInformation.setLoadingPath(file);
            }
        }
        setConnectionInformation();
    }
    
    public void loadProperties() {
        removeListeners();
        GenericJDBCURL url = new GenericJDBCURL(connectionInformation.getURL());
        locationCombo.setText(connectionInformation.getLoadingPath());
        databaseCombo.setText(url.getNode());
        updateDatabaseName();
        if (!(url.getProperties().indexOf(CREATE_EQUALS_TRUE_TEXT) > -1)) {
            createCheck.setSelection(false);
        }
        if (displayUpgrade && (url.getProperties().indexOf(CREATE_EQUALS_TRUE_TEXT) > -1)) {
            upgradeCheck.setSelection(true);
        }
        updateURL();
        addListeners();
        setConnectionInformation();
    }

    protected class GenericJDBCURL {
        protected String subprotocol = ""; //$NON-NLS-1$

        protected String node = ""; //$NON-NLS-1$

        protected String properties = ""; //$NON-NLS-1$

        /**
         * @param url
         */
        public GenericJDBCURL(String url) {
            parseURL(url);
        }

        /**
         * @return Returns the node.
         */
        public String getNode() {
            return node;
        }

        /**
         * @return Returns the subprotocol.
         */
        public String getSubprotocol() {
            return subprotocol;
        }

        protected void parseURL(String url)
        {
            try {
                String remainingURL = url.substring(url.indexOf(':') + 1);
                this.subprotocol = remainingURL.substring(0, remainingURL.indexOf(':'));
                remainingURL = remainingURL.substring(remainingURL.indexOf(':') + 1);
                if (remainingURL.indexOf(';') > -1) {
                    this.node = remainingURL.substring(0, remainingURL.indexOf(';'));
                    remainingURL = remainingURL.substring(remainingURL.indexOf(';') + 1);
                    this.properties = remainingURL;
                } else {
                    this.node = remainingURL;
                }
            } catch (Exception e) {
            }
        }

        /**
         * @return Returns the properties.
         */
        public String getProperties() {
            return properties;
        }
    }
}