module Msf

###
#
# This module provides methods for sending and receiving
# raw packets. It should be preferred over the soon-to-be
# deprecated Rex::Socket::Ip and Msf::Exploite::Remote::Ip
# mixins.
#
# Please see the pcaprub documentation for more information
# on how to use capture objects.
#
###

module Exploit::Capture

	#
	# Initializes an instance of an exploit module that captures traffic
	#

	def initialize(info = {})
		super

		register_options(
			[
				OptPath.new('PCAPFILE', [false, 'The name of the PCAP capture file to process']),
				OptString.new('INTERFACE', [false, 'The name of the interface']),
				OptString.new('FILTER', [false, 'The filter string for capturing traffic']),
				OptInt.new('SNAPLEN', [true, 'The number of bytes to capture', 65535]),
				OptInt.new('TIMEOUT', [true, 'The number of seconds to wait for new data', 500]),
				Opt::RHOST

			], Msf::Exploit::Capture
		)

		register_advanced_options(
			[
				OptInt.new('UDP_SECRET', [true, 'The 32-bit cookie for UDP probe requests.', 1297303091]),
				OptAddress.new('GATEWAY', [false, 'The gateway IP address. This will be used rather than a random remote address for the UDP probe, if set.']),
				OptInt.new('NETMASK', [false, 'The local network mask. This is used to decide if an address is in the local network.', 24]),
			], Msf::Exploit::Capture
		)

		require 'racket'

		begin
			require 'pcaprub'
			@pcaprub_loaded = true
		rescue ::Exception => e
			@pcaprub_loaded = false
			@pcaprub_error  = e
		end

	end

	def stats_recv(pcap=self.capture)
		return(0) if not pcap
		pcap.stats['recv']
	end

	def stats_drop(pcap=self.capture)
		return(0) if not pcap
		pcap.stats['drop']
	end

	def stats_ifdrop(pcap=self.capture)
		return(0) if not pcap
		pcap.stats['ifdrop']
	end

	#
	# Opens a handle to the specified device
	#
	def open_pcap(opts={})
		check_pcaprub_loaded

		# Capture device
		dev = opts['INTERFACE'] || datastore['INTERFACE'] || nil
		len = (opts['SNAPLEN'] || datastore['SNAPLEN']  || 65535).to_i
		tim = (opts['TIMEOUT'] || datastore['TIMEOUT']  || 0).to_i
		fil = opts['FILTER'] || datastore['FILTER']
		arp = opts['ARPCAP'] || true

		# Look for a PCAP file
		cap = datastore['PCAPFILE'] || ''

		if(not cap.empty?)
			if(not File.exists?(cap))
				raise RuntimeError, "The PCAP file #{cap} could not be found"
			end
			self.capture = ::Pcap.open_offline(cap)
		else
			dev ||= ::Pcap.lookupdev
			system("ifconfig", dev, "up")
			self.capture = ::Pcap.open_live(dev, len, true, tim)
			if arp
				self.arp_capture = ::Pcap.open_live(dev, 512, true, tim)
				preamble = datastore['UDP_SECRET'].to_i
				arp_filter = "arp[6:2] = 2 or (udp[8:4] = #{preamble})"
				self.arp_capture.setfilter(arp_filter)
			end
		end

		if (not self.capture)
			raise RuntimeError, "Could not start the capture process"
		elsif (arp and !self.arp_capture)
			raise RuntimeError, "Could not start the ARP capture process"
		end

		self.capture.setfilter(fil) if fil
	end

	def close_pcap
		return if not self.capture
		self.capture = nil
		self.arp_capture = nil
		GC.start()
	end

	def capture_extract_ies(raw)
		set = {}
		ret = 0
		idx = 0
		len = 0

		while (idx < raw.length)
			len = raw[idx+1]
			return set if not len
			set[ raw[idx] ] ||= []
			set[ raw[idx] ].push(raw[idx + 2, len])
			idx += len + 2
		end

		return set
	end

	#
	# This monstrosity works around a series of bugs in the interrupt
	# signal handling of Ruby 1.9
	#
	def each_packet
		return if not capture
		begin
			@capture_count = 0
			reader = Thread.new do
				capture.each do |pkt|
					yield(pkt)
					@capture_count += 1
				end
			end
			reader.join
		rescue ::Exception
			raise $!
		ensure
			reader.kill if reader.alive?
		end

		@capture_count
	end

	# Injects a packet on the wire. For all injection-related functions, it's
	# on the module to open up a capture device first (this way, we don't
	# needlessly spawn new capture devices).
	def inject(pkt="",pcap=self.capture)
		check_pcaprub_loaded
		if not pcap
			raise RuntimeError, "Could not access the capture process (remember to open_pcap first!)"
		else
			pcap.inject(pkt)
		end
	end

	# Injects an Ethernet packet with an optional payload.
	def inject_eth(args={})
		eth_daddr = args[:eth_daddr] || "ff:ff:ff:ff:ff:ff"
		eth_saddr = args[:eth_saddr] || "00:00:00:00:00:00"
		eth_type  = args[:eth_type]  || 0x0800 # IP default
		payload   = args[:payload]
		pcap      = args[:pcap]      || self.capture
		n = Racket::Racket.new
		n.l2 = Racket::L2::Ethernet.new
		n.l2.dst_mac = eth_daddr
		n.l2.src_mac = eth_saddr
		n.l2.ethertype = eth_type
		pkt = n.pack
		pkt += payload if payload
		inject pkt,pcap
	end

	# Capture_sendto is intended to replace the old Rex::Socket::Ip.sendto method. It requires
	# a payload and a destination address. To send to the broadcast address, set bcast
	# to true (this will guarantee that packets will be sent even if ARP doesn't work
	# out).
	def capture_sendto(payload="", dhost=nil, bcast=false)
		raise RuntimeError, "Could not access the capture process (remember to open_pcap first!)" unless self.capture
		raise RuntimeError, "Must specify a host to sendto" unless dhost
		dst_mac,src_mac = lookup_eth(dhost)
		if dst_mac == nil and not bcast
			return false
		end
		inject_eth(:payload => payload, :eth_daddr => dst_mac, :eth_saddr => src_mac)
	end

	# Depending on what kind of packet you get, the resultant hash returned will
	# contain one or several Racket objects.
	def inject_reply(proto=:udp,pcap=self.capture)
		reply = nil
		to = (datastore['TIMEOUT'] || 500).to_f / 1000.0
		if not pcap
			raise RuntimeError, "Could not access the capture process (remember to open_pcap first!)"
		else
			begin
				::Timeout.timeout(to) do
					pcap.each do |r|
						eth = Racket::L2::Ethernet.new(r)
						case proto
						when :arp
							next if not eth.ethertype == 0x0806
							arp = Racket::L3::ARP.new(eth.payload)
							reply = {:raw => r, :eth => eth, :arp => arp}
							break
						when :ip
							next if not eth.ethertype == 0x0800
							ip = Racket::L3::IPv4.new(eth.payload)
							reply = {:raw => r, :eth => eth, :ip => ip}
							break
						when :udp
							ip = Racket::L3::IPv4.new(eth.payload)
							next if not ip.protocol == 17
							udp = Racket::L4::UDP.new(ip.payload)
							reply = {:raw => r, :eth => eth, :ip => ip, :udp => udp, :payload => udp.payload}
							break
						when :tcp
							ip = Racket::L3::IPv4.new(eth.payload)
							next if not ip.protocol == 6
							tcp = Racket::L4::TCP.new(ip.payload)
							reply = {:raw => r, :eth => eth, :ip => ip, :tcp => tcp, :payload => tcp.payload}
							break
						end
					end
				end
				rescue ::Timeout::Error
			end
		end
		return reply
	end

	# This ascertains the correct Ethernet addresses one should use to
	# ensure injected IP packets actually get where they are going, and
	# manages the self.arp_cache hash. It always uses self.arp_capture
	# do inject and capture packets, and will always first fire off a
	# UDP packet using the regular socket to learn the source host's
	# and gateway's mac addresses.
	def lookup_eth(addr=nil,iface=nil)
		raise RuntimeError, "Could not access the capture process." if not self.arp_capture

		self.arp_cache ||= {}
		self.dst_cache ||= {}

		return self.dst_cache[addr] if self.dst_cache[addr]

		if ! self.arp_cache[Rex::Socket.source_address(addr)]
			probe_gateway(addr)
		end

		src_mac = self.arp_cache[Rex::Socket.source_address(addr)]
		unless should_arp?(addr)
			dst_mac = self.arp_cache[:gateway]
		else
			dst_mac = self.arp_cache[addr] || arp(addr)
		end

		self.dst_cache[addr] = [dst_mac,src_mac]
	end

	def probe_gateway(addr)
		dst_host = (datastore['GATEWAY'] || IPAddr.new((rand(16777216) + 2969567232), Socket::AF_INET).to_s)
		dst_port = rand(30000)+1024
		preamble = [datastore['UDP_SECRET']].pack("N")
		secret = "#{preamble}#{Rex::Text.rand_text(rand(0xff)+1)}"
		UDPSocket.open.send(secret,0,dst_host,dst_port)
		begin
			to = (datastore['TIMEOUT'] || 1500).to_f / 1000.0
			::Timeout.timeout(to) do
				while(my_packet = inject_reply(:udp,self.arp_capture))
					if my_packet[:payload] == secret
						dst_mac = self.arp_cache[:gateway] = my_packet[:eth].dst_mac
						src_mac = self.arp_cache[Rex::Socket.source_address(addr)] = my_packet[:eth].src_mac
						return [dst_mac,src_mac]
					else
						next
					end
				end
			end
		rescue ::Timeout::Error
			# Well, that didn't work (this common on networks where there's no gatway, like
			# VMWare network interfaces. We'll need to use a fake source hardware address.
			self.arp_cache[Rex::Socket.source_address(addr)] = "00:00:00:00:00:00"
		end
	end

	# A pure-Ruby ARP exchange. It uses self.arp_capture to send and recv
	# packets, rather than self.capture.
	def arp(target_ip=nil)
		return self.arp_cache[target_ip] if self.arp_cache[target_ip]
		return self.arp_cache[:gateway] unless should_arp? target_ip
		source_ip = Rex::Socket.source_address(target_ip)
		raise RuntimeError, "Could not access the capture process." if not self.arp_capture
		n = arp_packet(target_ip,source_ip)
		inject_eth(:eth_type => 0x0806,
			:payload => n.pack,
			:pcap => self.arp_capture,
			:eth_saddr => self.arp_cache[Rex::Socket.source_address(target_ip)]
		)
		begin
			to = (datastore['TIMEOUT'] || 500).to_f / 1000.0
			::Timeout.timeout(to) do
				while (my_packet = inject_reply(:arp,self.arp_capture))
					if my_packet[:arp].spa == target_ip
						self.arp_cache[target_ip] = my_packet[:arp].sha
						return self.arp_cache[target_ip]
					else
						next
					end
				end
			end
		rescue ::Timeout::Error
		end
	end

	def arp_packet(target_ip,source_ip)
		n = Racket::Racket.new
		n.l3 = Racket::L3::ARP.new
		n.l3.opcode = 1
		n.l3.tpa = target_ip || datastore['RHOST']
		n.l3.spa = datastore['LHOST'] || source_ip
		my_eth = self.arp_cache[Rex::Socket.source_address(target_ip)]
		n.l3.sha = my_eth || "00:00:00:00:00:00"
		return n
	end

	# Allow modules to reset their arp caches arbitrarily.
	def expire_arpcache
		self.arp_cache = {}
	end

	# For compatabilty with Msf::Exploit::Remote::Ip
	def rhost
		datastore['RHOST']
	end

	def check_pcaprub_loaded
		unless @pcaprub_loaded
			print_status("The Pcaprub module is not available: #{@pcaprub_error}")
			raise RuntimeError, "Pcaprub not available"
		else
			true
		end
	end

	def lookupnet
		check_pcaprub_loaded
		dev  = datastore['INTERFACE'] || ::Pcap.lookupdev
		mask = datastore['NETMASK'] || 24
		begin
			my_net = IPAddr.new("#{Pcap.lookupnet(dev).first}/#{mask}")
		rescue RuntimeError => e
			@pcaprub_error = e
			print_status("Cannot stat device: #{@pcaprub_error}")
			raise RuntimeError, "Pcaprub error: #{@pcaprub_error}"
		end
		return my_net
	end

	def should_arp?(ip)
		@mydev  ||= datastore['INTERFACE'] || ::Pcap.lookupdev
		@mymask ||= datastore['NETMASK'] || 24
		@mynet  ||= lookupnet
		@mynet.include?(IPAddr.new(ip))
	end

	attr_accessor :capture, :arp_cache, :arp_capture, :dst_cache

end

end

