/*
 * Copyright 2005 - 2009  Zarafa B.V.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3, 
 * as published by the Free Software Foundation with the following additional 
 * term according to sec. 7:
 *  
 * According to sec. 7 of the GNU Affero General Public License, version
 * 3, the terms of the AGPL are supplemented with the following terms:
 * 
 * "Zarafa" is a registered trademark of Zarafa B.V. The licensing of
 * the Program under the AGPL does not imply a trademark license.
 * Therefore any rights, title and interest in our trademarks remain
 * entirely with us.
 * 
 * However, if you propagate an unmodified version of the Program you are
 * allowed to use the term "Zarafa" to indicate that you distribute the
 * Program. Furthermore you may use our trademarks where it is necessary
 * to indicate the intended purpose of a product or service provided you
 * use it in accordance with honest practices in industrial or commercial
 * matters.  If you want to propagate modified versions of the Program
 * under the name "Zarafa" or "Zarafa Server", you may only do so if you
 * have a written permission by Zarafa B.V. (to acquire a permission
 * please contact Zarafa at trademark@zarafa.com).
 * 
 * The interactive user interface of the software displays an attribution
 * notice containing the term "Zarafa" and/or the logo of Zarafa.
 * Interactive user interfaces of unmodified and modified versions must
 * display Appropriate Legal Notices according to sec. 5 of the GNU
 * Affero General Public License, version 3, when you propagate
 * unmodified or modified versions of the Program. In accordance with
 * sec. 7 b) of the GNU Affero General Public License, version 3, these
 * Appropriate Legal Notices must retain the logo of Zarafa or display
 * the words "Initial Development by Zarafa" if the display of the logo
 * is not reasonably feasible for technical reasons. The use of the logo
 * of Zarafa in Legal Notices is allowed for unmodified and modified
 * versions of the software.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifndef ECCONFIG_H
#define ECCONFIG_H

#include <map>
#include <set>
#include <list>
#include <string>
#include <string.h>
#include <pthread.h>
using namespace std;

struct configsetting_t {
	const char *szName;
	const char *szValue;
	unsigned short ulFlags;
#define CONFIGSETTING_ALIAS			0x0001
#define CONFIGSETTING_RELOADABLE	0x0002
#define CONFIGSETTING_UNUSED		0x0004
#define CONFIGSETTING_NONEMPTY		0x0008
	unsigned short ulGroup;
#define CONFIGGROUP_PROPMAP			0x0001
};

struct settingkey_t {
	char s[256];
	unsigned short ulFlags;
	unsigned short ulGroup;
};

struct settingcompare
{
	bool operator()(const settingkey_t &a, const settingkey_t &b) const
	{
		return strcmp(a.s, b.s) < 0;
	}
};

#define MAXLINELEN 4096

class ECConfig;

typedef map<settingkey_t, char *, settingcompare> settingmap_t;
typedef bool (ECConfig::*directive_func_t)(const char *, unsigned int);
typedef struct {
	const char			*lpszDirective;
	directive_func_t	fExecute;
} directive_t;

/*
 * Flags for the InitDefaults & InitConfigFile functions
 */
#define LOADSETTING_INITIALIZING		0x0001	/* ECConfig is initializing, turns on extra debug information */
#define LOADSETTING_UNKNOWN				0x0002	/* Allow adding new configuration options */
#define LOADSETTING_OVERWRITE			0x0004	/* Allow overwriting predefined configuration options */
#define LOADSETTING_OVERWRITE_GROUP		0x0008	/* Same as CONFIG_LOAD_OVERWRITE but only if options are in the same group */
#define LOADSETTING_OVERWRITE_RELOAD	0x0010	/* Same as CONFIG_LOAD_OVERWRITE but only if option is marked reloadable */

static const char *lpszDEFAULTDIRECTIVES[] = { "include", NULL };

class ECConfig {
public:
	ECConfig(const configsetting_t *lpDefaults, const char **lpszDirectives = lpszDEFAULTDIRECTIVES);
	~ECConfig();

	bool	LoadSettings(const char *szFilename);
	bool	ReloadSettings();

	void	AddSetting(const char *szName, const char *szValue, const unsigned int ulGroup = 0);

	char*	GetSetting(const char *szName);
	char*	GetSetting(const char *szName, char *equal, char *other);

	list<configsetting_t> GetSettingGroup(unsigned int ulGroup);

	bool	HasWarnings();
	list<string>* GetWarnings();
	bool	HasErrors();
	list<string>* GetErrors();

private:
	bool	InitDefaults(unsigned int ulFlags);
	bool	InitConfigFile(unsigned int ulFlags);
	bool	ReadConfigFile(const char *szFilename, unsigned int ulFlags, unsigned int ulGroup = 0);

	bool	HandleDirective(string &strLine, unsigned int ulFlags);
	bool	HandleInclude(const char *lpszArgs, unsigned int ulFlags);
	bool	HandlePropMap(const char *lpszArgs, unsigned int ulFlags);

	char*	GetMapEntry(settingmap_t *lpMap, const char *szName);
	char*	GetAlias(const char *szAlias);

	void	AddSetting(const configsetting_t *lpsConfig, unsigned int ulFlags);
	void	AddAlias(const configsetting_t *lpsAlias);

	void	CleanupMap(settingmap_t *lpMap);
	bool	CopyConfigSetting(const configsetting_t *lpsSetting, settingkey_t *lpsKey);
	bool	CopyConfigSetting(const settingkey_t *lpsKey, const char *szValue, configsetting_t *lpsSetting);

	int		SetPathTo(const char *lpszDir);
	void	RestorePath();

private:
	const configsetting_t *m_lpDefaults;
	const char* m_szConfigFile;
	list<string> m_lDirectives;

	/* m_mapSettings & m_mapAliases are protected by m_settingsLock */
	pthread_rwlock_t m_settingsRWLock;

	settingmap_t		m_mapSettings;
	settingmap_t		m_mapAliases;
	list<string>		warnings;
	list<string>		errors;
	set<string>			m_readFiles;
	list<string>		m_pathStack;

	static const directive_t	s_sDirectives[];
};

#endif // ECCONFIG_H
