<?php
/*********************************************************************************
 * TimeTrex is a Payroll and Time Management program developed by
 * TimeTrex Payroll Services Copyright (C) 2003 - 2012 TimeTrex Payroll Services.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by
 * the Free Software Foundation with the addition of the following permission
 * added to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED
 * WORK IN WHICH THE COPYRIGHT IS OWNED BY TIMETREX, TIMETREX DISCLAIMS THE
 * WARRANTY OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License along
 * with this program; if not, see http://www.gnu.org/licenses or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 *
 * You can contact TimeTrex headquarters at Unit 22 - 2475 Dobbin Rd. Suite
 * #292 Westbank, BC V4T 2E9, Canada or at email address info@timetrex.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License
 * version 3, these Appropriate Legal Notices must retain the display of the
 * "Powered by TimeTrex" logo. If the display of the logo is not reasonably
 * feasible for technical reasons, the Appropriate Legal Notices must display
 * the words "Powered by TimeTrex".
 ********************************************************************************/
/*
 * $Revision: 2196 $
 * $Id: APIPunch.class.php 2196 2008-10-14 16:08:54Z ipso $
 * $Date: 2008-10-14 09:08:54 -0700 (Tue, 14 Oct 2008) $
 */

/**
 * @package API_APIPunch
 */

/*


	This class has to treat punch and punch_control data as if they are one.

*/
class APIPunch extends APIFactory {
	protected $main_class = 'PunchFactory';

	public function __construct() {
		parent::__construct(); //Make sure parent constructor is always called.

		return TRUE;
	}

	function getUserPunch( $user_id = NULL, $epoch = NULL, $station_id = NULL, $company_id = NULL ) {
		if ( $epoch == '' ) {
			$epoch = TTDate::getTime();
		}

		if ( $user_id == '' ) {
			$user_id = $this->getCurrentUserObject()->getId();
		}

		if ( $company_id == '' ) {
			$company_id = $this->getCurrentCompanyObject()->getId();
		}

		if ( $station_id == '' ) {
			$station_id = getStationID(); //API.inc
		}

		//Must call APIStation->getCurrentStation( $station_id = NULL ) first, so the Station ID cookie can be set and passed to this.
		//Check if station is allowed.
		$current_station = FALSE;
		$slf = new StationListFactory();
		$slf->getByStationIdandCompanyId( $station_id, $company_id );
		if ( $slf->getRecordCount() == 1 ) {
			$current_station = $slf->getCurrent();
		}
		unset($slf);
		$slf = TTnew('ScheduleListFactory');

		Debug::Text('Station ID: '. $station_id .' User ID: '. $user_id .' Epoch: '. $epoch, __FILE__, __LINE__, __METHOD__,10);
		if ( is_object($current_station) AND $current_station->checkAllowed() == TRUE ) {
			Debug::Text('Station Allowed! ID: '. $station_id .' User ID: '. $user_id .' Epoch: '. $epoch, __FILE__, __LINE__, __METHOD__,10);
			//Get user object from ID.
			$ulf = TTNew('UserListFactory');
			$ulf->getByIdAndCompanyId( $user_id, $company_id );
			if ( $ulf->getRecordCount() == 1 ) {
				$user_obj = $ulf->getCurrent();

				$plf = TTNew('PunchListFactory');
				$plf->getPreviousPunchByUserIDAndEpoch( $user_id, $epoch );
				if ( $plf->getRecordCount() > 0 ) {
					$prev_punch_obj = $plf->getCurrent();
					$prev_punch_obj->setUser( $user_id );
					Debug::Text(' Found Previous Punch within Continuous Time from now: '. TTDate::getDate('DATe+TIME', $prev_punch_obj->getTimeStamp() ), __FILE__, __LINE__, __METHOD__,10);

					$branch_id = $prev_punch_obj->getPunchControlObject()->getBranch();
					$department_id = $prev_punch_obj->getPunchControlObject()->getDepartment();
					$job_id = $prev_punch_obj->getPunchControlObject()->getJob();
					$job_item_id = $prev_punch_obj->getPunchControlObject()->getJobItem();

					//Don't enable transfer by default if the previous punch was any OUT punch.
					//Transfer does the OUT punch for them, so if the previous punch is an OUT punch
					//we don't gain anything anyways.
					if ( $this->getPermissionObject()->Check('punch','default_transfer') AND $prev_punch_obj->getStatus() == 10 ) {
						$transfer = TRUE;
					} else {
						$transfer = FALSE;
					}

					if ( $branch_id == '' OR empty($branch_id)
							OR $department_id == '' OR empty($department_id)
							OR $job_id == '' OR empty($job_id)
							OR $job_item_id == '' OR empty($job_item_id) ) {
						Debug::Text(' Branch or department are null. ', __FILE__, __LINE__, __METHOD__,10);
						$s_obj = $slf->getScheduleObjectByUserIdAndEpoch( $user_id, $epoch );

						if ( is_object($s_obj) ) {
							Debug::Text(' Found Schedule!: ', __FILE__, __LINE__, __METHOD__,10);
							if ( $branch_id == '' OR empty($branch_id) ) {
								Debug::Text(' overrriding branch: '. $s_obj->getBranch(), __FILE__, __LINE__, __METHOD__,10);
								$branch_id = $s_obj->getBranch();
							}
							if ( $department_id == '' OR empty($department_id) ) {
								Debug::Text(' overrriding department: '. $s_obj->getDepartment(), __FILE__, __LINE__, __METHOD__,10);
								$department_id = $s_obj->getDepartment();
							}

							if ( $job_id == '' OR empty($job_id) ) {
								Debug::Text(' overrriding job: '. $s_obj->getJob(), __FILE__, __LINE__, __METHOD__,10);
								$job_id = $s_obj->getJob();
							}
							if ( $job_item_id == '' OR empty($job_item_id) ) {
								Debug::Text(' overrriding job item: '. $s_obj->getJobItem(), __FILE__, __LINE__, __METHOD__,10);
								$job_item_id = $s_obj->getJobItem();
							}
						}
					}

					$next_type = $prev_punch_obj->getNextType( $epoch ); //Detects breaks/lunches too.

					if ( $prev_punch_obj->getNextStatus() == 10 ) {
						//In punch - Carry over just certain data
						$data = array(
										'user_id' => $this->getCurrentUserObject()->getId(),
										'first_name' => $this->getCurrentUserObject()->getFirstName(),
										'last_name' => $this->getCurrentUserObject()->getLastName(),
										'type_id' => $next_type,
										'status_id' => $prev_punch_obj->getNextStatus(),
										'date_stamp' => TTDate::getAPIDate( 'DATE', $epoch ),
										'time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
										'punch_date' => TTDate::getAPIDate( 'DATE', $epoch ),
										'punch_time' => TTDate::getAPIDate( 'TIME', $epoch ),
										'original_time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
										'actual_time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
										'punch_control_id' => $prev_punch_obj->getNextPunchControlID(),
										'transfer' => $transfer,
										'branch_id' => $branch_id,
										'department_id' => $department_id,
										'job_id' => $job_id,
										'job_item_id' => $job_item_id,
										'quantity_id' => 0,
										'bad_quantity_id' => 0,
										'station_id' => $current_station->getId(),
										);
					} else {
						//Out punch
						$data = array(
										'user_id' => $this->getCurrentUserObject()->getId(),
										'first_name' => $this->getCurrentUserObject()->getFirstName(),
										'last_name' => $this->getCurrentUserObject()->getLastName(),
										'type_id' => $next_type,
										'status_id' => $prev_punch_obj->getNextStatus(),
										'date_stamp' => TTDate::getAPIDate( 'DATE', $epoch ),
										'time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
										'punch_date' => TTDate::getAPIDate( 'DATE', $epoch ),
										'punch_time' => TTDate::getAPIDate( 'TIME', $epoch ),
										'original_time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
										'actual_time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
										'punch_control_id' => $prev_punch_obj->getNextPunchControlID(),
										'transfer' => $transfer,
										'branch_id' => $branch_id,
										'department_id' => $department_id,
										'job_id' => $job_id,
										'job_item_id' => $job_item_id,
										'quantity_id' => (float)$prev_punch_obj->getPunchControlObject()->getQuantity(),
										'bad_quantity_id' => (float)$prev_punch_obj->getPunchControlObject()->getBadQuantity(),
										'note' => $prev_punch_obj->getPunchControlObject()->getNote(),
										'other_id1' => $prev_punch_obj->getPunchControlObject()->getOtherID1(),
										'other_id2' => $prev_punch_obj->getPunchControlObject()->getOtherID2(),
										'other_id3' => $prev_punch_obj->getPunchControlObject()->getOtherID3(),
										'other_id4' => $prev_punch_obj->getPunchControlObject()->getOtherID4(),
										'other_id5' => $prev_punch_obj->getPunchControlObject()->getOtherID5(),
										'station_id' => $current_station->getId(),
										);

					}
				} else {
					Debug::Text(' DID NOT Find Previous Punch within Continuous Time from now: ', __FILE__, __LINE__, __METHOD__,10);
					$branch_id = NULL;
					$department_id = NULL;
					$job_id = NULL;
					$job_item_id = NULL;

					$s_obj = $slf->getScheduleObjectByUserIdAndEpoch( $user_id, $epoch );
					if ( is_object($s_obj) ) {
						Debug::Text(' Found Schedule!: ', __FILE__, __LINE__, __METHOD__,10);
						$branch_id = $s_obj->getBranch();
						$department_id = $s_obj->getDepartment();
						$job_id = $s_obj->getJob();
						$job_item_id = $s_obj->getJobItem();
					} else {
						$branch_id = $user_obj->getDefaultBranch();
						$department_id = $user_obj->getDefaultDepartment();

						//Check station for default/forced settings.
						if ( is_object($current_station) ) {
							if ( $current_station->getDefaultBranch() !== FALSE AND $current_station->getDefaultBranch() != 0 ) {
								$branch_id = $current_station->getDefaultBranch();
							}
							if ( $current_station->getDefaultDepartment() !== FALSE AND $current_station->getDefaultDepartment() != 0 ) {
								$department_id = $current_station->getDefaultDepartment();
							}
							if ( $current_station->getDefaultJob() !== FALSE AND $current_station->getDefaultJob() != 0 ) {
								$job_id = $current_station->getDefaultJob();
							}
							if ( $current_station->getDefaultJobItem() !== FALSE AND $current_station->getDefaultJobItem() != 0 ) {
								$job_item_id = $current_station->getDefaultJobItem();
							}
						}
					}

					$data = array(
									'user_id' => $this->getCurrentUserObject()->getId(),
									'first_name' => $this->getCurrentUserObject()->getFirstName(),
									'last_name' => $this->getCurrentUserObject()->getLastName(),
									'type_id' => 10, //Normal
									'status_id' => 10, //In
									'date_stamp' => TTDate::getAPIDate( 'DATE', $epoch ),
									'time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
									'punch_date' => TTDate::getAPIDate( 'DATE', $epoch ),
									'punch_time' => TTDate::getAPIDate( 'TIME', $epoch ),
									'original_time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
									'actual_time_stamp' => TTDate::getAPIDate( 'DATE+TIME', $epoch ),
									'punch_control_id' => FALSE,
									'station_id' => $current_station->getId(),
									'transfer' => FALSE,
									'branch_id' => $branch_id,
									'department_id' => $department_id,
									'job_id' => $job_id,
									'job_item_id' => $job_item_id,
									'quantity' => 0,
									'bad_quantity' => 0,
									);
				}

				if ( isset($data ) ) {
					Debug::Arr($data, 'Punch Data: ', __FILE__, __LINE__, __METHOD__,10);
					return $this->returnHandler( $data );
				}
			}
		} else {
			Debug::Text('Station IS NOT Allowed! ID: '. $station_id .' User ID: '. $user_id .' Epoch: '. $epoch, __FILE__, __LINE__, __METHOD__,10);
			$validator_obj = new Validator();
			$validator_stats = array('total_records' => 1, 'valid_records' => 0 );

			$error_message = TTi18n::gettext('You are not authorized to punch in or out from this station!');
			$validator_obj->isTrue( 'user_name', FALSE, $error_message );
			$validator[0] = $validator_obj->getErrorsArray();

			return $this->returnHandler( FALSE, 'VALIDATION', TTi18n::getText('INVALID DATA'), $validator, $validator_stats );
		}

		return FALSE;
	}

	function setUserPunch( $data, $validate_only = FALSE ) {
		if ( !$this->getPermissionObject()->Check('punch','enabled')
				OR !( $this->getPermissionObject()->Check('punch','punch_in_out') ) ) {
			return $this->getPermissionObject()->PermissionDenied();
		}

		//Force proper settings.
		$data['user_id'] = $this->getCurrentUserObject()->getId();
		if ( isset($data['transfer']) AND $data['transfer'] == TRUE ) {
			$data['type_id'] = 10;
			$data['status_id'] = 10;
		}

		$validator_stats = array('total_records' => 1, 'valid_records' => 0 );

		$lf = TTnew( 'PunchFactory' );
		$lf->StartTransaction();

		//We can't set the PunchControlID from $data, otherwise findPunchControlID() will never work.
		//This causes transfer punches to fail.
		$tmp_punch_control_id = $data['punch_control_id'];
		unset($data['punch_control_id']);

		$lf->setObjectFromArray( $data );

		if ( isset($data['status_id']) AND $data['status_id'] == 20 AND $tmp_punch_control_id  != '' ) {
			$lf->setPunchControlID( $tmp_punch_control_id );
		} else {
			$lf->setPunchControlID( $lf->findPunchControlID() );
		}
		unset($tmp_punch_control_id);

		$key = 0;
		$is_valid = $lf->isValid();
		if ( $is_valid == TRUE ) {
			Debug::Text('Saving data...', __FILE__, __LINE__, __METHOD__, 10);
			if ( $validate_only == TRUE ) {
				$save_result[$key] = TRUE;
				$validator_stats['valid_records']++;
			} else {
				//Save Punch object and start on PunchControl
				if ( $save_result[$key] = $lf->Save( FALSE ) == TRUE ) {
					unset($data['id']); //ID must be removed so it doesn't get confused with PunchControlID
					Debug::Text('Saving PCF data... Punch Control ID: '. $lf->getPunchControlID(), __FILE__, __LINE__, __METHOD__, 10);
					$pcf = TTnew( 'PunchControlFactory' );

					$pcf->setId( $lf->getPunchControlID() );
					$pcf->setPunchObject( $lf );

					$pcf->setObjectFromArray( $data );

					$pcf->setEnableStrictJobValidation( TRUE );
					$pcf->setEnableCalcUserDateID( TRUE );
					$pcf->setEnableCalcTotalTime( TRUE );
					$pcf->setEnableCalcSystemTotalTime( TRUE );
					$pcf->setEnableCalcWeeklySystemTotalTime( TRUE );
					$pcf->setEnableCalcUserDateTotal( TRUE );
					$pcf->setEnableCalcException( TRUE );
					$pcf->setEnablePreMatureException( TRUE ); //Enable pre-mature exceptions at this point.

					Debug::Arr($lf->data, 'Punch Object: ', __FILE__, __LINE__, __METHOD__,10);
					Debug::Arr($pcf->data, 'Punch Control Object: ', __FILE__, __LINE__, __METHOD__,10);
					if ( $pcf->isValid() ) {
						$validator_stats['valid_records']++;
						if ( $pcf->Save( TRUE, TRUE ) != TRUE ) { //Force isNew() lookup.
							$is_valid = $pcf_valid = FALSE;
						}
					} else {
						$is_valid = $pcf_valid = FALSE;
					}
				}
			}
		}

		if ( $is_valid == FALSE ) {
			Debug::Text('Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);

			$lf->FailTransaction(); //Just rollback this single record, continue on to the rest.

			Debug::Text('PF Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);
			$validator[$key] = $lf->Validator->getErrorsArray();
			//Merge PCF validation errors onto array.
			if ( isset($pcf) AND $pcf_valid == FALSE ) {
				Debug::Text('PCF Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);
				$validator[$key] += $pcf->Validator->getErrorsArray();
			}
		}

		$lf->CommitTransaction();

		if ( $validator_stats['valid_records'] > 0 AND $validator_stats['total_records'] == $validator_stats['valid_records'] ) {
			if ( $validator_stats['total_records'] == 1 ) {
				return $this->returnHandler( $save_result[$key] ); //Single valid record
			} else {
				return $this->returnHandler( TRUE, 'SUCCESS', TTi18n::getText('MULTIPLE RECORDS SAVED'), $save_result, $validator_stats ); //Multiple valid records
			}
		} else {
			return $this->returnHandler( FALSE, 'VALIDATION', TTi18n::getText('INVALID DATA'), $validator, $validator_stats );
		}

		return FALSE;
	}

	/**
	 * Get default punch data for creating new punches.
	 * @return array
	 */
	function getPunchDefaultData( $user_id = NULL, $date = NULL, $punch_control_id = NULL, $previous_punch_id = NULL ) {
		$company_obj = $this->getCurrentCompanyObject();

		Debug::Text('Getting punch default data... User ID: '. $user_id .' Date: '. $date .' Punch Control ID: '. $punch_control_id .' Previous Punch Id: '. $previous_punch_id, __FILE__, __LINE__, __METHOD__,10);

		$data = array(
						'status_id' => 10,
						'type_id' => 10,
						'user_id' => $this->getCurrentUserObject()->getId(),
						'punch_time' => TTDate::getAPIDate( 'TIME', TTDate::strtotime( '12:00 PM' ) ),
						'branch_id' => $this->getCurrentUserObject()->getDefaultBranch(),
						'department_id' => $this->getCurrentUserObject()->getDefaultDepartment(),
					);

		//If user_id is specified, use their default branch/department.
		$ulf = TTnew( 'UserListFactory' );
		$ulf->getByIdAndCompanyId( $user_id, $company_obj->getID() );
		if ( $ulf->getRecordCount() == 1 ) {
			$user_obj = $ulf->getCurrent();

			$data['user_id'] = $user_obj->getID();
			$data['branch_id'] = $user_obj->getDefaultBranch();
			$data['department_id'] = $user_obj->getDefaultDepartment();
		}
		unset($ulf, $user_obj);

		//IF specified, get the previous punch object to determine the next punch type/status.
		if ( $previous_punch_id > 0 ) {
			$plf = TTnew('PunchListFactory');
			$plf->getByCompanyIDAndId( $company_obj->getId(), $previous_punch_id );
			if ( $plf->getRecordCount() == 1 ) {
				$prev_punch_obj = $plf->getCurrent();
				$data['type_id'] = $prev_punch_obj->getNextType();
				//$data['status_id'] = $prev_punch_obj->getNextStatus(); //Flex handles this.
				Debug::Text('Getting punch default data... Type ID: '. $data['type_id'], __FILE__, __LINE__, __METHOD__,10);
			}
			unset($plf, $prev_punch_obj);
		}

		return $this->returnHandler( $data );
	}

	/**
	 * Get all necessary dates for building the TimeSheet in a single call, this is mainly as a performance optimization.
	 * @param array $data filter data
	 * @return array
	 */
	function getTimeSheetDates( $base_date ) {
		$epoch = TTDate::parseDateTime( $base_date );

		if ( $epoch == '' ) {
			$epoch = TTDate::getTime();
		}

		$start_date = TTDate::getBeginWeekEpoch( $epoch, $this->getCurrentUserPreferenceObject()->getStartWeekDay() );
		$end_date = TTDate::getEndWeekEpoch( $epoch, $this->getCurrentUserPreferenceObject()->getStartWeekDay() );

		$retarr = array(
						'base_date' => $epoch,
						'start_date' => $start_date,
						'end_date' => $end_date,
						'base_display_date' => TTDate::getAPIDate('DATE', $epoch ),
						'start_display_date' => TTDate::getAPIDate('DATE', $start_date ),
						'end_display_date' => TTDate::getAPIDate('DATE', $end_date),
						);

		return $retarr;
	}

	/**
	 * Get punch data for one or more punches.
	 * @param array $data filter data
	 * @return array
	 */
	function getPunch( $data = NULL, $disable_paging = FALSE ) {
		if ( !$this->getPermissionObject()->Check('punch','enabled')
				OR !( $this->getPermissionObject()->Check('punch','view') OR $this->getPermissionObject()->Check('punch','view_own') OR $this->getPermissionObject()->Check('punch','view_child') ) ) {
			return $this->getPermissionObject()->PermissionDenied();
		}

		$data = $this->initializeFilterAndPager( $data, $disable_paging );

		$data['filter_data']['permission_children_ids'] = $this->getPermissionObject()->getPermissionChildren( 'punch', 'view' );

		//As a performance optimization to prevent the API from having to do additional date lookups, accept a single "date" field, that converts
		//into start/end dates.
		if ( isset($data['filter_data']['date']) AND $data['filter_data']['date'] != '' ) {
			$data['filter_data']['start_date'] = TTDate::getBeginWeekEpoch( $data['filter_data']['date'], $this->getCurrentUserPreferenceObject()->getStartWeekDay() );
			$data['filter_data']['end_date'] = TTDate::getEndWeekEpoch( $data['filter_data']['date'], $this->getCurrentUserPreferenceObject()->getStartWeekDay() );
		}

		//No filter data, restrict to last pay period as a performance optimization when hundreds of thousands of punches exist.
		//The issue with this though is that the API doesn't know what the filter criteria is, so it can't display this to the user.
		if ( count($data['filter_data']) == 1 AND !isset($data['filter_data']['pay_period_ids']) ) {
			Debug::Text('Adding default filter data...', __FILE__, __LINE__, __METHOD__, 10);
			$pplf = TTnew( 'PayPeriodListFactory' );
			$pplf->getByCompanyId( $this->getCurrentCompanyObject()->getId() );
			$pay_period_ids = array_keys((array)$pplf->getArrayByListFactory( $pplf, FALSE, FALSE ) );
			if ( isset($pay_period_ids[0]) AND isset($pay_period_ids[1]) ) {
				$data['filter_data']['pay_period_ids'] = array($pay_period_ids[0], $pay_period_ids[1]);
			}
			unset($pplf, $pay_period_ids);
		}

		$blf = TTnew( 'PunchListFactory' );
		$blf->getAPISearchByCompanyIdAndArrayCriteria( $this->getCurrentCompanyObject()->getId(), $data['filter_data'], $data['filter_items_per_page'], $data['filter_page'], NULL, $data['filter_sort'] );
		Debug::Text('Record Count: '. $blf->getRecordCount(), __FILE__, __LINE__, __METHOD__, 10);
		if ( $blf->getRecordCount() > 0 ) {
			$this->getProgressBarObject()->start( $this->getAMFMessageID(), $blf->getRecordCount() );

			$this->setPagerObject( $blf );

			foreach( $blf as $b_obj ) {
				$retarr[] = $b_obj->getObjectAsArray( $data['filter_columns'], $data['filter_data']['permission_children_ids'] );

				$this->getProgressBarObject()->set( $this->getAMFMessageID(), $blf->getCurrentRow() );
			}

			$this->getProgressBarObject()->stop( $this->getAMFMessageID() );

			return $this->returnHandler( $retarr );
		}

		return $this->returnHandler( TRUE ); //No records returned.
	}

	/**
	 * Get only the fields that are common across all records in the search criteria. Used for Mass Editing of records.
	 * @param array $data filter data
	 * @return array
	 */
	function getCommonPunchData( $data ) {
		return Misc::arrayIntersectByRow( $this->getPunch( $data, TRUE ) );
	}

	/**
	 * Validate punch data for one or more punches.
	 * @param array $data punch data
	 * @return array
	 */
	function validatePunch( $data ) {
		return $this->setPunch( $data, TRUE );
	}

	/**
	 * Set punch data for one or more punches.
	 * @param array $data punch data
	 * @return array
	 */
	function setPunch( $data, $validate_only = FALSE ) {
		$validate_only = (bool)$validate_only;

		if ( !is_array($data) ) {
			return $this->returnHandler( FALSE );
		}

		if ( !$this->getPermissionObject()->Check('punch','enabled')
				OR !( $this->getPermissionObject()->Check('punch','edit') OR $this->getPermissionObject()->Check('punch','edit_own') OR $this->getPermissionObject()->Check('punch','edit_child') OR $this->getPermissionObject()->Check('punch','add') ) ) {
			return  $this->getPermissionObject()->PermissionDenied();
		}

		if ( $validate_only == TRUE ) {
			Debug::Text('Validating Only!', __FILE__, __LINE__, __METHOD__, 10);
			$permission_children_ids = FALSE;
		} else {
			//Get Permission Hierarchy Children first, as this can be used for viewing, or editing.
			$permission_children_ids = $this->getPermissionChildren();
		}

		extract( $this->convertToMultipleRecords($data) );
		Debug::Text('Received data for: '. $total_records .' Punchs', __FILE__, __LINE__, __METHOD__, 10);
		Debug::Arr($data, 'Data: ', __FILE__, __LINE__, __METHOD__, 10);

		$validator_stats = array('total_records' => $total_records, 'valid_records' => 0 );
		if ( is_array($data) ) {
			$this->getProgressBarObject()->start( $this->getAMFMessageID(), $total_records );

			foreach( $data as $key => $row ) {
				$primary_validator = new Validator();
				$lf = TTnew( 'PunchListFactory' );
				$lf->StartTransaction();
				if ( isset($row['id']) AND $row['id'] > 0 ) {
					//Modifying existing object.
					//Get punch object, so we can only modify just changed data for specific records if needed.
					//Use the special getAPIByIdAndCompanyId() function as it returns additional columns needed for mass editing.
					//These additional columns break editing a single record if we make $lf the current object.
					$lf->getAPIByIdAndCompanyId( $row['id'], $this->getCurrentCompanyObject()->getId() );
					if ( $lf->getRecordCount() == 1 ) {
						//Object exists, check edit permissions
						if (
							  $validate_only == TRUE
							  OR
								(
								$this->getPermissionObject()->Check('punch','edit')
									OR ( $this->getPermissionObject()->Check('punch','edit_own') AND $this->getPermissionObject()->isOwner( $lf->getCurrent()->getCreatedBy(), $lf->getCurrent()->getID() ) === TRUE )
									OR ( $this->getPermissionObject()->Check('punch','edit_child') AND $this->getPermissionObject()->isChild( $lf->getCurrent()->getPunchControlObject()->getUserDateObject()->getUser(), $permission_children_ids ) === TRUE )
								) ) {

							Debug::Text('Row Exists, getting current data: ', $row['id'], __FILE__, __LINE__, __METHOD__, 10);
							//If we make the current object be $lf, it fails saving the punch because extra columns exist.
							//$lf = $lf->getCurrent();
							//$row = array_merge( $lf->getObjectAsArray( array('id' => TRUE,'user_id' => TRUE,'transfer' => TRUE,'type_id' => TRUE,'status_id' => TRUE,'time_stamp' => TRUE,'punch_control_id' => TRUE,'actual_time_stamp' => TRUE,'original_time_stamp' => TRUE,'schedule_id' => TRUE,'station_id' => TRUE,'longitude' => TRUE,'latitude' => TRUE,'deleted' => TRUE) ), $row );
							$row = array_merge( $lf->getCurrent()->getObjectAsArray(), $row );
						} else {
							$primary_validator->isTrue( 'permission', FALSE, TTi18n::gettext('Edit permission denied') );
						}
					} else {
						//Object doesn't exist.
						$primary_validator->isTrue( 'id', FALSE, TTi18n::gettext('Edit permission denied, record does not exist') );
					}
				} else {
					//Adding new object, check ADD permissions.
					$primary_validator->isTrue( 'permission', $this->getPermissionObject()->Check('punch','add'), TTi18n::gettext('Add permission denied') );
				}
				Debug::Arr($row, 'Data: ', __FILE__, __LINE__, __METHOD__, 10);

				$is_valid = $pcf_valid = $primary_validator->isValid();
				if ( $is_valid == TRUE ) { //Check to see if all permission checks passed before trying to save data.
					Debug::Text('Setting object data...', __FILE__, __LINE__, __METHOD__, 10);

					$lf->setObjectFromArray( $row );

					//Force Company ID to current company.
					//$lf->setCompany( $this->getCurrentCompanyObject()->getId() );

					$is_valid = $lf->isValid();
					if ( $is_valid == TRUE ) {
						Debug::Text('Saving data...', __FILE__, __LINE__, __METHOD__, 10);
						if ( $validate_only == TRUE ) {
							$save_result[$key] = TRUE;
							$validator_stats['valid_records']++;
						} else {
							//Save Punch object and start on PunchControl
							$save_result[$key] = $lf->Save( FALSE );
							if ( $save_result[$key] == TRUE ) {
								unset($row['id']); //ID must be removed so it doesn't get confused with PunchControlID
								Debug::Text('Saving PCF data... Punch Control ID: '. $lf->getPunchControlID(), __FILE__, __LINE__, __METHOD__, 10);
								$pcf = TTnew( 'PunchControlFactory' );
								$pcf->setId( $lf->getPunchControlID() );
								$pcf->setPunchObject( $lf );

								$pcf->setObjectFromArray( $row );

								$pcf->setEnableStrictJobValidation( TRUE );
								$pcf->setEnableCalcUserDateID( TRUE );
								$pcf->setEnableCalcTotalTime( TRUE );
								$pcf->setEnableCalcSystemTotalTime( TRUE );
								$pcf->setEnableCalcWeeklySystemTotalTime( TRUE );
								$pcf->setEnableCalcUserDateTotal( TRUE );
								$pcf->setEnableCalcException( TRUE );

								if ( $pcf->isValid() ) {
									$validator_stats['valid_records']++;
									if ( $pcf->Save( TRUE, TRUE ) != TRUE ) { //Force isNew() lookup.
										$is_valid = $pcf_valid = FALSE;
									}
								} else {
									$is_valid = $pcf_valid = FALSE;
								}
							}
							Debug::Text('Save Result ID: '. (int)$save_result[$key], __FILE__, __LINE__, __METHOD__, 10);
						}
					}
				}

				if ( $is_valid == FALSE ) {
					Debug::Text('Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);

					$lf->FailTransaction(); //Just rollback this single record, continue on to the rest.

					if ( $primary_validator->isValid() == FALSE ) {
						$validator[$key] = $primary_validator->getErrorsArray();
					} else {
						Debug::Text('PF Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);
						$validator[$key] = $lf->Validator->getErrorsArray();
						//Merge PCF validation errors onto array.
						if ( isset($pcf) AND $pcf_valid == FALSE ) {
							Debug::Text('PCF Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);
							$validator[$key] += $pcf->Validator->getErrorsArray();
						}
					}
				}

				$lf->CommitTransaction();

				$this->getProgressBarObject()->set( $this->getAMFMessageID(), $key );
			}

			$this->getProgressBarObject()->stop( $this->getAMFMessageID() );

			if ( $validator_stats['valid_records'] > 0 AND $validator_stats['total_records'] == $validator_stats['valid_records'] ) {
				if ( $validator_stats['total_records'] == 1 ) {
					return $this->returnHandler( $save_result[$key] ); //Single valid record
				} else {
					return $this->returnHandler( TRUE, 'SUCCESS', TTi18n::getText('MULTIPLE RECORDS SAVED'), $save_result, $validator_stats ); //Multiple valid records
				}
			} else {
				return $this->returnHandler( FALSE, 'VALIDATION', TTi18n::getText('INVALID DATA'), $validator, $validator_stats );
			}
		}

		return $this->returnHandler( FALSE );
	}

	/**
	 * Delete one or more punchs.
	 * @param array $data punch data
	 * @return array
	 */
	function deletePunch( $data ) {
		if ( is_numeric($data) ) {
			$data = array($data);
		}

		if ( !is_array($data) ) {
			return $this->returnHandler( FALSE );
		}

		if ( !$this->getPermissionObject()->Check('punch','enabled')
				OR !( $this->getPermissionObject()->Check('punch','delete') OR $this->getPermissionObject()->Check('punch','delete_own') OR $this->getPermissionObject()->Check('punch','delete_child') ) ) {
			return  $this->getPermissionObject()->PermissionDenied();
		}

		//Get Permission Hierarchy Children first, as this can be used for viewing, or editing.
		$permission_children_ids = $this->getPermissionChildren();

		Debug::Text('Received data for: '. count($data) .' Punchs', __FILE__, __LINE__, __METHOD__, 10);
		Debug::Arr($data, 'Data: ', __FILE__, __LINE__, __METHOD__, 10);

		$total_records = count($data);
		$validator_stats = array('total_records' => $total_records, 'valid_records' => 0 );
		if ( is_array($data) ) {
			$this->getProgressBarObject()->start( $this->getAMFMessageID(), $total_records );

			foreach( $data as $key => $id ) {
				$primary_validator = new Validator();
				$lf = TTnew( 'PunchListFactory' );
				$lf->StartTransaction();
				if ( is_numeric($id) ) {
					//Modifying existing object.
					//Get punch object, so we can only modify just changed data for specific records if needed.
					$lf->getByIdAndCompanyId( $id, $this->getCurrentCompanyObject()->getId() );
					if ( $lf->getRecordCount() == 1 ) {
						//Object exists, check edit permissions
						//NOTE: Make sure we pass the user the punch is assigned too for proper delete_child permissions to work correctly.
						if ( $this->getPermissionObject()->Check('punch','delete')
								OR ( $this->getPermissionObject()->Check('punch','delete_own') AND $this->getPermissionObject()->isOwner( $lf->getCurrent()->getCreatedBy(), $lf->getCurrent()->getID() ) === TRUE )
								OR ( $this->getPermissionObject()->Check('punch','delete_child') AND $this->getPermissionObject()->isChild( $lf->getCurrent()->getPunchControlObject()->getUserDateObject()->getUser(), $permission_children_ids ) === TRUE )) {
							Debug::Text('Record Exists, deleting record: '. $id, __FILE__, __LINE__, __METHOD__, 10);
							$lf = $lf->getCurrent();
						} else {
							$primary_validator->isTrue( 'permission', FALSE, TTi18n::gettext('Delete permission denied') );
						}
					} else {
						//Object doesn't exist.
						$primary_validator->isTrue( 'id', FALSE, TTi18n::gettext('Delete permission denied, record does not exist') );
					}
				} else {
					$primary_validator->isTrue( 'id', FALSE, TTi18n::gettext('Delete permission denied, record does not exist') );
				}

				//Debug::Arr($lf, 'AData: ', __FILE__, __LINE__, __METHOD__, 10);

				$is_valid = $primary_validator->isValid();
				if ( $is_valid == TRUE ) { //Check to see if all permission checks passed before trying to save data.
					Debug::Text('Attempting to delete record...', __FILE__, __LINE__, __METHOD__, 10);
					Debug::Arr($lf->data, 'Current Data: ', __FILE__, __LINE__, __METHOD__, 10);

					$lf->setUser( $lf->getPunchControlObject()->getUserDateObject()->getUser() );
					$lf->setDeleted(TRUE);

					$is_valid = $lf->isValid();
					if ( $is_valid == TRUE ) {
						$lf->setEnableCalcTotalTime( TRUE );
						$lf->setEnableCalcSystemTotalTime( TRUE );
						$lf->setEnableCalcWeeklySystemTotalTime( TRUE );
						$lf->setEnableCalcUserDateTotal( TRUE );
						$lf->setEnableCalcException( TRUE );

						Debug::Text('Record Deleted...', __FILE__, __LINE__, __METHOD__, 10);
						$save_result[$key] = $lf->Save();
						$validator_stats['valid_records']++;
					}
				}

				if ( $is_valid == FALSE ) {
					Debug::Text('Data is Invalid...', __FILE__, __LINE__, __METHOD__, 10);

					$lf->FailTransaction(); //Just rollback this single record, continue on to the rest.

					if ( $primary_validator->isValid() == FALSE ) {
						$validator[$key] = $primary_validator->getErrorsArray();
					} else {
						$validator[$key] = $lf->Validator->getErrorsArray();
					}
				}

				$lf->CommitTransaction();

				$this->getProgressBarObject()->set( $this->getAMFMessageID(), $key );
			}

			$this->getProgressBarObject()->stop( $this->getAMFMessageID() );

			if ( $validator_stats['valid_records'] > 0 AND $validator_stats['total_records'] == $validator_stats['valid_records'] ) {
				if ( $validator_stats['total_records'] == 1 ) {
					return $this->returnHandler( $save_result[$key] ); //Single valid record
				} else {
					return $this->returnHandler( TRUE, 'SUCCESS', TTi18n::getText('MULTIPLE RECORDS SAVED'), $save_result, $validator_stats ); //Multiple valid records
				}
			} else {
				return $this->returnHandler( FALSE, 'VALIDATION', TTi18n::getText('INVALID DATA'), $validator, $validator_stats );
			}
		}

		return $this->returnHandler( FALSE );
	}

	function getMealAndBreakTotalTime( $data, $disable_paging = FALSE  ) {
		return PunchFactory::calcMealAndBreakTotalTime( $this->getPunch( $data, TRUE ) );
	}
}
?>
