"""
$RCSfile: Generators.py,v $

Default generators for result objects

Author: Philipp von Weitershausen <philikon@philikon.de>

$Id: Generators.py,v 1.2 2003/03/23 04:41:41 cstrong Exp $
"""

from Interface import Base
from GeneratorRegistry import GeneratorRegistry

class IGenerator(Base):
    """
    Interface for Output Generators
    """

    def createObject(id, title, rawString, **kw):
        """
        Create the object and return its result
        """
    def getResult(obj, client, REQUEST=None, RESPONSE=None):
        """
        Call the object and return its result
        """

class DTMLMethodGenerator:

    __implements__ = IGenerator

    def createObject(self, id, title, rawString, **kw):
        from OFS.DTMLMethod import DTMLMethod
        resultObj = DTMLMethod(rawString)
        resultObj.id = id
        resultObj.title  = title
        for name, value in kw.items():
            setattr(resultObj, name, value)
        return resultObj

    def getResult(self, obj, client, REQUEST=None, RESPONSE=None):
        return obj.__of__(client)(client, REQUEST, RESPONSE)

DTMLMethodGenerator = DTMLMethodGenerator()
GeneratorRegistry.register("DTML Method", DTMLMethodGenerator)

class DTMLDocumentGenerator:

    __implements__ = IGenerator

    def createObject(self, id, title, rawString, **kw):
        from OFS.DTMLDocument import DTMLDocument
        resultObj = DTMLDocument(rawString)
        resultObj.id = id
        resultObj.title  = title
        for name, value in kw.items():
            setattr(resultObj, name, value)
        return resultObj

    def getResult(self, obj, client, REQUEST=None, RESPONSE=None):
        return obj.__of__(client)(client, REQUEST, RESPONSE)

DTMLDocumentGenerator = DTMLDocumentGenerator()
GeneratorRegistry.register("DTML Document", DTMLDocumentGenerator)

class FileGenerator:

    __implements__ = IGenerator

    def createObject(self, id, title, rawString, **kw):
        from OFS.Image import File
        content_type = kw.get("content_type", "text/plain")
        resultObj = File(id, title, rawString, content_type)
        for name, value in kw.items():
            setattr(resultObj, name, value)
        return resultObj

    def getResult(self, obj, client, REQUEST=None, RESPONSE=None):
        return obj.__of__(client).index_html(REQUEST, RESPONSE)

FileGenerator = FileGenerator()
GeneratorRegistry.register("File", FileGenerator)

class ZPTGenerator:

    __implements__ = IGenerator

    def createObject(self, id, title, rawString, **kw):
        from Products.PageTemplates.ZopePageTemplate import ZopePageTemplate
        content_type = kw.get("content_type", "text/html")
        resultObj = ZopePageTemplate(id, rawString, content_type)
        resultObj.pt_setTitle(title)
        for name, value in kw.items():
            setattr(resultObj, name, value)
        return resultObj

    def getResult(self, obj, client, REQUEST=None, RESPONSE=None):
        return obj.__of__(client)(client, REQUEST, RESPONSE)

ZPTGenerator = ZPTGenerator()
try:
    import Products.PageTemplates
    GeneratorRegistry.register("Page Template", ZPTGenerator)
except ImportError:
    pass

class ParsedXMLGenerator:

    __implements__ = IGenerator

    def createObject(self, id, title, rawString, **kw):
        from Products.ParsedXML.ParsedXML import ParsedXML
        content_type = kw.get("content_type", "text/xml")
        resultObj = ParsedXML(id, rawString, contentType = content_type)
        return resultObj

    def getResult(self, obj, client, REQUEST=None, RESPONSE=None):
        return obj.__of__(client).index_html(REQUEST, RESPONSE)

ParsedXMLGenerator = ParsedXMLGenerator()
try:
    import Products.ParsedXML
    GeneratorRegistry.register("Parsed XML", ParsedXMLGenerator)
except ImportError:
    pass
