#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

import fcntl
import getpass
import os
import pty
import shlex
import signal
import struct
import sys
import termios
import time
import tty
from select import select

import pyte

from fenrirscreenreader.core import debug
from fenrirscreenreader.core.eventData import FenrirEventType
from fenrirscreenreader.core.screenDriver import ScreenDriver as screenDriver
from fenrirscreenreader.utils import screen_utils


class FenrirScreen(pyte.Screen):
    def set_margins(self, *args, **kwargs):
        kwargs.pop("private", None)
        super(FenrirScreen, self).set_margins(*args, **kwargs)


class Terminal:
    def __init__(self, columns, lines, p_in):
        self.text = ""
        self.attributes = None
        self.screen = FenrirScreen(columns, lines)
        self.screen.write_process_input = lambda data: p_in.write(
            data.encode()
        )
        self.stream = pyte.ByteStream()
        self.stream.attach(self.screen)

    def feed(self, data):
        self.stream.feed(data)

    def update_attributes(self, initialize=False):
        buffer = self.screen.buffer
        lines = None
        if not initialize:
            lines = self.screen.dirty
        else:
            lines = range(self.screen.lines)
            self.attributes = [
                [
                    list(attribute[1:]) + [False, "default", "default"]
                    for attribute in line.values()
                ]
                for line in buffer.values()
            ]
        for y in lines:
            try:
                t = self.attributes[y]
            except Exception as e:
                # Terminal class doesn't have access to env, use fallback
                # logging
                print(
                    f"ptyDriver Terminal update_attributes: Error accessing "
                    f"attributes: {e}"
                )
                self.attributes.append([])

            self.attributes[y] = [
                list(attribute[1:]) + [False, "default", "default"]
                for attribute in (buffer[y].values())
            ]
            if len(self.attributes[y]) < self.screen.columns:
                diff = self.screen.columns - len(self.attributes[y])
                self.attributes[y] += [
                    [
                        "default",
                        "default",
                        False,
                        False,
                        False,
                        False,
                        False,
                        False,
                        "default",
                        "default",
                    ]
                ] * diff

    def resize(self, lines, columns):
        self.screen.resize(lines, columns)
        self.set_cursor()
        self.update_attributes(True)

    def set_cursor(self, x=-1, y=-1):
        x_pos = x
        y_pos = y
        if x_pos == -1:
            x_pos = self.screen.cursor.x
        if y_pos == -1:
            y_pos = self.screen.cursor.y
        self.screen.cursor.x = min(
            self.screen.cursor.x, self.screen.columns - 1
        )
        self.screen.cursor.y = min(self.screen.cursor.y, self.screen.lines - 1)

    def get_screen_content(self):
        cursor = self.screen.cursor
        self.text = "\n".join(self.screen.display)
        self.update_attributes(self.attributes is None)
        self.screen.dirty.clear()
        return {
            "cursor": (cursor.x, cursor.y),
            "lines": self.screen.lines,
            "columns": self.screen.columns,
            "text": self.text,
            "attributes": self.attributes.copy(),
            "screen": "pty",
            "screenUpdateTime": time.time(),
        }.copy()


class driver(screenDriver):
    def __init__(self):
        screenDriver.__init__(self)
        self.signalPipe = os.pipe()
        self.p_out = None
        self.terminal = None
        self.p_pid = -1
        signal.signal(signal.SIGWINCH, self.handle_sigwinch)

    def initialize(self, environment):
        self.env = environment
        self.command = self.env["runtime"]["SettingsManager"].get_setting(
            "general", "shell"
        )
        self.shortcutType = self.env["runtime"][
            "InputManager"
        ].get_shortcut_type()
        self.env["runtime"]["ProcessManager"].add_custom_event_thread(
            self.terminal_emulation
        )

    def get_curr_screen(self):
        self.env["screen"]["oldTTY"] = "pty"
        self.env["screen"]["newTTY"] = "pty"

    def inject_text_to_screen(self, msg_bytes, screen=None):
        if not screen:
            screen = self.p_out.fileno()
        if isinstance(msg_bytes, str):
            msg_bytes = bytes(msg_bytes, "UTF-8")
        os.write(screen, msg_bytes)

    def get_session_information(self):
        self.env["screen"]["autoIgnoreScreens"] = []
        self.env["general"]["prev_user"] = getpass.getuser()
        self.env["general"]["curr_user"] = getpass.getuser()

    def read_all(self, fd, timeout=0.3, interruptFd=None, len=65536):
        msg_bytes = b""
        fd_list = []
        fd_list += [fd]
        if interruptFd:
            fd_list += [interruptFd]
        starttime = time.time()
        while True:
            r = screen_utils.has_more_what(fd_list, 0.0001)
            # nothing more to read
            if fd not in r:
                break
            data = os.read(fd, len)
            if data == b"":
                raise EOFError
            msg_bytes += data
            # exit on interrupt available
            if interruptFd in r:
                break
            # respect timeout but wait a little bit of time to see if something
            # more is here
            if (time.time() - starttime) >= timeout:
                break
        return msg_bytes

    def open_terminal(self, columns, lines, command):
        p_pid, master_fd = pty.fork()
        if p_pid == 0:  # Child.
            argv = shlex.split(command)
            env = os.environ.copy()
            # values are VT100,xterm-256color,linux
            try:
                if env["TERM"] == "":
                    env["TERM"] = "linux"
            except Exception as e:
                # Child process doesn't have access to env, use fallback
                # logging
                print(
                    f"ptyDriver spawnTerminal: Error checking TERM environment: {e}"
                )
                env["TERM"] = "linux"
            os.execvpe(argv[0], argv, env)
        # File-like object for I/O with the child process aka command.
        p_out = os.fdopen(master_fd, "w+b", 0)
        return Terminal(columns, lines, p_out), p_pid, p_out

    def resize_terminal(self, fd):
        s = struct.pack("HHHH", 0, 0, 0, 0)
        s = fcntl.ioctl(0, termios.TIOCGWINSZ, s)
        fcntl.ioctl(fd, termios.TIOCSWINSZ, s)
        lines, columns, _, _ = struct.unpack("hhhh", s)
        return lines, columns

    def get_terminal_size(self, fd):
        s = struct.pack("HHHH", 0, 0, 0, 0)
        lines, columns, _, _ = struct.unpack(
            "HHHH", fcntl.ioctl(fd, termios.TIOCGWINSZ, s)
        )
        return lines, columns

    def handle_sigwinch(self, *args):
        os.write(self.signalPipe[1], b"w")

    def terminal_emulation(self, active, event_queue):
        try:
            old_attr = termios.tcgetattr(sys.stdin)
            tty.setraw(0)
            lines, columns = self.get_terminal_size(0)
            if self.command == "":
                self.command = screen_utils.get_shell()
            self.terminal, self.p_pid, self.p_out = self.open_terminal(
                columns, lines, self.command
            )
            lines, columns = self.resize_terminal(self.p_out)
            self.terminal.resize(lines, columns)
            fd_list = [sys.stdin, self.p_out, self.signalPipe[0]]
            while active.value:
                r, _, _ = select(fd_list, [], [], 1)
                # none
                if r == []:
                    continue
                # signals
                if self.signalPipe[0] in r:
                    os.read(self.signalPipe[0], 1)
                    lines, columns = self.resize_terminal(self.p_out)
                    self.terminal.resize(lines, columns)
                # input
                if sys.stdin in r:
                    try:
                        msg_bytes = self.read_all(sys.stdin.fileno(), len=4096)
                    except (EOFError, OSError):
                        event_queue.put(
                            {
                                "Type": FenrirEventType.stop_main_loop,
                                "data": None,
                            }
                        )
                        break
                    if self.shortcutType == "KEY":
                        try:
                            self.inject_text_to_screen(msg_bytes)
                        except Exception as e:
                            self.env["runtime"][
                                "DebugManager"
                            ].write_debug_out(
                                "ptyDriver getInputData: Error injecting text to screen: "
                                + str(e),
                                debug.DebugLevel.ERROR,
                            )
                            event_queue.put(
                                {
                                    "Type": FenrirEventType.stop_main_loop,
                                    "data": None,
                                }
                            )
                            break
                    else:
                        event_queue.put(
                            {
                                "Type": FenrirEventType.byte_input,
                                "data": msg_bytes,
                            }
                        )
                # output
                if self.p_out in r:
                    try:
                        msg_bytes = self.read_all(
                            self.p_out.fileno(), interruptFd=sys.stdin.fileno()
                        )
                    except (EOFError, OSError):
                        event_queue.put(
                            {
                                "Type": FenrirEventType.stop_main_loop,
                                "data": None,
                            }
                        )
                        break
                    # feed and send event bevore write, the pyte already has the right state
                    # so fenrir already can progress bevore os.write what
                    # should give some better reaction time
                    self.terminal.feed(msg_bytes)
                    event_queue.put(
                        {
                            "Type": FenrirEventType.screen_update,
                            "data": screen_utils.create_screen_event_data(
                                self.terminal.get_screen_content()
                            ),
                        }
                    )
                    self.inject_text_to_screen(
                        msg_bytes, screen=sys.stdout.fileno()
                    )
        except Exception as e:  # Process died?
            print(e)
            event_queue.put(
                {"Type": FenrirEventType.stop_main_loop, "data": None}
            )
        finally:
            os.kill(self.p_pid, signal.SIGTERM)
            self.p_out.close()
            termios.tcsetattr(sys.stdin, termios.TCSADRAIN, old_attr)
            event_queue.put(
                {"Type": FenrirEventType.stop_main_loop, "data": None}
            )
            sys.exit(0)

    def get_curr_application(self):
        pass
