package cache

import (
	"time"

	"github.com/gofiber/fiber/v3"
	"github.com/gofiber/utils/v2"
)

// Config defines the config for middleware.
type Config struct {
	// Storage is used to store the state of the middleware
	//
	// Default: an in-memory store for this process only
	Storage fiber.Storage

	// Next defines a function to skip this middleware when returned true.
	//
	// Optional. Default: nil
	Next func(c fiber.Ctx) bool

	// CacheInvalidator defines a function to invalidate the cache when returned true
	//
	// Optional. Default: nil
	CacheInvalidator func(fiber.Ctx) bool

	// Key allows you to generate custom keys, by default c.Path() is used
	//
	// Default: func(c fiber.Ctx) string {
	//   return utils.CopyString(c.Path())
	// }
	KeyGenerator func(fiber.Ctx) string

	// ExpirationGenerator allows you to generate a custom expiration per request.
	// If nil, the Expiration value is used.
	//
	// Default: nil
	ExpirationGenerator func(fiber.Ctx, *Config) time.Duration

	// CacheHeader header on response header, indicate cache status, with the following possible return value
	//
	// hit, miss, unreachable
	//
	// Optional. Default: X-Cache
	CacheHeader string

	// You can specify HTTP methods to cache.
	// The middleware just caches the routes of its methods in this slice.
	//
	// Default: []string{fiber.MethodGet, fiber.MethodHead}
	Methods []string

	// Expiration is the time that a cached response will live
	//
	// Optional. Default: 5 * time.Minute
	Expiration time.Duration

	// Max number of bytes of response bodies simultaneously stored in cache. When limit is reached,
	// entries with the nearest expiration are deleted to make room for new.
	// 0 means no limit
	//
	// Optional. Default: 1 * 1024 * 1024
	MaxBytes uint

	// DisableValueRedaction turns off masking cache keys in logs and error messages when set to true.
	//
	// Optional. Default: false
	DisableValueRedaction bool

	// DisableCacheControl disables client side caching if set to true
	//
	// Optional. Default: false
	DisableCacheControl bool

	// StoreResponseHeaders allows you to store additional headers generated by
	// next middlewares and handlers.
	//
	// Default: false
	StoreResponseHeaders bool
}

// ConfigDefault is the default config
var ConfigDefault = Config{
	Next:                  nil,
	Expiration:            5 * time.Minute,
	CacheHeader:           "X-Cache",
	DisableCacheControl:   false,
	CacheInvalidator:      nil,
	DisableValueRedaction: false,
	KeyGenerator: func(c fiber.Ctx) string {
		return utils.CopyString(c.Path())
	},
	ExpirationGenerator:  nil,
	StoreResponseHeaders: false,
	Storage:              nil,
	MaxBytes:             1 * 1024 * 1024,
	Methods:              []string{fiber.MethodGet, fiber.MethodHead},
}

// Helper function to set default values
func configDefault(config ...Config) Config {
	// Return default config if nothing provided
	if len(config) < 1 {
		return ConfigDefault
	}

	// Override default config
	cfg := config[0]

	// Set default values
	if cfg.Next == nil {
		cfg.Next = ConfigDefault.Next
	}
	if int(cfg.Expiration.Seconds()) == 0 {
		cfg.Expiration = ConfigDefault.Expiration
	}
	if cfg.CacheHeader == "" {
		cfg.CacheHeader = ConfigDefault.CacheHeader
	}
	if cfg.KeyGenerator == nil {
		cfg.KeyGenerator = ConfigDefault.KeyGenerator
	}
	if len(cfg.Methods) == 0 {
		cfg.Methods = ConfigDefault.Methods
	}
	return cfg
}
