-- Copyright 2020 United States Government as represented by the Administrator
-- of the National Aeronautics and Space Administration. All Rights Reserved.
--
-- Disclaimers
--
-- Licensed under the Apache License, Version 2.0 (the "License"); you may
-- not use this file except in compliance with the License. You may obtain a
-- copy of the License at
--
--      https://www.apache.org/licenses/LICENSE-2.0
--
-- Unless required by applicable law or agreed to in writing, software
-- distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
-- WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
-- License for the specific language governing permissions and limitations
-- under the License.
--
-- | CLI interface to the CStructs2Copilot subcommand
module CLI.CommandCStructs2Copilot
    (
      -- * Direct command access
      command
    , CommandOpts
    , ErrorCode

      -- * CLI
    , commandDesc
    , commandOptsParser
    )
  where

-- External imports
import Options.Applicative ( Parser, help, long, metavar, strOption )

-- External imports: command results
import Command.Result ( Result )

-- External imports: actions or commands supported
import Command.CStructs2Copilot ( ErrorCode, cstructs2Copilot )

-- * Command

-- | A Copilot correspondence for a C struct can be generated by providing
-- the path to the file that declares the struct.
newtype CommandOpts = CommandOpts
  { cstructsInputFile :: FilePath }

-- | Generate Copilot struct definitions and instances from structs defined in
-- a C header file.
--
-- This is just an uncurried version of "Command.CStructs2Copilot".
command :: CommandOpts -> IO (Result ErrorCode)
command c = cstructs2Copilot (cstructsInputFile c)

-- * CLI

-- | Command description for CLI help.
commandDesc :: String
commandDesc = "Generate Copilot structs from C structs"

-- | Subparser for the @structs@ command, used to generate Copilot declarations
-- from C structs.
commandOptsParser :: Parser CommandOpts
commandOptsParser = CommandOpts
  <$> strOption
        (  long "input-file"
        <> metavar "FILENAME"
        <> help strStructsInputFileArgDesc
        )

-- | Argument C header file to struct conversion command
strStructsInputFileArgDesc :: String
strStructsInputFileArgDesc = "C header file with struct definitions"
