
/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the msXpertSuite project.
 *
 * The msXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once

/////////////////////// StdLib includes


/////////////////////// Qt includes
#include <QWidget>

/////////////////////// QCustomPlot


/////////////////////// pappsomspp includes
#include <pappsomspp/gui/plotwidget/basetraceplotwidget.h>
#include <pappsomspp/core/trace/trace.h>


/////////////////////// Local includes
#include "BasePlotCompositeWidget.hpp"
#include "../nongui/MsRunDataSet.hpp"
#include "../nongui/ProcessingFlow.hpp"
#include "SavitzkyGolayFilterParamsDlg.hpp"

namespace MsXpS
{
namespace MineXpert
{

struct PeakLabel
{
  QCPItemText *rp_qcpItemText = nullptr;
  QPointF position;

  PeakLabel(QCustomPlot *plot_p, const QString text, const QPointF &position)
  {
    rp_qcpItemText = new QCPItemText(plot_p);
    rp_qcpItemText->setText(text);
    rp_qcpItemText->setLayer("plotsLayer");
    rp_qcpItemText->setColor(QColor("black"));

    rp_qcpItemText->setPositionAlignment(Qt::AlignBottom | Qt::AlignLeft);
    rp_qcpItemText->position->setType(QCPItemPosition::ptPlotCoords);
    rp_qcpItemText->position->setCoords(position.x(), position.y());
    rp_qcpItemText->setRotation(-60);
    rp_qcpItemText->setVisible(true);
  }
};

using PeakLabelSPtr = std::shared_ptr<PeakLabel>;

class BaseTracePlotWnd;
class MsFragmentationSpecDlg;

// The BaseTracePlotCompositeWidget class is aimed at receiving QCPGraph_s, not
// QCPColorMap_s. This class implements all the basic features related to the
// presence of QCPGraph_s, like adding a Trace plot, for example, which would
// not work if the plot were of type QCPColorMap...
class BaseTracePlotCompositeWidget: public BasePlotCompositeWidget
{
  Q_OBJECT

  friend class BasePlotWnd;
  friend class BaseTracePlotWnd;

  public:
  explicit BaseTracePlotCompositeWidget(QWidget *parent,
                                        const QString &x_axis_label,
                                        const QString &y_axis_label);

  virtual ~BaseTracePlotCompositeWidget();

  virtual QCPGraph *addTrace(const pappso::Trace &trace,
                             QCPAbstractPlottable *parent_plottable_p,
                             MsRunDataSetCstSPtr ms_run_data_set_csp,
                             const QString &sample_name,
                             const ProcessingFlow &processing_flow,
                             const QColor &color);

  virtual void
  moveMouseCursorToNextGraphPoint(const pappso::BasePlotContext &context);

  void showSavitzkyGolayFilterParamsDlg();

  void loadFilterParamsFromSettings(const QString &filter_name,
                                    QString &filter_parameters);
  void writeFilterParamsToSettings(const QString &filter_name,
                                   const QString &filter_parameters);

  void savitzkyGolayFilterParamsChanged(pappso::SavGolParams sav_gol_params);

  void runFilter(const QString &filter_name);
  void runSavitzkyGolayFilter();
  void runLowIntensityThresholdRemovalFilter(QWidgetAction *widget_action_p);
  void removeZeroIntensityDataPointsFilter();

  pappso::Trace pickPeaksOnlyOnVisibleMaxima();

  virtual void createLabellingSessionMenu();
  virtual void labelPeak(const QString &text, const QPointF &plot_point);
  virtual void labelPeaks(const pappso::Trace &trace);
  virtual void toggleAllPeakLabelsVisibility();
  virtual void removeFirstPeakLabel();
  virtual void removeLastPeakLabel();
  virtual void removeAllPeakLabels();
  virtual void exportPeakLabelsToClipboard();

  bool savePeakListToClipboard(bool only_in_visible_range = false);
  bool savePeakListToFile(bool only_in_visible_range = false);


  public slots:

  virtual void labellingSessionMenuPushButtonClicked();

  virtual void
  plotWidgetKeyPressEvent(const pappso::BasePlotContext &context) override;

  virtual void
  plotWidgetKeyReleaseEvent(const pappso::BasePlotContext &context) override;

  virtual void
  plotWidgetMousePressEvent(const pappso::BasePlotContext &context) override;

  virtual void
  plotWidgetMouseReleaseEvent(const pappso::BasePlotContext &context) override;

  virtual void setNoiseSampleTrace();
  virtual void resetNoiseSampleTrace();

  virtual void pinPointPeakApexCheckBoxStateChanged(Qt::CheckState state);

  virtual void
  savePeakListOnlyForVisibleRangeCheckBoxStateChanged(Qt::CheckState state);

  signals:

  void plotWidgetMouseReleaseEventSignal(QMouseEvent *event);

  protected:
  // These are the members that are specific to the BaseTrace plot composite
  // widget.
  QMenu *mp_labelSessionMenu          = nullptr;
  bool m_isSinglePointIntegrationMode = false;
  QCPRange m_integrationRange;
  SavitzkyGolayFilterParamsDlg *mp_savitzkyGolayFilterParamsDlg = nullptr;

  bool m_isPinPointingPeakApex = false;
  std::vector<PeakLabelSPtr> m_peakLabels;

  bool m_isSavePeakListOnlyForVisibleRange = false;

  virtual void createMainMenu() override;
  void configureOutIntegrationCheckButtons();
  pappso::Trace lowIntensitySignalRemoval(pappso::Trace &trace, double value);
};


} // namespace MineXpert

} // namespace MsXpS
