% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nnetar.R
\name{forecast.nnetar}
\alias{forecast.nnetar}
\title{Forecasting using neural network models}
\usage{
\method{forecast}{nnetar}(
  object,
  h = if (object$m > 1) 2 * object$m else 10,
  PI = FALSE,
  level = c(80, 95),
  fan = FALSE,
  xreg = NULL,
  lambda = object$lambda,
  bootstrap = FALSE,
  npaths = 1000,
  innov = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{nnetar} resulting from a call to
\code{\link[=nnetar]{nnetar()}}.}

\item{h}{Number of periods for forecasting. If \code{xreg} is used, \code{h}
is ignored and the number of forecast periods is set to the number of rows
of \code{xreg}.}

\item{PI}{If \code{TRUE}, prediction intervals are produced, otherwise only point
forecasts are calculated. If \code{PI} is \code{FALSE}, then \code{level},
\code{fan}, \code{bootstrap} and \code{npaths} are all ignored.}

\item{level}{Confidence levels for prediction intervals.}

\item{fan}{If \code{TRUE}, \code{level} is set to \code{seq(51, 99, by = 3)}.
This is suitable for fan plots.}

\item{xreg}{Future values of any regression variables. A numerical vector or
matrix of external regressors; it should not be a data frame.}

\item{lambda}{Box-Cox transformation parameter. If \code{lambda = "auto"},
then a transformation is automatically selected using \code{BoxCox.lambda}.
The transformation is ignored if NULL. Otherwise,
data transformed before model is estimated.}

\item{bootstrap}{If \code{TRUE}, then prediction intervals are produced by
simulation using resampled errors (rather than normally distributed errors). Ignored if \code{innov} is not \code{NULL}.}

\item{npaths}{Number of sample paths used in computing simulated prediction
intervals.}

\item{innov}{Values to use as innovations for prediction intervals. Must be
a matrix with \code{h} rows and \code{npaths} columns (vectors are coerced
into a matrix). If present, \code{bootstrap} is ignored.}

\item{...}{Additional arguments passed to \code{\link[=simulate.nnetar]{simulate.nnetar()}}.}
}
\value{
An object of class \code{forecast}.
}
\description{
Returns forecasts and other information for univariate neural network
models.
}
\details{
Prediction intervals are calculated through simulations and can be slow.
Note that if the network is too complex and overfits the data, the residuals
can be arbitrarily small; if used for prediction interval calculations, they
could lead to misleadingly small values. It is possible to use out-of-sample
residuals to ameliorate this, see examples.
}
\section{forecast class}{

An object of class \code{forecast} is a list usually containing at least
the following elements:
\describe{
\item{model}{A list containing information about the fitted model}
\item{method}{The name of the forecasting method as a character string}
\item{mean}{Point forecasts as a time series}
\item{lower}{Lower limits for prediction intervals}
\item{upper}{Upper limits for prediction intervals}
\item{level}{The confidence values associated with the prediction intervals}
\item{x}{The original time series.}
\item{residuals}{Residuals from the fitted model. For models with additive
errors, the residuals will be x minus the fitted values.}
\item{fitted}{Fitted values (one-step forecasts)}
}
The function \code{summary} can be used to obtain and print a summary of the
results, while the functions \code{plot} and \code{autoplot} produce plots of the forecasts and
prediction intervals. The generic accessors functions \code{fitted.values} and \code{residuals}
extract various useful features from the underlying model.
}

\examples{
## Fit & forecast model
fit <- nnetar(USAccDeaths, size = 2)
fcast <- forecast(fit, h = 20)
plot(fcast)

\dontrun{
## Include prediction intervals in forecast
fcast2 <- forecast(fit, h = 20, PI = TRUE, npaths = 100)
plot(fcast2)

## Set up out-of-sample innovations using cross-validation
fit_cv <- CVar(USAccDeaths, size = 2)
res_sd <- sd(fit_cv$residuals, na.rm = TRUE)
myinnovs <- rnorm(20 * 100, mean = 0, sd = res_sd)
## Forecast using new innovations
fcast3 <- forecast(fit, h = 20, PI = TRUE, npaths = 100, innov = myinnovs)
plot(fcast3)
}

}
\seealso{
\code{\link[=nnetar]{nnetar()}}.
}
\author{
Rob J Hyndman and Gabriel Caceres
}
\keyword{ts}
