#[derive(Debug, Clone, Copy)]
pub(crate) struct Id {
    pub usage: &'static str,
    pub ty: &'static str,
    pub version: Option<&'static str>,
    pub long_name: &'static str,
    pub short_name: &'static str,
}

impl Id {
    pub(crate) const fn new(
        usage: &'static str,
        ty: &'static str,
        version: Option<&'static str>,
        long_name: &'static str,
        short_name: &'static str,
    ) -> Self {
        //TODO: use gettext
        //https://github.com/storaged-project/udisks/blob/4f24c900383d3dc28022f62cab3eb434d19b6b82/udisks/udisksclient.c#L2025
        Self {
            usage,
            ty,
            version,
            long_name,
            short_name,
        }
    }
}

pub(crate) const ID_TYPES: [Id; 46] = [
    Id::new(
        "filesystem",
        "vfat",
        Some("FAT12"),
        "FAT (12-bit version)",
        "FAT",
    ),
    Id::new(
        "filesystem",
        "vfat",
        Some("FAT16"),
        "FAT (16-bit version)",
        "FAT",
    ),
    Id::new(
        "filesystem",
        "vfat",
        Some("FAT32"),
        "FAT (32-bit version)",
        "FAT",
    ),
    Id::new("filesystem", "vfat", Some("*"), "FAT (version %s)", "FAT"),
    Id::new("filesystem", "vfat", None, "FAT", "FAT"),
    Id::new("filesystem", "ntfs", Some("*"), "NTFS (version %s)", "NTFS"),
    Id::new("filesystem", "ntfs", None, "NTFS", "NTFS"),
    Id::new("filesystem", "hfs", None, "HFS", "HFS"),
    Id::new("filesystem", "hfsplus", None, "HFS+", "HFS+"),
    Id::new("filesystem", "apfs", None, "APFS", "APFS"),
    Id::new("filesystem", "ext2", Some("*"), "Ext2 (version %s)", "Ext2"),
    Id::new("filesystem", "ext2", None, "Ext2", "Ext2"),
    Id::new("filesystem", "ext3", Some("*"), "Ext3 (version %s)", "Ext3"),
    Id::new("filesystem", "ext3", None, "Ext3", "Ext3"),
    Id::new("filesystem", "ext4", Some("*"), "Ext4 (version %s)", "Ext4"),
    Id::new("filesystem", "ext4", None, "Ext4", "Ext4"),
    Id::new(
        "filesystem",
        "jdb",
        Some("*"),
        "Journal for Ext (version %s)",
        "JDB",
    ),
    Id::new("filesystem", "jdb", Some("*"), "Journal for Ext", "JDB"),
    Id::new("filesystem", "xfs", Some("*"), "XFS (version %s)", "XFS"),
    Id::new("filesystem", "xfs", None, "XFS", "XFS"),
    //TODO: No ID_FS_VERSION yet for btrfs...
    //https://github.com/storaged-project/udisks/blob/4f24c900383d3dc28022f62cab3eb434d19b6b82/udisks/udisksclient.c#L2045C6-L2045C45
    Id::new("filesystem", "btrfs", None, "Btrfs", "Btrfs"),
    Id::new(
        "filesystem",
        "iso9660",
        Some("*"),
        "ISO 9660 (version %s)",
        "ISO9660",
    ),
    Id::new("filesystem", "iso9660", None, "ISO 9660", "ISO9660"),
    Id::new("filesystem", "udf", Some("*"), "UDF (version %s)", "UDF"),
    Id::new("filesystem", "udf", None, "UDF", "UDF"),
    Id::new("filesystem", "exfat", None, "exFAT", "exFAT"),
    Id::new(
        "filesystem",
        "exfat",
        Some("*"),
        "exFAT (version %s)",
        "exFAT",
    ),
    Id::new("other", "swap", Some("*"), "Swap (version %s)", "Swap"),
    Id::new("other", "swap", None, "Swap", "Swap"),
    Id::new(
        "raid",
        "LVM2_member",
        Some("*"),
        "LVM2 Physical Volume (%s)",
        "LVM2 PV",
    ),
    Id::new(
        "raid",
        "LVM2_member",
        None,
        "LVM2 Physical Volume",
        "LVM2 PV",
    ),
    Id::new(
        "raid",
        "linux_raid_member",
        Some("*"),
        "Linux RAID Member (version %s)",
        "Linux RAID Member",
    ),
    Id::new(
        "raid",
        "linux_raid_member",
        None,
        "Linux RAID Member",
        "Linux RAID Member",
    ),
    Id::new(
        "raid",
        "zfs_member",
        Some("*"),
        "ZFS Device (ZPool version %s)",
        "ZFS (v%s)",
    ),
    Id::new("raid", "zfs_member", None, "ZFS Device", "ZFS"),
    Id::new(
        "raid",
        "isw_raid_member",
        Some("*"),
        "Intel Rapid Storage Technology enterprise RAID Member (version %s)",
        "Intel RSTe RAID Member (%s)",
    ),
    Id::new(
        "raid",
        "isw_raid_member",
        None,
        "Intel Rapid Storage Technology enterprise RAID Member",
        "Intel RSTe RAID Member",
    ),
    Id::new(
        "crypto",
        "crypto_unknown",
        None,
        "Possibly Encrypted",
        "Encrypted?",
    ),
    Id::new(
        "crypto",
        "crypto_LUKS",
        Some("*"),
        "LUKS Encryption (version %s)",
        "LUKS",
    ),
    Id::new("crypto", "crypto_LUKS", None, "LUKS Encryption", "LUKS"),
    Id::new(
        "crypto",
        "crypto_TCRYPT",
        None,
        "VeraCrypt/TrueCrypt Encryption",
        "VeraCrypt/TrueCrypt",
    ),
    Id::new(
        "crypto",
        "BitLocker",
        None,
        "BitLocker Encryption",
        "BitLocker",
    ),
    Id::new(
        "filesystem",
        "VMFS",
        Some("*"),
        "VMFS (version %s)",
        "VMFS (v%s)",
    ),
    Id::new("filesystem", "VMFS", None, "VMFS", "VMFS"),
    Id::new(
        "raid",
        "VMFS_volume_member",
        Some("*"),
        "VMFS Volume Member (version %s)",
        "VMFS Member (v%s)",
    ),
    Id::new(
        "raid",
        "VMFS_volume_member",
        None,
        "VMFS Volume Member",
        "VMFS Member",
    ),
];
