// @HEADER
// *****************************************************************************
//                    Teuchos: Common Tools Package
//
// Copyright 2004 NTESS and the Teuchos contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef TEUCHOS_STANDARD_CATCH_MACROS_HPP
#define TEUCHOS_STANDARD_CATCH_MACROS_HPP

#include "Teuchos_GlobalMPISession.hpp"
#include "Teuchos_FancyOStream.hpp"
#include "Teuchos_TypeNameTraits.hpp"
#include "Teuchos_stacktrace.hpp"
#include "Teuchos_Assert.hpp"


#ifdef HAVE_TEUCHOS_STACKTRACE
#  define TEUCHOS_GET_STORED_STACKTRACE() \
  (Teuchos::TestForException_getEnableStacktrace() \
    ? Teuchos::get_stored_stacktrace() + "\n" \
    : std::string() )
#else
#  define TEUCHOS_GET_STORED_STACKTRACE() ""
#endif


/** \brief Implementation of TEUCHOS_STANDARD_CATCH_STATEMENTS(...) that
 * allows toggle of showing stack trace.
 */
#define TEUCHOS_STANDARD_CATCH_STATEMENTS_IMPL(VERBOSE, ERR_STREAM, \
  SHOW_STACK_TRACE, SUCCESS_FLAG \
  ) \
  catch (const std::exception &excpt) { \
    if((VERBOSE)) { \
      std::ostringstream oss; \
      oss \
        << "\np="<<::Teuchos::GlobalMPISession::getRank() \
        <<": *** Caught standard std::exception of type \'" \
        <<Teuchos::concreteTypeName(excpt)<<"\' :\n\n"; \
      Teuchos::OSTab scsi_tab(oss); \
      if (SHOW_STACK_TRACE) { \
        scsi_tab.o() << TEUCHOS_GET_STORED_STACKTRACE(); \
      } \
      scsi_tab.o() << excpt.what() << std::endl; \
      (ERR_STREAM) << std::flush; \
      (ERR_STREAM) << oss.str(); \
      (SUCCESS_FLAG) = false; \
    } \
  } \
  catch (const int &excpt_code) { \
    if((VERBOSE)) { \
      std::ostringstream oss; \
      oss \
        << "\np="<<::Teuchos::GlobalMPISession::getRank() \
        << ": *** Caught an integer exception with value = " \
        << excpt_code << std::endl; \
      (ERR_STREAM) << std::flush; \
      (ERR_STREAM) << oss.str(); \
      (SUCCESS_FLAG) = false; \
    } \
  } \
  catch ( ... ) { \
    if ((VERBOSE)) { \
      std::ostringstream oss; \
      oss << "\np="<<::Teuchos::GlobalMPISession::getRank() \
          <<": *** Caught an unknown exception\n"; \
      (ERR_STREAM) << std::flush; \
      (ERR_STREAM) << oss.str(); \
      (SUCCESS_FLAG) = false; \
    } \
  }



/** \brief Simple macro that catches and reports standard exceptions and other
 * exceptions.
 *
 * \ingroup teuchos_language_support_grp
 *
 * This macro should be used to write simple <tt>main()</tt> program functions
 * wrapped in a try statement as:

 \code

 int main(...)
 {
   bool verbose = true;
   bool success = true;
   try {
     ...
   }
   TEUCHOS_STANDARD_CATCH_STATEMENTS(verbose, std::cerr, success);
   return ( success ? EXIT_SUCCESS : EXIT_FAILURE );
 }
 \endcode
 *
 * NOTE: This macro will print out stacktraces generated by exceptions thrown
 * with the TEUCHOS_TEST_FOR_EXCEPTION() family of macros (see the main doc
 * page for details).
 */
#define TEUCHOS_STANDARD_CATCH_STATEMENTS(VERBOSE, ERR_STREAM, SUCCESS_FLAG) \
  TEUCHOS_STANDARD_CATCH_STATEMENTS_IMPL(VERBOSE, ERR_STREAM, true, SUCCESS_FLAG)


#endif // TEUCHOS_STANDARD_CATCH_MACROS_HPP
