require 'spec_helper'

describe 'collectd::plugin::apache', type: :class do
  on_supported_os(baseline_os_hash).each do |os, facts|
    context "on #{os} " do
      let :facts do
        facts
      end

      let :pre_condition do
        'include collectd'
      end

      options = os_specific_options(facts)
      context 'default' do
        it 'defaults' do
          content = <<EOS
# Generated by Puppet
<LoadPlugin apache>
  Globals false
</LoadPlugin>

<Plugin "apache">
  <Instance "localhost">
    URL "http://localhost/mod_status?auto"
  </Instance>
</Plugin>

EOS

          is_expected.to compile.with_all_deps
          is_expected.to contain_file('apache.load').with(
            content: content,
            path: "#{options[:plugin_conf_dir]}/10-apache.conf"
          )
        end
      end

      options = os_specific_options(facts)
      context 'multiple instances' do
        let :params do
          {
            instances: {
              site1: {
                'url' => 'https://example.com',
                'user' => 'nobody',
                'password' => 'hidden',
                'verifypeer' => true,
                'verifyhost' => false,
                'cacert' => '/etc/barfoo/ca.crt'
              },
              site2: {
                'url' => 'https://another.example.com',
                'user' => 'nobody',
                'password' => 'secrets',
                'verifypeer' => false,
                'verifyhost' => true,
                'cacert' => '/etc/foobar/ca.crt',
                'sslciphers' => 'TLS_CHACHA20_POLY1305_SHA256:TLS_AES_128_GCM_SHA256',
                'timeout' => 120
              }
            }
          }
        end

        it 'overrides defaults' do
          content = <<EOS
# Generated by Puppet
<LoadPlugin apache>
  Globals false
</LoadPlugin>

<Plugin "apache">
  <Instance "site1">
    URL "https://example.com"
    User "nobody"
    Password "hidden"
    VerifyPeer true
    VerifyHost false
    CACert "/etc/barfoo/ca.crt"
  </Instance>
  <Instance "site2">
    URL "https://another.example.com"
    User "nobody"
    Password "secrets"
    VerifyPeer false
    VerifyHost true
    CACert "/etc/foobar/ca.crt"
    SSLCiphers "TLS_CHACHA20_POLY1305_SHA256:TLS_AES_128_GCM_SHA256"
    Timeout 120
  </Instance>
</Plugin>

EOS

          is_expected.to compile.with_all_deps
          is_expected.to contain_file('apache.load').with(
            content: content,
            path: "#{options[:plugin_conf_dir]}/10-apache.conf"
          )
        end
      end

      case facts[:os]['family']
      when 'RedHat'
        context ':manage_package => true on osfamily => RedHat' do
          let :params do
            {
              manage_package: true
            }
          end

          it 'Will manage collectd-apache' do
            is_expected.to compile.with_all_deps
            is_expected.to contain_package('collectd-apache').with(ensure: 'present',
                                                                   name: 'collectd-apache')
            is_expected.to contain_class('collectd')
            is_expected.to contain_file('apache.load')
          end
        end

        context ':manage_package => false on osfamily => RedHat' do
          let :params do
            {
              manage_package: false
            }
          end

          it 'Will not manage collectd-apache' do
            is_expected.to compile.with_all_deps
            is_expected.not_to contain_package('collectd-apache').with(ensure: 'present',
                                                                       name: 'collectd-apache')
            is_expected.to contain_class('collectd')
            is_expected.to contain_file('apache.load')
          end
        end
      end
    end
  end
end
