% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval-tidy.R
\name{eval_tidy}
\alias{eval_tidy}
\title{Evaluate an expression with quosures and pronoun support}
\usage{
eval_tidy(expr, data = NULL, env = caller_env())
}
\arguments{
\item{expr}{An expression to evaluate.}

\item{data}{A data frame, or named list or vector. Alternatively, a
data mask created with \code{\link[=as_data_mask]{as_data_mask()}} or \code{\link[=new_data_mask]{new_data_mask()}}.}

\item{env}{The environment in which to evaluate \code{expr}. This
environment is always ignored when evaluating quosures. Quosures
are evaluated in their own environment.}
}
\description{
\code{eval_tidy()} is a variant of \code{\link[base:eval]{base::eval()}} that powers the tidy
evaluation framework. Like \code{eval()} it accepts user data as
argument. If supplied, it evaluates its input \code{expr} in a \link[=as_data_mask]{data
mask}. In additon \code{eval_tidy()} supports:
\itemize{
\item \link[=quotation]{Quosures}. The expression wrapped in the quosure
evaluates in its original context (masked by \code{data} if supplied).
\item \link[=.data]{Pronouns}. If \code{data} is supplied, the \code{.env} and \code{.data}
pronouns are installed in the data mask. \code{.env} is a reference to
the calling environment and \code{.data} refers to the \code{data} argument.
These pronouns lets you be explicit about where to find
values and throw errors if you try to access non-existent values.
}
}
\section{Life cycle}{


\code{eval_tidy()} is stable.
}

\examples{
# With simple quoted expressions eval_tidy() works the same way as
# eval():
apple <- "apple"
kiwi <- "kiwi"
expr <- quote(paste(apple, kiwi))
expr

eval(expr)
eval_tidy(expr)

# Both accept a data mask as argument:
data <- list(apple = "CARROT", kiwi = "TOMATO")
eval(expr, data)
eval_tidy(expr, data)


# In addition eval_tidy() has support for quosures:
with_data <- function(data, expr) {
  quo <- enquo(expr)
  eval_tidy(quo, data)
}
with_data(NULL, apple)
with_data(data, apple)
with_data(data, list(apple, kiwi))

# Secondly eval_tidy() installs handy pronouns that allows users to
# be explicit about where to find symbols:
with_data(data, .data$apple)
with_data(data, .env$apple)


# Note that instead of using `.env` it is often equivalent and may
# be preferred to unquote a value. There are two differences. First
# unquoting happens earlier, when the quosure is created. Secondly,
# subsetting `.env` with the `$` operator may be brittle because
# `$` does not look through the parents of the environment.
#
# For instance using `.env$name` in a magrittr pipeline is an
# instance where this poses problem, because the magrittr pipe
# currently (as of v1.5.0) evaluates its operands in a *child* of
# the current environment (this child environment is where it
# defines the pronoun `.`).
\dontrun{
  data \%>\% with_data(!!kiwi)     # "kiwi"
  data \%>\% with_data(.env$kiwi)  # NULL
}
}
\seealso{
\link{quasiquotation} for the second leg of the tidy evaluation
framework.
}
