///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// Macbeth shall never vanquish'd be, until
// Great Birnam wood to high Dunsinane hill
// Shall come against him.
//                                                        Act IV - scene I
//
// "Fear not, till Birnam wood
// Do come to Dunsinane;"
//                                                        Act V - scene V
//                                                William Shakespeare
//                                                                        Macbeth
#include "rheolef.h"
using namespace rheolef;
using namespace std;

point x_hill (0.25, 0, 0);
Float slope = 0.01;
Float nu = 1e-3;
Float t_now = 0;

field 
compose2 (const field & f, const geomap & g)
{
  const space & Vh = f.get_space ();
  const space & Vh_g = g.get_space ();
  if (!(Vh.get_geo () == Vh_g.get_geo ()))
    error_macro ("Spaces do not match");
  field fog (Vh);

  vector<bool> todo(Vh.size ());
  for (field::size_type i = 0; i < Vh.size (); i++)
    todo[i] = true;
  const geo & G = Vh.get_geo ();
  geo::const_iterator K = G.begin ();
  geo::const_iterator last_K = G.end ();
  while (K != last_K)
    {
    tiny_vector < field::size_type > idx;
    tiny_vector < field::size_type > idx_g;
      Vh.set_dof (*K, idx, 0);
      Vh_g.set_dof (*K, idx_g, 0);

      if (idx.size () >= idx_g.size ())
	error_macro ("Spaces are not compatible");
    for (field::size_type i = 0; i < idx.size (); i++)
	{
	  fog.at (idx[i]) = f.evaluate (g.at (idx_g[i]));
	  //TO BE MODIFIED
	  todo[idx[i]] = false;
	}

      K++;
    }
  //TO BE MODIFIED
  for (field::size_type j = 0; j < Vh.size (); j++)
    if (todo[j])
      error_macro ("Node " << j << " not done");
  return fog;
}


Float 
phi_exact (const point & x, Float t)
{
  return slope / (slope + 4 * nu * t)
    * exp (-(sqr (x[0] * cos (t) + x[1] * sin (t) - x_hill[0])
	     + sqr (-x[0] * sin (t) + x[1] * cos (t) - x_hill[1])) / (slope + 4 * nu * t));
}
Float 
phi_now (const point & x)
{
  return phi_exact (x, t_now);
}
Float 
phi_ini (const point & x)
{
  return phi_exact (x, 0);
}

void 
usage (string codename)
{
  // EXERGO
  cerr << "\n"
    << " Macbeth shall never vanquish'd be, until\n"
    << " Great Birnam wood to high Dunsinane hill\n"
    << " Shall come against him.                 \n"
    << "                                         \n"
    << "                     William Shakespeare \n"
    << "                            Macbeth,IV-1 \n";

  cerr << "Usage : " << codename << " <mesh.geo> [-hill <x_center> <y_center>] [-slope <sigma>]\n"
    << "            [-T <Tmax>] [-nu <nu>] [-tabata] [-jacobian] [-implicit]\n"
    << "            [-deltat <dt>|-iter <nb>|-opt_iter] [-X1|-X2] [-P1|-P2]\n";
  exit (1);
}

Float 
get_x (const point & x)
{
  return x[0];
}
Float 
get_minus_y (const point & x)
{
  return -x[1];
}

geomap 
caracteristiques (space & Vh, Float dt, bool use_RK2)
{
  space VVh = Vh * Vh;
  field u (VVh);
  field uoX (VVh);

  u[0] = interpolate (Vh, get_minus_y);
  u[1] = interpolate (Vh, get_x);

  if (!use_RK2)
    return geomap (Vh, -dt * u);
  else
    {
      geomap X_half (Vh, -dt / 2 * u);
      uoX[0] = compose ((field) u[0], X_half);
      uoX[1] = compose ((field) u[1], X_half);
      return geomap (Vh, -dt * uoX);
    }
}

Float 
calcul (const geo & omega, const string & approx, const string & der_approx,
	const string & out_dir, Float deltat, int iter, bool w_RK2,
	bool w_tabata, bool w_jacobian, bool w_implicit)
{
  Float erreur = 0;
  Float erreur_im = 0;

  // Define spaces and forms
  space Vh (omega, approx);
  Vh.block ("boundary");
  space Mh (omega, der_approx);
  space MMh = Mh * Mh;

  form m (Vh, Vh, "mass");
  form inv_m (Mh, Mh, "inv_mass");
  form grad (Vh, MMh, "grad");
  form gradT = trans (grad);
  form lapl (Vh, Vh, "grad_grad");

  form op = m + deltat * nu / 2 * lapl;
  // operateur a inverser (toutes methodes)

  field phi = interpolate (Vh, phi_ini);
  field prec_phi (Vh);
  field prec_grad_phi (MMh);

  geomap X = caracteristiques (Vh, deltat, w_RK2);

  ssk < Float > op_fact = ldlt (op.uu);

  for (int im = 1; im - 1 < iter; im++)
    {
      t_now = im * deltat;

      cerr << "t=" << t_now;
      prec_phi = compose (phi, X);

      if (!w_tabata)
	{
	  // methode de pironneau
	  prec_grad_phi = grad * phi;
	  prec_grad_phi[0] = inv_m * (prec_grad_phi[0]);
	  prec_grad_phi[1] = inv_m * (prec_grad_phi[1]);
	}
      else
	{
	  // methode de tabata : ordre 1 
	  prec_grad_phi = grad * phi;
	  cerr << "-";
	  prec_grad_phi[0] = compose2 (inv_m * (prec_grad_phi[0]), X);
	  cerr << "-";
	  prec_grad_phi[1] = compose2 (inv_m * (prec_grad_phi[1]), X);
	  cerr << "-";

	  if (w_jacobian)
	    {			// ordre 2

	      if (!w_implicit)
		{		// calcul explicite de la jacobienne

		  field swap = (field) prec_grad_phi[0];
		  prec_grad_phi[0] = swap + deltat * (field) prec_grad_phi[1];
		  prec_grad_phi[1] = (field) prec_grad_phi[1] - deltat * swap;
		}
	      else
		{		// calcul implicite de la jacobienne

		  field grad_prec_phi = deltat / (1 + deltat) * grad * prec_phi;
		  prec_grad_phi = prec_grad_phi - grad_prec_phi;
		  prec_grad_phi = (1 + deltat) * prec_grad_phi;
		}
	    }
	}

      cerr << "-";
      phi.u = op_fact.solve (m.uu * prec_phi.u + m.ub * prec_phi.b
			     - op.ub * phi.b
			     - deltat * nu / 2 * (gradT.uu * prec_grad_phi.u + gradT.ub * prec_grad_phi.b));
      field e = phi - interpolate (Vh, phi_now);
      erreur_im = sqrt(m(e,e));
      erreur = (erreur_im > erreur) ? erreur_im : erreur;

      cerr << ", erreur=" << erreur_im << "\n";
      string out_filename = out_dir + "/" + itos (im) + ".mfield";
      ofstream fout (out_filename.c_str ());
      fout << catchmark ("phi") << phi
	<< catchmark ("phi_exact") << interpolate (Vh, phi_now);

    }
  return erreur;
}

int 
main (int argc, char **argv)
{
  // GET ARGS
  // defaults
  string out_dir = ".";
  Float T = 3.14159265;
  bool w_optdt = false;
  Float epsilon_opt = 1e-2;
  int maxiter_opt = 10;
  bool w_tabata = false;
  bool w_jacobian = false;
  bool w_implicit = false;
  Float deltat = 0.2;
  int iter = 0;
  bool use_deltat = false;
  bool use_iter = false;
  bool w_RK2 = true;
  string approx = "P1";
  string der_approx = "P0";


  if (argc == 2)
    if (argv[1][0] == '-')
      usage (argv[0]);
  if (argc == 1)
    usage (argv[0]);
  geo omega (argv[1]);
  omega.init_localizer (omega["boundary"]);

  for (int i = 2; i < argc; i++)
    {
      if (argv[i][0] != '-')
	usage (argv[0]);
      // Test here for no-value parameters
      if (strcmp (argv[i], "-tabata") == 0)
	w_tabata = true;
      else if (strcmp (argv[i], "-notabata") == 0)
	w_tabata = false;
      else if (strcmp (argv[i], "-jacobian") == 0)
	w_jacobian = true;
      else if (strcmp (argv[i], "-nojacobian") == 0)
	w_jacobian = false;
      else if (strcmp (argv[i], "-implicit") == 0)
	w_implicit = true;
      else if (strcmp (argv[i], "-noimplicit") == 0)
	w_implicit = false;
      else if (strcmp (argv[i], "-X1") == 0)
	w_RK2 = false;
      else if (strcmp (argv[i], "-X2") == 0)
	w_RK2 = true;
      else if (strcmp (argv[i], "-P1") == 0)
	{
	  approx = "P1";
	  der_approx = "P0";
	}
      else if (strcmp (argv[i], "-P2") == 0)
	{
	  approx = "P2";
	  der_approx = "P1d";
	}
      else if (strcmp (argv[i], "-iter_tabata") == 0 && !use_deltat && !use_iter)
	{
	  use_iter = true;
	  iter = (int) ceil ((5 * sqrt (sqrt (2.)) / 2 / sqrt (omega.hmax ())));
	}
      else if (strcmp (argv[i], "-opt_iter") == 0 && !use_deltat && !use_iter)
	{
	  w_optdt = true;
	  use_iter = true;
	}
      else
	// 1 valued pars
      if (i + 1 == argc)
	usage (argv[0]);
      else if (strcmp (argv[i], "-maxiter_opt") == 0)
	maxiter_opt = atoi (argv[++i]);
      else if (strcmp (argv[i], "-epsilon_opt") == 0)
	epsilon_opt = atof (argv[++i]);
      else if (strcmp (argv[i], "-slope") == 0)
	slope = atof (argv[++i]);
      else if (strcmp (argv[i], "-T") == 0)
	T = atof (argv[++i]);
      else if (strcmp (argv[i], "-deltat") == 0 && !use_iter)
	{
	  use_deltat = true;
	  deltat = atof (argv[++i]);
	}
      else if (strcmp (argv[i], "-iter") == 0 && !use_deltat && !use_iter) {
	  use_iter = true;
	  iter = atoi (argv[++i]);
      } else if (strcmp (argv[i], "-nu") == 0) {
	nu = atof (argv[++i]);
      } else if (i + 2 >= argc) { // 2 valued pars
	usage (argv[0]);
      } else if (strcmp (argv[i], "-hill") == 0) {
	int i1 = ++i;
	int i2 = ++i;
	x_hill = point (atof (argv[i1]), atof (argv[i2]), 0);
      } else { // error
	usage (argv[0]);
      }
  }
  if (!use_iter)
    iter = (int) ceil (T / deltat);
  else if (w_optdt)
    {
      Float e1 = 0, e2 = 0, e3 = 0;
      deltat /= 2;
      cerr << "Optimizing delta t\n";
      e1 = calcul (omega, approx, der_approx, out_dir, deltat, 20,
		   w_RK2, w_tabata, w_jacobian, w_implicit) / deltat;
      e2 = calcul (omega, approx, der_approx, out_dir, 2 * deltat, 20,
		   w_RK2, w_tabata, w_jacobian, w_implicit) / deltat / 2;
      e3 = calcul (omega, approx, der_approx, out_dir, 4 * deltat, 20,
		   w_RK2, w_tabata, w_jacobian, w_implicit) / deltat / 4;
      if (e1 < e3)
	while (e1 <= e2)
	  {
	    e3 = e2;
	    e2 = e1;
	    deltat /= 2;
	    e1 = calcul (omega, approx, der_approx, out_dir, deltat, 20,
			 w_RK2, w_tabata, w_jacobian, w_implicit) / deltat;
	  }
      if (e3 < e1)
	while (e3 <= e2)
	  {
	    e1 = e2;
	    e2 = e3;
	    deltat *= 2;
	    e3 = calcul (omega, approx, der_approx, out_dir, deltat * 4, 20,
		      w_RK2, w_tabata, w_jacobian, w_implicit) / deltat / 4;
	  }
      // now e2 is smaller than e1 and e3, start dichotomy
      Float step1 = deltat;
      Float step2 = deltat * 2;
      Float el;
      Float er;
      int it = 0;
      while ((it <= maxiter_opt) && (e1 + e3 - 2 * e2) / (e1 + e3) >= epsilon_opt)
	{
	  it++;
	  el = calcul (omega, approx, der_approx, out_dir, deltat + step1 / 2, 20,
	    w_RK2, w_tabata, w_jacobian, w_implicit) / (deltat + step1 / 2);
	  if (el <= e2)
	    {
	      e3 = e2;
	      e2 = el;
	      step2 = (step1 /= 2);
	    }
	  else
	    {
	      er = calcul (omega, approx, der_approx, out_dir, deltat + step1 + step2 / 2,
			   20, w_RK2, w_tabata, w_jacobian, w_implicit)
		/ (deltat + step1 + step2 / 2);
	      if (er <= e2)
		{
		  e1 = e2;
		  e2 = er;
		  deltat += step1;
		  step1 = (step2 /= 2);
		}
	      else
		{
		  e1 = el;
		  e3 = er;
		  deltat += (step1 /= 2);
		  step2 /= 2;
		}
	    }
	}
      deltat += step1;
      iter = (int) ceil (T / deltat);
      cerr << "The value " << deltat << " is the best quality/price ratio !\n";
    }
  else
    deltat = T / iter;

  if ((!w_tabata || !w_jacobian) && (w_jacobian || w_implicit))
    error_macro ("Unconsistent options for resolution");

  // Write synopsis
  string synopsis_filename = out_dir + "/synopsis.txt";
  ofstream synopsis (synopsis_filename.c_str ());
  synopsis << "Command line was : \n";
  for (int i = 0; i < argc; i++)
    synopsis << argv[i] << " ";

  synopsis << "\n\nParameters were taken as :\n"
    << "\n slope         :\t" << slope
    << "\n x_hill        :\t" << x_hill
    << "\n nu            :\t" << nu
    << "\n optimize dt   :\t" << (w_optdt ? (string) "yes" : (string) "no")
    << "\n delta_t       :\t" << deltat
    << "\n tabata ratio  :\t" << iter / (5 * sqrt (sqrt (2.)) / 2 / sqrt (omega.hmin ()))
    << "\n tabata ratio  :\t" << iter / (5 * sqrt (sqrt (2.)) / 2 / sqrt (omega.hmax ()))
    << "\n pironneau     :\t" << (!w_tabata ? (string) "yes" : (string) "no")
    << "\n tabata o(1)   :\t"
    << ((w_tabata && !w_jacobian) ? (string) "yes" : (string) "no")
    << "\n tabata o(2)   :\t"
    << ((w_tabata && w_jacobian && !w_implicit) ? (string) "yes" : (string) "no")
    << "\n tabata o(2)'  :\t"
    << ((w_tabata && w_jacobian && w_implicit) ? (string) "yes" : (string) "no")
    << "\n Runge-Kutta 2 :\t" << (w_RK2 ? (string) "yes" : (string) "no")
    << "\n approximations:\t" << approx << " and " << der_approx
    << "\n h_min :\t" << omega.hmin ()
    << "\n h_max :\t" << omega.hmax ()
    << "\n x_min :\t" << omega.xmin ()
    << "\n x_max :\t" << omega.xmax ()
    ;
  synopsis.close ();

  Float erreur = calcul (omega, approx, der_approx, out_dir, deltat, iter,
			 w_RK2, w_tabata, w_jacobian, w_implicit);

  numeric_flags<Float>::output_as_double(true); // when using bigfloat !
  cerr << "The end :\ndelta_t        erreur\n";
  cout << double(deltat) << "\t" << double(erreur) << "\n";
}
