///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/config.h"

#ifdef _RHEOLEF_HAVE_SPOOLES

#include "rheolef/ssk.h"
#include "rheolef/csr.h"
#include "rheolef/vec.h"
using namespace rheolef;
using namespace std;

#ifdef FREE // from malloc_dbg.h
#undef FREE // for spooles/Utilities/MM.h(67)
#endif

extern "C" {
#include <misc.h>
#include <FrontMtx.h>
#include <SymbFac.h>
}

struct chv_spooles_rep {

    int           _neqns;
    bool          _no_empty;
    FrontMtx      *_frontmtx;
    ETree         *_frontETree;
    SubMtxManager *_mtxmanager;
    IV            *_newToOldIV;
    IV            *_oldToNewIV;
    IVL           *_symbfacIVL;

// Does not pass with 64 bit compiling:
    typedef vector<int>::size_type  size_type;
// Does pass:
//    typedef unsigned int  size_type;

    typedef double       element_type;

    void constructor();
    void destructor();
    void factorize (
        size_type                            n,
        vector<size_type>::const_iterator    ia,
        vector<size_type>::const_iterator    ja,
        vector<element_type>::const_iterator a);
  
    void solve (
        vector<element_type>::const_iterator b,
        vector<element_type>::iterator       x,
        bool                                 use_perm = false);
};
void
chv_spooles_rep::constructor()
{
    _neqns      = 0;
    _frontmtx   = 0;
    _frontETree = 0;
    _mtxmanager = 0;
    _newToOldIV = 0;
    _oldToNewIV = 0;
    _symbfacIVL = 0;
}
void
chv_spooles_rep::destructor()
{
    if (_frontmtx)   FrontMtx_free(_frontmtx);
    if (_frontETree) ETree_free(_frontETree);
    if (_mtxmanager) SubMtxManager_free(_mtxmanager);
    if (_newToOldIV) IV_free(_newToOldIV);
    if (_oldToNewIV) IV_free(_oldToNewIV);
    if (_symbfacIVL) IVL_free(_symbfacIVL);
}
void
chv_spooles_rep::factorize (
    size_type                            n,
    vector<size_type>::const_iterator    ia,
    vector<size_type>::const_iterator    ja,
    vector<element_type>::const_iterator a)
{
    if (n == 0 || ia[n] == 0) {
        // empty matrix
        return;
    }
    int msglvl = 1;
    FILE *msgFile = stdout;
    
    // type -- type of entries
    //  1 -- real entries
    //  2 -- complex entries
    int type = SPOOLES_REAL;
    
    // symmetryflag -- type of symmetry in the matrix
    //   0 -- symmetric
    //   1 -- hermitian
    //   2 -- nonsymmetric
    int symmetryflag = 0;
    
    // pivotingflag -- type of pivoting in the factorization
    //   0 -- no pivoting
    //   1 -- pivoting
    int pivotingflag   = 0;
    
    // seed -- random number seed, used for ordering"
    int seed           = 10101;
    /*
       --------------------------------------------
       STEP 1: read the entries from the input
               and create the InpMtx object
       --------------------------------------------
    */
    int nrow = n;
    int ncol = n;
    int nent = n+(ia[n]-n)/2; // symmetric part
    _neqns = nrow ;
    InpMtx *mtxA = InpMtx_new() ;
    InpMtx_init(mtxA, INPMTX_BY_ROWS, type, nent, _neqns) ;
    for (unsigned int i = 0 ; i < n; i++) {
        int irow = i;
        for (unsigned int p = ia[i] ; p < ia[i+1]; p++ ) {
	  int jcol = ja[p];
          double value = a[p];
          if (jcol <= irow) {
	      InpMtx_inputRealEntry(mtxA, irow, jcol, value) ;
          }
       }
    }
#ifdef _RHEOLEF_HAVE_SPOOLES_2_0
    InpMtx_changeStorageMode(mtxA, INPMTX_BY_CHEVRONS);
#endif

    if ( msglvl > 1 ) {
       fprintf(msgFile, "\n\n input matrix") ;
       InpMtx_writeForHumanEye(mtxA, msgFile) ;
       fflush(msgFile) ;
    }
    /*
       -------------------------------------------------
       STEP 2 : find a low-fill ordering
       (1) create the Graph object
       (2) order the graph using multiple minimum degree
       -------------------------------------------------
    */
    Graph *graph = Graph_new() ;
    IVL *adjIVL = InpMtx_fullAdjacency(mtxA) ;
    int nedges = IVL_tsize(adjIVL) ;
    Graph_init2(graph, 0, _neqns, 0, nedges, _neqns, nedges, adjIVL,
                NULL, NULL) ;
    if ( msglvl > 1 ) {
       fprintf(msgFile, "\n\n graph of the input matrix") ;
       Graph_writeForHumanEye(graph, msgFile) ;
       fflush(msgFile) ;
    }
    _frontETree = orderViaMMD(graph, seed, msglvl, msgFile) ;
    if ( msglvl > 1 ) {
       fprintf(msgFile, "\n\n front tree from ordering") ;
       ETree_writeForHumanEye(_frontETree, msgFile) ;
       fflush(msgFile) ;
    }
    Graph_free(graph) ;
    /*
       -----------------------------------------------------
       STEP 3: get the permutation, permute the matrix and 
               front tree and get the symbolic factorization
       -----------------------------------------------------
    */
    _oldToNewIV = ETree_oldToNewVtxPerm(_frontETree) ;
    _newToOldIV = ETree_newToOldVtxPerm(_frontETree) ;
    int *oldToNew   = IV_entries(_oldToNewIV) ;
    int *newToOld   = IV_entries(_newToOldIV) ;
    ETree_permuteVertices(_frontETree, _oldToNewIV) ;
    InpMtx_permute(mtxA, oldToNew, oldToNew) ;
    InpMtx_mapToUpperTriangle(mtxA) ;
#ifndef _RHEOLEF_HAVE_SPOOLES_2_0 /* assume spooles 2.2 */
    InpMtx_changeCoordType  (mtxA, INPMTX_BY_CHEVRONS);
    InpMtx_changeStorageMode(mtxA, INPMTX_BY_VECTORS);
#endif
    _symbfacIVL = SymbFac_initFromInpMtx(_frontETree, mtxA) ;
    if ( msglvl > 1 ) {
       fprintf(msgFile, "\n\n old-to-new permutation vector") ;
       IV_writeForHumanEye(_oldToNewIV, msgFile) ;
       fprintf(msgFile, "\n\n new-to-old permutation vector") ;
       IV_writeForHumanEye(_newToOldIV, msgFile) ;
       fprintf(msgFile, "\n\n front tree after permutation") ;
       ETree_writeForHumanEye(_frontETree, msgFile) ;
       fprintf(msgFile, "\n\n input matrix after permutation") ;
       InpMtx_writeForHumanEye(mtxA, msgFile) ;
       fprintf(msgFile, "\n\n symbolic factorization") ;
       IVL_writeForHumanEye(_symbfacIVL, msgFile) ;
       fflush(msgFile) ;
    }
    /*
       ------------------------------------------
       STEP 4: initialize the front matrix object
       ------------------------------------------
    */
    _frontmtx   = FrontMtx_new();
    _mtxmanager = SubMtxManager_new();
    SubMtxManager_init(_mtxmanager, NO_LOCK, 0) ;
    FrontMtx_init(_frontmtx, _frontETree, _symbfacIVL, type, symmetryflag, 
                  FRONTMTX_DENSE_FRONTS, pivotingflag, NO_LOCK, 0, NULL, 
                  _mtxmanager, msglvl, msgFile) ;
    /*
       -----------------------------------------
       STEP 5: compute the numeric factorization
       -----------------------------------------
    */
    ChvManager *chvmanager = ChvManager_new() ;
    ChvManager_init(chvmanager, NO_LOCK, 1) ;
    double cpus[10] ;
    DVfill(10, cpus, 0.0) ;
    int stats[20] ;
    IVfill(20, stats, 0) ;
    double tau = 100;
#ifdef _RHEOLEF_HAVE_SPOOLES_2_0
    Chv *rootchv = FrontMtx_factorInpMtx(_frontmtx, mtxA, tau, 0.0, chvmanager,
                                    cpus, stats, msglvl, msgFile) ;
#else /* assume spooles 2.2 */
    int error = -1;
    Chv *rootchv = FrontMtx_factorInpMtx(_frontmtx, mtxA, tau, 0.0, chvmanager,
                                    &error, cpus, stats, msglvl, msgFile) ;
    if (error >= 0) {
       error_macro("error encountered at front " << error);
    }
#endif
    double fact_cpu = 0;
    for (int k = 0; k < 10; k++) {
	fact_cpu += cpus[k];
    }
    // cerr << "fact_cpu = " << fact_cpu << endl;
    
    ChvManager_free(chvmanager) ;
    if ( msglvl > 1 ) {
       fprintf(msgFile, "\n\n factor matrix") ;
       FrontMtx_writeForHumanEye(_frontmtx, msgFile) ;
       fflush(msgFile) ;
    }
    if ( rootchv != NULL ) {
       error_macro("matrix found to be singular");
    }

    InpMtx_free(mtxA) ;
    /*
       --------------------------------------
       STEP 6: post-process the factorization
       --------------------------------------
    */
    FrontMtx_postProcess(_frontmtx, msglvl, msgFile) ;
    if ( msglvl > 1 ) {
       fprintf(msgFile, "\n\n factor matrix after post-processing") ;
       FrontMtx_writeForHumanEye(_frontmtx, msgFile) ;
       fflush(msgFile) ;
    }
}
void
chv_spooles_rep::solve (
    vector<element_type>::const_iterator b,
    vector<element_type>::iterator       x,
    bool                                 use_perm)
{
    if (_frontmtx == 0) {
        // empty matrix
	fill (x, x+_neqns, numeric_limits<double>::max());
    }
    int msglvl = 1;
    FILE *msgFile = stdout;
    
    // type -- type of entries
    //  1 -- real entries
    //  2 -- complex entries
    int type = 1;
    
    /*
       -----------------------------------------
       STEP 7: read the right hand side matrix B
       -----------------------------------------
    */
    DenseMtx *mtxB = DenseMtx_new() ;
    int nrhs = 1;
    DenseMtx_init(mtxB, type, 0, 0, _neqns, nrhs, 1, _neqns);
    DenseMtx_zero(mtxB);
    for (int i = 0 ; i < _neqns; i++) {
       int irow = i;
       int jrhs = 0;
       double value = b[i];
       DenseMtx_setRealEntry(mtxB, irow, jrhs, value);
    }
    /*
       ---------------------------------------------------------
       STEP 8: permute the right hand side into the new ordering
       ---------------------------------------------------------
    */
    if (use_perm) {
        DenseMtx_permuteRows(mtxB, _oldToNewIV) ;
    }
    /*
       -------------------------------
       STEP 9: solve the linear system
       -------------------------------
    */
    DenseMtx *mtxX = DenseMtx_new() ;
    DenseMtx_init(mtxX, type, 0, 0, _neqns, nrhs, 1, _neqns) ;
    DenseMtx_zero(mtxX) ;
    double cpus[10] ;
    DVfill(10, cpus, 0.0) ;
    FrontMtx_solve(_frontmtx, mtxX, mtxB, _mtxmanager, cpus, msglvl, msgFile) ;
    double solve_cpu = 0;
    for (int k = 0; k < 10; k++) {
	solve_cpu += cpus[k];
    }
    // cerr << "solve_cpu = " << solve_cpu << endl;
    
    /*
       --------------------------------------------------------
       STEP 10: permute the solution into the original ordering
       --------------------------------------------------------
    */
    if (use_perm) {
        DenseMtx_permuteRows(mtxX, _newToOldIV) ;
    }
    // store in x[]
    for (int i = 0; i < _neqns; i++) {
	int irow = i;
	int jcol = 0;
        double value;
	DenseMtx_realEntry(mtxX, irow, jcol, &value);
	x[i] = value;
    }
    /*
       -----------
       free memory
       -----------
    */
    DenseMtx_free(mtxX) ;
    DenseMtx_free(mtxB) ;
}
inline
chv_spooles_rep& gerrep(void *p)
{
    return *((chv_spooles_rep*)p);
}
template<class T>
spooles_rep<T>::spooles_rep ()
{
    _p = new_macro(chv_spooles_rep);
    gerrep(_p).constructor();
}
template<class T>
spooles_rep<T>::~spooles_rep ()
{
    gerrep(_p).destructor();
    delete_macro(((chv_spooles_rep*)_p));
}
template<class T>
spooles_rep<T>::spooles_rep (const csr<T>& a)
{
    _p = new_macro(chv_spooles_rep);
    gerrep(_p).factorize (a.nrow(), a.ia().begin(), a.ja().begin(), a.a().begin());
}
template<class T>
void
spooles_rep<T>::factorize_ldlt()
{
    // already factorized: nothing to do
}
template<class T>
void
spooles_rep<T>::solve(const vec<T>& b, vec<T>& x) const
{
    if (x.size() == 0) return; // spooles core dump otherwise
    gerrep(_p).solve(b.begin(),x.begin(),true);
}
template<class T>
typename spooles_rep<T>::size_type
spooles_rep<T>::nnz () const
{
    fatal_macro("spooles_rep::nnz: not implemented");
    return 0;
}
template<class T>
typename spooles_rep<T>::size_type
spooles_rep<T>::nrow () const
{
    fatal_macro("spooles_rep::nrow: not implemented");
    return 0;
}
template<class T>
typename spooles_rep<T>::size_type
spooles_rep<T>::ncol () const
{
    fatal_macro("spooles_rep::ncol: not implemented");
    return 0;
}

// instanciation in library
template class ssk<double>;
template class spooles_rep<double>;
template ssk<double> ldlt (const csr<double>&);

#endif // _RHEOLEF_HAVE_SPOOLES
