/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - PADS ASCII netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t epads_ascii_net;

static int pads_ascii_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if ((rnd_strcasecmp(fmt, "pads_ascii") == 0) || (rnd_strcasecmp(fmt, "pads") == 0))
		return 100;
	return 0;
}

static void pads_ascii_assert_alnum_len(const char* const title,
	const char* const str, const int maxlen)
{
	const char* p = str;

	if(!memchr(str, 0, maxlen+1))
	{
		rnd_message(RND_MSG_ERROR, "pads_ascii: Broken output! %s is too long: %s\n",
			title, str);
	}

	while(*p)
	{
		const char ch = *p++;

		if('_'!=ch && '-'!=ch && !isalnum(ch))
		{
			rnd_message(RND_MSG_ERROR, "pads_ascii: Broken output! %s contains one or more non-alpanumeric characters: %s\n",
				title, str);
			return;
		}
	}
}

/* Export abstract components; exports refdes and footprint, but format */
/* does not support device/value/etc attribs */
static void pads_ascii_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);

		if(fp==NULL)
		{
			fp = "unknown_fp";
			rnd_message(RND_MSG_ERROR, "pads_ascii: no footprint is set for \"%s\"\n", refdes);
		}

		/* The reference designator may be up to 15 alphanumeric characters long. */
		pads_ascii_assert_alnum_len("Refdes", refdes, 15);

		/* The part type can be up to 40 alphanumeric characters long. */
		pads_ascii_assert_alnum_len("Footprint", fp, 40);

		fprintf(f, "%s  %s\r\n", refdes, fp);
	}
}

/* Export abstract nets; exports net's name and a list of refdes.pinnum pairs */
static void pads_ascii_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		int net_exported = 0; /* net had at least one connection so it was written */

		if (net->hdr.omit) continue;

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "pads_ascii: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "pads_ascii: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "pads_ascii: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					if(net_exported)
					{
						fputc(' ', f);
					}
					else
					{
						net_exported = 1;

						/* A signal may be up to 47 alphanumeric characters long. */
						if(!memchr(netname, 0, 48))
						{
							rnd_message(RND_MSG_ERROR, "pads_ascii: Broken output! Netname is too long: %s\n", netname);
						}

						fprintf(f, "*SIGNAL* %s\r\n", netname);
					}

					if(strchr(refdes, '.'))
					{
						rnd_message(RND_MSG_ERROR, "pads_ascii: Broken output! Refdes contains dot (.), and messes things up: %s\n", refdes);
					}

					fprintf(f, "%s.%s", refdes, my_num);
				}
			);
		}

		/* If the net got exported, close the opened connections line */
		if(net_exported)
		{
			fprintf(f, "\r\n");
		}
	}
}


/* Export netlist from the abstract model */
static int pads_ascii_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "!PADS-POWERPCB-V3.0-MILS!\r\n");

	fprintf(f, "*PART*\r\n");
	pads_ascii_export_comps(f, abs);
	fprintf(f, "*NET*\r\n");
	pads_ascii_export_nets(f, abs);

	fprintf(f, "*END*\r\n");
	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_pads_ascii(int ver_needed) { return 0; }

void pplg_uninit_export_pads_ascii(void)
{
	csch_plug_io_unregister(&epads_ascii_net);
	rnd_export_remove_opts_by_cookie(pads_ascii_cookie);
	rnd_hid_remove_hid(&pads_ascii_hid);
}

int pplg_init_export_pads_ascii(void)
{
	RND_API_CHK_VER;

	epads_ascii_net.name = "export to PADS ASCII";
	epads_ascii_net.export_prio = pads_ascii_export_prio;
	epads_ascii_net.export_project_abst = pads_ascii_export_project_abst;
	epads_ascii_net.ext_export_project = ".net";
	csch_plug_io_register(&epads_ascii_net);


	rnd_hid_nogui_init(&pads_ascii_hid);

	pads_ascii_hid.struct_size = sizeof(rnd_hid_t);
	pads_ascii_hid.name = "pads_ascii";
	pads_ascii_hid.description = "Exports project's PADS ASCII netlist";
	pads_ascii_hid.exporter = 1;

	pads_ascii_hid.get_export_options = pads_ascii_get_export_options;
	pads_ascii_hid.do_export = pads_ascii_do_export;
	pads_ascii_hid.parse_arguments = pads_ascii_parse_arguments;
	pads_ascii_hid.argument_array = pads_ascii_values;

	pads_ascii_hid.usage = pads_ascii_usage;

	rnd_hid_register_hid(&pads_ascii_hid);
	rnd_hid_load_defaults(&pads_ascii_hid, pads_ascii_options, NUM_OPTIONS);


	return 0;
}

