/* SPDX-License-Identifier: GPL-2.0 */
#ifndef __BLK_NULL_BLK_H
#define __BLK_NULL_BLK_H

#undef pr_fmt
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/blkdev.h>
#include <linux/slab.h>
#include <linux/blk-mq.h>
#include <linux/hrtimer.h>
#include <linux/configfs.h>
#include <linux/badblocks.h>
#include <linux/fault-inject.h>
#include <linux/spinlock.h>
#include <linux/mutex.h>

struct nullb_cmd {
	blk_status_t error;
	bool fake_timeout;
	struct nullb_queue *nq;
	struct hrtimer timer;
};

struct nullb_queue {
	struct nullb_device *dev;
	unsigned int requeue_selection;

	struct list_head poll_list;
	spinlock_t poll_lock;
};

struct nullb_zone {
	/*
	 * Zone lock to prevent concurrent modification of a zone write
	 * pointer position and condition: with memory backing, a write
	 * command execution may sleep on memory allocation. For this case,
	 * use mutex as the zone lock. Otherwise, use the spinlock for
	 * locking the zone.
	 */
	union {
		spinlock_t spinlock;
		struct mutex mutex;
	};
	enum blk_zone_type type;
	enum blk_zone_cond cond;
	sector_t start;
	sector_t wp;
	unsigned int len;
	unsigned int capacity;
};

struct nullb_device {
	struct nullb *nullb;
	struct config_group group;
#ifdef CONFIG_BLK_DEV_NULL_BLK_FAULT_INJECTION
	struct fault_config timeout_config;
	struct fault_config requeue_config;
	struct fault_config init_hctx_fault_config;
#endif
	struct radix_tree_root data; /* data stored in the disk */
	struct radix_tree_root cache; /* disk cache data */
	unsigned long flags; /* device flags */
	unsigned int curr_cache;
	struct badblocks badblocks;
	bool badblocks_once;
	bool badblocks_partial_io;

	unsigned int nr_zones;
	unsigned int nr_zones_imp_open;
	unsigned int nr_zones_exp_open;
	unsigned int nr_zones_closed;
	unsigned int imp_close_zone_no;
	struct nullb_zone *zones;
	sector_t zone_size_sects;
	bool need_zone_res_mgmt;
	spinlock_t zone_res_lock;

	unsigned long size; /* device size in MB */
	unsigned long completion_nsec; /* time in ns to complete a request */
	unsigned long cache_size; /* disk cache size in MB */
	unsigned long zone_size; /* zone size in MB if device is zoned */
	unsigned long zone_capacity; /* zone capacity in MB if device is zoned */
	unsigned int zone_nr_conv; /* number of conventional zones */
	unsigned int zone_max_open; /* max number of open zones */
	unsigned int zone_max_active; /* max number of active zones */
	unsigned int zone_append_max_sectors; /* Max sectors per zone append command */
	unsigned int submit_queues; /* number of submission queues */
	unsigned int prev_submit_queues; /* number of submission queues before change */
	unsigned int poll_queues; /* number of IOPOLL submission queues */
	unsigned int prev_poll_queues; /* number of IOPOLL submission queues before change */
	unsigned int home_node; /* home node for the device */
	unsigned int queue_mode; /* block interface */
	unsigned int blocksize; /* block size */
	unsigned int max_sectors; /* Max sectors per command */
	unsigned int irqmode; /* IRQ completion handler */
	unsigned int hw_queue_depth; /* queue depth */
	unsigned int index; /* index of the disk, only valid with a disk */
	unsigned int mbps; /* Bandwidth throttle cap (in MB/s) */
	bool blocking; /* blocking blk-mq device */
	bool use_per_node_hctx; /* use per-node allocation for hardware context */
	bool power; /* power on/off the device */
	bool memory_backed; /* if data is stored in memory */
	bool discard; /* if support discard */
	bool zoned; /* if device is zoned */
	bool zone_full; /* Initialize zones to be full */
	bool virt_boundary; /* virtual boundary on/off for the device */
	bool no_sched; /* no IO scheduler for the device */
	bool shared_tags; /* share tag set between devices for blk-mq */
	bool shared_tag_bitmap; /* use hostwide shared tags */
	bool fua; /* Support FUA */
	bool rotational; /* Fake rotational device */
};

struct nullb {
	struct nullb_device *dev;
	struct list_head list;
	unsigned int index;
	struct request_queue *q;
	struct gendisk *disk;
	struct blk_mq_tag_set *tag_set;
	struct blk_mq_tag_set __tag_set;
	atomic_long_t cur_bytes;
	struct hrtimer bw_timer;
	unsigned long cache_flush_pos;
	spinlock_t lock;

	struct nullb_queue *queues;
	char disk_name[DISK_NAME_LEN];
};

blk_status_t null_handle_discard(struct nullb_device *dev, sector_t sector,
				 sector_t nr_sectors);
blk_status_t null_process_cmd(struct nullb_cmd *cmd, enum req_op op,
			      sector_t sector, unsigned int nr_sectors);
blk_status_t null_handle_badblocks(struct nullb_cmd *cmd, sector_t sector,
				   unsigned int *nr_sectors);
blk_status_t null_handle_memory_backed(struct nullb_cmd *cmd, enum req_op op,
				       sector_t sector, sector_t nr_sectors);

#ifdef CONFIG_BLK_DEV_ZONED
int null_init_zoned_dev(struct nullb_device *dev, struct queue_limits *lim);
int null_register_zoned_dev(struct nullb *nullb);
void null_free_zoned_dev(struct nullb_device *dev);
int null_report_zones(struct gendisk *disk, sector_t sector,
		      unsigned int nr_zones,
		      struct blk_report_zones_args *args);
blk_status_t null_process_zoned_cmd(struct nullb_cmd *cmd, enum req_op op,
				    sector_t sector, sector_t nr_sectors);
size_t null_zone_valid_read_len(struct nullb *nullb,
				sector_t sector, unsigned int len);
ssize_t zone_cond_store(struct nullb_device *dev, const char *page,
			size_t count, enum blk_zone_cond cond);
#else
static inline int null_init_zoned_dev(struct nullb_device *dev,
		struct queue_limits *lim)
{
	pr_err("CONFIG_BLK_DEV_ZONED not enabled\n");
	return -EINVAL;
}
static inline int null_register_zoned_dev(struct nullb *nullb)
{
	return -ENODEV;
}
static inline void null_free_zoned_dev(struct nullb_device *dev) {}
static inline blk_status_t null_process_zoned_cmd(struct nullb_cmd *cmd,
			enum req_op op, sector_t sector, sector_t nr_sectors)
{
	return BLK_STS_NOTSUPP;
}
static inline size_t null_zone_valid_read_len(struct nullb *nullb,
					      sector_t sector,
					      unsigned int len)
{
	return len;
}
static inline ssize_t zone_cond_store(struct nullb_device *dev,
				      const char *page, size_t count,
				      enum blk_zone_cond cond)
{
	return -EOPNOTSUPP;
}
#define null_report_zones	NULL
#endif /* CONFIG_BLK_DEV_ZONED */
#endif /* __NULL_BLK_H */
