# -*- coding: UTF-8 -*-

# Phatch - Photo Batch Processor
# Copyright (C) 2007-2008 www.stani.be
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see http://www.gnu.org/licenses/
#
# Phatch recommends SPE (http://pythonide.stani.be) for editing python files.

# Embedded icon is designed by Igor Kekeljevic (http://www.admiror-ns.co.yu).

# Follows PEP8

# Always import this:
from core import models
from lib.reverse_translation import _t

#---PIL


def init():
    global Image, ImageChops, ImageDraw, imtools
    import Image
    import ImageChops
    import ImageDraw
    from lib import imtools


# Declare constants here

CROSS = _t('Cross')
ROUNDED = _t('Rounded')
SQUARE = _t('Square')

CORNERS = [ROUNDED, SQUARE, CROSS]
CORNER_ID = 'rounded_corner_r%d_f%d'
CROSS_POS = (CROSS, CROSS, CROSS, CROSS)
ROUNDED_POS = (ROUNDED, ROUNDED, ROUNDED, ROUNDED)
ROUNDED_RECTANGLE_ID = 'rounded_rectangle_r%d_f%d_s%s_p%s'


def round_image(image, cache={}, round_all=True, rounding_type=None,
        radius=100, opacity=255, pos=ROUNDED_POS, back_color='#FFFFFF'):
    if image.mode != 'RGBA':
        image = image.convert('RGBA')

    if round_all:
        pos = 4 * (rounding_type, )

    mask = create_rounded_rectangle(image.size, cache, radius, opacity, pos)

    imtools.paste(image, Image.new('RGB', image.size, back_color), (0, 0),
        ImageChops.invert(mask))
    image.putalpha(mask)
    return image


def create_rounded_rectangle(size=(600, 400), cache={}, radius=100,
        opacity=255, pos=ROUNDED_POS):
    #rounded_rectangle
    im_x, im_y = size
    rounded_rectangle_id = ROUNDED_RECTANGLE_ID % (radius, opacity, size, pos)
    if rounded_rectangle_id in cache:
        return cache[rounded_rectangle_id]
    else:
        #cross
        cross_id = ROUNDED_RECTANGLE_ID % (radius, opacity, size, CROSS_POS)
        if cross_id in cache:
            cross = cache[cross_id]
        else:
            cross = cache[cross_id] = Image.new('L', size, 0)
            draw = ImageDraw.Draw(cross)
            draw.rectangle((radius, 0, im_x - radius, im_y), fill=opacity)
            draw.rectangle((0, radius, im_x, im_y - radius), fill=opacity)
        if pos == CROSS_POS:
            return cross
        #corner
        corner_id = CORNER_ID % (radius, opacity)
        if corner_id in cache:
            corner = cache[corner_id]
        else:
            corner = cache[corner_id] = create_corner(radius, opacity)
        #rounded rectangle
        rectangle = Image.new('L', (radius, radius), 255)
        rounded_rectangle = cross.copy()
        for index, angle in enumerate(pos):
            if angle == CROSS:
                continue
            if angle == ROUNDED:
                element = corner
            else:
                element = rectangle
            if index % 2:
                x = im_x - radius
                element = element.transpose(Image.FLIP_LEFT_RIGHT)
            else:
                x = 0
            if index < 2:
                y = 0
            else:
                y = im_y - radius
                element = element.transpose(Image.FLIP_TOP_BOTTOM)
            imtools.paste(rounded_rectangle, element, (x, y))
        cache[rounded_rectangle_id] = rounded_rectangle
        return rounded_rectangle


def create_corner(radius=100, opacity=255, factor=2):
    corner = Image.new('L', (factor * radius, factor * radius), 0)
    draw = ImageDraw.Draw(corner)
    draw.pieslice((0, 0, 2 * factor * radius, 2 * factor * radius),
        180, 270, fill=opacity)
    corner = corner.resize((radius, radius), Image.ANTIALIAS)
    return corner

#---Phatch


class Action(models.Action):

    label = _t('Round')
    author = 'Stani'
    cache = True
    email = 'spe.stani.be@gmail.com'
    init = staticmethod(init)
    pil = staticmethod(round_image)
    version = '0.1'
    tags = [_t('default'), _t('filter')]
    __doc__ = _t('Rounded or crossed corners')

    def interface(self, fields):
        fields[_t('Radius')] = self.PixelField('2%',
            choices=self.SMALL_PIXELS)
        fields[_t('Same Method for All Corners')] = self.BooleanField(True)
        fields[_t('Method')] = self.ChoiceField(CORNERS[0], CORNERS)
        fields[_t('Top Left Corner')] = self.ChoiceField(CORNERS[0], CORNERS)
        fields[_t('Top Right Corner')] = self.ChoiceField(CORNERS[0], CORNERS)
        fields[_t('Bottom Left Corner')] = self.ChoiceField(
            CORNERS[0], CORNERS)
        fields[_t('Bottom Right Corner')] = self.ChoiceField(
            CORNERS[0], CORNERS)
        fields[_t('Background Color')] = self.ColorField('#FFFFFF')
        fields[_t('Opacity')] = self.SliderField(100, 1, 100)

    def get_relevant_field_labels(self):
        """If this method is present, Phatch will only show relevant
        fields.

        :returns: list of the field labels which are relevant
        :rtype: list of strings

        .. note::

            It is very important that the list of labels has EXACTLY
            the same order as defined in the interface method.
        """
        relevant = ['Radius', 'Same Method for All Corners']
        if self.get_field_string('Same Method for All Corners') \
                in ('yes', 'true'):
            relevant.append('Method')
        else:
            relevant.extend(['Top Left Corner', 'Top Right Corner',
                'Bottom Left Corner', 'Bottom Right Corner'])
        relevant.extend(['Background Color', 'Opacity'])
        return relevant

    def values(self, info):
        #get info (always get this)
        width, height = info['size']
        dpi = info['dpi']

        round_all = self.get_field('Same Method for All Corners', info)
        rounding_type = self.get_field('Method', info)

        average = (width + height / 2)
        radius = self.get_field_size('Radius', info, average, dpi)
        opacity = int((self.get_field('Opacity', info) / 100.0) * 255)
        pos = (
            self.get_field('Top Left Corner', info),
            self.get_field('Top Right Corner', info),
            self.get_field('Bottom Left Corner', info),
            self.get_field('Bottom Right Corner', info))

        return {
            'round_all': round_all,
            'rounding_type': rounding_type,
            'radius': radius,
            'opacity': opacity,
            'pos': pos,
            'back_color': self.get_field('Background Color', info)}

    icon = \
'x\xda\x01\x00\x0e\xff\xf1\x89PNG\r\n\x1a\n\x00\x00\x00\rIHDR\x00\x00\x000\
\x00\x00\x000\x08\x06\x00\x00\x00W\x02\xf9\x87\x00\x00\x00\x04sBIT\x08\x08\
\x08\x08|\x08d\x88\x00\x00\r\xb7IDATh\x81\xd5\x9a{p[\xd5\x9d\xc7?\xf7\xe8q\
\xadk\xc9\x8ec\xcb\x96\x9f\xb2e\x1b\xdb\xd8q\xb2\xc4\x93\x17\x03yB \xd0\xce\
\x06f\x81\x86vX:\xa5\x03\x85\xce\xf0\'\x9d\xed\xfe\xbf\x7f\xecc\xda\xce\xb6\
\x0c\xd3\x9d\x1d\x16\xda$` P\x08)I\xcc36\xa5\tq\x1c?\x82Md\xc7\x8e\x1f\xaa\
\xadH~I\x96t\x1f\xfb\xc7\x95d\xd9\xb1\xc3\xc3av\xf6\xcc\x9c\xb9\xf6=W\xf7|\
\xbf\xbf\xf3\xfd=\xce\x91$\xc30\xf8\xff\xdc\xac\xdf\xc9[w\xef\xb6\x12\x08<\
\x88\x10\x9b\x81l\x0c\xa3\x0bIz\x95\xee\xeek7{*\xe9\xa6\xaf@C\xc3\x06,\x96\
\xff\xc606/\x1b\t`\x18O\xd1\xdb\xfb\xc6\xcd\x9c\xee\xe6\x12hjz\x1a\xc3\xf87@\
\xbe\xc1S/03\xf3,##\xd1\x9b1\xe5\xcd!\xd0\xd4\xb4>\xaei\x7f\xb4\x0b\xb1\xff\
\xeb<\x1e\xd7\xf5A\xbb\xc5r?\xdd\xdd\xbdk\x9dZ\xac\xf5\x05\xa1[n\xb9+\xa6\
\xaa\x83_\x17<\x80]\x88*M\xd7\xbb&\xab\xab\x7f\xb1\xd6\xf9\xd7\xb4\x02\xfe\r\
\x1b\xfeP\xa5\xeb\x87\xa45\x00\x18\xc9\xc9\xb9P\xde\xd1\xb1\xe9\xdb~~MQ\xa8\
\xaf\xaajg\x9e\xcfG\xde\xfb\xef\x83\xaa.\x193jk\xf9\xdb\x86\r$,\x16\x8a._\
\xc6\xd6\xd5\xb5\xf4\x19\x9b\x8d\xe0\xde\xbd\x9c\xeb\xe9\xa9-_\x03\x865\xad\
\xc0\xbf\xd8l\xf7\xe4VT\xbc|\xcfC\x0f\xe5W\xb6\xb7#MM\x81\xd3\xc9\xc4\xbe}\
\x9c\xeb\xeb\xe3\xc2\x993,\x18\x06\r\xf5\xf5l\xdd\xb9\x93\xcaO?E\x8c\x8da\
\x14\x15\xe1\xdf\xba\x95\x13\x87\x0fO]\x1b\x1d\xfd\xc1?\'\x12\xa7\xfeO\x08H\
\x92d}r\xf7\xee\xbf/\xbbr\xe5\xa5\xbd{\xf6d5ee1h\xb3\xf1\xa7\xd6V\x06\xcb\
\xcb\x89z\xbd\x08\xab\x15[ @Qo/\xf7\xec\xdeMSq1}\x93\x93\xb4}\xf8\xa1\x11\
\xda\xb1\xe3\xd7\xff\xf1\xf2\xcb\xffd\x18\xc6\xfcwK\xa0\xb9\xd9G<>\xc9\xa5K\
\xb3\x99\xb7\x8f\x1c9\xf2\x93\xe9\xe9\xe9_\x0f\xf4\xf7;\xd4\xb66n\x99\x9b\
\xe3K`\xaa\xa5\x85\x9c\xfc|\xdcn7\x0e\x87\x83\xd9\xd9Y&\'\'\xb1\x9d=\x8bol\
\x0c\x7fA\x01\xb1\xad[\xa9\xf2\xf9\xa8\xa8\xa88VTT\xf4\x8f\xfb\xf6\xed\x9bN\
\xbf\xb8\xbe\xde\x85\x10\x7fGo\xefGk\'\xd0\xd8\xf8C\xe0w\xc0\x04\xf0\x10==\
\xe7\xdfz\xeb-\x97\x10\xe2w\x86a<\x1a\x8b\xc5\x98\x9c\x9cdhh\x88\xd0\xc8\x08\
\x92\xcbEqq1\xa5\xa5\xa5\xe4\xe4\xe4`\xb1XH$\x12\x84\xc3a\x86\x87\x87\t\x8e\
\x8fc\xcb\xce\xa6\xbc\xbc<\xfd\x8c\xd5j\x1d\x92$\xe9\x91\x03\x07\x0e\xfc\x85\
\x86\x86\xcd\x08q\x14\xf0\x01\xff\x8e\xc3\xf1\x0b\xce\x9eM|s\x02\x1b7f\x93H\
\xfc\'\x92\xf4X\xc6\xddX\xd8\xeb\xfd\xd5\'?\xfb\xd9\x03@\r\x80a\x18\xa8\xaa\
\xca\xf4\xf44\xc1`\x10\xbb\xddN~~>\xd9\xd9\xd9X,\x16R\xcfh\x9a\xc6\xcc\xcc\
\x0c\xa1P\x88\xac\xac,\xd6\xad[\x87\xc3\xe1@\x88t$W7\xbe\xf2\xca\xbb\xe5\xe7\
\xce\xed\x07\xec\x19s\xfe\x15\x8b\xe5\x11\xba\xba\xfc_\x9f@S\xd3F\xe0(\x86Q\
\xb7\xd2\x87\xc6\x9b\x9b\xe9|\xe0\x014\x87#}O\xd7uTUE\x92$l6[&0Rs\xe8\xbaN"\
\x91@\x08\x81\xc5bA\x92\x92\x018\x1a\xa5\xa5\xb5\x15Ow\xf7\xca\xc6\x84\x19\
\xe0\tzz^Y>p}"\xab\xab+\x00N\xae\x06^/)!RS\xc3\x9c\xdf\x8f$Ii0v\xbb\x1dEQP\
\x14\x05\x9b\xcd\x86\xc5bIw\xab\xd5\x8a\xc5b\xc1f\xb3\xe1p8\x90ey\t\x81X0\
\xc8\x98\xc7C\xbc\xa9i5\x029\xc0\x116l\xd8\xf5\x95\x04t\x8b\xe5_1\x0c\xf7\
\x8a\xaf\xc9\xca\xe2|E\x05o\x1e?\x8eTZ\x8a\x10\x02\xab\xd5\x9a\x06\x98y\xcd\
\xec\xcb\xc72\xc9\t!\xb0\x16\x15\xd118\xc8\xc9\xf9y\x8c\xd2\xd2\xd5HH\x9a\
\xa6\xbd\xb8\xb8l\xab\x10P\r\xe3\xc1\xd5\xde\x10\xf1zi\xbdx\x91\xa9\xfd\xfb\
\x89eX\xffF\x00\xbfj\xccb\xb1\x80\x10\xa8{\xf7\xd2\x06\\\xcd\xcd]mz,\x92TA}}\
\xed\xea\x04n\xbf\xdde\x97e\'V+\x88\xeb\xd5u\xa9\xb7\x97\x99\x96\x16\x8a\x8a\
\x8a\xc8\xc9\xc9\xc1n\xb7/\x01\xb5\xfc\xef\xd5\xc6\x96wY\x96q\xbb\xdd\xe4n\
\xd9\xc2\xd9\xbe\xbeU\t\x00\x8c\x1a\xc6\xed\x99\xff/)%~\xd2\xde\xce\xafv\xed\
";\x12\x01]\x07\xc30\xbb\xaa\x82\xaeS\x0bTVTPYY\x89\xcb\xe5J\xeb8\xd5\x81\
\xe5+\x0c,:\xb1a\x18\xd7\xf5\x94\x1f\xe5\xe7\xe7\xa3\xeb:\x8d\x92d\xce\xb9R\
\x93$:]\xae\xf5\x99"[b\xe6\xff\x02}\xa2\xa8H\'+\x0b\x1c\x0e\xb3ge\x81\xa2\
\x80\xa2\xe0R\x14n\x1f\x1b#777\xed\xa8_\xe5\x037\xb2~J\x82B\x08l6\x1b\xc5\
\xc1 \xb5\xba\xbe\xaa\xf5\xc3UU\x9c\x08\x87/K\x92\x94\xc6\xbd\xbc\x98s\xb4ML\
$J\xeb\xea\xe4\xac+WLK\xe8\xfa\x92\xbe\xfd\xdc9\xfa\xe6\xe7\x99\xfd\xe9O\x97\
X\xff\xabV 3\x94fZ?uU^{\x8dM\xa7O\xaf\xf8y\x00c\xfdz\xdaU\x95\x99\x96\x16\
\x17_~i\x05\xe2KV@\x92$\xc9\xe9tJ\x81M\x9b\xe4\x0f\x82A\xe6\xb6m[\\\x01\x87\
\xc3\\\x05\xa7\x93\x99}\xfb\x18\x8e\xc5n\xa8\xfb\xe5+p\xa3.\x84@\x08\xc1t$B\
\xe4\xae\xbb \x99\xfc2\xdbBS\x13g7n\xe4T,FAI\xc9]@\x9ae\xa6\x84\xac\xba\xae\
\xdbr\xab\xabi\x8b\xc58\xfc\xc6\x1bL\xec\xd8\x01n\xb7I\xc0\xe3a\xf4\x8e;\xf8\
ck+S\xd5\xd5\xe8\xba~\x9ddV\x03\xb9\xd2xf\x04\x13B0\xed\xf3\xf1\xea\xfb\xefs\
\xe5\xfe\xfb!/\xcfDd\xb31\xb5\x7f?\xc7C!\xfe\xf0\xc1\x07\x88\xdbn\xc3j\xb5\
\xaa)\x83\xa7%\x94\xd4\x94\xac\xaa\xea:Y\x96)\xde\xbb\x97\xee\xf3\xe7\x99\
\xf8\xedo9x\xe8\x105\r\r\xf4\x05\x02\xbc\xf5\xc2\x0bL\xef\xdbG}i)\x9a\xa6\
\xa5\x9d8\x95u\x97\xcbh\xb9\xf3\xea\xba\xbeD6zR\xef\x9a\xa6akjb\xfc\xda5\xfe\
\xe7\xf0a\xee\xbd\xf7^\x9ab1\xae\xba\xdd\xfc\xa9\xb5\x95\x9e\x8a\n\xecw\xdeI\
EI\t\x03\x03\x03S\x80\rH\x00F\xca\x07$@\x16B\xac\x9f\x98\x98\xa0\xb1\xb1\x11\
\x87\xc3\x81\xdf\xed\xe6\xa5w\xdf\xc5;?\xcfe\x97\x0b\xf5{\xdf\xc3[YIyy9\x8a\
\xa2\xa4\xad\'I\x12"\x91\xc0\x1e\n\x11\xf7xV\xd4\x7f:\xe2\xcc\xce\xa2Y\xad\
\xa8\xd6E\xf7\xb3Z\xad\xb8\\.J\x1a\x1a\xe8\x7f\xf0A\x8e\x9c>M\xa3\x10|\x19\
\x0c2\xbeu+yeex\xbd^\xa2\xd1(\xd1h\x14 \x0bX\x00\xf4L\x02V\x8b\xc5\xa2\x18\
\x86\x81\xdf\xef\xa7\xae\xae\x8e\xdc\xdc\\.\xe7\xe5\xf1\xe5\xc8\x08Y\xf9\xf9\
4\xf8|TWW\x93\x9f\x9f\x8f,\xcbi\xf0Ycc\x14\x1d>\x8cuz\x9a\xe9\x9d;\t\xef\xdf\
\x7f\x1dx\xc30Pzz(\xfe\xcdo\xd0d\x99\xe1g\x9f%ZR\x92&\xe1t:\xf1\xf9|\xc8\xb2\
\xcc\x80\xa2p\xee\x8b/\xd0\xb6l\xc1\xeb\xf1PVV\xc6\xcc\xcc\x0c\xd1h\x14M\xd3\
\xecI\xe5,J(E\xc20\x0c\x0b@$\x12\xa1\xb3\xb3\x93\x86\x86\x066o\xdeL\xb0\xb2\
\x12\xa7\xd3\x89\xc7\xe3\xc1\xe5r\xa5\x8b5!\x04\xeb?\xfa\x88\xdc\xcf>CRU\x90\
e\xd6\x9d9\x83\xe3\xf2eB\x0f<@\xbc\xa4\xc4\xb4\xfc\xec,\xeb\x8e\x1d#\xe7\xe4\
I\x88\xc7\xb1\xce\xceR\xfd\xcb_2\xfe\xa3\x1f\x11\xdc\xbd\x1b\xc30\xb0X,dgg\
\xe3\xf5zQ\x14\x85\xf1\xc2B\xacV+\x8a\xa20::J<\x1eO\xc9Od\xfa\xee\x920\xaai\
\x9aa\x18\x86.I\x920\x0c\x83\xde\xde^JJJhjjBQ\x94t\x11&\x84\xc0\xba\xb0\x80\
\xfb\xd5WQ\xe6\xe6\xccHe\xb3\x81\xa6\x81\xaa"_\xbb\x86\xe7\xf9\xe7\x89\xbb\
\xddh\x8aBVo/\xd2\xc2\x82\x19\x0c\xacVH$\x10\xf18\xa5/\xbdD\xf6\xc5\x8b\x0c\
\xff\xf8\xc7\xe0p\xa4\xb3raa!.\x97\x8b\x89\x89\t\x86\x86\x86\xc8\xac\x985M[\
\xb2\xf9^\x92\xc8b\xb1\x986???\x95yoll\x8c\xf6\xf6v\xa2\xd1\xe8\x92H23>N(/\
\x0f\xa3\xacl1\xe9\xa5\xc2m\xb2\xdb\xc3a\x1c\xc3\xc3H\xb2\xbc\xe4>\x8a\x02\
\xd9\xd9\xe0\xf1\x10)/g\xaa\xa3\xe3\xba\x9c288\xc8\xe0\xe0\xe0\x12\xf0\x00\
\x81@`d5\x02:\xa0MLL|\xc1\xb2\x16\x8dFikk\xc3\xef\xf7\xa7\xc3_vY\x19\xa7O\
\x9e\xe4/\xfd\xfd$jk\x17A-\x07\xb9\xd2}E!\xd1\xd0@ws3\xc7^\x7f\x1dKM\x8d\xa9\
aIbvv\x96\x8e\x8e\x0e&&&\x96\xc3 \x1e\x8f\xcfuvv\x0e$\xb1.!`\x00\x1a\x10{\
\xf3\xcd7\xdf\x8dD"s\xa9\x07R\x16\x11Bp\xe1\xc2\x05:::H$\x12`\xb7S\xf4\xf3\
\x9f\xf3\xe1\xe4$G\x9e\x7f\x9e\xd9\xaa*3~gg/\xf6\x14\xf8\xcc\x95q:\x99\xdb\
\xb6\x8d\xb6@\x80#\xc7\x8f3\xff\xc8#De\x19]\xd7\x19\x1a\x1a\xe2\xcc\x993D"\
\x91\xeb\xc0\x03\xb4\xb7\xb7\xbf\x16\n\x85f\x92XuH\xee\xc8\x92y\xc0\tT\x01-\
\x15\x15\x15w\x1c<x\xf0\xfbyyyy+eN\xa7\xd3\xc9\xb6m\xdb\xd0u\x9d\x9e\x9e\x1e\
.wt\x90u\xfc8\xff\xf0\xf4\xd3x\xd7\xafG\n\x06\xcd\x02PU!\x1e\x87D\xc2\xec\
\xb2L\xc0\xed\xe6\xd8\xef\x7f\xcf\x17ee8n\xbb\x8d\x9a\x9a\x1a*++\x19\x18\x18\
`|||E\xe0\x9a\xa6i\xed\xed\xed\xef\x9f>}\xfaU\xe0,p\x19\x985\x0cCO\x11\x900\
\x0fd=\xc0\xad@\xb3,\xcb\xd5\x07\x0f\x1e\xbcs\xc3\x86\r\xb7\xa46.\x99\xc5\
\x97\x10\x82\xc6\xc6F\x8a\x8b\x8b\xf1\xfb\xfd\xf4uu1\xfb\xca+\xeclnf\xc7\x81\
\x03\xd8\'\'aa!\r^-(\xa0od\x84\xb7\x8e\x1c\xe1\xda\xae]\x14\xd4\xd4P[[\x8b\
\xa2(tuu\x11\x8b\xc5V\x04\x1f\x0e\x87\xc3\xaf\xbf\xfe\xfa{\xc3\xc3\xc3]@\x17\
\xd0\x03\x8c\x011\xc30\x8c\xf4\x9eX\x92$\x0b\xe0\x02\xca\x81z\xa0\x16(\xdd\
\xb6m[\xe3\x81\x03\x07\xb6+\x8ab_\xa9\x86q\xbb\xdd477\x13\x0c\x06\xe9\xee\
\xee\xe6\xea\x9f\xffL\xc1\xc5\x8b\x1cx\xea)\xf2\n\x0b\xcd2!\x14\xa2\xfd\xf0a\
:\xa3Q\xd8\xb5\x8b\xb2\x8a\njkk\t\x85B\xf4\xf6\xf6\xaeZ\xc0\xf5\xf5\xf5\xf5\
\x1f;v\xec\xc3X,6\x0c\x0c\x00\x97\x80a`\xc60\x0c--\xa1$\x01\t3E\xe7\x00%\x98\
\xc7\x1a>\xa0\xcc\xe3\xf1\x94?\xf6\xd8c\xbbKJJ\xf2\x97\x97\xc1\xa9Z\xa7\xa5\
\xa5\x05\x9b\xcdFww7_|\xfe9\xd1\x93\'\x11\xa3\xa3\x10\x8f\x93(,$\xb6i\x13\
\xb9uu\xf8|>JKK\xe9\xed\xed%\x10\x08\xac\x08<\x1e\x8f\xc7O\x9d:u\xe6\xb3\xcf\
>\xbb\x00\x8c\x02\xfed\x1f\x05\xa6\x81\x84\x91\x04\xbe\xe4T"\x83\x84\x13p\
\x03\x15\x98~\xe1\x15B\x14=\xfa\xe8\xa3;\xb6l\xd9Ro\xb3\xd9\xa4\x94\xa4R\xd9\
X\x92$|>\x1f^\xaf\x17\xbf\xdf\xcf\xa5K\x97\x98\x9a\x9a"\x91H`\xb5Z)((\xa0\
\xba\xba\x1a!\x04\xe7\xcf\x9fgaa\x01u\xd9y*\xc0\xe4\xe4\xe4Tkk\xeb\xa9@ 0\
\x08\x8c$\x81\x0f\x03\x7f\x03f3\xc1_G \x83\x84\x15\xb3\xde\xc8\x03J\x93$\xaa\
\x80\xe2\xed\xdb\xb7\xdf\xfa\xf0\xc3\x0fow\xb9\\Y\x99r\x02\xb3tp:\x9d466\xa2\
\xeb:\xe1p\x98\x85\x85\x05dYF\x96e\xae\\\xb9\xc2\xc0\xc0\x00\x9a\xa6\xa1i\
\x9a\x19\xcd\x92\xcd0\x0c\xa3\xb3\xb3\xb3\xfb\xed\xb7\xdf\xfeD\xd3\xb41`\x08\
\x18LZ\xfd\x1a\x10\x05Tc\x19\xe0U\x0f\xb6\x92>a\xc7\x94\x94\x07\xf0&I\x94\
\x17\x14\x14\x94=\xf3\xcc3{|>\x9f;%\xa3\xe4g\xd04\r \xbdo\x96$\x89\xe9\xe9i\
\x86\x86\x86\x98\x9b\x9bCUU4MK_u]\'\x1a\x8dF\xdfy\xe7\x9d\x8f\xba\xbb\xbb{\
\x80\xabI\xe0C\x98\xa7\x813\x98\x0e\xab\xad\x88\xf3FG\x8b\x19\x92\xca\x06\n0\
\x1d\xbc*I\xa6\xe8\xf1\xc7\x1f\xdf\xb1g\xcf\x9e[\xedv\xbb\x10B\xa4KfUU\xd1u=\
\rt9\xe8\xcc\xeb\xf0\xf0\xf0Xkk\xeb\xe9p8|\x05S*~L\xe9L\x01\xf3\x98\x92Yu\
\x9f\xf9\x95g\xa3I\x12\x16\x16%U\xcc\xa2\xa4JZZZ\xea\x9e|\xf2\xc9;\xf2\xf2\
\xf2\x1c\xc0\x8a\xa0S\'r)\xd9\xe8\xbaN<\x1e\xd7?\xf9\xe4\x93\xcfO\x9c8\xf1)f\
X\x1cJZ~\x0c\x08a\x96\xcb\xd7I\xe6\x1b\x13\xc8 "X\x94T!\x19\x92\xca\xc9\xc9)\
y\xee\xb9\xe7\xf6\xd5\xd7\xd7{R W\xb3\xb8\xaa\xaa\x84\xc3\xe1\xf9\xa3G\x8f\
\xb6\xf5\xf7\xf7_\xc2\x94L\xcaQ\'0\x1d5\xbe\x9ad\xbe5\x81$\x89\x94\xa4\x14 \
\x1f(K\x92\xa8\x04\x8a\x0e\x1d:\xb4\xe5\xbe\xfb\xee\xdb(\x84\xb0d\xaeB&\xa9\
\xfe\xfe\xfe\x91\x17_|\xf1\xe4\xdc\xdc\xdc\x08\x8b\x92\xb9\n\x04Y\x94\xcc\
\xd7\x06\xf5\x8d\xbf\xe0H\x92\x10\x98\x92\xca\xc5\xcc\x19\x95I"\xa5\xf5\xf5\
\xf5UO<\xf1\xc4\x9d\x85\x85\x85\xebRrQU\x95H$\x92x\xef\xbd\xf7\xfez\xe2\xc4\
\x89\xb3\x98\x91%\xe5\xa8\xe3@\x18S2\xda7\x01\xff\xad\x08d\x10\x11\x98\xab\
\xe1\xc2\xcc\x19iI\t!\xdcw\xdf}w\xa3\xcf\xe7+\xb4\xdb\xed\xb6\xabW\xaf\x06?\
\xfe\xf8\xe3\xbe@ 0\x8c\xe9\xa0\x83\xc0\x15`\x92E\xc9\xac~ \xf4]\x10H\x92H\
\xe5\x0c\x07\xb0\x1e3gTbF+7\xa6\xd4\x00"\x98\x12\x19\xc6\xb4z*\xb6G\xf8\x1a\
\x8ez\xa3\xb6\xa6o)\r\xc30$IR1\xb5\xabb\xca`6\t\xb6\x103\xa3K\xc0\x1cf&\x1dK\
^\xa7\x81\x18\xa0\xaf\x05<\xdc\xc4\x9f\x1adH\xca\x81)\xab\x1cL?\x81Eb3\x98\
\x19\xf5\x86\xb1\xfd\x1b\xcd{3\x7f+\x91\xe1\xe0VL2\xa9\r\x93\x8ey\x8e\xa3r\
\x13\xac\x9e\xd9\xfe\x17Z)e=!o|\x97\x00\x00\x00\x00IEND\xaeB`\x82/\x91\xeeH'
