/*	Theater

PIRL CVS ID: Theater.java,v 1.46 2012/04/16 06:04:12 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Conductor.Maestro;

import	PIRL.Conductor.Processing_Changes;
import	PIRL.Messenger.*;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Configuration.Configuration_Exception;
import	PIRL.Utilities.Host;

import	java.io.IOException;
import	java.util.Hashtable;
import	java.util.Vector;
import	java.util.Iterator;


/**	A <i>Theater</i> provides basic Messenger connection and Message
	sending services with a Stage_Manager plus the Theater protocol
	facilities.
<p>
	This class is designed as a base class for the Remote_Theater and
	Local_Theater classes that implement a network extended Management
	interface via Messenger communication with a Stage_Manager class -
	its clients use the Remote_Theater - that conveys Messages via a
	Messenger to and from a Conductor instance that use the
	Local_Theater. These subclasses provide a Message based protocol for
	transmitting all the Management information between a remote client
	application and local Conductor application. The common Message
	parameter names used by the protocol are defined here along with
	Message {@link Message#Action() action name} to {@link
	#Action_Code(String) action code} mapping for efficient Message
	{@link
	Message_Delivered_Listener#Message_Delivered(Message_Delivered_Event)
	delivery} handler switching.
<p>
	This class can also be used as the primary interface for clients that
	will be working with the Stage_Manager directly rather than
	indirectly through the Management interface.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.46
	@see	Remote_Theater
	@see	Local_Theater
	@see	Stage_Manager
*/
public class Theater
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Conductor.Maestro.Theater (1.46 2012/04/16 06:04:12)";


public static final String
	THEATER_NAME						= "Theater";

public static final String

	ACTION_PARAMETER_NAME				= Message.ACTION_PARAMETER_NAME,

	//	Basic actions:

	IDENTIFY_ACTION						= Message.IDENTIFY_ACTION,
		KEY_PARAMETER_NAME					= Stage_Manager.KEY_PARAMETER_NAME,
	IDENTITY_ACTION						= Message.IDENTITY_ACTION,
	NACK_ACTION							= Message.NACK_ACTION,
	UNDELIVERABLE_ACTION				= Messenger.UNDELIVERABLE_ACTION,
		ORIGINAL_MESSAGE_PARAMETER_NAME		= Messenger.ORIGINAL_MESSAGE_PARAMETER_NAME,
	DONE_ACTION							= Message.DONE_ACTION,

	//	Common parameters:

	NAME_PARAMETER_NAME					= Message.NAME_PARAMETER_NAME,
	VALUE_PARAMETER_NAME				= "Value",
	CLASS_ID_PARAMETER_NAME				= Message.CLASS_ID_PARAMETER_NAME,
	EXPLANATION_PARAMETER_NAME			= Messenger.EXPLANATION_PARAMETER_NAME,
	EXCEPTION_PARAMETER_NAME			= Messenger.EXCEPTION_PARAMETER_NAME,
	THEATER_LOCATION_PARAMETER_NAME		= "Theater_Location",

	//	Stage_Manager actions:

	START_MESSENGER_REPORTING_ACTION	= Stage_Manager.START_MESSENGER_REPORTING_ACTION,
	STOP_MESSENGER_REPORTING_ACTION		= Stage_Manager.STOP_MESSENGER_REPORTING_ACTION,
	MESSENGERS_REPORT_ACTION			= Stage_Manager.MESSENGERS_REPORT_ACTION,
	START_CONDUCTOR_ACTION				= Stage_Manager.START_CONDUCTORS_ACTION,
	STATUS_REPORT_ACTION				= Stage_Manager.STATUS_REPORT_ACTION,
		MESSENGER_STATUS_PARAMETER_NAME
			= Stage_Manager.MESSENGER_STATUS_PARAMETER_NAME,
			FORWARDING_MESSENGERS_PARAMETER_NAME
				= Messenger.FORWARDING_MESSENGERS_PARAMETER_NAME,
			MESSAGES_SENT_PARAMETER_NAME
				= Stage_Manager.MESSAGES_SENT_PARAMETER_NAME,
			MESSAGE_BYTES_SENT_PARAMETER_NAME
				= Stage_Manager.MESSAGE_BYTES_SENT_PARAMETER_NAME,
			MESSAGES_SENT_DROPPED_PARAMETER_NAME
				= Stage_Manager.MESSAGES_SENT_DROPPED_PARAMETER_NAME,
			MESSAGES_RECEIVED_PARAMETER_NAME
				= Stage_Manager.MESSAGES_RECEIVED_PARAMETER_NAME,
			MESSAGE_BYTES_RECEIVED_PARAMETER_NAME
				= Stage_Manager.MESSAGE_BYTES_RECEIVED_PARAMETER_NAME,
			MESSAGES_RECEIVED_DROPPED_PARAMETER_NAME
				= Stage_Manager.MESSAGES_RECEIVED_DROPPED_PARAMETER_NAME,
		MEMORY_STATUS_PARAMETER_NAME
			= Stage_Manager.MEMORY_STATUS_PARAMETER_NAME,
			MEMORY_AVAILABLE_PARAMETER_NAME
				= Stage_Manager.MEMORY_AVAILABLE_PARAMETER_NAME,
			MEMORY_ALLOCATED_PARAMETER_NAME
				= Stage_Manager.MEMORY_ALLOCATED_PARAMETER_NAME,
			MEMORY_FREE_PARAMETER_NAME
				= Stage_Manager.MEMORY_FREE_PARAMETER_NAME,

	//	Management actions:

	CONDUCTOR_CONNECT_ACTION			= Stage_Manager.LINK_MESSENGER_ACTION,
	CONDUCTOR_DISCONNECT_ACTION			= Stage_Manager.UNLINK_MESSENGER_ACTION,
		ADDRESS_PARAMETER_NAME
			= Messenger.ADDRESS_PARAMETER_NAME,
		ROUTE_TO_PARAMETER_NAME
			= Stage_Manager.ROUTE_TO_PARAMETER_NAME,
	START_ACTION						= "Start",
	PROCESSING_STATE_ACTION				= "Processing_State",
	STOP_ACTION							= "Stop",
	QUIT_ACTION							= "Quit",

	CONFIGURATION_ACTION				= "Config_Report",
		CONFIGURATION_PARAMETER_NAME
			= "Configuration",

	SOURCES_ACTION						= "Sources",
	PROCEDURES_ACTION					= "Procedures",
		TABLE_PARAMETER_NAME
			= "Table",

	GET_POLL_INTERVAL_ACTION			= "Get_Poll_Interval",
	SET_POLL_INTERVAL_ACTION			= "Set_Poll_Interval",

	GET_RESOLVER_DEFAULT_VALUE_ACTION	= "Get_Resolver_Default_Value",
	SET_RESOLVER_DEFAULT_VALUE_ACTION	= "Set_Resolver_Default_Value",

	GET_STOP_ON_FAILURE_ACTION			= "Get_Stop_on_Failure",
	SET_STOP_ON_FAILURE_ACTION			= "Set_Stop_on_Failure",
	SEQUENTIAL_FAILURES_ACTION			= "Sequential_Failures",
	RESET_SEQUENTIAL_FAILURES_ACTION	= "Reset_Sequential_Failures",
	PROCESSING_EXCEPTION_ACTION			= "Processing_Exception",

	CONDUCTOR_STATE_ACTION				= "Conductor_State",
		SOURCE_RECORD_PARAMETER_NAME
			= "Source_Record",
		PROCEDURE_RECORD_PARAMETER_NAME
			= "Procedure_Record",
		SOURCES_REFRESHED_PARAMETER_NAME
			= "Sources_Refreshed",
		PROCEDURES_CHANGED_PARAMETER_NAME
			= "Procedures_Changed",
	ADD_PROCESSING_LISTENER_ACTION		= "Add_Processing_Listener",
	REMOVE_PROCESSING_LISTENER_ACTION	= "Remove_Processing_Listener",
	PROCESSING_CHANGES_ACTION			= "Processing_Changes",

	LOG_WRITER_ACTION					= Messenger_Styled_Writer.WRITE_ACTION,
		LOG_WRITTEN_PARAMETER_NAME
			= Messenger_Styled_Writer.WRITTEN_PARAMETER_NAME,
		LOG_STYLE_PARAMETER_GROUP
			= Messenger_Styled_Writer.STYLE_PARAMETER_GROUP,
	ADD_LOG_WRITER_ACTION				= "Add_Log_Writer",
	REMOVE_LOG_WRITER_ACTION			= "Remove_Log_Writer",
	ENABLE_LOG_WRITER_ACTION			= "Enable_Log_Writer";

public static final int

	//	Management Protocol actions:

	MINIMUM_PROTOCOL_CODE				= 100,

	IDENTIFY_CODE						= 100,
	CONDUCTOR_CONNECT_CODE				= 101,
	IDENTITY_CODE						= 102,
	START_CODE							= 103,
	PROCESSING_STATE_CODE				= 104,
	STOP_CODE							= 105,
	QUIT_CODE							= 106,
	CONFIGURATION_CODE					= 107,
	SOURCES_CODE						= 108,
	PROCEDURES_CODE						= 109,
	GET_POLL_INTERVAL_CODE				= 110,
	SET_POLL_INTERVAL_CODE				= 111,
	GET_RESOLVER_DEFAULT_VALUE_CODE		= 112,
	SET_RESOLVER_DEFAULT_VALUE_CODE		= 113,
	GET_STOP_ON_FAILURE_CODE			= 114,
	SET_STOP_ON_FAILURE_CODE			= 115,
	SEQUENTIAL_FAILURES_CODE			= 116,
	RESET_SEQUENTIAL_FAILURES_CODE		= 117,
	PROCESSING_EXCEPTION_CODE			= 118,
	CONDUCTOR_STATE_CODE				= 119,
	ADD_PROCESSING_LISTENER_CODE		= 120,
	REMOVE_PROCESSING_LISTENER_CODE		= 121,
	PROCESSING_CHANGES_CODE				= 122,
	LOG_WRITER_CODE						= 123,
	ADD_LOG_WRITER_CODE					= 124,
	REMOVE_LOG_WRITER_CODE				= 125,
	ENABLE_LOG_WRITER_CODE				= 126,
	CONDUCTOR_DISCONNECT_CODE			= 127,

	//	Non-Management protocol actions:
	
	DONE_CODE							= 0,
	NACK_CODE							= 1,
	UNDELIVERABLE_CODE					= 2,
	START_MESSENGER_REPORTING_CODE		= 3,
	STOP_MESSENGER_REPORTING_CODE		= 4,
	MESSENGERS_REPORT_CODE				= 5,
	START_CONDUCTOR_CODE				= 6,
	STATUS_REPORT_CODE					= 7;

private static Hashtable<String, Integer>
	Action_Codes						= new Hashtable<String, Integer> ();
static
	{
	Action_Codes.put (IDENTIFY_ACTION,
		new Integer  (IDENTIFY_CODE));
	Action_Codes.put (IDENTITY_ACTION,
		new Integer  (IDENTITY_CODE));
	Action_Codes.put (CONDUCTOR_CONNECT_ACTION,
		new Integer  (CONDUCTOR_CONNECT_CODE));
	Action_Codes.put (START_ACTION,
		new Integer  (START_CODE));
	Action_Codes.put (PROCESSING_STATE_ACTION,
		new Integer  (PROCESSING_STATE_CODE));
	Action_Codes.put (STOP_ACTION,
		new Integer  (STOP_CODE));
	Action_Codes.put (QUIT_ACTION,
		new Integer  (QUIT_CODE));
	Action_Codes.put (CONFIGURATION_ACTION,
		new Integer  (CONFIGURATION_CODE));
	Action_Codes.put (SOURCES_ACTION,
		new Integer  (SOURCES_CODE));
	Action_Codes.put (PROCEDURES_ACTION,
		new Integer  (PROCEDURES_CODE));
	Action_Codes.put (GET_POLL_INTERVAL_ACTION,
		new Integer  (GET_POLL_INTERVAL_CODE));
	Action_Codes.put (SET_POLL_INTERVAL_ACTION,
		new Integer  (SET_POLL_INTERVAL_CODE));
	Action_Codes.put (GET_RESOLVER_DEFAULT_VALUE_ACTION,
		new Integer  (GET_RESOLVER_DEFAULT_VALUE_CODE));
	Action_Codes.put (SET_RESOLVER_DEFAULT_VALUE_ACTION,
		new Integer  (SET_RESOLVER_DEFAULT_VALUE_CODE));
	Action_Codes.put (GET_STOP_ON_FAILURE_ACTION,
		new Integer  (GET_STOP_ON_FAILURE_CODE));
	Action_Codes.put (SET_STOP_ON_FAILURE_ACTION,
		new Integer  (SET_STOP_ON_FAILURE_CODE));
	Action_Codes.put (SEQUENTIAL_FAILURES_ACTION,
		new Integer  (SEQUENTIAL_FAILURES_CODE));
	Action_Codes.put (RESET_SEQUENTIAL_FAILURES_ACTION,
		new Integer  (RESET_SEQUENTIAL_FAILURES_CODE));
	Action_Codes.put (PROCESSING_EXCEPTION_ACTION,
		new Integer  (PROCESSING_EXCEPTION_CODE));
	Action_Codes.put (CONDUCTOR_STATE_ACTION,
		new Integer  (CONDUCTOR_STATE_CODE));
	Action_Codes.put (ADD_PROCESSING_LISTENER_ACTION,
		new Integer  (ADD_PROCESSING_LISTENER_CODE));
	Action_Codes.put (REMOVE_PROCESSING_LISTENER_ACTION,
		new Integer  (REMOVE_PROCESSING_LISTENER_CODE));
	Action_Codes.put (PROCESSING_CHANGES_ACTION,
		new Integer  (PROCESSING_CHANGES_CODE));
	Action_Codes.put (LOG_WRITER_ACTION,
		new Integer  (LOG_WRITER_CODE));
	Action_Codes.put (ADD_LOG_WRITER_ACTION,
		new Integer  (ADD_LOG_WRITER_CODE));
	Action_Codes.put (REMOVE_LOG_WRITER_ACTION,
		new Integer  (REMOVE_LOG_WRITER_CODE));
	Action_Codes.put (ENABLE_LOG_WRITER_ACTION,
		new Integer  (ENABLE_LOG_WRITER_CODE));
	Action_Codes.put (CONDUCTOR_DISCONNECT_ACTION,
		new Integer  (CONDUCTOR_DISCONNECT_CODE));

	Action_Codes.put (DONE_ACTION,
		new Integer  (DONE_CODE));
	Action_Codes.put (NACK_ACTION,
		new Integer  (NACK_CODE));
	Action_Codes.put (UNDELIVERABLE_ACTION,
		new Integer  (UNDELIVERABLE_CODE));
	Action_Codes.put (START_MESSENGER_REPORTING_ACTION,
		new Integer  (START_MESSENGER_REPORTING_CODE));
	Action_Codes.put (STOP_MESSENGER_REPORTING_ACTION,
		new Integer  (STOP_MESSENGER_REPORTING_CODE));
	Action_Codes.put (MESSENGERS_REPORT_ACTION,
		new Integer  (MESSENGERS_REPORT_CODE));
	Action_Codes.put (START_CONDUCTOR_ACTION,
		new Integer  (START_CONDUCTOR_CODE));
	Action_Codes.put (STATUS_REPORT_ACTION,
		new Integer  (STATUS_REPORT_CODE));
	}

private static final Message
	THEATER_IDENTITY
		= Message
		.Identity (THEATER_NAME)
		.Set (CLASS_ID_PARAMETER_NAME, ID);

private Messenger
	The_Messenger				= null;
private Message
	Stage_Manager_Identity		= null;
private boolean
	Opened						= false;


/**	The port number delimiter character.
<p>
	The delimiter, if present, occurs between the {@link #Host(String)
	hostname} and {@link #Port(String) port number} portions of a
	Theater {@link #Location(String) location} or {@link #Name(String)
	name}.
<p>
	<b>N.B.</b>: The delimiter and trailing port number portion of a
	Theater location or name may not be present. In this case the port
	number is the {@link #Default_Port() default port}.
*/
public static final char
	PORT_DELIMITER_CHAR		= ':';

/**	The default communications port number.
<p>
	The value is initialized to the {@link Stage_Manager#DEFAULT_PORT}.
*/
public static int
	Default_Port				= Stage_Manager.DEFAULT_PORT;

/**	The default maximum amount of time, in seconds, a Theater will wait
	for a Message to be received.
<p>
	@see	#Default_Receive_Timeout(int)
*/
public static final int
	DEFAULT_RECEIVE_TIMEOUT		= 10;

/**	The minimum amount of time, in seconds, a Theater will wait for a
	Message to be received.
<p>
	@see	#DEFAULT_RECEIVE_TIMEOUT
*/
public static final int
	MINIMUM_RECEIVE_TIMEOUT		= 5;
private static int
	Default_Receive_Timeout		= DEFAULT_RECEIVE_TIMEOUT;
private int
	Receive_Timeout				= Default_Receive_Timeout;

public static final String
	NL							= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF					= 0,
	DEBUG_CONSTRUCTORS			= 1 << 1,
	DEBUG_CONNECTION			= 1 << 2,
	DEBUG_MESSAGES				= 1 << 3,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct an unopened Theater.
*/
public Theater ()
{}

/**	Construct and open a Theater.
<p>
	@param	host	The name or IP address of the host system where the
		Stage_Manager is expected to be running. If null, "localhost"
		will be used.
	@param	port	The port number on the named host to use for
		establishing a communication channel to the Stage_Manager. If
		less than or equal to zero the {@link Stage_Manager#DEFAULT_PORT}
		will be used.
	@param	identity	An {@link Message#Identity(String) Identity}
		Message. <b>N.B.</b>: A {@link #KEY_PARAMETER_NAME} parameter
		may be required. If null an Identity using the {@link
		#THEATER_NAME} and no key parameter will be used.
	@throws	IOException	If a connection could not be established to
		the Stage_Manager. This will be a Theater_Protocol_Exception
		if there was a problem parsing any of the protocol messages.
	@see	#Open(String, int, Message)
*/
public Theater
	(
	String	host,
	int		port,
	Message	identity
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Theater");
Open (host, port, identity);
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("<<< Theater");
}

/**	Construct and open a Theater connected to a Stage_Manager on the
	local host at the default port.
<p>
	@param	identity	An {@link Message#Identity(String) Identity}
		Message. <b>N.B.</b>: A {@link #KEY_PARAMETER_NAME} parameter
		may be required. If null an Identity using the {@link
		#THEATER_NAME} and no key parameter will be used.
	@throws	IOException	If a connection could not be established to
		the Stage_Manager. This will be a Theater_Protocol_Exception
		if there was a problem parsing any of the protocol messages.
	@see	#Open(String, int, Message)
*/
public Theater
	(
	Message	identity
	)
	throws IOException
{this (null, 0, identity);}

/*==============================================================================
	Accessors
*/
/**	Get the Messenger used for communication.
<p>
	@return	A Messenger. This will be null if the communication channel
		has not yet been {@link #Open(String, int, Message) opened}.
*/
public Messenger Messenger ()
{return The_Messenger;}

/**	Get the default communications port number.
<p>
	@return	The default communications port number.
	@see	#Default_Port(int)
*/
public static int Default_Port ()
{return Default_Port;}

/**	Set the default communications port number.
<p>
	This port number will be used when an attempt is made to {@link
	#Open(String, int, Message) open} the Theater using a port number
	that is not positive.
<p>
	@param	port	The default communications port number. If not
		positive the {@link Stage_Manager#DEFAULT_PORT} is used.
*/
public static void Default_Port
	(
	int		port
	)
{
if (port <= 0)
	port = Stage_Manager.DEFAULT_PORT;
Default_Port = port;
}

/**	Get the default maximum amount of time, in seconds, a Theater will
	wait for a Message to be received.
<p>
	@return	The time, in seconds, when a timeout will occur while waiting
		for a Message to be received.
	@see	#Receive_Timeout()
*/
public static int Default_Receive_Timeout ()
{return Default_Receive_Timeout;}

/**	Set the default maximum amount of time, in seconds, a Theater will
	wait for a Message to be received.
<p>
	<b>N.B.</b>: This value is only effective when a new Theater is
	constructed.
<p>
	@param timeout	The time, in seconds, when a timeout will occur while
		waiting for a Message to be received. The value will be limited
		to be greater than or equal to the {@link
		#MINIMUM_RECEIVE_TIMEOUT}.
	@see	#Receive_Timeout()
*/
public static void Default_Receive_Timeout
	(
	int		timeout
	)
{
if (timeout < MINIMUM_RECEIVE_TIMEOUT)
	timeout = MINIMUM_RECEIVE_TIMEOUT;
Default_Receive_Timeout = timeout;
}

/**	Get the maximum amount of time, in seconds, a Theater will wait for a
	Message to be received.
<p>
	The value is initialized when a Theater is constructed to the
	{@link #Default_Receive_Timeout() default receive timeout}.
<p>
	@return	The time, in seconds, when a timeout will occur while waiting
		for a Message to be received.
	@see	#Receive_Timeout(int)
*/
public int Receive_Timeout ()
{return Receive_Timeout;}


/**	Set the maximum amount of time, in seconds, a Theater will wait for a
	Message to be received.
<p>
	@return	The time, in seconds, when a timeout will occur while waiting
		for a Message to be received. The value will be limited to be
		greater than or equal to the {@link #MINIMUM_RECEIVE_TIMEOUT}.
	@see	#Receive_Message()
*/
public Theater Receive_Timeout
	(
	int		timeout
	)
{
if (timeout < MINIMUM_RECEIVE_TIMEOUT)
	timeout = MINIMUM_RECEIVE_TIMEOUT;
Receive_Timeout = timeout;
return this;
}

/**	Get a protocol code for an action name.
<p>
	@return	A protocol code number. This will be -1 if the action name
		does not map to a protocol code.
*/
public static int Action_Code
	(
	String	action
	)
{
Integer
	code = Action_Codes.get (action);
if (code == null)
	return -1;
return code.intValue ();
}

/**	Get the Theater location.
<p>
	The Theater location is the Theater's {@link #Messenger() Messenger}
	{@link Messenger#Client_Hostname() client hostname} with the {@link
	Messenger#Client_Port() client port number}  following a colon (':')
	delimiter. This is the fully qualified location of a Theater.
<p>
	@return	The Theater location String. This will be null if no Messenger
		has yet been {@link #Open(String, int, Message) opened}.
	@see	#Location(String)
*/
public String Location ()
{
if (The_Messenger == null)
	return null;
String
	location = The_Messenger.Client_Hostname ();
if (location != null)
	location += ":" + The_Messenger.Client_Port ();
return location;
}

/**	Get the identity of the Theater's Stage_Manager.
<p>
	<b>N.B.</b>: The availability of a Stage_Manager identity does not
	imply that the Theater is {@link #Opened() open}. The most recent
	Stage_Manager identity that was obtained when the Theater was last
	{@link #Open(String, int, Message) opened} is provided.
<p>
	@return	A Message containing the identity of the Stage_Mangager for
		the Theater. This will be null if the Theater has yet to be
		opened for the first time.
*/
public Message Stage_Manager_Identity ()
{return Stage_Manager_Identity;}

/**	Get the description of this Theater.
<p>
	The first line of the description is the class {@link #ID}. The second
	line of the description provides the Theater location, if it is
	known, and whether or not the Theater is open. If the Theater has a
	{@link #Messenger() Messenger} its description is included on the
	following lines.
<p>
	@return	The Theater description String.
*/
public String toString ()
{
String
	string = ID + NL
		+ "The " + THEATER_NAME;
String
	location = Location ();
if (location != null)
	string += " at location " + location;

string += " is " + (Opened ? "" : "not ") + "open.";

if (The_Messenger != null)
	string += NL + The_Messenger;

return string;
}

/*==============================================================================
	Stage_Manager Connection
*/
/**	Open this Theater.
<p>
	If this Theater is not {@link #Opened() opened} a new Messenger is
	constructed with a communication connection to the Stage_Manager at
	the port on the specifed host.
<p>
	<b>N.B.</b>: The Messenger will that is constructed will be in
	synchronous message receive mode. The initial Stage_Manager handshake
	will be done. The handshake is initiated by an {@link
	#IDENTIFY_ACTION} Message {@link #Receive_Message() received} from
	the Stage_Manager with an {@link Message#Identity(String) Identity}
	Message sent in reply. <b.N.B.</b>: Unless the Stage_Manager will
	accept unauthenticated connections a {@link #KEY_PARAMETER_NAME}
	parameter with the required authentication value must be provided in
	which case the parameter of the same name from the Stage_Manager
	Identify Message will be used to encode the value from the Identity
	Message before it is sent.
<p>
	To start asynchronously listening for messages set the {@link
	#Employer(Message_Delivered_Listener) employer} and begin to {@link
	#Listen_for_Messages() listen for messages}.
<p>
	@param	host	The name or IP address of the host system where the
		Stage_Manager is expected to be running. If null, "localhost"
		will be used.
	@param	port	The port number on the named host to use for
		establishing a communication channel to the Stage_Manager. If
		less than or equal to zero the {@link #Default_Port()}
		will be used.
	@param	identity	An {@link Message#Identity(String) Identity}
		Message. <b>N.B.</b>: A {@link #KEY_PARAMETER_NAME} parameter
		may be required. If null an Identity using the {@link
		#THEATER_NAME} and no key parameter will be used.
	@throws	IOException	If a connection could not be established to
		the Stage_Manager. This will be a Theater_Protocol_Exception
		if there was a problem parsing any of the protocol messages.
*/
public void Open
	(
	String	host,
	int		port,
	Message	identity
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Theater.Open: host " + host + " port " + port);
if (Opened ())
	{
	if ((DEBUG & DEBUG_CONNECTION) != 0)
		System.out.println
			("    Already opened" + NL
			+"<<< Theater.Open");
	return;
	}

if (host == null)
	host = "localhost";
if (port <= 0)
	port = Default_Port;

if (identity == null)
	identity = THEATER_IDENTITY;

//	Copy the identity Message so it can be modified.
Message
	client_identity = null;
try {client_identity =
		new Message (identity)
			.Set (KEY_PARAMETER_NAME, null);}
catch (PVL_Exception exception)
	{
	throw new Theater_Protocol_Exception (ID + NL
		+ "The identity message could not be parsed." + NL
		+ exception.getMessage (),
		Theater_Protocol_Exception.INVALID_MESSAGE,
		exception);
	}

if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Constructing a Messenger on host "
			+ host + " port " + port + NL);
The_Messenger = new Messenger (host, port);
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    The_Messenger: " + The_Messenger + NL
		+"    Identity -" + NL
		+ client_identity);

//	Set the Messenger identity, sans key.
The_Messenger.Identity (client_identity);

//	Stage_Manager handshake:
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Stage_Manager handshake ...");

//	Set Opened true so messages can be received and sent.
Opened = true;

//	Receive the Identify Message.
Message
	identify = Receive_Message ();
if (identify == null)
	{
	if ((DEBUG & DEBUG_CONNECTION) != 0)
		System.out.println
			("    No " + IDENTIFY_ACTION + " message was received after "
				+ Receive_Timeout + " seconds");
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "The Theater could not be opened" + NL
		+ "because the connection handshake with the Stage_Manager failed." + NL
		+ "No " + IDENTIFY_ACTION + " message was received after "
			+ Receive_Timeout + " second"
			+ ((Receive_Timeout == 1) ? "." : "s."),
		Theater_Protocol_Exception.TIMEOUT);
	}
if (! IDENTIFY_ACTION.equals (identify.Action ()))
	{
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "The Theater could not be opened" + NL
		+ "because the connection handshake with the Stage_Manager failed." + NL
		+ "An " + IDENTIFY_ACTION
			+ " message was expected but not received -" + NL
		+ identify,
		Theater_Protocol_Exception.INVALID_MESSAGE);
	}

//	Send a copy of the Identity Message with key that will be encoded.
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Identify message received." + NL
		+"    Sending the identity message");
try {client_identity = new Message (identity);}
catch (PVL_Exception exception) {/* Already copied above */}
Send_Message
	(Stage_Manager.Authentication (identify, client_identity)
	.Set (ACTION_PARAMETER_NAME, IDENTITY_ACTION)	//	Just to be sure.
	.Reply_To (identify));

//	Receive the Stage_Manager response.
identify = Receive_Message ();
if (identify == null)
	{
	if ((DEBUG & DEBUG_CONNECTION) != 0)
		System.out.println
			("    No " + IDENTITY_ACTION + " message was received after "
				+ Receive_Timeout + " seconds");
	Close ();
	throw new Theater_Protocol_Exception (ID + NL
		+ "The Theater could not be opened" + NL
		+ "because the connection handshake with the Stage_Manager failed." + NL
		+ "No response to the " + IDENTITY_ACTION
			+ " message was received after "
			+ Receive_Timeout + " second"
			+ ((Receive_Timeout == 1) ? "." : "s."),
			Theater_Protocol_Exception.TIMEOUT);
	}
if (NACK_ACTION.equals (identify.Action ()))
	{
	if ((DEBUG & DEBUG_CONNECTION) != 0)
		System.out.println
			("    NACK message received -" + NL
			+ identify);
	Close ();
	String
		explanation = identify.Get (EXPLANATION_PARAMETER_NAME);
	throw new Theater_Protocol_Exception (ID + NL
		+ "The Theater could not be opened" + NL
		+ "because the connection handshake with the Stage_Manager failed." + NL
		+ "The identity information was rejected."
		+ ((explanation == null) ? "" : (NL + explanation)),
		Theater_Protocol_Exception.UNAUTHENTICATED);
	}
Stage_Manager_Identity = identify;

if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("    Stage_Manager identity received -" + NL
		+ Stage_Manager_Identity + NL
		+"<<< Theater.Open");
}

/**	Open this Theater.
<p>
	The {@link #Default_Port() default port} on the "localhost" will
	be used to open the communication connection.
<p>
	@param	identity	An {@link Message#Identity(String) Identity}
		Message. <b>N.B.</b>: A {@link #KEY_PARAMETER_NAME} parameter
		may be required. If null an Identity using the {@link
		#THEATER_NAME} and no key parameter will be used.
*/
public void Open
	(
	Message	identity
	)
	throws IOException
{Open (null, 0, identity);}

/**	Test if this Theater is opened.
<p>
	@return	true if a Messenger is active and has a connected
		communication channel to the Stage_Manager; false otherwise.
*/
public boolean Opened ()
{
if (Opened &&
	The_Messenger != null &&
    ! The_Messenger.Is_Connected ())
    Opened = false;
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">-< Theater.Opened: " + Opened);
return Opened;
}

/**	Set the {@link Messenger#Employer(Message_Delivered_Listener)
	Messenger employer} to receive asynchronous message delivery.
<p>
	@param	employer	A Message_Delivered_Listener employer object.
		If null, nothing is done.
	@return	This Theater object.
	@throws	IllegalStateException	If not yet {@link #Open(String,
		int, Message) opened}.
	@see	Message_Delivered_Listener
*/
protected Theater Employer
	(
	Message_Delivered_Listener	employer
	)
{
if (employer != null)
	{
	if (The_Messenger == null)
		throw new IllegalStateException (ID + NL
			+ "Can't set the employer until opened.");
	The_Messenger.Employer (employer);
	}
return this;
}

/**	Begin asynchronously listening for messages.
<p>
	The Messenger is told to {@link Messenger#Listen_for_Messages()
	listen for messages}.
<p>
	@return	true if the Messenger began asynchronously listening for
		messages; false if the Messenger is not connected to the
		communication channel.
	@throws	IllegalStateException	If not yet {@link #Open(String,
		int, Message) opened}.
*/
public boolean Listen_for_Messages ()
{
if (The_Messenger == null)
	throw new IllegalStateException (ID + NL
		+ "Can't listen for messages until opened.");
return The_Messenger.Listen_for_Messages () != null;
}

/**	Close this Theater.
<p>
	If this Theater is {@link #Opened() open} it's Messenger is given
	the {@link Messenger#Done(String) Done} message which will close the
	communication connection.
<p>
	@return	true if this Theater was open at the time the method
		was called; false if the Theater was already closed.
*/
public boolean Close ()
{
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		(">>> Theater.Close");
boolean
	closed = false;
if (Opened ())
	{
	if ((DEBUG & DEBUG_CONNECTION) != 0)
		System.out.println
			("    Closing");
	/*
		The Opened flag is set false *before* the Done signal is sent
		to The_Messenger to prevent a Close loop in response to the
		Done Message received as a result of the Done signal. When the
		Done Message is received and Close is called the call to
		Opened will return false.
	*/
	Opened = false;
	The_Messenger.Done (ID + NL
		+ "Close requested.");
	closed = true;
	}
if ((DEBUG & DEBUG_CONNECTION) != 0)
	System.out.println
		("<<< Theater.Close: " + closed);
return closed;
}

/*==============================================================================
	Messages
*/
/**	Send a Message.
<p>
	If not {@link #Opened() opened} or the Message is null nothing is
	done.
<p>
	A Message is sent via the {@link #Messenger() Messenger}. If an
	IOException occurs the Stage_Manager is (@link #Close()
	closed}. However, if the message can not be parsed the
	Stage_Manager is not closed.
<p>
	@param	message	A Message. If null nothing is done.
	@throws	IOException	If there was a problem send the message. This
		will be a Theater_Protocol_Exception if the message could
		not be parsed.
*/
public void Send_Message
	(
	Message		message
	)
	throws IOException
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Theater.Send_Message");
if (! Opened () ||
	message == null)
	{
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			("<<< Theater.Send_Message: "
				+ ((message == null) ? "No message" : "NOT CONNECTED"));
	return;
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(message.Routing () + NL
		+message);
try {The_Messenger.Send (message);}
catch (IOException exception)
	{
	Close ();
	throw exception;
	}
catch (PVL_Exception exception)
	{
	throw new Theater_Protocol_Exception (ID + NL
		+ "The message to be sent could not be parsed." + NL
		+ exception.getMessage (),
		Theater_Protocol_Exception.INVALID_MESSAGE,
		exception);
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("<<< Theater.Send_Message");
}

/**	Receive a Message.
<p>
	If not {@link #Opened() opened} nothing is done and null is returned.
<p>
	A Message is synchronously received from the {@link #Messenger()
	Messenger}. If the {@link #Receive_Timeout(int) receive timeout}
	expires null is returned. If an IOException occurs the Stage_Manager
	is (@link #Close() closed}. However, if the message
	can not be parsed the Stage_Manager is not closed.
<p>
	@return	A Message. This will be null if this Theater is not open or
		if the receive timeout occured.
	@throws	IOException	If there was a problem receiving a message. This
		will be a Theater_Protocol_Exception if the message could
		not be parsed.
*/
public Message Receive_Message ()
	throws IOException
{
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		(">>> Theater.Receive_Message");
if (! Opened ())
	{
	if ((DEBUG & DEBUG_MESSAGES) != 0)
		System.out.println
			("<<< Theater.Receive_Message: NOT CONNECTED");
	return null;
	}
Message
	message = null;
try {message = The_Messenger.Receive (Receive_Timeout);}
catch (IOException exception)
	{
	Close ();
	throw exception;
	}
catch (PVL_Exception exception)
	{
	throw new Theater_Protocol_Exception (ID + NL
		+ "The message received could not be parsed." + NL
		+ exception.getMessage (),
		Theater_Protocol_Exception.INVALID_MESSAGE,
		exception);
	}
if ((DEBUG & DEBUG_MESSAGES) != 0)
	System.out.println
		("    " + ((message == null) ?
			"No message recieved" :
			("Message received -" + NL
			+ message.Routing () + NL
			+ message)) + NL
		+"<<< Theater.Receive_Message");
return message;
}

/*==============================================================================
	Utilities
*/
/**	Provide an abbreviated theater location for a given theater
	location.
<p>
	The format of a theater location string is:
<p>
	<i>hostname</i>[<b><i>delimiter</i></b><i>port</i>]
<p>
	The <i>hostname</i> portion is the {@link #Name(String) name} obtained
	from the {@link #Host(String) host} part of the given location.
<p>
	The delimiter is the {@link #PORT_DELIMITER_CHAR}. If it is present
	in the given location the remaining port portion is compared to the
	Theater's {@link #Default_Port() default port}. If they are the same
	the delimiter and port portion are not included in the returned
	location String.
<p>
	@param	location	A theater location or name String. If null, or
		the emtpy String, null is returned.
	@return	A possibly modified theater location String.
	@see	#Location()
*/
public static String Location
	(
	String	location
	)
{
if (location == null ||
	location.length () == 0)
	return null;

String
	hostname = Name (Host (location)),
	port = null;
int
	index = location.indexOf (PORT_DELIMITER_CHAR);
if (index >= 0)
	{
	port = location.substring (index);
	if (! port.equals (String.valueOf (PORT_DELIMITER_CHAR)
			+ String.valueOf (Default_Port)))
		hostname += port;
	}
return hostname;
}

/**	Provide a fully qualified theater location.
<p>
	The format of a theater location string is:
<p>
	<i>host</i>[<b><i>delimiter</i></b><i>port</i>]
<p>
	The <i>host</i> may be a hostname, short or fully qualified, or IP
	address. If the hostname can be {@link Host#Full_Hostname(String)
	fully qualified} that name is used; otherwise the hostname portion of
	the location is used as given.
<p>
	<b>N.B.</b>: A hashmap of host names to fully qualified hostnames is
	maintained to avoid the, often very time consuming, DNS lookup of
	canonical hostnames each time this utility is called on a previously
	examined location.
<p>
	The optional <i>port</i> is the system port number used to
	connect to the Stage_Manager. If no port number is present in the
	given location the {@link #Default_Port() default port} is used. The
	delimiter is the {@link #PORT_DELIMITER_CHAR}.
<p>
	@param	location	A theater location or name String. If null, or
		the emtpy String, null is returned.
	@return	A possibly modified theater location String.
	@see	#Location(String)
*/
public static String Full_Location
	(
	String	location
	)
{
if (location == null ||
	location.length () == 0)
	return null;

String
	hostname,
	port = null;
int
	index = location.indexOf (PORT_DELIMITER_CHAR);
if (index >= 0)
	{
	if (index == 0 ||
		index == (location.length () - 1))
		//	Port delimiter at the begining or end of the string.
		return location;

	hostname = location.substring (0, index);
	port     = location.substring (++index);
	if (port.length () == 0)
		port = null;
	}
else
	hostname = location;

//	Check the hostnames map for a cached reference.
location = Hostnames_Map.get (hostname);
if (location == null)
	{
	//	Get the canonical hostname from the system.
	if ((location = Host.Full_Hostname (hostname)) == null)
		//	Unknown; use the unchanged hostname.
		location = hostname;

	//	Update the hostnames cache.
	Hostnames_Map.put (hostname, location);
	}

if (port == null)
	port = String.valueOf (Theater.Default_Port ());

location += PORT_DELIMITER_CHAR + port;
return location;
}


private static Hashtable<String, String>
	Hostnames_Map		= new Hashtable<String, String> ();


/**	Get the name portion of a Theater location.
<p>
	The name portion of a Theater location is the leading substring up
	to, but not including, the first period character ('.') in the
	String plus the trailing substring starting with the last {@link
	#PORT_DELIMITER_CHAR}, if present. If no period character is found
	the entire location is returned. Thus the {@link #Port(String) port
	number}, if present, is retained but only the {@link
	Host#Hostname(String) short hostname} (but without doing the system
	name lookup) is returned.
<p>
	However, if the name portion of the string is numeric, then the entire
	name is always returned since this is most likely an IP address.
<p>
	@param	location	A Theater location String. If null, null is
		returned.
	@return	The Theater name String.
*/
public static String Name
	(
	String	location
	)
{
if (location != null)
	{
	int
		name_index = location.indexOf ('.');
	if (name_index > 0)
		{
		String
			name = location.substring (0, name_index);

		//	Test for an IP address.
		try 
			{
			Integer.parseInt (name);
			// The location appears to be an IP address.
			return location;
			}
		catch (NumberFormatException exception) {}

		int
			port_index = location.lastIndexOf (PORT_DELIMITER_CHAR);
		if (port_index >= 0)
			location = name + location.substring (port_index);
		else
			location = name;
		}
	}
return location;
}

/**	Get the Theater host name for a location or name.
<p>
	@param	location	A Theater location or name String.
	@return	If the location String contains a {@link
		#PORT_DELIMITER_CHAR} the substring preceeding the delimiter is
		returned. Otherwise the entire String is returned. If the
		location is null, null is returned.
	@see	#Location(String)
*/
public static String Host
	(
	String	location
	)
{
if (location == null)
	return null;

int
	index = location.indexOf (PORT_DELIMITER_CHAR);
if (index >= 0)
	return location.substring (0, index);
return location;
}

/**	Get the Theater communications port number for a location.
<p>
	@param	location	A Theater location String. If null -1 is returned.
	@return	If the location String contains a port number following a
		{@link #PORT_DELIMITER_CHAR} that value is returned. Otherwise
		the {@link #Default_Port() default port} for a Theater is
		returned.
	@see	#Location(String)
*/
public static int Port
	(
	String	location
	)
{
if (location != null)
	{
	int
		index = location.indexOf (PORT_DELIMITER_CHAR);
	if (index >= 0 &&
		index < (location.length () - 1))
		{
		try {return Integer.parseInt (location.substring (++index));}
		catch (NumberFormatException exception) {}
		}
	else
		return Theater.Default_Port ();
	}
return -1;
}

/**	Generate a NACK - no acknowledge - Message with optional explanation
	and exception parameters.
<p>
	A {@link Message#NACK() NACK action Message} is obtained. If the
	explanation is non-null and not empty an {@link
	#EXPLANATION_PARAMETER_NAME} parameter is set in the Message with
	the explanation. If the exception is non-null an {@link
	#EXCEPTION_PARAMETER_NAME} parameter is set in the Message with the
	exception's String description.
<p>
	@param	explanation	An explanatory String. May be null.
	@param	exception	An Exception associated with the NACK. May be null.
	@return	A NACK action Message.
*/
public static Message NACK
	(
	String		explanation,
	Exception	exception
	)
{
Message
	message = Message.NACK ();
if (explanation != null &&
	explanation.length () != 0)
	message.Set (EXPLANATION_PARAMETER_NAME, explanation);
if (exception != null)
	message.Set (EXCEPTION_PARAMETER_NAME, exception.toString ());
return message;
}

/**	Assemble a Message containing Processing_Changes parameters.
<p>
	The Message will have an {@link #ACTION_PARAMETER_NAME} with the
	specified action value.
<p>
	Each Processing_Changes state variable that has a value indicating a
	changed Conductor state is entered as a parameter in the assembled
	Message:
<dl>
<dt>{@link #CONFIGURATION_PARAMETER_NAME}
<dd>A Parameter Group containing the current Conductor Configuration
	parameters. This will not be present if the changes {@link
	Processing_Changes#Configuration() Configuration} is null.

<dt>{@link #PROCESSING_STATE_ACTION}
<dd>The current Conductor processing state code. This not be present if
	the changes {@link Processing_Changes#Processing_State() processing
	state} is zero.

<dt>{@link #SOURCE_RECORD_PARAMETER_NAME}
<dd>The current Conductor source record as an Array of String Values.
	This will not be present if the changes {@link
	Processing_Changes#Source_Record() source record} is null.

<dt>{@link #PROCEDURE_RECORD_PARAMETER_NAME}
<dd>The current Conductor procedure record as an Array of String Values.
	This will not be present if the changes {@link
	Processing_Changes#Procedure_Record() procedure record} is null.

<dt>{@link #SOURCES_REFRESHED_PARAMETER_NAME}
<dd>A flag value of "true" indicating that the Conductor has just
	refreshed its cache of source records. This will not be present if
	the changes {@link Processing_Changes#Sources_Refreshed() sources
	refreshed} value is false.

<dt>{@link #PROCEDURES_CHANGED_PARAMETER_NAME}
<dd>A flag value of "true" indicating that the Conductor has just
	reloaded a modified procedures table. This will not be present if the
	changes {@link Processing_Changes#Procedures_Changed() procedures
	changed} value is false;

<dt>{@link #SEQUENTIAL_FAILURES_ACTION}
<dd>The current number of sequential source record processing failures
	that the Conductor encountered. This will not be present if the
	changes {@link Processing_Changes#Sequential_Failures() sequential
	failures} value is less than zero.

<dt>{@link #PROCESSING_EXCEPTION_ACTION}
<dd>A description of the last processing error that occured in the
	Conductor. This will not be present if the changes {@link
	Processing_Changes#Error_Condition() error condition} is null.

<dt>{@link #QUIT_ACTION}
<dd>A flag value of "true" indicating that the Conductor process is about
	to exit. This will not be present if the changes {@link
	Processing_Changes#Exiting() exiting} value is false.
<p>
	@param	action	A String naming the value of the {@link
		#ACTION_PARAMETER_NAME} parameter of the Message. If null,
		{@link #PROCESSING_CHANGES_ACTION} will be used.
	@param	changes	A Processing_Changes object. If null, null is returned.
	@return	A Message containing Processing_Changes parameters. This will
		be null if, and only if, the changes argument is null.
*/
public static Message Processing_Changes
	(
	String				action,
	Processing_Changes	changes
	)
{
if (changes == null)
	return null;

if (action == null)
	action = PROCESSING_CHANGES_ACTION;

Message
	message = Message.Action (action);

if (changes.Configuration () != null)
	message.Add (CONFIGURATION_PARAMETER_NAME,
		changes.Configuration ());

if (changes.Processing_State () != 0)
	message.Set (PROCESSING_STATE_ACTION,
		changes.Processing_State ());

if (changes.Source_Record () != null)
	{
	try {message.Set (SOURCE_RECORD_PARAMETER_NAME,
			changes.Source_Record ());}
	catch (PVL_Exception exception)
		{/* Invalid source record (shouldn't happen) */}
	}

if (changes.Procedure_Record () != null)
	{
	try {message.Set (PROCEDURE_RECORD_PARAMETER_NAME,
			changes.Procedure_Record ());}
	catch (PVL_Exception exception)
		{/* Invalid procedure record (shouldn't happen) */}
	}

if (changes.Sources_Refreshed ())
	message.Set (SOURCES_REFRESHED_PARAMETER_NAME, "true");

if (changes.Procedures_Changed ())
	message.Set (PROCEDURES_CHANGED_PARAMETER_NAME, "true");

if (changes.Sequential_Failures () >= 0)
	message.Set (SEQUENTIAL_FAILURES_ACTION,
		changes.Sequential_Failures ());

if (changes.Error_Condition () != null)
	message.Set (PROCESSING_EXCEPTION_ACTION,
		changes.Error_Condition ());

if (changes.Exiting ())
	message.Set (QUIT_ACTION, "true");

return message;
}

/**	Assemble a Processing_Changes object using Message parameter values.
<p>
	Each possible {@link #Processing_Changes(String, Processing_Changes)
	Processing_Changes Message} parameter is sought in the provided
	Message. For parameters that are found their values are used to set
	the corresponding Conductor processing state variable in a
	Processing_Changes object that is initialized with all its variables
	set to the unchanged value.
<p>
	@param	message	A Message containing Processing_Changes parameters.
		If null, null is returned.
	@return	A Processing_Changes object. This will be null if, and only if,
		the message argument is null.
*/
public static Processing_Changes Processing_Changes
	(
	Message		message
	)
{
if (message == null)
	return null;

Processing_Changes
	changes = new Processing_Changes ();
String
	value;

try {changes.Configuration
		(message.Find (CONFIGURATION_PARAMETER_NAME, Parameter.AGGREGATE));}
catch (Configuration_Exception exception)
	{/*	Protocol error */}

value = message.Get (PROCESSING_STATE_ACTION);
if (value != null)
	{
	try {changes.Processing_State (Integer.parseInt (value));}
	catch (NumberFormatException exception)
		{/*	Protocol error */}
	}

value = message.Get (SEQUENTIAL_FAILURES_ACTION);
if (value != null)
	{
	try {changes.Sequential_Failures (Integer.parseInt (value));}
	catch (NumberFormatException exception)
		{/*	Protocol error */}
	}

changes
	.Source_Record
		(Record (message.Value_of (SOURCE_RECORD_PARAMETER_NAME)))
	.Procedure_Record
		(Record (message.Value_of (PROCEDURE_RECORD_PARAMETER_NAME)))
	.Error_Condition
		(message.Get (PROCESSING_EXCEPTION_ACTION))
	.Sources_Refreshed
		("true".equals (message.Get (SOURCES_REFRESHED_PARAMETER_NAME)))
	.Procedures_Changed
		("true".equals (message.Get (PROCEDURES_CHANGED_PARAMETER_NAME)))
	.Exiting
		("true".equals (message.Get (QUIT_ACTION)));

return changes;
}

/**	Assemble a data table from an Array Value.
<p>
	Each Array entry is expected to an Array Value that is assembled
	into a Record Vector.
<p>
	@param	array	An Array of Array Values.
	@return	A Vector of String Vectors. This will be null if any of
		the Array entries can not be assembled into a Record Vector
		of Strings.
*/
public static Vector<Vector<String>> Table
	(
	Value	array
	)
{
if (array == null ||
	array.Array_Size () == 0)
	return null;

Vector<Vector<String>>
	table = new Vector<Vector<String>> (array.Array_Size ());
Vector<String>
	record;
Iterator
	records = array.iterator ();
while (records.hasNext ())
	{
	if ((record = Record ((Value)records.next ())) == null)
		return null;
	table.add (record);
	}
return table;
}

/**	Assemble a data table record from an Array Value.
<p>
	Each Array entry is converted into its String representation and
	added to the record Vector.
<p>
	@param	array	A Array Value.
	@return	A Vector of Strings. This will be null if any of the Array
		entries can not be converted to a String (for example, if the
		entry is an Array).
*/
public static Vector<String> Record
	(
	Value	array
	)
{
if (array == null ||
	! array.Is_Array ())
	return null;

Vector<String>
	record = new Vector<String> (array.Array_Size ());
Value
	field;
Iterator
	fields = array.iterator ();
while (fields.hasNext ())
	{
	try {record.add (((Value)fields.next ()).String_Data ());}
	catch (PVL_Exception exception)
		{
		//	The field is an Array!
		record = null;
		break;
		}
	}
return record;
}

}
