/*	Projection

PIRL CVS ID: Projection.java,v 1.8 2012/09/11 02:23:20 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Image_Tools;

import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Strings.Words;

import	java.util.NoSuchElementException;
import	java.lang.reflect.Constructor;
import	java.lang.reflect.InvocationTargetException;
import	java.lang.UnsupportedOperationException;
import	java.awt.Point;
import	java.awt.geom.Point2D;

//	For main.
import	PIRL.PVL.Parser;
import	PIRL.Utilities.Streams;

/**	A <i>Projection</i> is used to algorithmically map, or "project",
	from one image coordinate system to another while maintaining the
	spatial relationships of all coordinates.
<p>
	Spatial projections are used in many contexts. The typical example is
	when a camera image of curved surface is to be mapped as if the
	surface were flat. The commonly known Mercator projection of a map of
	the Earth is such a projection. There are many map projections. See
	Snyder, John P., "Map Projections - A Working Manual", U.S Geological
	Survey Professional Paper 1395, 1987 for a discussion of the map
	projection algorithm used in these classes.
<p>
	The core projection characteristic is an algorithm that maps
	coordinates from one coordinate system to another. The specific
	projection algorithms used - the implementation of a set of equations
	- will, of course, vary from one type of projection to another to
	achieve the desired mapping effect. In addition, the choice of
	projection is likey to depend on the range of coordinate values to be
	mapped - projections are likely to have increasing visual distortion
	effects as boundary conditions are approached - or the accuracy to be
	achieved - the implementing equations are like to make various
	simplifying assumptions. Nevertheless, all projections are expected
	to map coordinates between image and world coordinate systems. The
	former is in image pixel units of sample - horizontal distance from
	the left-most pixel - and and line - vertical distance from the
	top-most line - of a rectangular array of pixels. The latter is
	typically in planetary longitude - angular angular distance about the
	center of the planet relative to a zero-reference longitude - and
	latitude - angular angular distance about the center of the planet
	relative to the equator - usually measured in decimal degrees.
	However, other "world" coordinate systems are quite possible. In
	order to accommodate images of arbitrary location, an intermediate
	projeciton coordinate system is typically used by the projection
	equations, with a simple translational equation used to map between
	the projection coordinate system and the location of the image raster
	in that system.
<p>
	A specific projection, implemented as a subclass of Projection, will
	require various parameters need by the projection equations. These
	are supplied by a PVL Parameter Aggregate. Because this Projection
	implementation assumes that planetary projections are to be
	implemented by subclasses, the Parameter group used to construct a
	Projection collects the common set of parameter values required by
	the specific projection algorithms. If all the required parameters
	are not provided the default identity projection - in which
	coordinate values are not changed - is used.
<p>
	@author		Bradford Castalia UA/PIRL
	@version	1.8
*/
public class Projection
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Image_Tools.Projection (1.8 2012/09/11 02:23:20)";

//!	The projection name (override in subclass).
private static final String
	PROJECTION_NAME							= "Idenitity";

public static final String
	PROJECTION_PARAMETER_NAME				= "MAP_PROJECTION_TYPE",
		EQUIRECTANGULAR_PROJECTION_NAME			= "EQUIRECTANGULAR",
		POLAR_STEREOGRAPHIC_PROJECTION_NAME		= "POLAR STEREOGRAPHIC",
		POLARSTEREOGRAPHIC_PROJECTION_NAME		= "POLARSTEREOGRAPHIC",
	PROJECTION_LATITUDE_PARAMETER_NAME		= "PROJECTION_LATITUDE_TYPE",
		PLANETOCENTRIC_PROJECTION_NAME			= "PLANETOCENTRIC",
	EQUITORIAL_RADIUS_PARAMETER_NAME		= "A_AXIS_RADIUS",
	POLAR_RADIUS_PARAMETER_NAME				= "C_AXIS_RADIUS",
	METERS_PER_PIXEL_PARAMETER_NAME			= "MAP_SCALE",
	POSITIVE_LONGITUDE_PARAMETER_NAME		= "POSITIVE_LONGITUDE_DIRECTION",
		POSITIVE_LONGITUDE_EAST_NAME			= "EAST",
		POSITIVE_LONGITUDE_WEST_NAME			= "WEST",
	CENTER_LATITUDE_PARAMETER_NAME			= "CENTER_LATITUDE",
	CENTER_LONGITUDE_PARAMETER_NAME			= "CENTER_LONGITUDE",
	SAMPLE_OFFSET_PARAMETER_NAME			= "SAMPLE_PROJECTION_OFFSET",
	LINE_OFFSET_PARAMETER_NAME				= "LINE_PROJECTION_OFFSET",
	NOT_APPLICABLE_CONSTANT_PARAMETER_NAME	= "NOT_APPLICABLE_CONSTANT";

protected boolean
	Not_Identity			= false,
	Positive_West,							//	POSITIVE_LONGITUDE_DIRECTION
	Planetocentric;							//	PROJECTION_LATITUDE_TYPE

public static final double
	DEFAULT_NA				= -9998.0;

protected double
	Meters_per_Pixel		= 0.0,			//	MAP_SCALE
	Equitorial_Radius		= 0.0,			//	A_AXIS_RADIUS <km>
	Polar_Radius			= 0.0,			//	C_AXIS_RADIUS <km>
	NA						= DEFAULT_NA;	//	NOT_APPLICABLE_CONSTANT

/**	Projection planetographic center latitude in radians.
*/
protected double
	Center_Latitude			= 0.0;			//	CENTER_LATITUDE

/**	Projection center longitude in radians.
*/
protected double
	Center_Longitude		= 0.0;			//	CENTER_LONGITUDE

protected double
	Sample_Offset			= 0.0,			//	SAMPLE_PROJECTION_OFFSET
	Line_Offset				= 0.0;			//	LINE_PROJECTION_OFFSET

//	Derived values.
protected double
	Eccentricity;

public static final double
	PI_OVER_2				= StrictMath.PI * 0.5,
	PI_OVER_4				= StrictMath.PI * 0.25,
	DBL_EPSILON				= 2.2204460492503131E-16;

public static final boolean
	DEFAULT_USE_SPHERICAL	= false;

private static boolean
	Use_Spherical			= DEFAULT_USE_SPHERICAL;

private double
	Coefficient_PC_to_PG,
	Coefficient_PG_to_PC;


//  DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_CONVERTERS		= 1 << 2,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Projection from a Parameter Aggregate.
<p>
	@param	parameters	The Parameter Aggregate containing (at least) the
		minimally required parameter values. If all required parameters
		are not provided - including if the argument is null - the {@link
		#Is_Identity() identity projection flag} is set.
	@throws	PVL_Exception	If there is a problem reading the parameters
		or a parameter is found to have an invalid value.
*/
public Projection
	(
	Parameter	parameters
	)
	throws PVL_Exception
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Projection: " + parameters);
if (parameters == null)
	{
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("<<< Projection");
	return;
	}

try {NA =
		Find_Parameter (parameters, NOT_APPLICABLE_CONSTANT_PARAMETER_NAME)
			.Value ().double_Data ();}
catch (Exception exception) {/* Use the default */}

try
	{
	Meters_per_Pixel =
		Find_Parameter (parameters, METERS_PER_PIXEL_PARAMETER_NAME)
			.Value ().double_Data ();
	if (Meters_per_Pixel <= 0.0 ||
		Meters_per_Pixel == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + METERS_PER_PIXEL_PARAMETER_NAME + " value: "
				+ Meters_per_Pixel + '\n'
			+ "The value must be a valid non-zero positive.");
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("     Meters_per_Pixel = " + Meters_per_Pixel);

	Equitorial_Radius =
		Find_Parameter (parameters, EQUITORIAL_RADIUS_PARAMETER_NAME)
			.Value ().double_Data () * 1000.0;
	if (Equitorial_Radius <= 0.0 ||
		Equitorial_Radius == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + EQUITORIAL_RADIUS_PARAMETER_NAME + " value: "
				+ Equitorial_Radius + '\n'
			+ "The value must be a valid non-zero positive.");
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("    Equitorial_Radius = " + Equitorial_Radius);

	Polar_Radius =
		Find_Parameter (parameters, POLAR_RADIUS_PARAMETER_NAME)
			.Value ().double_Data () * 1000.0;
	if (Polar_Radius <= 0.0 ||
		Polar_Radius == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + POLAR_RADIUS_PARAMETER_NAME + " value: "
				+ Polar_Radius + '\n'
			+ "The value must be a valid non-zero positive.");
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("         Polar_Radius = " + Polar_Radius);

	//	Planetocentric_to_Planetographic conversion coefficient.
	Coefficient_PC_to_PG =
		(Equitorial_Radius / Polar_Radius) * (Equitorial_Radius / Polar_Radius);
	//	Planetographic_to_Planetocentric conversion coefficient.
	Coefficient_PG_to_PC =
		(Polar_Radius / Equitorial_Radius) * (Polar_Radius / Equitorial_Radius);

	Eccentricity = Eccentricity (Polar_Radius, Equitorial_Radius);
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("         Eccentricity = " + Eccentricity);

	String
		string =
			Find_Parameter (parameters, POSITIVE_LONGITUDE_PARAMETER_NAME)
				.Value ().String_Data ();
	if (string.equalsIgnoreCase (POSITIVE_LONGITUDE_WEST_NAME))
		Positive_West = true;
	else
	if (string.equalsIgnoreCase (POSITIVE_LONGITUDE_EAST_NAME))
		Positive_West = false;
	else
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + POSITIVE_LONGITUDE_PARAMETER_NAME + " value: "
				+ string + '\n'
			+ "The value must be \"" + POSITIVE_LONGITUDE_WEST_NAME
				+ "\" or \"" + POSITIVE_LONGITUDE_EAST_NAME + "\".");
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("        Positive_West = " + Positive_West);

	Sample_Offset =
		Find_Parameter (parameters, SAMPLE_OFFSET_PARAMETER_NAME)
			.Value ().double_Data ();
	if (Sample_Offset == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + SAMPLE_OFFSET_PARAMETER_NAME + " value: "
				+ Sample_Offset);
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("        Sample_Offset = " + Sample_Offset);

	Line_Offset =
		Find_Parameter (parameters, LINE_OFFSET_PARAMETER_NAME)
			.Value ().double_Data ();
	if (Line_Offset == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + LINE_OFFSET_PARAMETER_NAME + " value: "
				+ Line_Offset);
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("          Line_Offset = " + Line_Offset);

	try {Planetocentric = 
			Find_Parameter (parameters, PROJECTION_LATITUDE_PARAMETER_NAME)
				.Value ().String_Data ()
				.equalsIgnoreCase (PLANETOCENTRIC_PROJECTION_NAME);}
	catch (NoSuchElementException exception) {/* false */}
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("       Planetocentric = " + Planetocentric);

	Center_Longitude =
		Find_Parameter (parameters, CENTER_LONGITUDE_PARAMETER_NAME)
			.Value ().double_Data ();
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("     Center_Longitude = " + Center_Longitude);
	if (Center_Longitude == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + CENTER_LONGITUDE_PARAMETER_NAME + " value: "
				+ Center_Longitude);
	Center_Longitude = StrictMath.toRadians (Center_Longitude);
	if (Positive_West)
		Center_Longitude *= -1.0;
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("                        " + Center_Longitude);

	Center_Latitude = 
		Find_Parameter (parameters, CENTER_LATITUDE_PARAMETER_NAME)
			.Value ().double_Data ();
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("      Center_Latitude = " + Center_Latitude);
	if (Center_Latitude == NA)
		throw new PVL_Exception (ID + '\n'
			+ "Invalid " + CENTER_LATITUDE_PARAMETER_NAME + " value: "
				+ Center_Latitude);
	Center_Latitude = StrictMath.toRadians (Center_Latitude);
	if (Planetocentric)
		Center_Latitude = Planetocentric_to_Planetographic (Center_Latitude);
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("                        " + Center_Latitude);

	Not_Identity = true;
	}
catch (NoSuchElementException exception)
	{/*	Take this exception to mean that the image is not projected. */}
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Projection");
}

/**	Constructs a default identity Projection.
<p>
	The {@link #Is_Identity() identity projection flag} is set.
*/
public Projection ()
{}

/**	Creates a Projection object that uses an appropriate specific
	Projection subclass projection implementation.
<p>
	The choice of specific projection implementation is based on the
	value of the Parameter having the {@link #PROJECTION_PARAMETER_NAME}.
	This name is used to assemble a {@link #Projection_Class_Name(String)
	projection specific class name} to be loaded. Once loaded, a new
	instance is constructed using the parameters argument.
<p>
	As a special case, if the projection name is {@link
	#POLAR_STEREOGRAPHIC_PROJECTION_NAME} the "_Spherical" suffix is
	appended if {@link #Use_Spherical(boolean)} is enabled so the faster but
	slightly less accurate spherical form of the projection will be used;
	otherwise the "_Elliptical" suffix is appended so the elliptical form
	will be used. Also, if the name is {@link
	#POLARSTEREOGRAPHIC_PROJECTION_NAME} it will be converted to the
	{@link #POLAR_STEREOGRAPHIC_PROJECTION_NAME}
<p>
	@param	parameters	The Parameter Aggregate containing the projection
		definition parameter values. These parameters will be passed to
		the specific projection class to be constructed. If null, not an
		Aggregate Parameter, or there is no {@link
		#PROJECTION_PARAMETER_NAME} parameter a default Projection is
		constructed which will have the {@link #Is_Identity() identity
		projection flag} set.
	@return	A Projection or Projection subclass, Object.
	@throws	PVL_Exception	If there is a problem reading the parameters
		or a parameter is found to have an invalid value.
	@throws	NoSuchElementException	If any required parameter was missing
		from the provided parameters.
	@throws	UnsupportedOperationException	If the expected Projection
		subclass could not be loaded or instantiated.
	@throws	Throwable	If the constructor of the loaded projection class
		throws an unexpected exception.
	@see	Equirectangular_Projection
	@see	Polar_Stereographic_Elliptical_Projection
	@see	Polar_Stereographic_Spherical_Projection
*/
public static Projection Create
	(
	Parameter	parameters
	)
	throws
		PVL_Exception,
		NoSuchElementException,
		UnsupportedOperationException,
		Throwable
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Projection.Create: " + parameters);
Projection
	projection = null;
if (parameters == null ||
	! parameters.Is_Aggregate ())
	//	Identity Projection when no parameters.
	projection = new Projection ();
String
	projection_name = null;
try
	{
	Parameter
		projection_parameter
			= Find_Parameter (parameters, PROJECTION_PARAMETER_NAME);
	try {projection_name = projection_parameter.Value ().String_Data ();}
	catch (PVL_Exception except) {}
	}
catch (NoSuchElementException exception) {}
if (projection_name == null)
	//	Identity Projection when no valid projection name parameter.
	projection = new Projection ();
else
	{
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("    " + PROJECTION_PARAMETER_NAME + " = " + projection_name);
	if (projection_name.equalsIgnoreCase (POLAR_STEREOGRAPHIC_PROJECTION_NAME) ||
		projection_name.equalsIgnoreCase (POLARSTEREOGRAPHIC_PROJECTION_NAME))
		{
		if (Use_Spherical)
			projection_name += "_spherical";
		else
			projection_name += "_elliptical";
		}
	projection_name = Projection_Class_Name (projection_name);
	if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
		System.out.println
			("    Projection class name: " + projection_name);

	Class
		projection_class = null;
	try {projection_class = Class.forName (projection_name);}
	catch (ClassNotFoundException exception)
		{
		try {projection_class =
			Class.forName ("PIRL.Image_Tools." + projection_name);}
		catch (ClassNotFoundException except)
			{
			throw new UnsupportedOperationException (ID + '\n'
				+ "Unable to find the \"" + projection_name + "\" class.");
			}
		}
	try
		{
		Class[]
			parameter_class = new Class[1];
		parameter_class[0] = parameters.getClass ();
		Constructor
			projection_constructor =
				projection_class.getConstructor (parameter_class);
		Object
			argument[] = new Object[1];
		argument[0] = parameters;
		projection = (Projection)projection_constructor.newInstance (argument);
		if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
			System.out.println
				("    Projection: " + projection);
		}
	catch (InvocationTargetException exception)
		{
		Throwable
			throwable = exception.getCause ();
		if (throwable != null)
			throw throwable;
		throw exception;
		}
	catch (NoSuchElementException exception)
		{throw exception;}
	catch (Exception exception)
		{
		UnsupportedOperationException
			unsupported = new UnsupportedOperationException (ID + '\n'
				+ "Unable to load the \""
					+ projection_name + "\" projection class.");
		unsupported.initCause (exception);
		}
	}
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Projection.Create: " + parameters);
return projection;
}

/*==============================================================================
	Accessors
*/
public String Name ()
{return PROJECTION_NAME;}

public boolean Is_Identity ()
{return ! Not_Identity;}

public double Resolution ()
{return Meters_per_Pixel;}

public double Equitorial_Radius ()
{return Equitorial_Radius;}

public double Polar_Radius ()
{return Polar_Radius;}

public double Eccentricity ()
{return Eccentricity;}

public double Sample_Offset ()
{return Sample_Offset;}

public double Line_Offset ()
{return Line_Offset;}

public boolean Positive_West ()
{return Positive_West;}

public Projection Positive_West
	(
	boolean	positive_west
	)
{Positive_West = positive_west; return this;}

public boolean Planetocentric ()
{return Planetocentric;}

public Projection Planetocentric
	(
	boolean	planetocentric
	)
{Planetocentric = planetocentric; return this;}

public double Center_Latitude ()
{return Center_Latitude;}

public double Center_Longitude ()
{return Center_Longitude;}


public static boolean Use_Spherical
	(
	boolean	enabled
	)
{
boolean
	previous_state = Use_Spherical;
Use_Spherical = enabled;
return previous_state;
}

/*==============================================================================
	Converters
*/
/**	Get the world longitude,latitude coordinate for an image sample.line
	coordinate.
<p>
	This method simply returns a copy of the coordinate argument. A
	projection specific subclass will override this method.
<p>
	@param	image_coordinate	The image sample,line coordinate.
	@return	A copy of the image_coordinate.
*/
public Point2D.Double to_World
	(
	Point2D	image_coordinate
	)
{
if (image_coordinate == null)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to project world coordinates to image coordinates\n"
		+ "because the world coordinates point is null.");
return new Point2D.Double (image_coordinate.getX (), image_coordinate.getY ());
}

/**	Get the image sample,line coordinate for a world longitude,latitude
	coordinate.
<p>
	This method simply returns a copy of the coordinate argument. A
	projection specific subclass will override this method.
<p>
	@param	world_coordinate	The world longitude,latitude coordinate.
	@return	A copy of the world_coordinate.
*/
public Point to_Image
	(
	Point2D	world_coordinate
	)
{
if (world_coordinate == null)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to project image coordinates to world coordinates\n"
		+ "because the image coordinates point is null.");
return new Point ((int)world_coordinate.getX (), (int)world_coordinate.getY ());
}

/** Converts an angle in decimal degrees, to a degrees, minutes, seconds
	representation.
<p>
	The format of the degrees, minutes, seconds String representation is:
<p><blockquote>
	DDDd MMm SS.SSSs
</blockquote>
	where DDD is integer degrees, MM is integer minutes in the range 0 to
	59, and SS.SSS is seconds with fractional seconds in the range 0.0 to
	59.999 For example, 206.291 degrees is represented as "206d 17m
	27.600s". <b>N.B.</b>: A fixed field format is used in which leading
	spaces and trailing zeros are used to each value occurs at a specific
	location in the String representation.
<p>
	@param	angle	The angle in degrees to be represented.
	@return	The String representation.
*/
public static String Degrees_Minutes_Seconds
	(
	double	angle
	)
{
int
	degrees = (int)angle;
double
	fraction = Math.abs (angle - degrees) * 60.0;
int
	minutes = (int)fraction;
fraction = (fraction - minutes) * 60.0;
int
	seconds = (int)fraction;
int
	msecs = (int)((fraction - seconds) * 1000.0 + 0.5);
if (msecs >= 1000)
	{
	msecs -= 1000;
	seconds++;
	}
if (seconds >= 60)
	{
	seconds -= 60;
	minutes++;
	}
if (minutes >= 60)
	{
	minutes -= 60;
	degrees++;
	}
return
	((degrees < 10) ? "  " :
	((degrees < 100) ? " " : ""))
	+ degrees + "d "
	+
	((minutes < 10) ? " " : "")
	+ minutes + "m "
	+
	((seconds < 10) ? " " : "")
	+ seconds + "."
	+
	((msecs < 10) ? "00" :
	((msecs < 100) ? "0" : ""))
	+ msecs + "s";
}

/** Converts an angle in decimal degrees, to an hours, minutes, seconds
	representation.
<p>
	The format of the hours, minutes, seconds String representation is:
<p><blockquote>
	HHh MMm SS.SSSs
</blockquote>
	where HH is integer hours in the range 0 to 24, MM is integer minutes
	in the range 0 to 59, and SS.SSS is seconds with fractional seconds
	in the range 0.0 to 59.999 For example, 206.291 degrees is
	represented as "13h 45m  9.840s". <b>N.B.</b>: A fixed field format
	is used in which leading spaces and trailing zeros are used to each
	value occurs at a specific location in the String representation.
<p>
	@param	angle	The angle in degrees to be represented.
	@return	The String representation.
*/
public static String Hours_Minutes_Seconds
	(
	double	angle
	)
{
angle = To_360 (angle) / 15.0;
int
	hours = (int)angle;
double
	fraction = Math.abs (angle - hours) * 60.0;
int
	minutes = (int)fraction;
fraction = (fraction - minutes) * 60.0;
int
	seconds = (int)fraction;
int
	msecs = (int)((fraction - seconds) * 1000.0 + 0.5);
if (msecs >= 1000)
	{
	msecs -= 1000;
	seconds++;
	}
if (seconds >= 60)
	{
	seconds -= 60;
	minutes++;
	}
if (minutes >= 60)
	{
	minutes -= 60;
	hours++;
	}
return
	((hours < 10) ? " " : "")
	+ hours + "h "
	+
	((minutes < 10) ? " " : "")
	+ minutes + "m "
	+
	((seconds < 10) ? " " : "")
	+ seconds + "."
	+
	((msecs < 10) ? "00" :
	((msecs < 100) ? "0" : ""))
	+ msecs + "s";
}


public static double Decimal_Degrees
	(
	String	parts
	)
	throws IllegalArgumentException, NumberFormatException
{
double
	degrees;
Words
	words = new Words (parts);
String
	part = words.Next_Word ();
int
	index = part.length () - 1;
if (index < 0)
	throw new IllegalArgumentException (ID + '\n'
		+ "Unable to convert to decimal degrees: \"" + parts + '"');
char
	character = part.charAt (index);
if (character == 'd' ||
	character == 'D')
	{
	//	Degrees.
	try {degrees = Integer.parseInt (part.substring (0, index));}
	catch (NumberFormatException exception)
		{
		throw new NumberFormatException (ID + '\n'
			+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
			+ "Degrees (the first value) must be an integer.");
		}
	}
else
	{
	//	Hours.
	if (character == 'h' ||
		character == 'H')
		part = part.substring (0, index);
	try {degrees = Integer.parseInt (part) * 15;}	//	Convert to degrees.
	catch (NumberFormatException exception)
		{
		throw new NumberFormatException (ID + '\n'
			+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
			+ "Hours (the first value) must be an integer.");
		}
	}
degrees = To_360 (degrees);

part = words.Next_Word ();
if ((index = part.length () - 1) >= 0)
	{
	//	Minutes.
	character = part.charAt (index);
	if (character == 'm' ||
		character == 'M')
		part = part.substring (0, index);
	int
		minutes;
	try {minutes = Integer.parseInt (part);}
	catch (NumberFormatException exception)
		{
		throw new NumberFormatException (ID + '\n'
			+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
			+ "Minutes (the second value) must be an integer.");
		}
	if (minutes < 0 ||
		minutes > 59)
		throw new IllegalArgumentException (ID + '\n'
			+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
			+ "Minutes (the second value) must be in the range [0-60).");
	degrees += (double)minutes * (15.0 / 60.0);	//	Convert to degrees.

	part = words.Next_Word ();
	if ((index = part.length () - 1) >= 0)
		{
		//	Seconds.
		character = part.charAt (index);
		if (character == 's' ||
			character == 'S')
			part = part.substring (0, index);
		double
			seconds;
		try {seconds = Double.parseDouble (part);}
		catch (NumberFormatException exception)
			{
			throw new NumberFormatException (ID + '\n'
				+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
				+ "Seconds (the third value) must be a real number.");
			}
		if (seconds <   0.0 ||
			seconds >= 60.0)
			throw new IllegalArgumentException (ID + '\n'
				+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
				+ "Seconds (the third value) must be in the range [0-60).");
		degrees += (double)seconds * (15.0 / 3600.0);	//	Convert to degrees.

		part = words.Next_Word ();
		if (part.length () != 0)
			throw new IllegalArgumentException (ID + '\n'
				+ "Unable to convert to decimal degrees: \"" + parts + "\"\n"
				+ "Seconds (the third value) should be the last part.");
		}
	}
return degrees;
}


/**	Convert planetocentric latitude to planetographic latitude.
<p>
	@param	latitude	The latitude, in radians, to convert.
	@return	The converted latitude value.
*/
public double Planetocentric_to_Planetographic
	(
	double	latitude
	)
{
if (Not_Identity &&
	Math.abs (latitude) < PI_OVER_2)
	latitude =
		StrictMath.atan (StrictMath.tan (latitude) * Coefficient_PC_to_PG);
return latitude;
}

/**	Convert planetographic latitude to planetocentric latitude.
<p>
	@param	latitude	The latitude, in radians, to convert.
	@return	The converted latitude value.
*/
public double Planetographic_to_Planetocentric
	(
	double	latitude
	)
{
if (Not_Identity &&
	Math.abs (latitude) < PI_OVER_2)
	latitude =
		StrictMath.atan (StrictMath.tan (latitude) * Coefficient_PG_to_PC);
return latitude;
}


protected double Projection_X_to_Sample
	(
	double	projection_x
	)
{return (projection_x / Meters_per_Pixel) + Sample_Offset;}

protected double Projection_Y_to_Line
	(
	double	projection_y
	)
{return (projection_y / Meters_per_Pixel) - Line_Offset;}


protected double Sample_to_Projection_X
	(
	double	sample
	)
{return (sample - Sample_Offset) * Meters_per_Pixel;}


protected double Line_to_Projection_Y
	(
	double	line
	)
{return (-Line_Offset - line) * Meters_per_Pixel;}

/*==============================================================================
	Derived values
*/
/**	Calculate the planet eccentricity.
<p><blockquote>
	E = sqrt (1 - polar_radius**2 / equitorial_radius**2)
</blockquote>
<p>
	@param	equitorial_radius	The planet equitorial radius.
	@param	polar_radius	The planet polar radius.
	@return	The planet eccentricity.
*/
public static double Eccentricity
	(
	double	polar_radius,
	double	equitorial_radius
	)
{
return
	StrictMath.sqrt (1.0 -
		((polar_radius * polar_radius) /
		(equitorial_radius * equitorial_radius)));
}

/**	Calculate the radius of the planet at some latitude.
<p><blockquote>
	R = Re * Rp / sqrt (a**2 + b**2)
</blockquote>
	where:
<p><blockquote>
	Re = {@link Projection#Equitorial_Radius() Equitorial_Radius}
	Rp = {@link Projection#Polar_Radius() Polar_Radius}
	a = Re * sin (latitude)<br>
	b = Rp * cos (latitude)
</blockquote>
<p>
	@param	latitude	The latitude, in radians, at which the local
		radius is to be calculated.
	@return	The radius of the planet at the specified latitude.
*/
public double Local_Radius
	(
	double	latitude
	)
{
double
	radius = 0.0;
if (Not_Identity)
	{
	if (latitude == PI_OVER_2)
		radius = Polar_Radius;
	else
	if (latitude == 0.0)
		radius = Equitorial_Radius;
	else
		{
		double
			coefficient_e = Equitorial_Radius * StrictMath.sin (latitude),
			coefficient_p = Polar_Radius      * StrictMath.cos (latitude);
		radius = Equitorial_Radius * Polar_Radius / StrictMath.sqrt
			((coefficient_p * coefficient_p) + (coefficient_e * coefficient_e));
		}
	}
return radius;
}

/**	Ensure that a longitude value is in the range 0 to 360 degrees.
<p>
	A negative longitude value is repeatedly increased by 360 until
	it is no longer negative. A longitude value greater than or equal to 360
	is repeatedly decreased by 360 until it is less than 360.
<p>
	@param	longitude	The longitude value to map to the 0 to 360 degree
		domain.
	@return	The longitude value in the 0-360 degree domain.
*/
public static double To_360
	(
	double	longitude
	)
{
while (longitude <    0.0) longitude += 360.0;
while (longitude >= 360.0) longitude -= 360.0;
return longitude;
}

/**	Ensure that a longitude value is in the range -180 to 180 degrees.
<p>
	A longitude value less than -180 is repeatedly increased by 360 until
	it is greater than or equal to -180. A longitude value greater than
	or equal to 180 is repeatedly decreased by 360 until it is less than
	180.
<p>
	@param	longitude	The longitude value to map to the -180 to 180
		degree domain.
	@return	The longitude value in the -180 to 180 degree domain.
*/
public static double To_180
	(
	double	longitude
	)
{
while (longitude < -180.0) longitude += 360.0;
while (longitude >= 180.0) longitude -= 360.0;
return longitude;
}

/*==============================================================================
	Helpers
*/
/**	Produce a class name for a projection name.
<p>
	The projection name is first trimmed of leading and trailing
	whitespace and all characters are set to lowercase. If the result
	starts with "polar" but is not followed by a space (' ') or underbar
	('_') character an underbar is inserted following the "polar" prefix.
	Then any remaining space characters are changed to '_' characters.
	The first character and every character following an underbar
	character is set to uppercase. Finally the "_Projection" String is
	appended.
<p>
	@param	projection_name	The projection name String.
	@return	The class name String.
*/
public static String Projection_Class_Name
	(
	String	projection_name
	)
{
if (projection_name == null)
	return null;
StringBuffer
	class_name = new StringBuffer
		(projection_name = projection_name.trim ().toLowerCase ());

//	Special case.
if (projection_name.startsWith ("polar") &&
	class_name.length () > 5 &&
	class_name.charAt (5) != ' ' &&
	class_name.charAt (5) != '_')
	class_name.insert (5, "_");

if (class_name.length () != 0)
	{
	class_name.setCharAt (0, Character.toUpperCase (class_name.charAt (0)));
	int
		index = 0;
	while ((index = class_name.indexOf (" ", index)) > 0)
		class_name.setCharAt (index++, '_');
	index = 0;
	while ((index = class_name.indexOf ("_", index) + 1) > 0)
		class_name.setCharAt (index,
			Character.toUpperCase (class_name.charAt (index)));
	class_name.append ("_Projection");
	}
return class_name.toString ();
}

/**	Find a named parameter in PDS label parameters.

	@param	label	The PDS label parameters.
	@param	name	The name of the parameter to find.
	@return	The Parameter that was found.
	@throws	NoSuchElementException	If the parameter was not found.
*/
protected static Parameter Find_Parameter
	(
	Parameter	label,
	String		name
	)
	throws NoSuchElementException
{
Parameter
	parameter = label.Find (name);
if (parameter == null)
	throw new NoSuchElementException (ID + '\n'
		+ "The required \"" + name
			+ "\" parameter was not found in the PDS label.");
return parameter;
}

/*==============================================================================
	main
*/
public static void main
	(
	String[]	args
	)
{
System.out.println (ID);

if (args.length == 0)
	Usage ();

Parameter
	parameters = null;
boolean
	image_coordinates = false,
	spherical = false;
Point2D.Double
	coordinate = null;
int
	index;
String
	number;

for (int
		count = 0;
		count < args.length;
		count++)
	{
	if (args[count].length () == 0)
		continue;
	if (args[count].charAt (0) == '-' &&
		args[count].length () > 1)
		{
		switch (args[count].charAt (1))
			{
			case 'A':
			case 'a':
				if ((count + 1) == args.length)
					Usage ();
				double
					angle = 0;
				try {angle = Double.parseDouble (args[++count]);}
				catch (NumberFormatException exception)
					{
					try {angle = Decimal_Degrees (args[count]);}
					catch (Exception except)
						{
						System.out.println
							("Not a valid angle: " + args[count] + '\n'
							+ except.getMessage ());
						Usage ();
						}
					System.out.println
						("Angle = " + args[count]);
					break;
					}
				System.out.println
					("Angle = " + angle + '\n'
					+"        " + Projection.Hours_Minutes_Seconds (angle) + '\n'
					+"        " + Projection.Degrees_Minutes_Seconds (angle));
				break;
			case 'S':
			case 's':
				spherical = true;
				break;
			case 'I':
			case 'i':
				image_coordinates = true;
			case 'W':
			case 'w':
				if ((count + 1) == args.length ||
					coordinate != null)
					Usage ();
				coordinate = Coordinate (args[++count]);
				break;
			default:
				System.out.println ("Unknown option: " + args[count]);
				Usage ();
			}
		}
	else
		{
		if (parameters != null)
			Usage ();
		try {parameters =
			new Parameter (new Parser (Streams.Get_Stream (args[count])));}
		catch (PVL_Exception exception)
			{
			System.out.println
				("Unable to read PVL source: " + args[count] + '\n'
				+ exception.getMessage ());
			System.exit (2);
			}
		}
	}
if (parameters != null ||
	coordinate != null)
	{
	if (parameters == null ||
		coordinate == null)
		Usage ();

	try
		{
		Projection.Use_Spherical (spherical);
		Projection
			projection = Projection.Create (parameters);
		if (image_coordinates)
			System.out.println
				("Image coordinate: " + coordinate + '\n'
				+"World coordinate: " + projection.to_World (coordinate));
		else
			System.out.println
				("World coordinate: " + coordinate + '\n'
				+"Image coordinate: " + projection.to_Image (coordinate));
		}
	catch (Throwable exception)
		{
		exception.printStackTrace ();
		System.exit (3);
		}
	}
System.exit (0);
}

private static Point2D.Double Coordinate
	(
	String	coordinate
	)
{
int
	index = coordinate.indexOf (',');
if (index < 0)
	Usage ();
Point2D.Double
	point = null;
try
	{
	point = new Point2D.Double
		(
		Double.parseDouble (coordinate.substring (0, index)),
		Double.parseDouble (coordinate.substring (index + 1))
		);
	}
catch (NumberFormatException exception)
	{
	System.out.println
		("Not a valid coordinate: " + coordinate);
	Usage ();
	}
return point;
}


public static void Usage ()
{
System.out.println
	("Usage: Projection"
		+ " [-Angle <angle]"
		+ " [-Spherical]"
		+ " <PVL parameters>"
		+ " -Image x,y | -World lon,lat");
System.exit (1);
}

}
