/*	Messenger_Styled_Writer

PIRL CVS ID: Messenger_Styled_Writer.java,v 1.13 2012/04/16 06:11:36 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Messenger;

import	PIRL.Utilities.Styled_Writer;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.Value;
import	PIRL.Strings.String_Buffer;
import	PIRL.PVL.PVL_Exception;

import	java.io.Writer;
import	java.io.IOException;
import	java.io.EOFException;
import	javax.swing.text.AttributeSet;
import	javax.swing.text.SimpleAttributeSet;
import	javax.swing.text.StyleConstants;
import	javax.swing.text.TabSet;
import	javax.swing.text.TabStop;
import	java.awt.Color;
import	java.util.Hashtable;
import	java.util.Vector;
import	java.util.Iterator;
import	java.util.Enumeration;


/**	A <i>Messenger_Styled_Writer</i> is a Writer that encapsulates what
	is written, along with Styled_Writer information, into a Message
	that is sent via a Messenger.
<p>
	Each write operation produces a Message that is sent to the
	Messenger to which a Messenger_Styled_Writer is bound. The Message
	contains a "Write" Action parameter and a "Written" parameter with a
	string value that is what was written. Styled_Writer information, if
	present, is contained in a "Style" Aggregate parameter in which each
	member describes style properties.
<p>
	Messages are routed by the Messenger according to the route-to
	address list that may be provided. If no route-to list is provided
	Messages that are sent will be delivered to the receiving Messenger's
	Employer.
<p>
	No buffering is done; each write operation causes a message to be
	immediately sent.
<p>
	Message sending may be suspended. While a Messenger_Styled_Writer is
	suspended all write operations have no effect. Argument parameters
	will still be checked for validity, where applicable.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.13
	@see		Writer
	@see		Messenger
	@see		Message
*/
public class Messenger_Styled_Writer
	extends Writer
	implements Styled_Writer
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Utilities.Messenger_Styled_Writer (1.13 2012/04/16 06:11:36)";


/**	The {@link Message#ACTION_PARAMETER_NAME} value that identifies the
	Message sent to the Messenger as coming from a Messenger_Styled_Writer.
*/
public static final String
	WRITE_ACTION			= "Write";

/**	The name of the Message parameter whose value is what was written.
*/
public static final String
	WRITTEN_PARAMETER_NAME	= "Written";

/**	The name of the Parameter Group that contains the style parameters.
*/
public static final String
	STYLE_PARAMETER_GROUP	= "Style";

/**	The suspension state of the Writer.
*/
public volatile boolean
	Suspended				= false;

private Messenger
	The_Messenger;

private Value
	Route;


private static final String
	NL						= System.getProperty ("line.separator");


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_WRITE			= 1 << 2,
	DEBUG_SEND			= 1 << 3,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Messenger_Styled_Writer on a Messenger with a Message route
	address list.
<p>
	<b>N.B.</b>: Unless the messenger {@link Messenger#Is_Connected() is
	connected} to its communication channel this Messenger_Styled_Writer
	will be initially {@link #Closed() closed} and thus useless.
<p>
	@param	messenger	The Messenger to which Writer Messages will be sent.
	@param	route	An Array Value containing the route-to address list
		to be used when sending Messages. If null no message routing will
		be used.
	@throws	IllegalArgumentException	If the messenger is null or the
		route is not an Array of address Strings.
*/
public Messenger_Styled_Writer
	(
	Messenger	messenger,
	Value		route
	)
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Messenger_Styled_Writer");
if ((The_Messenger = messenger) == null)
	throw new IllegalArgumentException (ID + NL
		+ "Can't construct with a null Messenger.");
if (route == null)
	{
	try {Route = new Value ().Type (Value.SEQUENCE);}
	catch (PVL_Exception exception) {/* Can't happen */}
	}
else
	{
	try {Route = new Value (Message.Valid_Route (route));}
	catch (PVL_Exception exception)
		{
		throw new IllegalArgumentException (ID + NL
			+ "Can't construct with an invalid address route." + NL
			+ exception.getMessage ());
		}
	}
}

/**	Construct a Messenger_Styled_Writer on a Messenger.
<p>
	Unless the Messenger {@link Messenger#Is_Connected() is connected} to
	its communication channel this Messenger_Styled_Writer will be
	initially {@link #Closed() closed} and thus useless.
<p>
	<b.N.B.</b>: No Message routing will be used; Messages sent will be
	delivered to the receiving Messenger's Employer rather than being
	forwarded to another Messenger.
<p>
	@param	messenger	The Messenger to which Writer Messages will be sent.
	@throws	IllegalArgumentException	If the messenger is null.
*/
public Messenger_Styled_Writer
	(
	Messenger	messenger
	)
{this (messenger, null);}


private Messenger_Styled_Writer ()
{}

/*==============================================================================
	Accessors
*/
/**	Get the Messenger used to send what is written.
<p>
	@return	A Messenger.
*/
public Messenger Writer_Messenger ()
{return The_Messenger;}

/**	Get the message route address list.
<p>
	@return A Value that contains a copy of the message route address list.
*/
public Value Route ()
{
try {return new Value (Route);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return null;
}

/**	Set the message route address list.
<p>
	@param	route	An Array Value containing the route-to address list
		to be used when sending Messages. If null no message routing will
		be used. The Value is copied.
	@return	This Messenger_Styled_Writer.
	@throws	PVL_Exception	If the route is not an Array of STRING type
		Values.
*/
public Messenger_Styled_Writer Route
	(
	Value	route
	)
	throws PVL_Exception
{
if (route == null)
	Route.Remove_All ();
else
	Route = new Value (Message.Valid_Route (route));
return this;
}

/**	Test if output has been suspended.
<p>
	@return	true if output has been suspended; false otherwise.
*/
public boolean Suspended ()
{return Suspended;}

/**	Turn output suspension on or off.
<p>
	@param	suspend	If true subsequent output will be suspended; if
		false output will occur.
	@return	This Messenger_Styled_Writer.
*/
public Messenger_Styled_Writer Suspend
	(
	boolean	suspend
	)
{Suspended = suspend; return this;}

/**	Test if this Messenger_Styled_Writer is capable of sending write messages.
<p>
	@return	true if the {@link #Writer_Messenger() Messenger} bound to
		this Messenger_Styled_Writer is no longer {@link
		Messenger#Is_Connected() connected} to it communication channel;
		false if this Messenger_Styled_Writer is currently able to write
		to its Messenger.
*/
public boolean Closed ()
{return ! The_Messenger.Is_Connected ();}

/*==============================================================================
	Writer
*/
/**	Write a portion of an array of characters.
<p>
	If the characters array is null or zero length, or the amount to
	write is zero, nothing is done.
<p>
	@param	characters	The char array containing the characters to be
		written. If null or empty nothing is done.
	@param	offset	Array offset from which to start writing characters.
	@param	amount	The number of characters to write. If zero nothing is
		done.
	@throws	IndexOutOfBoundsException	If the offset or amount are
		negative or the offset plus the amount is greater than the length
		of the array.
	@throws	IOException	If the write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public void write
	(
	char[]	characters,
	int		offset,
	int		amount
	)
	throws IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">>> Messenger_Styled_Writer.write: " + offset + '/' + amount + " -" + NL
		+ characters);
if (characters == null ||
	characters.length == 0 ||
	amount == 0)
	{
	if ((DEBUG & DEBUG_WRITE) != 0)
		System.out.println
			("    Nothing to write." + NL
			+"<<< Messenger_Styled_Writer.write");
	return;
	}

if (offset < 0 ||
	amount < 0 ||
	(offset + amount) > characters.length)
	throw new IndexOutOfBoundsException (ID + NL
		+ "Invalid values for " + characters.length
			+ " character array content -" + NL
		+ "  Array offset " + offset + " and amount " + amount + '.');

Send (new String (characters, offset, amount));
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		("<<< Messenger_Styled_Writer.write");
}

/**	Write an array of characters.
<p>
	If the characters array is null or zero length nothing is done.
<p>
	@param	characters	The char array containing the characters to be
		written. If null or empty nothing is done.
	@throws	IOException	If the write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public void write
	(
	char[]	characters
	)
	throws IOException
{
if (characters == null)
	return;
Send (new String (characters));
}

/**	Write a character.
<p>
	<b>N.B.</b>: Characters are not buffered so writing characters one
	at a time, rather than {@link #write(char[]) as an array} or {@link
	#write(String) string}, is very inefficient.
<p>
	@param	character	The character to be written in the 16 low-order
		bits.
	@throws	IOException	If the write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public void write
	(
	int		character
	)
	throws IOException
{
char[]
	array = new char[1];
array[0] = (char)character;
Send (new String (array));
}

/**	Write a portion of a String.
<p>
	If the string is null or zero length, or the amount to write is
	zero, nothing is done.
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@param	offset	String offset from which to start writing characters.
	@param	amount	The number of characters to write. If zero nothing
		is done.
	@throws	IndexOutOfBoundsException	If the offset or amount are
		negative or the offset plus the amount is greater than the length
		of the string.
	@throws	IOException	If the write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public void write
	(
	String			string,
	int				offset,
	int				amount
	)
	throws IOException
{
if (string == null ||
	amount == 0)
	return;

if (offset < 0 ||
	amount < 0 ||
	(offset + amount) > string.length ())
	throw new IndexOutOfBoundsException (ID + NL
		+ "Invalid values for " + string.length ()
			+ " character String content -" + NL
		+ "  String offset " + offset + " and amount " + amount + '.');

Send (string.substring (offset, offset + amount));
}

/**	Write a String.
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@throws	IOException	If the write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public void write
	(
	String			string
	)
	throws IOException
{
if (string == null)
	return;
Send (string);
}

/**	Flush the writers.
<p>
	Because no buffering is done flush has no effect.
*/
public void flush ()
{}

/**	Close the writer.
<p>
	This method has no effect. Because writing is done by sending
	Messages via the {@link #Writer_Messenger() Messenger} bound to this 
	Messenger_Styled_Writer, and the Messenger may be in use by other
	objects, its communication channel should not be disconnected by the
	Messenger_Styled_Writer. To disconnect the Messenger tell it that it
	is {@link Messenger#Done(String) done}.
<p>
	@see	#Closed()
*/
public void close ()
{}

/*==============================================================================
	Styled_Writer
*/
/**	Write a styled String.
<p>
	The AttributeSet used to specify the display style may be generated
	using the StyleConstants functions. For example:
<p>
<blockquote><pre><code>
SimpleAttributeSet style;
StyleConstants.setBold (style, true);
StyleConstants.setForeground (style, Color.RED);
</code></pre></blockquote>
<p>
	The following styles, indicated by their StyleConstants functions, are
	recognized by a Messenger_Styled_Writer:
<p>
<ul>
<li>setAlignment
<li>setBackground
<li>setBidiLevel
<li>setBold
<li>setFirstLineIndent
<li>setFontFamily
<li>setFontSize
<li>setForeground
<li>setItalic
<li>setLeftIndent
<li>setLineSpacing
<li>setRightIndent
<li>setSpaceAbove
<li>setSpaceBelow
<li>setStrikeThrough
<li>setSubscript
<li>setSuperscript
<li>setTabSet
<li>setUnderline
</ul>
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@param	style	An AttributeSet describing the style to be used when
		displaying the text. If null, no style will be applied.
	@throws	IOException	If the Write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public Messenger_Styled_Writer Write
	(
	String			string,
	AttributeSet	style
	)
	throws IOException
{
if (string == null)
	return this;
Send (string, style);
return this;
}

/**	Write a plain String.
<p>
	Using this method is the same a using {@link #Write(String, AttributeSet)}
	with a null style, or using (@link #write(String)}.
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@throws	IOException	If the write message could not be sent. This
		will be an EOFException if this Messenger_Styled_Writer is
		{@link #Closed() closed}.
*/
public Messenger_Styled_Writer Write
	(
	String			string
	)
	throws IOException
{
if (string == null)
	return this;
Send (string);
return this;
}

/*==============================================================================
	Messenger
*/
private void Send
	(
	String	string
	)
	throws IOException
{Send (string, null);}


private void Send
	(
	String			string,
	AttributeSet	style
	)
	throws IOException
{
if (Suspended ||
	string.length () == 0)
	return;
if ((DEBUG & (DEBUG_SEND | DEBUG_WRITE)) != 0)
	System.out.println
		(">>> Messenger_Styled_Writer.Send: " + NL
		+ '"' + string + '"');
if (Closed ())
	throw new EOFException (ID + NL
		+ "The Messenger communication channel connection is closed.");

Message
	message = Message.Action (WRITE_ACTION)
		.Set (WRITTEN_PARAMETER_NAME,
			new String_Buffer (string).special_to_escape ().toString ())
		.Add (Parameter_Style (style));
try
	{
	message.Route_To (Route);
	if ((DEBUG & (DEBUG_SEND | DEBUG_WRITE)) != 0)
		System.out.println
			("    Sending to " + Route.Description () + NL
			+ message);
	The_Messenger.Send (message);
	}
catch (PVL_Exception exception)
	{
	throw new IOException (ID + NL
		+ "Unable to generate the PVL Message content for" + NL
		+ '"' + string + '"' + NL
		+ exception.getMessage ());
	}
if ((DEBUG & (DEBUG_SEND | DEBUG_WRITE)) != 0)
	System.out.println
		("<<< Messenger_Styled_Writer.Send");
}

/*------------------------------------------------------------------------------
	Parameter to/from AttributeSet Conversion
*/
/**	Generate a Parameter that specifies a set of text styles.
<p>
	The set of style attributes is represented by a {@link
	#STYLE_PARAMETER_GROUP} Parameter Aggregate in which each style
	attribute is reprented by Pararmeter tht the name of the attribute
	and its Assignment Value has the value of the attribute.
<p>
	@param	style	An AttributeSet containing a set of text style
		attributes. If null, null is returned.
	@return	A Parameter Aggregate specifying each style attribute.
	@see	#Style_Parameter(Parameter)
*/
public static Parameter Parameter_Style
	(
	AttributeSet	style
	)
{
if (style == null)
	return null;

Parameter
	parameter,
	group =
		new Parameter (STYLE_PARAMETER_GROUP)
			.Classification (Parameter.GROUP);
String
	name;
Enumeration
	names = style.getAttributeNames ();
while (names.hasMoreElements ())
	{
	Object
		object = names.nextElement ();
	parameter = new Parameter (object.toString ());
	object = style.getAttribute (object);

	try
		{
		if (object instanceof Color)
			{
			parameter.Value (new Value
				((long)((Color)object).getRGB () & 0xFFFFFFFFL).Base (16));
			}
		else
		if (object instanceof Boolean)
			parameter.Value (new Value
				(object.toString ()).Type (Value.IDENTIFIER));
		else
		if (object instanceof TabSet)
			{
			Value
				value = new Value ().Type (Value.SEQUENCE);
			TabSet
				tab_set = (TabSet)object;
			TabStop
				tab_stop;
			for (int
					index = 0,
					total = tab_set.getTabCount ();
					index < total;
					index++)
				{
				tab_stop = tab_set.getTab (index);
				value
					.Add (new Value ().Type (Value.SEQUENCE)
						.Add (new Value (tab_stop.getPosition ()))
						.Add (new Value (tab_stop.getAlignment ()))
						.Add (new Value (tab_stop.getLeader ())));
				}
			parameter.Value (value);
			}
		else
			//	String, Integer or Float.
			parameter.Value (new Value (object));

		group.Add (parameter);
		}
	catch (PVL_Exception exception)
		{/* Ignore problem cases */}
	}

return group;
}


private static final int
	ALIGNMENT_CODE			= 0,
	BACKGROUND_CODE			= 1,
	BIDI_LEVEL_CODE			= 2,
	BOLD_CODE				= 3,
	FIRST_LINE_INDENT_CODE	= 4,
	FONT_FAMILY_CODE		= 5,
	FONT_SIZE_CODE			= 6,
	FOREGROUND_CODE			= 7,
	ITALIC_CODE				= 8,
	LEFT_INDENT_CODE		= 9,
	LINE_SPACING_CODE		= 10,
	RIGHT_INDENT_CODE		= 11,
	SPACE_ABOVE_CODE		= 12,
	SPACE_BELOW_CODE		= 13,
	STRIKETHROUGH_CODE		= 14,
	SUBSCRIPT_CODE			= 15,
	SUPERSCRIPT_CODE		= 16,
	TAB_SET_CODE			= 17,
	UNDERLINE_CODE			= 18;

/**	Recognized {@link #STYLE_PARAMETER_GROUP} parameter names.
*/
public static final String
	ALIGNMENT_NAME			= StyleConstants.Alignment.toString (),
	BACKGROUND_NAME			= StyleConstants.Background.toString (),
	BIDI_LEVEL_NAME			= StyleConstants.BidiLevel.toString (),
	BOLD_NAME				= StyleConstants.Bold.toString (),
	FIRST_LINE_INDENT_NAME	= StyleConstants.FirstLineIndent.toString (),
	FONT_FAMILY_NAME		= StyleConstants.FontFamily.toString (),
	FONT_SIZE_NAME			= StyleConstants.FontSize.toString (),
	FOREGROUND_NAME			= StyleConstants.Foreground.toString (),
	ITALIC_NAME				= StyleConstants.Italic.toString (),
	LEFT_INDENT_NAME		= StyleConstants.LeftIndent.toString (),
	LINE_SPACING_NAME		= StyleConstants.LineSpacing.toString (),
	RIGHT_INDENT_NAME		= StyleConstants.RightIndent.toString (),
	SPACE_ABOVE_NAME		= StyleConstants.SpaceAbove.toString (),
	SPACE_BELOW_NAME		= StyleConstants.SpaceBelow.toString (),
	STRIKETHROUGH_NAME		= StyleConstants.StrikeThrough.toString (),
	SUBSCRIPT_NAME			= StyleConstants.Subscript.toString (),
	SUPERSCRIPT_NAME		= StyleConstants.Superscript.toString (),
	TAB_SET_NAME			= StyleConstants.TabSet.toString (),
	UNDERLINE_NAME			= StyleConstants.Underline.toString ();

private static Hashtable<String, Integer>
	Style_Name_Codes	= new Hashtable<String, Integer> ();
static
	{
	Style_Name_Codes.put (ALIGNMENT_NAME,
			 new Integer (ALIGNMENT_CODE));
	Style_Name_Codes.put (BACKGROUND_NAME,
			 new Integer (BACKGROUND_CODE));
	Style_Name_Codes.put (BIDI_LEVEL_NAME,
			 new Integer (BIDI_LEVEL_CODE));
	Style_Name_Codes.put (BOLD_NAME,
			 new Integer (BOLD_CODE));
	Style_Name_Codes.put (FIRST_LINE_INDENT_NAME,
			 new Integer (FIRST_LINE_INDENT_CODE));
	Style_Name_Codes.put (FONT_FAMILY_NAME,
			 new Integer (FONT_FAMILY_CODE));
	Style_Name_Codes.put (FONT_SIZE_NAME,
			 new Integer (FONT_SIZE_CODE));
	Style_Name_Codes.put (FOREGROUND_NAME,
			 new Integer (FOREGROUND_CODE));
	Style_Name_Codes.put (ITALIC_NAME,
			 new Integer (ITALIC_CODE));
	Style_Name_Codes.put (LEFT_INDENT_NAME,
			 new Integer (LEFT_INDENT_CODE));
	Style_Name_Codes.put (LINE_SPACING_NAME,
			 new Integer (LINE_SPACING_CODE));
	Style_Name_Codes.put (RIGHT_INDENT_NAME,
			 new Integer (RIGHT_INDENT_CODE));
	Style_Name_Codes.put (SPACE_ABOVE_NAME,
			 new Integer (SPACE_ABOVE_CODE));
	Style_Name_Codes.put (SPACE_BELOW_NAME,
			 new Integer (SPACE_BELOW_CODE));
	Style_Name_Codes.put (STRIKETHROUGH_NAME,
			 new Integer (STRIKETHROUGH_CODE));
	Style_Name_Codes.put (SUBSCRIPT_NAME,
			 new Integer (SUBSCRIPT_CODE));
	Style_Name_Codes.put (SUPERSCRIPT_NAME,
			 new Integer (SUPERSCRIPT_CODE));
	Style_Name_Codes.put (TAB_SET_NAME,
			 new Integer (TAB_SET_CODE));
	Style_Name_Codes.put (UNDERLINE_NAME,
			 new Integer (UNDERLINE_CODE));
	}


private static int Style_Name_Code
	(
	String	style_name
	)
{
Integer
	code = Style_Name_Codes.get (style_name);
if (code == null)
	return -1;
return code.intValue ();
}

/**	Generate a SimpleAttributeSet from a Parameter Aggregate that
	specifies a set of text style attributes.
<p>
	The name of each Parameter in the Aggregate is mapped to a
	StyleConstants function that is used to set an attribute in a
	SimpleAttributeSet using the Value of the Parameter. The following
	attribute names are recognized as text style specification Parameter
	names:
<p>
<ul>
<li>{@link #ALIGNMENT_NAME}
<li>{@link #BACKGROUND_NAME}
<li>{@link #BIDI_LEVEL_NAME}
<li>{@link #BOLD_NAME}
<li>{@link #FIRST_LINE_INDENT_NAME}
<li>{@link #FONT_FAMILY_NAME}
<li>{@link #FONT_SIZE_NAME}
<li>{@link #FOREGROUND_NAME}
<li>{@link #ITALIC_NAME}
<li>{@link #LEFT_INDENT_NAME}
<li>{@link #LINE_SPACING_NAME}
<li>{@link #RIGHT_INDENT_NAME}
<li>{@link #SPACE_ABOVE_NAME}
<li>{@link #SPACE_BELOW_NAME}
<li>{@link #STRIKETHROUGH_NAME}
<li>{@link #SUBSCRIPT_NAME}
<li>{@link #SUPERSCRIPT_NAME}
<li>{@link #TAB_SET_NAME}
<li>{@link #UNDERLINE_NAME}
</u>
<p>
	Parameters that have unrecognized names or that are not an
	Assignment of the appropriate Value are ignored.
<p>
	@param	group	A Parameter Aggregate containing a list of
		text style specification Parameters.
	@return	A SimpleAttributeSet contained a set of text styles.
	@see	#Parameter_Style(AttributeSet)
*/
public static SimpleAttributeSet Style_Parameter
	(
	Parameter	group
	)
{
if (group == null ||
	! group.Is_Aggregate ())
	return null;

SimpleAttributeSet
	style = new SimpleAttributeSet ();
Parameter
	parameter;
Iterator
	parameters = group.iterator ();
while (parameters.hasNext ())
	{
	parameter = (Parameter)parameters.next ();
	try
		{
		switch (Style_Name_Code (parameter.Name ()))
			{
			case ALIGNMENT_CODE:
				StyleConstants.setAlignment (style,
					(int)parameter.Value ().long_Data ());
				break;
			case BACKGROUND_CODE:
				StyleConstants.setBackground (style,
					new Color ((int)parameter.Value ().long_Data ()));
				break;
			case BIDI_LEVEL_CODE:
				StyleConstants.setBidiLevel (style,
					(int)parameter.Value ().long_Data ());
				break;
			case BOLD_CODE:
				StyleConstants.setBold (style,
					parameter.Value ().String_Data ().equals ("true"));
				break;
			case FIRST_LINE_INDENT_CODE:
				StyleConstants.setFirstLineIndent (style,
					(float)parameter.Value ().double_Data ());
				break;
			case FONT_FAMILY_CODE:
				StyleConstants.setFontFamily (style,
					parameter.Value ().String_Data ());
				break;
			case FONT_SIZE_CODE:
				StyleConstants.setFontSize (style,
					(int)parameter.Value ().long_Data ());
				break;
			case FOREGROUND_CODE:
				StyleConstants.setForeground (style,
					new Color ((int)parameter.Value ().long_Data ()));
				break;
			case ITALIC_CODE:
				StyleConstants.setItalic (style,
					parameter.Value ().String_Data ().equals ("true"));
				break;
			case LEFT_INDENT_CODE:
				StyleConstants.setLeftIndent (style,
					(float)parameter.Value ().double_Data ());
				break;
			case LINE_SPACING_CODE:
				StyleConstants.setLineSpacing (style,
					(float)parameter.Value ().double_Data ());
				break;
			case RIGHT_INDENT_CODE:
				StyleConstants.setRightIndent (style,
					(float)parameter.Value ().double_Data ());
				break;
			case SPACE_ABOVE_CODE:
				StyleConstants.setSpaceAbove (style,
					(float)parameter.Value ().double_Data ());
				break;
			case SPACE_BELOW_CODE:
				StyleConstants.setSpaceBelow (style,
					(float)parameter.Value ().double_Data ());
				break;
			case STRIKETHROUGH_CODE:
				StyleConstants.setStrikeThrough (style,
					parameter.Value ().String_Data ().equals ("true"));
				break;
			case SUBSCRIPT_CODE:
				StyleConstants.setSubscript (style,
					parameter.Value ().String_Data ().equals ("true"));
				break;
			case SUPERSCRIPT_CODE:
				StyleConstants.setSuperscript (style,
					parameter.Value ().String_Data ().equals ("true"));
				break;
			case TAB_SET_CODE:
				Vector<TabStop>
					tab_stops = new Vector<TabStop> ();
				Value
					args,
					value;
				Iterator
					values = parameter.Value ().iterator ();
				while (values.hasNext ())
					{
					args = (Value)values.next ();
					if ((value = args.Get (0)) == null)
						continue;
					float
						position = (float)value.double_Data ();
					if ((value = args.Get (1)) == null)
						continue;
					int
						alignment = (int)value.long_Data ();
					if ((value = args.Get (2)) == null)
						continue;
					int
						leader = (int)value.long_Data ();
					tab_stops.add (new TabStop (position, alignment, leader));
					}
				TabStop[]
					tabs = new TabStop[tab_stops.size ()];
				StyleConstants.setTabSet (style,
					new TabSet ((TabStop[])tab_stops.toArray (tabs)));
				break;
			case UNDERLINE_CODE:
				StyleConstants.setUnderline (style,
					parameter.Value ().String_Data ().equals ("true"));
				break;
			}
		}
	catch (PVL_Exception exception)
		{/* Ignore problem cases */}
	}
return style;
}


}
