/*	String_Buffer

PIRL CVS ID:	String_Buffer.java,v 1.21 2012/04/16 06:15:35 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Strings;

import	java.lang.StringBuffer;


/**	A <I>String_Buffer</I> is a wrapper for the StringBuffer class.
<p>
	The StringBuffer class is declared final so it can not be extended.
	The String_Buffer class implements all of the StringBuffer methods
	(by simply calling the corresponding method) plus numerous methods
	that provide extended and convenient capabilities along the usual
	lines of String manipulation. The wrapped methods all function
	identically to their StringBuffer counterparts, of course, except
	when a reference to this StringBuffer object is returned it is a
	reference to this String_Buffer object.
<p>
	@see		StringBuffer
	@author		Bradford Castalia, UA/PIRL
	@version	1.21 
*/
public class String_Buffer
{
public static final String
	ID = "PIRL.Strings.String_Buffer (1.21 2012/04/16 06:15:35)";

private StringBuffer	_StringBuffer_;

/*==============================================================================
	Constants
*/
public static final String
	NL	= System.getProperty ("line.separator");

public static final char
	QUESTIONABLE_CHARACTER	= '?';

/*==============================================================================
	Constructors
*/
/**	Constructs a String_Buffer with no characters in it and an initial
	capacity of 16 characters.
*/
public String_Buffer ()
	{_StringBuffer_ = new StringBuffer ();}

/**	Constructs a String_Buffer with no characters in it and an initial
	capacity specified by the length argument.
<p>
	@param	length	The initial capacity.
	@throws	NegativeArraySizeException	If the length argument is less than 0.
*/
public String_Buffer (int length)
	{_StringBuffer_ = new StringBuffer (length);}

/**	Constructs a String_Buffer so that it represents the same sequence
	of characters as the string argument; in other words, the initial
	contents of the String_Buffer is a copy of the argument string. The
	initial capacity of the String_Buffer is 16 plus the length of the
	string argument.
<p>
	@param	string	The initial contents of the buffer.
*/
public String_Buffer (String string)
	{_StringBuffer_ = new StringBuffer (string);}

/*==============================================================================
	StringBuffer Methods (wrapped)
*/
/**	@see	StringBuffer#append(boolean)
*/
public String_Buffer append (boolean b)
	{_StringBuffer_.append (b); return this;}

/**	@see	StringBuffer#append(char)
*/
public String_Buffer append (char c)
	{_StringBuffer_.append (c); return this;}

/**	Append a byte as a character.
<p>
	@param	b	The byte to be appended.
*/
public String_Buffer append (byte b)
{return append ((char)b & 0xFF);}

/**	@see	StringBuffer#append(char[])
*/
public String_Buffer append (char[] s)
	{_StringBuffer_.append (s); return this;}

/**	Append a byte array as characters.
<p>
	Equivalent to append (bytes, 0, bytes.length);
<p>
	@param	bytes	An array of bytes.
*/
public String_Buffer append (byte[] bytes)
{return append (bytes, 0, bytes.length);}


/**	@see	StringBuffer#append(char[], int, int)
*/
public String_Buffer append (char[] s, int offset, int length)
	{_StringBuffer_.append (s, offset, length); return this;}

/**	Append a set of bytes as characters.
<p>
	A convenience method for append (new String (bytes, offset, length));
<p>
	@param	bytes	An array of bytes.
	@param	offset	The byte array offset of the first byte to be appended.
	@param	length	The number of bytes to be appended.
	@throws	IndexOutOfBoundsException	If the offset and/or length index
		bytes outside array bounds.
*/
public String_Buffer append (byte[] bytes, int offset, int length)
	throws IndexOutOfBoundsException
{
if (bytes != null)
	append (new String (bytes, offset, length));
return this;
}

/**	@see	StringBuffer#append(double)
*/
public String_Buffer append (double d)
	{_StringBuffer_.append (d); return this;}

/**	@see	StringBuffer#append(float)
*/
public String_Buffer append (float f)
	{_StringBuffer_.append (f); return this;}

/**	@see	StringBuffer#append(int)
*/
public String_Buffer append (int i)
	{_StringBuffer_.append (i); return this;}

/**	@see	StringBuffer#append(long)
*/
public String_Buffer append (long l)
	{_StringBuffer_.append (l); return this;}

/**	@see	StringBuffer#append(Object)
*/
public String_Buffer append (Object o)
	{_StringBuffer_.append (o); return this;}

/**	@see	StringBuffer#append(String)
*/
public String_Buffer append (String s)
	{_StringBuffer_.append (s); return this;}

/**	@see	StringBuffer#capacity()
*/
public int capacity ()
	{return _StringBuffer_.capacity ();}

/**	@see	StringBuffer#charAt(int)
*/
public char charAt (int index)
	{return _StringBuffer_.charAt (index);}

/**	@see	StringBuffer#delete(int, int)
*/
public String_Buffer delete (int start, int end)
	{_StringBuffer_.delete (start, end); return this;}

/**	@see	StringBuffer#deleteCharAt(int)
*/
public String_Buffer deleteCharAt (int index)
	{_StringBuffer_.deleteCharAt (index); return this;}

/**	@see	StringBuffer#ensureCapacity(int)
*/
public void ensureCapacity (int minimumCapacity)
	{_StringBuffer_.ensureCapacity (minimumCapacity);}

/**	@see	StringBuffer#getChars(int, int, char[], int)
*/
public void getChars (int srcBegin, int srcEnd, char[] dst, int dstBegin)
	{_StringBuffer_.getChars (srcBegin, srcEnd, dst, dstBegin);}

/**	@see	StringBuffer#insert(int, boolean)
*/
public String_Buffer insert (int index, boolean b)
	{_StringBuffer_.insert (index, b); return this;}

/**	@see	StringBuffer#insert(int, char)
*/
public String_Buffer insert (int index, char c)
	{_StringBuffer_.insert (index, c); return this;}

/**	@see	StringBuffer#insert(int, char[])
*/
public String_Buffer insert (int index, char[] s)
	{_StringBuffer_.insert (index, s); return this;}

/**	@see	StringBuffer#insert(int, char[], int, int)
*/
public String_Buffer insert (int index, char[] s, int offset, int length)
	{_StringBuffer_.insert (index, s, offset, length); return this;}

/**	@see	StringBuffer#insert(int, double)
*/
public String_Buffer insert (int index, double d)
	{_StringBuffer_.insert (index, d); return this;}

/**	@see	StringBuffer#insert(int, float)
*/
public String_Buffer insert (int index, float f)
	{_StringBuffer_.insert (index, f); return this;}

/**	@see	StringBuffer#insert(int, int)
*/
public String_Buffer insert (int index, int i)
	{_StringBuffer_.insert (index, i); return this;}

/**	@see	StringBuffer#insert(int, long)
*/
public String_Buffer insert (int index, long l)
	{_StringBuffer_.insert (index, l); return this;}

/**	@see	StringBuffer#insert(int, Object)
*/
public String_Buffer insert (int index, Object o)
	{_StringBuffer_.insert (index, o); return this;}

/**	@see	StringBuffer#insert(int, String)
*/
public String_Buffer insert (int index, String s)
	{_StringBuffer_.insert (index, s); return this;}

/**	@see	StringBuffer#length()
*/
public int length ()
	{return _StringBuffer_.length ();}

/**	@see	StringBuffer#replace(int, int, String)
*/
public String_Buffer replace (int start, int end, String string)
	{_StringBuffer_.replace (start, end, string); return this;}

/**	@see	StringBuffer#reverse()
*/
public String_Buffer reverse ()
	{_StringBuffer_.reverse (); return this;}

/**	@see	StringBuffer#setCharAt(int, char)
*/
public void setCharAt (int index, char c)
	{_StringBuffer_.setCharAt (index, c);}

/**	@see	StringBuffer#setLength(int)
*/
public void setLength (int length)
	{_StringBuffer_.setLength (length);}

/**	@see	StringBuffer#substring(int)
*/
public String substring (int index)
	{return _StringBuffer_.substring (index);}

/**	@see	StringBuffer#substring(int, int)
*/
public String substring (int start, int end)
	{return _StringBuffer_.substring (start, end);}

/**	@see	StringBuffer#toString()
*/
public String toString ()
	{return _StringBuffer_.toString ();}

/*==============================================================================
	Extension Methods
*/
/**	Skips over any and all characters in the skip String, starting with
	the character at the specified index.
<p>
	@param	index	The index of the first character to test.
	@param	skip	The set of characters to be skipped.
	@return	The index of the character at or beyond the starting index
		that is not in the skip String, or the end
		(<CODE>length</CODE>) index if all remaining characters from
		the starting index are in the skip String.
	@see	String#indexOf(int)
	@see	String_Buffer#charAt(int)
*/
public int skip_over
	(
	int		index,
	String	skip
	)
{
int
	end = length ();

if (index < 0 || index >= end)
	return end;

while (index < end)
	if (skip.indexOf (charAt (index++)) < 0)
		//	The character isn't in the skip string.
		return --index;
return index;
}

/**	@see	#skip_over(int, String)
*/
public int skipOver (int index, String skip)
{return skip_over (index, skip);}


/**	Skips back over any and all characters in the skip String, starting
	with the character at the specified index.
<p>
	@param	index	The index of the first character to test.
	@param	skip	The set of characters to be skipped.
	@return	The index of the last character at or before the starting
		index that is not in the skip String, or -1 if all characters
		up to and including the starting index are in the skip String.
	@see	String#indexOf(int)
	@see	String_Buffer#charAt(int)
*/
public int skip_back_over
	(
	int		index,
	String	skip
	)
{
if (index >= length ())
	index = length () - 1;

while (index >= 0)
	if (skip.indexOf (charAt (index--)) < 0)
		//	The character isn't in the skip string.
		return ++index;
return index;
}

/**	@see	#skip_back_over(int, String)
*/
public int skipBackOver (int index, String skip)
{return skip_back_over (index, skip);}


/**	Skips until a character also in the find String is found, starting
	with the character at the specified index.
<p>
	@param	index	The index of the first character to test.
	@param	find	The set of potential characters to find.
	@return	The index of the character at or beyond the starting index
		that is in the find String, or -1 if all remaining characters
		from the starting index are not in the find String.
	@see	String#indexOf(int)
	@see	String_Buffer#charAt(int)
*/
public int skip_until
	(
	int		index,
	String	find
	)
{
int
	end = length ();

if (index < 0)
	return -1;

while (index < end)
	if (find.indexOf (charAt (index++)) >= 0)
		//	The character is in the find string.
		return --index;
return -1;
}

/**	@see	#skip_until(int, String)
*/
public int skipUntil (int index, String find)
{return skip_until (index, find);}


/**	Skips back until a character also in the find String is found,
	starting with the character at the specified index.
<p>
	@param	index	The index of the first character to test.
	@param	find	The set of potential characters to find.
	@return	The index of the last character at or before the starting
		index that is in the find String, or -1 if all characters up to
		and including the starting index are not in the find String.
	@see	String#indexOf(int)
	@see	String_Buffer#charAt(int)
*/
public int skip_back_until
	(
	int		index,
	String	find
	)
{
if (index >= length ())
	index = length () - 1;

while (index >= 0)
	if (find.indexOf (charAt (index--)) >= 0)
		//	The character is in the find string.
		return ++index;
return index;
}

/**	@see	#skip_back_until(int, String)
*/
public int skipBackUntil (int index, String find)
{return skip_until (index, find);}


/**	Gets the index of the next character matching the specified
	character, starting with the specified index.
<p>
	@param	index	The index of the first character to test.
	@param	character	The character to match.
	@return	The index of the next character at or beyond the starting
		index that matches the specified character, or -1 if all
		remaining characters do not match the specified character.
	@see	String#indexOf(int, int)
*/
public int index_of
	(
	int		index,
	char	character
	)
{
int
	end = length ();

if (index < 0)
	return -1;

while (index < end)
	if (charAt (index++) == character)
		//	Found the character.
		return --index;
return -1;
}

/**	@see	#index_of(int, char)
*/
public int indexOf (int index, char character)
{return index_of (index, character);}


/*	Return the index of the pattern String
	starting from the specified index of the string buffer;
	or -1 if the pattern isn't found.
*/
/**	Gets the index of the next occurance of the substring matching the
	pattern String, starting with the specified index.
<p>
	@param	index	The starting index from which to search for the pattern String.
	@param	pattern	The pattern String to match.
	@return	The index of the next substring at or beyond the starting
		index that matches the pattern String, or -1 if the pattern is
		not found.
	@see	String#indexOf(String, int)
*/
public int index_of
	(
	int		index,
	String	pattern
	)
{
try {return toString ().indexOf (pattern, index);}
catch (Exception exception) {return -1;}
}

/**	@see	#index_of(int, String)
*/
public int indexOf (int index, String pattern)
{return index_of (index, pattern);}

/*------------------------------------------------------------------------------
*/
/**	Tests if the pattern String is equal to the substring of the same length
	starting at the index.
<p>
	@param	index	The starting index from which to test for the pattern String.
	@param	pattern	The pattern String to match.
	@return	true if they are equal; false otherwise.
	@see	String#startsWith(String, int)
*/
public boolean equals
	(
	int		index,
	String	pattern
	)
{
try {return toString ().startsWith (pattern, index);}
catch (Exception exception) {return false;}
}


/**	Tests if the pattern String is equal regardless of case to the
	substring of the same length starting at the index.
<p>
	@param	index	The starting index from which to test for the pattern String.
	@param	pattern	The pattern String to match.
	@return	true if they are equal while ignoring case; false otherwise.
	@see	String#regionMatches(boolean, int, String, int, int)
*/
public boolean equals_ignore_case
	(
	int		index,
	String	pattern
	)
{return toString ().regionMatches (true, index, pattern, 0, pattern.length ());}

/**	@see	#equals_ignore_case(int, String)
*/
public boolean equalsIgnoreCase (int index, String pattern)
{return equals_ignore_case (index, pattern);}

/*------------------------------------------------------------------------------
*/
/**	Replaces all occurances of the old_string with the new_string,
	starting at the specified index.
<p>
	Recursive replacements are avoided: If the new_string contains the
	old_string, the new old_string remains in its location within the
	new_string; the search for an old_string continues at the location
	after where the new_string has been inserted.
<p>
	@param	index	The starting index from which to test for the old_string.
	@param	old_string	The substring to find.
	@param	new_string	The String to replace the old_string.
	@return	This, possibly modified, String_Buffer.
	@see	#index_of(int, String)
	@see	StringBuffer#replace(int, int, String)
*/
public String_Buffer replace
	(
	int		index,
	String	old_string,
	String	new_string
	)
{
while ((index = index_of (index, old_string)) >= 0)
	{
	replace (index, index + old_string.length (), new_string);
	index += new_string.length ();
	} 
return this;
}


/**	Replaces all sequences of any and all characters from the span String
	with the substitute String, starting at the specified index.
<p>
	Each replacement occurs on the longest sequence of characters all
	from the span String set.
<p>
	Recursive replacements are avoided: If the substitute contains any
	characters from the span the substitute remains unaffected; the
	search for another span continues at the location after where the
	substitute has been inserted.
<p>
	@param	index	The starting index from which to test for span characters.
	@param	span	The set of potential characters for replacement.
	@param	substitute	The String to replace each span sequence.
	@return	This, possibly modified, String_Buffer.
	@see	#skip_until(int, String)
	@see	#skip_over(int, String)
	@see	StringBuffer#replace(int, int, String)
*/
public String_Buffer replace_span
	(
	int		index,
	String	span,
	String	substitute
	)
{
while ((index = skip_until (index, span)) >= 0)
	{
	replace (index, skip_over (index, span), substitute);
	index += substitute.length ();
	}
return this;
}

/**	@see	#replace_span(int, String, String)
*/
public String_Buffer replaceSpan (int index, String span, String substitute)
{return replace_span (index, span, substitute);}


/**	Trims whitespace from both ends of a substring of the String_Buffer,
	starting at the start index and ending before the end index.
<p>
	@param	start	The index of the starting character of the substring.
	@param	end		The index of the end (last + 1) character of the substring.
	@return	This, possibly modified, String_Buffer.
	@see	StringBuffer#substring(int, int)
	@see	String#trim()
	@see	StringBuffer#replace(int, int, String)
*/
public String_Buffer trim
	(
	int		start,
	int		end
	)
{replace (start, end, substring (start, end).trim ()); return this;}

/**	Trims whitespace from both ends of the String_Buffer.
<p>
	@return	This, possibly modified, String_Buffer.
	@see	#trim(int, int)
*/
public String_Buffer trim ()
{return trim (0, length ());}

/**	Trims whitespace from the beginning of the String_Buffer.
<p>
	@return	This, possibly modified, String_Buffer.
	@see	Character#isWhitespace(char)
	@see	#delete(int, int)
*/
public String_Buffer trim_beginning ()
{
int
	index = 0,
	end = length ();
while (index < end)
	if (! Character.isWhitespace (charAt (index++)))
		//	Found a non-whitespace character.
		return delete (0, --index);
return this;
}

/**	Trims whitespace from the end of the String_Buffer.
<p>
	@return	This, possibly modified, String_Buffer.
	@see	Character#isWhitespace(char)
	@see	#delete(int, int)
*/
public String_Buffer trim_end ()
{
int
	index = length ();
while (--index >= 0)
	if (! Character.isWhitespace (charAt (index)))
		//	Found a non-whitespace character.
		return delete (++index, length ());
return this;
}

/**	Trims a character from both ends of a string.
<p>
	@param	character	The character to be trimmed off the ends.
	@return	This, possibly modified, String_Buffer.
*/
public String_Buffer trim
	(
	char	character
	)
{
if (length () > 0 &&
	charAt (0) == character)
	deleteCharAt (0);
if (length () > 0 &&
	charAt (length () - 1) == character)
	deleteCharAt (length () - 1);
return this;
}

/**	Trims a sequence of the same character from both ends of a string.
<p>
	@param	character	The character to be trimmed off the ends.
	@return	This, possibly modified, String_Buffer.
*/
public String_Buffer trim_all
	(
	char	character
	)
{
int
	index = 0;
for (index = 0;
	 index < length ();
	 index++)
	if (charAt (index) != character)
		break;
delete (0, index);
for (index = length ();
	 --index >= 0;)
	if (charAt (index) != character)
		break;
delete (++index, length ());
return this;
}

/**	Empties the buffer of all characters.
<p>
	@return	This String_Buffer.
*/
public String_Buffer clear ()
{return delete (0, length ());}

/**	Clean all unescaped occurances of a character from a String.
<p>
	Escaped characters, which are preceded by a backslash ('\') character,
	are always ignored. All other characters that match the character
	to be cleaned are removed from the String.
<p>
	The backslash character can be cleaned from the String. Escaped
	backslash characters will result in a single backslash character.
	Thus a String might be first cleaned of a non-backslash character
	and then cleaned of backslash characters to remove any escape
	characters remaining from the previous cleaning.
<p>
	@param	character	The char to be cleaned from the String.
	@return	This, possibly modified, String_Buffer.
	@see	#escape_to_special()
*/
public String_Buffer clean
	(
	char	character
	)
{
for (int
		index = 0;
		index < length ();
		index++)
	{
	char
		it = charAt (index);
	if (it == character)
		{
		deleteCharAt (index);
		if (character != '\\')
			--index;
		}
	else if (it == '\\')
		++index;
	}
return this;
}


/**	Substitutes escape sequences with special characters.
<p>
	The following escape sequences, and their corresponding special
	characters, are recognized:
<p>
	<BLOCKQUOTE>
	\b - Backspace (BS)<BR>
	\t - Horizontal tab (HT)<BR>
	\n - Newline (NL)<BR>
	\f - Form feed (FF)<BR>
	\r - Carriage return (CR)<BR>
	\X - The character X<BR>
	\0nnn - The character having the octal value nnn (0 <= nnn <= 177777)<BR>
	</BLOCKQUOTE>
<p>
	The escape sequences will be substituted for their corresponding
	special characters, which will shorten the length of the
	String_Buffer. All backslash characters, except those that are
	themselves escaped, will be removed from the String_Buffer.
<p>
	<b>N.B.</b>: If an octal valued escape sequence is immediately
	followed by an octal digit (0-7) the latter will be mistaken for part
	of the escape sequence unless the maximum 6 digit limit has been
	reached. Because of this it may be preferable to encode non-printable
	characters using standard {@link #to_character_references() character
	references}.
<p>
	@return	This String_Buffer. The conversion is done in-place.
	@see	#special_to_escape()
*/
public String_Buffer escape_to_special ()
{
int
	 index;
for (index = 0;
	 index < (length () - 1);
	 index++)
	{
	if (charAt (index) == '\\')
		{
		//	Delete the backslash.
		deleteCharAt (index);
		//	Check the escaped character.
		switch (charAt (index))
			{
			case 'b':
				//	BS
				setCharAt (index, '\b');
				break;
			case 't':
				//	HT
				setCharAt (index, '\t');
				break;
			case 'n':
				//	NL
				deleteCharAt (index);
				insert (index, NL);
				break;
			case 'f':
				//	FF
				setCharAt (index, '\f');
				break;
			case 'r':
				//	CR
				setCharAt (index, '\r');
				break;
			case '0':
				//	Octal value.
				char
					character,
					new_character = 0;
				int
					 end_index;
				for (end_index = index + 1;
					 end_index < length () &&
					(end_index - index) < 7;	//	No more that six digits.
					 end_index++)
					{
					//	Only octal digits are acceptable.
					if ((character = charAt (end_index)) > '7' ||
						 character < '0')
						break;
					new_character *= 8;
					new_character += character - 48;
					}
				if (new_character <= Character.MAX_VALUE)
					{
					setCharAt (index, new_character);
					delete (index + 1, end_index);
					}
				else
					//	Skip invalid escape sequence.
					index = end_index - 1;
				break;
			}
		}
	}
return this;
}

/**	Substitutes escape sequences with special characters.
<p>
	@param	string	The String to be filtered. If null, null is returned.
	@return	The filtered String.
	@see	#escape_to_special()
*/
public static String escape_to_special
	(
	String	string
	)
{
if (string != null)
	string = new String_Buffer (string)
		.escape_to_special ().toString ();
return string;
}


/**	Substitutes special characters with escape sequences.
<p>
	All control characters - less than ' ' (32) or greater than '~'
	(126) - will be substituted with escape sequences:
<p>
	<BLOCKQUOTE>
	\b - for backspace (BS, 8)<BR>
	\t - for horizontal tab (HT, 9)<BR>
	\n - for newline (NL, 10)<BR>
	\f - for form feed (FF, 12)<BR>
	\r - for carriage return (CR, 13)<BR>
	\0nnn - for all other special characters where nnn is the octal
		value of the character (1-6 digits).<BR>
	</BLOCKQUOTE>
<p>
	The escape sequences will substitute for their corresponding
	special characters, which will increase the length of the
	String_Buffer.
<p>
	<b>N.B.</b>: If an octal valued escape sequence is immediately
	followed by an octal digit (0-7) the latter will be mistaken for part
	of the escape sequence unless the maximum 6 digit limit has been
	reached. Because of this it may be preferable to encode non-printable
	characters using standard {@link #to_character_references() character
	references}.
<p>
	@return	This String_Buffer. The conversion is done in-place.
	@see	#escape_to_special()
*/
public String_Buffer special_to_escape ()
{
char
	character;
String
	escape;
int
	index = 0;
while (index < length ())
	{
	character = charAt (index);
	if (character < 32 ||
		character > 126)
		{
		//	Check the special character.
		switch (character)
			{
			case '\b':
				//	BS
				escape = "b";
				break;
			case '\t':
				//	HT
				escape = "t";
				break;
			case '\n':
				//	NL
				escape = "n";
				break;
			case '\f':
				//	FF
				escape = "f";
				break;
			case '\r':
				//	CR
				escape = "r";
				break;
			default:
				//	Octal value.
				escape = "0" + Integer.toString ((int)character, 8);
			}
		setCharAt (index++, '\\');
		insert (index, escape);
		index += escape.length ();
		}
	else
		index++;
	}
return this;
}

/**	Substitutes special characters with escape sequences.
<p>
	@param	string	The String to be filtered. If null, null is returned.
	@return	The filtered String.
	@see	#special_to_escape()
*/
public static String special_to_escape
	(
	String	string
	)
{
if (string != null)
	string = new String_Buffer (string)
		.special_to_escape ().toString ();
return string;
}


/**	Substitutes numeric character references with characters.
<p>
	All numeric character references will be replaced with their single
	character value. A numeric character references has the form:
<p>
	<BLOCKQUOTE>
	<b>&amp;#</b>[<b>X</b>]<i>N</i><b>;</b>
	</BLOCKQUOTE>
<p>
	Where <i>N</i> is the text representation of the character value. If
	the 'X' (case insensitive) is present the value is in hexadecimal
	representation, otherwise it is decimal. <b>N.B.</b>: The semicolon
	terminating the numerical character reference must be present as
	well as the &amp;# prefix that signals the presence of the
	reference.
<p>
	@return	This String_Buffer. The conversion is done in-place.
	@see	#to_character_references()
*/
public String_Buffer from_character_references ()
{
char
	previous = 0,
	character;
int
	 index = 0;
while (index < length ())
	{
	if ((character = charAt (index)) == '#' &&
		previous == '&')
		{
		if (++index == length ())
			break;
		int
			start = index - 2,
			begin = index,
			radix = 10;
		if ((character = charAt (index)) == 'X' ||
			character == 'x')
			{
			if (++index == length ())
				break;
			begin++;
			radix = 16;
			character = charAt (index);
			}
		while (Character.digit (character, radix) >= 0)
			{
			if (++index == length ())
				break;
			character = charAt (index);
			}
		if (index > start)
			{
			try
				{
				int
					value = Integer.parseInt (substring (begin, index), radix);
				if (value <= Character.MAX_VALUE)
					{
					setCharAt (start, (char)value);
					delete (start + 1, ++index);
					index = start;
					}
				}
			catch (NumberFormatException exception) {}
			}
		}
	index++;
	previous = character;
	}
return this;
}

/**	Substitutes numeric character references with characters.
<p>
	@param	string	The String to be filtered. If null, null is returned.
	@return	The filtered String.
	@see	#from_character_references()
*/
public static String from_character_references
	(
	String	string
	)
{
if (string != null)
	string = new String_Buffer (string)
		.from_character_references ().toString ();
return string;
}


/**	Substitutes non-printable characters with numeric character references.
<p>
	All non-printable characters - less than ' ' (32) or greater than
	'~' (126) - will be substituted with hexadecimal numeric character
	references. A hexadecimal numeric character references has the form:
<p>
	<BLOCKQUOTE>
	<b>&amp;#X</b><i>N</i><b>;</b>
	</BLOCKQUOTE>
<p>
	Where <i>N</i> is the hexadecimal (uppercase) representation of the
	character value.
<p>
	@return	This String_Buffer. The conversion is done in-place.
	@see	#from_character_references()
	@see	#special_to_escape()
*/
public String_Buffer to_character_references ()
{
char
	character;
String
	reference;
int
	index = 0;
while (index < length ())
	{
	character = charAt (index);
	if (character < 32 ||
		character > 126)
		{
		reference =
			"#X" + Integer.toString ((int)character, 16).toUpperCase () + ';';
		setCharAt (index++, '&');
		insert (index, reference);
		index += reference.length ();
		}
	else
		index++;
	}
return this;
}

/**	Substitutes non-printable characters with numeric character references.
<p>
	@param	string	The String to be filtered. If null, null is returned.
	@return	The filtered String.
	@see	#to_character_references()
*/
public static String to_character_references
	(
	String	string
	)
{
if (string != null)
	string = new String_Buffer (string)
		.to_character_references ().toString ();
return string;
}

/**	Substitute all non-ASCII and unprintable characters with a question mark.
<p>
	All non-printable characters - less than ' ' (32) or greater than
	'~' (126) - will be substituted with the {@link #QUESTIONABLE_CHARACTER
	question mark} ('?') character.
<p>
	@return	This String_Buffer. The conversion is done in-place.
*/
public String_Buffer to_printable_ASCII ()
{
char
	character;
int
	index = length ();
while (index-- != 0)
	{
	character = charAt (index);
	if (character < 32 ||
		character > 126)
		setCharAt (index, QUESTIONABLE_CHARACTER);
	}
return this;
}

/**	Substitute all non-ASCII and unprintable characters with a question mark.
<p>
	@param	string	The string to be filtered. If null, null is returned.
	@return	The filtered String.
	@see	#to_printable_ASCII()
*/
public static String to_printable_ASCII
	(
	String	string
	)
{
if (string != null)
	string = new String_Buffer (string)
		.to_printable_ASCII ().toString ();
return string;
}


}	//	End of class
